"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RESERVED_TUNNEL_INSIDE_CIDR = exports.VpnConnection = exports.VpnGateway = exports.VpnConnectionType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const net = require("net");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The VPN connection type.
 */
var VpnConnectionType;
(function (VpnConnectionType) {
    /**
     * The IPsec 1 VPN connection type.
     */
    VpnConnectionType["IPSEC_1"] = "ipsec.1";
    /**
     * Dummy member
     * TODO: remove once https://github.com/aws/jsii/issues/231 is fixed
     */
    VpnConnectionType["DUMMY"] = "dummy";
})(VpnConnectionType = exports.VpnConnectionType || (exports.VpnConnectionType = {}));
/**
 * The VPN Gateway that shall be added to the VPC
 *
 * @resource AWS::EC2::VPNGateway
 */
class VpnGateway extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpnGatewayProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        // This is 'Default' instead of 'Resource', because using 'Default' will generate
        // a logical ID for a VpnGateway which is exactly the same as the logical ID that used
        // to be created for the CfnVPNGateway (and 'Resource' would not do that).
        const vpnGW = new ec2_generated_1.CfnVPNGateway(this, 'Default', props);
        this.gatewayId = vpnGW.ref;
    }
}
exports.VpnGateway = VpnGateway;
_a = JSII_RTTI_SYMBOL_1;
VpnGateway[_a] = { fqn: "@aws-cdk/aws-ec2.VpnGateway", version: "1.155.0" };
/**
 * Define a VPN Connection
 *
 * @resource AWS::EC2::VPNConnection
 */
class VpnConnection extends core_1.Resource {
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpnConnectionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        if (!props.vpc.vpnGatewayId) {
            props.vpc.enableVpnGateway({
                type: 'ipsec.1',
                amazonSideAsn: props.asn,
            });
        }
        if (!core_1.Token.isUnresolved(props.ip) && !net.isIPv4(props.ip)) {
            throw new Error(`The \`ip\` ${props.ip} is not a valid IPv4 address.`);
        }
        const type = VpnConnectionType.IPSEC_1;
        const bgpAsn = props.asn || 65000;
        const customerGateway = new ec2_generated_1.CfnCustomerGateway(this, 'CustomerGateway', {
            bgpAsn,
            ipAddress: props.ip,
            type,
        });
        this.customerGatewayId = customerGateway.ref;
        this.customerGatewayAsn = bgpAsn;
        this.customerGatewayIp = props.ip;
        // Validate tunnel options
        if (props.tunnelOptions) {
            if (props.tunnelOptions.length > 2) {
                throw new Error('Cannot specify more than two `tunnelOptions`');
            }
            if (props.tunnelOptions.length === 2 && props.tunnelOptions[0].tunnelInsideCidr === props.tunnelOptions[1].tunnelInsideCidr) {
                throw new Error(`Same ${props.tunnelOptions[0].tunnelInsideCidr} \`tunnelInsideCidr\` cannot be used for both tunnels.`);
            }
            props.tunnelOptions.forEach((options, index) => {
                if (options.preSharedKey && options.preSharedKeySecret) {
                    throw new Error("Specify at most one of 'preSharedKey' and 'preSharedKeySecret'.");
                }
                if (options.preSharedKey && !core_1.Token.isUnresolved(options.preSharedKey) && !/^[a-zA-Z1-9._][a-zA-Z\d._]{7,63}$/.test(options.preSharedKey)) {
                    /* eslint-disable max-len */
                    throw new Error(`The \`preSharedKey\` ${options.preSharedKey} for tunnel ${index + 1} is invalid. Allowed characters are alphanumeric characters and ._. Must be between 8 and 64 characters in length and cannot start with zero (0).`);
                    /* eslint-enable max-len */
                }
                if (options.tunnelInsideCidr) {
                    if (exports.RESERVED_TUNNEL_INSIDE_CIDR.includes(options.tunnelInsideCidr)) {
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is a reserved inside CIDR.`);
                    }
                    if (!/^169\.254\.\d{1,3}\.\d{1,3}\/30$/.test(options.tunnelInsideCidr)) {
                        /* eslint-disable-next-line max-len */
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is not a size /30 CIDR block from the 169.254.0.0/16 range.`);
                    }
                }
            });
        }
        const vpnConnection = new ec2_generated_1.CfnVPNConnection(this, 'Resource', {
            type,
            customerGatewayId: customerGateway.ref,
            staticRoutesOnly: props.staticRoutes ? true : false,
            vpnGatewayId: props.vpc.vpnGatewayId,
            vpnTunnelOptionsSpecifications: (_c = props.tunnelOptions) === null || _c === void 0 ? void 0 : _c.map(t => {
                var _c, _d;
                return ({
                    preSharedKey: (_d = (_c = t.preSharedKeySecret) === null || _c === void 0 ? void 0 : _c.unsafeUnwrap()) !== null && _d !== void 0 ? _d : t.preSharedKey,
                    tunnelInsideCidr: t.tunnelInsideCidr,
                });
            }),
        });
        this.vpnId = vpnConnection.ref;
        if (props.staticRoutes) {
            props.staticRoutes.forEach(route => {
                new ec2_generated_1.CfnVPNConnectionRoute(this, `Route${route.replace(/[^\d]/g, '')}`, {
                    destinationCidrBlock: route,
                    vpnConnectionId: this.vpnId,
                });
            });
        }
    }
    /**
     * Return the given named metric for all VPN connections in the account/region.
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/VPN',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the tunnel state of all VPN connections in the account/region.
     *
     * @default average over 5 minutes
     */
    static metricAllTunnelState(props) {
        return this.metricAll('TunnelState', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the tunnel data in of all VPN connections in the account/region.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataIn(props) {
        return this.metricAll('TunnelDataIn', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the tunnel data out of all VPN connections.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataOut(props) {
        return this.metricAll('TunnelDataOut', { statistic: 'sum', ...props });
    }
}
exports.VpnConnection = VpnConnection;
_b = JSII_RTTI_SYMBOL_1;
VpnConnection[_b] = { fqn: "@aws-cdk/aws-ec2.VpnConnection", version: "1.155.0" };
exports.RESERVED_TUNNEL_INSIDE_CIDR = [
    '169.254.0.0/30',
    '169.254.1.0/30',
    '169.254.2.0/30',
    '169.254.3.0/30',
    '169.254.4.0/30',
    '169.254.5.0/30',
    '169.254.169.252/30',
];
//# sourceMappingURL=data:application/json;base64,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