"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateSubnet = exports.PublicSubnet = exports.RouterType = exports.Subnet = exports.Vpc = exports.DefaultInstanceTenancy = exports.SubnetType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const client_vpn_endpoint_1 = require("./client-vpn-endpoint");
const ec2_generated_1 = require("./ec2.generated");
const nat_1 = require("./nat");
const network_acl_1 = require("./network-acl");
const network_util_1 = require("./network-util");
const subnet_1 = require("./subnet");
const util_1 = require("./util");
const vpc_endpoint_1 = require("./vpc-endpoint");
const vpc_flow_logs_1 = require("./vpc-flow-logs");
const vpn_1 = require("./vpn");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
const VPC_SUBNET_SYMBOL = Symbol.for('@aws-cdk/aws-ec2.VpcSubnet');
/**
 * The type of Subnet
 */
var SubnetType;
(function (SubnetType) {
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC),
     * and as such, do not require NAT gateways.
     *
     * Isolated subnets can only connect to or be connected to from other
     * instances in the same VPC. A default VPC configuration will not include
     * isolated subnets.
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     *
     * @deprecated use `SubnetType.PRIVATE_ISOLATED`
     */
    SubnetType["ISOLATED"] = "Isolated";
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC),
     * and as such, do not require NAT gateways.
     *
     * Isolated subnets can only connect to or be connected to from other
     * instances in the same VPC. A default VPC configuration will not include
     * isolated subnets.
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     */
    SubnetType["PRIVATE_ISOLATED"] = "Isolated";
    /**
     * Subnet that routes to the internet, but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. NAT Gateway(s) are
     * required with this subnet type to route the Internet traffic through.
     * If a NAT Gateway is not required or desired, use `SubnetType.PRIVATE_ISOLATED` instead.
     *
     * By default, a NAT gateway is created in every public subnet for maximum availability.
     * Be aware that you will be charged for NAT gateways.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     *
     * @deprecated use `PRIVATE_WITH_NAT`
     */
    SubnetType["PRIVATE"] = "Private";
    /**
     * Subnet that routes to the internet (via a NAT gateway), but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. NAT Gateway(s) are
     * required with this subnet type to route the Internet traffic through.
     * If a NAT Gateway is not required or desired, use `SubnetType.PRIVATE_ISOLATED` instead.
     *
     * By default, a NAT gateway is created in every public subnet for maximum availability.
     * Be aware that you will be charged for NAT gateways.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     */
    SubnetType["PRIVATE_WITH_NAT"] = "Private";
    /**
     * Subnet connected to the Internet
     *
     * Instances in a Public subnet can connect to the Internet and can be
     * connected to from the Internet as long as they are launched with public
     * IPs (controlled on the AutoScalingGroup or other constructs that launch
     * instances).
     *
     * Public subnets route outbound traffic via an Internet Gateway.
     */
    SubnetType["PUBLIC"] = "Public";
})(SubnetType = exports.SubnetType || (exports.SubnetType = {}));
/**
 * A new or imported VPC
 */
class VpcBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Dependencies for NAT connectivity
         *
         * @deprecated - This value is no longer used.
         */
        this.natDependencies = new Array();
        /**
         * If this is set to true, don't error out on trying to select subnets
         */
        this.incompleteSubnetDefinition = false;
    }
    /**
     * Returns IDs of selected subnets
     */
    selectSubnets(selection = {}) {
        const subnets = this.selectSubnetObjects(selection);
        const pubs = new Set(this.publicSubnets);
        return {
            subnetIds: subnets.map(s => s.subnetId),
            get availabilityZones() { return subnets.map(s => s.availabilityZone); },
            internetConnectivityEstablished: tap(new CompositeDependable(), d => subnets.forEach(s => d.add(s.internetConnectivityEstablished))),
            subnets,
            hasPublic: subnets.some(s => pubs.has(s)),
            isPendingLookup: this.incompleteSubnetDefinition,
        };
    }
    /**
     * Adds a VPN Gateway to this VPC
     */
    enableVpnGateway(options) {
        var _e;
        if (this.vpnGatewayId) {
            throw new Error('The VPN Gateway has already been enabled.');
        }
        const vpnGateway = new vpn_1.VpnGateway(this, 'VpnGateway', {
            amazonSideAsn: options.amazonSideAsn,
            type: vpn_1.VpnConnectionType.IPSEC_1,
        });
        this._vpnGatewayId = vpnGateway.gatewayId;
        const attachment = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCVPNGW', {
            vpcId: this.vpcId,
            vpnGatewayId: this._vpnGatewayId,
        });
        // Propagate routes on route tables associated with the right subnets
        const vpnRoutePropagation = (_e = options.vpnRoutePropagation) !== null && _e !== void 0 ? _e : [{}];
        const routeTableIds = util_1.allRouteTableIds(util_1.flatten(vpnRoutePropagation.map(s => this.selectSubnets(s).subnets)));
        if (routeTableIds.length === 0) {
            core_1.Annotations.of(this).addError(`enableVpnGateway: no subnets matching selection: '${JSON.stringify(vpnRoutePropagation)}'. Select other subnets to add routes to.`);
        }
        const routePropagation = new ec2_generated_1.CfnVPNGatewayRoutePropagation(this, 'RoutePropagation', {
            routeTableIds,
            vpnGatewayId: this._vpnGatewayId,
        });
        // The AWS::EC2::VPNGatewayRoutePropagation resource cannot use the VPN gateway
        // until it has successfully attached to the VPC.
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpn-gatewayrouteprop.html
        routePropagation.node.addDependency(attachment);
    }
    /**
     * Adds a new VPN connection to this VPC
     */
    addVpnConnection(id, options) {
        return new vpn_1.VpnConnection(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new client VPN endpoint to this VPC
     */
    addClientVpnEndpoint(id, options) {
        return new client_vpn_endpoint_1.ClientVpnEndpoint(this, id, {
            ...options,
            vpc: this,
        });
    }
    /**
     * Adds a new interface endpoint to this VPC
     */
    addInterfaceEndpoint(id, options) {
        return new vpc_endpoint_1.InterfaceVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new gateway endpoint to this VPC
     */
    addGatewayEndpoint(id, options) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new flow log to this VPC
     */
    addFlowLog(id, options) {
        return new vpc_flow_logs_1.FlowLog(this, id, {
            resourceType: vpc_flow_logs_1.FlowLogResourceType.fromVpc(this),
            ...options,
        });
    }
    /**
     * Returns the id of the VPN Gateway (if enabled)
     */
    get vpnGatewayId() {
        return this._vpnGatewayId;
    }
    /**
     * Return the subnets appropriate for the placement strategy
     */
    selectSubnetObjects(selection = {}) {
        var _e;
        selection = this.reifySelectionDefaults(selection);
        if (selection.subnets !== undefined) {
            return selection.subnets;
        }
        let subnets;
        if (selection.subnetGroupName !== undefined) { // Select by name
            subnets = this.selectSubnetObjectsByName(selection.subnetGroupName);
        }
        else { // Or specify by type
            const type = selection.subnetType || SubnetType.PRIVATE_WITH_NAT;
            subnets = this.selectSubnetObjectsByType(type);
        }
        // Apply all the filters
        subnets = this.applySubnetFilters(subnets, (_e = selection.subnetFilters) !== null && _e !== void 0 ? _e : []);
        return subnets;
    }
    applySubnetFilters(subnets, filters) {
        let filtered = subnets;
        // Apply each filter in sequence
        for (const filter of filters) {
            filtered = filter.selectSubnets(filtered);
        }
        return filtered;
    }
    selectSubnetObjectsByName(groupName) {
        const allSubnets = [...this.publicSubnets, ...this.privateSubnets, ...this.isolatedSubnets];
        const subnets = allSubnets.filter(s => util_1.subnetGroupNameFromConstructId(s) === groupName);
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const names = Array.from(new Set(allSubnets.map(util_1.subnetGroupNameFromConstructId)));
            throw new Error(`There are no subnet groups with name '${groupName}' in this VPC. Available names: ${names}`);
        }
        return subnets;
    }
    selectSubnetObjectsByType(subnetType) {
        const allSubnets = {
            [SubnetType.PRIVATE_ISOLATED]: this.isolatedSubnets,
            [SubnetType.PRIVATE_WITH_NAT]: this.privateSubnets,
            [SubnetType.PUBLIC]: this.publicSubnets,
        };
        const subnets = allSubnets[subnetType];
        // Force merge conflict here with https://github.com/aws/aws-cdk/pull/4089
        // see ImportedVpc
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const availableTypes = Object.entries(allSubnets).filter(([_, subs]) => subs.length > 0).map(([typeName, _]) => typeName);
            throw new Error(`There are no '${subnetType}' subnet groups in this VPC. Available types: ${availableTypes}`);
        }
        return subnets;
    }
    /**
     * Validate the fields in a SubnetSelection object, and reify defaults if necessary
     *
     * In case of default selection, select the first type of PRIVATE, ISOLATED,
     * PUBLIC (in that order) that has any subnets.
     */
    reifySelectionDefaults(placement) {
        var _e;
        if (placement.subnetName !== undefined) {
            if (placement.subnetGroupName !== undefined) {
                throw new Error('Please use only \'subnetGroupName\' (\'subnetName\' is deprecated and has the same behavior)');
            }
            else {
                core_1.Annotations.of(this).addWarning('Usage of \'subnetName\' in SubnetSelection is deprecated, use \'subnetGroupName\' instead');
            }
            placement = { ...placement, subnetGroupName: placement.subnetName };
        }
        const exclusiveSelections = ['subnets', 'subnetType', 'subnetGroupName'];
        const providedSelections = exclusiveSelections.filter(key => placement[key] !== undefined);
        if (providedSelections.length > 1) {
            throw new Error(`Only one of '${providedSelections}' can be supplied to subnet selection.`);
        }
        if (placement.subnetType === undefined && placement.subnetGroupName === undefined && placement.subnets === undefined) {
            // Return default subnet type based on subnets that actually exist
            let subnetType = this.privateSubnets.length
                ? SubnetType.PRIVATE_WITH_NAT : this.isolatedSubnets.length ? SubnetType.PRIVATE_ISOLATED : SubnetType.PUBLIC;
            placement = { ...placement, subnetType: subnetType };
        }
        // Establish which subnet filters are going to be used
        let subnetFilters = (_e = placement.subnetFilters) !== null && _e !== void 0 ? _e : [];
        // Backwards compatibility with existing `availabilityZones` and `onePerAz` functionality
        if (placement.availabilityZones !== undefined) { // Filter by AZs, if specified
            subnetFilters.push(subnet_1.SubnetFilter.availabilityZones(placement.availabilityZones));
        }
        if (!!placement.onePerAz) { // Ensure one per AZ if specified
            subnetFilters.push(subnet_1.SubnetFilter.onePerAz());
        }
        // Overwrite the provided placement filters and remove the availabilityZones and onePerAz properties
        placement = { ...placement, subnetFilters: subnetFilters, availabilityZones: undefined, onePerAz: undefined };
        const { availabilityZones, onePerAz, ...rest } = placement;
        return rest;
    }
}
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The default tenancy of instances launched into the VPC.
 */
var DefaultInstanceTenancy;
(function (DefaultInstanceTenancy) {
    /**
     * Instances can be launched with any tenancy.
     */
    DefaultInstanceTenancy["DEFAULT"] = "default";
    /**
     * Any instance launched into the VPC automatically has dedicated tenancy, unless you launch it with the default tenancy.
     */
    DefaultInstanceTenancy["DEDICATED"] = "dedicated";
})(DefaultInstanceTenancy = exports.DefaultInstanceTenancy || (exports.DefaultInstanceTenancy = {}));
/**
 * Define an AWS Virtual Private Cloud
 *
 * See the package-level documentation of this package for an overview
 * of the various dimensions in which you can configure your VPC.
 *
 * For example:
 *
 * ```ts
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   cidr: "10.0.0.0/16"
 * })
 *
 * // Iterate the private subnets
 * const selection = vpc.selectSubnets({
 *   subnetType: ec2.SubnetType.PRIVATE_WITH_NAT
 * });
 *
 * for (const subnet of selection.subnets) {
 *   // ...
 * }
 * ```
 *
 * @resource AWS::EC2::VPC
 */
class Vpc extends VpcBase {
    /**
     * Vpc creates a VPC that spans a whole region.
     * It will automatically divide the provided VPC CIDR range, and create public and private subnets per Availability Zone.
     * Network routing for the public subnets will be configured to allow outbound access directly via an Internet Gateway.
     * Network routing for the private subnets will be configured to allow outbound access via a set of resilient NAT Gateways (one per AZ).
     */
    constructor(scope, id, props = {}) {
        var _e;
        super(scope, id);
        /**
         * List of public subnets in this VPC
         */
        this.publicSubnets = [];
        /**
         * List of private subnets in this VPC
         */
        this.privateSubnets = [];
        /**
         * List of isolated subnets in this VPC
         */
        this.isolatedSubnets = [];
        /**
         * Subnet configurations for this VPC
         */
        this.subnetConfiguration = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpcProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const stack = core_1.Stack.of(this);
        // Can't have enabledDnsHostnames without enableDnsSupport
        if (props.enableDnsHostnames && !props.enableDnsSupport) {
            throw new Error('To use DNS Hostnames, DNS Support must be enabled, however, it was explicitly disabled.');
        }
        const cidrBlock = ifUndefined(props.cidr, Vpc.DEFAULT_CIDR_RANGE);
        if (core_1.Token.isUnresolved(cidrBlock)) {
            throw new Error('\'cidr\' property must be a concrete CIDR string, got a Token (we need to parse it for automatic subdivision)');
        }
        this.networkBuilder = new network_util_1.NetworkBuilder(cidrBlock);
        this.dnsHostnamesEnabled = props.enableDnsHostnames == null ? true : props.enableDnsHostnames;
        this.dnsSupportEnabled = props.enableDnsSupport == null ? true : props.enableDnsSupport;
        const instanceTenancy = props.defaultInstanceTenancy || 'default';
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
        // Define a VPC using the provided CIDR range
        this.resource = new ec2_generated_1.CfnVPC(this, 'Resource', {
            cidrBlock,
            enableDnsHostnames: this.dnsHostnamesEnabled,
            enableDnsSupport: this.dnsSupportEnabled,
            instanceTenancy,
        });
        this.vpcDefaultNetworkAcl = this.resource.attrDefaultNetworkAcl;
        this.vpcCidrBlockAssociations = this.resource.attrCidrBlockAssociations;
        this.vpcCidrBlock = this.resource.attrCidrBlock;
        this.vpcDefaultSecurityGroup = this.resource.attrDefaultSecurityGroup;
        this.vpcIpv6CidrBlocks = this.resource.attrIpv6CidrBlocks;
        core_1.Tags.of(this).add(NAME_TAG, props.vpcName || this.node.path);
        this.availabilityZones = stack.availabilityZones;
        const maxAZs = (_e = props.maxAzs) !== null && _e !== void 0 ? _e : 3;
        this.availabilityZones = this.availabilityZones.slice(0, maxAZs);
        this.vpcId = this.resource.ref;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
        }, stack);
        const defaultSubnet = props.natGateways === 0 ? Vpc.DEFAULT_SUBNETS_NO_NAT : Vpc.DEFAULT_SUBNETS;
        this.subnetConfiguration = ifUndefined(props.subnetConfiguration, defaultSubnet);
        const natGatewayPlacement = props.natGatewaySubnets || { subnetType: SubnetType.PUBLIC };
        const natGatewayCount = determineNatGatewayCount(props.natGateways, this.subnetConfiguration, this.availabilityZones.length);
        // subnetConfiguration must be set before calling createSubnets
        this.createSubnets();
        const allowOutbound = this.subnetConfiguration.filter(subnet => (subnet.subnetType !== SubnetType.PRIVATE_ISOLATED)).length > 0;
        // Create an Internet Gateway and attach it if necessary
        if (allowOutbound) {
            const igw = new ec2_generated_1.CfnInternetGateway(this, 'IGW', {});
            this.internetGatewayId = igw.ref;
            this._internetConnectivityEstablished.add(igw);
            const att = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCGW', {
                internetGatewayId: igw.ref,
                vpcId: this.resource.ref,
            });
            this.publicSubnets.forEach(publicSubnet => {
                publicSubnet.addDefaultInternetRoute(igw.ref, att);
            });
            // if gateways are needed create them
            if (natGatewayCount > 0) {
                const provider = props.natGatewayProvider || nat_1.NatProvider.gateway();
                this.createNatGateways(provider, natGatewayCount, natGatewayPlacement);
            }
        }
        if (props.vpnGateway && this.publicSubnets.length === 0 && this.privateSubnets.length === 0 && this.isolatedSubnets.length === 0) {
            throw new Error('Can not enable the VPN gateway while the VPC has no subnets at all');
        }
        if ((props.vpnConnections || props.vpnGatewayAsn) && props.vpnGateway === false) {
            throw new Error('Cannot specify `vpnConnections` or `vpnGatewayAsn` when `vpnGateway` is set to false.');
        }
        if (props.vpnGateway || props.vpnConnections || props.vpnGatewayAsn) {
            this.enableVpnGateway({
                amazonSideAsn: props.vpnGatewayAsn,
                type: vpn_1.VpnConnectionType.IPSEC_1,
                vpnRoutePropagation: props.vpnRoutePropagation,
            });
            const vpnConnections = props.vpnConnections || {};
            for (const [connectionId, connection] of Object.entries(vpnConnections)) {
                this.addVpnConnection(connectionId, connection);
            }
        }
        // Allow creation of gateway endpoints on VPC instantiation as those can be
        // immediately functional without further configuration. This is not the case
        // for interface endpoints where the security group must be configured.
        if (props.gatewayEndpoints) {
            const gatewayEndpoints = props.gatewayEndpoints || {};
            for (const [endpointId, endpoint] of Object.entries(gatewayEndpoints)) {
                this.addGatewayEndpoint(endpointId, endpoint);
            }
        }
        // Add flow logs to the VPC
        if (props.flowLogs) {
            const flowLogs = props.flowLogs || {};
            for (const [flowLogId, flowLog] of Object.entries(flowLogs)) {
                this.addFlowLog(flowLogId, flowLog);
            }
        }
    }
    /**
     * Import a VPC by supplying all attributes directly
     *
     * NOTE: using `fromVpcAttributes()` with deploy-time parameters (like a `Fn.importValue()` or
     * `CfnParameter` to represent a list of subnet IDs) sometimes accidentally works. It happens
     * to work for constructs that need a list of subnets (like `AutoScalingGroup` and `eks.Cluster`)
     * but it does not work for constructs that need individual subnets (like
     * `Instance`). See https://github.com/aws/aws-cdk/issues/4118 for more
     * information.
     *
     * Prefer to use `Vpc.fromLookup()` instead.
     */
    static fromVpcAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpcAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromVpcAttributes);
            }
            throw error;
        }
        return new ImportedVpc(scope, id, attrs, false);
    }
    /**
     * Import an existing VPC from by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use VPCs not defined in your CDK
     * application. If you are looking to share a VPC between stacks, you can
     * pass the `Vpc` object between stacks and use it as normal.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     *
     * The VPC information will be cached in `cdk.context.json` and the same VPC
     * will be used on future runs. To refresh the lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static fromLookup(scope, id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpcLookupOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromLookup);
            }
            throw error;
        }
        if (core_1.Token.isUnresolved(options.vpcId)
            || core_1.Token.isUnresolved(options.vpcName)
            || Object.values(options.tags || {}).some(core_1.Token.isUnresolved)
            || Object.keys(options.tags || {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to Vpc.fromLookup() must be concrete (no Tokens)');
        }
        const filter = makeTagFilter(options.tags);
        // We give special treatment to some tags
        if (options.vpcId) {
            filter['vpc-id'] = options.vpcId;
        }
        if (options.vpcName) {
            filter['tag:Name'] = options.vpcName;
        }
        if (options.isDefault !== undefined) {
            filter.isDefault = options.isDefault ? 'true' : 'false';
        }
        const overrides = {};
        if (options.region) {
            overrides.region = options.region;
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.VPC_PROVIDER,
            props: {
                ...overrides,
                filter,
                returnAsymmetricSubnets: true,
                subnetGroupNameTag: options.subnetGroupNameTag,
            },
            dummyValue: undefined,
        }).value;
        return new LookedUpVpc(scope, id, attributes || DUMMY_VPC_PROPS, attributes === undefined);
        /**
         * Prefixes all keys in the argument with `tag:`.`
         */
        function makeTagFilter(tags) {
            const result = {};
            for (const [name, value] of Object.entries(tags || {})) {
                result[`tag:${name}`] = value;
            }
            return result;
        }
    }
    /**
     * Adds a new S3 gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addS3Endpoint(id, subnets) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-ec2.Vpc#addS3Endpoint", "use `addGatewayEndpoint()` instead");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addS3Endpoint);
            }
            throw error;
        }
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.S3,
            vpc: this,
            subnets,
        });
    }
    /**
     * Adds a new DynamoDB gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addDynamoDbEndpoint(id, subnets) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-ec2.Vpc#addDynamoDbEndpoint", "use `addGatewayEndpoint()` instead");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addDynamoDbEndpoint);
            }
            throw error;
        }
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.DYNAMODB,
            vpc: this,
            subnets,
        });
    }
    createNatGateways(provider, natCount, placement) {
        const natSubnets = this.selectSubnetObjects(placement);
        for (const sub of natSubnets) {
            if (this.publicSubnets.indexOf(sub) === -1) {
                throw new Error(`natGatewayPlacement ${placement} contains non public subnet ${sub}`);
            }
        }
        provider.configureNat({
            vpc: this,
            natSubnets: natSubnets.slice(0, natCount),
            privateSubnets: this.privateSubnets,
        });
    }
    /**
     * createSubnets creates the subnets specified by the subnet configuration
     * array or creates the `DEFAULT_SUBNETS` configuration
     */
    createSubnets() {
        const remainingSpaceSubnets = [];
        for (const subnet of this.subnetConfiguration) {
            if (subnet.cidrMask === undefined) {
                remainingSpaceSubnets.push(subnet);
                continue;
            }
            this.createSubnetResources(subnet, subnet.cidrMask);
        }
        const totalRemaining = remainingSpaceSubnets.length * this.availabilityZones.length;
        const cidrMaskForRemaining = this.networkBuilder.maskForRemainingSubnets(totalRemaining);
        for (const subnet of remainingSpaceSubnets) {
            this.createSubnetResources(subnet, cidrMaskForRemaining);
        }
    }
    createSubnetResources(subnetConfig, cidrMask) {
        this.availabilityZones.forEach((zone, index) => {
            if (subnetConfig.reserved === true) {
                // For reserved subnets, just allocate ip space but do not create any resources
                this.networkBuilder.addSubnet(cidrMask);
                return;
            }
            // mapPublicIpOnLaunch true in Subnet.Public, false in Subnet.Private or Subnet.Isolated.
            let mapPublicIpOnLaunch = false;
            if (subnetConfig.subnetType !== SubnetType.PUBLIC && subnetConfig.mapPublicIpOnLaunch !== undefined) {
                throw new Error(`${subnetConfig.subnetType} subnet cannot include mapPublicIpOnLaunch parameter`);
            }
            if (subnetConfig.subnetType === SubnetType.PUBLIC) {
                mapPublicIpOnLaunch = (subnetConfig.mapPublicIpOnLaunch !== undefined)
                    ? subnetConfig.mapPublicIpOnLaunch
                    : true;
            }
            const name = util_1.subnetId(subnetConfig.name, index);
            const subnetProps = {
                availabilityZone: zone,
                vpcId: this.vpcId,
                cidrBlock: this.networkBuilder.addSubnet(cidrMask),
                mapPublicIpOnLaunch: mapPublicIpOnLaunch,
            };
            let subnet;
            switch (subnetConfig.subnetType) {
                case SubnetType.PUBLIC:
                    const publicSubnet = new PublicSubnet(this, name, subnetProps);
                    this.publicSubnets.push(publicSubnet);
                    subnet = publicSubnet;
                    break;
                case SubnetType.PRIVATE_WITH_NAT:
                    const privateSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.privateSubnets.push(privateSubnet);
                    subnet = privateSubnet;
                    break;
                case SubnetType.PRIVATE_ISOLATED:
                    const isolatedSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.isolatedSubnets.push(isolatedSubnet);
                    subnet = isolatedSubnet;
                    break;
                default:
                    throw new Error(`Unrecognized subnet type: ${subnetConfig.subnetType}`);
            }
            // These values will be used to recover the config upon provider import
            const includeResourceTypes = [ec2_generated_1.CfnSubnet.CFN_RESOURCE_TYPE_NAME];
            core_1.Tags.of(subnet).add(SUBNETNAME_TAG, subnetConfig.name, { includeResourceTypes });
            core_1.Tags.of(subnet).add(SUBNETTYPE_TAG, subnetTypeTagValue(subnetConfig.subnetType), { includeResourceTypes });
        });
    }
}
exports.Vpc = Vpc;
_a = JSII_RTTI_SYMBOL_1;
Vpc[_a] = { fqn: "@aws-cdk/aws-ec2.Vpc", version: "1.155.0" };
/**
 * The default CIDR range used when creating VPCs.
 * This can be overridden using VpcProps when creating a VPCNetwork resource.
 * e.g. new VpcResource(this, { cidr: '192.168.0.0./16' })
 */
Vpc.DEFAULT_CIDR_RANGE = '10.0.0.0/16';
/**
 * The default subnet configuration
 *
 * 1 Public and 1 Private subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE_WITH_NAT,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE_WITH_NAT),
    },
];
/**
 * The default subnet configuration if natGateways specified to be 0
 *
 * 1 Public and 1 Isolated Subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS_NO_NAT = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE_ISOLATED,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE_ISOLATED),
    },
];
const SUBNETTYPE_TAG = 'aws-cdk:subnet-type';
const SUBNETNAME_TAG = 'aws-cdk:subnet-name';
function subnetTypeTagValue(type) {
    switch (type) {
        case SubnetType.PUBLIC: return 'Public';
        case SubnetType.PRIVATE_WITH_NAT: return 'Private';
        case SubnetType.PRIVATE_ISOLATED: return 'Isolated';
    }
}
/**
 * Represents a new VPC subnet resource
 *
 * @resource AWS::EC2::Subnet
 */
class Subnet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Parts of this VPC subnet
         */
        this.dependencyElements = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_SubnetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        Object.defineProperty(this, VPC_SUBNET_SYMBOL, { value: true });
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.availabilityZone = props.availabilityZone;
        this.ipv4CidrBlock = props.cidrBlock;
        const subnet = new ec2_generated_1.CfnSubnet(this, 'Subnet', {
            vpcId: props.vpcId,
            cidrBlock: props.cidrBlock,
            availabilityZone: props.availabilityZone,
            mapPublicIpOnLaunch: props.mapPublicIpOnLaunch,
        });
        this.subnetId = subnet.ref;
        this.subnetVpcId = subnet.attrVpcId;
        this.subnetAvailabilityZone = subnet.attrAvailabilityZone;
        this.subnetIpv6CidrBlocks = subnet.attrIpv6CidrBlocks;
        this.subnetOutpostArn = subnet.attrOutpostArn;
        // subnet.attrNetworkAclAssociationId is the default ACL after the subnet
        // was just created. However, the ACL can be replaced at a later time.
        this._networkAcl = network_acl_1.NetworkAcl.fromNetworkAclId(this, 'Acl', subnet.attrNetworkAclAssociationId);
        this.subnetNetworkAclAssociationId = core_1.Lazy.string({ produce: () => this._networkAcl.networkAclId });
        this.node.defaultChild = subnet;
        const table = new ec2_generated_1.CfnRouteTable(this, 'RouteTable', {
            vpcId: props.vpcId,
        });
        this.routeTable = { routeTableId: table.ref };
        // Associate the public route table for this subnet, to this subnet
        new ec2_generated_1.CfnSubnetRouteTableAssociation(this, 'RouteTableAssociation', {
            subnetId: this.subnetId,
            routeTableId: table.ref,
        });
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
    }
    static isVpcSubnet(x) {
        return VPC_SUBNET_SYMBOL in x;
    }
    static fromSubnetAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_SubnetAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSubnetAttributes);
            }
            throw error;
        }
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * Import existing subnet from id.
     */
    // eslint-disable-next-line @typescript-eslint/no-shadow
    static fromSubnetId(scope, id, subnetId) {
        return this.fromSubnetAttributes(scope, id, { subnetId });
    }
    /**
     * Create a default route that points to a passed IGW, with a dependency
     * on the IGW's attachment to the VPC.
     *
     * @param gatewayId the logical ID (ref) of the gateway attached to your VPC
     * @param gatewayAttachment the gateway attachment construct to be added as a dependency
     */
    addDefaultInternetRoute(gatewayId, gatewayAttachment) {
        const route = new ec2_generated_1.CfnRoute(this, 'DefaultRoute', {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: '0.0.0.0/0',
            gatewayId,
        });
        route.node.addDependency(gatewayAttachment);
        // Since the 'route' depends on the gateway attachment, just
        // depending on the route is enough.
        this._internetConnectivityEstablished.add(route);
    }
    /**
     * Network ACL associated with this Subnet
     *
     * Upon creation, this is the default ACL which allows all traffic, except
     * explicit DENY entries that you add.
     *
     * You can replace it with a custom ACL which denies all traffic except
     * the explicit ALLOW entries that you add by creating a `NetworkAcl`
     * object and calling `associateNetworkAcl()`.
     */
    get networkAcl() {
        return this._networkAcl;
    }
    /**
     * Adds an entry to this subnets route table that points to the passed NATGatewayId
     * @param natGatewayId The ID of the NAT gateway
     */
    addDefaultNatRoute(natGatewayId) {
        this.addRoute('DefaultRoute', {
            routerType: RouterType.NAT_GATEWAY,
            routerId: natGatewayId,
            enablesInternetConnectivity: true,
        });
    }
    /**
     * Adds an entry to this subnets route table
     */
    addRoute(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_AddRouteOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRoute);
            }
            throw error;
        }
        if (options.destinationCidrBlock && options.destinationIpv6CidrBlock) {
            throw new Error('Cannot specify both \'destinationCidrBlock\' and \'destinationIpv6CidrBlock\'');
        }
        const route = new ec2_generated_1.CfnRoute(this, id, {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: options.destinationCidrBlock || (options.destinationIpv6CidrBlock === undefined ? '0.0.0.0/0' : undefined),
            destinationIpv6CidrBlock: options.destinationIpv6CidrBlock,
            [routerTypeToPropName(options.routerType)]: options.routerId,
        });
        if (options.enablesInternetConnectivity) {
            this._internetConnectivityEstablished.add(route);
        }
    }
    associateNetworkAcl(id, networkAcl) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_INetworkAcl(networkAcl);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.associateNetworkAcl);
            }
            throw error;
        }
        this._networkAcl = networkAcl;
        const scope = core_2.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_2.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + core_1.Names.nodeUniqueId(other.node), {
            networkAcl,
            subnet: this,
        });
    }
}
exports.Subnet = Subnet;
_b = JSII_RTTI_SYMBOL_1;
Subnet[_b] = { fqn: "@aws-cdk/aws-ec2.Subnet", version: "1.155.0" };
/**
 * Type of router used in route
 */
var RouterType;
(function (RouterType) {
    /**
     * Egress-only Internet Gateway
     */
    RouterType["EGRESS_ONLY_INTERNET_GATEWAY"] = "EgressOnlyInternetGateway";
    /**
     * Internet Gateway
     */
    RouterType["GATEWAY"] = "Gateway";
    /**
     * Instance
     */
    RouterType["INSTANCE"] = "Instance";
    /**
     * NAT Gateway
     */
    RouterType["NAT_GATEWAY"] = "NatGateway";
    /**
     * Network Interface
     */
    RouterType["NETWORK_INTERFACE"] = "NetworkInterface";
    /**
     * VPC peering connection
     */
    RouterType["VPC_PEERING_CONNECTION"] = "VpcPeeringConnection";
})(RouterType = exports.RouterType || (exports.RouterType = {}));
function routerTypeToPropName(routerType) {
    return ({
        [RouterType.EGRESS_ONLY_INTERNET_GATEWAY]: 'egressOnlyInternetGatewayId',
        [RouterType.GATEWAY]: 'gatewayId',
        [RouterType.INSTANCE]: 'instanceId',
        [RouterType.NAT_GATEWAY]: 'natGatewayId',
        [RouterType.NETWORK_INTERFACE]: 'networkInterfaceId',
        [RouterType.VPC_PEERING_CONNECTION]: 'vpcPeeringConnectionId',
    })[routerType];
}
/**
 * Represents a public VPC subnet resource
 */
class PublicSubnet extends Subnet {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_PublicSubnetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
    }
    static fromPublicSubnetAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_PublicSubnetAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromPublicSubnetAttributes);
            }
            throw error;
        }
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * Creates a new managed NAT gateway attached to this public subnet.
     * Also adds the EIP for the managed NAT.
     * @returns A ref to the the NAT Gateway ID
     */
    addNatGateway(eipAllocationId) {
        // Create a NAT Gateway in this public subnet
        const ngw = new ec2_generated_1.CfnNatGateway(this, 'NATGateway', {
            subnetId: this.subnetId,
            allocationId: eipAllocationId !== null && eipAllocationId !== void 0 ? eipAllocationId : new ec2_generated_1.CfnEIP(this, 'EIP', {
                domain: 'vpc',
            }).attrAllocationId,
        });
        return ngw;
    }
}
exports.PublicSubnet = PublicSubnet;
_c = JSII_RTTI_SYMBOL_1;
PublicSubnet[_c] = { fqn: "@aws-cdk/aws-ec2.PublicSubnet", version: "1.155.0" };
/**
 * Represents a private VPC subnet resource
 */
class PrivateSubnet extends Subnet {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_PrivateSubnetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
    }
    static fromPrivateSubnetAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_PrivateSubnetAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromPrivateSubnetAttributes);
            }
            throw error;
        }
        return new ImportedSubnet(scope, id, attrs);
    }
}
exports.PrivateSubnet = PrivateSubnet;
_d = JSII_RTTI_SYMBOL_1;
PrivateSubnet[_d] = { fqn: "@aws-cdk/aws-ec2.PrivateSubnet", version: "1.155.0" };
function ifUndefined(value, defaultValue) {
    return value !== null && value !== void 0 ? value : defaultValue;
}
class ImportedVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
        }, core_1.Stack.of(this));
        this.cidr = props.vpcCidrBlock;
        this.availabilityZones = props.availabilityZones;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        // None of the values may be unresolved list tokens
        for (const k of Object.keys(props)) {
            if (Array.isArray(props[k]) && core_1.Token.isUnresolved(props[k])) {
                core_1.Annotations.of(this).addWarning(`fromVpcAttributes: '${k}' is a list token: the imported VPC will not work with constructs that require a list of subnets at synthesis time. Use 'Vpc.fromLookup()' or 'Fn.importListValue' instead.`);
            }
        }
        /* eslint-disable max-len */
        const pub = new util_1.ImportSubnetGroup(props.publicSubnetIds, props.publicSubnetNames, props.publicSubnetRouteTableIds, SubnetType.PUBLIC, this.availabilityZones, 'publicSubnetIds', 'publicSubnetNames', 'publicSubnetRouteTableIds');
        const priv = new util_1.ImportSubnetGroup(props.privateSubnetIds, props.privateSubnetNames, props.privateSubnetRouteTableIds, SubnetType.PRIVATE_WITH_NAT, this.availabilityZones, 'privateSubnetIds', 'privateSubnetNames', 'privateSubnetRouteTableIds');
        const iso = new util_1.ImportSubnetGroup(props.isolatedSubnetIds, props.isolatedSubnetNames, props.isolatedSubnetRouteTableIds, SubnetType.PRIVATE_ISOLATED, this.availabilityZones, 'isolatedSubnetIds', 'isolatedSubnetNames', 'isolatedSubnetRouteTableIds');
        /* eslint-enable max-len */
        this.publicSubnets = pub.import(this);
        this.privateSubnets = priv.import(this);
        this.isolatedSubnets = iso.import(this);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not supplied when creating this VPC');
        }
        return this.cidr;
    }
}
class LookedUpVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
        }, core_1.Stack.of(this));
        this.cidr = props.vpcCidrBlock;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        const subnetGroups = props.subnetGroups || [];
        const availabilityZones = Array.from(new Set(flatMap(subnetGroups, subnetGroup => {
            return subnetGroup.subnets.map(subnet => subnet.availabilityZone);
        })));
        availabilityZones.sort((az1, az2) => az1.localeCompare(az2));
        this.availabilityZones = availabilityZones;
        this.publicSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PUBLIC);
        this.privateSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PRIVATE);
        this.isolatedSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.ISOLATED);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            // Value might be cached from an old CLI version, so bumping the CX API protocol to
            // force the value to exist would not have helped.
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not found when looking up this VPC. Use a newer version of the CDK CLI and clear the old context value.');
        }
        return this.cidr;
    }
    extractSubnetsOfType(subnetGroups, subnetGroupType) {
        return flatMap(subnetGroups.filter(subnetGroup => subnetGroup.type === subnetGroupType), subnetGroup => this.subnetGroupToSubnets(subnetGroup));
    }
    subnetGroupToSubnets(subnetGroup) {
        const ret = new Array();
        for (let i = 0; i < subnetGroup.subnets.length; i++) {
            const vpcSubnet = subnetGroup.subnets[i];
            ret.push(Subnet.fromSubnetAttributes(this, `${subnetGroup.name}Subnet${i + 1}`, {
                availabilityZone: vpcSubnet.availabilityZone,
                subnetId: vpcSubnet.subnetId,
                routeTableId: vpcSubnet.routeTableId,
                ipv4CidrBlock: vpcSubnet.cidr,
            }));
        }
        return ret;
    }
}
function flatMap(xs, fn) {
    const ret = new Array();
    for (const x of xs) {
        ret.push(...fn(x));
    }
    return ret;
}
class CompositeDependable {
    constructor() {
        this.dependables = new Array();
        const self = this;
        core_1.DependableTrait.implement(this, {
            get dependencyRoots() {
                const ret = new Array();
                for (const dep of self.dependables) {
                    ret.push(...core_1.DependableTrait.get(dep).dependencyRoots);
                }
                return ret;
            },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(dep) {
        this.dependables.push(dep);
    }
}
/**
 * Invoke a function on a value (for its side effect) and return the value
 */
function tap(x, fn) {
    fn(x);
    return x;
}
class ImportedSubnet extends core_1.Resource {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        if (!attrs.routeTableId) {
            // The following looks a little weird, but comes down to:
            //
            // * Is the subnetId itself unresolved ({ Ref: Subnet }); or
            // * Was it the accidentally extracted first element of a list-encoded
            //   token? ({ Fn::ImportValue: Subnets } => ['#{Token[1234]}'] =>
            //   '#{Token[1234]}'
            //
            // There's no other API to test for the second case than to the string back into
            // a list and see if the combination is Unresolved.
            //
            // In both cases we can't output the subnetId literally into the metadata (because it'll
            // be useless). In the 2nd case even, if we output it to metadata, the `resolve()` call
            // that gets done on the metadata will even `throw`, because the '#{Token}' value will
            // occur in an illegal position (not in a list context).
            const ref = core_1.Token.isUnresolved(attrs.subnetId) || core_1.Token.isUnresolved([attrs.subnetId])
                ? `at '${constructs_1.Node.of(scope).path}/${id}'`
                : `'${attrs.subnetId}'`;
            // eslint-disable-next-line max-len
            core_1.Annotations.of(this).addWarning(`No routeTableId was provided to the subnet ${ref}. Attempting to read its .routeTable.routeTableId will return null/undefined. (More info: https://github.com/aws/aws-cdk/pull/3171)`);
        }
        this._ipv4CidrBlock = attrs.ipv4CidrBlock;
        this._availabilityZone = attrs.availabilityZone;
        this.subnetId = attrs.subnetId;
        this.routeTable = {
            // Forcing routeTableId to pretend non-null to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            routeTableId: attrs.routeTableId,
        };
    }
    get availabilityZone() {
        if (!this._availabilityZone) {
            // eslint-disable-next-line max-len
            throw new Error('You cannot reference a Subnet\'s availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()');
        }
        return this._availabilityZone;
    }
    get ipv4CidrBlock() {
        if (!this._ipv4CidrBlock) {
            // tslint:disable-next-line: max-line-length
            throw new Error('You cannot reference an imported Subnet\'s IPv4 CIDR if it was not supplied. Add the ipv4CidrBlock when importing using Subnet.fromSubnetAttributes()');
        }
        return this._ipv4CidrBlock;
    }
    associateNetworkAcl(id, networkAcl) {
        const scope = core_2.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_2.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + core_1.Names.nodeUniqueId(other.node), {
            networkAcl,
            subnet: this,
        });
    }
}
/**
 * Determine (and validate) the NAT gateway count w.r.t. the rest of the subnet configuration
 *
 * We have the following requirements:
 *
 * - NatGatewayCount = 0 ==> there are no private subnets
 * - NatGatewayCount > 0 ==> there must be public subnets
 *
 * Do we want to require that there are private subnets if there are NatGateways?
 * They seem pointless but I see no reason to prevent it.
 */
function determineNatGatewayCount(requestedCount, subnetConfig, azCount) {
    const hasPrivateSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PRIVATE_WITH_NAT && !c.reserved);
    const hasPublicSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PUBLIC);
    const count = requestedCount !== undefined ? Math.min(requestedCount, azCount) : (hasPrivateSubnets ? azCount : 0);
    if (count === 0 && hasPrivateSubnets) {
        // eslint-disable-next-line max-len
        throw new Error('If you do not want NAT gateways (natGateways=0), make sure you don\'t configure any PRIVATE subnets in \'subnetConfiguration\' (make them PUBLIC or ISOLATED instead)');
    }
    if (count > 0 && !hasPublicSubnets) {
        // eslint-disable-next-line max-len
        throw new Error(`If you configure PRIVATE subnets in 'subnetConfiguration', you must also configure PUBLIC subnets to put the NAT gateways into (got ${JSON.stringify(subnetConfig)}.`);
    }
    return count;
}
/**
 * There are returned when the provider has not supplied props yet
 *
 * It's only used for testing and on the first run-through.
 */
const DUMMY_VPC_PROPS = {
    availabilityZones: [],
    vpcCidrBlock: '1.2.3.4/5',
    isolatedSubnetIds: undefined,
    isolatedSubnetNames: undefined,
    isolatedSubnetRouteTableIds: undefined,
    privateSubnetIds: undefined,
    privateSubnetNames: undefined,
    privateSubnetRouteTableIds: undefined,
    publicSubnetIds: undefined,
    publicSubnetNames: undefined,
    publicSubnetRouteTableIds: undefined,
    subnetGroups: [
        {
            name: 'Public',
            type: cxapi.VpcSubnetGroupType.PUBLIC,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 's-12345',
                    routeTableId: 'rtb-12345s',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 's-67890',
                    routeTableId: 'rtb-67890s',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Private',
            type: cxapi.VpcSubnetGroupType.PRIVATE,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Isolated',
            type: cxapi.VpcSubnetGroupType.ISOLATED,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
    ],
    vpcId: 'vpc-12345',
};
//# sourceMappingURL=data:application/json;base64,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