"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Connections = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const peer_1 = require("./peer");
/**
 * Manage the allowed network connections for constructs with Security Groups.
 *
 * Security Groups can be thought of as a firewall for network-connected
 * devices. This class makes it easy to allow network connections to and
 * from security groups, and between security groups individually. When
 * establishing connectivity between security groups, it will automatically
 * add rules in both security groups
 *
 * This object can manage one or more security groups.
 */
class Connections {
    constructor(props = {}) {
        /**
         * Underlying securityGroup for this Connections object, if present
         *
         * May be empty if this Connections object is not managing a SecurityGroup,
         * but simply representing a Connectable peer.
         */
        this._securityGroups = new ReactiveList();
        /**
         * The rule that defines how to represent this peer in a security group
         */
        this._securityGroupRules = new ReactiveList();
        /**
         * When doing bidirectional grants between Connections, make sure we don't recursive infinitely
         */
        this.skip = false;
        /**
         * When doing bidirectional grants between Security Groups in different stacks, put the rule on the other SG
         */
        this.remoteRule = false;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_ConnectionsProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.connections = this;
        this._securityGroups.push(...(props.securityGroups || []));
        this._securityGroupRules.push(...this._securityGroups.asArray());
        if (props.peer) {
            this._securityGroupRules.push(props.peer);
        }
        this.defaultPort = props.defaultPort;
    }
    get securityGroups() {
        return this._securityGroups.asArray();
    }
    /**
     * Add a security group to the list of security groups managed by this object
     */
    addSecurityGroup(...securityGroups) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_ISecurityGroup(securityGroups);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSecurityGroup);
            }
            throw error;
        }
        for (const securityGroup of securityGroups) {
            this._securityGroups.push(securityGroup);
            this._securityGroupRules.push(securityGroup);
        }
    }
    /**
     * Allow connections to the peer on the given port
     */
    allowTo(other, portRange, description) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_IConnectable(other);
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(portRange);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.allowTo);
            }
            throw error;
        }
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addEgressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowFrom(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * Allow connections from the peer on the given port
     */
    allowFrom(other, portRange, description) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_IConnectable(other);
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(portRange);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.allowFrom);
            }
            throw error;
        }
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowTo(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * Allow hosts inside the security group to connect to each other on the given port
     */
    allowInternally(portRange, description) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(portRange);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.allowInternally);
            }
            throw error;
        }
        this._securityGroups.forEachAndForever(securityGroup => {
            this._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description);
                securityGroup.addEgressRule(rule, portRange, description);
            });
        });
    }
    /**
     * Allow to all IPv4 ranges
     */
    allowToAnyIpv4(portRange, description) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(portRange);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.allowToAnyIpv4);
            }
            throw error;
        }
        this.allowTo(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * Allow from any IPv4 ranges
     */
    allowFromAnyIpv4(portRange, description) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(portRange);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.allowFromAnyIpv4);
            }
            throw error;
        }
        this.allowFrom(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * Allow connections from the peer on our default port
     *
     * Even if the peer has a default port, we will always use our default port.
     */
    allowDefaultPortFrom(other, description) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_IConnectable(other);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.allowDefaultPortFrom);
            }
            throw error;
        }
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFrom(): this resource has no default port');
        }
        this.allowFrom(other, this.defaultPort, description);
    }
    /**
     * Allow hosts inside the security group to connect to each other
     */
    allowDefaultPortInternally(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortInternally(): this resource has no default port');
        }
        this.allowInternally(this.defaultPort, description);
    }
    /**
     * Allow default connections from all IPv4 ranges
     */
    allowDefaultPortFromAnyIpv4(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFromAnyIpv4(): this resource has no default port');
        }
        this.allowFromAnyIpv4(this.defaultPort, description);
    }
    /**
     * Allow connections to the security group on their default port
     */
    allowToDefaultPort(other, description) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_IConnectable(other);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.allowToDefaultPort);
            }
            throw error;
        }
        if (other.connections.defaultPort === undefined) {
            throw new Error('Cannot call allowToDefaultPort(): other resource has no default port');
        }
        this.allowTo(other, other.connections.defaultPort, description);
    }
    /**
     * Allow connections from the peer on our default port
     *
     * Even if the peer has a default port, we will always use our default port.
     */
    allowDefaultPortTo(other, description) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_IConnectable(other);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.allowDefaultPortTo);
            }
            throw error;
        }
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortTo(): this resource has no default port');
        }
        this.allowTo(other, this.defaultPort, description);
    }
}
exports.Connections = Connections;
_a = JSII_RTTI_SYMBOL_1;
Connections[_a] = { fqn: "@aws-cdk/aws-ec2.Connections", version: "1.155.0" };
class ReactiveList {
    constructor() {
        this.elements = new Array();
        this.listeners = new Array();
    }
    push(...xs) {
        this.elements.push(...xs);
        for (const listener of this.listeners) {
            for (const x of xs) {
                listener(x);
            }
        }
    }
    forEachAndForever(listener) {
        for (const element of this.elements) {
            listener(element);
        }
        this.listeners.push(listener);
    }
    asArray() {
        return this.elements.slice();
    }
    get length() {
        return this.elements.length;
    }
}
//# sourceMappingURL=data:application/json;base64,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