"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitConfig = exports.CloudFormationInit = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const machine_image_1 = require("./machine-image");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * A CloudFormation-init configuration
 */
class CloudFormationInit {
    constructor(configSets, configs) {
        this._configSets = {};
        this._configs = {};
        Object.assign(this._configSets, configSets);
        Object.assign(this._configs, configs);
    }
    /**
     * Build a new config from a set of Init Elements
     */
    static fromElements(...elements) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_InitElement(elements);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromElements);
            }
            throw error;
        }
        return CloudFormationInit.fromConfig(new InitConfig(elements));
    }
    /**
     * Use an existing InitConfig object as the default and only config
     */
    static fromConfig(config) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_InitConfig(config);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromConfig);
            }
            throw error;
        }
        return CloudFormationInit.fromConfigSets({
            configSets: {
                default: ['config'],
            },
            configs: { config },
        });
    }
    /**
     * Build a CloudFormationInit from config sets
     */
    static fromConfigSets(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_ConfigSetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromConfigSets);
            }
            throw error;
        }
        return new CloudFormationInit(props.configSets, props.configs);
    }
    /**
     * Add a config with the given name to this CloudFormationInit object
     */
    addConfig(configName, config) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_InitConfig(config);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addConfig);
            }
            throw error;
        }
        if (this._configs[configName]) {
            throw new Error(`CloudFormationInit already contains a config named '${configName}'`);
        }
        this._configs[configName] = config;
    }
    /**
     * Add a config set with the given name to this CloudFormationInit object
     *
     * The new configset will reference the given configs in the given order.
     */
    addConfigSet(configSetName, configNames = []) {
        if (this._configSets[configSetName]) {
            throw new Error(`CloudFormationInit already contains a configSet named '${configSetName}'`);
        }
        const unk = configNames.filter(c => !this._configs[c]);
        if (unk.length > 0) {
            throw new Error(`Unknown configs referenced in definition of '${configSetName}': ${unk}`);
        }
        this._configSets[configSetName] = [...configNames];
    }
    /**
     * Attach the CloudFormation Init config to the given resource
     *
     * As an app builder, use `instance.applyCloudFormationInit()` or
     * `autoScalingGroup.applyCloudFormationInit()` to trigger this method.
     *
     * This method does the following:
     *
     * - Renders the `AWS::CloudFormation::Init` object to the given resource's
     *   metadata, potentially adding a `AWS::CloudFormation::Authentication` object
     *   next to it if required.
     * - Updates the instance role policy to be able to call the APIs required for
     *   `cfn-init` and `cfn-signal` to work, and potentially add permissions to download
     *   referenced asset and bucket resources.
     * - Updates the given UserData with commands to execute the `cfn-init` script.
     */
    attach(attachedResource, attachOptions) {
        var _c, _d, _e, _f, _g;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_AttachInitOptions(attachOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.attach);
            }
            throw error;
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.UNKNOWN) {
            throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
        }
        const CFN_INIT_METADATA_KEY = 'AWS::CloudFormation::Init';
        if (attachedResource.getMetadata(CFN_INIT_METADATA_KEY) !== undefined) {
            throw new Error(`Cannot bind CfnInit: resource '${attachedResource.node.path}' already has '${CFN_INIT_METADATA_KEY}' attached`);
        }
        // Note: This will not reflect mutations made after attaching.
        const bindResult = this.bind(attachedResource.stack, attachOptions);
        attachedResource.addMetadata(CFN_INIT_METADATA_KEY, bindResult.configData);
        // Need to resolve the various tokens from assets in the config,
        // as well as include any asset hashes provided so the fingerprint is accurate.
        const resolvedConfig = attachedResource.stack.resolve(bindResult.configData);
        const fingerprintInput = { config: resolvedConfig, assetHash: bindResult.assetHash };
        const fingerprint = contentHash(JSON.stringify(fingerprintInput)).slice(0, 16);
        attachOptions.instanceRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
            resources: [core_1.Aws.STACK_ID],
        }));
        if (bindResult.authData) {
            attachedResource.addMetadata('AWS::CloudFormation::Authentication', bindResult.authData);
        }
        // To identify the resources that have the metadata and where the signal
        // needs to be sent, we need { region, stackName, logicalId }
        let resourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${attachedResource.logicalId}`;
        const signalResource = (_d = (_c = attachOptions.signalResource) === null || _c === void 0 ? void 0 : _c.logicalId) !== null && _d !== void 0 ? _d : attachedResource.logicalId;
        let notifyResourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${signalResource}`;
        // If specified in attachOptions, include arguments in cfn-init/cfn-signal commands
        if (attachOptions.includeUrl) {
            resourceLocator = `${resourceLocator} --url https://cloudformation.${core_1.Aws.REGION}.${core_1.Aws.URL_SUFFIX}`;
            notifyResourceLocator = `${notifyResourceLocator} --url https://cloudformation.${core_1.Aws.REGION}.${core_1.Aws.URL_SUFFIX}`;
        }
        if (attachOptions.includeRole) {
            resourceLocator = `${resourceLocator} --role ${attachOptions.instanceRole.roleName}`;
            notifyResourceLocator = `${notifyResourceLocator} --role ${attachOptions.instanceRole.roleName}`;
        }
        const configSets = ((_e = attachOptions.configSets) !== null && _e !== void 0 ? _e : ['default']).join(',');
        const printLog = (_f = attachOptions.printLog) !== null && _f !== void 0 ? _f : true;
        if ((_g = attachOptions.embedFingerprint) !== null && _g !== void 0 ? _g : true) {
            // It just so happens that the comment char is '#' for both bash and PowerShell
            attachOptions.userData.addCommands(`# fingerprint: ${fingerprint}`);
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.WINDOWS) {
            const errCode = attachOptions.ignoreFailures ? '0' : '$LASTEXITCODE';
            attachOptions.userData.addCommands(...[
                `cfn-init.exe -v ${resourceLocator} -c ${configSets}`,
                `cfn-signal.exe -e ${errCode} ${notifyResourceLocator}`,
                ...(printLog ? ['type C:\\cfn\\log\\cfn-init.log'] : []),
            ]);
        }
        else {
            const errCode = attachOptions.ignoreFailures ? '0' : '$?';
            attachOptions.userData.addCommands(...[
                // Run a subshell without 'errexit', so we can signal using the exit code of cfn-init
                '(',
                '  set +e',
                `  /opt/aws/bin/cfn-init -v ${resourceLocator} -c ${configSets}`,
                `  /opt/aws/bin/cfn-signal -e ${errCode} ${notifyResourceLocator}`,
                ...(printLog ? ['  cat /var/log/cfn-init.log >&2'] : []),
                ')',
            ]);
        }
    }
    bind(scope, options) {
        const nonEmptyConfigs = mapValues(this._configs, c => c.isEmpty() ? undefined : c);
        const configNameToBindResult = mapValues(nonEmptyConfigs, c => c._bind(scope, options));
        return {
            configData: {
                configSets: mapValues(this._configSets, configNames => configNames.filter(name => nonEmptyConfigs[name] !== undefined)),
                ...mapValues(configNameToBindResult, c => c.config),
            },
            authData: Object.values(configNameToBindResult).map(c => c.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(Object.values(configNameToBindResult).map(c => c.assetHash)),
        };
    }
}
exports.CloudFormationInit = CloudFormationInit;
_a = JSII_RTTI_SYMBOL_1;
CloudFormationInit[_a] = { fqn: "@aws-cdk/aws-ec2.CloudFormationInit", version: "1.155.0" };
/**
 * A collection of configuration elements
 */
class InitConfig {
    constructor(elements) {
        this.elements = new Array();
        this.add(...elements);
    }
    /**
     * Whether this configset has elements or not
     */
    isEmpty() {
        return this.elements.length === 0;
    }
    /**
     * Add one or more elements to the config
     */
    add(...elements) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_InitElement(elements);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.add);
            }
            throw error;
        }
        this.elements.push(...elements);
    }
    /**
     * Called when the config is applied to an instance.
     * Creates the CloudFormation representation of the Init config and handles any permissions and assets.
     * @internal
     */
    _bind(scope, options) {
        const bindOptions = {
            instanceRole: options.instanceRole,
            platform: this.initPlatformFromOSType(options.platform),
            scope,
        };
        const packageConfig = this.bindForType(cfn_init_internal_1.InitElementType.PACKAGE, bindOptions);
        const groupsConfig = this.bindForType(cfn_init_internal_1.InitElementType.GROUP, bindOptions);
        const usersConfig = this.bindForType(cfn_init_internal_1.InitElementType.USER, bindOptions);
        const sourcesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SOURCE, bindOptions);
        const filesConfig = this.bindForType(cfn_init_internal_1.InitElementType.FILE, bindOptions);
        const commandsConfig = this.bindForType(cfn_init_internal_1.InitElementType.COMMAND, bindOptions);
        // Must be last!
        const servicesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SERVICE, bindOptions);
        const allConfig = [packageConfig, groupsConfig, usersConfig, sourcesConfig, filesConfig, commandsConfig, servicesConfig];
        const authentication = allConfig.map(c => c === null || c === void 0 ? void 0 : c.authentication).reduce(deepMerge, undefined);
        const assetHash = combineAssetHashesOrUndefined(allConfig.map(c => c === null || c === void 0 ? void 0 : c.assetHash));
        return {
            config: {
                packages: packageConfig === null || packageConfig === void 0 ? void 0 : packageConfig.config,
                groups: groupsConfig === null || groupsConfig === void 0 ? void 0 : groupsConfig.config,
                users: usersConfig === null || usersConfig === void 0 ? void 0 : usersConfig.config,
                sources: sourcesConfig === null || sourcesConfig === void 0 ? void 0 : sourcesConfig.config,
                files: filesConfig === null || filesConfig === void 0 ? void 0 : filesConfig.config,
                commands: commandsConfig === null || commandsConfig === void 0 ? void 0 : commandsConfig.config,
                services: servicesConfig === null || servicesConfig === void 0 ? void 0 : servicesConfig.config,
            },
            authentication,
            assetHash,
        };
    }
    bindForType(elementType, renderOptions) {
        var _c;
        const elements = this.elements.filter(elem => elem.elementType === elementType);
        if (elements.length === 0) {
            return undefined;
        }
        const bindResults = elements.map((e, index) => e._bind({ index, ...renderOptions }));
        return {
            config: (_c = bindResults.map(r => r.config).reduce(deepMerge, undefined)) !== null && _c !== void 0 ? _c : {},
            authentication: bindResults.map(r => r.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(bindResults.map(r => r.assetHash)),
        };
    }
    initPlatformFromOSType(osType) {
        switch (osType) {
            case machine_image_1.OperatingSystemType.LINUX: {
                return cfn_init_internal_1.InitPlatform.LINUX;
            }
            case machine_image_1.OperatingSystemType.WINDOWS: {
                return cfn_init_internal_1.InitPlatform.WINDOWS;
            }
            default: {
                throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
            }
        }
    }
}
exports.InitConfig = InitConfig;
_b = JSII_RTTI_SYMBOL_1;
InitConfig[_b] = { fqn: "@aws-cdk/aws-ec2.InitConfig", version: "1.155.0" };
/**
 * Deep-merge objects and arrays
 *
 * Treat arrays as sets, removing duplicates. This is acceptable for rendering
 * cfn-inits, not applicable elsewhere.
 */
function deepMerge(target, src) {
    var _c, _d;
    if (target == null) {
        return src;
    }
    if (src == null) {
        return target;
    }
    for (const [key, value] of Object.entries(src)) {
        if (Array.isArray(value)) {
            if (target[key] && !Array.isArray(target[key])) {
                throw new Error(`Trying to merge array [${value}] into a non-array '${target[key]}'`);
            }
            target[key] = Array.from(new Set([
                ...(_c = target[key]) !== null && _c !== void 0 ? _c : [],
                ...value,
            ]));
            continue;
        }
        if (typeof value === 'object' && value) {
            target[key] = deepMerge((_d = target[key]) !== null && _d !== void 0 ? _d : {}, value);
            continue;
        }
        if (value !== undefined) {
            target[key] = value;
        }
    }
    return target;
}
/**
 * Map a function over values of an object
 *
 * If the mapping function returns undefined, remove the key
 */
function mapValues(xs, fn) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        const mapped = fn(v);
        if (mapped !== undefined) {
            ret[k] = mapped;
        }
    }
    return ret;
}
// Combines all input asset hashes into one, or if no hashes are present, returns undefined.
function combineAssetHashesOrUndefined(hashes) {
    const hashArray = hashes.filter((x) => x !== undefined);
    return hashArray.length > 0 ? hashArray.join('') : undefined;
}
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2ZuLWluaXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZm4taW5pdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxpQ0FBaUM7QUFDakMsd0NBQXdDO0FBQ3hDLHdDQUFpRDtBQUVqRCxtREFBc0Q7QUFDdEQsbUVBQWdIO0FBT2hIOztHQUVHO0FBQ0gsTUFBYSxrQkFBa0I7SUE4QjdCLFlBQW9CLFVBQW9DLEVBQUUsT0FBbUM7UUFINUUsZ0JBQVcsR0FBNkIsRUFBRSxDQUFDO1FBQzNDLGFBQVEsR0FBK0IsRUFBRSxDQUFDO1FBR3pELE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUM1QyxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLENBQUM7S0FDdkM7SUFoQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUFDLEdBQUcsUUFBdUI7Ozs7Ozs7Ozs7UUFDbkQsT0FBTyxrQkFBa0IsQ0FBQyxVQUFVLENBQUMsSUFBSSxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztLQUNoRTtJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxNQUFrQjs7Ozs7Ozs7OztRQUN6QyxPQUFPLGtCQUFrQixDQUFDLGNBQWMsQ0FBQztZQUN2QyxVQUFVLEVBQUU7Z0JBQ1YsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDO2FBQ3BCO1lBQ0QsT0FBTyxFQUFFLEVBQUUsTUFBTSxFQUFFO1NBQ3BCLENBQUMsQ0FBQztLQUNKO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsY0FBYyxDQUFDLEtBQXFCOzs7Ozs7Ozs7O1FBQ2hELE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztLQUNoRTtJQVVEOztPQUVHO0lBQ0ksU0FBUyxDQUFDLFVBQWtCLEVBQUUsTUFBa0I7Ozs7Ozs7Ozs7UUFDckQsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQzdCLE1BQU0sSUFBSSxLQUFLLENBQUMsdURBQXVELFVBQVUsR0FBRyxDQUFDLENBQUM7U0FDdkY7UUFDRCxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxHQUFHLE1BQU0sQ0FBQztLQUNwQztJQUVEOzs7O09BSUc7SUFDSSxZQUFZLENBQUMsYUFBcUIsRUFBRSxjQUF3QixFQUFFO1FBQ25FLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLDBEQUEwRCxhQUFhLEdBQUcsQ0FBQyxDQUFDO1NBQzdGO1FBRUQsTUFBTSxHQUFHLEdBQUcsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3ZELElBQUksR0FBRyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDbEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxnREFBZ0QsYUFBYSxNQUFNLEdBQUcsRUFBRSxDQUFDLENBQUM7U0FDM0Y7UUFFRCxJQUFJLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsR0FBRyxXQUFXLENBQUMsQ0FBQztLQUNwRDtJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLE1BQU0sQ0FBQyxnQkFBNkIsRUFBRSxhQUFnQzs7Ozs7Ozs7Ozs7UUFDM0UsSUFBSSxhQUFhLENBQUMsUUFBUSxLQUFLLG1DQUFtQixDQUFDLE9BQU8sRUFBRTtZQUMxRCxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7U0FDM0U7UUFFRCxNQUFNLHFCQUFxQixHQUFHLDJCQUEyQixDQUFDO1FBRTFELElBQUksZ0JBQWdCLENBQUMsV0FBVyxDQUFDLHFCQUFxQixDQUFDLEtBQUssU0FBUyxFQUFFO1lBQ3JFLE1BQU0sSUFBSSxLQUFLLENBQUMsa0NBQWtDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxJQUFJLGtCQUFrQixxQkFBcUIsWUFBWSxDQUFDLENBQUM7U0FDbEk7UUFFRCw4REFBOEQ7UUFDOUQsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDcEUsZ0JBQWdCLENBQUMsV0FBVyxDQUFDLHFCQUFxQixFQUFFLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUUzRSxnRUFBZ0U7UUFDaEUsK0VBQStFO1FBQy9FLE1BQU0sY0FBYyxHQUFHLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzdFLE1BQU0sZ0JBQWdCLEdBQUcsRUFBRSxNQUFNLEVBQUUsY0FBYyxFQUFFLFNBQVMsRUFBRSxVQUFVLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDckYsTUFBTSxXQUFXLEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFL0UsYUFBYSxDQUFDLFlBQVksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEUsT0FBTyxFQUFFLENBQUMsc0NBQXNDLEVBQUUsK0JBQStCLENBQUM7WUFDbEYsU0FBUyxFQUFFLENBQUMsVUFBRyxDQUFDLFFBQVEsQ0FBQztTQUMxQixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksVUFBVSxDQUFDLFFBQVEsRUFBRTtZQUN2QixnQkFBZ0IsQ0FBQyxXQUFXLENBQUMscUNBQXFDLEVBQUUsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQzFGO1FBRUQsd0VBQXdFO1FBQ3hFLDZEQUE2RDtRQUM3RCxJQUFJLGVBQWUsR0FBRyxZQUFZLFVBQUcsQ0FBQyxNQUFNLFlBQVksVUFBRyxDQUFDLFVBQVUsZUFBZSxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUNsSCxNQUFNLGNBQWMsZUFBRyxhQUFhLENBQUMsY0FBYywwQ0FBRSxTQUFTLG1DQUFJLGdCQUFnQixDQUFDLFNBQVMsQ0FBQztRQUM3RixJQUFJLHFCQUFxQixHQUFHLFlBQVksVUFBRyxDQUFDLE1BQU0sWUFBWSxVQUFHLENBQUMsVUFBVSxlQUFlLGNBQWMsRUFBRSxDQUFDO1FBRTVHLG1GQUFtRjtRQUNuRixJQUFJLGFBQWEsQ0FBQyxVQUFVLEVBQUU7WUFDNUIsZUFBZSxHQUFHLEdBQUcsZUFBZSxpQ0FBaUMsVUFBRyxDQUFDLE1BQU0sSUFBSSxVQUFHLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDcEcscUJBQXFCLEdBQUcsR0FBRyxxQkFBcUIsaUNBQWlDLFVBQUcsQ0FBQyxNQUFNLElBQUksVUFBRyxDQUFDLFVBQVUsRUFBRSxDQUFDO1NBQ2pIO1FBQ0QsSUFBSSxhQUFhLENBQUMsV0FBVyxFQUFFO1lBQzdCLGVBQWUsR0FBRyxHQUFHLGVBQWUsV0FBVyxhQUFhLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ3JGLHFCQUFxQixHQUFHLEdBQUcscUJBQXFCLFdBQVcsYUFBYSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztTQUNsRztRQUVELE1BQU0sVUFBVSxHQUFHLE9BQUMsYUFBYSxDQUFDLFVBQVUsbUNBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUN2RSxNQUFNLFFBQVEsU0FBRyxhQUFhLENBQUMsUUFBUSxtQ0FBSSxJQUFJLENBQUM7UUFFaEQsVUFBSSxhQUFhLENBQUMsZ0JBQWdCLG1DQUFJLElBQUksRUFBRTtZQUMxQywrRUFBK0U7WUFDL0UsYUFBYSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLFdBQVcsRUFBRSxDQUFDLENBQUM7U0FDckU7UUFFRCxJQUFJLGFBQWEsQ0FBQyxRQUFRLEtBQUssbUNBQW1CLENBQUMsT0FBTyxFQUFFO1lBQzFELE1BQU0sT0FBTyxHQUFHLGFBQWEsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDO1lBQ3JFLGFBQWEsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUNoQyxHQUFHO2dCQUNELG1CQUFtQixlQUFlLE9BQU8sVUFBVSxFQUFFO2dCQUNyRCxxQkFBcUIsT0FBTyxJQUFJLHFCQUFxQixFQUFFO2dCQUN2RCxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLGlDQUFpQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzthQUN6RCxDQUNGLENBQUM7U0FDSDthQUFNO1lBQ0wsTUFBTSxPQUFPLEdBQUcsYUFBYSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7WUFDMUQsYUFBYSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQ2hDLEdBQUc7Z0JBQ0QscUZBQXFGO2dCQUNyRixHQUFHO2dCQUNILFVBQVU7Z0JBQ1YsOEJBQThCLGVBQWUsT0FBTyxVQUFVLEVBQUU7Z0JBQ2hFLGdDQUFnQyxPQUFPLElBQUkscUJBQXFCLEVBQUU7Z0JBQ2xFLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsaUNBQWlDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2dCQUN4RCxHQUFHO2FBQ0osQ0FDRixDQUFDO1NBQ0g7S0FDRjtJQUVPLElBQUksQ0FBQyxLQUFnQixFQUFFLE9BQTBCO1FBQ3ZELE1BQU0sZUFBZSxHQUFHLFNBQVMsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRW5GLE1BQU0sc0JBQXNCLEdBQUcsU0FBUyxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFFeEYsT0FBTztZQUNMLFVBQVUsRUFBRTtnQkFDVixVQUFVLEVBQUUsU0FBUyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsV0FBVyxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxLQUFLLFNBQVMsQ0FBQyxDQUFDO2dCQUN2SCxHQUFHLFNBQVMsQ0FBQyxzQkFBc0IsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUM7YUFDcEQ7WUFDRCxRQUFRLEVBQUUsTUFBTSxDQUFDLE1BQU0sQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQztZQUN2RyxTQUFTLEVBQUUsNkJBQTZCLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUN0RyxDQUFDO0tBQ0g7O0FBM0tILGdEQTZLQzs7O0FBRUQ7O0dBRUc7QUFDSCxNQUFhLFVBQVU7SUFHckIsWUFBWSxRQUF1QjtRQUZsQixhQUFRLEdBQUcsSUFBSSxLQUFLLEVBQWUsQ0FBQztRQUduRCxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7S0FDdkI7SUFFRDs7T0FFRztJQUNJLE9BQU87UUFDWixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxLQUFLLENBQUMsQ0FBQztLQUNuQztJQUVEOztPQUVHO0lBQ0ksR0FBRyxDQUFDLEdBQUcsUUFBdUI7Ozs7Ozs7Ozs7UUFDbkMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsR0FBRyxRQUFRLENBQUMsQ0FBQztLQUNqQztJQUVEOzs7O09BSUc7SUFDSSxLQUFLLENBQUMsS0FBZ0IsRUFBRSxPQUEwQjtRQUN2RCxNQUFNLFdBQVcsR0FBRztZQUNsQixZQUFZLEVBQUUsT0FBTyxDQUFDLFlBQVk7WUFDbEMsUUFBUSxFQUFFLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDO1lBQ3ZELEtBQUs7U0FDTixDQUFDO1FBRUYsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxtQ0FBZSxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUM3RSxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLG1DQUFlLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQzFFLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsbUNBQWUsQ0FBQyxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDeEUsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxtQ0FBZSxDQUFDLE1BQU0sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUM1RSxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLG1DQUFlLENBQUMsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ3hFLE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsbUNBQWUsQ0FBQyxPQUFPLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDOUUsZ0JBQWdCO1FBQ2hCLE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsbUNBQWUsQ0FBQyxPQUFPLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFFOUUsTUFBTSxTQUFTLEdBQUcsQ0FBQyxhQUFhLEVBQUUsWUFBWSxFQUFFLFdBQVcsRUFBRSxhQUFhLEVBQUUsV0FBVyxFQUFFLGNBQWMsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUN6SCxNQUFNLGNBQWMsR0FBRyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxhQUFELENBQUMsdUJBQUQsQ0FBQyxDQUFFLGNBQWMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDMUYsTUFBTSxTQUFTLEdBQUcsNkJBQTZCLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsYUFBRCxDQUFDLHVCQUFELENBQUMsQ0FBRSxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBRWxGLE9BQU87WUFDTCxNQUFNLEVBQUU7Z0JBQ04sUUFBUSxFQUFFLGFBQWEsYUFBYixhQUFhLHVCQUFiLGFBQWEsQ0FBRSxNQUFNO2dCQUMvQixNQUFNLEVBQUUsWUFBWSxhQUFaLFlBQVksdUJBQVosWUFBWSxDQUFFLE1BQU07Z0JBQzVCLEtBQUssRUFBRSxXQUFXLGFBQVgsV0FBVyx1QkFBWCxXQUFXLENBQUUsTUFBTTtnQkFDMUIsT0FBTyxFQUFFLGFBQWEsYUFBYixhQUFhLHVCQUFiLGFBQWEsQ0FBRSxNQUFNO2dCQUM5QixLQUFLLEVBQUUsV0FBVyxhQUFYLFdBQVcsdUJBQVgsV0FBVyxDQUFFLE1BQU07Z0JBQzFCLFFBQVEsRUFBRSxjQUFjLGFBQWQsY0FBYyx1QkFBZCxjQUFjLENBQUUsTUFBTTtnQkFDaEMsUUFBUSxFQUFFLGNBQWMsYUFBZCxjQUFjLHVCQUFkLGNBQWMsQ0FBRSxNQUFNO2FBQ2pDO1lBQ0QsY0FBYztZQUNkLFNBQVM7U0FDVixDQUFDO0tBQ0g7SUFFTyxXQUFXLENBQUMsV0FBNEIsRUFBRSxhQUE2Qzs7UUFDN0YsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxDQUFDO1FBQ2hGLElBQUksUUFBUSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFBRSxPQUFPLFNBQVMsQ0FBQztTQUFFO1FBRWhELE1BQU0sV0FBVyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRXJGLE9BQU87WUFDTCxNQUFNLFFBQUUsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxtQ0FBSSxFQUFFO1lBQ3pFLGNBQWMsRUFBRSxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDO1lBQ25GLFNBQVMsRUFBRSw2QkFBNkIsQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQzVFLENBQUM7S0FDSDtJQUVPLHNCQUFzQixDQUFDLE1BQTJCO1FBQ3hELFFBQVEsTUFBTSxFQUFFO1lBQ2QsS0FBSyxtQ0FBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDOUIsT0FBTyxnQ0FBWSxDQUFDLEtBQUssQ0FBQzthQUMzQjtZQUNELEtBQUssbUNBQW1CLENBQUMsT0FBTyxDQUFDLENBQUM7Z0JBQ2hDLE9BQU8sZ0NBQVksQ0FBQyxPQUFPLENBQUM7YUFDN0I7WUFDRCxPQUFPLENBQUMsQ0FBQztnQkFDUCxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7YUFDM0U7U0FDRjtLQUNGOztBQXRGSCxnQ0F1RkM7OztBQWlCRDs7Ozs7R0FLRztBQUNILFNBQVMsU0FBUyxDQUFDLE1BQTRCLEVBQUUsR0FBeUI7O0lBQ3hFLElBQUksTUFBTSxJQUFJLElBQUksRUFBRTtRQUFFLE9BQU8sR0FBRyxDQUFDO0tBQUU7SUFDbkMsSUFBSSxHQUFHLElBQUksSUFBSSxFQUFFO1FBQUUsT0FBTyxNQUFNLENBQUM7S0FBRTtJQUVuQyxLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUM5QyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDeEIsSUFBSSxNQUFNLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFO2dCQUM5QyxNQUFNLElBQUksS0FBSyxDQUFDLDBCQUEwQixLQUFLLHVCQUF1QixNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDO2FBQ3ZGO1lBQ0QsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUM7Z0JBQy9CLFNBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxtQ0FBSSxFQUFFO2dCQUNwQixHQUFHLEtBQUs7YUFDVCxDQUFDLENBQUMsQ0FBQztZQUNKLFNBQVM7U0FDVjtRQUNELElBQUksT0FBTyxLQUFLLEtBQUssUUFBUSxJQUFJLEtBQUssRUFBRTtZQUN0QyxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsU0FBUyxPQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsbUNBQUksRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ2xELFNBQVM7U0FDVjtRQUNELElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUN2QixNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO1NBQ3JCO0tBQ0Y7SUFFRCxPQUFPLE1BQU0sQ0FBQztBQUNoQixDQUFDO0FBRUQ7Ozs7R0FJRztBQUNILFNBQVMsU0FBUyxDQUFPLEVBQXFCLEVBQUUsRUFBMkI7SUFDekUsTUFBTSxHQUFHLEdBQXNCLEVBQUUsQ0FBQztJQUNsQyxLQUFLLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsRUFBRTtRQUN2QyxNQUFNLE1BQU0sR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDckIsSUFBSSxNQUFNLEtBQUssU0FBUyxFQUFFO1lBQ3hCLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxNQUFNLENBQUM7U0FDakI7S0FDRjtJQUNELE9BQU8sR0FBRyxDQUFDO0FBQ2IsQ0FBQztBQUVELDRGQUE0RjtBQUM1RixTQUFTLDZCQUE2QixDQUFDLE1BQThCO0lBQ25FLE1BQU0sU0FBUyxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQWUsRUFBRSxDQUFDLENBQUMsS0FBSyxTQUFTLENBQUMsQ0FBQztJQUNyRSxPQUFPLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7QUFDL0QsQ0FBQztBQUVELFNBQVMsV0FBVyxDQUFDLE9BQWU7SUFDbEMsT0FBTyxNQUFNLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDbkUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNyeXB0byBmcm9tICdjcnlwdG8nO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgQXdzLCBDZm5SZXNvdXJjZSB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgSW5pdEVsZW1lbnQgfSBmcm9tICcuL2Nmbi1pbml0LWVsZW1lbnRzJztcbmltcG9ydCB7IE9wZXJhdGluZ1N5c3RlbVR5cGUgfSBmcm9tICcuL21hY2hpbmUtaW1hZ2UnO1xuaW1wb3J0IHsgSW5pdEJpbmRPcHRpb25zLCBJbml0RWxlbWVudENvbmZpZywgSW5pdEVsZW1lbnRUeXBlLCBJbml0UGxhdGZvcm0gfSBmcm9tICcuL3ByaXZhdGUvY2ZuLWluaXQtaW50ZXJuYWwnO1xuaW1wb3J0IHsgVXNlckRhdGEgfSBmcm9tICcuL3VzZXItZGF0YSc7XG5cbi8vIGtlZXAgdGhpcyBpbXBvcnQgc2VwYXJhdGUgZnJvbSBvdGhlciBpbXBvcnRzIHRvIHJlZHVjZSBjaGFuY2UgZm9yIG1lcmdlIGNvbmZsaWN0cyB3aXRoIHYyLW1haW5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1kdXBsaWNhdGUtaW1wb3J0cywgaW1wb3J0L29yZGVyXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuLyoqXG4gKiBBIENsb3VkRm9ybWF0aW9uLWluaXQgY29uZmlndXJhdGlvblxuICovXG5leHBvcnQgY2xhc3MgQ2xvdWRGb3JtYXRpb25Jbml0IHtcbiAgLyoqXG4gICAqIEJ1aWxkIGEgbmV3IGNvbmZpZyBmcm9tIGEgc2V0IG9mIEluaXQgRWxlbWVudHNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUVsZW1lbnRzKC4uLmVsZW1lbnRzOiBJbml0RWxlbWVudFtdKTogQ2xvdWRGb3JtYXRpb25Jbml0IHtcbiAgICByZXR1cm4gQ2xvdWRGb3JtYXRpb25Jbml0LmZyb21Db25maWcobmV3IEluaXRDb25maWcoZWxlbWVudHMpKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVc2UgYW4gZXhpc3RpbmcgSW5pdENvbmZpZyBvYmplY3QgYXMgdGhlIGRlZmF1bHQgYW5kIG9ubHkgY29uZmlnXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Db25maWcoY29uZmlnOiBJbml0Q29uZmlnKTogQ2xvdWRGb3JtYXRpb25Jbml0IHtcbiAgICByZXR1cm4gQ2xvdWRGb3JtYXRpb25Jbml0LmZyb21Db25maWdTZXRzKHtcbiAgICAgIGNvbmZpZ1NldHM6IHtcbiAgICAgICAgZGVmYXVsdDogWydjb25maWcnXSxcbiAgICAgIH0sXG4gICAgICBjb25maWdzOiB7IGNvbmZpZyB9LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEJ1aWxkIGEgQ2xvdWRGb3JtYXRpb25Jbml0IGZyb20gY29uZmlnIHNldHNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUNvbmZpZ1NldHMocHJvcHM6IENvbmZpZ1NldFByb3BzKTogQ2xvdWRGb3JtYXRpb25Jbml0IHtcbiAgICByZXR1cm4gbmV3IENsb3VkRm9ybWF0aW9uSW5pdChwcm9wcy5jb25maWdTZXRzLCBwcm9wcy5jb25maWdzKTtcbiAgfVxuXG4gIHByaXZhdGUgcmVhZG9ubHkgX2NvbmZpZ1NldHM6IFJlY29yZDxzdHJpbmcsIHN0cmluZ1tdPiA9IHt9O1xuICBwcml2YXRlIHJlYWRvbmx5IF9jb25maWdzOiBSZWNvcmQ8c3RyaW5nLCBJbml0Q29uZmlnPiA9IHt9O1xuXG4gIHByaXZhdGUgY29uc3RydWN0b3IoY29uZmlnU2V0czogUmVjb3JkPHN0cmluZywgc3RyaW5nW10+LCBjb25maWdzOiBSZWNvcmQ8c3RyaW5nLCBJbml0Q29uZmlnPikge1xuICAgIE9iamVjdC5hc3NpZ24odGhpcy5fY29uZmlnU2V0cywgY29uZmlnU2V0cyk7XG4gICAgT2JqZWN0LmFzc2lnbih0aGlzLl9jb25maWdzLCBjb25maWdzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBjb25maWcgd2l0aCB0aGUgZ2l2ZW4gbmFtZSB0byB0aGlzIENsb3VkRm9ybWF0aW9uSW5pdCBvYmplY3RcbiAgICovXG4gIHB1YmxpYyBhZGRDb25maWcoY29uZmlnTmFtZTogc3RyaW5nLCBjb25maWc6IEluaXRDb25maWcpIHtcbiAgICBpZiAodGhpcy5fY29uZmlnc1tjb25maWdOYW1lXSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDbG91ZEZvcm1hdGlvbkluaXQgYWxyZWFkeSBjb250YWlucyBhIGNvbmZpZyBuYW1lZCAnJHtjb25maWdOYW1lfSdgKTtcbiAgICB9XG4gICAgdGhpcy5fY29uZmlnc1tjb25maWdOYW1lXSA9IGNvbmZpZztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBjb25maWcgc2V0IHdpdGggdGhlIGdpdmVuIG5hbWUgdG8gdGhpcyBDbG91ZEZvcm1hdGlvbkluaXQgb2JqZWN0XG4gICAqXG4gICAqIFRoZSBuZXcgY29uZmlnc2V0IHdpbGwgcmVmZXJlbmNlIHRoZSBnaXZlbiBjb25maWdzIGluIHRoZSBnaXZlbiBvcmRlci5cbiAgICovXG4gIHB1YmxpYyBhZGRDb25maWdTZXQoY29uZmlnU2V0TmFtZTogc3RyaW5nLCBjb25maWdOYW1lczogc3RyaW5nW10gPSBbXSkge1xuICAgIGlmICh0aGlzLl9jb25maWdTZXRzW2NvbmZpZ1NldE5hbWVdKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENsb3VkRm9ybWF0aW9uSW5pdCBhbHJlYWR5IGNvbnRhaW5zIGEgY29uZmlnU2V0IG5hbWVkICcke2NvbmZpZ1NldE5hbWV9J2ApO1xuICAgIH1cblxuICAgIGNvbnN0IHVuayA9IGNvbmZpZ05hbWVzLmZpbHRlcihjID0+ICF0aGlzLl9jb25maWdzW2NdKTtcbiAgICBpZiAodW5rLmxlbmd0aCA+IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5rbm93biBjb25maWdzIHJlZmVyZW5jZWQgaW4gZGVmaW5pdGlvbiBvZiAnJHtjb25maWdTZXROYW1lfSc6ICR7dW5rfWApO1xuICAgIH1cblxuICAgIHRoaXMuX2NvbmZpZ1NldHNbY29uZmlnU2V0TmFtZV0gPSBbLi4uY29uZmlnTmFtZXNdO1xuICB9XG5cbiAgLyoqXG4gICAqIEF0dGFjaCB0aGUgQ2xvdWRGb3JtYXRpb24gSW5pdCBjb25maWcgdG8gdGhlIGdpdmVuIHJlc291cmNlXG4gICAqXG4gICAqIEFzIGFuIGFwcCBidWlsZGVyLCB1c2UgYGluc3RhbmNlLmFwcGx5Q2xvdWRGb3JtYXRpb25Jbml0KClgIG9yXG4gICAqIGBhdXRvU2NhbGluZ0dyb3VwLmFwcGx5Q2xvdWRGb3JtYXRpb25Jbml0KClgIHRvIHRyaWdnZXIgdGhpcyBtZXRob2QuXG4gICAqXG4gICAqIFRoaXMgbWV0aG9kIGRvZXMgdGhlIGZvbGxvd2luZzpcbiAgICpcbiAgICogLSBSZW5kZXJzIHRoZSBgQVdTOjpDbG91ZEZvcm1hdGlvbjo6SW5pdGAgb2JqZWN0IHRvIHRoZSBnaXZlbiByZXNvdXJjZSdzXG4gICAqICAgbWV0YWRhdGEsIHBvdGVudGlhbGx5IGFkZGluZyBhIGBBV1M6OkNsb3VkRm9ybWF0aW9uOjpBdXRoZW50aWNhdGlvbmAgb2JqZWN0XG4gICAqICAgbmV4dCB0byBpdCBpZiByZXF1aXJlZC5cbiAgICogLSBVcGRhdGVzIHRoZSBpbnN0YW5jZSByb2xlIHBvbGljeSB0byBiZSBhYmxlIHRvIGNhbGwgdGhlIEFQSXMgcmVxdWlyZWQgZm9yXG4gICAqICAgYGNmbi1pbml0YCBhbmQgYGNmbi1zaWduYWxgIHRvIHdvcmssIGFuZCBwb3RlbnRpYWxseSBhZGQgcGVybWlzc2lvbnMgdG8gZG93bmxvYWRcbiAgICogICByZWZlcmVuY2VkIGFzc2V0IGFuZCBidWNrZXQgcmVzb3VyY2VzLlxuICAgKiAtIFVwZGF0ZXMgdGhlIGdpdmVuIFVzZXJEYXRhIHdpdGggY29tbWFuZHMgdG8gZXhlY3V0ZSB0aGUgYGNmbi1pbml0YCBzY3JpcHQuXG4gICAqL1xuICBwdWJsaWMgYXR0YWNoKGF0dGFjaGVkUmVzb3VyY2U6IENmblJlc291cmNlLCBhdHRhY2hPcHRpb25zOiBBdHRhY2hJbml0T3B0aW9ucykge1xuICAgIGlmIChhdHRhY2hPcHRpb25zLnBsYXRmb3JtID09PSBPcGVyYXRpbmdTeXN0ZW1UeXBlLlVOS05PV04pIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGF0dGFjaCBDbG91ZEZvcm1hdGlvbkluaXQgdG8gYW4gdW5rbm93biBPUyB0eXBlJyk7XG4gICAgfVxuXG4gICAgY29uc3QgQ0ZOX0lOSVRfTUVUQURBVEFfS0VZID0gJ0FXUzo6Q2xvdWRGb3JtYXRpb246OkluaXQnO1xuXG4gICAgaWYgKGF0dGFjaGVkUmVzb3VyY2UuZ2V0TWV0YWRhdGEoQ0ZOX0lOSVRfTUVUQURBVEFfS0VZKSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBiaW5kIENmbkluaXQ6IHJlc291cmNlICcke2F0dGFjaGVkUmVzb3VyY2Uubm9kZS5wYXRofScgYWxyZWFkeSBoYXMgJyR7Q0ZOX0lOSVRfTUVUQURBVEFfS0VZfScgYXR0YWNoZWRgKTtcbiAgICB9XG5cbiAgICAvLyBOb3RlOiBUaGlzIHdpbGwgbm90IHJlZmxlY3QgbXV0YXRpb25zIG1hZGUgYWZ0ZXIgYXR0YWNoaW5nLlxuICAgIGNvbnN0IGJpbmRSZXN1bHQgPSB0aGlzLmJpbmQoYXR0YWNoZWRSZXNvdXJjZS5zdGFjaywgYXR0YWNoT3B0aW9ucyk7XG4gICAgYXR0YWNoZWRSZXNvdXJjZS5hZGRNZXRhZGF0YShDRk5fSU5JVF9NRVRBREFUQV9LRVksIGJpbmRSZXN1bHQuY29uZmlnRGF0YSk7XG5cbiAgICAvLyBOZWVkIHRvIHJlc29sdmUgdGhlIHZhcmlvdXMgdG9rZW5zIGZyb20gYXNzZXRzIGluIHRoZSBjb25maWcsXG4gICAgLy8gYXMgd2VsbCBhcyBpbmNsdWRlIGFueSBhc3NldCBoYXNoZXMgcHJvdmlkZWQgc28gdGhlIGZpbmdlcnByaW50IGlzIGFjY3VyYXRlLlxuICAgIGNvbnN0IHJlc29sdmVkQ29uZmlnID0gYXR0YWNoZWRSZXNvdXJjZS5zdGFjay5yZXNvbHZlKGJpbmRSZXN1bHQuY29uZmlnRGF0YSk7XG4gICAgY29uc3QgZmluZ2VycHJpbnRJbnB1dCA9IHsgY29uZmlnOiByZXNvbHZlZENvbmZpZywgYXNzZXRIYXNoOiBiaW5kUmVzdWx0LmFzc2V0SGFzaCB9O1xuICAgIGNvbnN0IGZpbmdlcnByaW50ID0gY29udGVudEhhc2goSlNPTi5zdHJpbmdpZnkoZmluZ2VycHJpbnRJbnB1dCkpLnNsaWNlKDAsIDE2KTtcblxuICAgIGF0dGFjaE9wdGlvbnMuaW5zdGFuY2VSb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVTdGFja1Jlc291cmNlJywgJ2Nsb3VkZm9ybWF0aW9uOlNpZ25hbFJlc291cmNlJ10sXG4gICAgICByZXNvdXJjZXM6IFtBd3MuU1RBQ0tfSURdLFxuICAgIH0pKTtcblxuICAgIGlmIChiaW5kUmVzdWx0LmF1dGhEYXRhKSB7XG4gICAgICBhdHRhY2hlZFJlc291cmNlLmFkZE1ldGFkYXRhKCdBV1M6OkNsb3VkRm9ybWF0aW9uOjpBdXRoZW50aWNhdGlvbicsIGJpbmRSZXN1bHQuYXV0aERhdGEpO1xuICAgIH1cblxuICAgIC8vIFRvIGlkZW50aWZ5IHRoZSByZXNvdXJjZXMgdGhhdCBoYXZlIHRoZSBtZXRhZGF0YSBhbmQgd2hlcmUgdGhlIHNpZ25hbFxuICAgIC8vIG5lZWRzIHRvIGJlIHNlbnQsIHdlIG5lZWQgeyByZWdpb24sIHN0YWNrTmFtZSwgbG9naWNhbElkIH1cbiAgICBsZXQgcmVzb3VyY2VMb2NhdG9yID0gYC0tcmVnaW9uICR7QXdzLlJFR0lPTn0gLS1zdGFjayAke0F3cy5TVEFDS19OQU1FfSAtLXJlc291cmNlICR7YXR0YWNoZWRSZXNvdXJjZS5sb2dpY2FsSWR9YDtcbiAgICBjb25zdCBzaWduYWxSZXNvdXJjZSA9IGF0dGFjaE9wdGlvbnMuc2lnbmFsUmVzb3VyY2U/LmxvZ2ljYWxJZCA/PyBhdHRhY2hlZFJlc291cmNlLmxvZ2ljYWxJZDtcbiAgICBsZXQgbm90aWZ5UmVzb3VyY2VMb2NhdG9yID0gYC0tcmVnaW9uICR7QXdzLlJFR0lPTn0gLS1zdGFjayAke0F3cy5TVEFDS19OQU1FfSAtLXJlc291cmNlICR7c2lnbmFsUmVzb3VyY2V9YDtcblxuICAgIC8vIElmIHNwZWNpZmllZCBpbiBhdHRhY2hPcHRpb25zLCBpbmNsdWRlIGFyZ3VtZW50cyBpbiBjZm4taW5pdC9jZm4tc2lnbmFsIGNvbW1hbmRzXG4gICAgaWYgKGF0dGFjaE9wdGlvbnMuaW5jbHVkZVVybCkge1xuICAgICAgcmVzb3VyY2VMb2NhdG9yID0gYCR7cmVzb3VyY2VMb2NhdG9yfSAtLXVybCBodHRwczovL2Nsb3VkZm9ybWF0aW9uLiR7QXdzLlJFR0lPTn0uJHtBd3MuVVJMX1NVRkZJWH1gO1xuICAgICAgbm90aWZ5UmVzb3VyY2VMb2NhdG9yID0gYCR7bm90aWZ5UmVzb3VyY2VMb2NhdG9yfSAtLXVybCBodHRwczovL2Nsb3VkZm9ybWF0aW9uLiR7QXdzLlJFR0lPTn0uJHtBd3MuVVJMX1NVRkZJWH1gO1xuICAgIH1cbiAgICBpZiAoYXR0YWNoT3B0aW9ucy5pbmNsdWRlUm9sZSkge1xuICAgICAgcmVzb3VyY2VMb2NhdG9yID0gYCR7cmVzb3VyY2VMb2NhdG9yfSAtLXJvbGUgJHthdHRhY2hPcHRpb25zLmluc3RhbmNlUm9sZS5yb2xlTmFtZX1gO1xuICAgICAgbm90aWZ5UmVzb3VyY2VMb2NhdG9yID0gYCR7bm90aWZ5UmVzb3VyY2VMb2NhdG9yfSAtLXJvbGUgJHthdHRhY2hPcHRpb25zLmluc3RhbmNlUm9sZS5yb2xlTmFtZX1gO1xuICAgIH1cblxuICAgIGNvbnN0IGNvbmZpZ1NldHMgPSAoYXR0YWNoT3B0aW9ucy5jb25maWdTZXRzID8/IFsnZGVmYXVsdCddKS5qb2luKCcsJyk7XG4gICAgY29uc3QgcHJpbnRMb2cgPSBhdHRhY2hPcHRpb25zLnByaW50TG9nID8/IHRydWU7XG5cbiAgICBpZiAoYXR0YWNoT3B0aW9ucy5lbWJlZEZpbmdlcnByaW50ID8/IHRydWUpIHtcbiAgICAgIC8vIEl0IGp1c3Qgc28gaGFwcGVucyB0aGF0IHRoZSBjb21tZW50IGNoYXIgaXMgJyMnIGZvciBib3RoIGJhc2ggYW5kIFBvd2VyU2hlbGxcbiAgICAgIGF0dGFjaE9wdGlvbnMudXNlckRhdGEuYWRkQ29tbWFuZHMoYCMgZmluZ2VycHJpbnQ6ICR7ZmluZ2VycHJpbnR9YCk7XG4gICAgfVxuXG4gICAgaWYgKGF0dGFjaE9wdGlvbnMucGxhdGZvcm0gPT09IE9wZXJhdGluZ1N5c3RlbVR5cGUuV0lORE9XUykge1xuICAgICAgY29uc3QgZXJyQ29kZSA9IGF0dGFjaE9wdGlvbnMuaWdub3JlRmFpbHVyZXMgPyAnMCcgOiAnJExBU1RFWElUQ09ERSc7XG4gICAgICBhdHRhY2hPcHRpb25zLnVzZXJEYXRhLmFkZENvbW1hbmRzKFxuICAgICAgICAuLi5bXG4gICAgICAgICAgYGNmbi1pbml0LmV4ZSAtdiAke3Jlc291cmNlTG9jYXRvcn0gLWMgJHtjb25maWdTZXRzfWAsXG4gICAgICAgICAgYGNmbi1zaWduYWwuZXhlIC1lICR7ZXJyQ29kZX0gJHtub3RpZnlSZXNvdXJjZUxvY2F0b3J9YCxcbiAgICAgICAgICAuLi4ocHJpbnRMb2cgPyBbJ3R5cGUgQzpcXFxcY2ZuXFxcXGxvZ1xcXFxjZm4taW5pdC5sb2cnXSA6IFtdKSxcbiAgICAgICAgXSxcbiAgICAgICk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IGVyckNvZGUgPSBhdHRhY2hPcHRpb25zLmlnbm9yZUZhaWx1cmVzID8gJzAnIDogJyQ/JztcbiAgICAgIGF0dGFjaE9wdGlvbnMudXNlckRhdGEuYWRkQ29tbWFuZHMoXG4gICAgICAgIC4uLltcbiAgICAgICAgICAvLyBSdW4gYSBzdWJzaGVsbCB3aXRob3V0ICdlcnJleGl0Jywgc28gd2UgY2FuIHNpZ25hbCB1c2luZyB0aGUgZXhpdCBjb2RlIG9mIGNmbi1pbml0XG4gICAgICAgICAgJygnLFxuICAgICAgICAgICcgIHNldCArZScsXG4gICAgICAgICAgYCAgL29wdC9hd3MvYmluL2Nmbi1pbml0IC12ICR7cmVzb3VyY2VMb2NhdG9yfSAtYyAke2NvbmZpZ1NldHN9YCxcbiAgICAgICAgICBgICAvb3B0L2F3cy9iaW4vY2ZuLXNpZ25hbCAtZSAke2VyckNvZGV9ICR7bm90aWZ5UmVzb3VyY2VMb2NhdG9yfWAsXG4gICAgICAgICAgLi4uKHByaW50TG9nID8gWycgIGNhdCAvdmFyL2xvZy9jZm4taW5pdC5sb2cgPiYyJ10gOiBbXSksXG4gICAgICAgICAgJyknLFxuICAgICAgICBdLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGJpbmQoc2NvcGU6IENvbnN0cnVjdCwgb3B0aW9uczogQXR0YWNoSW5pdE9wdGlvbnMpOiB7IGNvbmZpZ0RhdGE6IGFueSwgYXV0aERhdGE6IGFueSwgYXNzZXRIYXNoPzogYW55IH0ge1xuICAgIGNvbnN0IG5vbkVtcHR5Q29uZmlncyA9IG1hcFZhbHVlcyh0aGlzLl9jb25maWdzLCBjID0+IGMuaXNFbXB0eSgpID8gdW5kZWZpbmVkIDogYyk7XG5cbiAgICBjb25zdCBjb25maWdOYW1lVG9CaW5kUmVzdWx0ID0gbWFwVmFsdWVzKG5vbkVtcHR5Q29uZmlncywgYyA9PiBjLl9iaW5kKHNjb3BlLCBvcHRpb25zKSk7XG5cbiAgICByZXR1cm4ge1xuICAgICAgY29uZmlnRGF0YToge1xuICAgICAgICBjb25maWdTZXRzOiBtYXBWYWx1ZXModGhpcy5fY29uZmlnU2V0cywgY29uZmlnTmFtZXMgPT4gY29uZmlnTmFtZXMuZmlsdGVyKG5hbWUgPT4gbm9uRW1wdHlDb25maWdzW25hbWVdICE9PSB1bmRlZmluZWQpKSxcbiAgICAgICAgLi4ubWFwVmFsdWVzKGNvbmZpZ05hbWVUb0JpbmRSZXN1bHQsIGMgPT4gYy5jb25maWcpLFxuICAgICAgfSxcbiAgICAgIGF1dGhEYXRhOiBPYmplY3QudmFsdWVzKGNvbmZpZ05hbWVUb0JpbmRSZXN1bHQpLm1hcChjID0+IGMuYXV0aGVudGljYXRpb24pLnJlZHVjZShkZWVwTWVyZ2UsIHVuZGVmaW5lZCksXG4gICAgICBhc3NldEhhc2g6IGNvbWJpbmVBc3NldEhhc2hlc09yVW5kZWZpbmVkKE9iamVjdC52YWx1ZXMoY29uZmlnTmFtZVRvQmluZFJlc3VsdCkubWFwKGMgPT4gYy5hc3NldEhhc2gpKSxcbiAgICB9O1xuICB9XG5cbn1cblxuLyoqXG4gKiBBIGNvbGxlY3Rpb24gb2YgY29uZmlndXJhdGlvbiBlbGVtZW50c1xuICovXG5leHBvcnQgY2xhc3MgSW5pdENvbmZpZyB7XG4gIHByaXZhdGUgcmVhZG9ubHkgZWxlbWVudHMgPSBuZXcgQXJyYXk8SW5pdEVsZW1lbnQ+KCk7XG5cbiAgY29uc3RydWN0b3IoZWxlbWVudHM6IEluaXRFbGVtZW50W10pIHtcbiAgICB0aGlzLmFkZCguLi5lbGVtZW50cyk7XG4gIH1cblxuICAvKipcbiAgICogV2hldGhlciB0aGlzIGNvbmZpZ3NldCBoYXMgZWxlbWVudHMgb3Igbm90XG4gICAqL1xuICBwdWJsaWMgaXNFbXB0eSgpIHtcbiAgICByZXR1cm4gdGhpcy5lbGVtZW50cy5sZW5ndGggPT09IDA7XG4gIH1cblxuICAvKipcbiAgICogQWRkIG9uZSBvciBtb3JlIGVsZW1lbnRzIHRvIHRoZSBjb25maWdcbiAgICovXG4gIHB1YmxpYyBhZGQoLi4uZWxlbWVudHM6IEluaXRFbGVtZW50W10pIHtcbiAgICB0aGlzLmVsZW1lbnRzLnB1c2goLi4uZWxlbWVudHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIENhbGxlZCB3aGVuIHRoZSBjb25maWcgaXMgYXBwbGllZCB0byBhbiBpbnN0YW5jZS5cbiAgICogQ3JlYXRlcyB0aGUgQ2xvdWRGb3JtYXRpb24gcmVwcmVzZW50YXRpb24gb2YgdGhlIEluaXQgY29uZmlnIGFuZCBoYW5kbGVzIGFueSBwZXJtaXNzaW9ucyBhbmQgYXNzZXRzLlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfYmluZChzY29wZTogQ29uc3RydWN0LCBvcHRpb25zOiBBdHRhY2hJbml0T3B0aW9ucyk6IEluaXRFbGVtZW50Q29uZmlnIHtcbiAgICBjb25zdCBiaW5kT3B0aW9ucyA9IHtcbiAgICAgIGluc3RhbmNlUm9sZTogb3B0aW9ucy5pbnN0YW5jZVJvbGUsXG4gICAgICBwbGF0Zm9ybTogdGhpcy5pbml0UGxhdGZvcm1Gcm9tT1NUeXBlKG9wdGlvbnMucGxhdGZvcm0pLFxuICAgICAgc2NvcGUsXG4gICAgfTtcblxuICAgIGNvbnN0IHBhY2thZ2VDb25maWcgPSB0aGlzLmJpbmRGb3JUeXBlKEluaXRFbGVtZW50VHlwZS5QQUNLQUdFLCBiaW5kT3B0aW9ucyk7XG4gICAgY29uc3QgZ3JvdXBzQ29uZmlnID0gdGhpcy5iaW5kRm9yVHlwZShJbml0RWxlbWVudFR5cGUuR1JPVVAsIGJpbmRPcHRpb25zKTtcbiAgICBjb25zdCB1c2Vyc0NvbmZpZyA9IHRoaXMuYmluZEZvclR5cGUoSW5pdEVsZW1lbnRUeXBlLlVTRVIsIGJpbmRPcHRpb25zKTtcbiAgICBjb25zdCBzb3VyY2VzQ29uZmlnID0gdGhpcy5iaW5kRm9yVHlwZShJbml0RWxlbWVudFR5cGUuU09VUkNFLCBiaW5kT3B0aW9ucyk7XG4gICAgY29uc3QgZmlsZXNDb25maWcgPSB0aGlzLmJpbmRGb3JUeXBlKEluaXRFbGVtZW50VHlwZS5GSUxFLCBiaW5kT3B0aW9ucyk7XG4gICAgY29uc3QgY29tbWFuZHNDb25maWcgPSB0aGlzLmJpbmRGb3JUeXBlKEluaXRFbGVtZW50VHlwZS5DT01NQU5ELCBiaW5kT3B0aW9ucyk7XG4gICAgLy8gTXVzdCBiZSBsYXN0IVxuICAgIGNvbnN0IHNlcnZpY2VzQ29uZmlnID0gdGhpcy5iaW5kRm9yVHlwZShJbml0RWxlbWVudFR5cGUuU0VSVklDRSwgYmluZE9wdGlvbnMpO1xuXG4gICAgY29uc3QgYWxsQ29uZmlnID0gW3BhY2thZ2VDb25maWcsIGdyb3Vwc0NvbmZpZywgdXNlcnNDb25maWcsIHNvdXJjZXNDb25maWcsIGZpbGVzQ29uZmlnLCBjb21tYW5kc0NvbmZpZywgc2VydmljZXNDb25maWddO1xuICAgIGNvbnN0IGF1dGhlbnRpY2F0aW9uID0gYWxsQ29uZmlnLm1hcChjID0+IGM/LmF1dGhlbnRpY2F0aW9uKS5yZWR1Y2UoZGVlcE1lcmdlLCB1bmRlZmluZWQpO1xuICAgIGNvbnN0IGFzc2V0SGFzaCA9IGNvbWJpbmVBc3NldEhhc2hlc09yVW5kZWZpbmVkKGFsbENvbmZpZy5tYXAoYyA9PiBjPy5hc3NldEhhc2gpKTtcblxuICAgIHJldHVybiB7XG4gICAgICBjb25maWc6IHtcbiAgICAgICAgcGFja2FnZXM6IHBhY2thZ2VDb25maWc/LmNvbmZpZyxcbiAgICAgICAgZ3JvdXBzOiBncm91cHNDb25maWc/LmNvbmZpZyxcbiAgICAgICAgdXNlcnM6IHVzZXJzQ29uZmlnPy5jb25maWcsXG4gICAgICAgIHNvdXJjZXM6IHNvdXJjZXNDb25maWc/LmNvbmZpZyxcbiAgICAgICAgZmlsZXM6IGZpbGVzQ29uZmlnPy5jb25maWcsXG4gICAgICAgIGNvbW1hbmRzOiBjb21tYW5kc0NvbmZpZz8uY29uZmlnLFxuICAgICAgICBzZXJ2aWNlczogc2VydmljZXNDb25maWc/LmNvbmZpZyxcbiAgICAgIH0sXG4gICAgICBhdXRoZW50aWNhdGlvbixcbiAgICAgIGFzc2V0SGFzaCxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBiaW5kRm9yVHlwZShlbGVtZW50VHlwZTogSW5pdEVsZW1lbnRUeXBlLCByZW5kZXJPcHRpb25zOiBPbWl0PEluaXRCaW5kT3B0aW9ucywgJ2luZGV4Jz4pOiBJbml0RWxlbWVudENvbmZpZyB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3QgZWxlbWVudHMgPSB0aGlzLmVsZW1lbnRzLmZpbHRlcihlbGVtID0+IGVsZW0uZWxlbWVudFR5cGUgPT09IGVsZW1lbnRUeXBlKTtcbiAgICBpZiAoZWxlbWVudHMubGVuZ3RoID09PSAwKSB7IHJldHVybiB1bmRlZmluZWQ7IH1cblxuICAgIGNvbnN0IGJpbmRSZXN1bHRzID0gZWxlbWVudHMubWFwKChlLCBpbmRleCkgPT4gZS5fYmluZCh7IGluZGV4LCAuLi5yZW5kZXJPcHRpb25zIH0pKTtcblxuICAgIHJldHVybiB7XG4gICAgICBjb25maWc6IGJpbmRSZXN1bHRzLm1hcChyID0+IHIuY29uZmlnKS5yZWR1Y2UoZGVlcE1lcmdlLCB1bmRlZmluZWQpID8/IHt9LFxuICAgICAgYXV0aGVudGljYXRpb246IGJpbmRSZXN1bHRzLm1hcChyID0+IHIuYXV0aGVudGljYXRpb24pLnJlZHVjZShkZWVwTWVyZ2UsIHVuZGVmaW5lZCksXG4gICAgICBhc3NldEhhc2g6IGNvbWJpbmVBc3NldEhhc2hlc09yVW5kZWZpbmVkKGJpbmRSZXN1bHRzLm1hcChyID0+IHIuYXNzZXRIYXNoKSksXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgaW5pdFBsYXRmb3JtRnJvbU9TVHlwZShvc1R5cGU6IE9wZXJhdGluZ1N5c3RlbVR5cGUpOiBJbml0UGxhdGZvcm0ge1xuICAgIHN3aXRjaCAob3NUeXBlKSB7XG4gICAgICBjYXNlIE9wZXJhdGluZ1N5c3RlbVR5cGUuTElOVVg6IHtcbiAgICAgICAgcmV0dXJuIEluaXRQbGF0Zm9ybS5MSU5VWDtcbiAgICAgIH1cbiAgICAgIGNhc2UgT3BlcmF0aW5nU3lzdGVtVHlwZS5XSU5ET1dTOiB7XG4gICAgICAgIHJldHVybiBJbml0UGxhdGZvcm0uV0lORE9XUztcbiAgICAgIH1cbiAgICAgIGRlZmF1bHQ6IHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgYXR0YWNoIENsb3VkRm9ybWF0aW9uSW5pdCB0byBhbiB1bmtub3duIE9TIHR5cGUnKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBDbG91ZEZvcm1hdGlvbkluaXQud2l0aENvbmZpZ1NldHNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb25maWdTZXRQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgZGVmaW5pdGlvbnMgb2YgZWFjaCBjb25maWcgc2V0XG4gICAqL1xuICByZWFkb25seSBjb25maWdTZXRzOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmdbXT47XG5cbiAgLyoqXG4gICAqIFRoZSBzZXRzIG9mIGNvbmZpZ3MgdG8gcGljayBmcm9tXG4gICAqL1xuICByZWFkb25seSBjb25maWdzOiBSZWNvcmQ8c3RyaW5nLCBJbml0Q29uZmlnPjtcbn1cblxuLyoqXG4gKiBEZWVwLW1lcmdlIG9iamVjdHMgYW5kIGFycmF5c1xuICpcbiAqIFRyZWF0IGFycmF5cyBhcyBzZXRzLCByZW1vdmluZyBkdXBsaWNhdGVzLiBUaGlzIGlzIGFjY2VwdGFibGUgZm9yIHJlbmRlcmluZ1xuICogY2ZuLWluaXRzLCBub3QgYXBwbGljYWJsZSBlbHNld2hlcmUuXG4gKi9cbmZ1bmN0aW9uIGRlZXBNZXJnZSh0YXJnZXQ/OiBSZWNvcmQ8c3RyaW5nLCBhbnk+LCBzcmM/OiBSZWNvcmQ8c3RyaW5nLCBhbnk+KSB7XG4gIGlmICh0YXJnZXQgPT0gbnVsbCkgeyByZXR1cm4gc3JjOyB9XG4gIGlmIChzcmMgPT0gbnVsbCkgeyByZXR1cm4gdGFyZ2V0OyB9XG5cbiAgZm9yIChjb25zdCBba2V5LCB2YWx1ZV0gb2YgT2JqZWN0LmVudHJpZXMoc3JjKSkge1xuICAgIGlmIChBcnJheS5pc0FycmF5KHZhbHVlKSkge1xuICAgICAgaWYgKHRhcmdldFtrZXldICYmICFBcnJheS5pc0FycmF5KHRhcmdldFtrZXldKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRyeWluZyB0byBtZXJnZSBhcnJheSBbJHt2YWx1ZX1dIGludG8gYSBub24tYXJyYXkgJyR7dGFyZ2V0W2tleV19J2ApO1xuICAgICAgfVxuICAgICAgdGFyZ2V0W2tleV0gPSBBcnJheS5mcm9tKG5ldyBTZXQoW1xuICAgICAgICAuLi50YXJnZXRba2V5XSA/PyBbXSxcbiAgICAgICAgLi4udmFsdWUsXG4gICAgICBdKSk7XG4gICAgICBjb250aW51ZTtcbiAgICB9XG4gICAgaWYgKHR5cGVvZiB2YWx1ZSA9PT0gJ29iamVjdCcgJiYgdmFsdWUpIHtcbiAgICAgIHRhcmdldFtrZXldID0gZGVlcE1lcmdlKHRhcmdldFtrZXldID8/IHt9LCB2YWx1ZSk7XG4gICAgICBjb250aW51ZTtcbiAgICB9XG4gICAgaWYgKHZhbHVlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRhcmdldFtrZXldID0gdmFsdWU7XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIHRhcmdldDtcbn1cblxuLyoqXG4gKiBNYXAgYSBmdW5jdGlvbiBvdmVyIHZhbHVlcyBvZiBhbiBvYmplY3RcbiAqXG4gKiBJZiB0aGUgbWFwcGluZyBmdW5jdGlvbiByZXR1cm5zIHVuZGVmaW5lZCwgcmVtb3ZlIHRoZSBrZXlcbiAqL1xuZnVuY3Rpb24gbWFwVmFsdWVzPEEsIEI+KHhzOiBSZWNvcmQ8c3RyaW5nLCBBPiwgZm46ICh4OiBBKSA9PiBCIHwgdW5kZWZpbmVkKTogUmVjb3JkPHN0cmluZywgQj4ge1xuICBjb25zdCByZXQ6IFJlY29yZDxzdHJpbmcsIEI+ID0ge307XG4gIGZvciAoY29uc3QgW2ssIHZdIG9mIE9iamVjdC5lbnRyaWVzKHhzKSkge1xuICAgIGNvbnN0IG1hcHBlZCA9IGZuKHYpO1xuICAgIGlmIChtYXBwZWQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0W2tdID0gbWFwcGVkO1xuICAgIH1cbiAgfVxuICByZXR1cm4gcmV0O1xufVxuXG4vLyBDb21iaW5lcyBhbGwgaW5wdXQgYXNzZXQgaGFzaGVzIGludG8gb25lLCBvciBpZiBubyBoYXNoZXMgYXJlIHByZXNlbnQsIHJldHVybnMgdW5kZWZpbmVkLlxuZnVuY3Rpb24gY29tYmluZUFzc2V0SGFzaGVzT3JVbmRlZmluZWQoaGFzaGVzOiAoc3RyaW5nIHwgdW5kZWZpbmVkKVtdKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgY29uc3QgaGFzaEFycmF5ID0gaGFzaGVzLmZpbHRlcigoeCk6IHggaXMgc3RyaW5nID0+IHggIT09IHVuZGVmaW5lZCk7XG4gIHJldHVybiBoYXNoQXJyYXkubGVuZ3RoID4gMCA/IGhhc2hBcnJheS5qb2luKCcnKSA6IHVuZGVmaW5lZDtcbn1cblxuZnVuY3Rpb24gY29udGVudEhhc2goY29udGVudDogc3RyaW5nKSB7XG4gIHJldHVybiBjcnlwdG8uY3JlYXRlSGFzaCgnc2hhMjU2JykudXBkYXRlKGNvbnRlbnQpLmRpZ2VzdCgnaGV4Jyk7XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYXR0YWNoaW5nIGEgQ2xvdWRGb3JtYXRpb25Jbml0IHRvIGEgcmVzb3VyY2VcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBdHRhY2hJbml0T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBJbnN0YW5jZSByb2xlIG9mIHRoZSBjb25zdW1pbmcgaW5zdGFuY2Ugb3IgZmxlZXRcbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlUm9sZTogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBJbmNsdWRlIC0tdXJsIGFyZ3VtZW50IHdoZW4gcnVubmluZyBjZm4taW5pdCBhbmQgY2ZuLXNpZ25hbCBjb21tYW5kc1xuICAgKlxuICAgKiBUaGlzIHdpbGwgYmUgdGhlIGNsb3VkZm9ybWF0aW9uIGVuZHBvaW50IGluIHRoZSBkZXBsb3llZCByZWdpb25cbiAgICogZS5nLiBodHRwczovL2Nsb3VkZm9ybWF0aW9uLnVzLWVhc3QtMS5hbWF6b25hd3MuY29tXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBpbmNsdWRlVXJsPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogSW5jbHVkZSAtLXJvbGUgYXJndW1lbnQgd2hlbiBydW5uaW5nIGNmbi1pbml0IGFuZCBjZm4tc2lnbmFsIGNvbW1hbmRzXG4gICAqXG4gICAqIFRoaXMgd2lsbCBiZSB0aGUgSUFNIGluc3RhbmNlIHByb2ZpbGUgYXR0YWNoZWQgdG8gdGhlIEVDMiBpbnN0YW5jZVxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgaW5jbHVkZVJvbGU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBPUyBQbGF0Zm9ybSB0aGUgaW5pdCBjb25maWcgd2lsbCBiZSB1c2VkIGZvclxuICAgKi9cbiAgcmVhZG9ubHkgcGxhdGZvcm06IE9wZXJhdGluZ1N5c3RlbVR5cGU7XG5cbiAgLyoqXG4gICAqIFVzZXJEYXRhIHRvIGFkZCBjb21tYW5kcyB0b1xuICAgKi9cbiAgcmVhZG9ubHkgdXNlckRhdGE6IFVzZXJEYXRhO1xuXG4gIC8qKlxuICAgKiBDb25maWdTZXQgdG8gYWN0aXZhdGVcbiAgICpcbiAgICogQGRlZmF1bHQgWydkZWZhdWx0J11cbiAgICovXG4gIHJlYWRvbmx5IGNvbmZpZ1NldHM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogV2hldGhlciB0byBlbWJlZCBhIGhhc2ggaW50byB0aGUgdXNlckRhdGFcbiAgICpcbiAgICogSWYgYHRydWVgICh0aGUgZGVmYXVsdCksIGEgaGFzaCBvZiB0aGUgY29uZmlnIHdpbGwgYmUgZW1iZWRkZWQgaW50byB0aGVcbiAgICogVXNlckRhdGEsIHNvIHRoYXQgaWYgdGhlIGNvbmZpZyBjaGFuZ2VzLCB0aGUgVXNlckRhdGEgY2hhbmdlcyBhbmRcbiAgICogdGhlIGluc3RhbmNlIHdpbGwgYmUgcmVwbGFjZWQuXG4gICAqXG4gICAqIElmIGBmYWxzZWAsIG5vIHN1Y2ggaGFzaCB3aWxsIGJlIGVtYmVkZGVkLCBhbmQgaWYgdGhlIENsb3VkRm9ybWF0aW9uIEluaXRcbiAgICogY29uZmlnIGNoYW5nZXMgbm90aGluZyB3aWxsIGhhcHBlbiB0byB0aGUgcnVubmluZyBpbnN0YW5jZS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZW1iZWRGaW5nZXJwcmludD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFByaW50IHRoZSByZXN1bHRzIG9mIHJ1bm5pbmcgY2ZuLWluaXQgdG8gdGhlIEluc3RhbmNlIFN5c3RlbSBMb2dcbiAgICpcbiAgICogQnkgZGVmYXVsdCwgdGhlIG91dHB1dCBvZiBydW5uaW5nIGNmbi1pbml0IGlzIHdyaXR0ZW4gdG8gYSBsb2cgZmlsZVxuICAgKiBvbiB0aGUgaW5zdGFuY2UuIFNldCB0aGlzIHRvIGB0cnVlYCB0byBwcmludCBpdCB0byB0aGUgU3lzdGVtIExvZ1xuICAgKiAodmlzaWJsZSBmcm9tIHRoZSBFQzIgQ29uc29sZSksIGBmYWxzZWAgdG8gbm90IHByaW50IGl0LlxuICAgKlxuICAgKiAoQmUgYXdhcmUgdGhhdCB0aGUgc3lzdGVtIGxvZyBpcyByZWZyZXNoZWQgYXQgY2VydGFpbiBwb2ludHMgaW5cbiAgICogdGltZSBvZiB0aGUgaW5zdGFuY2UgbGlmZSBjeWNsZSwgYW5kIHN1Y2Nlc3NmdWwgZXhlY3V0aW9uIG1heVxuICAgKiBub3QgYWx3YXlzIHNob3cgdXApLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBwcmludExvZz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIERvbid0IGZhaWwgdGhlIGluc3RhbmNlIGNyZWF0aW9uIHdoZW4gY2ZuLWluaXQgZmFpbHNcbiAgICpcbiAgICogWW91IGNhbiB1c2UgdGhpcyB0byBwcmV2ZW50IENsb3VkRm9ybWF0aW9uIGZyb20gcm9sbGluZyBiYWNrIHdoZW5cbiAgICogaW5zdGFuY2VzIGZhaWwgdG8gc3RhcnQgdXAsIHRvIGhlbHAgaW4gZGVidWdnaW5nLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgaWdub3JlRmFpbHVyZXM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGVuIHByb3ZpZGVkLCBzaWduYWxzIHRoaXMgcmVzb3VyY2UgaW5zdGVhZCBvZiB0aGUgYXR0YWNoZWQgcmVzb3VyY2VcbiAgICpcbiAgICogWW91IGNhbiB1c2UgdGhpcyB0byBzdXBwb3J0IHNpZ25hbGluZyBMYXVuY2hUZW1wbGF0ZSB3aGlsZSBhdHRhY2hpbmcgQXV0b1NjYWxpbmdHcm91cFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGlmIHRoaXMgcHJvcGVydHkgaXMgdW5kZWZpbmVkIGNmbi1zaWduYWwgc2lnbmFscyB0aGUgYXR0YWNoZWQgcmVzb3VyY2VcbiAgICovXG4gIHJlYWRvbmx5IHNpZ25hbFJlc291cmNlPzogQ2ZuUmVzb3VyY2U7XG59XG4iXX0=