"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxUbuntuComponents = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("./common");
const common_2 = require("../common");
/**
 * Components for Ubuntu Linux that can be used with AWS Image Builder based builders. These cannot be used by {@link CodeBuildImageBuilder}.
 */
class LinuxUbuntuComponents {
    static requiredPackages(scope, id, architecture) {
        let archUrl;
        if (architecture.is(common_2.Architecture.X86_64)) {
            archUrl = 'amd64';
        }
        else if (architecture.is(common_2.Architecture.ARM64)) {
            archUrl = 'arm64';
        }
        else {
            throw new Error(`Unsupported architecture for required packages: ${architecture.name}`);
        }
        return new common_1.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'Required packages',
            description: 'Install packages required for GitHub Runner and upgrade all packages',
            commands: [
                'set -ex',
                'apt-get update',
                'DEBIAN_FRONTEND=noninteractive apt-get upgrade -y',
                'DEBIAN_FRONTEND=noninteractive apt-get install -y curl sudo jq bash zip unzip iptables software-properties-common ca-certificates',
                `curl -sfLo /tmp/amazon-cloudwatch-agent.deb https://s3.amazonaws.com/amazoncloudwatch-agent/ubuntu/${archUrl}/latest/amazon-cloudwatch-agent.deb`,
                'dpkg -i -E /tmp/amazon-cloudwatch-agent.deb',
                'rm /tmp/amazon-cloudwatch-agent.deb',
            ],
        });
    }
    static runnerUser(scope, id, _architecture) {
        return new common_1.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'GitHub Runner user',
            description: 'Install latest version of AWS CLI',
            commands: [
                'set -ex',
                'addgroup runner',
                'adduser --system --disabled-password --home /home/runner --ingroup runner runner',
                'usermod -aG sudo runner',
                'echo "%sudo   ALL=(ALL:ALL) NOPASSWD: ALL" > /etc/sudoers.d/runner',
            ],
        });
    }
    static awsCli(scope, id, architecture) {
        let archUrl;
        if (architecture.is(common_2.Architecture.X86_64)) {
            archUrl = 'x86_64';
        }
        else if (architecture.is(common_2.Architecture.ARM64)) {
            archUrl = 'aarch64';
        }
        else {
            throw new Error(`Unsupported architecture for awscli: ${architecture.name}`);
        }
        return new common_1.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'AWS CLI',
            description: 'Install latest version of AWS CLI',
            commands: [
                'set -ex',
                `curl -fsSL "https://awscli.amazonaws.com/awscli-exe-linux-${archUrl}.zip" -o awscliv2.zip`,
                'unzip -q awscliv2.zip',
                './aws/install',
                'rm -rf awscliv2.zip aws',
            ],
        });
    }
    static githubCli(scope, id, _architecture) {
        return new common_1.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'GitHub CLI',
            description: 'Install latest version of gh',
            commands: [
                'set -ex',
                'curl -fsSL https://cli.github.com/packages/githubcli-archive-keyring.gpg | sudo dd of=/usr/share/keyrings/githubcli-archive-keyring.gpg',
                'echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/githubcli-archive-keyring.gpg] ' +
                    '  https://cli.github.com/packages stable main" | sudo tee /etc/apt/sources.list.d/github-cli.list > /dev/null',
                'apt-get update',
                'DEBIAN_FRONTEND=noninteractive apt-get install -y gh',
            ],
        });
    }
    static git(scope, id, _architecture) {
        return new common_1.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'Git',
            description: 'Install latest version of git',
            commands: [
                'set -ex',
                'add-apt-repository ppa:git-core/ppa',
                'apt-get update',
                'DEBIAN_FRONTEND=noninteractive apt-get install -y git',
            ],
        });
    }
    static githubRunner(scope, id, runnerVersion, architecture) {
        let versionCommand;
        if (runnerVersion.is(common_2.RunnerVersion.latest())) {
            versionCommand = 'RUNNER_VERSION=`curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest | grep -oE "[^/v]+$"`';
        }
        else {
            versionCommand = `RUNNER_VERSION='${runnerVersion.version}'`;
        }
        let archUrl;
        if (architecture.is(common_2.Architecture.X86_64)) {
            archUrl = 'x64';
        }
        else if (architecture.is(common_2.Architecture.ARM64)) {
            archUrl = 'arm64';
        }
        else {
            throw new Error(`Unsupported architecture for GitHub Runner: ${architecture.name}`);
        }
        return new common_1.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'GitHub Actions Runner',
            description: 'Install latest version of GitHub Actions Runner',
            commands: [
                'set -ex',
                versionCommand,
                `curl -fsSLO "https://github.com/actions/runner/releases/download/v\${RUNNER_VERSION}/actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                `tar xzf "actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                `rm actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz`,
                './bin/installdependencies.sh',
            ],
        });
    }
    static docker(scope, id, _architecture) {
        return new common_1.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'Docker',
            description: 'Install latest version of Docker',
            commands: [
                'set -ex',
                'curl -fsSL https://download.docker.com/linux/ubuntu/gpg | sudo gpg --dearmor -o /usr/share/keyrings/docker.gpg',
                'echo ' +
                    '  "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker.gpg] https://download.docker.com/linux/ubuntu ' +
                    '  $(lsb_release -cs) stable" | sudo tee /etc/apt/sources.list.d/docker.list > /dev/null',
                'apt-get update',
                'DEBIAN_FRONTEND=noninteractive apt-get install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin',
                'usermod -aG docker runner',
                'ln -s /usr/libexec/docker/cli-plugins/docker-compose /usr/bin/docker-compose',
            ],
        });
    }
    static extraCertificates(scope, id, path) {
        return new common_1.ImageBuilderComponent(scope, id, {
            platform: 'Linux',
            displayName: 'Extra certificates',
            description: 'Install self-signed certificates to provide access to GitHub Enterprise Server',
            commands: [
                'set -ex',
                'cp certs/certs.pem /usr/local/share/ca-certificates/github-enterprise-server.crt',
                'update-ca-certificates',
            ],
            assets: [
                {
                    path: 'certs',
                    asset: new aws_cdk_lib_1.aws_s3_assets.Asset(scope, `${id} Asset`, { path }),
                },
            ],
        });
    }
}
exports.LinuxUbuntuComponents = LinuxUbuntuComponents;
_a = JSII_RTTI_SYMBOL_1;
LinuxUbuntuComponents[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.LinuxUbuntuComponents", version: "0.8.4" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGludXgtY29tcG9uZW50cy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9wcm92aWRlcnMvaW1hZ2UtYnVpbGRlcnMvbGludXgtY29tcG9uZW50cy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZDQUF5RDtBQUV6RCxxQ0FBaUQ7QUFDakQsc0NBQXdEO0FBRXhEOztHQUVHO0FBQ0gsTUFBYSxxQkFBcUI7SUFDekIsTUFBTSxDQUFDLGdCQUFnQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFlBQTBCO1FBQ3JGLElBQUksT0FBTyxDQUFDO1FBQ1osSUFBSSxZQUFZLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDeEMsT0FBTyxHQUFHLE9BQU8sQ0FBQztTQUNuQjthQUFNLElBQUksWUFBWSxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQzlDLE9BQU8sR0FBRyxPQUFPLENBQUM7U0FDbkI7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsbURBQW1ELFlBQVksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1NBQ3pGO1FBRUQsT0FBTyxJQUFJLDhCQUFxQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDMUMsUUFBUSxFQUFFLE9BQU87WUFDakIsV0FBVyxFQUFFLG1CQUFtQjtZQUNoQyxXQUFXLEVBQUUsc0VBQXNFO1lBQ25GLFFBQVEsRUFBRTtnQkFDUixTQUFTO2dCQUNULGdCQUFnQjtnQkFDaEIsbURBQW1EO2dCQUNuRCxtSUFBbUk7Z0JBQ25JLHNHQUFzRyxPQUFPLHFDQUFxQztnQkFDbEosNkNBQTZDO2dCQUM3QyxxQ0FBcUM7YUFDdEM7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU0sTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxhQUEyQjtRQUNoRixPQUFPLElBQUksOEJBQXFCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUMxQyxRQUFRLEVBQUUsT0FBTztZQUNqQixXQUFXLEVBQUUsb0JBQW9CO1lBQ2pDLFdBQVcsRUFBRSxtQ0FBbUM7WUFDaEQsUUFBUSxFQUFFO2dCQUNSLFNBQVM7Z0JBQ1QsaUJBQWlCO2dCQUNqQixrRkFBa0Y7Z0JBQ2xGLHlCQUF5QjtnQkFDekIsb0VBQW9FO2FBQ3JFO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVNLE1BQU0sQ0FBQyxNQUFNLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsWUFBMEI7UUFDM0UsSUFBSSxPQUFPLENBQUM7UUFDWixJQUFJLFlBQVksQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUN4QyxPQUFPLEdBQUcsUUFBUSxDQUFDO1NBQ3BCO2FBQU0sSUFBSSxZQUFZLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDOUMsT0FBTyxHQUFHLFNBQVMsQ0FBQztTQUNyQjthQUFNO1lBQ0wsTUFBTSxJQUFJLEtBQUssQ0FBQyx3Q0FBd0MsWUFBWSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7U0FDOUU7UUFFRCxPQUFPLElBQUksOEJBQXFCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUMxQyxRQUFRLEVBQUUsT0FBTztZQUNqQixXQUFXLEVBQUUsU0FBUztZQUN0QixXQUFXLEVBQUUsbUNBQW1DO1lBQ2hELFFBQVEsRUFBRTtnQkFDUixTQUFTO2dCQUNULDZEQUE2RCxPQUFPLHVCQUF1QjtnQkFDM0YsdUJBQXVCO2dCQUN2QixlQUFlO2dCQUNmLHlCQUF5QjthQUMxQjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGFBQTJCO1FBQy9FLE9BQU8sSUFBSSw4QkFBcUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQzFDLFFBQVEsRUFBRSxPQUFPO1lBQ2pCLFdBQVcsRUFBRSxZQUFZO1lBQ3pCLFdBQVcsRUFBRSw4QkFBOEI7WUFDM0MsUUFBUSxFQUFFO2dCQUNSLFNBQVM7Z0JBQ1QseUlBQXlJO2dCQUN6SSw0R0FBNEc7b0JBQzVHLCtHQUErRztnQkFDL0csZ0JBQWdCO2dCQUNoQixzREFBc0Q7YUFDdkQ7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU0sTUFBTSxDQUFDLEdBQUcsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxhQUEyQjtRQUN6RSxPQUFPLElBQUksOEJBQXFCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUMxQyxRQUFRLEVBQUUsT0FBTztZQUNqQixXQUFXLEVBQUUsS0FBSztZQUNsQixXQUFXLEVBQUUsK0JBQStCO1lBQzVDLFFBQVEsRUFBRTtnQkFDUixTQUFTO2dCQUNULHFDQUFxQztnQkFDckMsZ0JBQWdCO2dCQUNoQix1REFBdUQ7YUFDeEQ7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU0sTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxhQUE0QixFQUFFLFlBQTBCO1FBQy9HLElBQUksY0FBc0IsQ0FBQztRQUMzQixJQUFJLGFBQWEsQ0FBQyxFQUFFLENBQUMsc0JBQWEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFO1lBQzVDLGNBQWMsR0FBRyx3SEFBd0gsQ0FBQztTQUMzSTthQUFNO1lBQ0wsY0FBYyxHQUFHLG1CQUFtQixhQUFhLENBQUMsT0FBTyxHQUFHLENBQUM7U0FDOUQ7UUFFRCxJQUFJLE9BQU8sQ0FBQztRQUNaLElBQUksWUFBWSxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ3hDLE9BQU8sR0FBRyxLQUFLLENBQUM7U0FDakI7YUFBTSxJQUFJLFlBQVksQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUM5QyxPQUFPLEdBQUcsT0FBTyxDQUFDO1NBQ25CO2FBQU07WUFDTCxNQUFNLElBQUksS0FBSyxDQUFDLCtDQUErQyxZQUFZLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztTQUNyRjtRQUVELE9BQU8sSUFBSSw4QkFBcUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQzFDLFFBQVEsRUFBRSxPQUFPO1lBQ2pCLFdBQVcsRUFBRSx1QkFBdUI7WUFDcEMsV0FBVyxFQUFFLGlEQUFpRDtZQUM5RCxRQUFRLEVBQUU7Z0JBQ1IsU0FBUztnQkFDVCxjQUFjO2dCQUNkLDZHQUE2RyxPQUFPLDZCQUE2QjtnQkFDakosaUNBQWlDLE9BQU8sNkJBQTZCO2dCQUNyRSwyQkFBMkIsT0FBTyw0QkFBNEI7Z0JBQzlELDhCQUE4QjthQUMvQjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSxNQUFNLENBQUMsTUFBTSxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGFBQTJCO1FBQzVFLE9BQU8sSUFBSSw4QkFBcUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQzFDLFFBQVEsRUFBRSxPQUFPO1lBQ2pCLFdBQVcsRUFBRSxRQUFRO1lBQ3JCLFdBQVcsRUFBRSxrQ0FBa0M7WUFDL0MsUUFBUSxFQUFFO2dCQUNSLFNBQVM7Z0JBQ1QsZ0hBQWdIO2dCQUNoSCxPQUFPO29CQUNQLCtIQUErSDtvQkFDL0gseUZBQXlGO2dCQUN6RixnQkFBZ0I7Z0JBQ2hCLCtHQUErRztnQkFDL0csMkJBQTJCO2dCQUMzQiw4RUFBOEU7YUFDL0U7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU0sTUFBTSxDQUFDLGlCQUFpQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLElBQVk7UUFDeEUsT0FBTyxJQUFJLDhCQUFxQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDMUMsUUFBUSxFQUFFLE9BQU87WUFDakIsV0FBVyxFQUFFLG9CQUFvQjtZQUNqQyxXQUFXLEVBQUUsZ0ZBQWdGO1lBQzdGLFFBQVEsRUFBRTtnQkFDUixTQUFTO2dCQUNULGtGQUFrRjtnQkFDbEYsd0JBQXdCO2FBQ3pCO1lBQ0QsTUFBTSxFQUFFO2dCQUNOO29CQUNFLElBQUksRUFBRSxPQUFPO29CQUNiLEtBQUssRUFBRSxJQUFJLDJCQUFTLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsUUFBUSxFQUFFLEVBQUUsSUFBSSxFQUFFLENBQUM7aUJBQzNEO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDOztBQXBLSCxzREFxS0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBhd3NfczNfYXNzZXRzIGFzIHMzX2Fzc2V0cyB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgSW1hZ2VCdWlsZGVyQ29tcG9uZW50IH0gZnJvbSAnLi9jb21tb24nO1xuaW1wb3J0IHsgQXJjaGl0ZWN0dXJlLCBSdW5uZXJWZXJzaW9uIH0gZnJvbSAnLi4vY29tbW9uJztcblxuLyoqXG4gKiBDb21wb25lbnRzIGZvciBVYnVudHUgTGludXggdGhhdCBjYW4gYmUgdXNlZCB3aXRoIEFXUyBJbWFnZSBCdWlsZGVyIGJhc2VkIGJ1aWxkZXJzLiBUaGVzZSBjYW5ub3QgYmUgdXNlZCBieSB7QGxpbmsgQ29kZUJ1aWxkSW1hZ2VCdWlsZGVyfS5cbiAqL1xuZXhwb3J0IGNsYXNzIExpbnV4VWJ1bnR1Q29tcG9uZW50cyB7XG4gIHB1YmxpYyBzdGF0aWMgcmVxdWlyZWRQYWNrYWdlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZSkge1xuICAgIGxldCBhcmNoVXJsO1xuICAgIGlmIChhcmNoaXRlY3R1cmUuaXMoQXJjaGl0ZWN0dXJlLlg4Nl82NCkpIHtcbiAgICAgIGFyY2hVcmwgPSAnYW1kNjQnO1xuICAgIH0gZWxzZSBpZiAoYXJjaGl0ZWN0dXJlLmlzKEFyY2hpdGVjdHVyZS5BUk02NCkpIHtcbiAgICAgIGFyY2hVcmwgPSAnYXJtNjQnO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuc3VwcG9ydGVkIGFyY2hpdGVjdHVyZSBmb3IgcmVxdWlyZWQgcGFja2FnZXM6ICR7YXJjaGl0ZWN0dXJlLm5hbWV9YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbWFnZUJ1aWxkZXJDb21wb25lbnQoc2NvcGUsIGlkLCB7XG4gICAgICBwbGF0Zm9ybTogJ0xpbnV4JyxcbiAgICAgIGRpc3BsYXlOYW1lOiAnUmVxdWlyZWQgcGFja2FnZXMnLFxuICAgICAgZGVzY3JpcHRpb246ICdJbnN0YWxsIHBhY2thZ2VzIHJlcXVpcmVkIGZvciBHaXRIdWIgUnVubmVyIGFuZCB1cGdyYWRlIGFsbCBwYWNrYWdlcycsXG4gICAgICBjb21tYW5kczogW1xuICAgICAgICAnc2V0IC1leCcsXG4gICAgICAgICdhcHQtZ2V0IHVwZGF0ZScsXG4gICAgICAgICdERUJJQU5fRlJPTlRFTkQ9bm9uaW50ZXJhY3RpdmUgYXB0LWdldCB1cGdyYWRlIC15JyxcbiAgICAgICAgJ0RFQklBTl9GUk9OVEVORD1ub25pbnRlcmFjdGl2ZSBhcHQtZ2V0IGluc3RhbGwgLXkgY3VybCBzdWRvIGpxIGJhc2ggemlwIHVuemlwIGlwdGFibGVzIHNvZnR3YXJlLXByb3BlcnRpZXMtY29tbW9uIGNhLWNlcnRpZmljYXRlcycsXG4gICAgICAgIGBjdXJsIC1zZkxvIC90bXAvYW1hem9uLWNsb3Vkd2F0Y2gtYWdlbnQuZGViIGh0dHBzOi8vczMuYW1hem9uYXdzLmNvbS9hbWF6b25jbG91ZHdhdGNoLWFnZW50L3VidW50dS8ke2FyY2hVcmx9L2xhdGVzdC9hbWF6b24tY2xvdWR3YXRjaC1hZ2VudC5kZWJgLFxuICAgICAgICAnZHBrZyAtaSAtRSAvdG1wL2FtYXpvbi1jbG91ZHdhdGNoLWFnZW50LmRlYicsXG4gICAgICAgICdybSAvdG1wL2FtYXpvbi1jbG91ZHdhdGNoLWFnZW50LmRlYicsXG4gICAgICBdLFxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBydW5uZXJVc2VyKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIF9hcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZSkge1xuICAgIHJldHVybiBuZXcgSW1hZ2VCdWlsZGVyQ29tcG9uZW50KHNjb3BlLCBpZCwge1xuICAgICAgcGxhdGZvcm06ICdMaW51eCcsXG4gICAgICBkaXNwbGF5TmFtZTogJ0dpdEh1YiBSdW5uZXIgdXNlcicsXG4gICAgICBkZXNjcmlwdGlvbjogJ0luc3RhbGwgbGF0ZXN0IHZlcnNpb24gb2YgQVdTIENMSScsXG4gICAgICBjb21tYW5kczogW1xuICAgICAgICAnc2V0IC1leCcsXG4gICAgICAgICdhZGRncm91cCBydW5uZXInLFxuICAgICAgICAnYWRkdXNlciAtLXN5c3RlbSAtLWRpc2FibGVkLXBhc3N3b3JkIC0taG9tZSAvaG9tZS9ydW5uZXIgLS1pbmdyb3VwIHJ1bm5lciBydW5uZXInLFxuICAgICAgICAndXNlcm1vZCAtYUcgc3VkbyBydW5uZXInLFxuICAgICAgICAnZWNobyBcIiVzdWRvICAgQUxMPShBTEw6QUxMKSBOT1BBU1NXRDogQUxMXCIgPiAvZXRjL3N1ZG9lcnMuZC9ydW5uZXInLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgYXdzQ2xpKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGFyY2hpdGVjdHVyZTogQXJjaGl0ZWN0dXJlKSB7XG4gICAgbGV0IGFyY2hVcmw7XG4gICAgaWYgKGFyY2hpdGVjdHVyZS5pcyhBcmNoaXRlY3R1cmUuWDg2XzY0KSkge1xuICAgICAgYXJjaFVybCA9ICd4ODZfNjQnO1xuICAgIH0gZWxzZSBpZiAoYXJjaGl0ZWN0dXJlLmlzKEFyY2hpdGVjdHVyZS5BUk02NCkpIHtcbiAgICAgIGFyY2hVcmwgPSAnYWFyY2g2NCc7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5zdXBwb3J0ZWQgYXJjaGl0ZWN0dXJlIGZvciBhd3NjbGk6ICR7YXJjaGl0ZWN0dXJlLm5hbWV9YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbWFnZUJ1aWxkZXJDb21wb25lbnQoc2NvcGUsIGlkLCB7XG4gICAgICBwbGF0Zm9ybTogJ0xpbnV4JyxcbiAgICAgIGRpc3BsYXlOYW1lOiAnQVdTIENMSScsXG4gICAgICBkZXNjcmlwdGlvbjogJ0luc3RhbGwgbGF0ZXN0IHZlcnNpb24gb2YgQVdTIENMSScsXG4gICAgICBjb21tYW5kczogW1xuICAgICAgICAnc2V0IC1leCcsXG4gICAgICAgIGBjdXJsIC1mc1NMIFwiaHR0cHM6Ly9hd3NjbGkuYW1hem9uYXdzLmNvbS9hd3NjbGktZXhlLWxpbnV4LSR7YXJjaFVybH0uemlwXCIgLW8gYXdzY2xpdjIuemlwYCxcbiAgICAgICAgJ3VuemlwIC1xIGF3c2NsaXYyLnppcCcsXG4gICAgICAgICcuL2F3cy9pbnN0YWxsJyxcbiAgICAgICAgJ3JtIC1yZiBhd3NjbGl2Mi56aXAgYXdzJyxcbiAgICAgIF0sXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGdpdGh1YkNsaShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBfYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpIHtcbiAgICByZXR1cm4gbmV3IEltYWdlQnVpbGRlckNvbXBvbmVudChzY29wZSwgaWQsIHtcbiAgICAgIHBsYXRmb3JtOiAnTGludXgnLFxuICAgICAgZGlzcGxheU5hbWU6ICdHaXRIdWIgQ0xJJyxcbiAgICAgIGRlc2NyaXB0aW9uOiAnSW5zdGFsbCBsYXRlc3QgdmVyc2lvbiBvZiBnaCcsXG4gICAgICBjb21tYW5kczogW1xuICAgICAgICAnc2V0IC1leCcsXG4gICAgICAgICdjdXJsIC1mc1NMIGh0dHBzOi8vY2xpLmdpdGh1Yi5jb20vcGFja2FnZXMvZ2l0aHViY2xpLWFyY2hpdmUta2V5cmluZy5ncGcgfCBzdWRvIGRkIG9mPS91c3Ivc2hhcmUva2V5cmluZ3MvZ2l0aHViY2xpLWFyY2hpdmUta2V5cmluZy5ncGcnLFxuICAgICAgICAnZWNobyBcImRlYiBbYXJjaD0kKGRwa2cgLS1wcmludC1hcmNoaXRlY3R1cmUpIHNpZ25lZC1ieT0vdXNyL3NoYXJlL2tleXJpbmdzL2dpdGh1YmNsaS1hcmNoaXZlLWtleXJpbmcuZ3BnXSAnICtcbiAgICAgICAgJyAgaHR0cHM6Ly9jbGkuZ2l0aHViLmNvbS9wYWNrYWdlcyBzdGFibGUgbWFpblwiIHwgc3VkbyB0ZWUgL2V0Yy9hcHQvc291cmNlcy5saXN0LmQvZ2l0aHViLWNsaS5saXN0ID4gL2Rldi9udWxsJyxcbiAgICAgICAgJ2FwdC1nZXQgdXBkYXRlJyxcbiAgICAgICAgJ0RFQklBTl9GUk9OVEVORD1ub25pbnRlcmFjdGl2ZSBhcHQtZ2V0IGluc3RhbGwgLXkgZ2gnLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZ2l0KHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIF9hcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZSkge1xuICAgIHJldHVybiBuZXcgSW1hZ2VCdWlsZGVyQ29tcG9uZW50KHNjb3BlLCBpZCwge1xuICAgICAgcGxhdGZvcm06ICdMaW51eCcsXG4gICAgICBkaXNwbGF5TmFtZTogJ0dpdCcsXG4gICAgICBkZXNjcmlwdGlvbjogJ0luc3RhbGwgbGF0ZXN0IHZlcnNpb24gb2YgZ2l0JyxcbiAgICAgIGNvbW1hbmRzOiBbXG4gICAgICAgICdzZXQgLWV4JyxcbiAgICAgICAgJ2FkZC1hcHQtcmVwb3NpdG9yeSBwcGE6Z2l0LWNvcmUvcHBhJyxcbiAgICAgICAgJ2FwdC1nZXQgdXBkYXRlJyxcbiAgICAgICAgJ0RFQklBTl9GUk9OVEVORD1ub25pbnRlcmFjdGl2ZSBhcHQtZ2V0IGluc3RhbGwgLXkgZ2l0JyxcbiAgICAgIF0sXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGdpdGh1YlJ1bm5lcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBydW5uZXJWZXJzaW9uOiBSdW5uZXJWZXJzaW9uLCBhcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZSkge1xuICAgIGxldCB2ZXJzaW9uQ29tbWFuZDogc3RyaW5nO1xuICAgIGlmIChydW5uZXJWZXJzaW9uLmlzKFJ1bm5lclZlcnNpb24ubGF0ZXN0KCkpKSB7XG4gICAgICB2ZXJzaW9uQ29tbWFuZCA9ICdSVU5ORVJfVkVSU0lPTj1gY3VybCAtdyBcIiV7cmVkaXJlY3RfdXJsfVwiIC1mc1MgaHR0cHM6Ly9naXRodWIuY29tL2FjdGlvbnMvcnVubmVyL3JlbGVhc2VzL2xhdGVzdCB8IGdyZXAgLW9FIFwiW14vdl0rJFwiYCc7XG4gICAgfSBlbHNlIHtcbiAgICAgIHZlcnNpb25Db21tYW5kID0gYFJVTk5FUl9WRVJTSU9OPScke3J1bm5lclZlcnNpb24udmVyc2lvbn0nYDtcbiAgICB9XG5cbiAgICBsZXQgYXJjaFVybDtcbiAgICBpZiAoYXJjaGl0ZWN0dXJlLmlzKEFyY2hpdGVjdHVyZS5YODZfNjQpKSB7XG4gICAgICBhcmNoVXJsID0gJ3g2NCc7XG4gICAgfSBlbHNlIGlmIChhcmNoaXRlY3R1cmUuaXMoQXJjaGl0ZWN0dXJlLkFSTTY0KSkge1xuICAgICAgYXJjaFVybCA9ICdhcm02NCc7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5zdXBwb3J0ZWQgYXJjaGl0ZWN0dXJlIGZvciBHaXRIdWIgUnVubmVyOiAke2FyY2hpdGVjdHVyZS5uYW1lfWApO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1hZ2VCdWlsZGVyQ29tcG9uZW50KHNjb3BlLCBpZCwge1xuICAgICAgcGxhdGZvcm06ICdMaW51eCcsXG4gICAgICBkaXNwbGF5TmFtZTogJ0dpdEh1YiBBY3Rpb25zIFJ1bm5lcicsXG4gICAgICBkZXNjcmlwdGlvbjogJ0luc3RhbGwgbGF0ZXN0IHZlcnNpb24gb2YgR2l0SHViIEFjdGlvbnMgUnVubmVyJyxcbiAgICAgIGNvbW1hbmRzOiBbXG4gICAgICAgICdzZXQgLWV4JyxcbiAgICAgICAgdmVyc2lvbkNvbW1hbmQsXG4gICAgICAgIGBjdXJsIC1mc1NMTyBcImh0dHBzOi8vZ2l0aHViLmNvbS9hY3Rpb25zL3J1bm5lci9yZWxlYXNlcy9kb3dubG9hZC92XFwke1JVTk5FUl9WRVJTSU9OfS9hY3Rpb25zLXJ1bm5lci1saW51eC0ke2FyY2hVcmx9LVxcJHtSVU5ORVJfVkVSU0lPTn0udGFyLmd6XCJgLFxuICAgICAgICBgdGFyIHh6ZiBcImFjdGlvbnMtcnVubmVyLWxpbnV4LSR7YXJjaFVybH0tXFwke1JVTk5FUl9WRVJTSU9OfS50YXIuZ3pcImAsXG4gICAgICAgIGBybSBhY3Rpb25zLXJ1bm5lci1saW51eC0ke2FyY2hVcmx9LVxcJHtSVU5ORVJfVkVSU0lPTn0udGFyLmd6YCxcbiAgICAgICAgJy4vYmluL2luc3RhbGxkZXBlbmRlbmNpZXMuc2gnLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZG9ja2VyKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIF9hcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZSkge1xuICAgIHJldHVybiBuZXcgSW1hZ2VCdWlsZGVyQ29tcG9uZW50KHNjb3BlLCBpZCwge1xuICAgICAgcGxhdGZvcm06ICdMaW51eCcsXG4gICAgICBkaXNwbGF5TmFtZTogJ0RvY2tlcicsXG4gICAgICBkZXNjcmlwdGlvbjogJ0luc3RhbGwgbGF0ZXN0IHZlcnNpb24gb2YgRG9ja2VyJyxcbiAgICAgIGNvbW1hbmRzOiBbXG4gICAgICAgICdzZXQgLWV4JyxcbiAgICAgICAgJ2N1cmwgLWZzU0wgaHR0cHM6Ly9kb3dubG9hZC5kb2NrZXIuY29tL2xpbnV4L3VidW50dS9ncGcgfCBzdWRvIGdwZyAtLWRlYXJtb3IgLW8gL3Vzci9zaGFyZS9rZXlyaW5ncy9kb2NrZXIuZ3BnJyxcbiAgICAgICAgJ2VjaG8gJyArXG4gICAgICAgICcgIFwiZGViIFthcmNoPSQoZHBrZyAtLXByaW50LWFyY2hpdGVjdHVyZSkgc2lnbmVkLWJ5PS91c3Ivc2hhcmUva2V5cmluZ3MvZG9ja2VyLmdwZ10gaHR0cHM6Ly9kb3dubG9hZC5kb2NrZXIuY29tL2xpbnV4L3VidW50dSAnICtcbiAgICAgICAgJyAgJChsc2JfcmVsZWFzZSAtY3MpIHN0YWJsZVwiIHwgc3VkbyB0ZWUgL2V0Yy9hcHQvc291cmNlcy5saXN0LmQvZG9ja2VyLmxpc3QgPiAvZGV2L251bGwnLFxuICAgICAgICAnYXB0LWdldCB1cGRhdGUnLFxuICAgICAgICAnREVCSUFOX0ZST05URU5EPW5vbmludGVyYWN0aXZlIGFwdC1nZXQgaW5zdGFsbCAteSBkb2NrZXItY2UgZG9ja2VyLWNlLWNsaSBjb250YWluZXJkLmlvIGRvY2tlci1jb21wb3NlLXBsdWdpbicsXG4gICAgICAgICd1c2VybW9kIC1hRyBkb2NrZXIgcnVubmVyJyxcbiAgICAgICAgJ2xuIC1zIC91c3IvbGliZXhlYy9kb2NrZXIvY2xpLXBsdWdpbnMvZG9ja2VyLWNvbXBvc2UgL3Vzci9iaW4vZG9ja2VyLWNvbXBvc2UnLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZXh0cmFDZXJ0aWZpY2F0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcGF0aDogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG5ldyBJbWFnZUJ1aWxkZXJDb21wb25lbnQoc2NvcGUsIGlkLCB7XG4gICAgICBwbGF0Zm9ybTogJ0xpbnV4JyxcbiAgICAgIGRpc3BsYXlOYW1lOiAnRXh0cmEgY2VydGlmaWNhdGVzJyxcbiAgICAgIGRlc2NyaXB0aW9uOiAnSW5zdGFsbCBzZWxmLXNpZ25lZCBjZXJ0aWZpY2F0ZXMgdG8gcHJvdmlkZSBhY2Nlc3MgdG8gR2l0SHViIEVudGVycHJpc2UgU2VydmVyJyxcbiAgICAgIGNvbW1hbmRzOiBbXG4gICAgICAgICdzZXQgLWV4JyxcbiAgICAgICAgJ2NwIGNlcnRzL2NlcnRzLnBlbSAvdXNyL2xvY2FsL3NoYXJlL2NhLWNlcnRpZmljYXRlcy9naXRodWItZW50ZXJwcmlzZS1zZXJ2ZXIuY3J0JyxcbiAgICAgICAgJ3VwZGF0ZS1jYS1jZXJ0aWZpY2F0ZXMnLFxuICAgICAgXSxcbiAgICAgIGFzc2V0czogW1xuICAgICAgICB7XG4gICAgICAgICAgcGF0aDogJ2NlcnRzJyxcbiAgICAgICAgICBhc3NldDogbmV3IHMzX2Fzc2V0cy5Bc3NldChzY29wZSwgYCR7aWR9IEFzc2V0YCwgeyBwYXRoIH0pLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcbiAgfVxufVxuIl19