"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const common_1 = require("./common");
const linux_components_1 = require("./linux-components");
const windows_components_1 = require("./windows-components");
const delete_ami_function_1 = require("../../lambdas/delete-ami-function");
const utils_1 = require("../../utils");
const common_2 = require("../common");
/**
 * Image builder recipe for Amazon Machine Image (AMI).
 */
class AmiRecipe extends common_1.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        const name = common_1.uniqueImageBuilderName(this);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        let parentAmi;
        let workingDirectory;
        if (props.platform == 'Linux') {
            let archUrl;
            if (props.architecture.is(common_2.Architecture.X86_64)) {
                archUrl = 'amd64';
            }
            else if (props.architecture.is(common_2.Architecture.ARM64)) {
                archUrl = 'arm64';
            }
            else {
                throw new Error(`Unsupported architecture for parent AMI: ${props.architecture.name}`);
            }
            parentAmi = aws_cdk_lib_1.aws_ec2.MachineImage.fromSsmParameter(`/aws/service/canonical/ubuntu/server/focal/stable/current/${archUrl}/hvm/ebs-gp2/ami-id`, {
                os: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX,
            }).getImage(this).imageId;
            workingDirectory = '/home/runner';
        }
        else if (props.platform == 'Windows') {
            parentAmi = aws_cdk_lib_1.aws_ec2.MachineImage.latestWindows(aws_cdk_lib_1.aws_ec2.WindowsVersion.WINDOWS_SERVER_2022_ENGLISH_FULL_CONTAINERSLATEST).getImage(this).imageId;
            workingDirectory = 'C:/'; // must exist or Image Builder fails and must not be empty or git will stall installing from the default windows\system32
        }
        else {
            throw new Error(`Unsupported AMI recipe platform: ${props.platform}`);
        }
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnImageRecipe(this, 'Recipe', {
            name: name,
            version: this.version('ImageRecipe', name, {
                platform: props.platform,
                components,
                parentAmi,
            }),
            parentImage: parentAmi,
            components,
            workingDirectory,
        });
        this.arn = recipe.attrArn;
        this.name = name;
    }
}
/**
 * An AMI builder that uses AWS Image Builder to build AMIs pre-baked with all the GitHub Actions runner requirements. Builders can be used with {@link Ec2Runner}.
 *
 * Each builder re-runs automatically at a set interval to make sure the AMIs contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the AMI used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the EC2 provider, use:
 *
 * ```
 * const builder = new AmiBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * builder.addComponent(new ImageBuilderComponent(scope, id, {
 *   platform: 'Linux',
 *   displayName: 'p7zip',
 *   description: 'Install some more packages',
 *   commands: [
 *     'set -ex',
 *     'apt-get install p7zip',
 *   ],
 * }));
 * new Ec2Runner(this, 'EC2 provider', {
 *     label: 'custom-ec2',
 *     amiBuilder: builder,
 * });
 * ```
 */
class AmiBuilder extends common_1.ImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, {
            os: props?.os,
            supportedOs: [common_2.Os.LINUX, common_2.Os.WINDOWS],
            architecture: props?.architecture,
            supportedArchitectures: [common_2.Architecture.X86_64, common_2.Architecture.ARM64],
            instanceType: props?.instanceType,
            vpc: props?.vpc,
            securityGroups: props?.securityGroup ? [props.securityGroup] : props?.securityGroups,
            subnetSelection: props?.subnetSelection,
            logRemovalPolicy: props?.logRemovalPolicy,
            logRetention: props?.logRetention,
            runnerVersion: props?.runnerVersion,
            rebuildInterval: props?.rebuildInterval,
            imageTypeName: 'AMI',
        });
        // add all basic components
        if (this.os.is(common_2.Os.WINDOWS)) {
            this.addBaseWindowsComponents(props?.installDocker ?? true);
        }
        else if (this.os.is(common_2.Os.LINUX)) {
            this.addBaseLinuxComponents(props?.installDocker ?? true);
        }
    }
    addBaseWindowsComponents(installDocker) {
        this.addComponent(windows_components_1.WindowsComponents.cloudwatchAgent(this, 'CloudWatch agent'));
        this.addComponent(windows_components_1.WindowsComponents.awsCli(this, 'AWS CLI'));
        this.addComponent(windows_components_1.WindowsComponents.githubCli(this, 'GitHub CLI'));
        this.addComponent(windows_components_1.WindowsComponents.git(this, 'git'));
        this.addComponent(windows_components_1.WindowsComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion));
        if (installDocker) {
            this.addComponent(windows_components_1.WindowsComponents.docker(this, 'Docker'));
        }
    }
    addBaseLinuxComponents(installDocker) {
        this.addComponent(linux_components_1.LinuxUbuntuComponents.requiredPackages(this, 'Upgrade packages and install basics', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.runnerUser(this, 'User', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.awsCli(this, 'AWS CLI', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.githubCli(this, 'GitHub CLI', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.git(this, 'git', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion, this.architecture));
        if (installDocker) {
            this.addComponent(linux_components_1.LinuxUbuntuComponents.docker(this, 'Docker', this.architecture));
        }
    }
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component) {
        if (this.boundAmi) {
            throw new Error('AMI is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components = [component].concat(this.components);
    }
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component) {
        if (this.boundAmi) {
            throw new Error('AMI is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components.push(component);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        if (this.platform == 'Linux') {
            this.prependComponent(linux_components_1.LinuxUbuntuComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else if (this.platform == 'Windows') {
            this.prependComponent(windows_components_1.WindowsComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else {
            throw new Error(`Unknown platform: ${this.platform}`);
        }
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the AMI builder.
     */
    bind() {
        if (this.boundAmi) {
            return this.boundAmi;
        }
        const launchTemplate = new aws_cdk_lib_1.aws_ec2.LaunchTemplate(this, 'Launch template');
        const stackName = cdk.Stack.of(this).stackName;
        const builderName = this.node.path;
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Distribution', {
            name: common_1.uniqueImageBuilderName(this),
            description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    amiDistributionConfiguration: {
                        Name: `${cdk.Names.uniqueResourceName(this, {
                            maxLength: 100,
                            separator: '-',
                            allowedSpecialCharacters: '_-',
                        })}-{{ imagebuilder:buildDate }}`,
                        AmiTags: {
                            'Name': this.node.id,
                            'GitHubRunners:Stack': stackName,
                            'GitHubRunners:Builder': builderName,
                        },
                    },
                    launchTemplateConfigurations: [
                        {
                            launchTemplateId: launchTemplate.launchTemplateId,
                        },
                    ],
                },
            ],
        });
        const recipe = new AmiRecipe(this, 'Ami Recipe', {
            platform: this.platform,
            components: this.components,
            architecture: this.architecture,
        });
        const log = this.createLog(recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
        ]);
        this.createImage(infra, dist, log, recipe.arn, undefined);
        this.createPipeline(infra, dist, log, recipe.arn, undefined);
        this.boundAmi = {
            launchTemplate: launchTemplate,
            architecture: this.architecture,
            os: this.os,
            logGroup: log,
            runnerVersion: this.runnerVersion,
        };
        this.imageCleaner(launchTemplate, stackName, builderName);
        return this.boundAmi;
    }
    imageCleaner(launchTemplate, stackName, builderName) {
        const deleter = utils_1.singletonLambda(delete_ami_function_1.DeleteAmiFunction, this, 'delete-ami', {
            description: 'Delete old GitHub Runner AMIs',
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ec2:DescribeLaunchTemplateVersions', 'ec2:DescribeImages', 'ec2:DeregisterImage', 'ec2:DeleteSnapshot'],
                    resources: ['*'],
                }),
            ],
            timeout: cdk.Duration.minutes(5),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        // delete old AMIs on schedule
        const eventRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Delete AMI Schedule', {
            schedule: aws_cdk_lib_1.aws_events.Schedule.rate(cdk.Duration.days(1)),
            description: `Delete old AMIs for ${builderName}`,
        });
        eventRule.addTarget(new aws_cdk_lib_1.aws_events_targets.LambdaFunction(deleter, {
            event: aws_cdk_lib_1.aws_events.RuleTargetInput.fromObject({
                RequestType: 'Scheduled',
                LaunchTemplateId: launchTemplate.launchTemplateId,
                StackName: stackName,
                BuilderName: builderName,
            }),
        }));
        // delete all AMIs when this construct is removed
        new aws_cdk_lib_1.CustomResource(this, 'AMI Deleter', {
            serviceToken: deleter.functionArn,
            resourceType: 'Custom::AmiDeleter',
            properties: {
                StackName: stackName,
                BuilderName: builderName,
            },
        });
    }
}
exports.AmiBuilder = AmiBuilder;
_a = JSII_RTTI_SYMBOL_1;
AmiBuilder[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.AmiBuilder", version: "0.8.4" };
//# sourceMappingURL=data:application/json;base64,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