"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseProvider = exports.Os = exports.Architecture = exports.RunnerVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * Defines desired GitHub Actions runner version.
 */
class RunnerVersion {
    constructor(version) {
        this.version = version;
    }
    /**
     * Use the latest version available at the time the runner provider image is built.
     */
    static latest() {
        return new RunnerVersion('latest');
    }
    /**
     * Use a specific version.
     *
     * @see https://github.com/actions/runner/releases
     *
     * @param version GitHub Runner version
     */
    static specific(version) {
        return new RunnerVersion(version);
    }
    /**
     * Check if two versions are the same.
     *
     * @param other version to compare
     */
    is(other) {
        return this.version == other.version;
    }
}
exports.RunnerVersion = RunnerVersion;
_a = JSII_RTTI_SYMBOL_1;
RunnerVersion[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.RunnerVersion", version: "0.8.4" };
/**
 * CPU architecture enum for an image.
 */
class Architecture {
    constructor(name) {
        this.name = name;
    }
    static of(architecture) {
        return new Architecture(architecture);
    }
    /**
    * Checks if the given architecture is the same as this one.
    *
    * @param arch architecture to compare
    */
    is(arch) {
        return arch.name == this.name;
    }
    /**
     * Checks if this architecture is in a given list.
     *
     * @param arches architectures to check
     */
    isIn(arches) {
        for (const arch of arches) {
            if (this.is(arch)) {
                return true;
            }
        }
        return false;
    }
    /**
     * Checks if a given EC2 instance type matches this architecture.
     *
     * @param instanceType instance type to check
     */
    instanceTypeMatch(instanceType) {
        if (instanceType.architecture == aws_cdk_lib_1.aws_ec2.InstanceArchitecture.X86_64) {
            return this.is(Architecture.X86_64);
        }
        if (instanceType.architecture == aws_cdk_lib_1.aws_ec2.InstanceArchitecture.ARM_64) {
            return this.is(Architecture.ARM64);
        }
        throw new Error('Unknown instance type architecture');
    }
}
exports.Architecture = Architecture;
_b = JSII_RTTI_SYMBOL_1;
Architecture[_b] = { fqn: "@cloudsnorkel/cdk-github-runners.Architecture", version: "0.8.4" };
/**
 * ARM64
 */
Architecture.ARM64 = Architecture.of('ARM64');
/**
 * X86_64
 */
Architecture.X86_64 = Architecture.of('X86_64');
/**
 * OS enum for an image.
 */
class Os {
    constructor(name) {
        this.name = name;
    }
    static of(os) {
        return new Os(os);
    }
    /**
    * Checks if the given OS is the same as this one.
    *
    * @param os OS to compare
    */
    is(os) {
        return os.name == this.name;
    }
    /**
     * Checks if this OS is in a given list.
     *
     * @param oses list of OS to check
     */
    isIn(oses) {
        for (const os of oses) {
            if (this.is(os)) {
                return true;
            }
        }
        return false;
    }
}
exports.Os = Os;
_c = JSII_RTTI_SYMBOL_1;
Os[_c] = { fqn: "@cloudsnorkel/cdk-github-runners.Os", version: "0.8.4" };
/**
* Linux
*/
Os.LINUX = Os.of('Linux');
/**
* Windows
*/
Os.WINDOWS = Os.of('Windows');
/**
 * Base class for all providers with common methods used by all providers.
 *
 * @internal
 */
class BaseProvider extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.retryOptions = props?.retryOptions;
    }
    labelsFromProperties(defaultLabel, propsLabel, propsLabels) {
        if (propsLabels && propsLabel) {
            throw new Error('Must supply either `label` or `labels` in runner properties, but not both. Try removing the `label` property.');
        }
        if (propsLabels) {
            return propsLabels;
        }
        if (propsLabel) {
            return [propsLabel];
        }
        return [defaultLabel];
    }
    addRetry(task, errors) {
        if (this.retryOptions?.retry ?? true) {
            task.addRetry({
                errors,
                interval: this.retryOptions?.interval ?? aws_cdk_lib_1.Duration.minutes(1),
                maxAttempts: this.retryOptions?.maxAttempts ?? 10,
                backoffRate: this.retryOptions?.backoffRate ?? 1.3,
            });
        }
    }
}
exports.BaseProvider = BaseProvider;
//# sourceMappingURL=data:application/json;base64,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