"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const github_1 = require("./github");
async function getRunnerId(octokit, owner, repo, name, idleOnly) {
    let page = 1;
    while (true) {
        const runners = await octokit.request('GET /repos/{owner}/{repo}/actions/runners?per_page=100&page={page}', {
            page: page,
            owner: owner,
            repo: repo,
        });
        if (runners.data.runners.length == 0) {
            return;
        }
        for (const runner of runners.data.runners) {
            if (runner.name == name) {
                if (idleOnly) {
                    if (!runner.busy) {
                        return runner.id;
                    }
                    else {
                        console.log('Runner is busy, no need to delete.');
                        return;
                    }
                }
                return runner.id;
            }
        }
        page++;
    }
}
class RunnerBusy extends Error {
    constructor(msg) {
        super(msg);
        this.name = 'RunnerBusy';
        Object.setPrototypeOf(this, RunnerBusy.prototype);
    }
}
exports.handler = async function (event) {
    const { octokit } = await github_1.getOctokit(event.installationId);
    // find runner id
    const runnerId = await getRunnerId(octokit, event.owner, event.repo, event.runnerName, event.idleOnly);
    if (!runnerId) {
        console.error(`Unable to find runner id for ${event.owner}/${event.repo}:${event.runnerName}`);
        return;
    }
    console.log(`Runner ${event.runnerName} has id #${runnerId}`);
    // delete runner (it usually gets deleted by ./run.sh, but it stopped prematurely if we're here).
    // it seems like runners are automatically removed after a timeout, if they first accepted a job.
    // we try removing it anyway for cases where a job wasn't accepted, and just in case it wasn't removed.
    // repos have a limited number of self-hosted runners, so we can't leave dead ones behind.
    try {
        await octokit.rest.actions.deleteSelfHostedRunnerFromRepo({
            owner: event.owner,
            repo: event.repo,
            runner_id: runnerId,
        });
    }
    catch (e) {
        const reqError = e;
        if (reqError.message.includes('is still running a job')) {
            throw new RunnerBusy(reqError.message);
        }
        else {
            throw e;
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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