import pathlib

import pytest

import pystv
from pystv import RaceInfo
from pystv import RaceMetadata as RM

PATH = pathlib.Path(__file__).parent

PARSE_HEADER_DATA = [
    (["Q0 [A00]"], [(RM("Q0", 1, ["A00"]), slice(0, 1))]),
    (["T", "Q0 [A00]", "Q0 [A01]"], [(RM("Q0", 1, ["A00", "A01"]), slice(1, 3))]),
    (["Q0 [A00]", "Q0 [A01]"], [(RM("Q0", 1, ["A00", "A01"]), slice(0, 2))]),
    (
        ["Q0 [A00]", "Q0 [A01]", "Q1 [A10]", "Q1 [A11]"],
        [
            (RM("Q0", 1, ["A00", "A01"]), slice(0, 2)),
            (RM("Q1", 1, ["A10", "A11"]), slice(2, 4)),
        ],
    ),
    (
        ["T", "Q0 [A00]", "Q0 [A01]", "T", "Q1 [A10]", "Q1 [A11]", "T"],
        [
            (RM("Q0", 1, ["A00", "A01"]), slice(1, 3)),
            (RM("Q1", 1, ["A10", "A11"]), slice(4, 6)),
        ],
    ),
    (
        ["Q0 (2 winners) [A00]", "Q0 (2 winners) [A01]", "Q0 (2 winners) [A02]"],
        [(RM("Q0", 2, ["A00", "A01", "A02"]), slice(0, 3))],
    ),
    (["Q0 [A00] "], []),
    (["Q0[A00]"], []),
    (["Q0  [A00]"], [(RM("Q0", 1, ["A00"]), slice(0, 1))]),
    (["Q0 (2  winners) [A00]"], [(RM("Q0", 2, ["A00"]), slice(0, 1))]),
    (["Q0  (2 winners) [A00]"], [(RM("Q0", 2, ["A00"]), slice(0, 1))]),
    (["Q0  (2 winners)  [A00]"], [(RM("Q0", 2, ["A00"]), slice(0, 1))]),
]


@pytest.mark.parametrize("header,expected", PARSE_HEADER_DATA)
def test_parse_header(header, expected):
    assert pystv.parse_header(header) == expected


def test_google_form_csv():
    actual = pystv.parse_google_form_csv(PATH / "transform_testdata.csv")

    metadata = RM(
        "What is your favorite season?", 1, ["Spring", "Summer", "Autumn", "Winter"]
    )
    expected = [
        RaceInfo(
            metadata,
            [
                [1, 0, 0, 0],
                [1, 3, 2, 4],
                [2, 0, 0, 0],
                [2, 3, 4, 1],
                [3, 1, 4, 2],
                [3, 2, 4, 1],
            ],
            [1, 1, 1, 2, 1, 2],
        )
    ]

    assert actual == expected


def test_google_form_csv_weighted():
    actual = pystv.parse_google_form_csv(PATH / "transform_weighted_testdata.csv")

    metadata = RM(
        "What is your favorite season?", 1, ["Spring", "Summer", "Autumn", "Winter"]
    )
    expected = [
        RaceInfo(
            metadata,
            [
                [1, 0, 0, 0],
                [1, 3, 2, 4],
                [2, 0, 0, 0],
                [2, 3, 4, 1],
                [3, 1, 4, 2],
                [3, 2, 4, 1],
            ],
            [3, 1, 2, 3, 4, 3],
        )
    ]

    assert actual == expected
