use std::collections::HashMap;

use alloy::rpc::types::Filter;
use alloy::signers::{local::PrivateKeySigner, Signer};
use alloy_primitives::{Address, FixedBytes, U256};
use alloy_provider::{Provider, ProviderBuilder};
use alloy_sol_types::{sol, SolEvent};
pub mod config;
pub mod proxy;
use tracing::info;
pub use CollateralUpgradeable::{Deposit, Reclaimed, Slashed};

#[cfg(test)]
mod tests;

use config::{CollateralNetworkConfig, MAX_BLOCKS_PER_SCAN};

sol!(
    #[allow(missing_docs)]
    #[sol(
        rpc,
        bytecode = "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"
    )]
    CollateralUpgradeable,
    "./src/CollateralUpgradableABI.json"
);

#[derive(Debug, Clone)]
pub struct Reclaim {
    pub hotkey: [u8; 32],
    pub node_id: [u8; 16],
    pub miner: Address,
    pub amount: U256,
    pub deny_timeout: u64,
}

impl From<(FixedBytes<32>, FixedBytes<16>, Address, U256, u64)> for Reclaim {
    fn from(tuple: (FixedBytes<32>, FixedBytes<16>, Address, U256, u64)) -> Self {
        Self {
            hotkey: {
                let mut hk = [0u8; 32];
                hk.copy_from_slice(tuple.0.as_slice());
                hk
            },
            node_id: {
                let mut b = [0u8; 16];
                b.copy_from_slice(tuple.1.as_slice());
                b
            },
            miner: tuple.2,
            amount: tuple.3,
            deny_timeout: tuple.4,
        }
    }
}

pub enum CollateralEvent {
    Deposit(CollateralUpgradeable::Deposit),
    Reclaimed(CollateralUpgradeable::Reclaimed),
    Slashed(CollateralUpgradeable::Slashed),
}

// get the collateral contract instance with custom network config
pub async fn get_collateral(
    private_key: &str,
    network_config: &CollateralNetworkConfig,
) -> Result<
    CollateralUpgradeable::CollateralUpgradeableInstance<impl alloy_provider::Provider>,
    anyhow::Error,
> {
    let mut signer: PrivateKeySigner = private_key.parse()?;
    signer.set_chain_id(Some(network_config.chain_id));

    let provider = ProviderBuilder::new()
        .wallet(signer)
        .connect(&network_config.rpc_url)
        .await?;

    let proxied = CollateralUpgradeable::new(network_config.contract_address, provider);

    Ok(proxied)
}

pub async fn scan_events(
    from_block: u64,
    network_config: &CollateralNetworkConfig,
) -> Result<(u64, HashMap<u64, Vec<CollateralEvent>>), anyhow::Error> {
    let provider = ProviderBuilder::new()
        .connect(&network_config.rpc_url)
        .await?;
    let current_block = provider.get_block_number().await?.saturating_sub(1);

    if from_block > current_block {
        return Err(anyhow::anyhow!(
            "from_block must be less than current_block"
        ));
    }

    let mut to_block = from_block + MAX_BLOCKS_PER_SCAN;

    if to_block > current_block {
        to_block = current_block;
    }
    scan_events_with_scope(from_block, to_block, network_config).await
}

pub async fn scan_events_with_scope(
    from_block: u64,
    to_block: u64,
    network_config: &CollateralNetworkConfig,
) -> Result<(u64, HashMap<u64, Vec<CollateralEvent>>), anyhow::Error> {
    let provider = ProviderBuilder::new()
        .connect(&network_config.rpc_url)
        .await?;

    let filter = Filter::new()
        .address(network_config.contract_address)
        .from_block(from_block)
        .to_block(to_block);

    let logs = provider.get_logs(&filter).await?;

    let mut result: HashMap<u64, Vec<CollateralEvent>> = HashMap::new();

    for log in logs {
        if log.removed {
            continue;
        }

        let topics = log.inner.topics();
        let topic0 = topics.first();
        let block_number = log
            .block_number
            .ok_or(anyhow::anyhow!("Block number not available in event"))?;

        let block_result = result.get_mut(&block_number);

        let event = match topic0 {
            Some(sig) if sig == &CollateralUpgradeable::Deposit::SIGNATURE_HASH => {
                let deposit = CollateralUpgradeable::Deposit::decode_raw_log(
                    topics,
                    log.data().data.as_ref(),
                )?;
                Some(CollateralEvent::Deposit(deposit))
            }
            Some(sig) if sig == &CollateralUpgradeable::Reclaimed::SIGNATURE_HASH => {
                let reclaimed = CollateralUpgradeable::Reclaimed::decode_raw_log(
                    topics,
                    log.data().data.as_ref(),
                )?;
                Some(CollateralEvent::Reclaimed(reclaimed))
            }
            Some(sig) if sig == &CollateralUpgradeable::Slashed::SIGNATURE_HASH => {
                let slashed = CollateralUpgradeable::Slashed::decode_raw_log(
                    topics,
                    log.data().data.as_ref(),
                )?;
                Some(CollateralEvent::Slashed(slashed))
            }
            _ => None,
        };

        if let Some(event) = event {
            match block_result {
                Some(events) => {
                    events.push(event);
                }
                None => {
                    result.insert(block_number, vec![event]);
                }
            }
        }
    }

    info!(
        "Scanned blocks {} to {}, {} events are found",
        from_block,
        to_block,
        result.values().map(|v| v.len()).sum::<usize>()
    );
    Ok((to_block, result))
}
// transactions
pub async fn deposit(
    private_key: &str,
    hotkey: [u8; 32],
    node_id: [u8; 16],
    amount: U256,
    network_config: &CollateralNetworkConfig,
) -> Result<(), anyhow::Error> {
    let contract = get_collateral(private_key, network_config).await?;

    let tx = contract
        .deposit(
            FixedBytes::from_slice(&hotkey),
            FixedBytes::from_slice(&node_id),
        )
        .value(amount);
    let tx = tx.send().await?;
    let receipt = tx.get_receipt().await?;
    tracing::info!("{receipt:?}");
    Ok(())
}

pub async fn deposit_with_config(
    private_key: &str,
    hotkey: [u8; 32],
    node_id: [u8; 16],
    amount: U256,
    network_config: &CollateralNetworkConfig,
) -> Result<(), anyhow::Error> {
    let contract = get_collateral(private_key, network_config).await?;

    let tx = contract
        .deposit(
            FixedBytes::from_slice(&hotkey),
            FixedBytes::from_slice(&node_id),
        )
        .value(amount);
    let tx = tx.send().await?;
    let receipt = tx.get_receipt().await?;
    tracing::info!("{receipt:?}");
    Ok(())
}

pub async fn reclaim_collateral(
    private_key: &str,
    hotkey: [u8; 32],
    node_id: [u8; 16],
    url: &str,
    url_content_md5_checksum: u128,
    network_config: &CollateralNetworkConfig,
) -> Result<(), anyhow::Error> {
    let contract = get_collateral(private_key, network_config).await?;

    let tx = contract.reclaimCollateral(
        FixedBytes::from_slice(&hotkey),
        FixedBytes::from_slice(&node_id),
        url.to_string(),
        FixedBytes::from_slice(&url_content_md5_checksum.to_be_bytes()),
    );
    let tx = tx.send().await?;
    tx.get_receipt().await?;
    Ok(())
}

pub async fn finalize_reclaim(
    private_key: &str,
    reclaim_request_id: U256,
    network_config: &CollateralNetworkConfig,
) -> Result<(), anyhow::Error> {
    let contract = get_collateral(private_key, network_config).await?;

    let tx = contract.finalizeReclaim(reclaim_request_id);
    let tx = tx.send().await?;
    tx.get_receipt().await?;
    Ok(())
}

pub async fn deny_reclaim(
    private_key: &str,
    reclaim_request_id: U256,
    url: &str,
    url_content_md5_checksum: u128,
    network_config: &CollateralNetworkConfig,
) -> Result<(), anyhow::Error> {
    let contract = get_collateral(private_key, network_config).await?;

    let tx = contract.denyReclaimRequest(
        reclaim_request_id,
        url.to_string(),
        FixedBytes::from_slice(&url_content_md5_checksum.to_be_bytes()),
    );
    let tx = tx.send().await?;
    tx.get_receipt().await?;
    Ok(())
}

pub async fn slash_collateral(
    private_key: &str,
    hotkey: [u8; 32],
    node_id: [u8; 16],
    url: &str,
    url_content_md5_checksum: u128,
    network_config: &CollateralNetworkConfig,
) -> Result<(), anyhow::Error> {
    let contract = get_collateral(private_key, network_config).await?;

    let tx = contract.slashCollateral(
        FixedBytes::from_slice(&hotkey),
        FixedBytes::from_slice(&node_id),
        url.to_string(),
        FixedBytes::from_slice(&url_content_md5_checksum.to_be_bytes()),
    );
    let tx = tx.send().await?;
    tx.get_receipt().await?;
    Ok(())
}

// Get methods

pub async fn netuid(network_config: &CollateralNetworkConfig) -> Result<u16, anyhow::Error> {
    let provider = ProviderBuilder::new()
        .connect(&network_config.rpc_url)
        .await?;
    let contract = CollateralUpgradeable::new(network_config.contract_address, provider);
    let netuid = contract.NETUID().call().await?;
    Ok(netuid)
}

pub async fn trustee(network_config: &CollateralNetworkConfig) -> Result<Address, anyhow::Error> {
    let provider = ProviderBuilder::new()
        .connect(&network_config.rpc_url)
        .await?;
    let contract = CollateralUpgradeable::new(network_config.contract_address, provider);
    let trustee = contract.TRUSTEE().call().await?;
    Ok(trustee)
}

pub async fn decision_timeout(
    network_config: &CollateralNetworkConfig,
) -> Result<u64, anyhow::Error> {
    let provider = ProviderBuilder::new()
        .connect(&network_config.rpc_url)
        .await?;
    let contract = CollateralUpgradeable::new(network_config.contract_address, provider);
    let decision_timeout = contract.DECISION_TIMEOUT().call().await?;
    Ok(decision_timeout)
}

pub async fn min_collateral_increase(
    network_config: &CollateralNetworkConfig,
) -> Result<U256, anyhow::Error> {
    let provider = ProviderBuilder::new()
        .connect(&network_config.rpc_url)
        .await?;
    let contract = CollateralUpgradeable::new(network_config.contract_address, provider);
    let min_collateral_increase = contract.MIN_COLLATERAL_INCREASE().call().await?;
    Ok(min_collateral_increase)
}

pub async fn node_to_miner(
    hotkey: [u8; 32],
    node_id: [u8; 16],
    network_config: &CollateralNetworkConfig,
) -> Result<Address, anyhow::Error> {
    let provider = ProviderBuilder::new()
        .connect(&network_config.rpc_url)
        .await?;
    let contract = CollateralUpgradeable::new(network_config.contract_address, provider);
    // let node_bytes = node_id.to_be_bytes();
    let node_to_miner = contract
        .nodeToMiner(
            FixedBytes::from_slice(&hotkey),
            FixedBytes::from_slice(&node_id),
        )
        .call()
        .await?;
    Ok(node_to_miner)
}

pub async fn collaterals(
    hotkey: [u8; 32],
    node_id: [u8; 16],
    network_config: &CollateralNetworkConfig,
) -> Result<U256, anyhow::Error> {
    let provider = ProviderBuilder::new()
        .connect(&network_config.rpc_url)
        .await?;
    let contract = CollateralUpgradeable::new(network_config.contract_address, provider);
    let collaterals = contract
        .collaterals(
            FixedBytes::from_slice(&hotkey),
            FixedBytes::from_slice(&node_id),
        )
        .call()
        .await?;
    Ok(collaterals)
}

pub async fn reclaims(
    reclaim_request_id: U256,
    network_config: &CollateralNetworkConfig,
) -> Result<Reclaim, anyhow::Error> {
    let provider = ProviderBuilder::new()
        .connect(&network_config.rpc_url)
        .await?;
    let contract = CollateralUpgradeable::new(network_config.contract_address, provider);
    let result = contract.reclaims(reclaim_request_id).call().await?;
    let reclaim = Reclaim::from((
        result.hotkey,
        result.nodeId,
        result.miner,
        result.amount,
        result.denyTimeout,
    ));
    Ok(reclaim)
}
