//! Python type exposures for Basilica SDK responses
//!
//! This module provides PyO3 bindings for response types, enabling
//! direct attribute access with full IDE autocomplete support.

// pyo3-stub-gen uses deprecated PyO3 APIs internally, we need to allow them
#![cfg_attr(feature = "stub-gen", allow(deprecated))]

use basilica_sdk::types::{
    AvailabilityInfo as SdkAvailabilityInfo, AvailableNode as SdkAvailableNode,
    CpuSpec as SdkCpuSpec, GpuRequirements as SdkGpuRequirements, GpuSpec as SdkGpuSpec,
    ListAvailableNodesQuery as SdkListAvailableNodesQuery, ListRentalsQuery as SdkListRentalsQuery,
    NodeDetails as SdkNodeDetails, NodeSelection as SdkNodeSelection,
    PortMappingRequest as SdkPortMappingRequest, RentalState, RentalStatus as SdkRentalStatus,
    RentalStatusWithSshResponse as SdkRentalStatusWithSshResponse,
    ResourceRequirementsRequest as SdkResourceRequirementsRequest, SshAccess as SdkSshAccess,
    StartRentalApiRequest as SdkStartRentalApiRequest, VolumeMountRequest as SdkVolumeMountRequest,
};
use basilica_validator::rental::RentalResponse as SdkRentalResponse;
use pyo3::prelude::*;
#[cfg(feature = "stub-gen")]
use pyo3_stub_gen_derive::{gen_stub_pyclass, gen_stub_pyclass_enum, gen_stub_pymethods};
use std::collections::HashMap;

/// SSH access information for a rental
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct SshAccess {
    #[pyo3(get)]
    pub host: String,
    #[pyo3(get)]
    pub port: u16,
    #[pyo3(get)]
    pub user: String,
}

impl From<SdkSshAccess> for SshAccess {
    fn from(ssh: SdkSshAccess) -> Self {
        Self {
            host: ssh.host,
            port: ssh.port,
            user: ssh.username,
        }
    }
}

/// GPU specification details
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct GpuSpec {
    #[pyo3(get)]
    pub name: String,
    #[pyo3(get)]
    pub memory_gb: u32,
    #[pyo3(get)]
    pub compute_capability: String,
}

impl From<SdkGpuSpec> for GpuSpec {
    fn from(spec: SdkGpuSpec) -> Self {
        Self {
            name: spec.name,
            memory_gb: spec.memory_gb,
            compute_capability: spec.compute_capability,
        }
    }
}

/// CPU specification details
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct CpuSpec {
    #[pyo3(get)]
    pub cores: u32,
    #[pyo3(get)]
    pub model: String,
    #[pyo3(get)]
    pub memory_gb: u32,
}

impl From<SdkCpuSpec> for CpuSpec {
    fn from(spec: SdkCpuSpec) -> Self {
        Self {
            cores: spec.cores,
            model: spec.model,
            memory_gb: spec.memory_gb,
        }
    }
}

/// Node details including GPU and CPU specifications
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct NodeDetails {
    #[pyo3(get)]
    pub id: String,
    #[pyo3(get)]
    pub gpu_specs: Vec<GpuSpec>,
    #[pyo3(get)]
    pub cpu_specs: CpuSpec,
    #[pyo3(get)]
    pub location: Option<String>,
}

impl From<SdkNodeDetails> for NodeDetails {
    fn from(details: SdkNodeDetails) -> Self {
        Self {
            id: details.id,
            gpu_specs: details.gpu_specs.into_iter().map(Into::into).collect(),
            cpu_specs: details.cpu_specs.into(),
            location: details.location,
        }
    }
}

/// Rental status enumeration
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct RentalStatus {
    #[pyo3(get)]
    pub state: String,
    #[pyo3(get)]
    pub message: Option<String>,
}

impl From<SdkRentalStatus> for RentalStatus {
    fn from(status: SdkRentalStatus) -> Self {
        let state = match status {
            SdkRentalStatus::Pending => "Pending",
            SdkRentalStatus::Active => "Active",
            SdkRentalStatus::Terminated => "Terminated",
            SdkRentalStatus::Failed => "Failed",
        };

        Self {
            state: state.to_string(),
            message: None,
        }
    }
}

/// Response from starting a rental
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct RentalResponse {
    #[pyo3(get)]
    pub rental_id: String,
    #[pyo3(get)]
    pub ssh_credentials: Option<String>,
    #[pyo3(get)]
    pub container_id: String,
    #[pyo3(get)]
    pub container_name: String,
    #[pyo3(get)]
    pub status: String,
}

impl From<SdkRentalResponse> for RentalResponse {
    fn from(response: SdkRentalResponse) -> Self {
        Self {
            rental_id: response.rental_id,
            ssh_credentials: response.ssh_credentials,
            container_id: response.container_info.container_id,
            container_name: response.container_info.container_name,
            status: response.container_info.status,
        }
    }
}

/// Full rental status response with SSH credentials (matches API response)
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct RentalStatusWithSshResponse {
    #[pyo3(get)]
    pub rental_id: String,
    #[pyo3(get)]
    pub status: RentalStatus,
    #[pyo3(get)]
    pub node: NodeDetails,
    #[pyo3(get)]
    pub ssh_credentials: Option<String>,
    #[pyo3(get)]
    pub created_at: String,
    #[pyo3(get)]
    pub updated_at: String,
}

impl From<SdkRentalStatusWithSshResponse> for RentalStatusWithSshResponse {
    fn from(response: SdkRentalStatusWithSshResponse) -> Self {
        Self {
            rental_id: response.rental_id,
            status: response.status.into(),
            node: response.node.into(),
            ssh_credentials: response.ssh_credentials,
            created_at: response.created_at.to_rfc3339(),
            updated_at: response.updated_at.to_rfc3339(),
        }
    }
}

/// Availability information for an node
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct AvailabilityInfo {
    #[pyo3(get)]
    pub available_until: Option<String>,
    #[pyo3(get)]
    pub verification_score: f64,
    #[pyo3(get)]
    pub uptime_percentage: f64,
}

impl From<SdkAvailabilityInfo> for AvailabilityInfo {
    fn from(info: SdkAvailabilityInfo) -> Self {
        Self {
            available_until: info.available_until.map(|dt| dt.to_rfc3339()),
            verification_score: info.verification_score,
            uptime_percentage: info.uptime_percentage,
        }
    }
}

/// Available node with details and availability info
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct AvailableNode {
    #[pyo3(get)]
    pub node: NodeDetails,
    #[pyo3(get)]
    pub availability: AvailabilityInfo,
}

impl From<SdkAvailableNode> for AvailableNode {
    fn from(node: SdkAvailableNode) -> Self {
        Self {
            node: node.node.into(),
            availability: node.availability.into(),
        }
    }
}

/// Health check response
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct HealthCheckResponse {
    #[pyo3(get)]
    pub status: String,
    #[pyo3(get)]
    pub version: String,
    #[pyo3(get)]
    pub timestamp: String,
    #[pyo3(get)]
    pub healthy_validators: usize,
    #[pyo3(get)]
    pub total_validators: usize,
}

impl From<basilica_sdk::types::HealthCheckResponse> for HealthCheckResponse {
    fn from(response: basilica_sdk::types::HealthCheckResponse) -> Self {
        Self {
            status: response.status,
            version: response.version,
            timestamp: response.timestamp.to_rfc3339(),
            healthy_validators: response.healthy_validators,
            total_validators: response.total_validators,
        }
    }
}

// Request types for Python bindings

/// GPU requirements for node selection
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct GpuRequirements {
    #[pyo3(get, set)]
    pub gpu_count: u32,
    #[pyo3(get, set)]
    pub gpu_type: Option<String>,
    #[pyo3(get, set)]
    pub min_memory_gb: u32,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl GpuRequirements {
    #[new]
    #[pyo3(signature = (gpu_count, min_memory_gb, gpu_type=None))]
    fn new(gpu_count: u32, min_memory_gb: u32, gpu_type: Option<String>) -> Self {
        Self {
            gpu_count,
            gpu_type,
            min_memory_gb,
        }
    }
}

impl From<GpuRequirements> for SdkGpuRequirements {
    fn from(req: GpuRequirements) -> Self {
        Self {
            gpu_count: req.gpu_count,
            gpu_type: req.gpu_type,
            min_memory_gb: req.min_memory_gb,
        }
    }
}

/// Node selection strategy
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass_enum)]
#[pyclass]
#[derive(Clone)]
pub enum NodeSelection {
    NodeId { node_id: String },
    ExactGpuConfiguration { gpu_requirements: GpuRequirements },
}

impl From<NodeSelection> for SdkNodeSelection {
    fn from(selection: NodeSelection) -> Self {
        match selection {
            NodeSelection::NodeId { node_id } => SdkNodeSelection::NodeId { node_id },
            NodeSelection::ExactGpuConfiguration { gpu_requirements } => {
                SdkNodeSelection::ExactGpuConfiguration {
                    gpu_requirements: gpu_requirements.into(),
                }
            }
        }
    }
}

/// Port mapping request
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct PortMappingRequest {
    #[pyo3(get, set)]
    pub container_port: u32,
    #[pyo3(get, set)]
    pub host_port: u32,
    #[pyo3(get, set)]
    pub protocol: String,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl PortMappingRequest {
    #[new]
    #[pyo3(signature = (container_port, host_port, protocol=None))]
    fn new(container_port: u32, host_port: u32, protocol: Option<String>) -> Self {
        Self {
            container_port,
            host_port,
            protocol: protocol.unwrap_or_else(|| "tcp".to_string()),
        }
    }
}

impl From<PortMappingRequest> for SdkPortMappingRequest {
    fn from(port: PortMappingRequest) -> Self {
        Self {
            container_port: port.container_port,
            host_port: port.host_port,
            protocol: port.protocol,
        }
    }
}

/// Resource requirements request
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct ResourceRequirementsRequest {
    #[pyo3(get, set)]
    pub cpu_cores: f64,
    #[pyo3(get, set)]
    pub memory_mb: i64,
    #[pyo3(get, set)]
    pub storage_mb: i64,
    #[pyo3(get, set)]
    pub gpu_count: u32,
    #[pyo3(get, set)]
    pub gpu_types: Vec<String>,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl ResourceRequirementsRequest {
    #[new]
    #[pyo3(signature = (cpu_cores=1.0, memory_mb=1024, storage_mb=10240, gpu_count=0, gpu_types=None))]
    fn new(
        cpu_cores: f64,
        memory_mb: i64,
        storage_mb: i64,
        gpu_count: u32,
        gpu_types: Option<Vec<String>>,
    ) -> Self {
        Self {
            cpu_cores,
            memory_mb,
            storage_mb,
            gpu_count,
            gpu_types: gpu_types.unwrap_or_default(),
        }
    }
}

impl From<ResourceRequirementsRequest> for SdkResourceRequirementsRequest {
    fn from(req: ResourceRequirementsRequest) -> Self {
        Self {
            cpu_cores: req.cpu_cores,
            memory_mb: req.memory_mb,
            storage_mb: req.storage_mb,
            gpu_count: req.gpu_count,
            gpu_types: req.gpu_types,
        }
    }
}

impl Default for ResourceRequirementsRequest {
    fn default() -> Self {
        Self {
            cpu_cores: 1.0,
            memory_mb: 1024,
            storage_mb: 10240,
            gpu_count: 0,
            gpu_types: vec![],
        }
    }
}

/// Volume mount request
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct VolumeMountRequest {
    #[pyo3(get, set)]
    pub host_path: String,
    #[pyo3(get, set)]
    pub container_path: String,
    #[pyo3(get, set)]
    pub read_only: bool,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl VolumeMountRequest {
    #[new]
    #[pyo3(signature = (host_path, container_path, read_only=false))]
    fn new(host_path: String, container_path: String, read_only: bool) -> Self {
        Self {
            host_path,
            container_path,
            read_only,
        }
    }
}

impl From<VolumeMountRequest> for SdkVolumeMountRequest {
    fn from(vol: VolumeMountRequest) -> Self {
        Self {
            host_path: vol.host_path,
            container_path: vol.container_path,
            read_only: vol.read_only,
        }
    }
}

/// Start rental API request
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct StartRentalApiRequest {
    #[pyo3(get, set)]
    pub node_selection: NodeSelection,
    #[pyo3(get, set)]
    pub container_image: String,
    #[pyo3(get, set)]
    pub environment: HashMap<String, String>,
    #[pyo3(get, set)]
    pub ports: Vec<PortMappingRequest>,
    #[pyo3(get, set)]
    pub resources: ResourceRequirementsRequest,
    #[pyo3(get, set)]
    pub command: Vec<String>,
    #[pyo3(get, set)]
    pub volumes: Vec<VolumeMountRequest>,
    #[pyo3(get, set)]
    pub no_ssh: bool,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl StartRentalApiRequest {
    #[new]
    #[pyo3(signature = (node_selection, container_image, environment=None, ports=None, resources=None, command=None, volumes=None, no_ssh=false))]
    #[allow(clippy::too_many_arguments)]
    fn new(
        node_selection: NodeSelection,
        container_image: String,
        environment: Option<HashMap<String, String>>,
        ports: Option<Vec<PortMappingRequest>>,
        resources: Option<ResourceRequirementsRequest>,
        command: Option<Vec<String>>,
        volumes: Option<Vec<VolumeMountRequest>>,
        no_ssh: bool,
    ) -> Self {
        Self {
            node_selection,
            container_image,
            environment: environment.unwrap_or_default(),
            ports: ports.unwrap_or_default(),
            resources: resources.unwrap_or_default(),
            command: command.unwrap_or_default(),
            volumes: volumes.unwrap_or_default(),
            no_ssh,
        }
    }
}

impl From<StartRentalApiRequest> for SdkStartRentalApiRequest {
    fn from(req: StartRentalApiRequest) -> Self {
        Self {
            node_selection: req.node_selection.into(),
            container_image: req.container_image,
            environment: req.environment,
            ports: req.ports.into_iter().map(Into::into).collect(),
            resources: req.resources.into(),
            command: req.command,
            volumes: req.volumes.into_iter().map(Into::into).collect(),
            no_ssh: req.no_ssh,
        }
    }
}

/// Query parameters for listing available nodes
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone, Default)]
pub struct ListAvailableNodesQuery {
    #[pyo3(get, set)]
    pub available: Option<bool>,
    #[pyo3(get, set)]
    pub min_gpu_memory: Option<u32>,
    #[pyo3(get, set)]
    pub gpu_type: Option<String>,
    #[pyo3(get, set)]
    pub min_gpu_count: Option<u32>,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl ListAvailableNodesQuery {
    #[new]
    #[pyo3(signature = (available=None, min_gpu_memory=None, gpu_type=None, min_gpu_count=None))]
    fn new(
        available: Option<bool>,
        min_gpu_memory: Option<u32>,
        gpu_type: Option<String>,
        min_gpu_count: Option<u32>,
    ) -> Self {
        Self {
            available,
            min_gpu_memory,
            gpu_type,
            min_gpu_count,
        }
    }
}

impl From<ListAvailableNodesQuery> for SdkListAvailableNodesQuery {
    fn from(query: ListAvailableNodesQuery) -> Self {
        Self {
            available: query.available,
            min_gpu_memory: query.min_gpu_memory,
            gpu_type: query.gpu_type,
            min_gpu_count: query.min_gpu_count,
            location: None, // Python SDK doesn't support location filtering yet
        }
    }
}

/// Query parameters for listing rentals
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone, Default)]
pub struct ListRentalsQuery {
    #[pyo3(get, set)]
    pub status: Option<String>, // We'll use String for the enum
    #[pyo3(get, set)]
    pub gpu_type: Option<String>,
    #[pyo3(get, set)]
    pub min_gpu_count: Option<u32>,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl ListRentalsQuery {
    #[new]
    #[pyo3(signature = (status=None, gpu_type=None, min_gpu_count=None))]
    fn new(status: Option<String>, gpu_type: Option<String>, min_gpu_count: Option<u32>) -> Self {
        Self {
            status,
            gpu_type,
            min_gpu_count,
        }
    }
}

impl From<ListRentalsQuery> for SdkListRentalsQuery {
    fn from(query: ListRentalsQuery) -> Self {
        let status = query.status.and_then(|s| match s.to_lowercase().as_str() {
            "provisioning" => Some(RentalState::Provisioning),
            "active" => Some(RentalState::Active),
            "stopping" => Some(RentalState::Stopping),
            "stopped" => Some(RentalState::Stopped),
            "failed" => Some(RentalState::Failed),
            _ => None,
        });

        Self {
            status,
            gpu_type: query.gpu_type,
            min_gpu_count: query.min_gpu_count,
        }
    }
}

// Deployment types

use basilica_sdk::types::{
    CreateDeploymentRequest as SdkCreateDeploymentRequest,
    DeleteDeploymentResponse as SdkDeleteDeploymentResponse,
    DeploymentListResponse as SdkDeploymentListResponse,
    DeploymentResponse as SdkDeploymentResponse, DeploymentSummary as SdkDeploymentSummary,
    EnvVar as SdkEnvVar, GpuRequirementsSpec as SdkGpuRequirementsSpec,
    PersistentStorageSpec as SdkPersistentStorageSpec, PodInfo as SdkPodInfo,
    ReplicaStatus as SdkReplicaStatus, ResourceRequirements as SdkResourceRequirements,
    StorageBackend as SdkStorageBackend, StorageSpec as SdkStorageSpec,
};

/// Environment variable for container deployments
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct EnvVar {
    #[pyo3(get, set)]
    pub name: String,
    #[pyo3(get, set)]
    pub value: String,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl EnvVar {
    #[new]
    fn new(name: String, value: String) -> Self {
        Self { name, value }
    }
}

impl From<EnvVar> for SdkEnvVar {
    fn from(env: EnvVar) -> Self {
        Self {
            name: env.name,
            value: env.value,
        }
    }
}

impl From<SdkEnvVar> for EnvVar {
    fn from(env: SdkEnvVar) -> Self {
        Self {
            name: env.name,
            value: env.value,
        }
    }
}

/// GPU requirements specification for container deployments
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct GpuRequirementsSpec {
    #[pyo3(get, set)]
    pub count: u32,
    #[pyo3(get, set)]
    pub model: Vec<String>,
    #[pyo3(get, set)]
    pub min_cuda_version: Option<String>,
    #[pyo3(get, set)]
    pub min_gpu_memory_gb: Option<u32>,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl GpuRequirementsSpec {
    #[new]
    #[pyo3(signature = (count, model, min_cuda_version=None, min_gpu_memory_gb=None))]
    fn new(
        count: u32,
        model: Vec<String>,
        min_cuda_version: Option<String>,
        min_gpu_memory_gb: Option<u32>,
    ) -> Self {
        Self {
            count,
            model,
            min_cuda_version,
            min_gpu_memory_gb,
        }
    }
}

impl From<GpuRequirementsSpec> for SdkGpuRequirementsSpec {
    fn from(gpu: GpuRequirementsSpec) -> Self {
        Self {
            count: gpu.count,
            model: gpu.model,
            min_cuda_version: gpu.min_cuda_version,
            min_gpu_memory_gb: gpu.min_gpu_memory_gb,
        }
    }
}

impl From<SdkGpuRequirementsSpec> for GpuRequirementsSpec {
    fn from(gpu: SdkGpuRequirementsSpec) -> Self {
        Self {
            count: gpu.count,
            model: gpu.model,
            min_cuda_version: gpu.min_cuda_version,
            min_gpu_memory_gb: gpu.min_gpu_memory_gb,
        }
    }
}

/// Resource requirements for container deployments
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct ResourceRequirements {
    #[pyo3(get, set)]
    pub cpu: String,
    #[pyo3(get, set)]
    pub memory: String,
    #[pyo3(get, set)]
    pub gpus: Option<GpuRequirementsSpec>,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl ResourceRequirements {
    #[new]
    #[pyo3(signature = (cpu, memory, gpus=None))]
    fn new(cpu: String, memory: String, gpus: Option<GpuRequirementsSpec>) -> Self {
        Self { cpu, memory, gpus }
    }
}

impl From<ResourceRequirements> for SdkResourceRequirements {
    fn from(res: ResourceRequirements) -> Self {
        Self {
            cpu: res.cpu,
            memory: res.memory,
            gpus: res.gpus.map(|g| g.into()),
        }
    }
}

impl From<SdkResourceRequirements> for ResourceRequirements {
    fn from(res: SdkResourceRequirements) -> Self {
        Self {
            cpu: res.cpu,
            memory: res.memory,
            gpus: res.gpus.map(|g| g.into()),
        }
    }
}

/// Replica status for deployments
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct ReplicaStatus {
    #[pyo3(get)]
    pub desired: u32,
    #[pyo3(get)]
    pub ready: u32,
}

impl From<SdkReplicaStatus> for ReplicaStatus {
    fn from(status: SdkReplicaStatus) -> Self {
        Self {
            desired: status.desired,
            ready: status.ready,
        }
    }
}

/// Pod information
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct PodInfo {
    #[pyo3(get)]
    pub name: String,
    #[pyo3(get)]
    pub status: String,
    #[pyo3(get)]
    pub node: Option<String>,
}

impl From<SdkPodInfo> for PodInfo {
    fn from(pod: SdkPodInfo) -> Self {
        Self {
            name: pod.name,
            status: pod.status,
            node: pod.node,
        }
    }
}

/// Storage backend type
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass_enum)]
#[pyclass(eq, eq_int)]
#[derive(Clone, PartialEq)]
#[allow(clippy::upper_case_acronyms)]
pub enum StorageBackend {
    R2,
    S3,
    GCS,
}

impl From<StorageBackend> for SdkStorageBackend {
    fn from(backend: StorageBackend) -> Self {
        match backend {
            StorageBackend::R2 => SdkStorageBackend::R2,
            StorageBackend::S3 => SdkStorageBackend::S3,
            StorageBackend::GCS => SdkStorageBackend::GCS,
        }
    }
}

/// Persistent storage specification
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct PersistentStorageSpec {
    #[pyo3(get, set)]
    pub enabled: bool,
    #[pyo3(get, set)]
    pub backend: StorageBackend,
    #[pyo3(get, set)]
    pub bucket: String,
    #[pyo3(get, set)]
    pub region: Option<String>,
    #[pyo3(get, set)]
    pub endpoint: Option<String>,
    #[pyo3(get, set)]
    pub credentials_secret: Option<String>,
    #[pyo3(get, set)]
    pub sync_interval_ms: u64,
    #[pyo3(get, set)]
    pub cache_size_mb: usize,
    #[pyo3(get, set)]
    pub mount_path: String,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl PersistentStorageSpec {
    #[new]
    #[pyo3(signature = (enabled, backend, bucket="", region=None, endpoint=None, credentials_secret=None, sync_interval_ms=1000, cache_size_mb=1024, mount_path="/data"))]
    #[allow(clippy::too_many_arguments)]
    fn new(
        enabled: bool,
        backend: StorageBackend,
        bucket: &str,
        region: Option<String>,
        endpoint: Option<String>,
        credentials_secret: Option<String>,
        sync_interval_ms: u64,
        cache_size_mb: usize,
        mount_path: &str,
    ) -> Self {
        Self {
            enabled,
            backend,
            bucket: bucket.to_string(),
            region,
            endpoint,
            credentials_secret,
            sync_interval_ms,
            cache_size_mb,
            mount_path: mount_path.to_string(),
        }
    }
}

impl From<PersistentStorageSpec> for SdkPersistentStorageSpec {
    fn from(spec: PersistentStorageSpec) -> Self {
        Self {
            enabled: spec.enabled,
            backend: spec.backend.into(),
            bucket: spec.bucket,
            region: spec.region,
            endpoint: spec.endpoint,
            credentials_secret: spec.credentials_secret,
            sync_interval_ms: spec.sync_interval_ms,
            cache_size_mb: spec.cache_size_mb,
            mount_path: spec.mount_path,
        }
    }
}

/// Storage specification
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct StorageSpec {
    #[pyo3(get, set)]
    pub persistent: Option<PersistentStorageSpec>,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl StorageSpec {
    #[new]
    #[pyo3(signature = (persistent=None))]
    fn new(persistent: Option<PersistentStorageSpec>) -> Self {
        Self { persistent }
    }
}

impl From<StorageSpec> for SdkStorageSpec {
    fn from(spec: StorageSpec) -> Self {
        Self {
            persistent: spec.persistent.map(Into::into),
        }
    }
}

/// Create deployment request
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct CreateDeploymentRequest {
    #[pyo3(get, set)]
    pub instance_name: String,
    #[pyo3(get, set)]
    pub image: String,
    #[pyo3(get, set)]
    pub replicas: u32,
    #[pyo3(get, set)]
    pub port: u32,
    #[pyo3(get, set)]
    pub command: Option<Vec<String>>,
    #[pyo3(get, set)]
    pub args: Option<Vec<String>>,
    #[pyo3(get, set)]
    pub env: Option<HashMap<String, String>>,
    #[pyo3(get, set)]
    pub resources: Option<ResourceRequirements>,
    #[pyo3(get, set)]
    pub ttl_seconds: Option<u32>,
    #[pyo3(get, set)]
    pub public: bool,
    #[pyo3(get, set)]
    pub storage: Option<StorageSpec>,
}

#[cfg_attr(feature = "stub-gen", gen_stub_pymethods)]
#[pymethods]
impl CreateDeploymentRequest {
    #[new]
    #[pyo3(signature = (instance_name, image, replicas, port, command=None, args=None, env=None, resources=None, ttl_seconds=None, public=true, storage=None))]
    #[allow(clippy::too_many_arguments)]
    fn new(
        instance_name: String,
        image: String,
        replicas: u32,
        port: u32,
        command: Option<Vec<String>>,
        args: Option<Vec<String>>,
        env: Option<HashMap<String, String>>,
        resources: Option<ResourceRequirements>,
        ttl_seconds: Option<u32>,
        public: bool,
        storage: Option<StorageSpec>,
    ) -> Self {
        Self {
            instance_name,
            image,
            replicas,
            port,
            command,
            args,
            env,
            resources,
            ttl_seconds,
            public,
            storage,
        }
    }
}

impl From<CreateDeploymentRequest> for SdkCreateDeploymentRequest {
    fn from(req: CreateDeploymentRequest) -> Self {
        Self {
            instance_name: req.instance_name,
            image: req.image,
            replicas: req.replicas,
            port: req.port,
            command: req.command,
            args: req.args,
            env: req.env,
            resources: req.resources.map(Into::into),
            ttl_seconds: req.ttl_seconds,
            public: req.public,
            storage: req.storage.map(Into::into),
            enable_billing: true,
            queue_name: None,
            suspended: false,
            priority: None,
        }
    }
}

/// Deployment response
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct DeploymentResponse {
    #[pyo3(get)]
    pub instance_name: String,
    #[pyo3(get)]
    pub user_id: String,
    #[pyo3(get)]
    pub namespace: String,
    #[pyo3(get)]
    pub state: String,
    #[pyo3(get)]
    pub url: String,
    #[pyo3(get)]
    pub replicas: ReplicaStatus,
    #[pyo3(get)]
    pub created_at: String,
    #[pyo3(get)]
    pub updated_at: Option<String>,
    #[pyo3(get)]
    pub pods: Option<Vec<PodInfo>>,
}

impl From<SdkDeploymentResponse> for DeploymentResponse {
    fn from(response: SdkDeploymentResponse) -> Self {
        Self {
            instance_name: response.instance_name,
            user_id: response.user_id,
            namespace: response.namespace,
            state: response.state,
            url: response.url,
            replicas: response.replicas.into(),
            created_at: response.created_at,
            updated_at: response.updated_at,
            pods: response
                .pods
                .map(|pods| pods.into_iter().map(Into::into).collect()),
        }
    }
}

/// Deployment summary for list responses
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct DeploymentSummary {
    #[pyo3(get)]
    pub instance_name: String,
    #[pyo3(get)]
    pub state: String,
    #[pyo3(get)]
    pub url: String,
    #[pyo3(get)]
    pub replicas: ReplicaStatus,
    #[pyo3(get)]
    pub created_at: String,
}

impl From<SdkDeploymentSummary> for DeploymentSummary {
    fn from(summary: SdkDeploymentSummary) -> Self {
        Self {
            instance_name: summary.instance_name,
            state: summary.state,
            url: summary.url,
            replicas: summary.replicas.into(),
            created_at: summary.created_at,
        }
    }
}

/// List deployments response
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct DeploymentListResponse {
    #[pyo3(get)]
    pub deployments: Vec<DeploymentSummary>,
    #[pyo3(get)]
    pub total: usize,
}

impl From<SdkDeploymentListResponse> for DeploymentListResponse {
    fn from(response: SdkDeploymentListResponse) -> Self {
        Self {
            deployments: response.deployments.into_iter().map(Into::into).collect(),
            total: response.total,
        }
    }
}

/// Delete deployment response
#[cfg_attr(feature = "stub-gen", gen_stub_pyclass)]
#[pyclass]
#[derive(Clone)]
pub struct DeleteDeploymentResponse {
    #[pyo3(get)]
    pub instance_name: String,
    #[pyo3(get)]
    pub state: String,
    #[pyo3(get)]
    pub message: String,
}

impl From<SdkDeleteDeploymentResponse> for DeleteDeploymentResponse {
    fn from(response: SdkDeleteDeploymentResponse) -> Self {
        Self {
            instance_name: response.instance_name,
            state: response.state,
            message: response.message,
        }
    }
}
