"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const lambda = require("../lib");
/**
 * Use a class as test fixture
 *
 * setUp() will be mapped to the (synchronous) constructor. tearDown(cb) will be called if available.
 */
function classFixture(klass) {
    let fixture;
    const ret = {
        setUp(cb) {
            fixture = new klass();
            cb();
        },
        tearDown(cb) {
            if (fixture.tearDown) {
                fixture.tearDown(cb);
            }
            else {
                cb();
            }
        },
    };
    const testNames = Reflect.ownKeys(klass.prototype).filter(m => m !== 'tearDown' && m !== 'constructor');
    for (const testName of testNames) {
        ret[testName] = (test) => fixture[testName](test);
    }
    return ret;
}
class SomethingConnectable {
    constructor(connections) {
        this.connections = connections;
    }
}
module.exports = {
    'lambda in a VPC': classFixture(class Henk {
        constructor() {
            // GIVEN
            this.app = new cdk.App();
            this.stack = new cdk.Stack(this.app, 'stack');
            this.vpc = new ec2.Vpc(this.stack, 'VPC');
            // WHEN
            this.lambda = new lambda.Function(this.stack, 'Lambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS_10_X,
                vpc: this.vpc,
                allowAllOutbound: false,
            });
        }
        'has subnet and securitygroup'(test) {
            // THEN
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        { 'Fn::GetAtt': ['LambdaSecurityGroupE74659A1', 'GroupId'] },
                    ],
                    SubnetIds: [
                        { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                        { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                    ],
                },
            }));
            test.done();
        }
        'has securitygroup that is passed in props'(test) {
            // WHEN
            new lambda.Function(this.stack, 'LambdaWithCustomSG', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS_10_X,
                vpc: this.vpc,
                securityGroup: new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupX', { vpc: this.vpc }),
            });
            // THEN
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        { 'Fn::GetAtt': ['CustomSecurityGroupX6C7F3A78', 'GroupId'] },
                    ],
                    SubnetIds: [
                        { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                        { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                    ],
                },
            }));
            test.done();
        }
        'has all the securitygroups that are passed as a list of SG in props'(test) {
            // WHEN
            new lambda.Function(this.stack, 'LambdaWithCustomSGList', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS_10_X,
                vpc: this.vpc,
                securityGroups: [
                    new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupA', { vpc: this.vpc }),
                    new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupB', { vpc: this.vpc }),
                ],
            });
            // THEN
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        { 'Fn::GetAtt': ['CustomSecurityGroupA267F62DE', 'GroupId'] },
                        { 'Fn::GetAtt': ['CustomSecurityGroupB1118D0D5', 'GroupId'] },
                    ],
                    SubnetIds: [
                        { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                        { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                    ],
                },
            }));
            test.done();
        }
        'fails if both of securityGroup and securityGroups are passed in props at once'(test) {
            // THEN
            test.throws(() => {
                new lambda.Function(this.stack, 'LambdaWithWrongProps', {
                    code: new lambda.InlineCode('foo'),
                    handler: 'index.handler',
                    runtime: lambda.Runtime.NODEJS_10_X,
                    vpc: this.vpc,
                    securityGroup: new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupB', { vpc: this.vpc }),
                    securityGroups: [
                        new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupC', { vpc: this.vpc }),
                        new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupD', { vpc: this.vpc }),
                    ],
                });
            }, /Only one of the function props, securityGroup or securityGroups, is allowed/);
            test.done();
        }
        'participates in Connections objects'(test) {
            // GIVEN
            const securityGroup = new ec2.SecurityGroup(this.stack, 'SomeSecurityGroup', { vpc: this.vpc });
            const somethingConnectable = new SomethingConnectable(new ec2.Connections({ securityGroups: [securityGroup] }));
            // WHEN
            this.lambda.connections.allowTo(somethingConnectable, ec2.Port.allTcp(), 'Lambda can call connectable');
            // THEN: Lambda can connect to SomeSecurityGroup
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
                GroupId: { 'Fn::GetAtt': ['LambdaSecurityGroupE74659A1', 'GroupId'] },
                IpProtocol: 'tcp',
                Description: 'Lambda can call connectable',
                DestinationSecurityGroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
                FromPort: 0,
                ToPort: 65535,
            }));
            // THEN: SomeSecurityGroup accepts connections from Lambda
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                Description: 'Lambda can call connectable',
                FromPort: 0,
                GroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
                SourceSecurityGroupId: { 'Fn::GetAtt': ['LambdaSecurityGroupE74659A1', 'GroupId'] },
                ToPort: 65535,
            }));
            test.done();
        }
        'can still make Connections after export/import'(test) {
            // GIVEN
            const stack2 = new cdk.Stack(this.app, 'stack2');
            const securityGroup = new ec2.SecurityGroup(stack2, 'SomeSecurityGroup', { vpc: this.vpc });
            const somethingConnectable = new SomethingConnectable(new ec2.Connections({ securityGroups: [securityGroup] }));
            // WHEN
            somethingConnectable.connections.allowFrom(this.lambda.connections, ec2.Port.allTcp(), 'Lambda can call connectable');
            // THEN: SomeSecurityGroup accepts connections from Lambda
            assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
                GroupId: {
                    'Fn::ImportValue': 'stack:ExportsOutputFnGetAttLambdaSecurityGroupE74659A1GroupId8F3EC6F1',
                },
                IpProtocol: 'tcp',
                Description: 'Lambda can call connectable',
                DestinationSecurityGroupId: {
                    'Fn::GetAtt': [
                        'SomeSecurityGroupEF219AD6',
                        'GroupId',
                    ],
                },
                FromPort: 0,
                ToPort: 65535,
            }));
            // THEN: Lambda can connect to SomeSecurityGroup
            assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                Description: 'Lambda can call connectable',
                FromPort: 0,
                GroupId: {
                    'Fn::GetAtt': [
                        'SomeSecurityGroupEF219AD6',
                        'GroupId',
                    ],
                },
                SourceSecurityGroupId: {
                    'Fn::ImportValue': 'stack:ExportsOutputFnGetAttLambdaSecurityGroupE74659A1GroupId8F3EC6F1',
                },
                ToPort: 65535,
            }));
            test.done();
        }
    }),
    'lambda without VPC throws Error upon accessing connections'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const lambdaFn = new lambda.Function(stack, 'Lambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        test.throws(() => {
            lambdaFn.connections.allowToAnyIpv4(ec2.Port.allTcp(), 'Reach for the world Lambda!');
        });
        test.done();
    },
    'picking public subnets is not allowed'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        test.throws(() => {
            new lambda.Function(stack, 'Lambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS_10_X,
                vpc,
                vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
            });
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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