"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const lambda = require("../lib");
module.exports = {
    'can add same singleton Lambda multiple times, only instantiated once in template'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        for (let i = 0; i < 5; i++) {
            new lambda.SingletonFunction(stack, `Singleton${i}`, {
                uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
                code: new lambda.InlineCode('def hello(): pass'),
                runtime: lambda.Runtime.PYTHON_2_7,
                handler: 'index.hello',
                timeout: cdk.Duration.minutes(5),
            });
        }
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [
                                {
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'lambda.amazonaws.com' },
                                },
                            ],
                            Version: '2012-10-17',
                        },
                        ManagedPolicyArns: [
                            {
                                'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']],
                            },
                        ],
                    },
                },
                SingletonLambda84c0de93353f42179b0b45b6c993251a840BCC38: {
                    Type: 'AWS::Lambda::Function',
                    Properties: {
                        Code: {
                            ZipFile: 'def hello(): pass',
                        },
                        Handler: 'index.hello',
                        Role: { 'Fn::GetAtt': ['SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235', 'Arn'] },
                        Runtime: 'python2.7',
                        Timeout: 300,
                    },
                    DependsOn: ['SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235'],
                },
            },
        }));
        test.done();
    },
    'dependencies are correctly added'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
            timeout: cdk.Duration.minutes(5),
        });
        const dependency = new iam.User(stack, 'dependencyUser');
        // WHEN
        singleton.addDependency(dependency);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            DependsOn: [
                'dependencyUser1B9CB07E',
                'SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'dependsOn are correctly added'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
            timeout: cdk.Duration.minutes(5),
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        singleton.dependOn(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::User', {
            DependsOn: [
                'SingletonLambda84c0de93353f42179b0b45b6c993251a840BCC38',
                'SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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