"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const function_1 = require("./function");
const function_base_1 = require("./function-base");
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @resource AWS::Lambda::Function
 */
class SingletonFunction extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureLambda(props);
        this.permissionsNode = this.lambdaFunction.node;
        this.functionArn = this.lambdaFunction.functionArn;
        this.functionName = this.lambdaFunction.functionName;
        this.role = this.lambdaFunction.role;
        this.grantPrincipal = this.lambdaFunction.grantPrincipal;
        this.canCreatePermissions = true; // Doesn't matter, addPermission is overriden anyway
    }
    addPermission(name, permission) {
        return this.lambdaFunction.addPermission(name, permission);
    }
    /**
     * Using node.addDependency() does not work on this method as the underlying lambda function is modeled
     * as a singleton across the stack. Use this method instead to declare dependencies.
     */
    addDependency(...up) {
        this.lambdaFunction.node.addDependency(...up);
    }
    /**
     * The SingletonFunction construct cannot be added as a dependency of another construct using
     * node.addDependency(). Use this method instead to declare this as a dependency of another construct.
     */
    dependOn(down) {
        down.node.addDependency(this.lambdaFunction);
    }
    ensureLambda(props) {
        const constructName = (props.lambdaPurpose || 'SingletonLambda') + slugify(props.uuid);
        const existing = cdk.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            // Just assume this is true
            return existing;
        }
        return new function_1.Function(cdk.Stack.of(this), constructName, props);
    }
}
exports.SingletonFunction = SingletonFunction;
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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