"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const event_invoke_config_1 = require("./event-invoke-config");
const event_source_mapping_1 = require("./event-source-mapping");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
class FunctionBase extends core_1.Resource {
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     * @param permission The permission to grant to this Lambda function. @see Permission for details.
     */
    addPermission(id, permission) {
        if (!this.canCreatePermissions) {
            // FIXME: Report metadata
            return;
        }
        const principal = this.parsePermissionPrincipal(permission.principal);
        const action = permission.action || 'lambda:InvokeFunction';
        const scope = permission.scope || this;
        new lambda_generated_1.CfnPermission(scope, id, {
            action,
            principal,
            functionName: this.functionArn,
            eventSourceToken: permission.eventSourceToken,
            sourceAccount: permission.sourceAccount,
            sourceArn: permission.sourceArn,
        });
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    get connections() {
        if (!this._connections) {
            // tslint:disable-next-line:max-line-length
            throw new Error('Only VPC-associated Lambda Functions have security groups to manage. Supply the "vpc" parameter when creating the Lambda, or "securityGroupId" when importing it.');
        }
        return this._connections;
    }
    get latestVersion() {
        // Dynamic to avoid infinite recursion when creating the LatestVersion instance...
        return new LatestVersion(this);
    }
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    get isBoundToVpc() {
        return !!this._connections;
    }
    addEventSourceMapping(id, options) {
        return new event_source_mapping_1.EventSourceMapping(this, id, {
            target: this,
            ...options,
        });
    }
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['lambda:InvokeFunction'],
            resourceArns: [this.functionArn],
            // Fake resource-like object on which to call addToResourcePolicy(), which actually
            // calls addPermission()
            resource: {
                addToResourcePolicy: (_statement) => {
                    // Couldn't add permissions to the principal, so add them locally.
                    const identifier = `Invoke${grantee.grantPrincipal}`; // calls the .toString() of the princpal
                    this.addPermission(identifier, {
                        principal: grantee.grantPrincipal,
                        action: 'lambda:InvokeFunction',
                    });
                    return { statementAdded: true, policyDependable: this.node.findChild(identifier) };
                },
                node: this.node,
            },
        });
    }
    /**
     * Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     *
     *     import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     *     myFunction.addEventSource(new SqsEventSource(myQueue));
     *
     * @param source The event source to bind to this function
     */
    addEventSource(source) {
        source.bind(this);
    }
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this,
            ...options,
        });
    }
    parsePermissionPrincipal(principal) {
        if (!principal) {
            return undefined;
        }
        // use duck-typing, not instance of
        if ('accountId' in principal) {
            return principal.accountId;
        }
        if ('service' in principal) {
            return principal.service;
        }
        if ('arn' in principal) {
            return principal.arn;
        }
        throw new Error(`Invalid principal type for Lambda permission statement: ${principal.constructor.name}. ` +
            'Supported: AccountPrincipal, ArnPrincipal, ServicePrincipal');
    }
}
exports.FunctionBase = FunctionBase;
class QualifiedFunctionBase extends FunctionBase {
    constructor() {
        super(...arguments);
        this.permissionsNode = this.node;
    }
    get latestVersion() {
        return this.lambda.latestVersion;
    }
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the qualified function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this.lambda,
            qualifier: this.qualifier,
            ...options,
        });
    }
}
exports.QualifiedFunctionBase = QualifiedFunctionBase;
/**
 * The $LATEST version of a function, useful when attempting to create aliases.
 */
class LatestVersion extends FunctionBase {
    constructor(lambda) {
        super(lambda, '$LATEST');
        this.version = '$LATEST';
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.lambda = lambda;
    }
    get functionArn() {
        return `${this.lambda.functionArn}:${this.version}`;
    }
    get functionName() {
        return `${this.lambda.functionName}:${this.version}`;
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get latestVersion() {
        return this;
    }
    get role() {
        return this.lambda.role;
    }
    addAlias(aliasName, options = {}) {
        return util_1.addAlias(this, this, aliasName, options);
    }
}
//# sourceMappingURL=data:application/json;base64,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