# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['thquickjs']

package_data = \
{'': ['*']}

install_requires = \
['quickjs>=1.17.0,<2.0.0', 'thresult>=0.9.2,<0.10.0']

setup_kwargs = {
    'name': 'thquickjs',
    'version': '0.9.0',
    'description': 'TangledHub QuickJs wrapper library',
    'long_description': "[![License][bsd3-image]][bsd3-url]\n[![Build][build-image]]()\n[![Coverage][coverage-image]]()\n\n# thquickjs\n\n## Overview\nTangledHub library for creating context for JavaScript and Python code.\nQuickJS is a small and embeddable Javascript engine. \nSafely evaluate untrusted Javascript. \nCreate and manipulate values inside the QuickJS runtime. \nExpose host functions to the QuickJS runtime\n\n## Licencing\nthquickjs is licensed under the BSD license. Check the [LICENSE file](LICENCE.md) for details\n\n## Installation\n```bash\npip install thquickjs\n```\n\n---\n\n## Testing\nRun in console:\n```bash\ndocker-compose build thquickjs-test ; docker-compose run --rm thquickjs-test\n```\n\n## Building\nRun in console:\n```bash\ndocker-compose build thquickjs-build ; docker-compose run --rm thquickjs-build\n```\n\n## Publish\nRun in console:\n```bash\ndocker-compose build thquickjs-publish ; docker-compose run --rm thquickjs-publish\n```\n\n---\n\n## Usage\n\n### Evaluate JavaScript code\nCreate instance of QuickJS. Use that instance to evaluate JavaScript code. \nJavaScript code can be in string or file.\nJavaScript code (variables, functions) can be accessed from quickjs context.\n```python\n'''\n    setup and create instance of QuickJS object\n    - eval code function evaluate JS code\n    - get function obtain a function from quickjs context\n    - js function call in python code\n\n    params:\n        eval:\n            code: string(js code)\n        \n        get:\n            function_name: str\n\n    returns:\n        get:\n            function: js function\n'''\nfrom thquickjs.thquickjs import QuickJS\n\n# create QuckJS object\nqjs: QuickJS = QuickJS()\n\n# example js code\ncode = '''\n    f = function(x) {\n        return 40 + x;\n    }\n    \n    f1 = function(x, y) {\n        return x + y;\n    }\n'''\n\n# evaluate JS code\nqjs.eval(code).unwrap()\n\n# obtain a function from quickjs context\nfunc = qjs.get('f1').unwrap()  # js function in python\n\n# invoke function from JS code in Python code\nresult = func(2, 3)  # returns 5\n```\n\nJavaScript code can be in separate file. \nNote that file extension doesn't have to be *.js*.\n\n```python\n'''\n    setup and create instance of QuickJS object\n    - eval code function evaluate JS code from file\n    - get function obtain a function from quickjs context\n    - js function call in python code\n\n    params:\n        eval:\n            code: string(js code)\n        \n        get:\n            function_name: str\n\n    returns:\n        get:\n            function: js function\n'''\nfrom thquickjs.thquickjs import QuickJS\n\n# create QuckJS object\nqjs = QuickJS()\n\n# value of code is content of file \ncode = '''\n    f = function(x) {\n        return 40 + x;\n    }\n    \n    f1 = function(x, y) {\n        return x + y;\n    }\n'''\n\nfile_name: str = file_name = 'abc.txt'\n\nwith open(file_name, 'r') as reader:\n    content = reader.read()\n\n# evaluate JS code\nqjs.eval(content).unwrap()\n\n# obtain a function from quickjs context\nfunc = qjs.get('f1').unwrap()  # js function in python\n\n# invoke function from JS code in Python code\nresult = func(2, 3)  # returns 5\n```\n\nTry to get non-existing variable from context will return None.\n\n```python\nfrom thquickjs.thquickjs import QuickJS\n\n# create QuckJS object\nqjs = QuickJS()\n\n# example js code\ncode = '''\n    f = function(x) {\n        return 40 + x;\n    }\n    \n    f1 = function(x, y) {\n        return x + y;\n    }\n'''\n\n# evaluate JS code\nqjs.eval(code).unwrap()\n\n# try to get non-existing variable from context will return None\nfunc = qjs.get('a').unwrap()  # None\n```\n### Parsing js modules\n```python\n# create QuckJS object\nqjs = QuickJS()\n\n# import lodash\npath = os.path.join('/deps' 'vendor', 'lodash.js')\n\n# import module by specified path\nqjs.import_js_module(path).unwrap()\n\n# use lodash\ncode = '''\n    var a = _.range(10);\n    a;\n'''\n\n# evaluate JS code\nres = qjs.eval(code, as_json=True).unwrap()\n# [0, 1, 2, 3, 4, 5, 6, 7, 8, 9]\n```\n\n### Add values to thquickjs context\nVariables can be added in thquickjs context.\n\n```python\nfrom thquickjs.thquickjs import QuickJS\n\n# create QucikJS object\nqjs: QuickJS = QuickJS()\n\n# set variable and value to context\nqjs.set('x', 8, ).unwrap()\n\n# get value by given variable name\nv = qjs.get('x').unwrap()  # v is 8\n\n# change value of variable x in context\nqjs.set('x', 12, ).unwrap()\n\n# get value by given variable name\nv = qjs.get('x').unwrap()  # v is 12\n\n```\n\nAdd Python callable in thquickjs context\n\n```python\nfrom thquickjs.thquickjs import QuickJS\n\n# create QucikJS object\nqjs = QuickJS()\n\npy_name = 'pylam'\npy_func = lambda x: x * 10\n\n# adding Python callable in context\nqjs.add_callable(py_name, py_func).unwrap()\n\n# get function by given variable name\nf = qjs.get(py_name).unwrap()  # v is 8\n\n# result\nresult = f(5)  # returns value of 50\n```\n\n### Handling errors\nIn previous examples, using unwrap() will return appropriate value or rise Exception. \nUsing *unwrap_or(v: Any)* or *unwrap_value()* will prevent rising exceptions and terminating program.\n\nHandling errors when parsing JavaScript code\n\n```python\nfrom _quickjs import JSException\nfrom thquickjs.thquickjs import QuickJS\n\n# create QuckJS object\nqjs = QuickJS()\n\n# example of unparsable js code\ncode = '''unparsable js code'''\n\n# standard way to handle exception using try and except blocks\ntry:\n    # evaluate unparsable JS code\n    qjs_func: _quickjs.Object = qjs.eval(code).unwrap()\nexcept JSException as e:\n    pass\n\n# evaluation of js code and unwraps_value - returns exception as string\nqjs_func = qjs.eval(code).unwrap_value()\n\n# unwrap_or method sets default value in case of exception\n# qjs_func in case of exception will have default value\nqjs_func = qjs.eval(code).unwrap_or('default value in case of exception') \n```\n\nHandling errors when adding Python callable\n\n```python\nfrom thquickjs.thquickjs import QuickJS\n\n## handling errors\n\n# create QucikJS object\nqjs: QuickJS = QuickJS()\n\n\n# handling error with try/except\ndef add_ten(n: int) -> int:\n    return n + 10\n\n\npy_name = 'pylam'\n\ntry:\n    # add callable to context\n    f = qjs.add_callable(py_name, 'unparsable').unwrap()\nexcept TypeError as e:\n    f = add_ten\n\n\n# handling error with unwrap_value\ndef add_two(n: int) -> int:\n    return n + 2\n\n\npy_name = 'pylam'\n\n# in case of Err, variable f will contain error message\nf = qjs.add_callable(py_name, 'unparsable').unwrap_value()\n\n# in case of Err, add_two function will be assigned to variable f\nf = qjs.add_callable(py_name, 'unparsable').unwrap_or(add_two)\nf(3)  # result 5\n```\n\nUsing unwrap_value() and unwrap_or(v: Any) when there is no exceptions, \nwill have no effects on return value.\n\n```python\nfrom thquickjs.thquickjs import QuickJS\n\n# create QucikJS object\nqjs = QuickJS()\n\n# set variable and value to context\nqjs.set('x', 8, ).unwrap()\n\n# get value by given variable name\nv = qjs.get('x').unwrap()  # v is 8\n\n# change value of variable x in context\nqjs.set('x', 12, ).unwrap()\n\n# get value by given variable name\nv = qjs.get('x').unwrap()  # v is 12\n\n# if unwrap doesn't rise Exception, unwrap_value will return expected value\nv1 = qjs.get('x').unwrap_value()  # v1 is 12\n\n# if unwrap doesn't rise Exception, unwrap_or will have no effects on return value\nv2 = qjs.get('x').unwrap_or(11)  # v2 is 12\n\n```\n### Set memory limit\nUse *set_memory_limit* method to set available memory for thquickjs.\nTo get informations about memory use *memory* method.\n```python\nqjs: QuickJS = QuickJS()\n\nqjs.set_memory_limit(memory_limit=1024000)\n\n# get informations about memory - used, available etc.\nres: dict = qjs.memory()\n```\n\n\n### Set time limit\nTo set time limit in context, use *set_time_limit* method.\n```python\nqjs = QuickJS()\n\nqjs.set_time_limit(time_limit=600)\n```\n\n<!-- Links -->\n\n<!-- Badges -->\n[bsd3-image]: https://img.shields.io/badge/License-BSD_3--Clause-blue.svg\n[bsd3-url]: https://opensource.org/licenses/BSD-3-Clause\n[build-image]: https://img.shields.io/badge/build-success-brightgreen\n[coverage-image]: https://img.shields.io/badge/Coverage-100%25-green\n",
    'author': 'TangledHub',
    'author_email': 'info@tangledgroup.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/tangledlabs/thqyickjs',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
