# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['flake8_type_checking']

package_data = \
{'': ['*']}

install_requires = \
['flake8']

entry_points = \
{'flake8.extension': ['TCH = flake8_type_checking.plugin:Plugin']}

setup_kwargs = {
    'name': 'flake8-type-checking',
    'version': '1.0.0',
    'description': 'A flake8 plugin for managing type-checking imports & forward references',
    'long_description': '<a href="https://pypi.org/project/flake8-type-checking/">\n    <img src="https://img.shields.io/pypi/v/flake8-type-checking.svg" alt="Package version">\n</a>\n<a href="https://codecov.io/gh/sondrelg/flake8-type-checking">\n    <img src="https://codecov.io/gh/sondrelg/flake8-type-checking/branch/master/graph/badge.svg" alt="Code coverage">\n</a>\n<a href="https://pypi.org/project/flake8-type-checking/">\n    <img src="https://github.com/sondrelg/flake8-type-checking/actions/workflows/testing.yml/badge.svg" alt="Test status">\n</a>\n<a href="https://pypi.org/project/flake8-type-checking/">\n    <img src="https://img.shields.io/badge/python-3.7%2B-blue" alt="Supported Python versions">\n</a>\n<a href="http://mypy-lang.org/">\n    <img src="http://www.mypy-lang.org/static/mypy_badge.svg" alt="Checked with mypy">\n</a>\n\n# flake8-type-checking\n\nLets you know which imports to put in [type-checking](https://docs.python.org/3/library/typing.html#typing.TYPE_CHECKING) blocks.\n\nFor the imports you\'ve already defined inside type-checking blocks, it can\nalso help you manage [forward references](https://www.python.org/dev/peps/pep-0484/#forward-references)\nusing [PEP 484](https://www.python.org/dev/peps/pep-0484) or [PEP 563](https://www.python.org/dev/peps/pep-0563/) style references.\n\n## Codes\n\n\n| Code   | Description                                         |\n|--------|-----------------------------------------------------|\n| TC001 | Move import into a type-checking block  |\n| TC002 | Move third-party import into a type-checking block |\n| TC003 | Found multiple type checking blocks |\n| TC004 | Move import out of type-checking block. Import is used for more than type hinting. |\n\n### Forward reference codes\n\nThese code ranges are opt-in. They represent two different ways of solving the same problem,\nso please only choose one.\n\n`TCH100` and `TCH101` manage forward references by taking advantage of\n[postponed evaluation of annotations](https://www.python.org/dev/peps/pep-0563/).\n\n| Code   | Description                                         |\n|--------|-----------------------------------------------------|\n| TC100 | Add \'from \\_\\_future\\_\\_ import annotations\' import |\n| TC101 | Annotation does not need to be a string literal |\n\n`TCH200` and `TCH201` manage forward references using [string literals](https://www.python.org/dev/peps/pep-0484/#forward-references).\n\n| Code   | Description                                         |\n|--------|-----------------------------------------------------|\n| TC200 | Annotation needs to be made into a string literal |\n| TC201 | Annotation does not need to be a string literal |\n\nTo one of the ranges, just specify the code in your flake8 config:\n\n```\n[flake8]\nmax-line-length = 80\nmax-complexity = 12\n...\nignore = E501\nselect = C,E,F,W,..., TC, TC2  # or TC1\n```\n\n## Rationale\n\nGood type hinting requires a lot of imports, which can increase the risk of\n[import cycles](https://mypy.readthedocs.io/en/stable/runtime_troubles.html?highlight=TYPE_CHECKING#import-cycles)\nin your project.\nThe recommended way of preventing this problem is to use `typing.TYPE_CHECKING` blocks\nto guard these types of imports.\n\nBoth `TC001` and `TC002` help alleviate this problem; the reason there are two\ncodes instead of one, is because the import cycles rarely occur from\nlibrary/third-party imports, so this artificial split provides a way to filter down\nthe total pool of imports for users that want to guard against import cycles,\nbut don\'t want to manage every import in their projects *this* strictly.\n\nOnce imports are guarded, they will no longer be evaluated during runtime. The\nconsequence of this is that these imports can no longer be treated as if they\nwere imported outside the block. Instead we need to use [forward references](https://www.python.org/dev/peps/pep-0484/#forward-references).\n\nFor Python version `>= 3.7`, there are actually two ways of solving this issue.\nYou can either make your annotations string literals, or you can use a `__futures__` import to enable [postponed evaluation of annotations](https://www.python.org/dev/peps/pep-0563/).\nSee [this](https://stackoverflow.com/a/55344418/8083459) excellent stackoverflow answer\nfor a better explanation of the differences.\n\n## Installation\n\n```shell\npip install flake8-type-checking\n```\n\n## Examples\n\n**Bad code**\n\n`models/a.py`\n```python\nfrom models.b import B\n\nclass A(Model):\n    def foo(self, b: B): ...\n```\n\n`models/b.py`\n```python\nfrom models.a import A\n\nclass B(Model):\n    def bar(self, a: A): ...\n```\n\nWill result in these errors\n\n```shell\n>> a.py: TC002: Move third-party import \'models.b.B\' into a type-checking block\n>> b.py: TC002: Move third-party import \'models.a.A\' into a type-checking block\n```\n\nand consequently trigger these errors if imports are purely moved into type-checking block, without proper forward reference handling\n\n```shell\n>> a.py: TC100: Add \'from __future__ import annotations\' import\n>> b.py: TC100: Add \'from __future__ import annotations\' import\n```\n\nor\n\n```shell\n>> a.py: TC200: Annotation \'B\' needs to be made into a string literal\n>> b.py: TC200: Annotation \'A\' needs to be made into a string literal\n```\n\n**Good code**\n\n`models/a.py`\n```python\nfrom typing import TYPE_CHECKING\n\nif TYPE_CHECKING:\n    from models.b import B\n\nclass A(Model):\n    def foo(self, b: \'B\'): ...\n```\n`models/b.py`\n```python\n# TCH1\nfrom __future__ import annotations\n\nfrom typing import TYPE_CHECKING\n\nif TYPE_CHECKING:\n    from models.a import A\n\nclass B(Model):\n    def bar(self, a: A): ...\n```\n\nor\n\n```python\n# TC2\nfrom typing import TYPE_CHECKING\n\nif TYPE_CHECKING:\n    from models.a import A\n\nclass B(Model):\n    def bar(self, a: \'A\'): ...\n```\n\n## As a pre-commit hook\n\nYou can run this flake8 plugin as a [pre-commit](https://github.com/pre-commit/pre-commit) hook:\n\n```yaml\n- repo: https://gitlab.com/pycqa/flake8\n  rev: 3.7.8\n  hooks:\n    - id: flake8\n      additional_dependencies: [ flake8-type-checking ]\n```\n\n## Supporting the project\n\nContributions are always welcome, and leaving a ⭐️\nis always useful as it helps raise the profile of the repo 🚀\n',
    'author': 'Sondre Lillebø Gundersen',
    'author_email': 'sondrelg@live.no',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/snok',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
