"""Dockerfile Generator"""

import os
import re
from ..execution import commandBuilder

FILE_PATH = "{path}/Dockerfile"

PY_INSTALL = "RUN [\"pip\", \"install\", \"{dep}\"]\n"
PY_INSTALL_VERSION = "RUN [\"pip\", \"install\", \"{depName}=={version}\"]\n"
PY_INSTALL_GITHUB = "RUN [\"pip\", \"install\", \"git+{depPath}\"]\n"
PY_INSTALL_FILE = "COPY {depPath} {depPath}\n"
PY_INSTALL_FILE += "RUN [\"pip\", \"install\", \"/app/{depPath}\"]\n"
PY_R_INSTALL = "RUN [\"pip3\", \"install\", \"{dep}\"]\n"
PY_R_INSTALL_VERSION = "RUN [\"pip3\", \"install\", \"{depName}=={version}\"]\n"
PY_R_INSTALL_GITHUB = "RUN [\"pip3\", \"install\", \"git+{depPath}\"]\n"
PY_R_INSTALL_FILE = "COPY {depPath} {depPath}\n"
PY_R_INSTALL_FILE += "RUN [\"pip3\", \"install\", \"/app/{depPath}\"]\n"
R_INSTALL = "RUN [\"Rscript\", \"-e\", \"install.packages('{dep}', repo='https://cloud.r-project.org'); library({dep})\"]\n"
R_INSTALL_VERSION = "RUN [\"Rscript\", \"-e\", \"library(devtools); install_version('{depName}', version='{version}', repos='http://cran.us.r-project.org'); library({depName})\"]\n"
R_INSTALL_GITHUB = "RUN [\"Rscript\", \"-e\", \"library(devtools); install_github('{depPath}'); library({depName})\"]\n"
R_INSTALL_FILE = "COPY {depPath} {depPath}\n"
R_INSTALL_FILE += "RUN [\"Rscript\", \"-e\", \"install.packages('/app/{depPath}', repos=NULL, type='source'); library({depName})\"]\n"
TIMEZONE = "ENV TZ={timezone}\nRUN ln -snf /usr/share/zoneinfo/$TZ /etc/localtime && echo $TZ > /etc/timezone\n"

DOCKERFILE = """
# This Dockerfile was generated by Skelebot
# Editing this file manually is not advised as all changes will be overwritten by Skelebot

"""

def buildDockerfile(config):
    """Generates the Dockerfile based on values from the Config object"""

    # Setup the basics of all dockerfiless
    docker = DOCKERFILE
    docker += "FROM {baseImage}\n".format(baseImage=config.getBaseImage())
    docker += "MAINTAINER {maintainer} <{contact}>\n".format(maintainer=config.maintainer, contact=config.contact)
    docker += "WORKDIR /app\n"

    if (config.timezone is not None):
        docker += TIMEZONE.format(timezone=config.timezone)

    # Add language dependencies
    if (config.language == "Python"):
        for dep in config.dependencies:
            depSplit = dep.split(":", maxsplit=1)
            if ("github:" in dep):
                docker += PY_INSTALL_GITHUB.format(depPath=depSplit[1])
            elif ("file:" in dep):
                docker += PY_INSTALL_FILE.format(depPath=depSplit[1])
            # if using PIP version specifiers, will be handled as a standard case
            elif dep.count("=") == 1 and not re.search(r"[!<>~]", dep):
                verSplit = dep.split("=")
                docker += PY_INSTALL_VERSION.format(depName=verSplit[0], version=verSplit[1])
            else:
                docker += PY_INSTALL.format(dep=dep)
    if (config.language == "R"):
        for dep in config.dependencies:
            depSplit = dep.split(":")
            if ("github:" in dep):
                docker += R_INSTALL_GITHUB.format(depPath=depSplit[1], depName=depSplit[2])
            elif ("file:" in dep):
                docker += R_INSTALL_FILE.format(depPath=depSplit[1], depName=depSplit[2])
            elif ("=" in dep):
                verSplit = dep.split("=")
                docker += R_INSTALL_VERSION.format(depName=verSplit[0], version=verSplit[1])
            else:
                docker += R_INSTALL.format(dep=dep)

    if (config.language == "R+Python"):
        for dep in config.dependencies["Python"]:
            depSplit = dep.split(":", maxsplit=1)
            if ("github:" in dep):
                docker += PY_R_INSTALL_GITHUB.format(depPath=depSplit[1])
            elif ("file:" in dep):
                docker += PY_R_INSTALL_FILE.format(depPath=depSplit[1])
            # if using PIP version specifiers, will be handled as a standard case
            elif dep.count("=") == 1 and not re.search(r"[!<>~]", dep):
                verSplit = dep.split("=")
                docker += PY_R_INSTALL_VERSION.format(depName=verSplit[0], version=verSplit[1])
            else:
                docker += PY_R_INSTALL.format(dep=dep)
        for dep in config.dependencies["R"]:
            depSplit = dep.split(":")
            if ("github:" in dep):
                docker += R_INSTALL_GITHUB.format(depPath=depSplit[1], depName=depSplit[2])
            elif ("file:" in dep):
                docker += R_INSTALL_FILE.format(depPath=depSplit[1], depName=depSplit[2])
            elif ("=" in dep):
                verSplit = dep.split("=")
                docker += R_INSTALL_VERSION.format(depName=verSplit[0], version=verSplit[1])
            else:
                docker += R_INSTALL.format(dep=dep)

    # Copy the project into the /app folder of the Docker Image
    # Ignores anything in the .dockerignore file of the project
    docker += "COPY . /app\n"

    # Run any custom global commands
    for command in config.commands:
        docker += "RUN {command}\n".format(command=command)

    # Pull in any additional dockerfile updates from the components
    for component in config.components:
        docker += component.appendDockerfile()

    # Set the CMD to execute the primary job by default (if there is one)
    for job in config.jobs:
        if config.primaryJob == job.name:

            if "CMD" == config.primaryExe:
                command = commandBuilder.build(config, job, None)
                docker += "CMD /bin/bash -c \"{command}\"\n".format(command=command)
            elif "ENTRYPOINT" == config.primaryExe:
                # ENTRYPOINT execution ignores any job and global arguments and parameters
                job_args = job.args
                job_params = job.params
                config_params = config.params
                job.args, job.params, config.params = [], [], []

                command = commandBuilder.build(config, job, None)
                commandParts = command.split(" ")
                docker += 'ENTRYPOINT ["{}"]\n'.format('", "'.join(commandParts))

                # Restore job and config for any downstream jobs
                job.args, job.params, config.params = job_args, job_params, config_params

            break

    dockerfile = open(FILE_PATH.format(path=os.getcwd()), "w")
    dockerfile.write(docker)
    dockerfile.close()
