import { __rest } from "tslib";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import React, { forwardRef, useState } from 'react';
import clsx from 'clsx';
import { fireCancelableEvent, isPlainLeftClick } from '../../../internal/events';
import { useUniqueId } from '../../../internal/hooks/use-unique-id';
import useFocusVisible from '../../../internal/hooks/focus-visible';
import InternalIcon from '../../../icon/internal';
import { useNavigate } from './router';
import styles from '../../styles.css.js';
const ListItem = ({ children, startIcon, endIcon }) => {
    return (React.createElement(React.Fragment, null,
        startIcon && React.createElement("span", { className: styles['overflow-menu-list-item-icon'] }, startIcon),
        React.createElement("span", { className: styles['overflow-menu-list-item-text'] }, children),
        endIcon && endIcon));
};
const LinkItem = forwardRef(({ children, external, href, startIcon, endIcon, onFollow, context, testId }, ref) => {
    const focusVisible = useFocusVisible();
    const rel = external ? 'noopener noreferrer' : undefined;
    const target = external ? '_blank' : undefined;
    const anchorProps = {
        rel,
        target,
        href,
        onClick(event) {
            if (isPlainLeftClick(event)) {
                onFollow === null || onFollow === void 0 ? void 0 : onFollow(event);
            }
        },
    };
    const buttonProps = {
        role: 'button',
        tabIndex: 0,
        onKeyDown(event) {
            if (event.key === ' ') {
                event.preventDefault();
            }
        },
        onKeyUp(event) {
            if (event.key === ' ' || event.key === 'Enter') {
                onFollow === null || onFollow === void 0 ? void 0 : onFollow(event);
            }
        },
        onClick: onFollow,
    };
    return (React.createElement("a", Object.assign({ ref: ref, className: clsx(styles['overflow-menu-control'], styles['overflow-menu-control-link'], context && styles[`overflow-menu-control-${context}`]) }, (typeof href === 'string' ? anchorProps : buttonProps), focusVisible, (testId ? { 'data-testid': testId } : {})),
        React.createElement(ListItem, { startIcon: startIcon, endIcon: endIcon }, children)));
});
const ButtonItem = forwardRef(({ children, startIcon, endIcon, onFollow: onClick, testId }, ref) => {
    const focusVisible = useFocusVisible();
    return (React.createElement("button", Object.assign({ ref: ref, className: styles['overflow-menu-control'], onClick: onClick }, focusVisible, (typeof testId === 'string' ? { 'data-testid': testId } : {})),
        React.createElement(ListItem, { startIcon: startIcon, endIcon: endIcon }, children)));
});
const NavigationItem = forwardRef((_a, ref) => {
    var { startIcon, children, index, testId } = _a, definition = __rest(_a, ["startIcon", "children", "index", "testId"]);
    const navigate = useNavigate();
    return (React.createElement(ButtonItem, { ref: ref, startIcon: startIcon, endIcon: React.createElement(InternalIcon, { name: "angle-right" }), testId: testId, onFollow: () => navigate('dropdown-menu', {
            definition,
            headerText: definition.text || definition.title,
            headerSecondaryText: definition.description,
            utilityIndex: index,
        }) }, children));
});
const ExpandableItem = (_a) => {
    var { children, onItemClick } = _a, definition = __rest(_a, ["children", "onItemClick"]);
    const focusVisible = useFocusVisible();
    const [expanded, setExpanded] = useState(false);
    const headerId = useUniqueId('overflow-menu-item');
    return (React.createElement(React.Fragment, null,
        React.createElement("button", Object.assign({ className: clsx(styles['overflow-menu-control'], styles['overflow-menu-control-expandable-menu-trigger']), onClick: () => setExpanded(value => !value), "aria-expanded": expanded }, focusVisible),
            React.createElement(ListItem, { endIcon: React.createElement("span", { className: clsx(styles.icon, expanded && styles.expanded) },
                    React.createElement(InternalIcon, { name: "caret-up-filled" })) },
                React.createElement("span", { id: headerId }, children))),
        expanded && (React.createElement("ul", { className: clsx(styles['overflow-menu-list'], styles['overflow-menu-list-submenu']), "aria-labelledby": headerId }, definition.items.map((item, index) => {
            const isGroup = typeof item.items !== 'undefined';
            return (React.createElement("li", { key: index, className: clsx(styles[`overflow-menu-list-item`], styles[`overflow-menu-list-item-dropdown-menu`]) }, dropdownComponentFactory(item, isGroup, onItemClick)));
        })))));
};
function utilityComponentFactory(utility, index, ref) {
    const label = utility.text || utility.title;
    const hasIcon = !!utility.iconName || !!utility.iconUrl || !!utility.iconAlt || !!utility.iconSvg;
    const startIcon = hasIcon && (React.createElement(InternalIcon, { name: utility.iconName, url: utility.iconUrl, alt: utility.iconAlt, svg: utility.iconSvg }));
    switch (utility.type) {
        case 'button': {
            const handleClick = (event) => {
                fireCancelableEvent(utility.onClick, {}, event);
            };
            if (utility.variant === 'primary-button') {
                return (React.createElement(ButtonItem, { ref: ref, startIcon: startIcon, onFollow: handleClick, testId: `__${index}` }, label));
            }
            return (React.createElement(LinkItem, { ref: ref, startIcon: startIcon, href: utility.href, external: utility.external, testId: `__${index}`, onFollow: handleClick },
                label,
                utility.external && (React.createElement(React.Fragment, null,
                    ' ',
                    React.createElement("span", { "aria-label": utility.externalIconAriaLabel, role: utility.externalIconAriaLabel ? 'img' : undefined },
                        React.createElement(InternalIcon, { name: "external", size: "normal" }))))));
        }
        case 'menu-dropdown': {
            return (React.createElement(NavigationItem, Object.assign({ ref: ref, startIcon: startIcon, index: index }, utility, { testId: `__${index}` }), label));
        }
    }
}
function dropdownComponentFactory(item, expandable, onItemClick) {
    const label = item.text;
    const hasIcon = !!item.iconName || !!item.iconUrl || !!item.iconAlt || !!item.iconSvg;
    const startIcon = hasIcon && (React.createElement(InternalIcon, { name: item.iconName, url: item.iconUrl, alt: item.iconAlt, svg: item.iconSvg }));
    if (expandable) {
        return (React.createElement(ExpandableItem, Object.assign({}, item, { onItemClick: onItemClick }), label));
    }
    return (React.createElement(LinkItem, { startIcon: startIcon, href: item.href, external: item.external, context: "dropdown-menu", testId: item.id, onFollow: () => onItemClick(item) },
        label,
        item.external && (React.createElement(React.Fragment, null,
            ' ',
            React.createElement("span", { "aria-label": item.externalIconAriaLabel, role: item.externalIconAriaLabel ? 'img' : undefined },
                React.createElement(InternalIcon, { name: "external", size: "normal" }))))));
}
export const UtilityMenuItem = forwardRef((_a, ref) => {
    var { index } = _a, props = __rest(_a, ["index"]);
    return (React.createElement("li", { className: clsx(styles[`overflow-menu-list-item`], styles[`overflow-menu-list-item-utility`]) }, utilityComponentFactory(props, index, ref)));
});
export const SubmenuItem = (props) => {
    const expandable = typeof props.items !== 'undefined';
    return (React.createElement("li", { className: clsx(styles[`overflow-menu-list-item`], styles[`overflow-menu-list-item-submenu`], expandable && styles[`overflow-menu-list-item-expandable`]) }, dropdownComponentFactory(props, expandable, props.onItemClick)));
};
//# sourceMappingURL=menu-item.js.map