# Locating distributional changes in piece-wise stationary time-series with long-range dependencies

Implementations of the change-point estimation algorithms proposed in the following two papers.

 - A. Khaleghi, D. Ryabko, Asymptotically Consistent Estimation of the Number of Change Points in Highly Dependent Time Series In Proceedings of the International Conference on Machine Learning, 2014.

 - A. Khaleghi, D. Ryabko, Locating Changes in Highly-Dependent Data with an Unknown Number of Change-Points, In Proceedings of Neural Information Processing Systems, 2012.


written in Go and distributed in [Go](#go) and [Python](#python).


## Python
### Installing

The Python package is precompiled for Windows (32/64 bit), MacOs (64 bit) and many Linux (32/64 bit) distributions. If you want to use the code on system for which there are no precompiled binaries like Raspberry Pis or Android phones head to the [Compiling](#compiling-the-go-code) section in the Appendix.

If Python 3 is installed on your system simply run the following code

```
python3 -m pip i PyChest
```

### How to use

To find changepoint positions in a sequence (given as either a python list or a numpy array), with a minimum distance of `0.03 * len(sequence)` between changepoints which have been generated by `2` different processes simply call:

```Python
import PyChest
estimates = PyChest.find_changepoints(sequence, 0.03, 2)
```

`estimates` will now be a python list containing a list of changepoints in increasing order. The list can be emtpy. 

If you do not know the number of generating processes you can call the `list_estimator` instead like following

```Python
import PyChest
estimates = PyChest.list_estimator(sequence, 0.03)
```

Now `estimates` is a list of changepoints at least `0.03 * len(sequence)` apart in decreasing likelihood. The list can be empty. 

## Go
### Installing
Install the Go source code by entering the following line in the console:
```
go get github.com/LukasZierahn/GoChest
```

Go will now complain that the directory layout is unexpected. That is because of the duality of it being a Python and Go package and can safely be ignored.

### How to use
 
 To find changepoint positions in a sequence (given as a slice), with a minimum distance of `0.03 * len(sequence)` between changepoints which have been generated by `2` different processes simply call:

```go
import (
    "github.com/LukasZierahn/GoChest/GoChest"
)
func main() {
    sequence := make([]float64, 0)

    // fill sequence somehow

    estimates = GoChest.FindChangepoints(sequence, 0.03, 2)
}
```

`estimates` will now be a slice containing a list of changepoints in increasing order. The slice can be emtpy. 

If you do not know the number of generating processes you can call the `ListEstimator` instead like following

```go
import (
    "github.com/LukasZierahn/GoChest/GoChest"
)

func main() {
    sequence := make([]float64, 0)

    // fill sequence somehow

    estimates = GoChest.ListEstimator(sequence, 0.03)
}
```

Now `estimates` is a slice of changepoints at least `0.03 * len(sequence)` apart in decreasing likelihood. The slice can be empty. 


## How to cite

If you use this package, please cite the following papers:

 - A. Khaleghi, D. Ryabko, Asymptotically Consistent Estimation of the Number of Change Points in Highly Dependent Time Series In Proceedings of the International Conference on Machine Learning, 2014.

 - A. Khaleghi, D. Ryabko, Locating Changes in Highly-Dependent Data with an Unknown Number of Change-Points, In Proceedings of Neural Information Processing Systems, 2012.



## Appendix
### Compiling the Go code
You should not need to compile the Go code yourself if you just want to use the package on Windows, MacOS or a common Unix distribution, you can find instructions how to install the code here: [Python](#python).

If you want to compile the Go code yourself you need to install [Go](https://golang.org/) first, then clone and enter the repository:
```
git clone https://github.com/LukasZierahn/GoChest
cd GoChest
```

Next you will need to actually compile the code, for Windows enter the following line in the console to do that
```bat
go build -buildmode=c-shared -o PyChestBuild/GoChest.dll CWrapper.go
```
And for MacOS and Linux
```bat
go build -buildmode=c-shared -o PyChestBuild/GoChest.so CWrapper.go
```

Your compiled c-shared library is now situated in the `PyChestBuild` folder. To complete the installation with simply type:

```
python3 -m pip install .
```

The package will now prefer your compiled library over the precompiled libraries.

#### Cross Compiling
Cross compiling, for example to make the package available on Android phones, is a little more complicated. In order to build the code you will need to enable cgo specifically and provide a c-compiler for your target platform. Your final command should look something like this:
```
CGO_CFLAGS="-g -O2 -w" CGO_ENABLED=1 GOOS=windows GOARCH=amd64 CC=ValidCCompilerForThePlatform go build -buildmode=c-shared -o PyChestBuild/GoChest.dll CWrapper.go
```
Which would compile for windows/amd64 architecture given that the `CC=ValidCCompilerForThePlatform` is replaced by an actual c-compiler for windows/amd64.

I recommend compiling on system wherever possible.
