"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ToSqs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const constructs_1 = require("constructs");
const defaults = require("@aws-solutions-constructs/core");
const s3n = require("aws-cdk-lib/aws-s3-notifications");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const core_1 = require("@aws-solutions-constructs/core");
/**
 * @summary The S3ToSqs class.
 */
class S3ToSqs extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the S3ToSqs class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {S3ToSqsProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // All our tests are based upon this behavior being on, so we're setting
        // context here rather than assuming the client will set it
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        defaults.CheckProps(props);
        let bucket;
        let enableEncryptionParam = props.enableEncryptionWithCustomerManagedKey;
        if (props.enableEncryptionWithCustomerManagedKey === undefined ||
            props.enableEncryptionWithCustomerManagedKey === true) {
            enableEncryptionParam = true;
        }
        // Setup the S3 bucket
        if (!props.existingBucketObj) {
            const buildS3BucketResponse = defaults.buildS3Bucket(this, {
                bucketProps: props.bucketProps,
                loggingBucketProps: props.loggingBucketProps,
                logS3AccessLogs: props.logS3AccessLogs
            });
            this.s3Bucket = buildS3BucketResponse.bucket;
            this.s3LoggingBucket = buildS3BucketResponse.loggingBucket;
            bucket = this.s3Bucket;
        }
        else {
            bucket = props.existingBucketObj;
        }
        this.s3BucketInterface = bucket;
        // Setup the dead letter queue, if applicable
        this.deadLetterQueue = defaults.buildDeadLetterQueue(this, {
            existingQueueObj: props.existingQueueObj,
            deployDeadLetterQueue: props.deployDeadLetterQueue,
            deadLetterQueueProps: props.deadLetterQueueProps,
            maxReceiveCount: props.maxReceiveCount
        });
        // Setup the queue
        const buildQueueResponse = defaults.buildQueue(this, 'queue', {
            existingQueueObj: props.existingQueueObj,
            queueProps: props.queueProps,
            deadLetterQueue: this.deadLetterQueue,
            enableEncryptionWithCustomerManagedKey: enableEncryptionParam,
            encryptionKey: props.encryptionKey,
            encryptionKeyProps: props.encryptionKeyProps
        });
        this.sqsQueue = buildQueueResponse.queue;
        this.encryptionKey = buildQueueResponse.key;
        // Setup the S3 bucket event types
        let s3EventTypes;
        if (!props.s3EventTypes) {
            s3EventTypes = defaults.defaultS3NotificationEventTypes;
        }
        else {
            s3EventTypes = props.s3EventTypes;
        }
        // Setup the S3 bucket event filters
        let s3Eventfilters = [];
        if (props.s3EventFilters) {
            s3Eventfilters = props.s3EventFilters;
        }
        // Setup the S3 bucket event notifications
        s3EventTypes.forEach(type => bucket.addEventNotification(type, new s3n.SqsDestination(this.sqsQueue), ...s3Eventfilters));
        core_1.addCfnNagS3BucketNotificationRulesToSuppress(aws_cdk_lib_1.Stack.of(this), 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834');
    }
}
exports.S3ToSqs = S3ToSqs;
_a = JSII_RTTI_SYMBOL_1;
S3ToSqs[_a] = { fqn: "@aws-solutions-constructs/aws-s3-sqs.S3ToSqs", version: "2.42.0" };
//# sourceMappingURL=data:application/json;base64,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