# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/oco2peak-nc4_convert.ipynb (unless otherwise specified).

__all__ = ['get_file_list', 'get_nc4_raw_content', 'get_np_table', 'columns', 'get_dataframe', 'get_pattern_yearmonth',
           'process_files']

# Cell
import os
import glob
from netCDF4 import Dataset
import numpy as np
import pandas as pd
from fastprogress.fastprogress import master_bar, progress_bar

# Cell
def get_file_list(nc4_directory, pattern='*.nc4'):
    nc4_directory = os.path.realpath(nc4_directory)
    return glob.glob(nc4_directory + '/' + pattern)

# Cell
def get_nc4_raw_content(one_file):
    return Dataset(one_file, 'r')

# Cell
columns=['flag','sounding_id', 'latitude', 'longitude', 'xco2', 'xco2_uncert', 'orbit', 'windspeed_u', 'windspeed_v',
    'surface_pressure_apriori', 'surface_pressure', 'altitude', 'land_water_indicator', 'land_fraction', 'tcwv', 'tcwv_apriori', 'tcwv_uncertainty']
def get_np_table(one_file):
    # Open the file
    try:
        file_nc = Dataset(one_file, 'r')
    except:
        print('ERROR reading', one_file)
        return np.empty((0,len(columns)))
    # Documentation of data : https://docserver.gesdisc.eosdis.nasa.gov/public/project/OCO/OCO2_DUG.V9.pdf
    #print(file_nc)
    if file_nc.BuildId[0:3] == 'B10' :
        np_table = np.column_stack((file_nc.variables['xco2_quality_flag'],file_nc.variables['sounding_id'],file_nc.variables['latitude'],file_nc.variables['longitude'],
            file_nc.variables['xco2'],file_nc.variables['xco2_uncertainty'],file_nc.groups['Sounding'].variables['orbit'],
            file_nc.groups['Meteorology']['windspeed_u_met'], file_nc.groups['Meteorology']['windspeed_v_met'],
            file_nc.groups['Meteorology']['psurf_apriori_o2a'], file_nc.groups['Retrieval']['psurf'],
            file_nc.groups['Sounding']['altitude'], file_nc.groups['Sounding']['land_water_indicator'], file_nc.groups['Sounding']['land_fraction'],
            file_nc.groups['Retrieval']['tcwv'], file_nc.groups['Retrieval']['tcwv_apriori'], file_nc.groups['Retrieval']['tcwv_uncertainty']))
    else:
        np_table = np.column_stack((file_nc.variables['xco2_quality_flag'],file_nc.variables['sounding_id'],file_nc.variables['latitude'],file_nc.variables['longitude'],
            file_nc.variables['xco2'],file_nc.variables['xco2_uncertainty'],file_nc.groups['Sounding'].variables['orbit'],
            file_nc.groups['Meteorology']['windspeed_u_met'], file_nc.groups['Meteorology']['windspeed_v_met'],
            file_nc.groups['Meteorology']['psurf_apriori'], file_nc.groups['Retrieval']['psurf'], file_nc.groups['Sounding']['altitude'], file_nc.groups['Sounding']['land_water_indicator'],
            file_nc.groups['Sounding']['land_fraction'],
            file_nc.groups['Retrieval']['tcwv'], file_nc.groups['Retrieval']['tcwv_apriori'], file_nc.groups['Retrieval']['tcwv_uncertainty']))#))
    return np_table

# Cell
def get_dataframe(nc4_list, master_progress_bar = None):
    global columns
    month_data = np.empty((0,len(columns)))
    # Loop over the files
    if master_progress_bar is None:
        master_progress_bar = master_bar([0])
        for _ in master_progress_bar: None

    for one_file in progress_bar(nc4_list, parent=master_progress_bar):
        np_table = get_np_table(one_file)
        month_data = np.concatenate((month_data, np_table), axis=0)

    if(month_data.size == 0):
        return pd.DataFrame(columns=columns)
    df = pd.DataFrame(month_data, columns=columns)
    # using dictionary to convert specific columns (https://www.geeksforgeeks.org/change-data-type-for-one-or-more-columns-in-pandas-dataframe/)
    convert_dict = {'sounding_id': int,
                    'orbit': int
                }
    df = df.astype(convert_dict)
    # Remove bad quality
    df=df[df['flag']==0]
    # Remove flag
    df.drop(['flag'], axis=1, inplace=True)
    return df

# Cell
def get_pattern_yearmonth():
    '''
    Generate a list of every YYMM from 2014 to 2020
    '''
    years_months = []
    for year in range(14, 20+1):
        for month in range(1,12+1):
            years_months.append(str(year)+str(month).zfill(2))
    return years_months

def process_files(input_dir, output_dir, patterns):
    '''
    Process all NC4 file corresponding to the patterns list.
    '''
    if len(patterns) < 1:
        raise Exception("ERROR You must give an array pattern !")
    master_progress_bar = master_bar(patterns)
    for pattern in master_progress_bar:
        # Get the file list in directory
        nc4_list = get_file_list(input_dir, pattern='oco2_LtCO2_'+pattern+"*.nc4")
        master_progress_bar.write(f'Files to process for {pattern} : {len(nc4_list)}')
        if len(nc4_list) > 1:
            #master_progress_bar.write(f'Loading {pattern}')
            df = get_dataframe(nc4_list, master_progress_bar)
            master_progress_bar.write(f'Saving {pattern} to disk...')
            df.to_csv(output_dir + 'oco2_'+pattern+'.csv.bz2', sep=';', index=False, compression='bz2')
            del(df)
        else:
            master_progress_bar.write(f'WARNING : No file for {pattern}')