"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlane = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class KongControlPlane extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // TODO : Dont know why, but pod SA still wants node to lookup Describe* calls on PCA
        props.nodegroup.role.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
                'secretsmanager:ListSecretVersionIds',
            ],
        }));
        const external_secrets_helm = props.cluster.addHelmChart('ExternalSecretsHandler', {
            repository: 'https://external-secrets.github.io/kubernetes-external-secrets/',
            chart: 'kubernetes-external-secrets',
            release: 'kubernetes-external-secrets',
            namespace: props.namespace,
            wait: true,
            values: {
                securityContext: {
                    fsGroup: 65534,
                },
                env: {
                    AWS_REGION: aws_cdk_lib_1.Stack.of(this).region,
                },
            },
        });
        const secretNameParts = aws_cdk_lib_1.Fn.split('-', (_a = props.rds.secret) === null || _a === void 0 ? void 0 : _a.secretName);
        const secretNameWithoutSuffix = aws_cdk_lib_1.Fn.join('-', [aws_cdk_lib_1.Fn.select(0, secretNameParts), aws_cdk_lib_1.Fn.select(1, secretNameParts)]);
        const secret_provider_class = {
            apiVersion: 'kubernetes-client.io/v1',
            kind: 'ExternalSecret',
            metadata: {
                name: 'rds-secrets',
                namespace: props.namespace,
            },
            spec: {
                backendType: 'secretsManager',
                data: [
                    {
                        key: secretNameWithoutSuffix,
                        name: 'password',
                        property: 'password',
                    },
                    {
                        key: secretNameWithoutSuffix,
                        name: 'host',
                        property: 'host',
                    },
                    {
                        key: secretNameWithoutSuffix,
                        name: 'username',
                        property: 'username',
                    },
                    {
                        key: secretNameWithoutSuffix,
                        name: 'dbname',
                        property: 'dbname',
                    },
                ],
            },
        };
        const license_key_manifest = {
            apiVersion: 'kubernetes-client.io/v1',
            kind: 'ExternalSecret',
            metadata: {
                name: 'kong-license',
                namespace: props.namespace,
            },
            spec: {
                backendType: 'secretsManager',
                data: [
                    {
                        key: props.license_secret_name,
                        name: 'license',
                    },
                ],
            },
        };
        const kong_rds_secrets = props.cluster.addManifest('KongRdsSecrets', secret_provider_class, license_key_manifest);
        kong_rds_secrets.node.addDependency(external_secrets_helm);
        const kong_helm = props.cluster.addHelmChart('KongCpHelm', {
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                ingressController: {
                    enabled: true,
                    installCRDs: false,
                    image: {
                        repository: 'kong/kubernetes-ingress-controller',
                        tag: '1.3.1-alpine',
                    },
                },
                image: {
                    repository: 'kong/kong-gateway',
                    tag: '2.5.0.0-alpine',
                },
                env: {
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.crt',
                    cluster_cert_key: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.key',
                    cluster_ca_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/ca.crt',
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'username',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'password',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'host',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: 'true',
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                        servicePort: 8005,
                        containerPort: 8005,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'cluster.kong-cp.internal',
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'telemetry.kong-cp.internal',
                    },
                },
                proxy: {
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=proxy',
                        'external-dns.alpha.kubernetes.io/hostname': 'proxy.kong-cp.internal',
                    },
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    http: {
                        enabled: true,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': 'admin.kong-cp.internal',
                    },
                },
                enterprise: {
                    enabled: true,
                    license_secret: 'kong-license',
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'manager.kong-cp.internal',
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    'kong-cp-cluster-issuer-secret',
                ],
            },
        });
        kong_helm.node.addDependency(kong_rds_secrets);
        //TODO : https://github.com/aws/aws-cdk/issues/14933
        // const kong_admin = new KubernetesObjectValue(this, 'KongAdminAttribute', {
        //   cluster: props.cluster,
        //   objectType: 'service',
        //   objectName: 'kong-kong-admin',
        //   jsonPath: '.status.loadBalancer.ingress[0].hostname',
        //   timeout: Duration.minutes(30),
        //   objectNamespace: props.namespace,
        // });
        // const kong_telemetry = new KubernetesObjectValue(this, 'KongTelemetryAttribute', {
        //   cluster: props.cluster,
        //   objectType: 'service',
        //   objectName: 'kong-kong-clustertelemetry',
        //   jsonPath: '.status.loadBalancer.ingress[0].hostname',
        //   timeout: Duration.minutes(30),
        //   objectNamespace: props.namespace,
        // });
        // const kong_cluster = new KubernetesObjectValue(this, 'KongClusterAttribute', {
        //   cluster: props.cluster,
        //   objectType: 'service',
        //   objectName: 'kong-kong-cluster',
        //   jsonPath: '.status.loadBalancer.ingress[0].hostname',
        //   timeout: Duration.minutes(30),
        //   objectNamespace: props.namespace,
        // });
        // // const kong_cluster= props.cluster.getServiceLoadBalancerAddress('kong-kong-cluster', { namespace: props.namespace }) ;
        // // const kong_telemetry = props.cluster.getServiceLoadBalancerAddress('kong-kong-clustertelemetry', { namespace: props.namespace }) ;
        // new CfnOutput(this, 'KongAdminOutput', { value: kong_admin.value, description: 'Kong Admin Endpoint' });
        // new CfnOutput(this, 'KongTelemetryOutput', { value: kong_telemetry.value, description: 'Kong Telemetry Endpoint' });
        // new CfnOutput(this, 'KongClusterOutput', { value: kong_cluster.value, description: 'Kong Cluster Endpoint' });
    }
}
exports.KongControlPlane = KongControlPlane;
//# sourceMappingURL=data:application/json;base64,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