# -*- coding: utf-8 -*-
# #############################################################################
# Copyright (C) 2025 manatlan manatlan[at]gmail(dot)com
#
# MIT licence
#
# https://github.com/manatlan/reqman4
# #############################################################################
import logging
import json
import html
import httpx
from urllib.parse import unquote

# reqman imports
from . import common

logger = logging.getLogger(__name__)



def prettify(body:bytes) -> str:
    if not body:
        return ""
    else:
        try:
            return json.dumps(eval(body.decode()), indent=2, sort_keys=True,ensure_ascii=False)
        except:
            try:
                return json.dumps(json.loads(body), indent=2, sort_keys=True,ensure_ascii=False)
            except:
                try:
                    return html.escape(body.decode())
                except:
                    return html.escape(str(body))

def generate_base() -> str:
    return """
<meta charset="UTF-8">
<style>
body {font-family: 'Inter', sans-serif;}
h2 {color:blue}
h3 {width:100%;padding:0px;margin:0px}
div.request {margin-left:10px}
div.click {cursor:pointer;background:#F0F0F0;border-radius:4px;padding:4px}
div.request.hide div.detail {display:None}
div.detail {padding-left:10px}
.hideresponse {display:None}
pre {padding:4px;border:1px solid #CCC;max-height:300px;margin:2px;width:99%;display:block;overflow:auto;background:#F8F8F8;font-size:0.8em;border-radius:4px}
pre.request {}
pre.response {}
div.doc {}
span.status {float:right;color:#888}
ul.tests li.True {color:green}
ul.tests li.False {color:red}
ul.tests li.None {color:#D00;font-weight:800}
div.final {position:fixed; top:0px;right:0px;background:white;padding:4px;border-radius:4px}
</style>
"""

def generate_section(file:str) -> str:
    return f"<h2>{file}</h2>"

def generate_error(ex:Exception):
    if isinstance(ex,common.ExeError):
        return f"""
            <h3 style='color:red;border-left:8px solid red;padding-left:8px'>{html.escape(ex.message)}
                <div style='color:black;font-size:0.9em'>in "{html.escape(ex.filename)}" at line {ex.line}</div>
            </h3>
        """
    else:
        return f"<h3 style='color:red'>{html.escape(str(ex))}</h3>"


def generate_request(r:common.Result) -> str:
    def h(d:httpx.Headers) -> str:
        ll = list(dict(d).items())
        return "\n".join( [f"<b>{k}:</b> {v}" for k,v in ll] )
    def c(body:bytes) -> str:
        r=prettify(body)
        return "\n\n"+r if r else ""
    def t(ll:list[common.TestResult]) -> str:
        items = []
        for tr in ll:
            items.append(f"""<li class={tr.ok} title="{html.escape(tr.ctx)}">{str(tr)} : {tr.text}</li>""")
        return "\n".join(items)

    if r.response.status_code<=0:
        status = "❌"  
        class_hide_response="hideresponse"
    else: 
        status = str(r.response.status_code)
        class_hide_response=""

    try:
        elapsed = r.response.elapsed
    except:
        try:
            elapsed = r.response.error  #TODO: do better here
        except:
            elapsed = "test-server"
    return f"""
<div class="request hide">
    <div class="click" onclick="this.parentElement.classList.toggle('hide')" title="Click to show/hide details">
        <h3>{r.request.method} {unquote( str(r.request.url) )} <span class="status" title="{elapsed}">{status}</span></h3>
        <div class="doc">{r.doc}</div>
    </div>

    <div class="detail">
<pre class="request" title="request">
{r.request.method} {unquote( str(r.request.url) )}
{h(r.request.headers)}{c(r.request.content)}
</pre>
<span class='{class_hide_response}'>➔ {r.response.http_version} {r.response.status_code} {r.response.reason_phrase}
<pre class="response" title="response">
{h(r.response.headers)}{c(r.response.content)}
</pre>
</span>
    </div>

    <ul class="tests">
        {t(r.tests)}
    </ul>
</div>
"""

def generate_final(switchs:tuple, nb_ok:int, nb_tests:int, error) -> str:
    switch = ", ".join(switchs)
    if error:
        msg="ERROR"
    else:
        msg=f"{nb_ok}/{nb_tests}"

    title = f"<title>{switch or ''} {msg}</title>"
    return f"<div class='final'>{switch+'<br>' if switch else ''}{msg}</div>" + title


