from typing import Dict, List, Optional, Union

import terrascript.core as core


@core.schema
class ClientData(core.Schema):

    comment: Optional[Union[str, core.StringOut]] = core.attr(str, default=None)

    upload_end: Optional[Union[str, core.StringOut]] = core.attr(str, default=None, computed=True)

    upload_size: Optional[Union[float, core.FloatOut]] = core.attr(
        float, default=None, computed=True
    )

    upload_start: Optional[Union[str, core.StringOut]] = core.attr(str, default=None, computed=True)

    def __init__(
        self,
        *,
        comment: Optional[Union[str, core.StringOut]] = None,
        upload_end: Optional[Union[str, core.StringOut]] = None,
        upload_size: Optional[Union[float, core.FloatOut]] = None,
        upload_start: Optional[Union[str, core.StringOut]] = None,
    ):
        super().__init__(
            args=ClientData.Args(
                comment=comment,
                upload_end=upload_end,
                upload_size=upload_size,
                upload_start=upload_start,
            ),
        )

    @core.schema_args
    class Args(core.SchemaArgs):
        comment: Optional[Union[str, core.StringOut]] = core.arg(default=None)

        upload_end: Optional[Union[str, core.StringOut]] = core.arg(default=None)

        upload_size: Optional[Union[float, core.FloatOut]] = core.arg(default=None)

        upload_start: Optional[Union[str, core.StringOut]] = core.arg(default=None)


@core.schema
class UserBucket(core.Schema):

    s3_bucket: Union[str, core.StringOut] = core.attr(str)

    s3_key: Union[str, core.StringOut] = core.attr(str)

    def __init__(
        self,
        *,
        s3_bucket: Union[str, core.StringOut],
        s3_key: Union[str, core.StringOut],
    ):
        super().__init__(
            args=UserBucket.Args(
                s3_bucket=s3_bucket,
                s3_key=s3_key,
            ),
        )

    @core.schema_args
    class Args(core.SchemaArgs):
        s3_bucket: Union[str, core.StringOut] = core.arg()

        s3_key: Union[str, core.StringOut] = core.arg()


@core.schema
class DiskContainer(core.Schema):

    description: Optional[Union[str, core.StringOut]] = core.attr(str, default=None)

    format: Union[str, core.StringOut] = core.attr(str)

    url: Optional[Union[str, core.StringOut]] = core.attr(str, default=None)

    user_bucket: Optional[UserBucket] = core.attr(UserBucket, default=None)

    def __init__(
        self,
        *,
        format: Union[str, core.StringOut],
        description: Optional[Union[str, core.StringOut]] = None,
        url: Optional[Union[str, core.StringOut]] = None,
        user_bucket: Optional[UserBucket] = None,
    ):
        super().__init__(
            args=DiskContainer.Args(
                format=format,
                description=description,
                url=url,
                user_bucket=user_bucket,
            ),
        )

    @core.schema_args
    class Args(core.SchemaArgs):
        description: Optional[Union[str, core.StringOut]] = core.arg(default=None)

        format: Union[str, core.StringOut] = core.arg()

        url: Optional[Union[str, core.StringOut]] = core.arg(default=None)

        user_bucket: Optional[UserBucket] = core.arg(default=None)


@core.resource(type="aws_ebs_snapshot_import", namespace="aws_ebs")
class SnapshotImport(core.Resource):

    arn: Union[str, core.StringOut] = core.attr(str, computed=True)

    client_data: Optional[ClientData] = core.attr(ClientData, default=None)

    data_encryption_key_id: Union[str, core.StringOut] = core.attr(str, computed=True)

    description: Optional[Union[str, core.StringOut]] = core.attr(str, default=None, computed=True)

    disk_container: DiskContainer = core.attr(DiskContainer)

    encrypted: Optional[Union[bool, core.BoolOut]] = core.attr(bool, default=None)

    id: Union[str, core.StringOut] = core.attr(str, computed=True)

    kms_key_id: Optional[Union[str, core.StringOut]] = core.attr(str, default=None)

    outpost_arn: Union[str, core.StringOut] = core.attr(str, computed=True)

    owner_alias: Union[str, core.StringOut] = core.attr(str, computed=True)

    owner_id: Union[str, core.StringOut] = core.attr(str, computed=True)

    permanent_restore: Optional[Union[bool, core.BoolOut]] = core.attr(bool, default=None)

    role_name: Optional[Union[str, core.StringOut]] = core.attr(str, default=None)

    storage_tier: Optional[Union[str, core.StringOut]] = core.attr(str, default=None, computed=True)

    tags: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = core.attr(
        str, default=None, kind=core.Kind.map
    )

    tags_all: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = core.attr(
        str, default=None, computed=True, kind=core.Kind.map
    )

    temporary_restore_days: Optional[Union[int, core.IntOut]] = core.attr(int, default=None)

    volume_id: Union[str, core.StringOut] = core.attr(str, computed=True)

    volume_size: Union[int, core.IntOut] = core.attr(int, computed=True)

    def __init__(
        self,
        resource_name: str,
        *,
        disk_container: DiskContainer,
        client_data: Optional[ClientData] = None,
        description: Optional[Union[str, core.StringOut]] = None,
        encrypted: Optional[Union[bool, core.BoolOut]] = None,
        kms_key_id: Optional[Union[str, core.StringOut]] = None,
        permanent_restore: Optional[Union[bool, core.BoolOut]] = None,
        role_name: Optional[Union[str, core.StringOut]] = None,
        storage_tier: Optional[Union[str, core.StringOut]] = None,
        tags: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = None,
        tags_all: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = None,
        temporary_restore_days: Optional[Union[int, core.IntOut]] = None,
        depends_on: Optional[Union[List[str], core.ArrayOut[core.StringOut]]] = None,
        provider: Optional[Union[str, core.StringOut]] = None,
        lifecycle: Optional[core.Lifecycle] = None,
    ):
        super().__init__(
            name=resource_name,
            args=SnapshotImport.Args(
                disk_container=disk_container,
                client_data=client_data,
                description=description,
                encrypted=encrypted,
                kms_key_id=kms_key_id,
                permanent_restore=permanent_restore,
                role_name=role_name,
                storage_tier=storage_tier,
                tags=tags,
                tags_all=tags_all,
                temporary_restore_days=temporary_restore_days,
                depends_on=depends_on,
                provider=provider,
                lifecycle=lifecycle,
            ),
        )

    @core.schema_args
    class Args(core.Resource.Args):
        client_data: Optional[ClientData] = core.arg(default=None)

        description: Optional[Union[str, core.StringOut]] = core.arg(default=None)

        disk_container: DiskContainer = core.arg()

        encrypted: Optional[Union[bool, core.BoolOut]] = core.arg(default=None)

        kms_key_id: Optional[Union[str, core.StringOut]] = core.arg(default=None)

        permanent_restore: Optional[Union[bool, core.BoolOut]] = core.arg(default=None)

        role_name: Optional[Union[str, core.StringOut]] = core.arg(default=None)

        storage_tier: Optional[Union[str, core.StringOut]] = core.arg(default=None)

        tags: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = core.arg(default=None)

        tags_all: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = core.arg(
            default=None
        )

        temporary_restore_days: Optional[Union[int, core.IntOut]] = core.arg(default=None)
