"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GatewayRouteSpec = exports.GatewayRouteHostnameMatch = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const http_route_path_match_1 = require("./http-route-path-match");
const utils_1 = require("./private/utils");
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * Used to generate host name matching methods.
 */
class GatewayRouteHostnameMatch {
    /**
     * The value of the host name must match the specified value exactly.
     *
     * @param name The exact host name to match on
     */
    static exactly(name) {
        return new GatewayRouteHostnameMatchImpl({ exact: name });
    }
    /**
     * The value of the host name with the given name must end with the specified characters.
     *
     * @param suffix The specified ending characters of the host name to match on
     */
    static endsWith(suffix) {
        return new GatewayRouteHostnameMatchImpl({ suffix });
    }
}
exports.GatewayRouteHostnameMatch = GatewayRouteHostnameMatch;
_a = JSII_RTTI_SYMBOL_1;
GatewayRouteHostnameMatch[_a] = { fqn: "@aws-cdk/aws-appmesh.GatewayRouteHostnameMatch", version: "1.167.0" };
class GatewayRouteHostnameMatchImpl extends GatewayRouteHostnameMatch {
    constructor(matchProperty) {
        super();
        this.matchProperty = matchProperty;
    }
    bind(_scope) {
        return {
            hostnameMatch: this.matchProperty,
        };
    }
}
/**
 * Used to generate specs with different protocols for a GatewayRoute
 */
class GatewayRouteSpec {
    /**
     * Creates an HTTP Based GatewayRoute
     *
     * @param options - no http gateway route
     */
    static http(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_HttpGatewayRouteSpecOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.http);
            }
            throw error;
        }
        return new HttpGatewayRouteSpec(options, shared_interfaces_1.Protocol.HTTP);
    }
    /**
     * Creates an HTTP2 Based GatewayRoute
     *
     * @param options - no http2 gateway route
     */
    static http2(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_HttpGatewayRouteSpecOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.http2);
            }
            throw error;
        }
        return new HttpGatewayRouteSpec(options, shared_interfaces_1.Protocol.HTTP2);
    }
    /**
     * Creates an gRPC Based GatewayRoute
     *
     * @param options - no grpc gateway route
     */
    static grpc(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appmesh_GrpcGatewayRouteSpecOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grpc);
            }
            throw error;
        }
        return new GrpcGatewayRouteSpec(options);
    }
}
exports.GatewayRouteSpec = GatewayRouteSpec;
_b = JSII_RTTI_SYMBOL_1;
GatewayRouteSpec[_b] = { fqn: "@aws-cdk/aws-appmesh.GatewayRouteSpec", version: "1.167.0" };
class HttpGatewayRouteSpec extends GatewayRouteSpec {
    constructor(options, protocol) {
        super();
        this.routeTarget = options.routeTarget;
        this.routeType = protocol;
        this.match = options.match;
        this.priority = options.priority;
    }
    bind(scope) {
        const pathMatchConfig = (this.match?.path ?? http_route_path_match_1.HttpGatewayRoutePathMatch.startsWith('/')).bind(scope);
        const rewriteRequestHostname = this.match?.rewriteRequestHostname;
        const prefixPathRewrite = pathMatchConfig.prefixPathRewrite;
        const wholePathRewrite = pathMatchConfig.wholePathRewrite;
        const httpConfig = {
            match: {
                prefix: pathMatchConfig.prefixPathMatch,
                path: pathMatchConfig.wholePathMatch,
                hostname: this.match?.hostname?.bind(scope).hostnameMatch,
                method: this.match?.method,
                headers: this.match?.headers?.map(header => header.bind(scope).headerMatch),
                queryParameters: this.match?.queryParameters?.map(queryParameter => queryParameter.bind(scope).queryParameterMatch),
            },
            action: {
                target: {
                    virtualService: {
                        virtualServiceName: this.routeTarget.virtualServiceName,
                    },
                },
                rewrite: rewriteRequestHostname !== undefined || prefixPathRewrite || wholePathRewrite
                    ? {
                        hostname: rewriteRequestHostname === undefined
                            ? undefined
                            : {
                                defaultTargetHostname: rewriteRequestHostname ? 'ENABLED' : 'DISABLED',
                            },
                        prefix: prefixPathRewrite,
                        path: wholePathRewrite,
                    }
                    : undefined,
            },
        };
        return {
            priority: this.priority,
            httpSpecConfig: this.routeType === shared_interfaces_1.Protocol.HTTP ? httpConfig : undefined,
            http2SpecConfig: this.routeType === shared_interfaces_1.Protocol.HTTP2 ? httpConfig : undefined,
        };
    }
}
class GrpcGatewayRouteSpec extends GatewayRouteSpec {
    constructor(options) {
        super();
        this.match = options.match;
        this.routeTarget = options.routeTarget;
        this.priority = options.priority;
    }
    bind(scope) {
        const metadataMatch = this.match.metadata;
        utils_1.validateGrpcGatewayRouteMatch(this.match);
        utils_1.validateGrpcMatchArrayLength(metadataMatch);
        return {
            grpcSpecConfig: {
                match: {
                    serviceName: this.match.serviceName,
                    hostname: this.match.hostname?.bind(scope).hostnameMatch,
                    metadata: metadataMatch?.map(metadata => metadata.bind(scope).headerMatch),
                },
                action: {
                    target: {
                        virtualService: {
                            virtualServiceName: this.routeTarget.virtualServiceName,
                        },
                    },
                    rewrite: this.match.rewriteRequestHostname === undefined
                        ? undefined
                        : {
                            hostname: {
                                defaultTargetHostname: this.match.rewriteRequestHostname ? 'ENABLED' : 'DISABLED',
                            },
                        },
                },
            },
            priority: this.priority,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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