"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretTargetAttachment = exports.AttachmentTargetType = exports.Secret = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/fr_fr/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'],
            resourceArns: [this.secretArn],
            scope: this,
        });
        if (versionStages != null && result.principalStatement) {
            result.principalStatement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/fr_fr/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    grantWrite(grantee) {
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:PutSecretValue', 'secretsmanager:UpdateSecret'],
            resourceArns: [this.secretArn],
            scope: this,
        });
        if (this.encryptionKey) {
            // See https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantEncrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    get secretValue() {
        return this.secretValueFromJson('');
    }
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.secretName,
        });
        this.autoCreatePolicy = true;
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: props.generateSecretString || {},
            name: this.physicalName,
        });
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.encryptionKey = props.encryptionKey;
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principle = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(principle);
        (_b = this.encryptionKey) === null || _b === void 0 ? void 0 : _b.grant(principle, 'kms:CreateGrant', 'kms:DescribeKey');
    }
    static fromSecretArn(scope, id, secretArn) {
        return Secret.fromSecretAttributes(scope, id, { secretArn });
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = attrs.secretArn;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     *
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Attach a target to this secret
     *
     * @param target The target to attach
     * @returns An attached secret
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
exports.Secret = Secret;
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    /**
     * A database instance
     *
     * @deprecated use RDS_DB_INSTANCE instead
     */
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * A database cluster
     *
     * @deprecated use RDS_DB_CLUSTER instead
     */
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::Redshift::Cluster
     */
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    /**
     * AWS::DocDB::DBInstance
     */
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    /**
     * AWS::DocDB::DBCluster
     */
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.autoCreatePolicy = true;
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType,
        });
        this.encryptionKey = props.secret.encryptionKey;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
//# sourceMappingURL=data:application/json;base64,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