"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretRotation = exports.SecretRotationApplication = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const serverless = require("../../aws-sam"); // Automatically re-written from '@aws-cdk/aws-sam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A secret rotation serverless application.
 */
class SecretRotationApplication {
    constructor(applicationId, semanticVersion, options) {
        this.applicationId = `arn:aws:serverlessrepo:us-east-1:297356227824:applications/${applicationId}`;
        this.semanticVersion = semanticVersion;
        this.isMultiUser = options && options.isMultiUser;
    }
}
exports.SecretRotationApplication = SecretRotationApplication;
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the single user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the multi user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the single user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the multi user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the single user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the multi user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the single user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the multi user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the single user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the multi user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the single user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the multi user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the single user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the multi user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Secret rotation for a service or database
 */
class SecretRotation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.target.connections.defaultPort) {
            throw new Error('The `target` connections must have a default port range.');
        }
        if (props.application.isMultiUser && !props.masterSecret) {
            throw new Error('The `masterSecret` must be specified for application using the multi user scheme.');
        }
        // Max length of 64 chars, get the last 64 chars
        const uniqueId = this.node.uniqueId;
        const rotationFunctionName = uniqueId.substring(Math.max(uniqueId.length - 64, 0), uniqueId.length);
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
        });
        props.target.connections.allowDefaultPortFrom(securityGroup);
        const parameters = {
            endpoint: `https://secretsmanager.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}`,
            functionName: rotationFunctionName,
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds.join(','),
            vpcSecurityGroupIds: securityGroup.securityGroupId,
        };
        if (props.secret.encryptionKey) {
            parameters.kmsKeyArn = props.secret.encryptionKey.keyArn;
        }
        if (props.masterSecret) {
            parameters.masterSecretArn = props.masterSecret.secretArn;
            if (props.masterSecret.encryptionKey) {
                parameters.masterSecretKmsKeyArn = props.masterSecret.encryptionKey.keyArn;
            }
        }
        const application = new serverless.CfnApplication(this, 'Resource', {
            location: props.application,
            parameters,
        });
        // This creates a CF a dependency between the rotation schedule and the
        // serverless application. This is needed because it's the application
        // that creates the Lambda permission to invoke the function.
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html
        const rotationLambda = lambda.Function.fromFunctionArn(this, 'RotationLambda', core_1.Token.asString(application.getAtt('Outputs.RotationLambdaARN')));
        props.secret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
            automaticallyAfter: props.automaticallyAfter,
        });
        // Prevent secrets deletions when rotation is in place
        props.secret.denyAccountRootDelete();
        if (props.masterSecret) {
            props.masterSecret.denyAccountRootDelete();
        }
    }
}
exports.SecretRotation = SecretRotation;
//# sourceMappingURL=data:application/json;base64,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