"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PackageJsonManager = void 0;
const fs = require("fs");
const path = require("path");
const util_1 = require("./util");
/**
 * A package.json manager to act on the closest package.json file.
 *
 * Configuring the bundler requires to manipulate the package.json and then
 * restore it.
 */
class PackageJsonManager {
    constructor() {
        const pkgPath = util_1.findClosestPathContaining('package.json');
        if (!pkgPath) {
            throw new Error('Cannot find a `package.json` in this project.');
        }
        this.pkgPath = path.join(pkgPath, 'package.json');
        this.pkg = fs.readFileSync(this.pkgPath);
        this.pkgJson = JSON.parse(this.pkg.toString());
    }
    /**
     * Update the package.json
     */
    update(data) {
        fs.writeFileSync(this.pkgPath, JSON.stringify({
            ...this.pkgJson,
            ...data,
        }, null, 2));
    }
    /**
     * Restore the package.json to the original
     */
    restore() {
        fs.writeFileSync(this.pkgPath, this.pkg);
    }
    /**
     * Extract versions for a list of modules
     */
    getVersions(modules) {
        var _a, _b, _c;
        const dependencies = {};
        const allDependencies = {
            ...(_a = this.pkgJson.dependencies) !== null && _a !== void 0 ? _a : {},
            ...(_b = this.pkgJson.devDependencies) !== null && _b !== void 0 ? _b : {},
            ...(_c = this.pkgJson.peerDependencies) !== null && _c !== void 0 ? _c : {},
        };
        for (const mod of modules) {
            if (!allDependencies[mod]) {
                throw new Error(`Cannot extract version for ${mod} in package.json`);
            }
            dependencies[mod] = allDependencies[mod];
        }
        return dependencies;
    }
}
exports.PackageJsonManager = PackageJsonManager;
//# sourceMappingURL=data:application/json;base64,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