"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const fs = require("fs");
const path = require("path");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f;
        // Find project root
        const projectRoot = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findClosestPathContaining(`.git${path.sep}`);
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        // Bundling image derived from runtime bundling image (lambci)
        const image = cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
            buildArgs: {
                IMAGE: options.runtime.bundlingDockerImage.image,
                PARCEL_VERSION: (_b = options.parcelVersion) !== null && _b !== void 0 ? _b : '2.0.0-beta.1',
            },
        });
        const packageJsonManager = new package_json_manager_1.PackageJsonManager();
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_c = options.externalModules) !== null && _c !== void 0 ? _c : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_d = options.nodeModules) !== null && _d !== void 0 ? _d : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            'cdk-lambda': `${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/index.js`,
            'targets': {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_e = options.sourceMaps) !== null && _e !== void 0 ? _e : false,
                    minify: (_f = options.minify) !== null && _f !== void 0 ? _f : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Entry file path relative to container path
        const containerEntryPath = path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, path.relative(projectRoot, path.resolve(options.entry)));
        const parcelCommand = `parcel build ${containerEntryPath.replace(/\\/g, '/')} --target cdk-lambda${options.cacheDir ? ' --cache-dir /parcel-cache' : ''}`;
        let installer = Installer.NPM;
        let lockfile;
        let depsCommand = '';
        if (dependencies) {
            // Create a dummy package.json for dependencies that we need to install
            fs.writeFileSync(path.join(projectRoot, '.package.json'), JSON.stringify({ dependencies }));
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, LockFile.YARN))) {
                installer = Installer.YARN;
                lockfile = LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, LockFile.NPM))) {
                lockfile = LockFile.NPM;
            }
            // Move dummy package.json and lock file then install
            depsCommand = chain([
                `mv ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/.package.json ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/package.json`,
                lockfile ? `cp ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/${lockfile} ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${lockfile}` : '',
                `cd ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR} && ${installer} install`,
            ]);
        }
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                image,
                command: ['bash', '-c', chain([parcelCommand, depsCommand])],
                environment: options.parcelEnvironment,
                volumes: options.cacheDir
                    ? [{ containerPath: '/parcel-cache', hostPath: options.cacheDir }]
                    : [],
                workingDirectory: path.dirname(containerEntryPath).replace(/\\/g, '/'),
            },
        });
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile || (LockFile = {}));
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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