"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MachineImageType = exports.DefaultCapacityType = exports.CoreDnsComputeType = exports.NodeType = exports.EksOptimizedImage = exports.Cluster = void 0;
const autoscaling = require("../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const fs = require("fs");
const path = require("path");
const YAML = require("yaml");
const aws_auth_1 = require("./aws-auth");
const cluster_resource_1 = require("./cluster-resource");
const eks_generated_1 = require("./eks.generated");
const fargate_profile_1 = require("./fargate-profile");
const helm_chart_1 = require("./helm-chart");
const k8s_patch_1 = require("./k8s-patch");
const k8s_resource_1 = require("./k8s-resource");
const kubectl_provider_1 = require("./kubectl-provider");
const managed_nodegroup_1 = require("./managed-nodegroup");
const service_account_1 = require("./service-account");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 */
class Cluster extends core_1.Resource {
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param name the name of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * If the cluster has one (or more) FargateProfiles associated, this array
         * will hold a reference to each.
         */
        this._fargateProfiles = [];
        const stack = core_1.Stack.of(this);
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        this.version = props.version;
        this.tagSubnets();
        // this is the role used by EKS when interacting with AWS resources
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(443),
        });
        // Get subnetIds for all selected subnets
        const placements = props.vpcSubnets || [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE }];
        const subnetIds = [...new Set(Array().concat(...placements.map(s => this.vpc.selectSubnets(s).subnetIds)))];
        const clusterProps = {
            name: this.physicalName,
            roleArn: this.role.roleArn,
            version: props.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds,
            },
        };
        let resource;
        this.kubectlEnabled = props.kubectlEnabled === undefined ? true : props.kubectlEnabled;
        if (this.kubectlEnabled) {
            resource = new cluster_resource_1.ClusterResource(this, 'Resource', clusterProps);
            this._clusterResource = resource;
        }
        else {
            resource = new eks_generated_1.CfnCluster(this, 'Resource', clusterProps);
        }
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, cluster_resource_1.clusterArnComponents(this.physicalName));
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        this.clusterSecurityGroupId = resource.attrClusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = resource.attrEncryptionConfigKeyArn;
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // map the IAM role to the `system:masters` group.
        if (props.mastersRole) {
            if (!this.kubectlEnabled) {
                throw new Error('Cannot specify a "masters" role if kubectl is disabled');
            }
            this.awsAuth.addMastersRole(props.mastersRole);
            if (props.outputMastersRoleArn) {
                new core_1.CfnOutput(this, 'MastersRoleArn', { value: props.mastersRole.roleArn });
            }
            commonCommandOptions.push(`--role-arn ${props.mastersRole.roleArn}`);
        }
        // allocate default capacity if non-zero (or default).
        const minCapacity = props.defaultCapacity === undefined ? DEFAULT_CAPACITY_COUNT : props.defaultCapacity;
        if (minCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = props.defaultCapacityType === DefaultCapacityType.EC2 ?
                this.addCapacity('DefaultCapacity', { instanceType, minCapacity }) : undefined;
            this.defaultNodegroup = props.defaultCapacityType !== DefaultCapacityType.EC2 ?
                this.addNodegroup('DefaultCapacity', { instanceType, minSize: minCapacity }) : undefined;
        }
        const outputConfigCommand = props.outputConfigCommand === undefined ? true : props.outputConfigCommand;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
        if (this.kubectlEnabled) {
            this.defineCoreDnsComputeType(props.coreDnsComputeType || CoreDnsComputeType.EC2);
        }
    }
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     */
    addCapacity(id, options) {
        if (options.machineImageType === MachineImageType.BOTTLEROCKET && options.bootstrapOptions !== undefined) {
            throw new Error('bootstrapOptions is not supported for Bottlerocket');
        }
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: options.machineImageType === MachineImageType.BOTTLEROCKET ?
                new BottleRocketImage() :
                new EksOptimizedImage({
                    nodeType: nodeTypeForInstanceType(options.instanceType),
                    kubernetesVersion: this.version,
                }),
            updateType: options.updateType || autoscaling.UpdateType.ROLLING_UPDATE,
            instanceType: options.instanceType,
        });
        this.addAutoScalingGroup(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled,
            machineImageType: options.machineImageType,
        });
        if (nodeTypeForInstanceType(options.instanceType) === NodeType.INFERENTIA) {
            this.addNeuronDevicePlugin();
        }
        return asg;
    }
    /**
     * Add managed nodegroup to this Amazon EKS cluster
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @param id The ID of the nodegroup
     * @param options options for creating a new nodegroup
     */
    addNodegroup(id, options) {
        return new managed_nodegroup_1.Nodegroup(this, `Nodegroup${id}`, {
            cluster: this,
            ...options,
        });
    }
    /**
     * Add compute capacity to this EKS cluster in the form of an AutoScalingGroup
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    addAutoScalingGroup(autoScalingGroup, options) {
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        const bootstrapEnabled = options.bootstrapEnabled !== undefined ? options.bootstrapEnabled : true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error('Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false');
        }
        if (bootstrapEnabled) {
            const userData = options.machineImageType === MachineImageType.BOTTLEROCKET ?
                user_data_1.renderBottlerocketUserData(this) :
                user_data_1.renderAmazonLinuxUserData(this.clusterName, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        core_1.Tag.add(autoScalingGroup, `kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true,
        });
        if (options.mapRole === true && !this.kubectlEnabled) {
            throw new Error('Cannot map instance IAM role to RBAC if kubectl is disabled for the cluster');
        }
        // do not attempt to map the role if `kubectl` is not enabled for this
        // cluster or if `mapRole` is set to false. By default this should happen.
        const mapRole = options.mapRole === undefined ? true : options.mapRole;
        if (mapRole && this.kubectlEnabled) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.awsAuth.addRoleMapping(autoScalingGroup.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        else {
            // since we are not mapping the instance role to RBAC, synthesize an
            // output so it can be pasted into `aws-auth-cm.yaml`
            new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
                value: autoScalingGroup.role.roleArn,
            });
        }
        // if this is an ASG with spot instances, install the spot interrupt handler (only if kubectl is enabled).
        if (autoScalingGroup.spotPrice && this.kubectlEnabled) {
            this.addSpotInterruptHandler();
        }
    }
    /**
     * Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     */
    get awsAuth() {
        if (!this.kubectlEnabled) {
            throw new Error('Cannot define aws-auth mappings if kubectl is disabled');
        }
        if (!this._awsAuth) {
            this._awsAuth = new aws_auth_1.AwsAuth(this, 'AwsAuth', { cluster: this });
        }
        return this._awsAuth;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer url.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuerUrl() {
        if (!this._clusterResource) {
            throw new Error('unable to obtain OpenID Connect issuer URL. Cluster must be kubectl-enabled');
        }
        return this._clusterResource.attrOpenIdConnectIssuerUrl;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuer() {
        if (!this._clusterResource) {
            throw new Error('unable to obtain OpenID Connect issuer. Cluster must be kubectl-enabled');
        }
        return this._clusterResource.attrOpenIdConnectIssuer;
    }
    /**
     * An `OpenIdConnectProvider` resource associated with this cluster, and which can be used
     * to link this cluster to AWS IAM.
     *
     * A provider will only be defined if this property is accessed (lazy initialization).
     */
    get openIdConnectProvider() {
        if (!this.kubectlEnabled) {
            throw new Error('Cannot specify a OpenID Connect Provider if kubectl is disabled');
        }
        if (!this._openIdConnectProvider) {
            this._openIdConnectProvider = new iam.OpenIdConnectProvider(this, 'OpenIdConnectProvider', {
                url: this.clusterOpenIdConnectIssuerUrl,
                clientIds: ['sts.amazonaws.com'],
                /**
                 * For some reason EKS isn't validating the root certificate but a intermediat certificate
                 * which is one level up in the tree. Because of the a constant thumbprint value has to be
                 * stated with this OpenID Connect provider. The certificate thumbprint is the same for all the regions.
                 */
                thumbprints: ['9e99a48a9960b14926bb7f3b02e22da2b0ab7280'],
            });
        }
        return this._openIdConnectProvider;
    }
    /**
     * Defines a Kubernetes resource in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest
     * @param manifest a list of Kubernetes resource specifications
     * @returns a `KubernetesResource` object.
     * @throws If `kubectlEnabled` is `false`
     */
    addResource(id, ...manifest) {
        return new k8s_resource_1.KubernetesResource(this, `manifest-${id}`, { cluster: this, manifest });
    }
    /**
     * Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` object
     * @throws If `kubectlEnabled` is `false`
     */
    addChart(id, options) {
        return new helm_chart_1.HelmChart(this, `chart-${id}`, { cluster: this, ...options });
    }
    /**
     * Adds a Fargate profile to this cluster.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param id the id of this profile
     * @param options profile options
     */
    addFargateProfile(id, options) {
        return new fargate_profile_1.FargateProfile(this, `fargate-profile-${id}`, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Adds a service account to this cluster.
     *
     * @param id the id of this service account
     * @param options service account options
     */
    addServiceAccount(id, options = {}) {
        return new service_account_1.ServiceAccount(this, id, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Returns the role ARN for the cluster creation role for kubectl-enabled
     * clusters.
     * @param assumedBy The IAM that will assume this role. If omitted, the
     * creation role will be returned witout modification of its trust policy.
     *
     * @internal
     */
    get _kubectlCreationRole() {
        if (!this._clusterResource) {
            throw new Error('Unable to perform this operation since kubectl is not enabled for this cluster');
        }
        return this._clusterResource.creationRole;
    }
    /**
     * Returns the custom resource provider for kubectl-related resources.
     * @internal
     */
    get _kubectlProvider() {
        if (!this._clusterResource) {
            throw new Error('Unable to perform this operation since kubectl is not enabled for this cluster');
        }
        const uid = '@aws-cdk/aws-eks.KubectlProvider';
        const provider = this.stack.node.tryFindChild(uid) || new kubectl_provider_1.KubectlProvider(this.stack, uid);
        // allow the kubectl provider to assume the cluster creation role.
        this._clusterResource.addTrustedRole(provider.role);
        return provider;
    }
    /**
     * Internal API used by `FargateProfile` to keep inventory of Fargate profiles associated with
     * this cluster, for the sake of ensuring the profiles are created sequentially.
     *
     * @returns the list of FargateProfiles attached to this cluster, including the one just attached.
     * @internal
     */
    _attachFargateProfile(fargateProfile) {
        this._fargateProfiles.push(fargateProfile);
        return this._fargateProfiles;
    }
    /**
     * Installs the AWS spot instance interrupt handler on the cluster if it's not
     * already added.
     */
    addSpotInterruptHandler() {
        if (!this._spotInterruptHandler) {
            this._spotInterruptHandler = this.addChart('spot-interrupt-handler', {
                chart: 'aws-node-termination-handler',
                version: '0.7.3',
                repository: 'https://aws.github.io/eks-charts',
                namespace: 'kube-system',
                values: {
                    'nodeSelector.lifecycle': user_data_1.LifecycleLabel.SPOT,
                },
            });
        }
        return this._spotInterruptHandler;
    }
    /**
     * Installs the Neuron device plugin on the cluster if it's not
     * already added.
     */
    addNeuronDevicePlugin() {
        if (!this._neuronDevicePlugin) {
            const fileContents = fs.readFileSync(path.join(__dirname, 'addons/neuron-device-plugin.yaml'), 'utf8');
            const sanitized = YAML.parse(fileContents);
            this._neuronDevicePlugin = this.addResource('NeuronDevicePlugin', sanitized);
        }
        return this._neuronDevicePlugin;
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) ? '' : ` ${subnet.subnetId}`;
                    this.node.addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                subnet.node.applyAspect(new core_1.Tag(tag, '1'));
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, 'kubernetes.io/role/internal-elb');
        tagAllSubnets('public', this.vpc.publicSubnets, 'kubernetes.io/role/elb');
    }
    /**
     * Patches the CoreDNS deployment configuration and sets the "eks.amazonaws.com/compute-type"
     * annotation to either "ec2" or "fargate". Note that if "ec2" is selected, the resource is
     * omitted/removed, since the cluster is created with the "ec2" compute type by default.
     */
    defineCoreDnsComputeType(type) {
        if (!this.kubectlEnabled) {
            throw new Error('kubectl must be enabled in order to define the compute type for CoreDNS');
        }
        // ec2 is the "built in" compute type of the cluster so if this is the
        // requested type we can simply omit the resource. since the resource's
        // `restorePatch` is configured to restore the value to "ec2" this means
        // that deletion of the resource will change to "ec2" as well.
        if (type === CoreDnsComputeType.EC2) {
            return;
        }
        // this is the json patch we merge into the resource based off of:
        // https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns
        const renderPatch = (computeType) => ({
            spec: {
                template: {
                    metadata: {
                        annotations: {
                            'eks.amazonaws.com/compute-type': computeType,
                        },
                    },
                },
            },
        });
        new k8s_patch_1.KubernetesPatch(this, 'CoreDnsComputeTypePatch', {
            cluster: this,
            resourceName: 'deployment/coredns',
            resourceNamespace: 'kube-system',
            applyPatch: renderPatch(CoreDnsComputeType.FARGATE),
            restorePatch: renderPatch(CoreDnsComputeType.EC2),
        });
    }
}
exports.Cluster = Cluster;
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.connections = new ec2.Connections();
        this.vpc = ec2.Vpc.fromVpcAttributes(this, 'VPC', props.vpc);
        this.clusterName = props.clusterName;
        this.clusterEndpoint = props.clusterEndpoint;
        this.clusterArn = props.clusterArn;
        this.clusterCertificateAuthorityData = props.clusterCertificateAuthorityData;
        this.clusterSecurityGroupId = props.clusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = props.clusterEncryptionConfigKeyArn;
        let i = 1;
        for (const sgProps of props.securityGroups) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgProps.securityGroupId));
            i++;
        }
    }
}
/**
 * Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM
 */
class EksOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props = {}) {
        var _a, _b;
        this.nodeType = (_a = props.nodeType) !== null && _a !== void 0 ? _a : NodeType.STANDARD;
        this.kubernetesVersion = (_b = props.kubernetesVersion) !== null && _b !== void 0 ? _b : LATEST_KUBERNETES_VERSION;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/eks/optimized-ami/${this.kubernetesVersion}/`
            + (this.nodeType === NodeType.STANDARD ? 'amazon-linux-2/' : '')
            + (this.nodeType === NodeType.GPU ? 'amazon-linux-2-gpu/' : '')
            + (this.nodeType === NodeType.INFERENTIA ? 'amazon-linux-2-gpu/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.forLinux(),
        };
    }
}
exports.EksOptimizedImage = EksOptimizedImage;
/**
 * Construct an Bottlerocket image from the latest AMI published in SSM
 */
class BottleRocketImage {
    /**
     * Constructs a new instance of the BottleRocketImage class.
     */
    constructor() {
        // only 1.15 is currently available
        this.kubernetesVersion = '1.15';
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/bottlerocket/aws-k8s-${this.kubernetesVersion}/x86_64/latest/image_id`;
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.custom(''),
        };
    }
}
// MAINTAINERS: use ./scripts/kube_bump.sh to update LATEST_KUBERNETES_VERSION
const LATEST_KUBERNETES_VERSION = '1.14';
/**
 * Whether the worker nodes should support GPU or just standard instances
 */
var NodeType;
(function (NodeType) {
    /**
     * Standard instances
     */
    NodeType["STANDARD"] = "Standard";
    /**
     * GPU instances
     */
    NodeType["GPU"] = "GPU";
    /**
     * Inferentia instances
     */
    NodeType["INFERENTIA"] = "INFERENTIA";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * The type of compute resources to use for CoreDNS.
 */
var CoreDnsComputeType;
(function (CoreDnsComputeType) {
    /**
     * Deploy CoreDNS on EC2 instances.
     */
    CoreDnsComputeType["EC2"] = "ec2";
    /**
     * Deploy CoreDNS on Fargate-managed instances.
     */
    CoreDnsComputeType["FARGATE"] = "fargate";
})(CoreDnsComputeType = exports.CoreDnsComputeType || (exports.CoreDnsComputeType = {}));
/**
 * The default capacity type for the cluster
 */
var DefaultCapacityType;
(function (DefaultCapacityType) {
    /**
     * managed node group
     */
    DefaultCapacityType[DefaultCapacityType["NODEGROUP"] = 0] = "NODEGROUP";
    /**
     * EC2 autoscaling group
     */
    DefaultCapacityType[DefaultCapacityType["EC2"] = 1] = "EC2";
})(DefaultCapacityType = exports.DefaultCapacityType || (exports.DefaultCapacityType = {}));
/**
 * The machine image type
 */
var MachineImageType;
(function (MachineImageType) {
    /**
     * Amazon EKS-optimized Linux AMI
     */
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    /**
     * Bottlerocket AMI
     */
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
const GPU_INSTANCETYPES = ['p2', 'p3', 'g4'];
const INFERENTIA_INSTANCETYPES = ['inf1'];
function nodeTypeForInstanceType(instanceType) {
    return GPU_INSTANCETYPES.includes(instanceType.toString().substring(0, 2)) ? NodeType.GPU :
        INFERENTIA_INSTANCETYPES.includes(instanceType.toString().substring(0, 4)) ? NodeType.INFERENTIA :
            NodeType.STANDARD;
}
//# sourceMappingURL=data:application/json;base64,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