"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystem = exports.ThroughputMode = exports.PerformanceMode = exports.LifecyclePolicy = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const access_point_1 = require("./access-point");
const efs_generated_1 = require("./efs.generated");
// tslint:disable:max-line-length
/**
 * EFS Lifecycle Policy, if a file is not accessed for given days, it will move to EFS Infrequent Access.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-lifecyclepolicies
 */
// tslint:enable
var LifecyclePolicy;
(function (LifecyclePolicy) {
    /**
     * After 7 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_7_DAYS"] = 0] = "AFTER_7_DAYS";
    /**
     * After 14 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_14_DAYS"] = 1] = "AFTER_14_DAYS";
    /**
     * After 30 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_30_DAYS"] = 2] = "AFTER_30_DAYS";
    /**
     * After 60 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_60_DAYS"] = 3] = "AFTER_60_DAYS";
    /**
     * After 90 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_90_DAYS"] = 4] = "AFTER_90_DAYS";
})(LifecyclePolicy = exports.LifecyclePolicy || (exports.LifecyclePolicy = {}));
/**
 * EFS Performance mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-efs-filesystem-performancemode
 */
var PerformanceMode;
(function (PerformanceMode) {
    /**
     * This is the general purpose performance mode for most file systems.
     */
    PerformanceMode["GENERAL_PURPOSE"] = "generalPurpose";
    /**
     * This performance mode can scale to higher levels of aggregate throughput and operations per second with a
     * tradeoff of slightly higher latencies.
     */
    PerformanceMode["MAX_IO"] = "maxIO";
})(PerformanceMode = exports.PerformanceMode || (exports.PerformanceMode = {}));
/**
 * EFS Throughput mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-throughputmode
 */
var ThroughputMode;
(function (ThroughputMode) {
    /**
     *  This mode on Amazon EFS scales as the size of the file system in the standard storage class grows.
     */
    ThroughputMode["BURSTING"] = "bursting";
    /**
     * This mode can instantly provision the throughput of the file system (in MiB/s) independent of the amount of data stored.
     */
    ThroughputMode["PROVISIONED"] = "provisioned";
})(ThroughputMode = exports.ThroughputMode || (exports.ThroughputMode = {}));
/**
 * The Elastic File System implementation of IFileSystem.
 * It creates a new, empty file system in Amazon Elastic File System (Amazon EFS).
 * It also creates mount target (AWS::EFS::MountTarget) implicitly to mount the
 * EFS file system on an Amazon Elastic Compute Cloud (Amazon EC2) instance or another resource.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html
 *
 * @resource AWS::EFS::FileSystem
 */
class FileSystem extends core_1.Resource {
    /**
     * Constructor for creating a new EFS FileSystem.
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (props.throughputMode === ThroughputMode.PROVISIONED && props.provisionedThroughputPerSecond === undefined) {
            throw new Error('Property provisionedThroughputPerSecond is required when throughputMode is PROVISIONED');
        }
        const filesystem = new efs_generated_1.CfnFileSystem(this, 'Resource', {
            encrypted: props.encrypted,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lifecyclePolicies: (props.lifecyclePolicy ? Array.of({
                transitionToIa: LifecyclePolicy[props.lifecyclePolicy],
            }) : undefined),
            performanceMode: props.performanceMode,
            throughputMode: props.throughputMode,
            provisionedThroughputInMibps: (_a = props.provisionedThroughputPerSecond) === null || _a === void 0 ? void 0 : _a.toMebibytes(),
        });
        filesystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = filesystem.ref;
        core_1.Tag.add(this, 'Name', props.fileSystemName || this.node.path);
        const securityGroup = (props.securityGroup || new ec2.SecurityGroup(this, 'EfsSecurityGroup', {
            vpc: props.vpc,
        }));
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        const subnets = props.vpc.selectSubnets(props.vpcSubnets);
        // We now have to create the mount target for each of the mentioned subnet
        let mountTargetCount = 0;
        subnets.subnetIds.forEach((subnetId) => {
            new efs_generated_1.CfnMountTarget(this, 'EfsMountTarget' + (++mountTargetCount), {
                fileSystemId: this.fileSystemId,
                securityGroups: Array.of(securityGroup.securityGroupId),
                subnetId,
            });
        });
    }
    /**
     * Import an existing File System from the given properties.
     */
    static fromFileSystemAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.fileSystemId = attrs.fileSystemId;
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * create access point from this filesystem
     */
    addAccessPoint(id, accessPointOptions = {}) {
        return new access_point_1.AccessPoint(this, id, {
            fileSystem: this,
            ...accessPointOptions,
        });
    }
}
exports.FileSystem = FileSystem;
/**
 * The default port File System listens on.
 */
FileSystem.DEFAULT_PORT = 2049;
//# sourceMappingURL=data:application/json;base64,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