"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PropagatedTagSource = exports.DeploymentControllerType = exports.LaunchType = exports.BaseService = exports.ListenerConfig = void 0;
const appscaling = require("../../../aws-applicationautoscaling"); // Automatically re-written from '@aws-cdk/aws-applicationautoscaling'
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const elbv2 = require("../../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cloudmap = require("../../../aws-servicediscovery"); // Automatically re-written from '@aws-cdk/aws-servicediscovery'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_definition_1 = require("../base/task-definition");
const ecs_generated_1 = require("../ecs.generated");
const scalable_task_count_1 = require("./scalable-task-count");
/**
 * Base class for configuring listener when registering targets.
 */
class ListenerConfig {
    /**
     * Create a config for adding target group to ALB listener.
     */
    static applicationListener(listener, props) {
        return new ApplicationListenerConfig(listener, props);
    }
    /**
     * Create a config for adding target group to NLB listener.
     */
    static networkListener(listener, props) {
        return new NetworkListenerConfig(listener, props);
    }
}
exports.ListenerConfig = ListenerConfig;
/**
 * Class for configuring application load balancer listener when registering targets.
 */
class ApplicationListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        const props = this.props || {};
        const protocol = props.protocol;
        const port = props.port !== undefined ? props.port : (protocol === undefined ? 80 :
            (protocol === elbv2.ApplicationProtocol.HTTPS ? 443 : 80));
        this.listener.addTargets(id, {
            ...props,
            targets: [
                service.loadBalancerTarget({
                    ...target,
                }),
            ],
            port,
        });
    }
}
/**
 * Class for configuring network load balancer listener when registering targets.
 */
class NetworkListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        const port = this.props !== undefined ? this.props.port : 80;
        this.listener.addTargets(id, {
            ...this.props,
            targets: [
                service.loadBalancerTarget({
                    ...target,
                }),
            ],
            port,
        });
    }
}
/**
 * The base class for Ec2Service and FargateService services.
 */
class BaseService extends core_1.Resource {
    /**
     * Constructs a new instance of the BaseService class.
     */
    constructor(scope, id, props, additionalProps, taskDefinition) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.serviceName,
        });
        /**
         * The security groups which manage the allowed network traffic for the service.
         */
        this.connections = new ec2.Connections();
        /**
         * A list of Elastic Load Balancing load balancer objects, containing the load balancer name, the container
         * name (as it appears in a container definition), and the container port to access from the load balancer.
         */
        this.loadBalancers = new Array();
        /**
         * The details of the service discovery registries to assign to this service.
         * For more information, see Service Discovery.
         */
        this.serviceRegistries = new Array();
        this.taskDefinition = taskDefinition;
        this.resource = new ecs_generated_1.CfnService(this, 'Service', {
            desiredCount: props.desiredCount,
            serviceName: this.physicalName,
            loadBalancers: core_1.Lazy.anyValue({ produce: () => this.loadBalancers }, { omitEmptyArray: true }),
            deploymentConfiguration: {
                maximumPercent: props.maxHealthyPercent || 200,
                minimumHealthyPercent: props.minHealthyPercent === undefined ? 50 : props.minHealthyPercent,
            },
            propagateTags: props.propagateTags === PropagatedTagSource.NONE ? undefined : props.propagateTags,
            enableEcsManagedTags: props.enableECSManagedTags === undefined ? false : props.enableECSManagedTags,
            deploymentController: props.deploymentController,
            launchType: ((_a = props.deploymentController) === null || _a === void 0 ? void 0 : _a.type) === DeploymentControllerType.EXTERNAL ? undefined : props.launchType,
            healthCheckGracePeriodSeconds: this.evaluateHealthGracePeriod(props.healthCheckGracePeriod),
            /* role: never specified, supplanted by Service Linked Role */
            networkConfiguration: core_1.Lazy.anyValue({ produce: () => this.networkConfiguration }, { omitEmptyArray: true }),
            serviceRegistries: core_1.Lazy.anyValue({ produce: () => this.serviceRegistries }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (((_b = props.deploymentController) === null || _b === void 0 ? void 0 : _b.type) === DeploymentControllerType.EXTERNAL) {
            this.node.addWarning('taskDefinition and launchType are blanked out when using external deployment controller.');
        }
        this.serviceArn = this.getResourceArnAttribute(this.resource.ref, {
            service: 'ecs',
            resource: 'service',
            resourceName: `${props.cluster.clusterName}/${this.physicalName}`,
        });
        this.serviceName = this.getResourceNameAttribute(this.resource.attrName);
        this.cluster = props.cluster;
        if (props.cloudMapOptions) {
            this.enableCloudMap(props.cloudMapOptions);
        }
    }
    /**
     * The CloudMap service created for this service, if any.
     */
    get cloudMapService() {
        return this.cloudmapService;
    }
    /**
     * This method is called to attach this service to an Application Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToApplicationTargetGroup(targetGroup) {
        return this.defaultLoadBalancerTarget.attachToApplicationTargetGroup(targetGroup);
    }
    /**
     * Registers the service as a target of a Classic Load Balancer (CLB).
     *
     * Don't call this. Call `loadBalancer.addTarget()` instead.
     */
    attachToClassicLB(loadBalancer) {
        return this.defaultLoadBalancerTarget.attachToClassicLB(loadBalancer);
    }
    /**
     * Return a load balancing target for a specific container and port.
     *
     * Use this function to create a load balancer target if you want to load balance to
     * another container than the first essential container or the first mapped port on
     * the container.
     *
     * Use the return value of this function where you would normally use a load balancer
     * target, instead of the `Service` object itself.
     *
     * @example
     *
     * listener.addTargets('ECS', {
     *   port: 80,
     *   targets: [service.loadBalancerTarget({
     *     containerName: 'MyContainer',
     *     containerPort: 1234,
     *   })],
     * });
     */
    loadBalancerTarget(options) {
        const self = this;
        const target = this.taskDefinition._validateTarget(options);
        const connections = self.connections;
        return {
            attachToApplicationTargetGroup(targetGroup) {
                targetGroup.registerConnectable(self, self.taskDefinition._portRangeFromPortMapping(target.portMapping));
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            attachToNetworkTargetGroup(targetGroup) {
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            connections,
            attachToClassicLB(loadBalancer) {
                return self.attachToELB(loadBalancer, target.containerName, target.portMapping.containerPort);
            },
        };
    }
    /**
     * Use this function to create all load balancer targets to be registered in this service, add them to
     * target groups, and attach target groups to listeners accordingly.
     *
     * Alternatively, you can use `listener.addTargets()` to create targets and add them to target groups.
     *
     * @example
     *
     * service.registerLoadBalancerTargets(
     *   {
     *     containerName: 'web',
     *     containerPort: 80,
     *     newTargetGroupId: 'ECS',
     *     listener: ecs.ListenerConfig.applicationListener(listener, {
     *       protocol: elbv2.ApplicationProtocol.HTTPS
     *     }),
     *   },
     * )
     */
    registerLoadBalancerTargets(...targets) {
        for (const target of targets) {
            target.listener.addTargets(target.newTargetGroupId, {
                containerName: target.containerName,
                containerPort: target.containerPort,
                protocol: target.protocol,
            }, this);
        }
    }
    /**
     * This method is called to attach this service to a Network Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToNetworkTargetGroup(targetGroup) {
        return this.defaultLoadBalancerTarget.attachToNetworkTargetGroup(targetGroup);
    }
    /**
     * An attribute representing the minimum and maximum task count for an AutoScalingGroup.
     */
    autoScaleTaskCount(props) {
        if (this.scalableTaskCount) {
            throw new Error('AutoScaling of task count already enabled for this service');
        }
        return this.scalableTaskCount = new scalable_task_count_1.ScalableTaskCount(this, 'TaskCount', {
            serviceNamespace: appscaling.ServiceNamespace.ECS,
            resourceId: `service/${this.cluster.clusterName}/${this.serviceName}`,
            dimension: 'ecs:service:DesiredCount',
            role: this.makeAutoScalingRole(),
            ...props,
        });
    }
    /**
     * Enable CloudMap service discovery for the service
     *
     * @returns The created CloudMap service
     */
    enableCloudMap(options) {
        const sdNamespace = options.cloudMapNamespace !== undefined ? options.cloudMapNamespace : this.cluster.defaultCloudMapNamespace;
        if (sdNamespace === undefined) {
            throw new Error('Cannot enable service discovery if a Cloudmap Namespace has not been created in the cluster.');
        }
        // Determine DNS type based on network mode
        const networkMode = this.taskDefinition.networkMode;
        if (networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a service discovery if NetworkMode is None. Use Bridge, Host or AwsVpc instead.');
        }
        // Bridge or host network mode requires SRV records
        let dnsRecordType = options.dnsRecordType;
        if (networkMode === task_definition_1.NetworkMode.BRIDGE || networkMode === task_definition_1.NetworkMode.HOST) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.SRV;
            }
            if (dnsRecordType !== cloudmap.DnsRecordType.SRV) {
                throw new Error('SRV records must be used when network mode is Bridge or Host.');
            }
        }
        // Default DNS record type for AwsVpc network mode is A Records
        if (networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.A;
            }
        }
        // If the task definition that your service task specifies uses the AWSVPC network mode and a type SRV DNS record is
        // used, you must specify a containerName and containerPort combination
        const containerName = dnsRecordType === cloudmap.DnsRecordType.SRV ? this.taskDefinition.defaultContainer.containerName : undefined;
        const containerPort = dnsRecordType === cloudmap.DnsRecordType.SRV ? this.taskDefinition.defaultContainer.containerPort : undefined;
        const cloudmapService = new cloudmap.Service(this, 'CloudmapService', {
            namespace: sdNamespace,
            name: options.name,
            dnsRecordType: dnsRecordType,
            customHealthCheck: { failureThreshold: options.failureThreshold || 1 },
            dnsTtl: options.dnsTtl,
        });
        const serviceArn = cloudmapService.serviceArn;
        // add Cloudmap service to the ECS Service's serviceRegistry
        this.addServiceRegistry({
            arn: serviceArn,
            containerName,
            containerPort,
        });
        this.cloudmapService = cloudmapService;
        return cloudmapService;
    }
    /**
     * This method returns the specified CloudWatch metric name for this service.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ECS',
            metricName,
            dimensions: { ClusterName: this.cluster.clusterName, ServiceName: this.serviceName },
            ...props,
        }).attachTo(this);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory utilization.
     *
     * @default average over 5 minutes
     */
    metricMemoryUtilization(props) {
        return this.metric('MemoryUtilization', props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU utilization.
     *
     * @default average over 5 minutes
     */
    metricCpuUtilization(props) {
        return this.metric('CPUUtilization', props);
    }
    /**
     * This method is called to create a networkConfiguration.
     * @deprecated use configureAwsVpcNetworkingWithSecurityGroups instead.
     */
    // tslint:disable-next-line:max-line-length
    configureAwsVpcNetworking(vpc, assignPublicIp, vpcSubnets, securityGroup) {
        if (vpcSubnets === undefined) {
            vpcSubnets = assignPublicIp ? { subnetType: ec2.SubnetType.PUBLIC } : {};
        }
        if (securityGroup === undefined) {
            securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', { vpc });
        }
        this.connections.addSecurityGroup(securityGroup);
        this.networkConfiguration = {
            awsvpcConfiguration: {
                assignPublicIp: assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: vpc.selectSubnets(vpcSubnets).subnetIds,
                securityGroups: core_1.Lazy.listValue({ produce: () => [securityGroup.securityGroupId] }),
            },
        };
    }
    /**
     * This method is called to create a networkConfiguration.
     */
    // tslint:disable-next-line:max-line-length
    configureAwsVpcNetworkingWithSecurityGroups(vpc, assignPublicIp, vpcSubnets, securityGroups) {
        if (vpcSubnets === undefined) {
            vpcSubnets = assignPublicIp ? { subnetType: ec2.SubnetType.PUBLIC } : {};
        }
        if (securityGroups === undefined || securityGroups.length === 0) {
            securityGroups = [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc })];
        }
        securityGroups.forEach((sg) => { this.connections.addSecurityGroup(sg); }, this);
        this.networkConfiguration = {
            awsvpcConfiguration: {
                assignPublicIp: assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: vpc.selectSubnets(vpcSubnets).subnetIds,
                securityGroups: securityGroups.map((sg) => sg.securityGroupId),
            },
        };
    }
    renderServiceRegistry(registry) {
        return {
            registryArn: registry.arn,
            containerName: registry.containerName,
            containerPort: registry.containerPort,
        };
    }
    /**
     * Shared logic for attaching to an ELB
     */
    attachToELB(loadBalancer, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            throw new Error('Cannot use a Classic Load Balancer if NetworkMode is AwsVpc. Use Host or Bridge instead.');
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a Classic Load Balancer if NetworkMode is None. Use Host or Bridge instead.');
        }
        this.loadBalancers.push({
            loadBalancerName: loadBalancer.loadBalancerName,
            containerName,
            containerPort,
        });
    }
    /**
     * Shared logic for attaching to an ELBv2
     */
    attachToELBv2(targetGroup, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a load balancer if NetworkMode is None. Use Bridge, Host or AwsVpc instead.');
        }
        this.loadBalancers.push({
            targetGroupArn: targetGroup.targetGroupArn,
            containerName,
            containerPort,
        });
        // Service creation can only happen after the load balancer has
        // been associated with our target group(s), so add ordering dependency.
        this.resource.node.addDependency(targetGroup.loadBalancerAttached);
        const targetType = this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC ? elbv2.TargetType.IP : elbv2.TargetType.INSTANCE;
        return { targetType };
    }
    get defaultLoadBalancerTarget() {
        return this.loadBalancerTarget({
            containerName: this.taskDefinition.defaultContainer.containerName,
        });
    }
    /**
     * Generate the role that will be used for autoscaling this service
     */
    makeAutoScalingRole() {
        // Use a Service Linked Role.
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            region: '',
            service: 'iam',
            resource: 'role/aws-service-role/ecs.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_ECSService',
        }));
    }
    /**
     * Associate Service Discovery (Cloud Map) service
     */
    addServiceRegistry(registry) {
        const sr = this.renderServiceRegistry(registry);
        this.serviceRegistries.push(sr);
    }
    /**
     *  Return the default grace period when load balancers are configured and
     *  healthCheckGracePeriod is not already set
     */
    evaluateHealthGracePeriod(providedHealthCheckGracePeriod) {
        return core_1.Lazy.anyValue({
            produce: () => providedHealthCheckGracePeriod !== undefined ? providedHealthCheckGracePeriod.toSeconds() :
                this.loadBalancers.length > 0 ? 60 :
                    undefined,
        });
    }
}
exports.BaseService = BaseService;
/**
 * The launch type of an ECS service
 */
var LaunchType;
(function (LaunchType) {
    /**
     * The service will be launched using the EC2 launch type
     */
    LaunchType["EC2"] = "EC2";
    /**
     * The service will be launched using the FARGATE launch type
     */
    LaunchType["FARGATE"] = "FARGATE";
})(LaunchType = exports.LaunchType || (exports.LaunchType = {}));
/**
 * The deployment controller type to use for the service.
 */
var DeploymentControllerType;
(function (DeploymentControllerType) {
    /**
     * The rolling update (ECS) deployment type involves replacing the current
     * running version of the container with the latest version.
     */
    DeploymentControllerType["ECS"] = "ECS";
    /**
     * The blue/green (CODE_DEPLOY) deployment type uses the blue/green deployment model powered by AWS CodeDeploy
     */
    DeploymentControllerType["CODE_DEPLOY"] = "CODE_DEPLOY";
    /**
     * The external (EXTERNAL) deployment type enables you to use any third-party deployment controller
     */
    DeploymentControllerType["EXTERNAL"] = "EXTERNAL";
})(DeploymentControllerType = exports.DeploymentControllerType || (exports.DeploymentControllerType = {}));
/**
 * Propagate tags from either service or task definition
 */
var PropagatedTagSource;
(function (PropagatedTagSource) {
    /**
     * Propagate tags from service
     */
    PropagatedTagSource["SERVICE"] = "SERVICE";
    /**
     * Propagate tags from task definition
     */
    PropagatedTagSource["TASK_DEFINITION"] = "TASK_DEFINITION";
    /**
     * Do not propagate
     */
    PropagatedTagSource["NONE"] = "NONE";
})(PropagatedTagSource = exports.PropagatedTagSource || (exports.PropagatedTagSource = {}));
//# sourceMappingURL=data:application/json;base64,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