"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataResourceType = exports.Trail = exports.ReadWriteType = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudtrail_generated_1 = require("./cloudtrail.generated");
/**
 * Types of events that CloudTrail can log
 */
var ReadWriteType;
(function (ReadWriteType) {
    /**
     * Read-only events include API operations that read your resources,
     * but don't make changes.
     * For example, read-only events include the Amazon EC2 DescribeSecurityGroups
     * and DescribeSubnets API operations.
     */
    ReadWriteType["READ_ONLY"] = "ReadOnly";
    /**
     * Write-only events include API operations that modify (or might modify)
     * your resources.
     * For example, the Amazon EC2 RunInstances and TerminateInstances API
     * operations modify your instances.
     */
    ReadWriteType["WRITE_ONLY"] = "WriteOnly";
    /**
     * All events
     */
    ReadWriteType["ALL"] = "All";
    /**
     * No events
     */
    ReadWriteType["NONE"] = "None";
})(ReadWriteType = exports.ReadWriteType || (exports.ReadWriteType = {}));
/**
 * Cloud trail allows you to log events that happen in your AWS account
 * For example:
 *
 * import { CloudTrail } from '@aws-cdk/aws-cloudtrail'
 *
 * const cloudTrail = new CloudTrail(this, 'MyTrail');
 *
 * NOTE the above example creates an UNENCRYPTED bucket by default,
 * If you are required to use an Encrypted bucket you can supply a preconfigured bucket
 * via TrailProps
 *
 */
class Trail extends core_1.Resource {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id, {
            physicalName: props.trailName,
        });
        this.eventSelectors = [];
        const cloudTrailPrincipal = new iam.ServicePrincipal('cloudtrail.amazonaws.com');
        this.s3bucket = props.bucket || new s3.Bucket(this, 'S3', { encryption: s3.BucketEncryption.UNENCRYPTED });
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.bucketArn],
            actions: ['s3:GetBucketAcl'],
            principals: [cloudTrailPrincipal],
        }));
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.arnForObjects(`${props.s3KeyPrefix ? `${props.s3KeyPrefix}/` : ''}AWSLogs/${core_1.Stack.of(this).account}/*`)],
            actions: ['s3:PutObject'],
            principals: [cloudTrailPrincipal],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        let logsRole;
        if (props.sendToCloudWatchLogs) {
            if (props.cloudWatchLogGroup) {
                this.logGroup = props.cloudWatchLogGroup;
            }
            else {
                this.logGroup = new logs.LogGroup(this, 'LogGroup', {
                    retention: (_a = props.cloudWatchLogsRetention) !== null && _a !== void 0 ? _a : logs.RetentionDays.ONE_YEAR,
                });
            }
            logsRole = new iam.Role(this, 'LogsRole', { assumedBy: cloudTrailPrincipal });
            logsRole.addToPolicy(new iam.PolicyStatement({
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: [this.logGroup.logGroupArn],
            }));
        }
        if (props.managementEvents) {
            let managementEvent;
            if (props.managementEvents === ReadWriteType.NONE) {
                managementEvent = {
                    includeManagementEvents: false,
                };
            }
            else {
                managementEvent = {
                    includeManagementEvents: true,
                    readWriteType: props.managementEvents,
                };
            }
            this.eventSelectors.push(managementEvent);
        }
        if (props.kmsKey && props.encryptionKey) {
            throw new Error('Both kmsKey and encryptionKey must not be specified. Use only encryptionKey');
        }
        // TODO: not all regions support validation. Use service configuration data to fail gracefully
        const trail = new cloudtrail_generated_1.CfnTrail(this, 'Resource', {
            isLogging: true,
            enableLogFileValidation: props.enableFileValidation == null ? true : props.enableFileValidation,
            isMultiRegionTrail: props.isMultiRegionTrail == null ? true : props.isMultiRegionTrail,
            includeGlobalServiceEvents: props.includeGlobalServiceEvents == null ? true : props.includeGlobalServiceEvents,
            trailName: this.physicalName,
            kmsKeyId: (_c = (_b = props.encryptionKey) === null || _b === void 0 ? void 0 : _b.keyArn) !== null && _c !== void 0 ? _c : (_d = props.kmsKey) === null || _d === void 0 ? void 0 : _d.keyArn,
            s3BucketName: this.s3bucket.bucketName,
            s3KeyPrefix: props.s3KeyPrefix,
            cloudWatchLogsLogGroupArn: (_e = this.logGroup) === null || _e === void 0 ? void 0 : _e.logGroupArn,
            cloudWatchLogsRoleArn: logsRole === null || logsRole === void 0 ? void 0 : logsRole.roleArn,
            snsTopicName: (_f = props.snsTopic) === null || _f === void 0 ? void 0 : _f.topicName,
            eventSelectors: this.eventSelectors,
        });
        this.trailArn = this.getResourceArnAttribute(trail.attrArn, {
            service: 'cloudtrail',
            resource: 'trail',
            resourceName: this.physicalName,
        });
        this.trailSnsTopicArn = trail.attrSnsTopicArn;
        // Add a dependency on the bucket policy being updated, CloudTrail will test this upon creation.
        if (this.s3bucket.policy) {
            trail.node.addDependency(this.s3bucket.policy);
        }
        // If props.sendToCloudWatchLogs is set to true then the trail needs to depend on the created logsRole
        // so that it can create the log stream for the log group. This ensures the logsRole is created and propagated
        // before the trail tries to create the log stream.
        if (logsRole !== undefined) {
            trail.node.addDependency(logsRole);
        }
    }
    /**
     * Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     */
    static onEvent(scope, id, options = {}) {
        const rule = new events.Rule(scope, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            detailType: ['AWS API Call via CloudTrail'],
        });
        return rule;
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an Event Selector for filtering events that match either S3 or Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param dataResourceValues the list of data resource ARNs to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addEventSelector(dataResourceType, dataResourceValues, options = {}) {
        if (dataResourceValues.length > 250) {
            throw new Error('A maximum of 250 data elements can be in one event selector');
        }
        if (this.eventSelectors.length > 5) {
            throw new Error('A maximum of 5 event selectors are supported per trail.');
        }
        this.eventSelectors.push({
            dataResources: [{
                    type: dataResourceType,
                    values: dataResourceValues,
                }],
            includeManagementEvents: options.includeManagementEvents,
            readWriteType: options.readWriteType,
        });
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds a Lambda Data Event Selector for filtering events that match Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param handlers the list of lambda function handlers whose data events should be logged (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addLambdaEventSelector(handlers, options = {}) {
        if (handlers.length === 0) {
            return;
        }
        const dataResourceValues = handlers.map((h) => h.functionArn);
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, dataResourceValues, options);
    }
    /**
     * Log all Lamda data events for all lambda functions the account.
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @default false
     */
    logAllLambdaDataEvents(options = {}) {
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, [`arn:${this.stack.partition}:lambda`], options);
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an S3 Data Event Selector for filtering events that match S3 operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param s3Selector the list of S3 bucket with optional prefix to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addS3EventSelector(s3Selector, options = {}) {
        if (s3Selector.length === 0) {
            return;
        }
        const dataResourceValues = s3Selector.map((sel) => { var _a; return `${sel.bucket.bucketArn}/${(_a = sel.objectPrefix) !== null && _a !== void 0 ? _a : ''}`; });
        return this.addEventSelector(DataResourceType.S3_OBJECT, dataResourceValues, options);
    }
    /**
     * Log all S3 data events for all objects for all buckets in the account.
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @default false
     */
    logAllS3DataEvents(options = {}) {
        return this.addEventSelector(DataResourceType.S3_OBJECT, [`arn:${this.stack.partition}:s3:::`], options);
    }
    /**
     * Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     *
     * @deprecated - use Trail.onEvent()
     */
    onCloudTrailEvent(id, options = {}) {
        return Trail.onEvent(this, id, options);
    }
}
exports.Trail = Trail;
/**
 * Resource type for a data event
 */
var DataResourceType;
(function (DataResourceType) {
    /**
     * Data resource type for Lambda function
     */
    DataResourceType["LAMBDA_FUNCTION"] = "AWS::Lambda::Function";
    /**
     * Data resource type for S3 objects
     */
    DataResourceType["S3_OBJECT"] = "AWS::S3::Object";
})(DataResourceType = exports.DataResourceType || (exports.DataResourceType = {}));
//# sourceMappingURL=data:application/json;base64,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