"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const batch_generated_1 = require("./batch.generated");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 */
class JobDefinition extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const jobDef = new batch_generated_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container),
            type: 'container',
            nodeProperties: props.nodeProps
                ? { mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps),
                    numNodes: props.nodeProps.count }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
        });
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * Imports an existing batch job definition by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobDefinitionArn
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = core_1.Stack.of(scope);
        const jobDefName = stack.parseArn(jobDefinitionArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return undefined;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    buildJobContainer(container) {
        if (container === undefined) {
            return undefined;
        }
        return {
            command: container.command,
            environment: this.deserializeEnvVariables(container.environment),
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            memory: container.memoryLimitMiB || 4,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            resourceRequirements: container.gpuCount
                ? [{ type: 'GPU', value: String(container.gpuCount) }]
                : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            vcpus: container.vcpus || 1,
            volumes: container.volumes,
        };
    }
    buildNodeRangeProps(multiNodeProps) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
}
exports.JobDefinition = JobDefinition;
//# sourceMappingURL=data:application/json;base64,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