"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessLogFormat = exports.AccessLogField = exports.LogGroupLogDestination = void 0;
/**
 * Use CloudWatch Logs as a custom access log destination for API Gateway.
 */
class LogGroupLogDestination {
    constructor(logGroup) {
        this.logGroup = logGroup;
    }
    /**
     * Binds this destination to the CloudWatch Logs.
     */
    bind(_stage) {
        return {
            destinationArn: this.logGroup.logGroupArn,
        };
    }
}
exports.LogGroupLogDestination = LogGroupLogDestination;
/**
 * $context variables that can be used to customize access log pattern.
 */
class AccessLogField {
    /**
     * The API owner's AWS account ID.
     */
    static contextAccountId() {
        return '$context.identity.accountId';
    }
    /**
     * The identifier API Gateway assigns to your API.
     */
    static contextApiId() {
        return '$context.apiId';
    }
    /**
     * A property of the claims returned from the Amazon Cognito user pool after the method caller is successfully authenticated.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html
     *
     * @param property A property key of the claims.
     */
    static contextAuthorizerClaims(property) {
        return `$context.authorizer.claims.${property}`;
    }
    /**
     * The principal user identification associated with the token sent by the client and returned
     * from an API Gateway Lambda authorizer (formerly known as a custom authorizer).
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     */
    static contextAuthorizerPrincipalId() {
        return '$context.authorizer.principalId';
    }
    /**
     * The stringified value of the specified key-value pair of the `context` map returned from an API Gateway Lambda authorizer function.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     * @param property key of the context map.
     */
    static contextAuthorizer(property) {
        return `$context.authorizer.${property}`;
    }
    /**
     * The AWS endpoint's request ID.
     */
    static contextAwsEndpointRequestId() {
        return '$context.awsEndpointRequestId';
    }
    /**
     * The full domain name used to invoke the API. This should be the same as the incoming `Host` header.
     */
    static contextDomainName() {
        return '$context.domainName';
    }
    /**
     * The first label of the `$context.domainName`. This is often used as a caller/customer identifier.
     */
    static contextDomainPrefix() {
        return '$context.domainPrefix';
    }
    /**
     * A string containing an API Gateway error message.
     */
    static contextErrorMessage() {
        return '$context.error.message';
    }
    /**
     * The quoted value of $context.error.message, namely "$context.error.message".
     */
    static contextErrorMessageString() {
        return '$context.error.messageString';
    }
    /**
     * A type of GatewayResponse. This variable can only be used for simple variable substitution in a GatewayResponse body-mapping template,
     * which is not processed by the Velocity Template Language engine, and in access logging.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-websocket-api-logging.html
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html
     */
    static contextErrorResponseType() {
        return '$context.error.responseType';
    }
    /**
     * A string containing a detailed validation error message.
     */
    static contextErrorValidationErrorString() {
        return '$context.error.validationErrorString';
    }
    /**
     * The extended ID that API Gateway assigns to the API request, which contains more useful information for debugging/troubleshooting.
     */
    static contextExtendedRequestId() {
        return '$context.extendedRequestId';
    }
    /**
     * The HTTP method used. Valid values include: `DELETE`, `GET`, `HEAD`, `OPTIONS`, `PATCH`, `POST`, and `PUT`.
     */
    static contextHttpMethod() {
        return '$context.httpMethod';
    }
    /**
     * The AWS account ID associated with the request.
     */
    static contextIdentityAccountId() {
        return '$context.identity.accountId';
    }
    /**
     * For API methods that require an API key, this variable is the API key associated with the method request.
     * For methods that don't require an API key, this variable is
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html
     */
    static contextIdentityApiKey() {
        return '$context.identity.apiKey';
    }
    /**
     * The API key ID associated with an API request that requires an API key.
     */
    static contextIdentityApiKeyId() {
        return '$context.identity.apiKeyId';
    }
    /**
     * The principal identifier of the caller making the request.
     */
    static contextIdentityCaller() {
        return '$context.identity.caller';
    }
    /**
     * The Amazon Cognito authentication provider used by the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html
     */
    static contextIdentityCognitoAuthenticationProvider() {
        return '$context.identity.cognitoAuthenticationProvider';
    }
    /**
     * The Amazon Cognito authentication type of the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoAuthenticationType() {
        return '$context.identity.cognitoAuthenticationType';
    }
    /**
     * The Amazon Cognito identity ID of the caller making the request. Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoIdentityId() {
        return '$context.identity.cognitoIdentityId';
    }
    /**
     * The Amazon Cognito identity pool ID of the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoIdentityPoolId() {
        return '$context.identity.cognitoIdentityPoolId';
    }
    /**
     * The AWS organization ID.
     */
    static contextIdentityPrincipalOrgId() {
        return '$context.identity.principalOrgId';
    }
    /**
     * The source IP address of the TCP connection making the request to API Gateway.
     * Warning: You should not trust this value if there is any chance that the `X-Forwarded-For` header could be forged.
     */
    static contextIdentitySourceIp() {
        return '$context.identity.sourceIp';
    }
    /**
     * The principal identifier of the user making the request. Used in Lambda authorizers.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-output.html
     */
    static contextIdentityUser() {
        return '$context.identity.user';
    }
    /**
     * The User-Agent header of the API caller.
     */
    static contextIdentityUserAgent() {
        return '$context.identity.userAgent';
    }
    /**
     * The Amazon Resource Name (ARN) of the effective user identified after authentication.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
     */
    static contextIdentityUserArn() {
        return '$context.identity.userArn';
    }
    /**
     * The request path.
     * For example, for a non-proxy request URL of https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child,
     * this value is /{stage}/root/child.
     */
    static contextPath() {
        return '$context.path';
    }
    /**
     * The request protocol, for example, HTTP/1.1.
     */
    static contextProtocol() {
        return '$context.protocol';
    }
    /**
     * The ID that API Gateway assigns to the API request.
     */
    static contextRequestId() {
        return '$context.requestId';
    }
    /**
     * The request header override.
     * If this parameter is defined, it contains the headers to be used instead of the HTTP Headers that are defined in the Integration Request pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param headerName
     */
    static contextRequestOverrideHeader(headerName) {
        return `$context.requestOverride.header.${headerName}`;
    }
    /**
     * The request path override. If this parameter is defined,
     * it contains the request path to be used instead of the URL Path Parameters that are defined in the Integration Request pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param pathName
     */
    static contextRequestOverridePath(pathName) {
        return `$context.requestOverride.path.${pathName}`;
    }
    /**
     * The request query string override.
     * If this parameter is defined, it contains the request query strings to be used instead
     * of the URL Query String Parameters that are defined in the Integration Request pane.
     *
     * @param querystringName
     */
    static contextRequestOverrideQuerystring(querystringName) {
        return `$context.requestOverride.querystring.${querystringName}`;
    }
    /**
     * The response header override.
     * If this parameter is defined, it contains the header to be returned instead of the Response header
     * that is defined as the Default mapping in the Integration Response pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param headerName
     */
    static contextResponseOverrideHeader(headerName) {
        return `$context.responseOverride.header.${headerName}`;
    }
    /**
     * The response status code override.
     * If this parameter is defined, it contains the status code to be returned instead of the Method response status
     * that is defined as the Default mapping in the Integration Response pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     */
    static contextResponseOverrideStatus() {
        return '$context.responseOverride.status';
    }
    /**
     * The CLF-formatted request time (dd/MMM/yyyy:HH:mm:ss +-hhmm).
     */
    static contextRequestTime() {
        return '$context.requestTime';
    }
    /**
     * The Epoch-formatted request time.
     */
    static contextRequestTimeEpoch() {
        return '$context.requestTimeEpoch';
    }
    /**
     * The identifier that API Gateway assigns to your resource.
     */
    static contextResourceId() {
        return '$context.resourceId';
    }
    /**
     * The path to your resource.
     * For example, for the non-proxy request URI of `https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child`,
     * The $context.resourcePath value is `/root/child`.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-create-api-step-by-step.html
     */
    static contextResourcePath() {
        return '$context.resourcePath';
    }
    /**
     * The deployment stage of the API request (for example, `Beta` or `Prod`).
     */
    static contextStage() {
        return '$context.stage';
    }
    /**
     * The response received from AWS WAF: `WAF_ALLOW` or `WAF_BLOCK`. Will not be set if the stage is not associated with a web ACL.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     */
    static contextWafResponseCode() {
        return '$context.wafResponseCode';
    }
    /**
     * The complete ARN of the web ACL that is used to decide whether to allow or block the request.
     * Will not be set if the stage is not associated with a web ACL.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     */
    static contextWebaclArn() {
        return '$context.webaclArn';
    }
    /**
     * The trace ID for the X-Ray trace.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-enabling-xray.html
     */
    static contextXrayTraceId() {
        return '$context.xrayTraceId';
    }
    /**
     * The authorizer latency in ms.
     */
    static contextAuthorizerIntegrationLatency() {
        return '$context.authorizer.integrationLatency';
    }
    /**
     * The integration latency in ms.
     */
    static contextIntegrationLatency() {
        return '$context.integrationLatency';
    }
    /**
     * For Lambda proxy integration, this parameter represents the status code returned from AWS Lambda,
     * not from the backend Lambda function.
     */
    static contextIntegrationStatus() {
        return '$context.integrationStatus.';
    }
    /**
     * The response latency in ms.
     */
    static contextResponseLatency() {
        return '$context.responseLatency';
    }
    /**
     * The response payload length.
     */
    static contextResponseLength() {
        return '$context.responseLength';
    }
    /**
     * The method response status.
     */
    static contextStatus() {
        return '$context.status';
    }
}
exports.AccessLogField = AccessLogField;
/**
 * factory methods for access log format.
 */
class AccessLogFormat {
    constructor(format) {
        this.format = format;
    }
    /**
     * Custom log format.
     * You can create any log format string. You can easily get the $ context variable by using the methods of AccessLogField.
     * @param format
     * @example custom(JSON.stringify({
     *  requestId: AccessLogField.contextRequestId(),
     *  sourceIp: AccessLogField.contextIdentitySourceIp(),
     *  method: AccessLogFiled.contextHttpMethod(),
     *  userContext: {
     *    sub: AccessLogField.contextAuthorizerClaims('sub'),
     *    email: AccessLogField.contextAuthorizerClaims('email')
     *  }
     * }))
     */
    static custom(format) {
        return new AccessLogFormat(format);
    }
    /**
     * Generate Common Log Format.
     */
    static clf() {
        const requester = [AccessLogField.contextIdentitySourceIp(), AccessLogField.contextIdentityCaller(), AccessLogField.contextIdentityUser()].join(' ');
        const requestTime = AccessLogField.contextRequestTime();
        const request = [AccessLogField.contextHttpMethod(), AccessLogField.contextResourcePath(), AccessLogField.contextProtocol()].join(' ');
        const status = [AccessLogField.contextStatus(), AccessLogField.contextResponseLength(), AccessLogField.contextRequestId()].join(' ');
        return new AccessLogFormat(`${requester} [${requestTime}] "${request}" ${status}`);
    }
    /**
     * Access log will be produced in the JSON format with a set of fields most useful in the access log. All fields are turned on by default with the
     * option to turn off specific fields.
     */
    static jsonWithStandardFields(fields = {
        ip: true,
        user: true,
        caller: true,
        requestTime: true,
        httpMethod: true,
        resourcePath: true,
        status: true,
        protocol: true,
        responseLength: true,
    }) {
        return this.custom(JSON.stringify({
            requestId: AccessLogField.contextRequestId(),
            ip: fields.ip ? AccessLogField.contextIdentitySourceIp() : undefined,
            user: fields.user ? AccessLogField.contextIdentityUser() : undefined,
            caller: fields.caller ? AccessLogField.contextIdentityCaller() : undefined,
            requestTime: fields.requestTime ? AccessLogField.contextRequestTime() : undefined,
            httpMethod: fields.httpMethod ? AccessLogField.contextHttpMethod() : undefined,
            resourcePath: fields.resourcePath ? AccessLogField.contextResourcePath() : undefined,
            status: fields.status ? AccessLogField.contextStatus() : undefined,
            protocol: fields.protocol ? AccessLogField.contextProtocol() : undefined,
            responseLength: fields.responseLength ? AccessLogField.contextResponseLength() : undefined,
        }));
    }
    /**
     * Output a format string to be used with CloudFormation.
     */
    toString() {
        return this.format;
    }
}
exports.AccessLogFormat = AccessLogFormat;
//# sourceMappingURL=data:application/json;base64,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