"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const logger_1 = require("@aws-lambda-powertools/logger");
const client_codepipeline_1 = require("@aws-sdk/client-codepipeline");
const client_eventbridge_1 = require("@aws-sdk/client-eventbridge");
const REPOSITORY_TYPE = process.env.REPOSITORY_TYPE || '';
const EVENT_SOURCE_NAME = process.env.EVENT_SOURCE_NAME || '';
const logger = new logger_1.Logger();
const codePipeline = new client_codepipeline_1.CodePipelineClient({});
const eventBridge = new client_eventbridge_1.EventBridgeClient({});
exports.handler = async (event) => {
    logger.info('Event', { event });
    const { pipeline: pipelineName, 'execution-id': pipelineExecutionId } = event.detail;
    const state = transformStateName(REPOSITORY_TYPE, event.detail.state);
    if (!state) {
        logger.warn('Ignoring unsupported state change');
        return;
    }
    const execution = await codePipeline.send(new client_codepipeline_1.GetPipelineExecutionCommand({
        pipelineName,
        pipelineExecutionId,
    }));
    const commitSha = execution.pipelineExecution?.artifactRevisions?.[0].revisionId;
    if (!commitSha) {
        logger.warn('Commit hash not found', { execution });
    }
    await eventBridge.send(new client_eventbridge_1.PutEventsCommand({
        Entries: [{
                Source: EVENT_SOURCE_NAME,
                DetailType: event['detail-type'],
                Detail: JSON.stringify({
                    'pipeline-name': pipelineName,
                    'execution-id': pipelineExecutionId,
                    'state': state,
                    'commit-sha': commitSha,
                }),
            }],
    }));
};
const transformStateName = (repositoryType, state) => {
    switch (repositoryType.toLowerCase()) {
        case 'github':
            return transformStateNameForGitHub(state);
        case 'bitbucket':
            return transformStateNameForBitbucket(state);
        default:
            return null;
    }
};
const transformStateNameForGitHub = (state) => {
    switch (state) {
        case 'STARTED':
            return 'pending';
        case 'SUCCEEDED':
            return 'success';
        case 'FAILED':
            return 'failure';
        case 'STOPPED':
            return 'error';
        default:
            return null;
    }
};
const transformStateNameForBitbucket = (state) => {
    switch (state) {
        case 'STARTED':
            return 'INPROGRESS';
        case 'SUCCEEDED':
            return 'SUCCESSFUL';
        case 'FAILED':
            return state;
        case 'STOPPED':
            return state;
        default:
            return null;
    }
};
//# sourceMappingURL=data:application/json;base64,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