"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MainPipeline = void 0;
const aws_events_1 = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const constructs_1 = require("constructs");
const customNodejsFunction_1 = require("./customNodejsFunction");
const path = require("path");
const notificationsTopic_1 = require("./notificationsTopic");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const lodash_1 = require("lodash");
const environment_1 = require("../util/environment");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const types_1 = require("../util/types");
const appStage_1 = require("./appStage");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
class MainPipeline extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const source = pipelines_1.CodePipelineSource.codeCommit(props.codeCommitRepository, props.repository.defaultBranch);
        const pipeline = new pipelines_1.CodePipeline(this, 'Pipeline', lodash_1.merge({
            pipelineName: aws_cdk_lib_1.Stack.of(this).stackName,
            synth: new pipelines_1.ShellStep('Synth', {
                input: source,
                installCommands: [
                    ...(props.commands.preInstall || []),
                    ...(props.commands.install || []),
                ],
                commands: [
                    ...(props.commands.buildAndTest || []),
                    ...props.commands.synthPipeline,
                ],
                primaryOutputDirectory: props.cdkOutputDirectory,
            }),
            crossAccountKeys: true,
            codeBuildDefaults: props.codeBuild,
        }, props.codePipeline));
        props.pipeline.forEach(step => {
            if (this.isWave(step)) {
                this.addWaveDeployment(pipeline, step, props.stacks);
            }
            else {
                this.addEnvironmentDeployment(pipeline, step, props.stacks);
            }
        });
        pipeline.buildPipeline();
        this.createPipelineBuildNotifications(pipeline, props.repository.host, props.repositoryApiDestination);
        this.failuresTopic = this.createPipelineFailuresTopic(pipeline, props.projectName);
    }
    isWave(waveOrEnvironment) {
        return 'wave' in waveOrEnvironment;
    }
    addWaveDeployment(pipeline, step, stacks) {
        const wave = pipeline.addWave(step.wave);
        step.environments.forEach(env => {
            const environmentDeployment = {
                ...env,
                pre: [
                    ...(step.preEachEnvironment || []),
                    ...(env.pre || []),
                ],
                post: [
                    ...(env.post || []),
                    ...(step.postEachEnvironment || []),
                ],
            };
            this.addEnvironmentDeployment(wave, environmentDeployment, stacks, `Wave${capitalizeFirstLetter(step.wave)}`, { WAVE_NAME: step.wave });
        });
        if (step.pre && step.pre.length > 0) {
            wave.addPre(new pipelines_1.ShellStep(`PreWave${capitalizeFirstLetter(step.wave)}`, {
                env: { WAVE_NAME: step.wave },
                commands: step.pre,
            }));
        }
        if (step.post && step.post.length > 0) {
            wave.addPost(new pipelines_1.ShellStep(`PostWave${capitalizeFirstLetter(step.wave)}`, {
                env: { WAVE_NAME: step.wave },
                commands: step.post,
            }));
        }
    }
    addEnvironmentDeployment(parent, step, stacks, idPrefix = '', envVariables) {
        const stage = parent.addStage(new appStage_1.AppStage(this, `${idPrefix}DeployEnv${capitalizeFirstLetter(step.environment)}`, {
            envName: step.environment,
            env: environment_1.getEnvironmentConfig(this, step.environment),
            stacks,
        }));
        if (step.pre && step.pre.length > 0) {
            stage.addPre(new pipelines_1.ShellStep(`${idPrefix}PreEnv${capitalizeFirstLetter(step.environment)}`, {
                env: {
                    ...envVariables,
                    ENV_NAME: step.environment,
                },
                commands: step.pre,
            }));
        }
        if (step.post && step.post.length > 0) {
            stage.addPost(new pipelines_1.ShellStep(`${idPrefix}PostEnv${capitalizeFirstLetter(step.environment)}`, {
                env: {
                    ...envVariables,
                    ENV_NAME: step.environment,
                },
                commands: step.post,
            }));
        }
    }
    createPipelineFailuresTopic(pipeline, projectName) {
        const failuresTopic = new notificationsTopic_1.NotificationsTopic(this, 'PipelineFailuresTopic', {
            projectName: projectName,
            notificationName: 'pipelineFailures',
        });
        // for better visibility, use EventBridge Rules instead of CodeStar Notifications that are generated with pipeline.notifyOn()
        pipeline.pipeline.onStateChange('OnPipelineFailure', {
            eventPattern: {
                detail: {
                    state: ['FAILED'],
                },
            },
            target: new targets.SnsTopic(failuresTopic.topic),
        });
        return failuresTopic.topic;
    }
    /**
     * To send CodePipeline build status back to repository:
     * - trigger Lambda function on CodePipeline state change events,
     * - in Lambda:
     *   - get CodePipeline execution details to get commit SHA,
     *   - send custom event to EventBridge including the commit SHA,
     * - use EventBridge to send build status to repository.
     */
    createPipelineBuildNotifications(pipeline, repositoryType, repositoryApiDestination) {
        const pipelineBuildStatusEventsSourceName = `${aws_cdk_lib_1.Stack.of(this).stackName}.pipelineBuildStatus`;
        const pipelineBuildStatusFunction = new customNodejsFunction_1.CustomNodejsFunction(this, 'PipelineBuildStatus', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambda', 'pipelineBuildStatus')),
            environment: {
                'REPOSITORY_TYPE': repositoryType,
                'EVENT_SOURCE_NAME': pipelineBuildStatusEventsSourceName,
            },
        });
        pipelineBuildStatusFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['codepipeline:GetPipelineExecution'],
            resources: [`arn:aws:codepipeline:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:${pipeline.pipeline.pipelineName}`],
        }));
        pipelineBuildStatusFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['events:PutEvents'],
            resources: [`arn:aws:events:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:event-bus/default`],
            conditions: {
                StringEquals: {
                    'events:source': pipelineBuildStatusEventsSourceName,
                },
            },
        }));
        pipeline.pipeline.onStateChange('OnPipelineStateChange', {
            target: new targets.LambdaFunction(pipelineBuildStatusFunction),
        });
        new aws_events_1.Rule(this, 'SendPipelineStatusToRepositoryRule', {
            eventPattern: {
                source: [pipelineBuildStatusEventsSourceName],
                detailType: ['CodePipeline Pipeline Execution State Change'],
            },
            targets: [
                new targets.ApiDestination(repositoryApiDestination, {
                    pathParameterValues: ['$.detail.commit-sha'],
                    event: this.createStatusEvent(repositoryType),
                }),
            ],
        });
    }
    createStatusEvent(repositoryType) {
        switch (repositoryType) {
            case 'github':
                return aws_events_1.RuleTargetInput.fromObject({
                    'state': aws_events_1.EventField.fromPath('$.detail.state'),
                    'target_url': `https://${aws_events_1.EventField.fromPath('$.region')}.console.aws.amazon.com/codesuite/codepipeline/pipelines/${aws_events_1.EventField.fromPath('$.detail.pipeline-name')}/executions/${aws_events_1.EventField.fromPath('$.detail.execution-id')}`,
                    'context': aws_events_1.EventField.fromPath('$.detail.pipeline-name'),
                });
            case 'bitbucket':
                return aws_events_1.RuleTargetInput.fromObject({
                    'key': 'AWS-PIPELINE-BUILD',
                    'state': aws_events_1.EventField.fromPath('$.detail.state'),
                    'name': aws_events_1.EventField.fromPath('$.detail.pipeline-name'),
                    'description': 'AWS CodePipeline',
                    'url': `https://${aws_events_1.EventField.fromPath('$.region')}.console.aws.amazon.com/codesuite/codepipeline/pipelines/${aws_events_1.EventField.fromPath('$.detail.pipeline-name')}/executions/${aws_events_1.EventField.fromPath('$.detail.execution-id')}`,
                });
            default:
                return types_1.assertUnreachable(repositoryType);
        }
    }
}
exports.MainPipeline = MainPipeline;
const capitalizeFirstLetter = (str) => str.charAt(0).toUpperCase() + str.slice(1);
//# sourceMappingURL=data:application/json;base64,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