"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FeatureBranchBuilds = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const constructs_1 = require("constructs");
const customNodejsFunction_1 = require("./customNodejsFunction");
const path = require("path");
const notificationsTopic_1 = require("./notificationsTopic");
const types_1 = require("../util/types");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
class FeatureBranchBuilds extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const source = aws_codebuild_1.Source.codeCommit({ repository: props.codeCommitRepository });
        const deployProject = this.createDeployProject(source, props.codeBuild, props.commands, props.codeCommitRepository, props.repository.defaultBranch);
        this.createDeployNotifications(deployProject, props.repository.host, props.repositoryApiDestination);
        this.failuresTopic = this.createBuildFailuresTopic(deployProject, props.projectName);
        this.createDestroyProject(source, props.codeBuild, props.commands, props.codeCommitRepository, props.repository.defaultBranch);
    }
    createDeployProject(source, codeBuild, commands, repository, defaultBranch) {
        const deployProject = new aws_codebuild_1.Project(this, 'DeployProject', {
            projectName: `${aws_cdk_lib_1.Stack.of(this).stackName}-featureBranch-deploy`,
            source,
            timeout: codeBuild.timeout,
            environment: codeBuild.buildEnvironment,
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: [
                            'ENV_NAME=$(echo ${BRANCH_NAME} | awk \'{print tolower($0)}\')',
                            ...(commands.preInstall || []),
                            ...(commands.install || []),
                        ],
                    },
                    build: {
                        commands: [
                            ...(commands.buildAndTest || []),
                            ...commands.deployEnvironment,
                        ],
                    },
                },
            }),
        });
        this.grantAssumeCDKRoles(deployProject);
        repository.onCommit('OnBranchCommit', this.createProjectTriggerOptions(deployProject, defaultBranch, true));
        return deployProject;
    }
    createDeployNotifications(deployProject, repositoryType, repositoryApiDestination) {
        const deployStatusEventSourceName = `${aws_cdk_lib_1.Stack.of(this).stackName}.featureBranchDeployStatus`;
        const deployStatusFunction = new customNodejsFunction_1.CustomNodejsFunction(this, 'DeployStatus', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambda', 'featureBranchDeployStatus')),
            environment: {
                'REPOSITORY_TYPE': repositoryType,
                'EVENT_SOURCE_NAME': deployStatusEventSourceName,
            },
        });
        deployStatusFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['events:PutEvents'],
            resources: [`arn:aws:events:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:event-bus/default`],
            conditions: {
                StringEquals: {
                    'events:source': deployStatusEventSourceName,
                },
            },
        }));
        deployProject.onStateChange('OnDeployStateChange', {
            target: new targets.LambdaFunction(deployStatusFunction),
        });
        new aws_events_1.Rule(this, 'SendDeployStatusToRepositoryRule', {
            eventPattern: {
                source: [deployStatusEventSourceName],
                detailType: ['CodeBuild Build State Change'],
            },
            targets: [
                new targets.ApiDestination(repositoryApiDestination, {
                    pathParameterValues: ['$.detail.commit-sha'],
                    event: this.createStatusEvent(repositoryType),
                }),
            ],
        });
    }
    createStatusEvent(repositoryType) {
        switch (repositoryType) {
            case 'github':
                return aws_events_1.RuleTargetInput.fromObject({
                    'state': aws_events_1.EventField.fromPath('$.detail.state'),
                    'target_url': `https://${aws_events_1.EventField.fromPath('$.region')}.console.aws.amazon.com/codesuite/codebuild/projects/${aws_events_1.EventField.fromPath('$.detail.project-name')}/build/${aws_events_1.EventField.fromPath('$.detail.build-id')}`,
                    'context': aws_events_1.EventField.fromPath('$.detail.project-name'),
                });
            case 'bitbucket':
                return aws_events_1.RuleTargetInput.fromObject({
                    'key': 'AWS-PIPELINE-BUILD',
                    'state': aws_events_1.EventField.fromPath('$.detail.state'),
                    'name': aws_events_1.EventField.fromPath('$.detail.project-name'),
                    'description': 'Feature branch deployment on AWS CodeBuild',
                    'url': `https://${aws_events_1.EventField.fromPath('$.region')}.console.aws.amazon.com/codesuite/codebuild/projects/${aws_events_1.EventField.fromPath('$.detail.project-name')}/build/${aws_events_1.EventField.fromPath('$.detail.build-id')}`,
                });
            default:
                return types_1.assertUnreachable(repositoryType);
        }
    }
    createBuildFailuresTopic(deployProject, projectName) {
        const failuresTopic = new notificationsTopic_1.NotificationsTopic(this, 'FeatureBranchBuildFailuresTopic', {
            projectName: projectName,
            notificationName: 'featureBranchBuildFailures',
        });
        // for better visibility, use EventBridge Rules instead of CodeStar Notifications that are generated with project.notifyOnBuildFailed()
        deployProject.onBuildFailed('OnFeatureBuildFailure', {
            target: new targets.SnsTopic(failuresTopic.topic),
        });
        return failuresTopic.topic;
    }
    createDestroyProject(source, codeBuild, commands, repository, defaultBranch) {
        const destroyProject = new aws_codebuild_1.Project(this, 'DestroyProject', {
            projectName: `${aws_cdk_lib_1.Stack.of(this).stackName}-featureBranch-destroy`,
            source,
            timeout: codeBuild.timeout,
            environment: codeBuild.buildEnvironment,
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: [
                            'ENV_NAME=$(echo ${BRANCH_NAME} | awk \'{print tolower($0)}\')',
                            ...(commands.preInstall || []),
                            ...(commands.install || []),
                        ],
                    },
                    build: {
                        commands: [
                            ...commands.destroyEnvironment,
                        ],
                    },
                },
            }),
        });
        this.grantAssumeCDKRoles(destroyProject);
        repository.onReferenceDeleted('OnBranchRemoval', this.createProjectTriggerOptions(destroyProject, defaultBranch));
        return destroyProject;
    }
    createProjectTriggerOptions(targetProject, defaultBranch, withSourceVersion = false) {
        return {
            eventPattern: {
                detail: {
                    referenceType: ['branch'],
                    referenceName: [
                        { 'anything-but': [defaultBranch] },
                    ],
                    referenceFullName: [
                        { 'anything-but': { prefix: 'refs/remotes/' } },
                    ],
                },
            },
            target: new aws_events_targets_1.CodeBuildProject(targetProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    sourceVersion: withSourceVersion ? aws_events_1.EventField.fromPath('$.detail.commitId') : undefined,
                    environmentVariablesOverride: [
                        {
                            name: 'BRANCH_NAME',
                            value: aws_events_1.EventField.fromPath('$.detail.referenceName'),
                            type: 'PLAINTEXT',
                        },
                    ],
                }),
            }),
        };
    }
    grantAssumeCDKRoles(project) {
        const qualifier = aws_cdk_lib_1.Stack.of(this).synthesizer.bootstrapQualifier || 'hnb659fds';
        project.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-deploy-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-file-publishing-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-image-publishing-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
                `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/cdk-${qualifier}-lookup-role-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`,
            ],
        }));
    }
}
exports.FeatureBranchBuilds = FeatureBranchBuilds;
//# sourceMappingURL=data:application/json;base64,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