"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CIStack = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lodash_1 = require("lodash");
const aws_codecommit_1 = require("aws-cdk-lib/aws-codecommit");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const featureBranchBuilds_1 = require("./constructs/featureBranchBuilds");
const mainPipeline_1 = require("./constructs/mainPipeline");
const types_1 = require("./util/types");
const applicationProps_1 = require("./applicationProps");
const aws_chatbot_1 = require("aws-cdk-lib/aws-chatbot");
const notificationsTopic_1 = require("./constructs/notificationsTopic");
const defaultCommands = {
    npm: {
        install: [
            'npm install --location=global aws-cdk@2',
            'npm ci',
        ],
    },
    pnpm: {
        install: [
            'npm install --location=global aws-cdk@2 pnpm',
            'pnpm install --frozen-lockfile',
        ],
    },
};
class CIStack extends aws_cdk_lib_1.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: `${props.projectName}-ci`,
            ...props,
        });
        const resolvedProps = this.resolveProps(props);
        const repository = this.createCodeCommitRepository(props.projectName);
        const repositoryApiDestination = this.createRepositoryApiDestination(props.projectName, props.repository);
        const mainPipeline = new mainPipeline_1.MainPipeline(this, 'MainPipeline', {
            ...resolvedProps,
            codeCommitRepository: repository,
            repositoryApiDestination,
        });
        const featureBranchBuilds = new featureBranchBuilds_1.FeatureBranchBuilds(this, 'FeatureBranchBuilds', {
            ...resolvedProps,
            codeCommitRepository: repository,
            repositoryApiDestination,
        });
        if (resolvedProps.slackNotifications.workspaceId && resolvedProps.slackNotifications.channelId) {
            this.createSlackNotifications(resolvedProps.projectName, resolvedProps.slackNotifications, mainPipeline.failuresTopic, featureBranchBuilds.failuresTopic);
        }
    }
    resolveProps(props) {
        if (props.packageManager) {
            lodash_1.merge(applicationProps_1.defaultProps, { commands: defaultCommands[props.packageManager] });
        }
        return lodash_1.defaultsDeep(lodash_1.cloneDeep(props), applicationProps_1.defaultProps);
    }
    createCodeCommitRepository(projectName) {
        const repository = new aws_codecommit_1.Repository(this, 'Repository', {
            repositoryName: projectName,
        });
        const mirrorUser = new aws_iam_1.User(this, 'RepositoryMirrorUser', {
            userName: `${this.stackName}-repository-mirror-user`,
        });
        repository.grantPullPush(mirrorUser);
        return repository;
    }
    createRepositoryApiDestination(projectName, repository) {
        switch (repository.host) {
            case 'github':
                return new aws_events_1.ApiDestination(this, 'GitHubDestination', {
                    connection: new aws_events_1.Connection(this, 'GitHubConnection', {
                        authorization: aws_events_1.Authorization.apiKey('Authorization', aws_cdk_lib_1.SecretValue.secretsManager(`/${projectName}/githubAuthorization`, { jsonField: 'header' })),
                        description: 'GitHub repository connection',
                        headerParameters: {
                            'Accept': aws_events_1.HttpParameter.fromString('application/vnd.github+json'),
                            'X-GitHub-Api-Version': aws_events_1.HttpParameter.fromString('2022-11-28'),
                        },
                    }),
                    endpoint: `https://api.github.com/repos/${repository.name}/statuses/*`,
                    httpMethod: aws_events_1.HttpMethod.POST,
                });
            case 'bitbucket':
                return new aws_events_1.ApiDestination(this, 'BitbucketDestination', {
                    connection: new aws_events_1.Connection(this, 'BitbucketConnection', {
                        authorization: aws_events_1.Authorization.apiKey('Authorization', aws_cdk_lib_1.SecretValue.secretsManager(`/${projectName}/bitbucketAuthorization`, { jsonField: 'header' })),
                        description: 'Bitbucket repository connection',
                    }),
                    endpoint: `https://api.bitbucket.org/2.0/repositories/${repository.name}/commit/*/statuses/build`,
                    httpMethod: aws_events_1.HttpMethod.POST,
                });
            default:
                return types_1.assertUnreachable(repository.host);
        }
    }
    createSlackNotifications(projectName, config, mainPipelineFailuresTopic, featureBranchFailuresTopic) {
        const alarmsTopic = new notificationsTopic_1.NotificationsTopic(this, 'SlackAlarmsTopic', {
            projectName: projectName,
            notificationName: 'slackAlarms',
        });
        const slack = new aws_chatbot_1.SlackChannelConfiguration(this, 'SlackChannelConfiguration', {
            slackChannelConfigurationName: this.stackName,
            slackWorkspaceId: config.workspaceId,
            slackChannelId: config.channelId,
            notificationTopics: [
                alarmsTopic.topic,
                config.mainPipelineFailures ? mainPipelineFailuresTopic : undefined,
                config.featureBranchFailures ? featureBranchFailuresTopic : undefined,
            ].filter(types_1.notEmpty),
        });
        slack.role?.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchReadOnlyAccess'));
    }
}
exports.CIStack = CIStack;
//# sourceMappingURL=data:application/json;base64,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