"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for rule events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName]
            }
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule compliance events.
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * Imports an existing rule.
     *
     * @param configRuleName the name of the rule
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Restrict scope of changes to a specific resource.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param type the resource type
     * @param identifier the resource identifier
     */
    scopeToResource(type, identifier) {
        this.scopeTo({
            complianceResourceId: identifier,
            complianceResourceTypes: [type],
        });
    }
    /**
     * Restrict scope of changes to specific resource types.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param types resource types
     */
    scopeToResources(...types) {
        this.scopeTo({
            complianceResourceTypes: types
        });
    }
    /**
     * Restrict scope of changes to a specific tag.
     *
     * @param key the tag key
     * @param value the tag value
     */
    scopeToTag(key, value) {
        this.scopeTo({
            tagKey: key,
            tagValue: value
        });
    }
    scopeTo(scope) {
        if (!this.isManaged && !this.isCustomWithChanges) {
            throw new Error('Cannot scope rule when `configurationChanges` is set to false.');
        }
        this.scope = scope;
    }
}
/**
 * The maximum frequency at which the AWS Config rule runs evaluations.
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    /**
     * 1 hour.
     */
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    /**
     * 3 hours.
     */
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    /**
     * 6 hours.
     */
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    /**
     * 12 hours.
     */
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    /**
     * 24 hours.
     */
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * A new managed rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier
            }
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
exports.ManagedRule = ManagedRule;
/**
 * A new custom rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'ConfigurationItemChangeNotification'
            });
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'OversizedConfigurationItemChangeNotification'
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: 'aws.config',
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: 'ScheduledNotification'
            });
        }
        props.lambdaFunction.addPermission('Permission', {
            principal: new iam.ServicePrincipal('config.amazonaws.com')
        });
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn
            }
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
exports.CustomRule = CustomRule;
//# sourceMappingURL=data:application/json;base64,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