"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginRequestQueryStringBehavior = exports.OriginRequestHeaderBehavior = exports.OriginRequestCookieBehavior = exports.OriginRequestPolicy = void 0;
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * A Origin Request Policy configuration.
 *
 * @stability stable
 * @resource AWS::CloudFront::OriginRequestPolicy
 */
class OriginRequestPolicy extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d;
        super(scope, id, {
            physicalName: props.originRequestPolicyName,
        });
        const originRequestPolicyName = (_a = props.originRequestPolicyName) !== null && _a !== void 0 ? _a : core_1.Names.uniqueId(this);
        if (!core_1.Token.isUnresolved(originRequestPolicyName) && !originRequestPolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'originRequestPolicyName' can only include '-', '_', and alphanumeric characters, got: '${props.originRequestPolicyName}'`);
        }
        const cookies = (_b = props.cookieBehavior) !== null && _b !== void 0 ? _b : OriginRequestCookieBehavior.none();
        const headers = (_c = props.headerBehavior) !== null && _c !== void 0 ? _c : OriginRequestHeaderBehavior.none();
        const queryStrings = (_d = props.queryStringBehavior) !== null && _d !== void 0 ? _d : OriginRequestQueryStringBehavior.none();
        const resource = new cloudfront_generated_1.CfnOriginRequestPolicy(this, 'Resource', {
            originRequestPolicyConfig: {
                name: originRequestPolicyName,
                comment: props.comment,
                cookiesConfig: {
                    cookieBehavior: cookies.behavior,
                    cookies: cookies.cookies,
                },
                headersConfig: {
                    headerBehavior: headers.behavior,
                    headers: headers.headers,
                },
                queryStringsConfig: {
                    queryStringBehavior: queryStrings.behavior,
                    queryStrings: queryStrings.queryStrings,
                },
            },
        });
        this.originRequestPolicyId = resource.ref;
    }
    /**
     * Imports a Origin Request Policy from its id.
     *
     * @stability stable
     */
    static fromOriginRequestPolicyId(scope, id, originRequestPolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.originRequestPolicyId = originRequestPolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed origin request policy. */
    static fromManagedOriginRequestPolicy(managedOriginRequestPolicyId) {
        return new class {
            constructor() {
                this.originRequestPolicyId = managedOriginRequestPolicyId;
            }
        }();
    }
}
exports.OriginRequestPolicy = OriginRequestPolicy;
/**
 * This policy includes only the User-Agent and Referer headers.
 *
 * It doesn’t include any query strings or cookies.
 *
 * @stability stable
 */
OriginRequestPolicy.USER_AGENT_REFERER_HEADERS = OriginRequestPolicy.fromManagedOriginRequestPolicy('acba4595-bd28-49b8-b9fe-13317c0390fa');
/**
 * This policy includes the header that enables cross-origin resource sharing (CORS) requests when the origin is a custom origin.
 *
 * @stability stable
 */
OriginRequestPolicy.CORS_CUSTOM_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('59781a5b-3903-41f3-afcb-af62929ccde1');
/**
 * This policy includes the headers that enable cross-origin resource sharing (CORS) requests when the origin is an Amazon S3 bucket.
 *
 * @stability stable
 */
OriginRequestPolicy.CORS_S3_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('88a5eaf4-2fd4-4709-b370-b4c650ea3fcf');
/**
 * This policy includes all values (query strings, headers, and cookies) in the viewer request.
 *
 * @stability stable
 */
OriginRequestPolicy.ALL_VIEWER = OriginRequestPolicy.fromManagedOriginRequestPolicy('216adef6-5c7f-47e4-b989-5492eafa07d3');
/**
 * This policy is designed for use with an origin that is an AWS Elemental MediaTailor endpoint.
 *
 * @stability stable
 */
OriginRequestPolicy.ELEMENTAL_MEDIA_TAILOR = OriginRequestPolicy.fromManagedOriginRequestPolicy('775133bc-15f2-49f9-abea-afb2e0bf67d2');
/**
 * Ddetermines whether any cookies in viewer requests (and if so, which cookies) are included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class OriginRequestCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * Cookies in viewer requests are not included in requests that CloudFront sends to the origin.
     *
     * Any cookies that are listed in a CachePolicy are still included in origin requests.
     *
     * @stability stable
     */
    static none() { return new OriginRequestCookieBehavior('none'); }
    /**
     * All cookies in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all() { return new OriginRequestCookieBehavior('all'); }
    /**
     * Only the provided `cookies` are included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new OriginRequestCookieBehavior('whitelist', cookies);
    }
}
exports.OriginRequestCookieBehavior = OriginRequestCookieBehavior;
/**
 * Determines whether any HTTP headers (and if so, which headers) are included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class OriginRequestHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /**
     * HTTP headers are not included in requests that CloudFront sends to the origin.
     *
     * Any headers that are listed in a CachePolicy are still included in origin requests.
     *
     * @stability stable
     */
    static none() { return new OriginRequestHeaderBehavior('none'); }
    /**
     * All HTTP headers in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * Additionally, any additional CloudFront headers provided are included; the additional headers are added by CloudFront.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-cloudfront-headers.html
     * @stability stable
     */
    static all(...cloudfrontHeaders) {
        if (cloudfrontHeaders.length > 0) {
            if (!cloudfrontHeaders.every(header => header.startsWith('CloudFront-'))) {
                throw new Error('additional CloudFront headers passed to `OriginRequestHeaderBehavior.all()` must begin with \'CloudFront-\'');
            }
            return new OriginRequestHeaderBehavior('allViewerAndWhitelistCloudFront', cloudfrontHeaders);
        }
        else {
            return new OriginRequestHeaderBehavior('allViewer');
        }
    }
    /**
     * Listed headers are included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        return new OriginRequestHeaderBehavior('whitelist', headers);
    }
}
exports.OriginRequestHeaderBehavior = OriginRequestHeaderBehavior;
/**
 * Determines whether any URL query strings in viewer requests (and if so, which query strings) are included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class OriginRequestQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * Query strings in viewer requests are not included in requests that CloudFront sends to the origin.
     *
     * Any query strings that are listed in a CachePolicy are still included in origin requests.
     *
     * @stability stable
     */
    static none() { return new OriginRequestQueryStringBehavior('none'); }
    /**
     * All query strings in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all() { return new OriginRequestQueryStringBehavior('all'); }
    /**
     * Only the provided `queryStrings` are included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new OriginRequestQueryStringBehavior('whitelist', queryStrings);
    }
}
exports.OriginRequestQueryStringBehavior = OriginRequestQueryStringBehavior;
//# sourceMappingURL=data:application/json;base64,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