# -----------------------------------------------------------------
# -----------------------------------------------------------------
#
#  Mandrake lib (mlib) for calculating randomized dependence coefficient
#
# -----------------------------------------------------------------
# -----------------------------------------------------------------

import numpy as N


# The "ordinary" correlation coefficient, -1 to +1, linear relationships
def pearson_correlation(x, y):
    return N.corrcoef(x, y)[0, 1]


# Performs an ensemble of rdc calls and takes a function of the results before returning
# This is an attempt to correct for the ~1% of spurious values that can be generated by rdc
def rdc(x, y, num_iterations=10, func=N.median, f=N.sin, k=20, s=1 / 6.):
    return func([randomized_dependence_coefficient(x, y, f=f, k=k, s=s) for i in range(num_iterations)])


def randomized_dependence_coefficient(x, y, f=N.sin, k=20, s=1 / 6.):
    """

    Implements the Randomized Dependence Coefficient
    David Lopez-Paz, Philipp Hennig, Bernhard Schoelkopf

    http://papers.nips.cc/paper/5138-the-randomized-dependence-coefficient.pdf

    Computes the Randomized Dependence Coefficient
    x,y: numpy arrays 1-D or 2-D
         If 1-D, size (samples,)
         If 2-D, size (samples, variables)
    f:   function to use for random projection
    k:   number of random projections to use
    s:   scale parameter

    According to the paper, the coefficient should be relatively insensitive to
    the settings of the last three parameters.

    Base use case
    >>> N.random.seed(0)
    >>> randomized_dependence_coefficient(range(100), N.random.random(100))
    0.33...

    Protect ourselves from pure relationships
    >>> N.random.seed(0)
    >>> randomized_dependence_coefficient(range(100), range(100))
    1.0
    >>> N.random.seed(0)
    >>> randomized_dependence_coefficient([1,]*100,[1,]*100)
    1.0
    >>> N.random.seed(0)
    >>> randomized_dependence_coefficient([1,]*100,[2,]*100)
    0.99...

    NaN case
    >>> N.random.seed(0)
    >>> randomized_dependence_coefficient([1,2,3,4,N.nan],[1,2,3,4,5])
    nan

    Size errors
    >>> randomized_dependence_coefficient([],[1,])
    Traceback (most recent call last):
    Exception: Arrays must have matching size

    Tiny input cases
    >>> N.random.seed(0)
    >>> randomized_dependence_coefficient([],[])
    nan
    >>> N.random.seed(0)
    >>> randomized_dependence_coefficient([1,],[2,])
    nan
    >>> N.random.seed(0)
    >>> randomized_dependence_coefficient(range(20),N.random.random(20))
    Traceback (most recent call last):
    Exception: RDC is not reliable with short vectors less than 50 (likely scales with k)

    """

    from scipy.stats import rankdata

    x = N.array(x)
    y = N.array(y)

    if len(x) != len(y):
        raise Exception("Arrays must have matching size")

    # Handle zero or singleton length arrays
    if len(x) < 2:
        return N.nan

    # Handle nan's
    if N.any(N.isnan(x)) or N.any(N.isnan(y)):
        return N.nan

    # Immediately handle identical x,y case for speed and to avoid singular matrices
    if N.all(x == y):
        return 1.0

    # Handle short vectors
    if len(x) < 50:
        raise Exception("RDC is not reliable with short vectors less than 50 (likely scales with k)")

    if len(x.shape) == 1: x = x.reshape((-1, 1))
    if len(y.shape) == 1: y = y.reshape((-1, 1))

    # Copula Transformation
    cx = N.column_stack([rankdata(xc, method='ordinal') for xc in x.T]) / float(x.size)
    cy = N.column_stack([rankdata(yc, method='ordinal') for yc in y.T]) / float(y.size)

    # Add a vector of ones so that w.x + b is just a dot product
    O = N.ones(cx.shape[0])
    X = N.column_stack([cx, O])
    Y = N.column_stack([cy, O])

    # Random linear projections
    Rx = (s / X.shape[1]) * N.random.randn(X.shape[1], k)
    Ry = (s / Y.shape[1]) * N.random.randn(Y.shape[1], k)
    X = N.dot(X, Rx)
    Y = N.dot(Y, Ry)

    # Apply non-linear function to random projections
    fX = f(X)
    fY = f(Y)

    # Compute full covariance matrix
    C = N.cov(N.hstack([fX, fY]).T)

    # Due to numerical issues, if k is too large,
    # then rank(fX) < k or rank(fY) < k, so we need
    # to find the largest k such that the eigenvalues
    # (canonical correlations) are real-valued
    k0 = k
    lb = 1
    ub = k

    while True:
        # Compute canonical correlations
        Cxx = C[:k, :k]
        Cyy = C[k0:k0 + k, k0:k0 + k]
        Cxy = C[:k, k0:k0 + k]
        Cyx = C[k0:k0 + k, :k]

        if len(x) == 20:
            print(N.any(N.isnan(Cxx)), N.any(N.isnan(Cyy)), N.any(N.isnan(Cxy)), N.any(N.isnan(Cyx)))
            print(Cxx.shape)

        # Handle if inversion goes singular... this means there is a 1=1 degeneracy, and thus the rdc is 1.0
        try:
            eigs = N.linalg.eigvals(N.dot(N.dot(N.linalg.pinv(Cxx), Cxy),
                                          N.dot(N.linalg.pinv(Cyy), Cyx)))
        except N.linalg.linalg.LinAlgError as err:
            if str(err) == 'Singular matrix':
                eigs = [1.0, ]
                break
            else:
                raise

        # Binary search if k is too large
        if not (N.all(N.isreal(eigs)) and 0 <= N.min(eigs) and N.max(eigs) <= 1):
            ub -= 1
            k = (ub + lb) // 2
            continue
        if lb == ub:
            break
        lb = k
        if ub == lb + 1:
            k = ub
        else:
            k = (ub + lb) // 2

    return N.sqrt(N.max(eigs))


if __name__ == "__main__":
    import doctest

    N.random.seed(0)
    randomized_dependence_coefficient([1, ] * 100, [2, ] * 100)
    # randomized_dependence_coefficient(range(100), range(100))
    # randomized_dependence_coefficient(range(100), N.random.random(100))

    doctest.testmod(optionflags=doctest.ELLIPSIS)
