# Complete implementation of VirtualBox's COM API with a Pythonic interface.
#
# Note: Commenting, and API structure generation was carved from
#       VirtualBox project's VirtualBox.xidl Main API definition.
#
from __future__ import absolute_import
from .library_base import Enum
from .library_base import Interface
from .library_base import VBoxError

# Py2 and Py3 compatibility
try:
    import __builtin__ as builtin
except:
    import builtins as builtin
try:
    basestring = basestring
except:
    basestring = (str, bytes)
try:
    baseinteger = (int, long)
except:
    baseinteger = (int,)


__doc__ = """\
  Welcome to the **VirtualBox Main API documentation**. This documentation
describes the so-called *VirtualBox Main API* which comprises all public
COM interfaces and components provided by the VirtualBox server and by the
VirtualBox client library.

VirtualBox employs a client-server design, meaning that whenever any part of
VirtualBox is running -- be it the Qt GUI, the VBoxManage command-line
interface or any virtual machine --, a dedicated server process named
VBoxSVC runs in the background. This allows multiple processes working with
VirtualBox to cooperate without conflicts. These processes communicate to each
other using inter-process communication facilities provided by the COM
implementation of the host computer.

On Windows platforms, the VirtualBox Main API uses Microsoft COM, a native COM
implementation. On all other platforms, Mozilla XPCOM, an open-source COM
implementation, is used.

All the parts that a typical VirtualBox user interacts with (the Qt GUI
and the VBoxManage command-line interface) are technically
front-ends to the Main API and only use the interfaces that are documented
in this Main API documentation. This ensures that, with any given release
version of VirtualBox, all capabilities of the product that could be useful
to an external client program are always exposed by way of this API.

The VirtualBox Main API (also called the *VirtualBox COM library*)
contains two public component classes:
:py:class:`IVirtualBox` and :py:class:`ISession`, which
implement IVirtualBox and ISession interfaces respectively. These two classes
are of supreme importance and will be needed in order for any front-end
program to do anything useful. It is recommended to read the documentation of
the mentioned interfaces first.

The :py:class:`IVirtualBox` class is a singleton. This means that
there can be only one object of this class on the local machine at any given
time. This object is a parent of many other objects in the VirtualBox COM
library and lives in the VBoxSVC process. In fact, when you create an instance
of the :py:class:`IVirtualBox`, the COM subsystem checks if the VBoxSVC
process is already running, starts it if not, and returns you a reference to
the VirtualBox object created in this process. When the last reference
to this object is released, the VBoxSVC process ends (with a 5 second delay to
protect from too frequent restarts).

The :py:class:`ISession` class is a regular component. You can create
as many Session objects as you need but all of them will live in a
process which issues the object instantiation call. Session objects
represent virtual machine sessions which are used to configure virtual
machines and control their execution.

The naming of methods and attributes is very clearly defined: they all start
with a lowercase letter (except if they start with an acronym), and are using
CamelCase style otherwise. This naming only applies to the IDL description,
and is modified by the various language bindings (some convert the first
character to upper case, some not). See the SDK reference for more details
about how to call a method or attribute from a specific programming language.
"""


vbox_version = "6.1.16"
lib_version = 1.3
lib_app_uuid = "819B4D85-9CEE-493C-B6FC-64FFE759B3C9"
lib_uuid = "d7569351-1750-46f0-936e-bd127d5bc264"
xidl_hash = "09a60b2cdcbf61f994e4eb01863c81d1"


class VBoxErrorObjectNotFound(VBoxError):
    """Object corresponding to the supplied arguments does not exist."""

    name = "VBOX_E_OBJECT_NOT_FOUND"
    value = 0x80BB0001


class VBoxErrorInvalidVmState(VBoxError):
    """Current virtual machine state prevents the operation."""

    name = "VBOX_E_INVALID_VM_STATE"
    value = 0x80BB0002


class VBoxErrorVmError(VBoxError):
    """Virtual machine error occurred attempting the operation."""

    name = "VBOX_E_VM_ERROR"
    value = 0x80BB0003


class VBoxErrorFileError(VBoxError):
    """File not accessible or erroneous file contents."""

    name = "VBOX_E_FILE_ERROR"
    value = 0x80BB0004


class VBoxErrorIprtError(VBoxError):
    """Runtime subsystem error."""

    name = "VBOX_E_IPRT_ERROR"
    value = 0x80BB0005


class VBoxErrorPdmError(VBoxError):
    """Pluggable Device Manager error."""

    name = "VBOX_E_PDM_ERROR"
    value = 0x80BB0006


class VBoxErrorInvalidObjectState(VBoxError):
    """Current object state prohibits operation."""

    name = "VBOX_E_INVALID_OBJECT_STATE"
    value = 0x80BB0007


class VBoxErrorHostError(VBoxError):
    """Host operating system related error."""

    name = "VBOX_E_HOST_ERROR"
    value = 0x80BB0008


class VBoxErrorNotSupported(VBoxError):
    """Requested operation is not supported."""

    name = "VBOX_E_NOT_SUPPORTED"
    value = 0x80BB0009


class VBoxErrorXmlError(VBoxError):
    """Invalid XML found."""

    name = "VBOX_E_XML_ERROR"
    value = 0x80BB000A


class VBoxErrorInvalidSessionState(VBoxError):
    """Current session state prohibits operation."""

    name = "VBOX_E_INVALID_SESSION_STATE"
    value = 0x80BB000B


class VBoxErrorObjectInUse(VBoxError):
    """Object being in use prohibits operation."""

    name = "VBOX_E_OBJECT_IN_USE"
    value = 0x80BB000C


class VBoxErrorPasswordIncorrect(VBoxError):
    """A provided password was incorrect."""

    name = "VBOX_E_PASSWORD_INCORRECT"
    value = 0x80BB000D


class VBoxErrorMaximumReached(VBoxError):
    """A maximum has been reached."""

    name = "VBOX_E_MAXIMUM_REACHED"
    value = 0x80BB000E


class VBoxErrorGstctlGuestError(VBoxError):
    """Guest Control reported an error from the guest side."""

    name = "VBOX_E_GSTCTL_GUEST_ERROR"
    value = 0x80BB000F


class VBoxErrorTimeout(VBoxError):
    """The operation ran into an explicitly requested timeout."""

    name = "VBOX_E_TIMEOUT"
    value = 0x80BB0010


class OleErrorFail(VBoxError):
    """Unspecified error"""

    name = "E_FAIL"
    value = 0x80004005


class OleErrorNointerface(VBoxError):
    """No such interface supported"""

    name = "E_NOINTERFACE"
    value = 0x80004002


class OleErrorAccessdenied(VBoxError):
    """General access denied error"""

    name = "E_ACCESSDENIED"
    value = 0x80070005


class OleErrorNotimpl(VBoxError):
    """Not implemented"""

    name = "E_NOTIMPL"
    value = 0x80004001


class OleErrorUnexpected(VBoxError):
    """Catastrophic failure"""

    name = "E_UNEXPECTED"
    value = 0x8000FFFF


class OleErrorInvalidarg(VBoxError):
    """One or more arguments are invalid"""

    name = "E_INVALIDARG"
    value = 0x80070057


class SettingsVersion(Enum):
    """Settings version of VirtualBox settings files. This is written to
    the "version" attribute of the root "VirtualBox" element in the settings
    file XML and indicates which VirtualBox version wrote the file.


    .. describe:: null(0)

            Null value, indicates invalid version.

    .. describe:: v1_0(1)

            Legacy settings version, not currently supported.

    .. describe:: v1_1(2)

            Legacy settings version, not currently supported.

    .. describe:: v1_2(3)

            Legacy settings version, not currently supported.

    .. describe:: v1_3pre(4)

            Legacy settings version, not currently supported.

    .. describe:: v1_3(5)

            Settings version "1.3", written by VirtualBox 2.0.12.

    .. describe:: v1_4(6)

            Intermediate settings version, understood by VirtualBox 2.1.x.

    .. describe:: v1_5(7)

            Intermediate settings version, understood by VirtualBox 2.1.x.

    .. describe:: v1_6(8)

            Settings version "1.6", written by VirtualBox 2.1.4 (at least).

    .. describe:: v1_7(9)

            Settings version "1.7", written by VirtualBox 2.2.x and 3.0.x.

    .. describe:: v1_8(10)

            Intermediate settings version "1.8", understood by VirtualBox 3.1.x.

    .. describe:: v1_9(11)

            Settings version "1.9", written by VirtualBox 3.1.x.

    .. describe:: v1_10(12)

            Settings version "1.10", written by VirtualBox 3.2.x.

    .. describe:: v1_11(13)

            Settings version "1.11", written by VirtualBox 4.0.x.

    .. describe:: v1_12(14)

            Settings version "1.12", written by VirtualBox 4.1.x.

    .. describe:: v1_13(15)

            Settings version "1.13", written by VirtualBox 4.2.x.

    .. describe:: v1_14(16)

            Settings version "1.14", written by VirtualBox 4.3.x.

    .. describe:: v1_15(17)

            Settings version "1.15", written by VirtualBox 5.0.x.

    .. describe:: v1_16(18)

            Settings version "1.16", written by VirtualBox 5.1.x.

    .. describe:: v1_17(19)

            Settings version "1.17", written by VirtualBox 6.0.x.

    .. describe:: v1_18(20)

            Settings version "1.18", written by VirtualBox 6.1.x.

    .. describe:: future(99999)

            Settings version greater than "1.15", written by a future VirtualBox version.

    """

    __uuid__ = "b4cc23c2-96f2-419d-830b-bd13c1135dfb"
    _enums = [
        ("Null", 0, """Null value, indicates invalid version."""),
        ("v1_0", 1, """Legacy settings version, not currently supported."""),
        ("v1_1", 2, """Legacy settings version, not currently supported."""),
        ("v1_2", 3, """Legacy settings version, not currently supported."""),
        ("v1_3pre", 4, """Legacy settings version, not currently supported."""),
        ("v1_3", 5, """Settings version "1.3", written by VirtualBox 2.0.12."""),
        (
            "v1_4",
            6,
            """Intermediate settings version, understood by VirtualBox 2.1.x.""",
        ),
        (
            "v1_5",
            7,
            """Intermediate settings version, understood by VirtualBox 2.1.x.""",
        ),
        (
            "v1_6",
            8,
            """Settings version "1.6", written by VirtualBox 2.1.4 (at least).""",
        ),
        (
            "v1_7",
            9,
            """Settings version "1.7", written by VirtualBox 2.2.x and 3.0.x.""",
        ),
        (
            "v1_8",
            10,
            """Intermediate settings version "1.8", understood by VirtualBox 3.1.x.""",
        ),
        ("v1_9", 11, """Settings version "1.9", written by VirtualBox 3.1.x."""),
        ("v1_10", 12, """Settings version "1.10", written by VirtualBox 3.2.x."""),
        ("v1_11", 13, """Settings version "1.11", written by VirtualBox 4.0.x."""),
        ("v1_12", 14, """Settings version "1.12", written by VirtualBox 4.1.x."""),
        ("v1_13", 15, """Settings version "1.13", written by VirtualBox 4.2.x."""),
        ("v1_14", 16, """Settings version "1.14", written by VirtualBox 4.3.x."""),
        ("v1_15", 17, """Settings version "1.15", written by VirtualBox 5.0.x."""),
        ("v1_16", 18, """Settings version "1.16", written by VirtualBox 5.1.x."""),
        ("v1_17", 19, """Settings version "1.17", written by VirtualBox 6.0.x."""),
        ("v1_18", 20, """Settings version "1.18", written by VirtualBox 6.1.x."""),
        (
            "Future",
            99999,
            """Settings version greater than "1.15", written by a future VirtualBox version.""",
        ),
    ]


class AccessMode(Enum):
    """Access mode for opening files.


    .. describe:: read_only(1)



    .. describe:: read_write(2)



    """

    __uuid__ = "1da0007c-ddf7-4be8-bcac-d84a1558785f"
    _enums = [
        ("ReadOnly", 1, """"""),
        ("ReadWrite", 2, """"""),
    ]


class MachineState(Enum):
    """Virtual machine execution state.

    This enumeration represents possible values of the :py:func:`IMachine.state`  attribute.

    Below is the basic virtual machine state diagram. It shows how the state
    changes during virtual machine execution. The text in square braces shows
    a method of the IConsole or IMachine interface that performs the given state
    transition.


    ::

         +---------[powerDown()] <- Stuck <--[failure]-+
         V                                             |
         +-> PoweredOff --+-->[powerUp()]--> Starting --+      | +-----[resume()]-----+
         |                |                             |      | V                    |
         |   Aborted -----+                             +--> Running --[pause()]--> Paused
         |                                              |      ^ |                   ^ |
         |   Saved -----------[powerUp()]--> Restoring -+      | |                   | |
         |     ^                                               | |                   | |
         |     |     +-----------------------------------------+-|-------------------+ +
         |     |     |                                           |                     |
         |     |     +- OnlineSnapshotting <--[takeSnapshot()]<--+---------------------+
         |     |                                                 |                     |
         |     +-------- Saving <--------[saveState()]<----------+---------------------+
         |                                                       |                     |
         +-------------- Stopping -------[powerDown()]<----------+---------------------+



    Note that states to the right from PoweredOff, Aborted and Saved in the
    above diagram are called *online VM states*. These states
    represent the virtual machine which is being executed in a dedicated
    process (usually with a GUI window attached to it where you can see the
    activity of the virtual machine and interact with it). There are two
    special pseudo-states, FirstOnline and LastOnline, that can be used in
    relational expressions to detect if the given machine state is online or
    not:


    ::

         if (machine.GetState() >= MachineState_FirstOnline &&
         machine.GetState() <= MachineState_LastOnline)
         {
         ...the machine is being executed...
         }



    When the virtual machine is in one of the online VM states (that is, being
    executed), only a few machine settings can be modified. Methods working
    with such settings contain an explicit note about that. An attempt to
    change any other setting or perform a modifying operation during this time
    will result in the @c VBOX_E_INVALID_VM_STATE error.

    All online states except Running, Paused and Stuck are transitional: they
    represent temporary conditions of the virtual machine that will last as
    long as the operation that initiated such a condition.

    The Stuck state is a special case. It means that execution of the machine
    has reached the "Guru Meditation" condition. This condition indicates an
    internal VMM (virtual machine manager) failure which may happen as a
    result of either an unhandled low-level virtual hardware exception or one
    of the recompiler exceptions (such as the *too-many-traps*
    condition).

    Note also that any online VM state may transit to the Aborted state. This
    happens if the process that is executing the virtual machine terminates
    unexpectedly (for example, crashes). Other than that, the Aborted state is
    equivalent to PoweredOff.

    There are also a few additional state diagrams that do not deal with
    virtual machine execution and therefore are shown separately. The states
    shown on these diagrams are called *offline VM states* (this includes
    PoweredOff, Aborted and Saved too).

    The first diagram shows what happens when a lengthy setup operation is
    being executed (such as :py:func:`IMachine.attach_device` ).


    ::

         +----------------------------------(same state as before the call)------+
         |                                                                       |
         +-> PoweredOff --+                                                      |
         |                |                                                      |
         |-> Aborted -----+-->[lengthy VM configuration call] --> SettingUp -----+
         |                |
         +-> Saved -------+



    The next two diagrams demonstrate the process of taking a snapshot of a
    powered off virtual machine, restoring the state to that as of a snapshot
    or deleting a snapshot, respectively.


    ::

         +----------------------------------(same state as before the call)------+
         |                                                                       |
         +-> PoweredOff --+                                                      |
         |                +-->[takeSnapshot()] ------------------> Snapshotting -+
         +-> Aborted -----+

         +-> PoweredOff --+
         |                |
         |   Aborted -----+-->[restoreSnapshot()    ]-------> RestoringSnapshot -+
         |                |   [deleteSnapshot()     ]-------> DeletingSnapshot --+
         +-> Saved -------+                                                      |
         |                                                                       |
         +---(Saved if restored from an online snapshot, PoweredOff otherwise)---+




    For whoever decides to touch this enum: In order to keep the
    comparisons involving FirstOnline and LastOnline pseudo-states valid,
    the numeric values of these states must be correspondingly updated if
    needed: for any online VM state, the condition
    FirstOnline <= state <= LastOnline must be
    @c true. The same relates to transient states for which
    the condition FirstOnline <= state <= LastOnline must be
    @c true.


    .. describe:: null(0)

            Null value (never used by the API).

    .. describe:: powered_off(1)

            The machine is not running and has no saved execution state; it has
            either never been started or been shut down successfully.

    .. describe:: saved(2)

            The machine is not currently running, but the execution state of the machine
            has been saved to an external file when it was running, from where
            it can be resumed.

    .. describe:: teleported(3)

            The machine was teleported to a different host (or process) and then
            powered off. Take care when powering it on again may corrupt resources
            it shares with the teleportation target (e.g. disk and network).

    .. describe:: aborted(4)

            The process running the machine has terminated abnormally. This may
            indicate a crash of the VM process in host execution context, or
            the VM process has been terminated externally.

    .. describe:: running(5)

            The machine is currently being executed.

            For whoever decides to touch this enum: In order to keep the
            comparisons in the old source code valid, this state must immediately
            precede the Paused state.

            @todo Lift this spectacularly wonderful restriction.

    .. describe:: paused(6)

            Execution of the machine has been paused.

            For whoever decides to touch this enum: In order to keep the
            comparisons in the old source code valid, this state must immediately
            follow the Running state.

            @todo Lift this spectacularly wonderful restriction.

    .. describe:: stuck(7)

            Execution of the machine has reached the "Guru Meditation"
            condition. This indicates a severe error in the hypervisor itself.

            bird: Why this uncool name? Could we rename it to "GuruMeditation" or
            "Guru", perhaps? Or are there some other VMM states that are
            intended to be lumped in here as well?

    .. describe:: teleporting(8)

            The machine is about to be teleported to a different host or process.
            It is possible to pause a machine in this state, but it will go to the
            @c TeleportingPausedVM state and it will not be
            possible to resume it again unless the teleportation fails.

    .. describe:: live_snapshotting(9)

            A live snapshot is being taken. The machine is running normally, but
            some of the runtime configuration options are inaccessible. Also, if
            paused while in this state it will transition to
            @c OnlineSnapshotting and it will not be resume the
            execution until the snapshot operation has completed.

    .. describe:: starting(10)

            Machine is being started after powering it on from a
            zero execution state.

    .. describe:: stopping(11)

            Machine is being normally stopped powering it off, or after the guest OS
            has initiated a shutdown sequence.

    .. describe:: saving(12)

            Machine is saving its execution state to a file.

    .. describe:: restoring(13)

            Execution state of the machine is being restored from a file
            after powering it on from the saved execution state.

    .. describe:: teleporting_paused_vm(14)

            The machine is being teleported to another host or process, but it is
            not running. This is the paused variant of the
            @c Teleporting state.

    .. describe:: teleporting_in(15)

            Teleporting the machine state in from another host or process.

    .. describe:: deleting_snapshot_online(16)

            Like @c DeletingSnapshot, but the merging of media is ongoing in
            the background while the machine is running.

    .. describe:: deleting_snapshot_paused(17)

            Like @c DeletingSnapshotOnline, but the machine was paused when the
            merging of differencing media was started.

    .. describe:: online_snapshotting(18)

            Like @c LiveSnapshotting, but the machine was paused when the
            merging of differencing media was started.

    .. describe:: restoring_snapshot(19)

            A machine snapshot is being restored; this typically does not take long.

    .. describe:: deleting_snapshot(20)

            A machine snapshot is being deleted; this can take a long time since this
            may require merging differencing media. This value indicates that the
            machine is not running while the snapshot is being deleted.

    .. describe:: setting_up(21)

            Lengthy setup operation is in progress.

    .. describe:: snapshotting(22)

            Taking an (offline) snapshot.

    .. describe:: first_online(5)

            Pseudo-state: first online state (for use in relational expressions).

    .. describe:: last_online(18)

            Pseudo-state: last online state (for use in relational expressions).

    .. describe:: first_transient(8)

            Pseudo-state: first transient state (for use in relational expressions).

    .. describe:: last_transient(22)

            Pseudo-state: last transient state (for use in relational expressions).

    """

    __uuid__ = "00bc01b5-00a4-48db-000a-9061008357aa"
    _enums = [
        ("Null", 0, """Null value (never used by the API)."""),
        (
            "PoweredOff",
            1,
            """The machine is not running and has no saved execution state; it has
            either never been started or been shut down successfully.""",
        ),
        (
            "Saved",
            2,
            """The machine is not currently running, but the execution state of the machine
            has been saved to an external file when it was running, from where
            it can be resumed.""",
        ),
        (
            "Teleported",
            3,
            """The machine was teleported to a different host (or process) and then
            powered off. Take care when powering it on again may corrupt resources
            it shares with the teleportation target (e.g. disk and network).""",
        ),
        (
            "Aborted",
            4,
            """The process running the machine has terminated abnormally. This may
            indicate a crash of the VM process in host execution context, or
            the VM process has been terminated externally.""",
        ),
        (
            "Running",
            5,
            """The machine is currently being executed.
            
            For whoever decides to touch this enum: In order to keep the
            comparisons in the old source code valid, this state must immediately
            precede the Paused state.
            
            @todo Lift this spectacularly wonderful restriction.""",
        ),
        (
            "Paused",
            6,
            """Execution of the machine has been paused.
            
            For whoever decides to touch this enum: In order to keep the
            comparisons in the old source code valid, this state must immediately
            follow the Running state.
            
            @todo Lift this spectacularly wonderful restriction.""",
        ),
        (
            "Stuck",
            7,
            """Execution of the machine has reached the "Guru Meditation"
            condition. This indicates a severe error in the hypervisor itself.
            
            bird: Why this uncool name? Could we rename it to "GuruMeditation" or
            "Guru", perhaps? Or are there some other VMM states that are
            intended to be lumped in here as well?""",
        ),
        (
            "Teleporting",
            8,
            """The machine is about to be teleported to a different host or process.
            It is possible to pause a machine in this state, but it will go to the
            @c TeleportingPausedVM state and it will not be
            possible to resume it again unless the teleportation fails.""",
        ),
        (
            "LiveSnapshotting",
            9,
            """A live snapshot is being taken. The machine is running normally, but
            some of the runtime configuration options are inaccessible. Also, if
            paused while in this state it will transition to
            @c OnlineSnapshotting and it will not be resume the
            execution until the snapshot operation has completed.""",
        ),
        (
            "Starting",
            10,
            """Machine is being started after powering it on from a
            zero execution state.""",
        ),
        (
            "Stopping",
            11,
            """Machine is being normally stopped powering it off, or after the guest OS
            has initiated a shutdown sequence.""",
        ),
        ("Saving", 12, """Machine is saving its execution state to a file."""),
        (
            "Restoring",
            13,
            """Execution state of the machine is being restored from a file
            after powering it on from the saved execution state.""",
        ),
        (
            "TeleportingPausedVM",
            14,
            """The machine is being teleported to another host or process, but it is
            not running. This is the paused variant of the
            @c Teleporting state.""",
        ),
        (
            "TeleportingIn",
            15,
            """Teleporting the machine state in from another host or process.""",
        ),
        (
            "DeletingSnapshotOnline",
            16,
            """Like @c DeletingSnapshot, but the merging of media is ongoing in
            the background while the machine is running.""",
        ),
        (
            "DeletingSnapshotPaused",
            17,
            """Like @c DeletingSnapshotOnline, but the machine was paused when the
            merging of differencing media was started.""",
        ),
        (
            "OnlineSnapshotting",
            18,
            """Like @c LiveSnapshotting, but the machine was paused when the
            merging of differencing media was started.""",
        ),
        (
            "RestoringSnapshot",
            19,
            """A machine snapshot is being restored; this typically does not take long.""",
        ),
        (
            "DeletingSnapshot",
            20,
            """A machine snapshot is being deleted; this can take a long time since this
            may require merging differencing media. This value indicates that the
            machine is not running while the snapshot is being deleted.""",
        ),
        ("SettingUp", 21, """Lengthy setup operation is in progress."""),
        ("Snapshotting", 22, """Taking an (offline) snapshot."""),
        (
            "FirstOnline",
            5,
            """Pseudo-state: first online state (for use in relational expressions).""",
        ),
        (
            "LastOnline",
            18,
            """Pseudo-state: last online state (for use in relational expressions).""",
        ),
        (
            "FirstTransient",
            8,
            """Pseudo-state: first transient state (for use in relational expressions).""",
        ),
        (
            "LastTransient",
            22,
            """Pseudo-state: last transient state (for use in relational expressions).""",
        ),
    ]


class SessionState(Enum):
    """Session state. This enumeration represents possible values of
    :py:func:`IMachine.session_state`  and :py:func:`ISession.state`
    attributes.


    .. describe:: null(0)

            Null value (never used by the API).

    .. describe:: unlocked(1)

            In :py:func:`IMachine.session_state` , this means that the machine
            is not locked for any sessions.

            In :py:func:`ISession.state` , this means that no machine is
            currently locked for this session.

    .. describe:: locked(2)

            In :py:func:`IMachine.session_state` , this means that the machine
            is currently locked for a session, whose process identifier can
            then be found in the :py:func:`IMachine.session_pid`  attribute.

            In :py:func:`ISession.state` , this means that a machine is
            currently locked for this session, and the mutable machine object
            can be found in the :py:func:`ISession.machine`  attribute
            (see :py:func:`IMachine.lock_machine`  for details).

    .. describe:: spawning(3)

            A new process is being spawned for the machine as a result of
            :py:func:`IMachine.launch_vm_process`  call. This state also occurs
            as a short transient state during an :py:func:`IMachine.lock_machine`
            call.

    .. describe:: unlocking(4)

            The session is being unlocked.

    """

    __uuid__ = "cf2700c0-ea4b-47ae-9725-7810114b94d8"
    _enums = [
        ("Null", 0, """Null value (never used by the API)."""),
        (
            "Unlocked",
            1,
            """In :py:func:`IMachine.session_state` , this means that the machine
            is not locked for any sessions.
            
            In :py:func:`ISession.state` , this means that no machine is
            currently locked for this session.""",
        ),
        (
            "Locked",
            2,
            """In :py:func:`IMachine.session_state` , this means that the machine
            is currently locked for a session, whose process identifier can
            then be found in the :py:func:`IMachine.session_pid`  attribute.
            
            In :py:func:`ISession.state` , this means that a machine is
            currently locked for this session, and the mutable machine object
            can be found in the :py:func:`ISession.machine`  attribute
            (see :py:func:`IMachine.lock_machine`  for details).""",
        ),
        (
            "Spawning",
            3,
            """A new process is being spawned for the machine as a result of
            :py:func:`IMachine.launch_vm_process`  call. This state also occurs
            as a short transient state during an :py:func:`IMachine.lock_machine` 
            call.""",
        ),
        ("Unlocking", 4, """The session is being unlocked."""),
    ]


class CPUArchitecture(Enum):
    """Basic CPU architecture types.


    .. describe:: any_p(0)

            Matches any CPU architecture.

    .. describe:: x86(1)

            32-bit (and 16-bit) x86.

    .. describe:: amd64(2)

            64-bit x86.  (Also known as x86-64 or x64.)

    """

    __uuid__ = "4a2c9915-12f1-43b2-bb84-e4bd4d5ca227"
    _enums = [
        ("Any", 0, """Matches any CPU architecture."""),
        ("x86", 1, """32-bit (and 16-bit) x86."""),
        ("AMD64", 2, """64-bit x86.  (Also known as x86-64 or x64.)"""),
    ]


class CPUPropertyType(Enum):
    """Virtual CPU property type. This enumeration represents possible values of the
    IMachine get- and setCPUProperty methods.


    .. describe:: null(0)

            Null value (never used by the API).

    .. describe:: pae(1)

            This setting determines whether VirtualBox will expose the Physical Address
            Extension (PAE) feature of the host CPU to the guest. Note that in case PAE
            is not available, it will not be reported.

    .. describe:: long_mode(2)

            This setting determines whether VirtualBox will advertise long mode
            (i.e. 64-bit guest support) and let the guest enter it.

    .. describe:: triple_fault_reset(3)

            This setting determines whether a triple fault within a guest will
            trigger an internal error condition and stop the VM (default) or reset
            the virtual CPU/VM and continue execution.

    .. describe:: apic(4)

            This setting determines whether an APIC is part of the virtual CPU.
            This feature can only be turned off when the X2APIC feature is off.

    .. describe:: x2_apic(5)

            This setting determines whether an x2APIC is part of the virtual CPU.
            Since this feature implies that the APIC feature is present, it
            automatically enables the APIC feature when set.

    .. describe:: ibpb_on_vm_exit(6)

            If set, force an indirect branch prediction barrier on VM exits if the
            host CPU supports it.  This setting will significantly slow down workloads
            causing many VM exits, so it is only recommended for situation where there
            is a real need to be paranoid.

    .. describe:: ibpb_on_vm_entry(7)

            If set, force an indirect branch prediction barrier on VM entry if the
            host CPU supports it.  This setting will significantly slow down workloads
            causing many VM exits, so it is only recommended for situation where there
            is a real need to be paranoid.

    .. describe:: hw_virt(8)

            Enabled the hardware virtualization (AMD-V/VT-x) feature on the guest CPU.
            This requires hardware virtualization on the host CPU.

    .. describe:: spec_ctrl(9)

            If set, the speculation control CPUID bits and MSRs, when available on the
            host, are exposed to the guest. Depending on the host CPU and operating
            system, this may significantly slow down workloads causing many VM exits.

    .. describe:: spec_ctrl_by_host(10)

            If set, the speculation controls are managed by the host. This is intended
            for guests which do not set the speculation controls themselves.
            Note! This has not yet been implemented beyond leaving everything to the host OS.

    .. describe:: l1_d_flush_on_emt_scheduling(11)

            If set and the host is affected by CVE-2018-3646, flushes the level 1 data
            cache when the EMT is scheduled to do ring-0 guest execution.  There could
            be a small performance penalty for certain typs of workloads.
            For security reasons this setting will be enabled by default.

    .. describe:: l1_d_flush_on_vm_entry(12)

            If set and the host is affected by CVE-2018-3646, flushes the level 1 data
            on every VM entry.  This setting may significantly slow down workloads
            causing many VM exits, so it is only recommended for situation where there
            is a real need to be paranoid.

    .. describe:: mds_clear_on_emt_scheduling(13)

            If set and the host is affected by CVE-2018-12126, CVE-2018-12127, or
            CVE-2018-12130, clears the relevant MDS buffers when the EMT is scheduled
            to do ring-0 guest execution.  There could be a small performance penalty
            for certain typs of workloads. For security reasons this setting will be
            enabled by default.

    .. describe:: mds_clear_on_vm_entry(14)

            If set and the host is affected by CVE-2018-12126, CVE-2018-12127, or
            CVE-2018-12130, clears the relevant MDS buffers on every VM entry.  This
            setting may slow down workloads causing many VM exits, so it is only
            recommended for situation where there is a real need to be paranoid.

    """

    __uuid__ = "3fcfe589-ca66-468f-e313-656f9d0b2eb6"
    _enums = [
        ("Null", 0, """Null value (never used by the API)."""),
        (
            "PAE",
            1,
            """This setting determines whether VirtualBox will expose the Physical Address
            Extension (PAE) feature of the host CPU to the guest. Note that in case PAE
            is not available, it will not be reported.""",
        ),
        (
            "LongMode",
            2,
            """This setting determines whether VirtualBox will advertise long mode
            (i.e. 64-bit guest support) and let the guest enter it.""",
        ),
        (
            "TripleFaultReset",
            3,
            """This setting determines whether a triple fault within a guest will
            trigger an internal error condition and stop the VM (default) or reset
            the virtual CPU/VM and continue execution.""",
        ),
        (
            "APIC",
            4,
            """This setting determines whether an APIC is part of the virtual CPU.
            This feature can only be turned off when the X2APIC feature is off.""",
        ),
        (
            "X2APIC",
            5,
            """This setting determines whether an x2APIC is part of the virtual CPU.
            Since this feature implies that the APIC feature is present, it
            automatically enables the APIC feature when set.""",
        ),
        (
            "IBPBOnVMExit",
            6,
            """If set, force an indirect branch prediction barrier on VM exits if the
            host CPU supports it.  This setting will significantly slow down workloads
            causing many VM exits, so it is only recommended for situation where there
            is a real need to be paranoid.""",
        ),
        (
            "IBPBOnVMEntry",
            7,
            """If set, force an indirect branch prediction barrier on VM entry if the
            host CPU supports it.  This setting will significantly slow down workloads
            causing many VM exits, so it is only recommended for situation where there
            is a real need to be paranoid.""",
        ),
        (
            "HWVirt",
            8,
            """Enabled the hardware virtualization (AMD-V/VT-x) feature on the guest CPU.
            This requires hardware virtualization on the host CPU.""",
        ),
        (
            "SpecCtrl",
            9,
            """If set, the speculation control CPUID bits and MSRs, when available on the
            host, are exposed to the guest. Depending on the host CPU and operating
            system, this may significantly slow down workloads causing many VM exits.""",
        ),
        (
            "SpecCtrlByHost",
            10,
            """If set, the speculation controls are managed by the host. This is intended
            for guests which do not set the speculation controls themselves.
            Note! This has not yet been implemented beyond leaving everything to the host OS.""",
        ),
        (
            "L1DFlushOnEMTScheduling",
            11,
            """If set and the host is affected by CVE-2018-3646, flushes the level 1 data
            cache when the EMT is scheduled to do ring-0 guest execution.  There could
            be a small performance penalty for certain typs of workloads.
            For security reasons this setting will be enabled by default.""",
        ),
        (
            "L1DFlushOnVMEntry",
            12,
            """If set and the host is affected by CVE-2018-3646, flushes the level 1 data
            on every VM entry.  This setting may significantly slow down workloads
            causing many VM exits, so it is only recommended for situation where there
            is a real need to be paranoid.""",
        ),
        (
            "MDSClearOnEMTScheduling",
            13,
            """If set and the host is affected by CVE-2018-12126, CVE-2018-12127, or
            CVE-2018-12130, clears the relevant MDS buffers when the EMT is scheduled
            to do ring-0 guest execution.  There could be a small performance penalty
            for certain typs of workloads. For security reasons this setting will be
            enabled by default.""",
        ),
        (
            "MDSClearOnVMEntry",
            14,
            """If set and the host is affected by CVE-2018-12126, CVE-2018-12127, or
            CVE-2018-12130, clears the relevant MDS buffers on every VM entry.  This
            setting may slow down workloads causing many VM exits, so it is only
            recommended for situation where there is a real need to be paranoid.""",
        ),
    ]


class HWVirtExPropertyType(Enum):
    """Hardware virtualization property type. This enumeration represents possible values
    for the :py:func:`IMachine.get_hw_virt_ex_property`  and
    :py:func:`IMachine.set_hw_virt_ex_property`  methods.


    .. describe:: null(0)

            Null value (never used by the API).

    .. describe:: enabled(1)

            Whether hardware virtualization (VT-x/AMD-V) is enabled at all. If
            such extensions are not available, they will not be used.

    .. describe:: vpid(2)

            Whether VT-x VPID is enabled. If this extension is not available, it will not be used.

    .. describe:: nested_paging(3)

            Whether Nested Paging is enabled. If this extension is not available, it will not be used.

    .. describe:: unrestricted_execution(4)

            Whether VT-x unrestricted execution is enabled. If this feature is not available, it will not be used.

    .. describe:: large_pages(5)

            Whether large page allocation is enabled; requires nested paging and a 64-bit host.

    .. describe:: force(6)

            Whether the VM should fail to start if hardware virtualization (VT-x/AMD-V) cannot be used. If
            not set, there will be an automatic fallback to software virtualization.

    .. describe:: use_native_api(7)

            Use the native hypervisor API instead of the VirtualBox one (HM) for VT-X/AMD-V.  This is
            ignored if :py:attr:`HWVirtExPropertyType.enabled`  isn't set.

    .. describe:: virt_vmsave_vmload(8)

            Whether AMD-V Virtualized VMSAVE/VMLOAD is enabled. If this feature is not available, it will not
            be used.

    """

    __uuid__ = "00069d9c-00b5-460c-00dd-64250024f7aa"
    _enums = [
        ("Null", 0, """Null value (never used by the API)."""),
        (
            "Enabled",
            1,
            """Whether hardware virtualization (VT-x/AMD-V) is enabled at all. If
            such extensions are not available, they will not be used.""",
        ),
        (
            "VPID",
            2,
            """Whether VT-x VPID is enabled. If this extension is not available, it will not be used.""",
        ),
        (
            "NestedPaging",
            3,
            """Whether Nested Paging is enabled. If this extension is not available, it will not be used.""",
        ),
        (
            "UnrestrictedExecution",
            4,
            """Whether VT-x unrestricted execution is enabled. If this feature is not available, it will not be used.""",
        ),
        (
            "LargePages",
            5,
            """Whether large page allocation is enabled; requires nested paging and a 64-bit host.""",
        ),
        (
            "Force",
            6,
            """Whether the VM should fail to start if hardware virtualization (VT-x/AMD-V) cannot be used. If
            not set, there will be an automatic fallback to software virtualization.""",
        ),
        (
            "UseNativeApi",
            7,
            """Use the native hypervisor API instead of the VirtualBox one (HM) for VT-X/AMD-V.  This is
            ignored if :py:attr:`HWVirtExPropertyType.enabled`  isn't set.""",
        ),
        (
            "VirtVmsaveVmload",
            8,
            """Whether AMD-V Virtualized VMSAVE/VMLOAD is enabled. If this feature is not available, it will not
            be used.""",
        ),
    ]


class ParavirtProvider(Enum):
    """The paravirtualized guest interface provider. This enumeration represents possible
    values for the :py:func:`IMachine.paravirt_provider`  attribute.


    .. describe:: none(0)

            No provider is used.

    .. describe:: default(1)

            A default provider is automatically chosen according to the guest OS type.

    .. describe:: legacy(2)

            Used for VMs which didn't used to have any provider settings. Usually
            interpreted as @c None for most VMs.

    .. describe:: minimal(3)

            A minimal set of features to expose to the paravirtualized guest.

    .. describe:: hyper_v(4)

            Microsoft Hyper-V.

    .. describe:: kvm(5)

            Linux KVM.

    """

    __uuid__ = "696453ec-3742-4a05-bead-658ccbf2c944"
    _enums = [
        ("None", 0, """No provider is used."""),
        (
            "Default",
            1,
            """A default provider is automatically chosen according to the guest OS type.""",
        ),
        (
            "Legacy",
            2,
            """Used for VMs which didn't used to have any provider settings. Usually
            interpreted as @c None for most VMs.""",
        ),
        (
            "Minimal",
            3,
            """A minimal set of features to expose to the paravirtualized guest.""",
        ),
        ("HyperV", 4, """Microsoft Hyper-V."""),
        ("KVM", 5, """Linux KVM."""),
    ]


class LockType(Enum):
    """Used with :py:func:`IMachine.lock_machine` .


    .. describe:: null(0)

            Placeholder value, do not use when obtaining a lock.

    .. describe:: shared(1)

            Request only a shared lock for remote-controlling the machine.
            Such a lock allows changing certain VM settings which can be safely
            modified for a running VM.

    .. describe:: write(2)

            Lock the machine for writing. This requests an exclusive lock, i.e.
            there cannot be any other API client holding any type of lock for this
            VM concurrently. Remember that a VM process counts as an API client
            which implicitly holds the equivalent of a shared lock during the
            entire VM runtime.

    .. describe:: vm(3)

            Lock the machine for writing, and create objects necessary for
            running a VM in this process.

    """

    __uuid__ = "678aaf14-2815-4c3e-b20a-e86ed0216498"
    _enums = [
        ("Null", 0, """Placeholder value, do not use when obtaining a lock."""),
        (
            "Shared",
            1,
            """Request only a shared lock for remote-controlling the machine.
            Such a lock allows changing certain VM settings which can be safely
            modified for a running VM.""",
        ),
        (
            "Write",
            2,
            """Lock the machine for writing. This requests an exclusive lock, i.e.
            there cannot be any other API client holding any type of lock for this
            VM concurrently. Remember that a VM process counts as an API client
            which implicitly holds the equivalent of a shared lock during the
            entire VM runtime.""",
        ),
        (
            "VM",
            3,
            """Lock the machine for writing, and create objects necessary for
            running a VM in this process.""",
        ),
    ]


class SessionType(Enum):
    """Session type. This enumeration represents possible values of the
    :py:func:`ISession.type_p`  attribute.


    .. describe:: null(0)

            Null value (never used by the API).

    .. describe:: write_lock(1)

            Session has acquired an exclusive write lock on a machine
            using :py:func:`IMachine.lock_machine` .

    .. describe:: remote(2)

            Session has launched a VM process using
            :py:func:`IMachine.launch_vm_process`

    .. describe:: shared(3)

            Session has obtained a link to another session using
            :py:func:`IMachine.lock_machine`

    """

    __uuid__ = "A13C02CB-0C2C-421E-8317-AC0E8AAA153A"
    _enums = [
        ("Null", 0, """Null value (never used by the API)."""),
        (
            "WriteLock",
            1,
            """Session has acquired an exclusive write lock on a machine
            using :py:func:`IMachine.lock_machine` .""",
        ),
        (
            "Remote",
            2,
            """Session has launched a VM process using
            :py:func:`IMachine.launch_vm_process` """,
        ),
        (
            "Shared",
            3,
            """Session has obtained a link to another session using
            :py:func:`IMachine.lock_machine` """,
        ),
    ]


class DeviceType(Enum):
    """Device type.


    .. describe:: null(0)

            Null value, may also mean "no device" (not allowed for
            :py:func:`IConsole.get_device_activity` ).

    .. describe:: floppy(1)

            Floppy device.

    .. describe:: dvd(2)

            CD/DVD-ROM device.

    .. describe:: hard_disk(3)

            Hard disk device.

    .. describe:: network(4)

            Network device.

    .. describe:: usb(5)

            USB device.

    .. describe:: shared_folder(6)

            Shared folder device.

    .. describe:: graphics3_d(7)

            Graphics device 3D activity.

    """

    __uuid__ = "cb977be1-d1fb-41f8-ad7e-951736c6cb3e"
    _enums = [
        (
            "Null",
            0,
            """Null value, may also mean "no device" (not allowed for
            :py:func:`IConsole.get_device_activity` ).""",
        ),
        ("Floppy", 1, """Floppy device."""),
        ("DVD", 2, """CD/DVD-ROM device."""),
        ("HardDisk", 3, """Hard disk device."""),
        ("Network", 4, """Network device."""),
        ("USB", 5, """USB device."""),
        ("SharedFolder", 6, """Shared folder device."""),
        ("Graphics3D", 7, """Graphics device 3D activity."""),
    ]


class DeviceActivity(Enum):
    """Device activity for :py:func:`IConsole.get_device_activity` .


    .. describe:: null(0)



    .. describe:: idle(1)



    .. describe:: reading(2)



    .. describe:: writing(3)



    """

    __uuid__ = "6FC8AEAA-130A-4eb5-8954-3F921422D707"
    _enums = [
        ("Null", 0, """"""),
        ("Idle", 1, """"""),
        ("Reading", 2, """"""),
        ("Writing", 3, """"""),
    ]


class ClipboardMode(Enum):
    """Host-Guest clipboard interchange mode.


    .. describe:: disabled(0)



    .. describe:: host_to_guest(1)



    .. describe:: guest_to_host(2)



    .. describe:: bidirectional(3)



    """

    __uuid__ = "33364716-4008-4701-8f14-be0fa3d62950"
    _enums = [
        ("Disabled", 0, """"""),
        ("HostToGuest", 1, """"""),
        ("GuestToHost", 2, """"""),
        ("Bidirectional", 3, """"""),
    ]


class DnDMode(Enum):
    """Drag and drop interchange mode.


    .. describe:: disabled(0)



    .. describe:: host_to_guest(1)



    .. describe:: guest_to_host(2)



    .. describe:: bidirectional(3)



    """

    __uuid__ = "07af8800-f936-4b33-9172-cd400e83c148"
    _enums = [
        ("Disabled", 0, """"""),
        ("HostToGuest", 1, """"""),
        ("GuestToHost", 2, """"""),
        ("Bidirectional", 3, """"""),
    ]


class Scope(Enum):
    """Scope of the operation.

    A generic enumeration used in various methods to define the action or
    argument scope.


    .. describe:: global_p(0)



    .. describe:: machine(1)



    .. describe:: session(2)



    """

    __uuid__ = "7c91096e-499e-4eca-9f9b-9001438d7855"
    _enums = [
        ("Global", 0, """"""),
        ("Machine", 1, """"""),
        ("Session", 2, """"""),
    ]


class BIOSBootMenuMode(Enum):
    """BIOS boot menu mode.


    .. describe:: disabled(0)



    .. describe:: menu_only(1)



    .. describe:: message_and_menu(2)



    """

    __uuid__ = "ae4fb9f7-29d2-45b4-b2c7-d579603135d5"
    _enums = [
        ("Disabled", 0, """"""),
        ("MenuOnly", 1, """"""),
        ("MessageAndMenu", 2, """"""),
    ]


class APICMode(Enum):
    """BIOS APIC initialization mode. If the hardware does not support the
    mode then the code falls back to a lower mode.


    .. describe:: disabled(0)



    .. describe:: apic(1)



    .. describe:: x2_apic(2)



    """

    __uuid__ = "c6884ba5-3cc4-4a92-a7f6-4410f9fd894e"
    _enums = [
        ("Disabled", 0, """"""),
        ("APIC", 1, """"""),
        ("X2APIC", 2, """"""),
    ]


class ProcessorFeature(Enum):
    """CPU features.


    .. describe:: hw_virt_ex(0)



    .. describe:: pae(1)



    .. describe:: long_mode(2)



    .. describe:: nested_paging(3)



    .. describe:: unrestricted_guest(4)



    .. describe:: nested_hw_virt(5)



    .. describe:: virt_vmsave_vmload(6)



    """

    __uuid__ = "0064dece-000e-4963-00f8-eb9b00674c8a"
    _enums = [
        ("HWVirtEx", 0, """"""),
        ("PAE", 1, """"""),
        ("LongMode", 2, """"""),
        ("NestedPaging", 3, """"""),
        ("UnrestrictedGuest", 4, """"""),
        ("NestedHWVirt", 5, """"""),
        ("VirtVmsaveVmload", 6, """"""),
    ]


class FirmwareType(Enum):
    """Firmware type.


    .. describe:: bios(1)

            BIOS Firmware.

    .. describe:: efi(2)

            EFI Firmware, bitness detected basing on OS type.

    .. describe:: efi32(3)

            EFI firmware, 32-bit.

    .. describe:: efi64(4)

            EFI firmware, 64-bit.

    .. describe:: efidual(5)

            EFI firmware, combined 32 and 64-bit.

    """

    __uuid__ = "b903f264-c230-483e-ac74-2b37ce60d371"
    _enums = [
        ("BIOS", 1, """BIOS Firmware."""),
        ("EFI", 2, """EFI Firmware, bitness detected basing on OS type."""),
        ("EFI32", 3, """EFI firmware, 32-bit."""),
        ("EFI64", 4, """EFI firmware, 64-bit."""),
        ("EFIDUAL", 5, """EFI firmware, combined 32 and 64-bit."""),
    ]


class PointingHIDType(Enum):
    """Type of pointing device used in a virtual machine.


    .. describe:: none(1)

            No mouse.

    .. describe:: ps2_mouse(2)

            PS/2 auxiliary device, a.k.a. mouse.

    .. describe:: usb_mouse(3)

            USB mouse (relative pointer).

    .. describe:: usb_tablet(4)

            USB tablet (absolute pointer).  Also enables a relative USB mouse in
            addition.

    .. describe:: combo_mouse(5)

            Combined device, working as PS/2 or USB mouse, depending on guest
            behavior.  Using this device can have negative performance implications.

    .. describe:: usb_multi_touch(6)

            USB multi-touch device.  Also enables the USB tablet and mouse devices.

    """

    __uuid__ = "19964e93-0050-45c4-9382-a7bccc53e666"
    _enums = [
        ("None", 1, """No mouse."""),
        ("PS2Mouse", 2, """PS/2 auxiliary device, a.k.a. mouse."""),
        ("USBMouse", 3, """USB mouse (relative pointer)."""),
        (
            "USBTablet",
            4,
            """USB tablet (absolute pointer).  Also enables a relative USB mouse in
            addition.""",
        ),
        (
            "ComboMouse",
            5,
            """Combined device, working as PS/2 or USB mouse, depending on guest
            behavior.  Using this device can have negative performance implications.""",
        ),
        (
            "USBMultiTouch",
            6,
            """USB multi-touch device.  Also enables the USB tablet and mouse devices.""",
        ),
    ]


class KeyboardHIDType(Enum):
    """Type of keyboard device used in a virtual machine.


    .. describe:: none(1)

            No keyboard.

    .. describe:: ps2_keyboard(2)

            PS/2 keyboard.

    .. describe:: usb_keyboard(3)

            USB keyboard.

    .. describe:: combo_keyboard(4)

            Combined device, working as PS/2 or USB keyboard, depending on guest behavior.
            Using of such device can have negative performance implications.

    """

    __uuid__ = "383e43d7-5c7c-4ec8-9cb8-eda1bccd6699"
    _enums = [
        ("None", 1, """No keyboard."""),
        ("PS2Keyboard", 2, """PS/2 keyboard."""),
        ("USBKeyboard", 3, """USB keyboard."""),
        (
            "ComboKeyboard",
            4,
            """Combined device, working as PS/2 or USB keyboard, depending on guest behavior.
            Using of such device can have negative performance implications.""",
        ),
    ]


class BitmapFormat(Enum):
    """Format of a bitmap. Generic values for formats used by
    the source bitmap, the screen shot or image update APIs.


    .. describe:: opaque(0)

            Unknown buffer format (the user may not assume any particular format of
            the buffer).

    .. describe:: bgr(542263106)

            Generic BGR format without alpha channel.
            Pixel layout depends on the number of bits per pixel:


            **32** - bits 31:24 undefined, bits 23:16 R, bits 15:8 G, bits 7:0 B.



            **16** - bits 15:11 R, bits 10:5 G, bits 4:0 B.

    .. describe:: bgr0(810698562)

            4 bytes per pixel: B, G, R, 0.

    .. describe:: bgra(1095911234)

            4 bytes per pixel: B, G, R, A.

    .. describe:: rgba(1094862674)

            4 bytes per pixel: R, G, B, A.

    .. describe:: png(541544016)

            PNG image.

    .. describe:: jpeg(1195724874)

            JPEG image.

    """

    __uuid__ = "afb2bf39-8b1e-4f9f-8948-d1b887f83eb0"
    _enums = [
        (
            "Opaque",
            0,
            """Unknown buffer format (the user may not assume any particular format of
            the buffer).""",
        ),
        (
            "BGR",
            542263106,
            """Generic BGR format without alpha channel.
            Pixel layout depends on the number of bits per pixel:
            
            
            **32** - bits 31:24 undefined, bits 23:16 R, bits 15:8 G, bits 7:0 B.
            
            
            
            **16** - bits 15:11 R, bits 10:5 G, bits 4:0 B.""",
        ),
        ("BGR0", 810698562, """4 bytes per pixel: B, G, R, 0."""),
        ("BGRA", 1095911234, """4 bytes per pixel: B, G, R, A."""),
        ("RGBA", 1094862674, """4 bytes per pixel: R, G, B, A."""),
        ("PNG", 541544016, """PNG image."""),
        ("JPEG", 1195724874, """JPEG image."""),
    ]


class PartitioningType(Enum):
    """The type of the disk partition.


    .. describe:: mbr(0)



    .. describe:: gpt(1)



    """

    __uuid__ = "64c4c806-8908-4c0b-9a51-2d7a0151321f"
    _enums = [
        ("MBR", 0, """"""),
        ("GPT", 1, """"""),
    ]


class PartitionType(Enum):
    """


    .. describe:: empty(0)

            Empty partition entry

    .. describe:: fat12(1)

            FAT12 if partition size less than 65536 sectors

    .. describe:: fat16(4)

            FAT16 if partition size less than 65536 sectors

    .. describe:: fat(6)

            FAT12 or FAT16 if partition size greater or equal than 65536 sectors

    .. describe:: ifs(7)

            NT and OS/2 installable file system, e.g. NTFS, exFAT, HPFS.

    .. describe:: fat32_chs(11)

            The FAT32 with CHS addressing.

    .. describe:: fat32_lba(12)

            The FAT32 with LBA addressing.

    .. describe:: fat16_b(14)

            The FAT16 with LBA addressing.

    .. describe:: extended(15)

            The extended partition with LBA addressing.

    .. describe:: windows_re(39)

            Windows Recovery Environment (RE) partition (hidden NTFS partition).

    .. describe:: linux_swap_old(66)

            The linux swap partition (old versions).

    .. describe:: linux_old(67)

            The linux native partition (old versions).

    .. describe:: dragon_fly_bsd_slice(108)

            The BSD slice.

    .. describe:: linux_swap(130)

            The linux swap partition.

    .. describe:: linux(131)

            The linux native partition.

    .. describe:: linux_extended(133)

            The linux extended partition.

    .. describe:: linux_lvm(142)

            The linux LVM partition.

    .. describe:: bsd_slice(165)

            The BSD slice.

    .. describe:: apple_ufs(168)

            The Apple UFS partition.

    .. describe:: apple_hfs(175)

            The Apple HFS partition.

    .. describe:: solaris(191)

            The Apple HFS partition.

    .. describe:: gpt(238)

            The GPT protective MBR partition.

    .. describe:: efi(239)

            The EFI system partition.

    .. describe:: unknown(256)

            Unknown partition type.

    .. describe:: mbr(257)

            MBR partition scheme.

    .. describe:: i_ffs(258)

            Intel Fast Flash (iFFS) partition.

    .. describe:: sony_boot(259)

            Sony boot partition.

    .. describe:: lenovo_boot(260)

            Lenovo boot partition.

    .. describe:: windows_msr(270)

            Microsoft Reserved Partition (MSR).

    .. describe:: windows_basic_data(271)

            Windows Basic data partition

    .. describe:: windows_ldm_meta(272)

            Windows Logical Disk Manager (LDM) metadata partition.

    .. describe:: windows_ldm_data(273)

            Windows Logical Disk Manager data partition.

    .. describe:: windows_recovery(274)

            Windows Recovery Environment.

    .. describe:: windows_storage_spaces(276)

            Windows Storage Spaces partition.

    .. describe:: windows_storage_replica(277)

            Windows Storage Replica partition.

    .. describe:: ibmgpfs(275)

            IBM General Parallel File System (GPFS) partition.

    .. describe:: linux_data(300)

            Linux filesystem data.

    .. describe:: linux_raid(301)

            Linux RAID partition.

    .. describe:: linux_root_x86(302)

            Linux root partition for x86.

    .. describe:: linux_root_amd64(303)

            Linux root partition for AMD64.

    .. describe:: linux_root_arm32(304)

            Linux root partition for ARM32.

    .. describe:: linux_root_arm64(305)

            Linux root partition for ARM64 / AArch64.

    .. describe:: linux_home(306)

            Linux /home partition.

    .. describe:: linux_srv(307)

            Linux /srv partition.

    .. describe:: linux_plain_dm_crypt(308)

            Linux plain dm-crypt partition.

    .. describe:: linux_luks(309)

            Linux unitified key setup (LUKS) partition.

    .. describe:: linux_reserved(310)

            Linux reserved partition.

    .. describe:: free_bsd_boot(330)

            FreeBSD boot partition.

    .. describe:: free_bsd_data(331)

            FreeBSD data partition.

    .. describe:: free_bsd_swap(332)

            FreeBSD swap partition.

    .. describe:: free_bsdufs(333)

            FreeBSD unix file system (UFS) partition.

    .. describe:: free_bsd_vinum(334)

            FreeBSD Vinum volume manager partition.

    .. describe:: free_bsdzfs(335)

            FreeBSD ZFS partition.

    .. describe:: free_bsd_unknown(359)

            Unknown FreeBSD partition.

    .. describe:: apple_hfs_plus(360)

            Apple hierarchical file system plus (HFS+) partition.

    .. describe:: apple_apfs(361)

            Apple APFS/FileFault container partition.

    .. describe:: apple_raid(362)

            Apple RAID partition.

    .. describe:: apple_raid_offline(363)

            Apple RAID partition, offline.

    .. describe:: apple_boot(364)

            Apple boot partition.

    .. describe:: apple_label(365)

            Apple label.

    .. describe:: apple_tv_recovery(366)

            Apple TV recovery partition.

    .. describe:: apple_core_storage(367)

            Apple Core Storage Containe.

    .. describe:: soft_raid_status(370)

            SoftRAID status.

    .. describe:: soft_raid_scratch(371)

            SoftRAID scratch.

    .. describe:: soft_raid_volume(372)

            SoftRAID volume.

    .. describe:: soft_raid_cache(373)

            SoftRAID cache.

    .. describe:: apple_unknown(389)

            Unknown Apple partition.

    .. describe:: solaris_boot(390)

            Solaris boot partition.

    .. describe:: solaris_root(391)

            Solaris root partition.

    .. describe:: solaris_swap(392)

            Solaris swap partition.

    .. describe:: solaris_backup(393)

            Solaris backup partition.

    .. describe:: solaris_usr(394)

            Solaris /usr partition.

    .. describe:: solaris_var(395)

            Solaris /var partition.

    .. describe:: solaris_home(396)

            Solaris /home partition.

    .. describe:: solaris_alt_sector(397)

            Solaris alternate sector.

    .. describe:: solaris_reserved(398)

            Solaris reserved partition.

    .. describe:: solaris_unknown(419)

            Unknown Solaris partition.

    .. describe:: net_bsd_swap(420)

            NetBSD swap partition.

    .. describe:: net_bsdffs(421)

            NetBSD fast file system (FFS) partition.

    .. describe:: net_bsdlfs(422)

            NetBSD log structured file system (LFS) partition.

    .. describe:: net_bsdraid(423)

            NetBSD RAID partition.

    .. describe:: net_bsd_concatenated(424)

            NetBSD concatenated partition.

    .. describe:: net_bsd_encrypted(425)

            NetBSD encrypted partition.

    .. describe:: net_bsd_unknown(449)

            Unknown NetBSD partition.

    .. describe:: chrome_os_kernel(450)

            Chrome OS kernel partition.

    .. describe:: chrome_os_root_fs(451)

            Chrome OS root file system partition.

    .. describe:: chrome_os_future(452)

            Chrome OS partition reserved for future.

    .. describe:: cont_lnx_usr(480)

            Container Linux /usr partition.

    .. describe:: cont_lnx_root(481)

            Container Linux resizable root filesystem partition.

    .. describe:: cont_lnx_reserved(482)

            Container Linux OEM customization partition.

    .. describe:: cont_lnx_root_raid(483)

            Container Linux root filesystem on RAID partition.

    .. describe:: haiku_bfs(510)

            Haiku BFS

    .. describe:: midnt_bsd_boot(540)

            MidnightBSD boot partition.

    .. describe:: midnt_bsd_data(541)

            MidnightBSD data partition.

    .. describe:: midnt_bsd_swap(542)

            MidnightBSD swap partition.

    .. describe:: midnt_bsdufs(543)

            MidnightBSD unix file system (UFS) partition.

    .. describe:: midnt_bsd_vium(544)

            MidnightBSD Vinum volume manager partition.

    .. describe:: midnt_bsdzfs(545)

            MidnightBSD ZFS partition.

    .. describe:: midnt_bsd_unknown(569)

            Unknown MidnightBSD partition.

    .. describe:: open_bsd_data(570)

            OpenBSD data partition.

    .. describe:: qnx_power_safe_fs(600)

            QNX power-safe file system partition.

    .. describe:: plan9(630)

            Plan 9 partition.

    .. describe:: vm_ware_vmk_core(660)

            VMWare ESX coredump partition.

    .. describe:: vm_ware_vmfs(661)

            VMWare ESX virtual machine file system (VMFS) partition.

    .. describe:: vm_ware_reserved(662)

            VMWare ESX reserved partition.

    .. describe:: vm_ware_unknown(689)

            Unknown VMWare partition.

    .. describe:: android_x86_bootloader(690)

            Android x86 bootloader partition.

    .. describe:: android_x86_bootloader2(691)

            Android x86 bootloader2 partition.

    .. describe:: android_x86_boot(692)

            Android x86 boot partition.

    .. describe:: android_x86_recovery(693)

            Android x86 recovery partition.

    .. describe:: android_x86_misc(694)

            Android x86 misc partition.

    .. describe:: android_x86_metadata(695)

            Android x86 metadata partition.

    .. describe:: android_x86_system(696)

            Android x86 system partition.

    .. describe:: android_x86_cache(697)

            Android x86 cache partition.

    .. describe:: android_x86_data(698)

            Android x86 data partition.

    .. describe:: android_x86_persistent(699)

            Android x86 persistent data partition.

    .. describe:: android_x86_vendor(700)

            Android x86 vendor partition.

    .. describe:: android_x86_config(701)

            Android x86 config partition.

    .. describe:: android_x86_factory(702)

            Android x86 factory partition.

    .. describe:: android_x86_factory_alt(703)

            Android x86 alternative factory partition.

    .. describe:: android_x86_fastboot(704)

            Android x86 fastboot partition.

    .. describe:: android_x86_oem(705)

            Android x86 OEM partition.

    .. describe:: android_arm_meta(720)

            Android ARM meta partition.

    .. describe:: android_arm_ext(721)

            Android ARM EXT partition.

    .. describe:: onie_boot(750)

            Open Network Install Environment (ONIE) boot partition.

    .. describe:: onie_config(751)

            Open Network Install Environment (ONIE) config partition.

    .. describe:: power_pc_prep(780)

            PowerPC PReP boot partition.

    .. describe:: xdg_shr_boot_config(810)

            freedesktop.org shared boot loader configuration partition.

    .. describe:: ceph_block(830)

            Ceph block partition.

    .. describe:: ceph_block_db(831)

            Ceph block DB partition.

    .. describe:: ceph_block_db_dmc(832)

            Ceph dm-crypt block DB partition.

    .. describe:: ceph_block_db_dmc_luks(833)

            Ceph dm-crypt Linux unitified key setup (LUKS) block DB partition.

    .. describe:: ceph_block_dmc(834)

            Ceph dm-crypt block partition.

    .. describe:: ceph_block_dmc_luks(835)

            Ceph dm-crypt Linux unitified key setup (LUKS) block partition.

    .. describe:: ceph_block_wa_log(836)

            Ceph block write-ahead log partition.

    .. describe:: ceph_block_wa_log_dmc(837)

            Ceph dm-crypt block write-ahead log partition.

    .. describe:: ceph_block_wa_log_dmc_luks(838)

            Ceph dm-crypt Linux unitified key setup (LUKS) block write-ahead log partition.

    .. describe:: ceph_disk(839)

            Ceph disk in creation partition.

    .. describe:: ceph_disk_dmc(840)

            Ceph dm-crypt disk in creation partition.

    .. describe:: ceph_journal(841)

            Ceph Journal partition.

    .. describe:: ceph_journal_dmc(842)

            Ceph dm-crypt journal partition.

    .. describe:: ceph_journal_dmc_luks(843)

            Ceph dm-crypt Linux unitified key setup (LUKS) journal partition.

    .. describe:: ceph_lockbox(844)

            Ceph Lockbox for dm-crypt keys partition.

    .. describe:: ceph_multipath_block1(845)

            Ceph multipath block 1 partition.

    .. describe:: ceph_multipath_block2(846)

            Ceph multipath block 2 partition.

    .. describe:: ceph_multipath_block_db(847)

            Ceph multipath block DB partition.

    .. describe:: ceph_multipath_b_lock_wa_log(848)

            Ceph multipath block write-ahead log partition.

    .. describe:: ceph_multipath_journal(849)

            Ceph multipath journal partition.

    .. describe:: ceph_multipath_osd(851)

            Ceph multipath object storage deamon (OSD) partition.

    .. describe:: ceph_osd(852)

            Ceph object storage deamon (OSD) partition.

    .. describe:: ceph_osd_dmc(853)

            Ceph dm-crypt object storage deamon (OSD) partition.

    .. describe:: ceph_osd_dmc_luks(854)

            Ceph dm-crypt Linux unitified key setup (LUKS) object storage deamon (OSD) partition.

    """

    __uuid__ = "84a6629c-8e9c-474c-adbb-21995671597f"
    _enums = [
        ("Empty", 0, """Empty partition entry"""),
        ("FAT12", 1, """FAT12 if partition size less than 65536 sectors"""),
        ("FAT16", 4, """FAT16 if partition size less than 65536 sectors"""),
        (
            "FAT",
            6,
            """FAT12 or FAT16 if partition size greater or equal than 65536 sectors""",
        ),
        ("IFS", 7, """NT and OS/2 installable file system, e.g. NTFS, exFAT, HPFS."""),
        ("FAT32CHS", 11, """The FAT32 with CHS addressing."""),
        ("FAT32LBA", 12, """The FAT32 with LBA addressing."""),
        ("FAT16B", 14, """The FAT16 with LBA addressing."""),
        ("Extended", 15, """The extended partition with LBA addressing."""),
        (
            "WindowsRE",
            39,
            """Windows Recovery Environment (RE) partition (hidden NTFS partition).""",
        ),
        ("LinuxSwapOld", 66, """The linux swap partition (old versions)."""),
        ("LinuxOld", 67, """The linux native partition (old versions)."""),
        ("DragonFlyBSDSlice", 108, """The BSD slice."""),
        ("LinuxSwap", 130, """The linux swap partition."""),
        ("Linux", 131, """The linux native partition."""),
        ("LinuxExtended", 133, """The linux extended partition."""),
        ("LinuxLVM", 142, """The linux LVM partition."""),
        ("BSDSlice", 165, """The BSD slice."""),
        ("AppleUFS", 168, """The Apple UFS partition."""),
        ("AppleHFS", 175, """The Apple HFS partition."""),
        ("Solaris", 191, """The Apple HFS partition."""),
        ("GPT", 238, """The GPT protective MBR partition."""),
        ("EFI", 239, """The EFI system partition."""),
        ("Unknown", 256, """Unknown partition type."""),
        ("MBR", 257, """MBR partition scheme."""),
        ("iFFS", 258, """Intel Fast Flash (iFFS) partition."""),
        ("SonyBoot", 259, """Sony boot partition."""),
        ("LenovoBoot", 260, """Lenovo boot partition."""),
        ("WindowsMSR", 270, """Microsoft Reserved Partition (MSR)."""),
        ("WindowsBasicData", 271, """Windows Basic data partition"""),
        (
            "WindowsLDMMeta",
            272,
            """Windows Logical Disk Manager (LDM) metadata partition.""",
        ),
        ("WindowsLDMData", 273, """Windows Logical Disk Manager data partition."""),
        ("WindowsRecovery", 274, """Windows Recovery Environment."""),
        ("WindowsStorageSpaces", 276, """Windows Storage Spaces partition."""),
        ("WindowsStorageReplica", 277, """Windows Storage Replica partition."""),
        ("IBMGPFS", 275, """IBM General Parallel File System (GPFS) partition."""),
        ("LinuxData", 300, """Linux filesystem data."""),
        ("LinuxRAID", 301, """Linux RAID partition."""),
        ("LinuxRootX86", 302, """Linux root partition for x86."""),
        ("LinuxRootAMD64", 303, """Linux root partition for AMD64."""),
        ("LinuxRootARM32", 304, """Linux root partition for ARM32."""),
        ("LinuxRootARM64", 305, """Linux root partition for ARM64 / AArch64."""),
        ("LinuxHome", 306, """Linux /home partition."""),
        ("LinuxSrv", 307, """Linux /srv partition."""),
        ("LinuxPlainDmCrypt", 308, """Linux plain dm-crypt partition."""),
        ("LinuxLUKS", 309, """Linux unitified key setup (LUKS) partition."""),
        ("LinuxReserved", 310, """Linux reserved partition."""),
        ("FreeBSDBoot", 330, """FreeBSD boot partition."""),
        ("FreeBSDData", 331, """FreeBSD data partition."""),
        ("FreeBSDSwap", 332, """FreeBSD swap partition."""),
        ("FreeBSDUFS", 333, """FreeBSD unix file system (UFS) partition."""),
        ("FreeBSDVinum", 334, """FreeBSD Vinum volume manager partition."""),
        ("FreeBSDZFS", 335, """FreeBSD ZFS partition."""),
        ("FreeBSDUnknown", 359, """Unknown FreeBSD partition."""),
        (
            "AppleHFSPlus",
            360,
            """Apple hierarchical file system plus (HFS+) partition.""",
        ),
        ("AppleAPFS", 361, """Apple APFS/FileFault container partition."""),
        ("AppleRAID", 362, """Apple RAID partition."""),
        ("AppleRAIDOffline", 363, """Apple RAID partition, offline."""),
        ("AppleBoot", 364, """Apple boot partition."""),
        ("AppleLabel", 365, """Apple label."""),
        ("AppleTvRecovery", 366, """Apple TV recovery partition."""),
        ("AppleCoreStorage", 367, """Apple Core Storage Containe."""),
        ("SoftRAIDStatus", 370, """SoftRAID status."""),
        ("SoftRAIDScratch", 371, """SoftRAID scratch."""),
        ("SoftRAIDVolume", 372, """SoftRAID volume."""),
        ("SoftRAIDCache", 373, """SoftRAID cache."""),
        ("AppleUnknown", 389, """Unknown Apple partition."""),
        ("SolarisBoot", 390, """Solaris boot partition."""),
        ("SolarisRoot", 391, """Solaris root partition."""),
        ("SolarisSwap", 392, """Solaris swap partition."""),
        ("SolarisBackup", 393, """Solaris backup partition."""),
        ("SolarisUsr", 394, """Solaris /usr partition."""),
        ("SolarisVar", 395, """Solaris /var partition."""),
        ("SolarisHome", 396, """Solaris /home partition."""),
        ("SolarisAltSector", 397, """Solaris alternate sector."""),
        ("SolarisReserved", 398, """Solaris reserved partition."""),
        ("SolarisUnknown", 419, """Unknown Solaris partition."""),
        ("NetBSDSwap", 420, """NetBSD swap partition."""),
        ("NetBSDFFS", 421, """NetBSD fast file system (FFS) partition."""),
        ("NetBSDLFS", 422, """NetBSD log structured file system (LFS) partition."""),
        ("NetBSDRAID", 423, """NetBSD RAID partition."""),
        ("NetBSDConcatenated", 424, """NetBSD concatenated partition."""),
        ("NetBSDEncrypted", 425, """NetBSD encrypted partition."""),
        ("NetBSDUnknown", 449, """Unknown NetBSD partition."""),
        ("ChromeOSKernel", 450, """Chrome OS kernel partition."""),
        ("ChromeOSRootFS", 451, """Chrome OS root file system partition."""),
        ("ChromeOSFuture", 452, """Chrome OS partition reserved for future."""),
        ("ContLnxUsr", 480, """Container Linux /usr partition."""),
        (
            "ContLnxRoot",
            481,
            """Container Linux resizable root filesystem partition.""",
        ),
        ("ContLnxReserved", 482, """Container Linux OEM customization partition."""),
        (
            "ContLnxRootRAID",
            483,
            """Container Linux root filesystem on RAID partition.""",
        ),
        ("HaikuBFS", 510, """Haiku BFS"""),
        ("MidntBSDBoot", 540, """MidnightBSD boot partition."""),
        ("MidntBSDData", 541, """MidnightBSD data partition."""),
        ("MidntBSDSwap", 542, """MidnightBSD swap partition."""),
        ("MidntBSDUFS", 543, """MidnightBSD unix file system (UFS) partition."""),
        ("MidntBSDVium", 544, """MidnightBSD Vinum volume manager partition."""),
        ("MidntBSDZFS", 545, """MidnightBSD ZFS partition."""),
        ("MidntBSDUnknown", 569, """Unknown MidnightBSD partition."""),
        ("OpenBSDData", 570, """OpenBSD data partition."""),
        ("QNXPowerSafeFS", 600, """QNX power-safe file system partition."""),
        ("Plan9", 630, """Plan 9 partition."""),
        ("VMWareVMKCore", 660, """VMWare ESX coredump partition."""),
        (
            "VMWareVMFS",
            661,
            """VMWare ESX virtual machine file system (VMFS) partition.""",
        ),
        ("VMWareReserved", 662, """VMWare ESX reserved partition."""),
        ("VMWareUnknown", 689, """Unknown VMWare partition."""),
        ("AndroidX86Bootloader", 690, """Android x86 bootloader partition."""),
        ("AndroidX86Bootloader2", 691, """Android x86 bootloader2 partition."""),
        ("AndroidX86Boot", 692, """Android x86 boot partition."""),
        ("AndroidX86Recovery", 693, """Android x86 recovery partition."""),
        ("AndroidX86Misc", 694, """Android x86 misc partition."""),
        ("AndroidX86Metadata", 695, """Android x86 metadata partition."""),
        ("AndroidX86System", 696, """Android x86 system partition."""),
        ("AndroidX86Cache", 697, """Android x86 cache partition."""),
        ("AndroidX86Data", 698, """Android x86 data partition."""),
        ("AndroidX86Persistent", 699, """Android x86 persistent data partition."""),
        ("AndroidX86Vendor", 700, """Android x86 vendor partition."""),
        ("AndroidX86Config", 701, """Android x86 config partition."""),
        ("AndroidX86Factory", 702, """Android x86 factory partition."""),
        ("AndroidX86FactoryAlt", 703, """Android x86 alternative factory partition."""),
        ("AndroidX86Fastboot", 704, """Android x86 fastboot partition."""),
        ("AndroidX86OEM", 705, """Android x86 OEM partition."""),
        ("AndroidARMMeta", 720, """Android ARM meta partition."""),
        ("AndroidARMExt", 721, """Android ARM EXT partition."""),
        (
            "ONIEBoot",
            750,
            """Open Network Install Environment (ONIE) boot partition.""",
        ),
        (
            "ONIEConfig",
            751,
            """Open Network Install Environment (ONIE) config partition.""",
        ),
        ("PowerPCPrep", 780, """PowerPC PReP boot partition."""),
        (
            "XDGShrBootConfig",
            810,
            """freedesktop.org shared boot loader configuration partition.""",
        ),
        ("CephBlock", 830, """Ceph block partition."""),
        ("CephBlockDB", 831, """Ceph block DB partition."""),
        ("CephBlockDBDmc", 832, """Ceph dm-crypt block DB partition."""),
        (
            "CephBlockDBDmcLUKS",
            833,
            """Ceph dm-crypt Linux unitified key setup (LUKS) block DB partition.""",
        ),
        ("CephBlockDmc", 834, """Ceph dm-crypt block partition."""),
        (
            "CephBlockDmcLUKS",
            835,
            """Ceph dm-crypt Linux unitified key setup (LUKS) block partition.""",
        ),
        ("CephBlockWALog", 836, """Ceph block write-ahead log partition."""),
        (
            "CephBlockWALogDmc",
            837,
            """Ceph dm-crypt block write-ahead log partition.""",
        ),
        (
            "CephBlockWALogDmcLUKS",
            838,
            """Ceph dm-crypt Linux unitified key setup (LUKS) block write-ahead log partition.""",
        ),
        ("CephDisk", 839, """Ceph disk in creation partition."""),
        ("CephDiskDmc", 840, """Ceph dm-crypt disk in creation partition."""),
        ("CephJournal", 841, """Ceph Journal partition."""),
        ("CephJournalDmc", 842, """Ceph dm-crypt journal partition."""),
        (
            "CephJournalDmcLUKS",
            843,
            """Ceph dm-crypt Linux unitified key setup (LUKS) journal partition.""",
        ),
        ("CephLockbox", 844, """Ceph Lockbox for dm-crypt keys partition."""),
        ("CephMultipathBlock1", 845, """Ceph multipath block 1 partition."""),
        ("CephMultipathBlock2", 846, """Ceph multipath block 2 partition."""),
        ("CephMultipathBlockDB", 847, """Ceph multipath block DB partition."""),
        (
            "CephMultipathBLockWALog",
            848,
            """Ceph multipath block write-ahead log partition.""",
        ),
        ("CephMultipathJournal", 849, """Ceph multipath journal partition."""),
        (
            "CephMultipathOSD",
            851,
            """Ceph multipath object storage deamon (OSD) partition.""",
        ),
        ("CephOSD", 852, """Ceph object storage deamon (OSD) partition."""),
        ("CephOSDDmc", 853, """Ceph dm-crypt object storage deamon (OSD) partition."""),
        (
            "CephOSDDmcLUKS",
            854,
            """Ceph dm-crypt Linux unitified key setup (LUKS) object storage deamon (OSD) partition.""",
        ),
    ]


class DHCPOption(Enum):
    """IPv4 netmask. Set to :py:func:`IDHCPServer.network_mask`  by default.


    .. describe:: subnet_mask(1)

            IPv4 netmask. Set to :py:func:`IDHCPServer.network_mask`  by default.

    .. describe:: time_offset(2)

            UTC offset in seconds (32-bit decimal value).

    .. describe:: routers(3)

            Space separated list of IPv4 router addresses.

    .. describe:: time_servers(4)

            Space separated list of IPv4 time server (RFC 868) addresses.

    .. describe:: name_servers(5)

            Space separated list of IPv4 name server (IEN 116) addresses.

    .. describe:: domain_name_servers(6)

            Space separated list of IPv4 DNS addresses.

    .. describe:: log_servers(7)

            Space separated list of IPv4 log server addresses.

    .. describe:: cookie_servers(8)

            Space separated list of IPv4 cookie server (RFC 865) addresses.

    .. describe:: lpr_servers(9)

            Space separated list of IPv4 line printer server (RFC 1179) addresses.

    .. describe:: impress_servers(10)

            Space separated list of IPv4 imagen impress server addresses.

    .. describe:: resourse_location_servers(11)

            Space separated list of IPv4 resource location (RFC 887) addresses.

    .. describe:: host_name(12)

            The client name. See RFC 1035 for character limits.

    .. describe:: boot_file_size(13)

            Number of 512 byte blocks making up the boot file (16-bit decimal value).

    .. describe:: merit_dump_file(14)

            Client core file.

    .. describe:: domain_name(15)

            Domain name for the client.

    .. describe:: swap_server(16)

            IPv4 address of the swap server that the client should use.

    .. describe:: root_path(17)

            The path to the root disk the client should use.

    .. describe:: extension_path(18)

            Path to a file containing additional DHCP options (RFC2123).

    .. describe:: ip_forwarding(19)

            Whether IP forwarding should be enabled by the client (boolean).

    .. describe:: opt_non_local_source_routing(20)

            Whether non-local datagrams should be forwarded by the client (boolean)

    .. describe:: policy_filter(21)

            List of IPv4 addresses and masks paris controlling non-local source routing.

    .. describe:: max_dgram_reassembly_size(22)

            The maximum datagram size the client should reassemble (16-bit decimal value).

    .. describe:: default_ipttl(23)

            The default time-to-leave on outgoing (IP) datagrams (8-bit decimal value).

    .. describe:: path_mtu_aging_timeout(24)

            RFC1191 path MTU discovery timeout value in seconds (32-bit decimal value).

    .. describe:: path_mtu_plateau_table(25)

            RFC1191 path MTU discovery size table, sorted in ascending order (list of 16-bit decimal values).

    .. describe:: interface_mtu(26)

            The MTU size for the interface (16-bit decimal value).

    .. describe:: all_subnets_are_local(27)

            Indicates whether the MTU size is the same for all subnets (boolean).

    .. describe:: broadcast_address(28)

            Broadcast address (RFC1122) for the client to use (IPv4 address).

    .. describe:: perform_mask_discovery(29)

            Whether to perform subnet mask discovery via ICMP (boolean).

    .. describe:: mask_supplier(30)

            Whether to respond to subnet mask requests via ICMP (boolean).

    .. describe:: perform_router_discovery(31)

            Whether to perform router discovery (RFC1256) (boolean).

    .. describe:: router_solicitation_address(32)

            Where to send router solicitation requests (RFC1256) (IPv4 address).

    .. describe:: static_route(33)

            List of network and router address pairs addresses.

    .. describe:: trailer_encapsulation(34)

            Whether to negotiate the use of trailers for ARP (RTF893) (boolean).

    .. describe:: arp_cache_timeout(35)

            The timeout in seconds for ARP cache entries (32-bit decimal value).

    .. describe:: ethernet_encapsulation(36)

            Whether to use IEEE 802.3 (RTF1042) rather than of v2 (RFC894) ethernet encapsulation (boolean).

    .. describe:: tcp_default_ttl(37)

            Default time-to-live for TCP sends (non-zero 8-bit decimal value).

    .. describe:: tcp_keepalive_interval(38)

            The interface in seconds between TCP keepalive messages (32-bit decimal value).

    .. describe:: tcp_keepalive_garbage(39)

            Whether to include a byte of garbage in TCP keepalive messages for backward compatibility (boolean).

    .. describe:: nis_domain(40)

            The NIS (Sun Network Information Services) domain name (string).

    .. describe:: nis_servers(41)

            Space separated list of IPv4 NIS server addresses.

    .. describe:: ntp_servers(42)

            Space separated list of IPv4 NTP (RFC1035) server addresses.

    .. describe:: vendor_specific_info(43)

            Vendor specific information. Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .

    .. describe:: net_bios_name_servers(44)

            Space separated list of IPv4 NetBIOS name server (NBNS) addresses (RFC1001,RFC1002).

    .. describe:: net_bios_datagram_servers(45)

            Space separated list of IPv4 NetBIOS datagram distribution server (NBDD) addresses (RFC1001,RFC1002).

    .. describe:: net_bios_node_type(46)

            NetBIOS node type (RFC1001,RFC1002): 1=B-node, 2=P-node, 4=M-node, and 8=H-node (8-bit decimal value).

    .. describe:: net_bios_scope(47)

            NetBIOS scope (RFC1001,RFC1002). Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .

    .. describe:: x_windows_font_servers(48)

            Space separated list of IPv4 X windows font server addresses.

    .. describe:: x_windows_display_manager(49)

            Space separated list of IPv4 X windows display manager addresses.

    .. describe:: net_ware_ip_domain_name(62)

            Netware IP domain name (RFC2242) (string).

    .. describe:: net_ware_ip_information(63)

            Netware IP information (RFC2242). Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .

    .. describe:: nis_plus_domain(64)

            The NIS+ domain name (string).

    .. describe:: nis_plus_servers(65)

            Space separated list of IPv4 NIS+ server addresses.

    .. describe:: tftp_server_name(66)

            TFTP server name (string).

    .. describe:: bootfile_name(67)

            Bootfile name (string).

    .. describe:: mobile_ip_home_agents(68)

            Space separated list of IPv4 mobile IP agent addresses.

    .. describe:: smtp_servers(69)

            Space separated list of IPv4 simple mail transport protocol (SMPT) server addresses.

    .. describe:: pop3_servers(70)

            Space separated list of IPv4 post office protocol 3 (POP3) server addresses.

    .. describe:: nntp_servers(71)

            Space separated list of IPv4 network news transport protocol (NTTP) server addresses.

    .. describe:: www_servers(72)

            Space separated list of default IPv4 world wide web (WWW) server addresses.

    .. describe:: finger_servers(73)

            Space separated list of default IPv4 finger server addresses.

    .. describe:: irc_servers(74)

            Space separated list of default IPv4 internet relay chat (IRC) server  addresses.

    .. describe:: street_talk_servers(75)

            Space separated list of IPv4 StreetTalk server addresses.

    .. describe:: stda_servers(76)

            Space separated list of IPv4 StreetTalk directory assistance (STDA) server addresses.

    .. describe:: slp_directory_agent(78)

            Addresses of one or more service location protocol (SLP) directory agent, and an indicator of whether their use is mandatory. Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .

    .. describe:: slp_service_scope(79)

            List of service scopes for the service location protocol (SLP) and whether using the list is mandator. Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .

    .. describe:: domain_search(119)

            Domain search list, see RFC3397 and section 4.1.4 in RFC1035 for encoding.  Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .

    """

    __uuid__ = "00f5b10f-0021-4513-00f7-5bf4000982bf"
    _enums = [
        (
            "SubnetMask",
            1,
            """IPv4 netmask. Set to :py:func:`IDHCPServer.network_mask`  by default.""",
        ),
        ("TimeOffset", 2, """UTC offset in seconds (32-bit decimal value)."""),
        ("Routers", 3, """Space separated list of IPv4 router addresses."""),
        (
            "TimeServers",
            4,
            """Space separated list of IPv4 time server (RFC 868) addresses.""",
        ),
        (
            "NameServers",
            5,
            """Space separated list of IPv4 name server (IEN 116) addresses.""",
        ),
        ("DomainNameServers", 6, """Space separated list of IPv4 DNS addresses."""),
        ("LogServers", 7, """Space separated list of IPv4 log server addresses."""),
        (
            "CookieServers",
            8,
            """Space separated list of IPv4 cookie server (RFC 865) addresses.""",
        ),
        (
            "LPRServers",
            9,
            """Space separated list of IPv4 line printer server (RFC 1179) addresses.""",
        ),
        (
            "ImpressServers",
            10,
            """Space separated list of IPv4 imagen impress server addresses.""",
        ),
        (
            "ResourseLocationServers",
            11,
            """Space separated list of IPv4 resource location (RFC 887) addresses.""",
        ),
        ("HostName", 12, """The client name. See RFC 1035 for character limits."""),
        (
            "BootFileSize",
            13,
            """Number of 512 byte blocks making up the boot file (16-bit decimal value).""",
        ),
        ("MeritDumpFile", 14, """Client core file."""),
        ("DomainName", 15, """Domain name for the client."""),
        (
            "SwapServer",
            16,
            """IPv4 address of the swap server that the client should use.""",
        ),
        ("RootPath", 17, """The path to the root disk the client should use."""),
        (
            "ExtensionPath",
            18,
            """Path to a file containing additional DHCP options (RFC2123).""",
        ),
        (
            "IPForwarding",
            19,
            """Whether IP forwarding should be enabled by the client (boolean).""",
        ),
        (
            "OptNonLocalSourceRouting",
            20,
            """Whether non-local datagrams should be forwarded by the client (boolean)""",
        ),
        (
            "PolicyFilter",
            21,
            """List of IPv4 addresses and masks paris controlling non-local source routing.""",
        ),
        (
            "MaxDgramReassemblySize",
            22,
            """The maximum datagram size the client should reassemble (16-bit decimal value).""",
        ),
        (
            "DefaultIPTTL",
            23,
            """The default time-to-leave on outgoing (IP) datagrams (8-bit decimal value).""",
        ),
        (
            "PathMTUAgingTimeout",
            24,
            """RFC1191 path MTU discovery timeout value in seconds (32-bit decimal value).""",
        ),
        (
            "PathMTUPlateauTable",
            25,
            """RFC1191 path MTU discovery size table, sorted in ascending order (list of 16-bit decimal values).""",
        ),
        (
            "InterfaceMTU",
            26,
            """The MTU size for the interface (16-bit decimal value).""",
        ),
        (
            "AllSubnetsAreLocal",
            27,
            """Indicates whether the MTU size is the same for all subnets (boolean).""",
        ),
        (
            "BroadcastAddress",
            28,
            """Broadcast address (RFC1122) for the client to use (IPv4 address).""",
        ),
        (
            "PerformMaskDiscovery",
            29,
            """Whether to perform subnet mask discovery via ICMP (boolean).""",
        ),
        (
            "MaskSupplier",
            30,
            """Whether to respond to subnet mask requests via ICMP (boolean).""",
        ),
        (
            "PerformRouterDiscovery",
            31,
            """Whether to perform router discovery (RFC1256) (boolean).""",
        ),
        (
            "RouterSolicitationAddress",
            32,
            """Where to send router solicitation requests (RFC1256) (IPv4 address).""",
        ),
        ("StaticRoute", 33, """List of network and router address pairs addresses."""),
        (
            "TrailerEncapsulation",
            34,
            """Whether to negotiate the use of trailers for ARP (RTF893) (boolean).""",
        ),
        (
            "ARPCacheTimeout",
            35,
            """The timeout in seconds for ARP cache entries (32-bit decimal value).""",
        ),
        (
            "EthernetEncapsulation",
            36,
            """Whether to use IEEE 802.3 (RTF1042) rather than of v2 (RFC894) ethernet encapsulation (boolean).""",
        ),
        (
            "TCPDefaultTTL",
            37,
            """Default time-to-live for TCP sends (non-zero 8-bit decimal value).""",
        ),
        (
            "TCPKeepaliveInterval",
            38,
            """The interface in seconds between TCP keepalive messages (32-bit decimal value).""",
        ),
        (
            "TCPKeepaliveGarbage",
            39,
            """Whether to include a byte of garbage in TCP keepalive messages for backward compatibility (boolean).""",
        ),
        (
            "NISDomain",
            40,
            """The NIS (Sun Network Information Services) domain name (string).""",
        ),
        ("NISServers", 41, """Space separated list of IPv4 NIS server addresses."""),
        (
            "NTPServers",
            42,
            """Space separated list of IPv4 NTP (RFC1035) server addresses.""",
        ),
        (
            "VendorSpecificInfo",
            43,
            """Vendor specific information. Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .""",
        ),
        (
            "NetBIOSNameServers",
            44,
            """Space separated list of IPv4 NetBIOS name server (NBNS) addresses (RFC1001,RFC1002).""",
        ),
        (
            "NetBIOSDatagramServers",
            45,
            """Space separated list of IPv4 NetBIOS datagram distribution server (NBDD) addresses (RFC1001,RFC1002).""",
        ),
        (
            "NetBIOSNodeType",
            46,
            """NetBIOS node type (RFC1001,RFC1002): 1=B-node, 2=P-node, 4=M-node, and 8=H-node (8-bit decimal value).""",
        ),
        (
            "NetBIOSScope",
            47,
            """NetBIOS scope (RFC1001,RFC1002). Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .""",
        ),
        (
            "XWindowsFontServers",
            48,
            """Space separated list of IPv4 X windows font server addresses.""",
        ),
        (
            "XWindowsDisplayManager",
            49,
            """Space separated list of IPv4 X windows display manager addresses.""",
        ),
        ("NetWareIPDomainName", 62, """Netware IP domain name (RFC2242) (string)."""),
        (
            "NetWareIPInformation",
            63,
            """Netware IP information (RFC2242). Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .""",
        ),
        ("NISPlusDomain", 64, """The NIS+ domain name (string)."""),
        (
            "NISPlusServers",
            65,
            """Space separated list of IPv4 NIS+ server addresses.""",
        ),
        ("TFTPServerName", 66, """TFTP server name (string)."""),
        ("BootfileName", 67, """Bootfile name (string)."""),
        (
            "MobileIPHomeAgents",
            68,
            """Space separated list of IPv4 mobile IP agent addresses.""",
        ),
        (
            "SMTPServers",
            69,
            """Space separated list of IPv4 simple mail transport protocol (SMPT) server addresses.""",
        ),
        (
            "POP3Servers",
            70,
            """Space separated list of IPv4 post office protocol 3 (POP3) server addresses.""",
        ),
        (
            "NNTPServers",
            71,
            """Space separated list of IPv4 network news transport protocol (NTTP) server addresses.""",
        ),
        (
            "WWWServers",
            72,
            """Space separated list of default IPv4 world wide web (WWW) server addresses.""",
        ),
        (
            "FingerServers",
            73,
            """Space separated list of default IPv4 finger server addresses.""",
        ),
        (
            "IRCServers",
            74,
            """Space separated list of default IPv4 internet relay chat (IRC) server  addresses.""",
        ),
        (
            "StreetTalkServers",
            75,
            """Space separated list of IPv4 StreetTalk server addresses.""",
        ),
        (
            "STDAServers",
            76,
            """Space separated list of IPv4 StreetTalk directory assistance (STDA) server addresses.""",
        ),
        (
            "SLPDirectoryAgent",
            78,
            """Addresses of one or more service location protocol (SLP) directory agent, and an indicator of whether their use is mandatory. Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .""",
        ),
        (
            "SLPServiceScope",
            79,
            """List of service scopes for the service location protocol (SLP) and whether using the list is mandator. Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .""",
        ),
        (
            "DomainSearch",
            119,
            """Domain search list, see RFC3397 and section 4.1.4 in RFC1035 for encoding.  Only accessible using :py:attr:`DHCPOptionEncoding.hex_p` .""",
        ),
    ]


class DHCPOptionEncoding(Enum):
    """Value format is specific to the option and generally user friendly.


    .. describe:: normal(0)

            Value format is specific to the option and generally user friendly.

    .. describe:: hex_p(1)

            Value format is a series of hex bytes (09314f3200fe), optionally colons
            as byte separators (9:31:4f:32::fe).

    """

    __uuid__ = "84b6d460-2838-4682-c0d6-ef5b573ef28a"
    _enums = [
        (
            "Normal",
            0,
            """Value format is specific to the option and generally user friendly.""",
        ),
        (
            "Hex",
            1,
            """Value format is a series of hex bytes (09314f3200fe), optionally colons
            as byte separators (9:31:4f:32::fe).""",
        ),
    ]


class DHCPConfigScope(Enum):
    """:py:func:`IDHCPServer.global_config`


    .. describe:: global_p(0)

            :py:func:`IDHCPServer.global_config`

    .. describe:: group(1)

            :py:func:`IDHCPServer.group_configs`

    .. describe:: machine_nic(2)

            :py:func:`IDHCPServer.individual_configs`

    .. describe:: mac(3)

            :py:func:`IDHCPServer.individual_configs`

    """

    __uuid__ = "469c42e4-b9ec-43f2-bdcb-9e9d1eb434ae"
    _enums = [
        ("Global", 0, """:py:func:`IDHCPServer.global_config` """),
        ("Group", 1, """:py:func:`IDHCPServer.group_configs` """),
        ("MachineNIC", 2, """:py:func:`IDHCPServer.individual_configs` """),
        ("MAC", 3, """:py:func:`IDHCPServer.individual_configs` """),
    ]


class DHCPGroupConditionType(Enum):
    """MAC address


    .. describe:: mac(0)

            MAC address

    .. describe:: mac_wildcard(1)

            MAC address wildcard pattern.

    .. describe:: vendor_class_id(2)

            Vendor class ID

    .. describe:: vendor_class_id_wildcard(3)

            Vendor class ID wildcard pattern.

    .. describe:: user_class_id(4)

            User class ID

    .. describe:: user_class_id_wildcard(5)

            User class ID wildcard pattern.

    """

    __uuid__ = "2cb9280f-ada2-4194-dee8-bfb8ad77119d"
    _enums = [
        ("MAC", 0, """MAC address"""),
        ("MACWildcard", 1, """MAC address wildcard pattern."""),
        ("vendorClassID", 2, """Vendor class ID"""),
        ("vendorClassIDWildcard", 3, """Vendor class ID wildcard pattern."""),
        ("userClassID", 4, """User class ID"""),
        ("userClassIDWildcard", 5, """User class ID wildcard pattern."""),
    ]


class VFSType(Enum):
    """Virtual file systems supported by VFSExplorer.


    .. describe:: file_p(1)



    .. describe:: cloud(2)



    .. describe:: s3(3)



    .. describe:: web_dav(4)



    """

    __uuid__ = "813999ba-b949-48a8-9230-aadc6285e2f2"
    _enums = [
        ("File", 1, """"""),
        ("Cloud", 2, """"""),
        ("S3", 3, """"""),
        ("WebDav", 4, """"""),
    ]


class ImportOptions(Enum):
    """Import options, used with :py:func:`IAppliance.import_machines` .


    .. describe:: keep_all_ma_cs(1)

            Don't generate new MAC addresses of the attached network adapters.

    .. describe:: keep_natma_cs(2)

            Don't generate new MAC addresses of the attached network adapters when they are using NAT.

    .. describe:: import_to_vdi(3)

            Import all disks to VDI format

    """

    __uuid__ = "0a981523-3b20-4004-8ee3-dfd322202ace"
    _enums = [
        (
            "KeepAllMACs",
            1,
            """Don't generate new MAC addresses of the attached network adapters.""",
        ),
        (
            "KeepNATMACs",
            2,
            """Don't generate new MAC addresses of the attached network adapters when they are using NAT.""",
        ),
        ("ImportToVDI", 3, """Import all disks to VDI format"""),
    ]


class ExportOptions(Enum):
    """Export options, used with :py:func:`IAppliance.write` .


    .. describe:: create_manifest(1)

            Write the optional manifest file (.mf) which is used for integrity
            checks prior import.

    .. describe:: export_dvd_images(2)

            Export DVD images. Default is not to export them as it is rarely
            needed for typical VMs.

    .. describe:: strip_all_ma_cs(3)

            Do not export any MAC address information. Default is to keep them
            to avoid losing information which can cause trouble after import, at the
            price of risking duplicate MAC addresses, if the import options are used
            to keep them.

    .. describe:: strip_all_non_natma_cs(4)

            Do not export any MAC address information, except for adapters
            using NAT. Default is to keep them to avoid losing information which can
            cause trouble after import, at the price of risking duplicate MAC
            addresses, if the import options are used to keep them.

    """

    __uuid__ = "8f45eb08-fd34-41ee-af95-a880bdee5554"
    _enums = [
        (
            "CreateManifest",
            1,
            """Write the optional manifest file (.mf) which is used for integrity
            checks prior import.""",
        ),
        (
            "ExportDVDImages",
            2,
            """Export DVD images. Default is not to export them as it is rarely
            needed for typical VMs.""",
        ),
        (
            "StripAllMACs",
            3,
            """Do not export any MAC address information. Default is to keep them
            to avoid losing information which can cause trouble after import, at the
            price of risking duplicate MAC addresses, if the import options are used
            to keep them.""",
        ),
        (
            "StripAllNonNATMACs",
            4,
            """Do not export any MAC address information, except for adapters
            using NAT. Default is to keep them to avoid losing information which can
            cause trouble after import, at the price of risking duplicate MAC
            addresses, if the import options are used to keep them.""",
        ),
    ]


class CertificateVersion(Enum):
    """X.509 certificate version numbers.


    .. describe:: v1(1)



    .. describe:: v2(2)



    .. describe:: v3(3)



    .. describe:: unknown(99)



    """

    __uuid__ = "9e232a99-51d0-4dbd-96a0-ffac4bc3e2a8"
    _enums = [
        ("V1", 1, """"""),
        ("V2", 2, """"""),
        ("V3", 3, """"""),
        ("Unknown", 99, """"""),
    ]


class VirtualSystemDescriptionType(Enum):
    """Used with :py:class:`IVirtualSystemDescription`  to describe the type of
    a configuration value.


    .. describe:: ignore(1)



    .. describe:: os(2)



    .. describe:: name(3)



    .. describe:: product(4)



    .. describe:: vendor(5)



    .. describe:: version(6)



    .. describe:: product_url(7)



    .. describe:: vendor_url(8)



    .. describe:: description(9)



    .. describe:: license_p(10)



    .. describe:: miscellaneous(11)



    .. describe:: cpu(12)



    .. describe:: memory(13)



    .. describe:: hard_disk_controller_ide(14)



    .. describe:: hard_disk_controller_sata(15)



    .. describe:: hard_disk_controller_scsi(16)



    .. describe:: hard_disk_controller_sas(17)



    .. describe:: hard_disk_image(18)



    .. describe:: floppy(19)



    .. describe:: cdrom(20)



    .. describe:: network_adapter(21)



    .. describe:: usb_controller(22)



    .. describe:: sound_card(23)



    .. describe:: settings_file(24)

            Optional, may be unset by the API caller. If this is changed by the
            API caller it defines the absolute path of the VM settings file and
            therefore also the VM folder with highest priority.

    .. describe:: base_folder(25)

            Optional, may be unset by the API caller. If set (and
            :py:attr:`VirtualSystemDescriptionType.settings_file`  is not changed),
            defines the VM base folder (taking the primary group into account if
            also set).

    .. describe:: primary_group(26)

            Optional, empty by default and may be unset by the API caller.
            Defines the primary group of the VM after import. May influence the
            selection of the VM folder. Additional groups may be configured later
            using :py:func:`IMachine.groups` , after importing.

    .. describe:: cloud_instance_shape(27)



    .. describe:: cloud_domain(28)



    .. describe:: cloud_boot_disk_size(29)



    .. describe:: cloud_bucket(30)



    .. describe:: cloud_ocivcn(31)



    .. describe:: cloud_public_ip(32)



    .. describe:: cloud_profile_name(33)



    .. describe:: cloud_oci_subnet(34)



    .. describe:: cloud_keep_object(35)



    .. describe:: cloud_launch_instance(36)



    .. describe:: cloud_instance_id(37)



    .. describe:: cloud_image_id(38)



    .. describe:: cloud_instance_state(39)



    .. describe:: cloud_image_state(40)



    .. describe:: cloud_instance_display_name(41)



    .. describe:: cloud_image_display_name(42)



    .. describe:: cloud_oci_launch_mode(43)



    .. describe:: cloud_private_ip(44)



    .. describe:: cloud_boot_volume_id(45)



    .. describe:: cloud_ocivcn_compartment(46)



    .. describe:: cloud_oci_subnet_compartment(47)



    .. describe:: cloud_public_ssh_key(48)



    .. describe:: booting_firmware(49)



    .. describe:: hard_disk_controller_virtio_scsi(60)



    """

    __uuid__ = "6e18c6d7-e1b7-4cb0-9706-1f31d18248f8"
    _enums = [
        ("Ignore", 1, """"""),
        ("OS", 2, """"""),
        ("Name", 3, """"""),
        ("Product", 4, """"""),
        ("Vendor", 5, """"""),
        ("Version", 6, """"""),
        ("ProductUrl", 7, """"""),
        ("VendorUrl", 8, """"""),
        ("Description", 9, """"""),
        ("License", 10, """"""),
        ("Miscellaneous", 11, """"""),
        ("CPU", 12, """"""),
        ("Memory", 13, """"""),
        ("HardDiskControllerIDE", 14, """"""),
        ("HardDiskControllerSATA", 15, """"""),
        ("HardDiskControllerSCSI", 16, """"""),
        ("HardDiskControllerSAS", 17, """"""),
        ("HardDiskImage", 18, """"""),
        ("Floppy", 19, """"""),
        ("CDROM", 20, """"""),
        ("NetworkAdapter", 21, """"""),
        ("USBController", 22, """"""),
        ("SoundCard", 23, """"""),
        (
            "SettingsFile",
            24,
            """Optional, may be unset by the API caller. If this is changed by the
            API caller it defines the absolute path of the VM settings file and
            therefore also the VM folder with highest priority.""",
        ),
        (
            "BaseFolder",
            25,
            """Optional, may be unset by the API caller. If set (and
            :py:attr:`VirtualSystemDescriptionType.settings_file`  is not changed),
            defines the VM base folder (taking the primary group into account if
            also set).""",
        ),
        (
            "PrimaryGroup",
            26,
            """Optional, empty by default and may be unset by the API caller.
            Defines the primary group of the VM after import. May influence the
            selection of the VM folder. Additional groups may be configured later
            using :py:func:`IMachine.groups` , after importing.""",
        ),
        ("CloudInstanceShape", 27, """"""),
        ("CloudDomain", 28, """"""),
        ("CloudBootDiskSize", 29, """"""),
        ("CloudBucket", 30, """"""),
        ("CloudOCIVCN", 31, """"""),
        ("CloudPublicIP", 32, """"""),
        ("CloudProfileName", 33, """"""),
        ("CloudOCISubnet", 34, """"""),
        ("CloudKeepObject", 35, """"""),
        ("CloudLaunchInstance", 36, """"""),
        ("CloudInstanceId", 37, """"""),
        ("CloudImageId", 38, """"""),
        ("CloudInstanceState", 39, """"""),
        ("CloudImageState", 40, """"""),
        ("CloudInstanceDisplayName", 41, """"""),
        ("CloudImageDisplayName", 42, """"""),
        ("CloudOCILaunchMode", 43, """"""),
        ("CloudPrivateIP", 44, """"""),
        ("CloudBootVolumeId", 45, """"""),
        ("CloudOCIVCNCompartment", 46, """"""),
        ("CloudOCISubnetCompartment", 47, """"""),
        ("CloudPublicSSHKey", 48, """"""),
        ("BootingFirmware", 49, """"""),
        ("HardDiskControllerVirtioSCSI", 60, """"""),
    ]


class VirtualSystemDescriptionValueType(Enum):
    """Used with :py:func:`IVirtualSystemDescription.get_values_by_type`  to describe the value
    type to fetch.


    .. describe:: reference(1)



    .. describe:: original(2)



    .. describe:: auto(3)



    .. describe:: extra_config(4)



    """

    __uuid__ = "56d9403f-3425-4118-9919-36f2a9b8c77c"
    _enums = [
        ("Reference", 1, """"""),
        ("Original", 2, """"""),
        ("Auto", 3, """"""),
        ("ExtraConfig", 4, """"""),
    ]


class RecordingDestination(Enum):
    """Recording destination enumeration.


    .. describe:: none(0)

            No destination.

    .. describe:: file_p(1)

            Destination is a regular file.

    """

    __uuid__ = "11E3F06B-DEC1-48B9-BDC4-1E618D72893C"
    _enums = [
        ("None", 0, """No destination."""),
        ("File", 1, """Destination is a regular file."""),
    ]


class RecordingFeature(Enum):
    """Recording features enumeration.


    .. describe:: none(0)

            No feature set.

    .. describe:: video(1)

            Video recording.

    .. describe:: audio(2)

            Audio recording.

    """

    __uuid__ = "A7DDC6A5-DAA8-4485-B860-E9F2E98A7794"
    _enums = [
        ("None", 0, """No feature set."""),
        ("Video", 1, """Video recording."""),
        ("Audio", 2, """Audio recording."""),
    ]


class RecordingAudioCodec(Enum):
    """Recording audio codec enumeration.


    .. describe:: none(0)

            No codec set.

    .. describe:: wav_pcm(1)

            WAV format, linear PCM, uncompressed.
            Not implemented yet.

    .. describe:: opus(2)

            Opus Audio.

    """

    __uuid__ = "0AEFF775-053A-42F8-9C00-E445107DBED8"
    _enums = [
        ("None", 0, """No codec set."""),
        (
            "WavPCM",
            1,
            """WAV format, linear PCM, uncompressed.
            Not implemented yet.""",
        ),
        ("Opus", 2, """Opus Audio."""),
    ]


class RecordingVideoCodec(Enum):
    """Recording video codec enumeration.


    .. describe:: none(0)

            No codec set.

    .. describe:: vp8(1)

            VP8 codec.

    .. describe:: vp9(2)

            VP9 codec. Not implemented yet.

    .. describe:: av1(3)

            AV1 codec. Not implemented yet.

    """

    __uuid__ = "663BFC39-AFFF-49FA-98DD-322A857E877B"
    _enums = [
        ("None", 0, """No codec set."""),
        ("VP8", 1, """VP8 codec."""),
        ("VP9", 2, """VP9 codec. Not implemented yet."""),
        ("AV1", 3, """AV1 codec. Not implemented yet."""),
    ]


class RecordingVideoScalingMethod(Enum):
    """Recording video scaling method enumeration.


    .. describe:: none(0)

            No scaling performed.

    .. describe:: nearest_neighbor(1)

            Performs scaling via nearest-neighbor interpolation.
            Not yet implemented.

    .. describe:: bilinear(2)

            Performs scaling via bilinear interpolation.
            Not yet implemented.

    .. describe:: bicubic(3)

            Performs scaling via bicubic interpolation.
            Not yet implemented.

    """

    __uuid__ = "5576D890-48EE-449A-A81B-B776233598B7"
    _enums = [
        ("None", 0, """No scaling performed."""),
        (
            "NearestNeighbor",
            1,
            """Performs scaling via nearest-neighbor interpolation.
            Not yet implemented.""",
        ),
        (
            "Bilinear",
            2,
            """Performs scaling via bilinear interpolation.
            Not yet implemented.""",
        ),
        (
            "Bicubic",
            3,
            """Performs scaling via bicubic interpolation.
            Not yet implemented.""",
        ),
    ]


class RecordingVideoRateControlMode(Enum):
    """Recording video rate control mode enumeration.


    .. describe:: cbr(0)

            Constant bit rate (CBR).

    .. describe:: vbr(1)

            Variable bit rate (VBR). Not yet implemented.

    """

    __uuid__ = "D4EFB692-9F98-4112-88D3-A16FBE2BF6A8"
    _enums = [
        ("CBR", 0, """Constant bit rate (CBR)."""),
        ("VBR", 1, """Variable bit rate (VBR). Not yet implemented."""),
    ]


class GraphicsControllerType(Enum):
    """Graphics controller type, used with :py:func:`IGraphicsAdapter.graphics_controller_type` .


    .. describe:: null(0)

            Reserved value, invalid.

    .. describe:: v_box_vga(1)

            VirtualBox VGA device.

    .. describe:: vmsvga(2)

            VMware SVGA II device.

    .. describe:: v_box_svga(3)

            VirtualBox VGA device with VMware SVGA II extensions.

    """

    __uuid__ = "3e009bb0-2b57-4283-a39b-4c363d4f0808"
    _enums = [
        ("Null", 0, """Reserved value, invalid."""),
        ("VBoxVGA", 1, """VirtualBox VGA device."""),
        ("VMSVGA", 2, """VMware SVGA II device."""),
        ("VBoxSVGA", 3, """VirtualBox VGA device with VMware SVGA II extensions."""),
    ]


class CleanupMode(Enum):
    """Cleanup mode, used with :py:func:`IMachine.unregister` .


    .. describe:: unregister_only(1)

            Unregister only the machine, but neither delete snapshots nor detach media.

    .. describe:: detach_all_return_none(2)

            Delete all snapshots and detach all media but return none; this will keep all media registered.

    .. describe:: detach_all_return_hard_disks_only(3)

            Delete all snapshots, detach all media and return hard disks for closing, but not removable media.

    .. describe:: full(4)

            Delete all snapshots, detach all media and return all media for closing.

    """

    __uuid__ = "67897c50-7cca-47a9-83f6-ce8fd8eb5441"
    _enums = [
        (
            "UnregisterOnly",
            1,
            """Unregister only the machine, but neither delete snapshots nor detach media.""",
        ),
        (
            "DetachAllReturnNone",
            2,
            """Delete all snapshots and detach all media but return none; this will keep all media registered.""",
        ),
        (
            "DetachAllReturnHardDisksOnly",
            3,
            """Delete all snapshots, detach all media and return hard disks for closing, but not removable media.""",
        ),
        (
            "Full",
            4,
            """Delete all snapshots, detach all media and return all media for closing.""",
        ),
    ]


class CloneMode(Enum):
    """Clone mode, used with :py:func:`IMachine.clone_to` .


    .. describe:: machine_state(1)

            Clone the state of the selected machine.

    .. describe:: machine_and_child_states(2)

            Clone the state of the selected machine and its child snapshots if present.

    .. describe:: all_states(3)

            Clone all states (including all snapshots) of the machine, regardless of the machine object used.

    """

    __uuid__ = "A7A159FE-5096-4B8D-8C3C-D033CB0B35A8"
    _enums = [
        ("MachineState", 1, """Clone the state of the selected machine."""),
        (
            "MachineAndChildStates",
            2,
            """Clone the state of the selected machine and its child snapshots if present.""",
        ),
        (
            "AllStates",
            3,
            """Clone all states (including all snapshots) of the machine, regardless of the machine object used.""",
        ),
    ]


class CloneOptions(Enum):
    """Clone options, used with :py:func:`IMachine.clone_to` .


    .. describe:: link(1)

            Create a clone VM where all virtual disks are linked to the original VM.

    .. describe:: keep_all_ma_cs(2)

            Don't generate new MAC addresses of the attached network adapters.

    .. describe:: keep_natma_cs(3)

            Don't generate new MAC addresses of the attached network adapters when they are using NAT.

    .. describe:: keep_disk_names(4)

            Don't change the disk names.

    .. describe:: keep_hw_uui_ds(5)

            Don't change UUID of the machine hardware.

    """

    __uuid__ = "22243f8e-96ab-497c-8cf0-f40a566c630b"
    _enums = [
        (
            "Link",
            1,
            """Create a clone VM where all virtual disks are linked to the original VM.""",
        ),
        (
            "KeepAllMACs",
            2,
            """Don't generate new MAC addresses of the attached network adapters.""",
        ),
        (
            "KeepNATMACs",
            3,
            """Don't generate new MAC addresses of the attached network adapters when they are using NAT.""",
        ),
        ("KeepDiskNames", 4, """Don't change the disk names."""),
        ("KeepHwUUIDs", 5, """Don't change UUID of the machine hardware."""),
    ]


class AutostopType(Enum):
    """Autostop types, used with :py:func:`IMachine.autostop_type` .


    .. describe:: disabled(1)

            Stopping the VM during system shutdown is disabled.

    .. describe:: save_state(2)

            The state of the VM will be saved when the system shuts down.

    .. describe:: power_off(3)

            The VM is powered off when the system shuts down.

    .. describe:: acpi_shutdown(4)

            An ACPI shutdown event is generated.

    """

    __uuid__ = "6bb96740-cf34-470d-aab2-2cd48ea2e10e"
    _enums = [
        ("Disabled", 1, """Stopping the VM during system shutdown is disabled."""),
        (
            "SaveState",
            2,
            """The state of the VM will be saved when the system shuts down.""",
        ),
        ("PowerOff", 3, """The VM is powered off when the system shuts down."""),
        ("AcpiShutdown", 4, """An ACPI shutdown event is generated."""),
    ]


class VMProcPriority(Enum):
    """Virtual machine process priorities.


    .. describe:: invalid(0)

            Invalid priority, do not use.

    .. describe:: default(1)

            Default process priority determined by the OS.

    .. describe:: flat(2)

            Assumes a scheduling policy which puts the process at the default
            priority and with all thread at the same priority

    .. describe:: low(3)

            Assumes a scheduling policy which puts the process mostly below the
            default priority of the host OS.

    .. describe:: normal(5)

            Assume a scheduling policy which shares the CPU resources fairly with
            other processes running with the default priority of the host OS.

    .. describe:: high(6)

            Assumes a scheduling policy which puts the task above the default
            priority of the host OS. This policy might easily cause other tasks
            in the system to starve.

    """

    __uuid__ = "6fa72dd5-19b7-46ba-bc52-f223c98c7d80"
    _enums = [
        ("Invalid", 0, """Invalid priority, do not use."""),
        ("Default", 1, """Default process priority determined by the OS."""),
        (
            "Flat",
            2,
            """Assumes a scheduling policy which puts the process at the default
            priority and with all thread at the same priority""",
        ),
        (
            "Low",
            3,
            """Assumes a scheduling policy which puts the process mostly below the
            default priority of the host OS.""",
        ),
        (
            "Normal",
            5,
            """Assume a scheduling policy which shares the CPU resources fairly with
            other processes running with the default priority of the host OS.""",
        ),
        (
            "High",
            6,
            """Assumes a scheduling policy which puts the task above the default
            priority of the host OS. This policy might easily cause other tasks
            in the system to starve.""",
        ),
    ]


class HostNetworkInterfaceMediumType(Enum):
    """Type of encapsulation. Ethernet encapsulation includes both wired and
    wireless Ethernet connections.
    :py:class:`IHostNetworkInterface`


    .. describe:: unknown(0)

            The type of interface cannot be determined.

    .. describe:: ethernet(1)

            Ethernet frame encapsulation.

    .. describe:: ppp(2)

            Point-to-point protocol encapsulation.

    .. describe:: slip(3)

            Serial line IP encapsulation.

    """

    __uuid__ = "1aa54aaf-2497-45a2-bfb1-8eb225e93d5b"
    _enums = [
        ("Unknown", 0, """The type of interface cannot be determined."""),
        ("Ethernet", 1, """Ethernet frame encapsulation."""),
        ("PPP", 2, """Point-to-point protocol encapsulation."""),
        ("SLIP", 3, """Serial line IP encapsulation."""),
    ]


class HostNetworkInterfaceStatus(Enum):
    """Current status of the interface.
    :py:class:`IHostNetworkInterface`


    .. describe:: unknown(0)

            The state of interface cannot be determined.

    .. describe:: up(1)

            The interface is fully operational.

    .. describe:: down(2)

            The interface is not functioning.

    """

    __uuid__ = "CC474A69-2710-434B-8D99-C38E5D5A6F41"
    _enums = [
        ("Unknown", 0, """The state of interface cannot be determined."""),
        ("Up", 1, """The interface is fully operational."""),
        ("Down", 2, """The interface is not functioning."""),
    ]


class HostNetworkInterfaceType(Enum):
    """Network interface type.


    .. describe:: bridged(1)



    .. describe:: host_only(2)



    """

    __uuid__ = "67431b00-9946-48a2-bc02-b25c5919f4f3"
    _enums = [
        ("Bridged", 1, """"""),
        ("HostOnly", 2, """"""),
    ]


class UpdateCheckType(Enum):
    """Which type of software to check whether a new version exists.


    .. describe:: none(0)

            No flags specified. Do not use this.

    .. describe:: virtual_box(1)



    .. describe:: extension_pack(2)



    .. describe:: guest_additions(3)



    """

    __uuid__ = "ac34bb91-6739-4791-b30f-ce84e57928bb"
    _enums = [
        ("None", 0, """No flags specified. Do not use this."""),
        ("VirtualBox", 1, """"""),
        ("ExtensionPack", 2, """"""),
        ("GuestAdditions", 3, """"""),
    ]


class ProxyMode(Enum):
    """Proxy setting: System (default), NoProxy and Manual. :py:func:`ISystemProperties.proxy_mode`


    .. describe:: system(0)

            Use the system proxy settings as far as possible.

    .. describe:: no_proxy(1)

            Direct connection to the Internet.

    .. describe:: manual(2)

            Use the manual proxy from :py:func:`ISystemProperties.proxy_url` .

    """

    __uuid__ = "885264b3-b517-40fc-ce46-36e3bae895a4"
    _enums = [
        ("System", 0, """Use the system proxy settings as far as possible."""),
        ("NoProxy", 1, """Direct connection to the Internet."""),
        (
            "Manual",
            2,
            """Use the manual proxy from :py:func:`ISystemProperties.proxy_url` .""",
        ),
    ]


class VBoxUpdateTarget(Enum):
    """The preferred release type used for determining whether a newer version of VirtualBox is available. :py:func:`ISystemProperties.v_box_update_target`


    .. describe:: stable(0)

            Stable releases: new maintenance and minor releases within the same major release version.  This is the default.

    .. describe:: all_releases(1)

            All releases: new maintenance, minor, and major releases.

    .. describe:: with_betas(2)

            With betas: new maintenance, minor, and major releases as well as beta releases.

    """

    __uuid__ = "900d4a76-7b08-4af8-a453-f331e783eaee"
    _enums = [
        (
            "Stable",
            0,
            """Stable releases: new maintenance and minor releases within the same major release version.  This is the default.""",
        ),
        (
            "AllReleases",
            1,
            """All releases: new maintenance, minor, and major releases.""",
        ),
        (
            "WithBetas",
            2,
            """With betas: new maintenance, minor, and major releases as well as beta releases.""",
        ),
    ]


class AdditionsFacilityType(Enum):
    """Guest Additions facility IDs.


    .. describe:: none(0)

            No/invalid facility.

    .. describe:: v_box_guest_driver(20)

            VirtualBox base driver (VBoxGuest).

    .. describe:: auto_logon(90)

            Auto-logon modules (VBoxGINA, VBoxCredProv, pam_vbox).

    .. describe:: v_box_service(100)

            VirtualBox system service (VBoxService).

    .. describe:: v_box_tray_client(101)

            VirtualBox desktop integration (VBoxTray on Windows, VBoxClient on non-Windows).

    .. describe:: seamless(1000)

            Seamless guest desktop integration.

    .. describe:: graphics(1100)

            Guest graphics mode. If not enabled, seamless rendering will not work, resize hints
            are not immediately acted on and guest display resizes are probably not initiated by
            the Guest Additions.

    .. describe:: monitor_attach(1101)

            Guest supports monitor hotplug.

    .. describe:: all_p(2147483646)

            All facilities selected.

    """

    __uuid__ = "c4b10d74-dd48-4ff4-9a40-785a2a389ade"
    _enums = [
        ("None", 0, """No/invalid facility."""),
        ("VBoxGuestDriver", 20, """VirtualBox base driver (VBoxGuest)."""),
        ("AutoLogon", 90, """Auto-logon modules (VBoxGINA, VBoxCredProv, pam_vbox)."""),
        ("VBoxService", 100, """VirtualBox system service (VBoxService)."""),
        (
            "VBoxTrayClient",
            101,
            """VirtualBox desktop integration (VBoxTray on Windows, VBoxClient on non-Windows).""",
        ),
        ("Seamless", 1000, """Seamless guest desktop integration."""),
        (
            "Graphics",
            1100,
            """Guest graphics mode. If not enabled, seamless rendering will not work, resize hints
            are not immediately acted on and guest display resizes are probably not initiated by
            the Guest Additions.""",
        ),
        ("MonitorAttach", 1101, """Guest supports monitor hotplug."""),
        ("All", 2147483646, """All facilities selected."""),
    ]


class AdditionsFacilityClass(Enum):
    """Guest Additions facility classes.


    .. describe:: none(0)

            No/invalid class.

    .. describe:: driver(10)

            Driver.

    .. describe:: service(30)

            System service.

    .. describe:: program(50)

            Program.

    .. describe:: feature(100)

            Feature.

    .. describe:: third_party(999)

            Third party.

    .. describe:: all_p(2147483646)

            All facility classes selected.

    """

    __uuid__ = "446451b2-c88d-4e5d-84c9-91bc7f533f5f"
    _enums = [
        ("None", 0, """No/invalid class."""),
        ("Driver", 10, """Driver."""),
        ("Service", 30, """System service."""),
        ("Program", 50, """Program."""),
        ("Feature", 100, """Feature."""),
        ("ThirdParty", 999, """Third party."""),
        ("All", 2147483646, """All facility classes selected."""),
    ]


class AdditionsFacilityStatus(Enum):
    """Guest Additions facility states.


    .. describe:: inactive(0)

            Facility is not active.

    .. describe:: paused(1)

            Facility has been paused.

    .. describe:: pre_init(20)

            Facility is preparing to initialize.

    .. describe:: init(30)

            Facility is initializing.

    .. describe:: active(50)

            Facility is up and running.

    .. describe:: terminating(100)

            Facility is shutting down.

    .. describe:: terminated(101)

            Facility successfully shut down.

    .. describe:: failed(800)

            Facility failed to start.

    .. describe:: unknown(999)

            Facility status is unknown.

    """

    __uuid__ = "ce06f9e1-394e-4fe9-9368-5a88c567dbde"
    _enums = [
        ("Inactive", 0, """Facility is not active."""),
        ("Paused", 1, """Facility has been paused."""),
        ("PreInit", 20, """Facility is preparing to initialize."""),
        ("Init", 30, """Facility is initializing."""),
        ("Active", 50, """Facility is up and running."""),
        ("Terminating", 100, """Facility is shutting down."""),
        ("Terminated", 101, """Facility successfully shut down."""),
        ("Failed", 800, """Facility failed to start."""),
        ("Unknown", 999, """Facility status is unknown."""),
    ]


class AdditionsRunLevelType(Enum):
    """Guest Additions run level type.


    .. describe:: none(0)

            Guest Additions are not loaded.

    .. describe:: system(1)

            Guest drivers are loaded.

    .. describe:: userland(2)

            Common components (such as application services) are loaded.

    .. describe:: desktop(3)

            Per-user desktop components are loaded.

    """

    __uuid__ = "a25417ee-a9dd-4f5b-b0dc-377860087754"
    _enums = [
        ("None", 0, """Guest Additions are not loaded."""),
        ("System", 1, """Guest drivers are loaded."""),
        (
            "Userland",
            2,
            """Common components (such as application services) are loaded.""",
        ),
        ("Desktop", 3, """Per-user desktop components are loaded."""),
    ]


class AdditionsUpdateFlag(Enum):
    """Guest Additions update flags.


    .. describe:: none(0)

            No flag set.

    .. describe:: wait_for_update_start_only(1)

            Starts the regular updating process and waits until the
            actual Guest Additions update inside the guest was started.
            This can be necessary due to needed interaction with the guest
            OS during the installation phase.

    """

    __uuid__ = "726a818d-18d6-4389-94e8-3e9e6826171a"
    _enums = [
        ("None", 0, """No flag set."""),
        (
            "WaitForUpdateStartOnly",
            1,
            """Starts the regular updating process and waits until the
            actual Guest Additions update inside the guest was started.
            This can be necessary due to needed interaction with the guest
            OS during the installation phase.""",
        ),
    ]


class GuestShutdownFlag(Enum):
    """Guest shutdown flags.


    .. describe:: none(0)

            No flag set.

    .. describe:: power_off(1)

            Performs a reboot after shutdown.

    .. describe:: reboot(2)

            Performs a reboot after shutdown.

    .. describe:: force(4)

            Force the system to shutdown/reboot regardless of objecting
            application or other stuff. This flag might not be realized on
            all systems.

    """

    __uuid__ = "28D19C9C-5862-4930-B29A-F117712B4864"
    _enums = [
        ("None", 0, """No flag set."""),
        ("PowerOff", 1, """Performs a reboot after shutdown."""),
        ("Reboot", 2, """Performs a reboot after shutdown."""),
        (
            "Force",
            4,
            """Force the system to shutdown/reboot regardless of objecting
            application or other stuff. This flag might not be realized on
            all systems.""",
        ),
    ]


class GuestSessionStatus(Enum):
    """Guest session status. This enumeration represents possible values of
    the :py:func:`IGuestSession.status`  attribute.


    .. describe:: undefined(0)

            Guest session is in an undefined state.

    .. describe:: starting(10)

            Guest session is being started.

    .. describe:: started(100)

            Guest session has been started.

    .. describe:: terminating(480)

            Guest session is being terminated.

    .. describe:: terminated(500)

            Guest session terminated normally.

    .. describe:: timed_out_killed(512)

            Guest session timed out and was killed.

    .. describe:: timed_out_abnormally(513)

            Guest session timed out and was not killed successfully.

    .. describe:: down(600)

            Service/OS is stopping, guest session was killed.

    .. describe:: error(800)

            Something went wrong.

    """

    __uuid__ = "ac2669da-4624-44f2-85b5-0b0bfb8d8673"
    _enums = [
        ("Undefined", 0, """Guest session is in an undefined state."""),
        ("Starting", 10, """Guest session is being started."""),
        ("Started", 100, """Guest session has been started."""),
        ("Terminating", 480, """Guest session is being terminated."""),
        ("Terminated", 500, """Guest session terminated normally."""),
        ("TimedOutKilled", 512, """Guest session timed out and was killed."""),
        (
            "TimedOutAbnormally",
            513,
            """Guest session timed out and was not killed successfully.""",
        ),
        ("Down", 600, """Service/OS is stopping, guest session was killed."""),
        ("Error", 800, """Something went wrong."""),
    ]


class GuestSessionWaitForFlag(Enum):
    """Guest session waiting flags.


    .. describe:: none(0)

            No waiting flags specified. Do not use this.

    .. describe:: start(1)

            Wait for the guest session being started.

    .. describe:: terminate(2)

            Wait for the guest session being terminated.

    .. describe:: status(4)

            Wait for the next guest session status change.

    """

    __uuid__ = "bb7a372a-f635-4e11-a81a-e707f3a52ef5"
    _enums = [
        ("None", 0, """No waiting flags specified. Do not use this."""),
        ("Start", 1, """Wait for the guest session being started."""),
        ("Terminate", 2, """Wait for the guest session being terminated."""),
        ("Status", 4, """Wait for the next guest session status change."""),
    ]


class GuestSessionWaitResult(Enum):
    """Guest session waiting results. Depending on the session waiting flags (for
    more information see :py:class:`GuestSessionWaitForFlag` ) the waiting result
    can vary based on the session's current status.

    To wait for a guest session to terminate after it has been
    created by :py:func:`IGuest.create_session`  one would specify
    GuestSessionWaitResult_Terminate.


    .. describe:: none(0)

            No result was returned. Not being used.

    .. describe:: start(1)

            The guest session has been started.

    .. describe:: terminate(2)

            The guest session has been terminated.

    .. describe:: status(3)

            The guest session has changed its status. The status then can
            be retrieved via :py:func:`IGuestSession.status` .

    .. describe:: error(4)

            Error while executing the process.

    .. describe:: timeout(5)

            The waiting operation timed out. This also will happen
            when no event has been occurred matching the
            current waiting flags in a :py:func:`IGuestSession.wait_for`  call.

    .. describe:: wait_flag_not_supported(6)

            A waiting flag specified in the :py:func:`IGuestSession.wait_for`  call
            is not supported by the guest.

    """

    __uuid__ = "c0f6a8a5-fdb6-42bf-a582-56c6f82bcd2d"
    _enums = [
        ("None", 0, """No result was returned. Not being used."""),
        ("Start", 1, """The guest session has been started."""),
        ("Terminate", 2, """The guest session has been terminated."""),
        (
            "Status",
            3,
            """The guest session has changed its status. The status then can
            be retrieved via :py:func:`IGuestSession.status` .""",
        ),
        ("Error", 4, """Error while executing the process."""),
        (
            "Timeout",
            5,
            """The waiting operation timed out. This also will happen
            when no event has been occurred matching the
            current waiting flags in a :py:func:`IGuestSession.wait_for`  call.""",
        ),
        (
            "WaitFlagNotSupported",
            6,
            """A waiting flag specified in the :py:func:`IGuestSession.wait_for`  call
            is not supported by the guest.""",
        ),
    ]


class GuestUserState(Enum):
    """State a guest user has been changed to.


    .. describe:: unknown(0)

            Unknown state. Not being used.

    .. describe:: logged_in(1)

            A guest user has been successfully logged into
            the guest OS.
            This property is not implemented yet!

    .. describe:: logged_out(2)

            A guest user has been successfully logged out
            of the guest OS.
            This property is not implemented yet!

    .. describe:: locked(3)

            A guest user has locked its account. This might
            include running a password-protected screensaver
            in the guest.
            This property is not implemented yet!

    .. describe:: unlocked(4)

            A guest user has unlocked its account.
            This property is not implemented yet!

    .. describe:: disabled(5)

            A guest user has been disabled by the guest OS.
            This property is not implemented yet!

    .. describe:: idle(6)

            A guest user currently is not using the guest OS.
            Currently only available for Windows guests since
            Windows 2000 SP2.
            On Windows guests this function currently only supports
            reporting contiguous idle times up to 49.7 days per user.
            The event will be triggered if a guest user is not active for
            at least 5 seconds. This threshold can be adjusted by either altering
            VBoxService's command line in the guest to
            --vminfo-user-idle-threshold
            , or by setting the per-VM guest property
            /VirtualBox/GuestAdd/VBoxService/--vminfo-user-idle-threshold
            with the RDONLYGUEST flag on the host. In both cases VBoxService needs
            to be restarted in order to get the changes applied.

    .. describe:: in_use(7)

            A guest user continued using the guest OS after
            being idle.

    .. describe:: created(8)

            A guest user has been successfully created.
            This property is not implemented yet!

    .. describe:: deleted(9)

            A guest user has been successfully deleted.
            This property is not implemented yet!

    .. describe:: session_changed(10)

            To guest OS has changed the session of a user.
            This property is not implemented yet!

    .. describe:: credentials_changed(11)

            To guest OS has changed the authentication
            credentials of a user. This might include changed passwords
            and authentication types.
            This property is not implemented yet!

    .. describe:: role_changed(12)

            To guest OS has changed the role of a user permanently,
            e.g. granting / denying administrative rights.
            This property is not implemented yet!

    .. describe:: group_added(13)

            To guest OS has added a user to a specific
            user group.
            This property is not implemented yet!

    .. describe:: group_removed(14)

            To guest OS has removed a user from a specific
            user group.
            This property is not implemented yet!

    .. describe:: elevated(15)

            To guest OS temporarily has elevated a user
            to perform a certain task.
            This property is not implemented yet!

    """

    __uuid__ = "b2a82b02-fd3d-4fc2-ba84-6ba5ac8be198"
    _enums = [
        ("Unknown", 0, """Unknown state. Not being used."""),
        (
            "LoggedIn",
            1,
            """A guest user has been successfully logged into
            the guest OS.
            This property is not implemented yet!""",
        ),
        (
            "LoggedOut",
            2,
            """A guest user has been successfully logged out
            of the guest OS.
            This property is not implemented yet!""",
        ),
        (
            "Locked",
            3,
            """A guest user has locked its account. This might
            include running a password-protected screensaver
            in the guest.
            This property is not implemented yet!""",
        ),
        (
            "Unlocked",
            4,
            """A guest user has unlocked its account.
            This property is not implemented yet!""",
        ),
        (
            "Disabled",
            5,
            """A guest user has been disabled by the guest OS.
            This property is not implemented yet!""",
        ),
        (
            "Idle",
            6,
            """A guest user currently is not using the guest OS.
            Currently only available for Windows guests since
            Windows 2000 SP2.
            On Windows guests this function currently only supports
            reporting contiguous idle times up to 49.7 days per user.
            The event will be triggered if a guest user is not active for
            at least 5 seconds. This threshold can be adjusted by either altering
            VBoxService's command line in the guest to
            --vminfo-user-idle-threshold 
            , or by setting the per-VM guest property
            /VirtualBox/GuestAdd/VBoxService/--vminfo-user-idle-threshold 
            with the RDONLYGUEST flag on the host. In both cases VBoxService needs
            to be restarted in order to get the changes applied.""",
        ),
        (
            "InUse",
            7,
            """A guest user continued using the guest OS after
            being idle.""",
        ),
        (
            "Created",
            8,
            """A guest user has been successfully created.
            This property is not implemented yet!""",
        ),
        (
            "Deleted",
            9,
            """A guest user has been successfully deleted.
            This property is not implemented yet!""",
        ),
        (
            "SessionChanged",
            10,
            """To guest OS has changed the session of a user.
            This property is not implemented yet!""",
        ),
        (
            "CredentialsChanged",
            11,
            """To guest OS has changed the authentication
            credentials of a user. This might include changed passwords
            and authentication types.
            This property is not implemented yet!""",
        ),
        (
            "RoleChanged",
            12,
            """To guest OS has changed the role of a user permanently,
            e.g. granting / denying administrative rights.
            This property is not implemented yet!""",
        ),
        (
            "GroupAdded",
            13,
            """To guest OS has added a user to a specific
            user group.
            This property is not implemented yet!""",
        ),
        (
            "GroupRemoved",
            14,
            """To guest OS has removed a user from a specific
            user group.
            This property is not implemented yet!""",
        ),
        (
            "Elevated",
            15,
            """To guest OS temporarily has elevated a user
            to perform a certain task.
            This property is not implemented yet!""",
        ),
    ]


class FileSeekOrigin(Enum):
    """What a file seek (:py:func:`IFile.seek` ) is relative to.


    .. describe:: begin(0)

            Seek from the beginning of the file.

    .. describe:: current(1)

            Seek from the current file position.

    .. describe:: end(2)

            Seek relative to the end of the file.  To seek to the position two
            bytes from the end of the file, specify -2 as the seek offset.

    """

    __uuid__ = "ad32f789-4279-4530-979c-f16892e1c263"
    _enums = [
        ("Begin", 0, """Seek from the beginning of the file."""),
        ("Current", 1, """Seek from the current file position."""),
        (
            "End",
            2,
            """Seek relative to the end of the file.  To seek to the position two
            bytes from the end of the file, specify -2 as the seek offset.""",
        ),
    ]


class ProcessInputFlag(Enum):
    """Guest process input flags.


    .. describe:: none(0)

            No flag set.

    .. describe:: end_of_file(1)

            End of file (input) reached.

    """

    __uuid__ = "5d38c1dd-2604-4ddf-92e5-0c0cdd3bdbd5"
    _enums = [
        ("None", 0, """No flag set."""),
        ("EndOfFile", 1, """End of file (input) reached."""),
    ]


class ProcessOutputFlag(Enum):
    """Guest process output flags for specifying which
    type of output to retrieve.


    .. describe:: none(0)

            No flags set. Get output from stdout.

    .. describe:: std_err(1)

            Get output from stderr.

    """

    __uuid__ = "9979e85a-52bb-40b7-870c-57115e27e0f1"
    _enums = [
        ("None", 0, """No flags set. Get output from stdout."""),
        ("StdErr", 1, """Get output from stderr."""),
    ]


class ProcessWaitForFlag(Enum):
    """Process waiting flags.


    .. describe:: none(0)

            No waiting flags specified. Do not use this.

    .. describe:: start(1)

            Wait for the process being started.

    .. describe:: terminate(2)

            Wait for the process being terminated.

    .. describe:: std_in(4)

            Wait for stdin becoming available.

    .. describe:: std_out(8)

            Wait for data becoming available on stdout.

    .. describe:: std_err(16)

            Wait for data becoming available on stderr.

    """

    __uuid__ = "23b550c7-78e1-437e-98f0-65fd9757bcd2"
    _enums = [
        ("None", 0, """No waiting flags specified. Do not use this."""),
        ("Start", 1, """Wait for the process being started."""),
        ("Terminate", 2, """Wait for the process being terminated."""),
        ("StdIn", 4, """Wait for stdin becoming available."""),
        ("StdOut", 8, """Wait for data becoming available on stdout."""),
        ("StdErr", 16, """Wait for data becoming available on stderr."""),
    ]


class ProcessWaitResult(Enum):
    """Process waiting results. Depending on the process waiting flags (for
    more information see :py:class:`ProcessWaitForFlag` ) the waiting result
    can vary based on the processes' current status.

    To wait for a guest process to terminate after it has been
    created by :py:func:`IGuestSession.process_create`  or :py:func:`IGuestSession.process_create_ex`
    one would specify ProcessWaitFor_Terminate.

    If a guest process has been started with ProcessCreateFlag_WaitForStdOut
    a client can wait with ProcessWaitResult_StdOut for new data to arrive on
    stdout; same applies for ProcessCreateFlag_WaitForStdErr and
    ProcessWaitResult_StdErr.


    .. describe:: none(0)

            No result was returned. Not being used.

    .. describe:: start(1)

            The process has been started.

    .. describe:: terminate(2)

            The process has been terminated.

    .. describe:: status(3)

            The process has changed its status. The status then can
            be retrieved via :py:func:`IProcess.status` .

    .. describe:: error(4)

            Error while executing the process.

    .. describe:: timeout(5)

            The waiting operation timed out. Also use if the guest process has
            timed out in the guest side (kill attempted).

    .. describe:: std_in(6)

            The process signalled that stdin became available for writing.

    .. describe:: std_out(7)

            Data on stdout became available for reading.

    .. describe:: std_err(8)

            Data on stderr became available for reading.

    .. describe:: wait_flag_not_supported(9)

            A waiting flag specified in the :py:func:`IProcess.wait_for`  call
            is not supported by the guest.

    """

    __uuid__ = "40719cbe-f192-4fe9-a231-6697b3c8e2b4"
    _enums = [
        ("None", 0, """No result was returned. Not being used."""),
        ("Start", 1, """The process has been started."""),
        ("Terminate", 2, """The process has been terminated."""),
        (
            "Status",
            3,
            """The process has changed its status. The status then can
            be retrieved via :py:func:`IProcess.status` .""",
        ),
        ("Error", 4, """Error while executing the process."""),
        (
            "Timeout",
            5,
            """The waiting operation timed out. Also use if the guest process has
            timed out in the guest side (kill attempted).""",
        ),
        (
            "StdIn",
            6,
            """The process signalled that stdin became available for writing.""",
        ),
        ("StdOut", 7, """Data on stdout became available for reading."""),
        ("StdErr", 8, """Data on stderr became available for reading."""),
        (
            "WaitFlagNotSupported",
            9,
            """A waiting flag specified in the :py:func:`IProcess.wait_for`  call
            is not supported by the guest.""",
        ),
    ]


class FileCopyFlag(Enum):
    """File copying flags.
    Not flags are implemented yet.


    .. describe:: none(0)

            No flag set.

    .. describe:: no_replace(1)

            Do not replace the destination file if it exists.
            This flag is not implemented yet.

    .. describe:: follow_links(2)

            Follow symbolic links.
            This flag is not implemented yet.

    .. describe:: update(4)

            Only copy when the source file is newer than the destination file
            or when the destination file is missing.
            This flag is not implemented yet.

    """

    __uuid__ = "791909d7-4c64-2fa4-4303-adb10658d347"
    _enums = [
        ("None", 0, """No flag set."""),
        (
            "NoReplace",
            1,
            """Do not replace the destination file if it exists.
            This flag is not implemented yet.""",
        ),
        (
            "FollowLinks",
            2,
            """Follow symbolic links.
            This flag is not implemented yet.""",
        ),
        (
            "Update",
            4,
            """Only copy when the source file is newer than the destination file
            or when the destination file is missing.
            This flag is not implemented yet.""",
        ),
    ]


class FsObjMoveFlag(Enum):
    """File moving flags.


    .. describe:: none(0)

            No flag set.

    .. describe:: replace(1)

            Replace the destination file, symlink, etc if it exists, however this
            does not allow replacing any directories.

    .. describe:: follow_links(2)

            Follow symbolic links in the final components or not (only applied to
            the given source and target paths, not to anything else).

    .. describe:: allow_directory_moves(4)

            Allow moving directories accross file system boundraries. Because it
            is could be a big undertaking, we require extra assurance that we
            should do it when requested.

    """

    __uuid__ = "2450a05d-80c6-4c96-9a17-94d73293ff86"
    _enums = [
        ("None", 0, """No flag set."""),
        (
            "Replace",
            1,
            """Replace the destination file, symlink, etc if it exists, however this
            does not allow replacing any directories.""",
        ),
        (
            "FollowLinks",
            2,
            """Follow symbolic links in the final components or not (only applied to
            the given source and target paths, not to anything else).""",
        ),
        (
            "AllowDirectoryMoves",
            4,
            """Allow moving directories accross file system boundraries. Because it
            is could be a big undertaking, we require extra assurance that we
            should do it when requested.""",
        ),
    ]


class DirectoryCreateFlag(Enum):
    """Directory creation flags.


    .. describe:: none(0)

            No flag set.

    .. describe:: parents(1)

            No error if existing, make parent directories as needed.

    """

    __uuid__ = "bd721b0e-ced5-4f79-b368-249897c32a36"
    _enums = [
        ("None", 0, """No flag set."""),
        ("Parents", 1, """No error if existing, make parent directories as needed."""),
    ]


class DirectoryCopyFlag(Enum):
    """Directory copying flags.
    Not flags are implemented yet.


    .. describe:: none(0)

            No flag set.

    .. describe:: copy_into_existing(1)

            Allow copying into an existing destination directory.

    """

    __uuid__ = "b5901856-d064-4fbc-ab06-2909ba106154"
    _enums = [
        ("None", 0, """No flag set."""),
        (
            "CopyIntoExisting",
            1,
            """Allow copying into an existing destination directory.""",
        ),
    ]


class DirectoryRemoveRecFlag(Enum):
    """Directory recursive removement flags.

    WARNING!! THE FLAGS ARE CURRENTLY IGNORED. THE METHOD APPLIES
    :py:attr:`DirectoryRemoveRecFlag.content_and_dir`  REGARDLESS
    OF THE INPUT.


    .. describe:: none(0)

            No flag set.

    .. describe:: content_and_dir(1)

            Delete the content of the directory and the directory itself.

    .. describe:: content_only(2)

            Only delete the content of the directory, omit the directory it self.

    """

    __uuid__ = "455aabf0-7692-48f6-9061-f21579b65769"
    _enums = [
        ("None", 0, """No flag set."""),
        (
            "ContentAndDir",
            1,
            """Delete the content of the directory and the directory itself.""",
        ),
        (
            "ContentOnly",
            2,
            """Only delete the content of the directory, omit the directory it self.""",
        ),
    ]


class FsObjRenameFlag(Enum):
    """Flags for use when renaming file system objects (files, directories,
    symlink, etc), see :py:func:`IGuestSession.fs_obj_rename` .


    .. describe:: no_replace(0)

            Do not replace any destination object.

    .. describe:: replace(1)

            This will attempt to replace any destination object other except
            directories. (The default is to fail if the destination exists.)

    """

    __uuid__ = "59bbf3a1-4e23-d7cf-05d5-ccae32080ed2"
    _enums = [
        ("NoReplace", 0, """Do not replace any destination object."""),
        (
            "Replace",
            1,
            """This will attempt to replace any destination object other except
            directories. (The default is to fail if the destination exists.)""",
        ),
    ]


class ProcessCreateFlag(Enum):
    """Guest process execution flags.
    The values are passed to the Guest Additions, so its not possible
    to change (move) or reuse values.here. See EXECUTEPROCESSFLAG_XXX
    in GuestControlSvc.h.


    .. describe:: none(0)

            No flag set.

    .. describe:: wait_for_process_start_only(1)

            Only use the specified timeout value to wait for starting the guest process - the guest
            process itself then uses an infinite timeout.

    .. describe:: ignore_orphaned_processes(2)

            Do not report an error when executed processes are still alive when VBoxService or the guest OS is shutting down.

    .. describe:: hidden(4)

            Do not show the started process according to the guest OS guidelines.

    .. describe:: profile(8)

            Utilize the user's profile data when exeuting a process. Only available for Windows guests at the moment.

    .. describe:: wait_for_std_out(16)

            The guest process waits until all data from stdout is read out.

    .. describe:: wait_for_std_err(32)

            The guest process waits until all data from stderr is read out.

    .. describe:: expand_arguments(64)

            Expands environment variables in process arguments.

            This is not yet implemented and is currently silently ignored.
            We will document the protocolVersion number for this feature once it
            appears, so don't use it till then.

    .. describe:: unquoted_arguments(128)

            Work around for Windows and OS/2 applications not following normal
            argument quoting and escaping rules. The arguments are passed to the
            application without any extra quoting, just a single space between each.
            Present since VirtualBox 4.3.28 and 5.0 beta 3.

    """

    __uuid__ = "C544CD2B-F02D-4886-9901-71C523DB8DC5"
    _enums = [
        ("None", 0, """No flag set."""),
        (
            "WaitForProcessStartOnly",
            1,
            """Only use the specified timeout value to wait for starting the guest process - the guest
            process itself then uses an infinite timeout.""",
        ),
        (
            "IgnoreOrphanedProcesses",
            2,
            """Do not report an error when executed processes are still alive when VBoxService or the guest OS is shutting down.""",
        ),
        (
            "Hidden",
            4,
            """Do not show the started process according to the guest OS guidelines.""",
        ),
        (
            "Profile",
            8,
            """Utilize the user's profile data when exeuting a process. Only available for Windows guests at the moment.""",
        ),
        (
            "WaitForStdOut",
            16,
            """The guest process waits until all data from stdout is read out.""",
        ),
        (
            "WaitForStdErr",
            32,
            """The guest process waits until all data from stderr is read out.""",
        ),
        (
            "ExpandArguments",
            64,
            """Expands environment variables in process arguments.
            
            This is not yet implemented and is currently silently ignored.
            We will document the protocolVersion number for this feature once it
            appears, so don't use it till then.""",
        ),
        (
            "UnquotedArguments",
            128,
            """Work around for Windows and OS/2 applications not following normal
            argument quoting and escaping rules. The arguments are passed to the
            application without any extra quoting, just a single space between each.
            Present since VirtualBox 4.3.28 and 5.0 beta 3.""",
        ),
    ]


class ProcessPriority(Enum):
    """Process priorities.


    .. describe:: invalid(0)

            Invalid priority, do not use.

    .. describe:: default(1)

            Default process priority determined by the OS.

    """

    __uuid__ = "ee8cac50-e232-49fe-806b-d1214d9c2e49"
    _enums = [
        ("Invalid", 0, """Invalid priority, do not use."""),
        ("Default", 1, """Default process priority determined by the OS."""),
    ]


class SymlinkType(Enum):
    """Symbolic link types.  This is significant when creating links on the
    Windows platform, ignored elsewhere.


    .. describe:: unknown(0)

            It is not known what is being targeted.

    .. describe:: directory(1)

            The link targets a directory.

    .. describe:: file_p(2)

            The link targets a file (or whatever else except directories).

    """

    __uuid__ = "37794668-f8f1-4714-98a5-6f8fa2ed0118"
    _enums = [
        ("Unknown", 0, """It is not known what is being targeted."""),
        ("Directory", 1, """The link targets a directory."""),
        (
            "File",
            2,
            """The link targets a file (or whatever else except directories).""",
        ),
    ]


class SymlinkReadFlag(Enum):
    """Symbolic link reading flags.


    .. describe:: none(0)

            No flags set.

    .. describe:: no_symlinks(1)

            Don't allow symbolic links as part of the path.

    """

    __uuid__ = "b7fe2b9d-790e-4b25-8adf-1ca33026931f"
    _enums = [
        ("None", 0, """No flags set."""),
        ("NoSymlinks", 1, """Don't allow symbolic links as part of the path."""),
    ]


class ProcessStatus(Enum):
    """Process execution statuses.


    .. describe:: undefined(0)

            Process is in an undefined state.

    .. describe:: starting(10)

            Process is being started.

    .. describe:: started(100)

            Process has been started.

    .. describe:: paused(110)

            Process has been paused.

    .. describe:: terminating(480)

            Process is being terminated.

    .. describe:: terminated_normally(500)

            Process terminated normally.

    .. describe:: terminated_signal(510)

            Process terminated via signal.

    .. describe:: terminated_abnormally(511)

            Process terminated abnormally.

    .. describe:: timed_out_killed(512)

            Process timed out and was killed.

    .. describe:: timed_out_abnormally(513)

            Process timed out and was not killed successfully.

    .. describe:: down(600)

            Service/OS is stopping, process was killed.

    .. describe:: error(800)

            Something went wrong.

    """

    __uuid__ = "4d52368f-5b48-4bfe-b486-acf89139b52f"
    _enums = [
        ("Undefined", 0, """Process is in an undefined state."""),
        ("Starting", 10, """Process is being started."""),
        ("Started", 100, """Process has been started."""),
        ("Paused", 110, """Process has been paused."""),
        ("Terminating", 480, """Process is being terminated."""),
        ("TerminatedNormally", 500, """Process terminated normally."""),
        ("TerminatedSignal", 510, """Process terminated via signal."""),
        ("TerminatedAbnormally", 511, """Process terminated abnormally."""),
        ("TimedOutKilled", 512, """Process timed out and was killed."""),
        (
            "TimedOutAbnormally",
            513,
            """Process timed out and was not killed successfully.""",
        ),
        ("Down", 600, """Service/OS is stopping, process was killed."""),
        ("Error", 800, """Something went wrong."""),
    ]


class ProcessInputStatus(Enum):
    """Process input statuses.


    .. describe:: undefined(0)

            Undefined state.

    .. describe:: broken(1)

            Input pipe is broken.

    .. describe:: available(10)

            Input pipe became available for writing.

    .. describe:: written(50)

            Data has been successfully written.

    .. describe:: overflow(100)

            Too much input data supplied, data overflow.

    """

    __uuid__ = "a4a0ef9c-29cc-4805-9803-c8215ae9da6c"
    _enums = [
        ("Undefined", 0, """Undefined state."""),
        ("Broken", 1, """Input pipe is broken."""),
        ("Available", 10, """Input pipe became available for writing."""),
        ("Written", 50, """Data has been successfully written."""),
        ("Overflow", 100, """Too much input data supplied, data overflow."""),
    ]


class PathStyle(Enum):
    """The path style of a system.
    (Values matches the RTPATH_STR_F_STYLE_XXX defines in iprt/path.h!)


    .. describe:: dos(1)

            DOS-style paths with forward and backward slashes, drive
            letters and UNC.  Known from DOS, OS/2 and Windows.

    .. describe:: unix(2)

            UNIX-style paths with forward slashes only.

    .. describe:: unknown(8)

            The path style is not known, most likely because the Guest Additions
            aren't active yet.

    """

    __uuid__ = "97303a5b-42e8-0a55-d16f-d2a92c295261"
    _enums = [
        (
            "DOS",
            1,
            """DOS-style paths with forward and backward slashes, drive
            letters and UNC.  Known from DOS, OS/2 and Windows.""",
        ),
        ("UNIX", 2, """UNIX-style paths with forward slashes only."""),
        (
            "Unknown",
            8,
            """The path style is not known, most likely because the Guest Additions
            aren't active yet.""",
        ),
    ]


class FileAccessMode(Enum):
    """File open access mode for use with :py:func:`IGuestSession.file_open`
    and :py:func:`IGuestSession.file_open_ex` .


    .. describe:: read_only(1)

            Open the file only with read access.

    .. describe:: write_only(2)

            Open the file only with write access.

    .. describe:: read_write(3)

            Open the file with both read and write access.

    .. describe:: append_only(4)

            Open the file for appending only, no read or seek access.
            Not yet implemented.

    .. describe:: append_read(5)

            Open the file for appending and read.  Writes always goes to the
            end of the file while reads are done at the current or specified file
            position.
            Not yet implemented.

    """

    __uuid__ = "231a578f-47fb-ea30-3b3e-8489558227f0"
    _enums = [
        ("ReadOnly", 1, """Open the file only with read access."""),
        ("WriteOnly", 2, """Open the file only with write access."""),
        ("ReadWrite", 3, """Open the file with both read and write access."""),
        (
            "AppendOnly",
            4,
            """Open the file for appending only, no read or seek access.
            Not yet implemented.""",
        ),
        (
            "AppendRead",
            5,
            """Open the file for appending and read.  Writes always goes to the
            end of the file while reads are done at the current or specified file
            position.
            Not yet implemented.""",
        ),
    ]


class FileOpenAction(Enum):
    """What action :py:func:`IGuestSession.file_open`  and :py:func:`IGuestSession.file_open_ex`
    should take whether the file being opened exists or not.


    .. describe:: open_existing(1)

            Opens an existing file, fails if no file exists. (Was "oe".)

    .. describe:: open_or_create(2)

            Opens an existing file, creates a new one if no file exists. (Was "oc".)

    .. describe:: create_new(3)

            Creates a new file is no file exists, fails if there is a file there already. (Was "ce".)

    .. describe:: create_or_replace(4)

            Creates a new file, replace any existing file. (Was "ca".)

            Currently undefined whether we will inherit mode and ACLs from the
            existing file or replace them.

    .. describe:: open_existing_truncated(5)

            Opens and truncate an existing file, fails if no file exists. (Was "ot".)

    .. describe:: append_or_create(99)

            Opens an existing file and places the file pointer at the end of
            the file, creates the file if it does not exist.  This action implies
            write access. (Was "oa".)

            <!-- @todo r=bird: See iprt/file.h, RTFILE_O_APPEND - not an action/disposition!
            Moving the file pointer to the end, is almost fine, but implying 'write' access
            isn't. That is something that is exclusively reserved for the opening mode. -->
            Deprecated. Only here for historical reasons. Do not use!

    """

    __uuid__ = "12bc97e2-4fc6-a8b4-4f84-0cbf4ab970d2"
    _enums = [
        (
            "OpenExisting",
            1,
            """Opens an existing file, fails if no file exists. (Was "oe".)""",
        ),
        (
            "OpenOrCreate",
            2,
            """Opens an existing file, creates a new one if no file exists. (Was "oc".)""",
        ),
        (
            "CreateNew",
            3,
            """Creates a new file is no file exists, fails if there is a file there already. (Was "ce".)""",
        ),
        (
            "CreateOrReplace",
            4,
            """Creates a new file, replace any existing file. (Was "ca".)
            
            Currently undefined whether we will inherit mode and ACLs from the
            existing file or replace them.""",
        ),
        (
            "OpenExistingTruncated",
            5,
            """Opens and truncate an existing file, fails if no file exists. (Was "ot".)""",
        ),
        (
            "AppendOrCreate",
            99,
            """Opens an existing file and places the file pointer at the end of
            the file, creates the file if it does not exist.  This action implies
            write access. (Was "oa".)
            
            <!-- @todo r=bird: See iprt/file.h, RTFILE_O_APPEND - not an action/disposition!
            Moving the file pointer to the end, is almost fine, but implying 'write' access
            isn't. That is something that is exclusively reserved for the opening mode. -->
            Deprecated. Only here for historical reasons. Do not use!""",
        ),
    ]


class FileSharingMode(Enum):
    """File sharing mode for :py:func:`IGuestSession.file_open_ex` .


    .. describe:: read(1)

            Only share read access to the file.

    .. describe:: write(2)

            Only share write access to the file.

    .. describe:: read_write(3)

            Share both read and write access to the file, but deny deletion.

    .. describe:: delete(4)

            Only share delete access, denying read and write.

    .. describe:: read_delete(5)

            Share read and delete access to the file, denying writing.

    .. describe:: write_delete(6)

            Share write and delete access to the file, denying reading.

    .. describe:: all_p(7)

            Share all access, i.e. read, write and delete, to the file.

    """

    __uuid__ = "f87dfe58-425b-c5ba-7d6d-22adeea25de1"
    _enums = [
        ("Read", 1, """Only share read access to the file."""),
        ("Write", 2, """Only share write access to the file."""),
        (
            "ReadWrite",
            3,
            """Share both read and write access to the file, but deny deletion.""",
        ),
        ("Delete", 4, """Only share delete access, denying read and write."""),
        (
            "ReadDelete",
            5,
            """Share read and delete access to the file, denying writing.""",
        ),
        (
            "WriteDelete",
            6,
            """Share write and delete access to the file, denying reading.""",
        ),
        ("All", 7, """Share all access, i.e. read, write and delete, to the file."""),
    ]


class FileOpenExFlag(Enum):
    """Open flags for :py:func:`IGuestSession.file_open_ex` .


    .. describe:: none(0)

            No flag set.

    """

    __uuid__ = "4671abd4-f70c-42aa-8542-6c169cb87a5c"
    _enums = [
        ("None", 0, """No flag set."""),
    ]


class FileStatus(Enum):
    """File statuses.


    .. describe:: undefined(0)

            File is in an undefined state.

    .. describe:: opening(10)

            Guest file is opening.

    .. describe:: open_p(100)

            Guest file has been successfully opened.

    .. describe:: closing(150)

            Guest file closing.

    .. describe:: closed(200)

            Guest file has been closed.

    .. describe:: down(600)

            Service/OS is stopping, guest file was closed.

    .. describe:: error(800)

            Something went wrong.

    """

    __uuid__ = "8c86468b-b97b-4080-8914-e29f5b0abd2c"
    _enums = [
        ("Undefined", 0, """File is in an undefined state."""),
        ("Opening", 10, """Guest file is opening."""),
        ("Open", 100, """Guest file has been successfully opened."""),
        ("Closing", 150, """Guest file closing."""),
        ("Closed", 200, """Guest file has been closed."""),
        ("Down", 600, """Service/OS is stopping, guest file was closed."""),
        ("Error", 800, """Something went wrong."""),
    ]


class FsObjType(Enum):
    """File system object (file) types.


    .. describe:: unknown(1)

            Used either if the object has type that is not in this enum, or
            if the type has not yet been determined or set.

    .. describe:: fifo(2)

            FIFO or named pipe, depending on the platform/terminology.

    .. describe:: dev_char(3)

            Character device.

    .. describe:: directory(4)

            Directory.

    .. describe:: dev_block(5)

            Block device.

    .. describe:: file_p(6)

            Regular file.

    .. describe:: symlink(7)

            Symbolic link.

    .. describe:: socket(8)

            Socket.

    .. describe:: white_out(9)

            A white-out file.  Found in union mounts where it is used for
            hiding files after deletion, I think.

    """

    __uuid__ = "34a0d1aa-491e-e209-e150-84964d6cee5f"
    _enums = [
        (
            "Unknown",
            1,
            """Used either if the object has type that is not in this enum, or
            if the type has not yet been determined or set.""",
        ),
        ("Fifo", 2, """FIFO or named pipe, depending on the platform/terminology."""),
        ("DevChar", 3, """Character device."""),
        ("Directory", 4, """Directory."""),
        ("DevBlock", 5, """Block device."""),
        ("File", 6, """Regular file."""),
        ("Symlink", 7, """Symbolic link."""),
        ("Socket", 8, """Socket."""),
        (
            "WhiteOut",
            9,
            """A white-out file.  Found in union mounts where it is used for
            hiding files after deletion, I think.""",
        ),
    ]


class DnDAction(Enum):
    """Possible actions of a drag'n drop operation.


    .. describe:: ignore(0)

            Do nothing.

    .. describe:: copy(1)

            Copy the item to the target.

    .. describe:: move(2)

            Move the item to the target.

    .. describe:: link(3)

            Link the item from within the target.

    """

    __uuid__ = "17609e74-778e-4d0e-8827-35f5230f287b"
    _enums = [
        ("Ignore", 0, """Do nothing."""),
        ("Copy", 1, """Copy the item to the target."""),
        ("Move", 2, """Move the item to the target."""),
        ("Link", 3, """Link the item from within the target."""),
    ]


class DirectoryOpenFlag(Enum):
    """Directory open flags.


    .. describe:: none(0)

            No flag set.

    .. describe:: no_symlinks(1)

            Don't allow symbolic links as part of the path.

    """

    __uuid__ = "5138837a-8fd2-4194-a1b0-08f7bc3949d0"
    _enums = [
        ("None", 0, """No flag set."""),
        ("NoSymlinks", 1, """Don't allow symbolic links as part of the path."""),
    ]


class MediumState(Enum):
    """Virtual medium state.
    :py:class:`IMedium`


    .. describe:: not_created(0)

            Associated medium storage does not exist (either was not created yet or
            was deleted).

    .. describe:: created(1)

            Associated storage exists and accessible; this gets set if the
            accessibility check performed by :py:func:`IMedium.refresh_state`
            was successful.

    .. describe:: locked_read(2)

            Medium is locked for reading (see :py:func:`IMedium.lock_read` ),
            no data modification is possible.

    .. describe:: locked_write(3)

            Medium is locked for writing (see :py:func:`IMedium.lock_write` ),
            no concurrent data reading or modification is possible.

    .. describe:: inaccessible(4)

            Medium accessibility check (see :py:func:`IMedium.refresh_state` ) has
            not yet been performed, or else, associated medium storage is not
            accessible. In the first case, :py:func:`IMedium.last_access_error`
            is empty, in the second case, it describes the error that occurred.

    .. describe:: creating(5)

            Associated medium storage is being created.

    .. describe:: deleting(6)

            Associated medium storage is being deleted.

    """

    __uuid__ = "ef41e980-e012-43cd-9dea-479d4ef14d13"
    _enums = [
        (
            "NotCreated",
            0,
            """Associated medium storage does not exist (either was not created yet or
            was deleted).""",
        ),
        (
            "Created",
            1,
            """Associated storage exists and accessible; this gets set if the
            accessibility check performed by :py:func:`IMedium.refresh_state` 
            was successful.""",
        ),
        (
            "LockedRead",
            2,
            """Medium is locked for reading (see :py:func:`IMedium.lock_read` ),
            no data modification is possible.""",
        ),
        (
            "LockedWrite",
            3,
            """Medium is locked for writing (see :py:func:`IMedium.lock_write` ),
            no concurrent data reading or modification is possible.""",
        ),
        (
            "Inaccessible",
            4,
            """Medium accessibility check (see :py:func:`IMedium.refresh_state` ) has
            not yet been performed, or else, associated medium storage is not
            accessible. In the first case, :py:func:`IMedium.last_access_error` 
            is empty, in the second case, it describes the error that occurred.""",
        ),
        ("Creating", 5, """Associated medium storage is being created."""),
        ("Deleting", 6, """Associated medium storage is being deleted."""),
    ]


class MediumType(Enum):
    """Virtual medium type. For each :py:class:`IMedium` , this defines how the medium is
    attached to a virtual machine (see :py:class:`IMediumAttachment` ) and what happens
    when a snapshot (see :py:class:`ISnapshot` ) is taken of a virtual machine which has
    the medium attached. At the moment DVD and floppy media are always of type "writethrough".


    .. describe:: normal(0)

            Normal medium (attached directly or indirectly, preserved
            when taking snapshots).

    .. describe:: immutable(1)

            Immutable medium (attached indirectly, changes are wiped out
            the next time the virtual machine is started).

    .. describe:: writethrough(2)

            Write through medium (attached directly, ignored when
            taking snapshots).

    .. describe:: shareable(3)

            Allow using this medium concurrently by several machines.
            Present since VirtualBox 3.2.0, and accepted since 3.2.8.

    .. describe:: readonly(4)

            A readonly medium, which can of course be used by several machines.
            Present and accepted since VirtualBox 4.0.

    .. describe:: multi_attach(5)

            A medium which is indirectly attached, so that one base medium can
            be used for several VMs which have their own differencing medium to
            store their modifications. In some sense a variant of Immutable
            with unset AutoReset flag in each differencing medium.
            Present and accepted since VirtualBox 4.0.

    """

    __uuid__ = "fe663fb5-c244-4e1b-9d81-c628b417dd04"
    _enums = [
        (
            "Normal",
            0,
            """Normal medium (attached directly or indirectly, preserved
            when taking snapshots).""",
        ),
        (
            "Immutable",
            1,
            """Immutable medium (attached indirectly, changes are wiped out
            the next time the virtual machine is started).""",
        ),
        (
            "Writethrough",
            2,
            """Write through medium (attached directly, ignored when
            taking snapshots).""",
        ),
        (
            "Shareable",
            3,
            """Allow using this medium concurrently by several machines.
            Present since VirtualBox 3.2.0, and accepted since 3.2.8.""",
        ),
        (
            "Readonly",
            4,
            """A readonly medium, which can of course be used by several machines.
            Present and accepted since VirtualBox 4.0.""",
        ),
        (
            "MultiAttach",
            5,
            """A medium which is indirectly attached, so that one base medium can
            be used for several VMs which have their own differencing medium to
            store their modifications. In some sense a variant of Immutable
            with unset AutoReset flag in each differencing medium.
            Present and accepted since VirtualBox 4.0.""",
        ),
    ]


class MediumVariant(Enum):
    """Virtual medium image variant. More than one flag may be set.
    :py:class:`IMedium`


    .. describe:: standard(0)

            No particular variant requested, results in using the backend default.

    .. describe:: vmdk_split2_g(1)

            VMDK image split in chunks of less than 2GByte.

    .. describe:: vmdk_raw_disk(2)

            VMDK image representing a raw disk.

    .. describe:: vmdk_stream_optimized(4)

            VMDK streamOptimized image. Special import/export format which is
            read-only/append-only.

    .. describe:: vmdk_esx(8)

            VMDK format variant used on ESX products.

    .. describe:: vdi_zero_expand(256)

            Fill new blocks with zeroes while expanding image file.

    .. describe:: fixed(65536)

            Fixed image. Only allowed for base images.

    .. describe:: diff(131072)

            Differencing image. Only allowed for child images.

    .. describe:: formatted(536870912)

            Special flag which requests formatting the disk image. Right now
            supported for floppy images only.

    .. describe:: no_create_dir(1073741824)

            Special flag which suppresses automatic creation of the subdirectory.
            Only used when passing the medium variant as an input parameter.

    """

    __uuid__ = "0282e97f-4ef3-4411-a8e0-47c384803cb6"
    _enums = [
        (
            "Standard",
            0,
            """No particular variant requested, results in using the backend default.""",
        ),
        ("VmdkSplit2G", 1, """VMDK image split in chunks of less than 2GByte."""),
        ("VmdkRawDisk", 2, """VMDK image representing a raw disk."""),
        (
            "VmdkStreamOptimized",
            4,
            """VMDK streamOptimized image. Special import/export format which is
            read-only/append-only.""",
        ),
        ("VmdkESX", 8, """VMDK format variant used on ESX products."""),
        (
            "VdiZeroExpand",
            256,
            """Fill new blocks with zeroes while expanding image file.""",
        ),
        ("Fixed", 65536, """Fixed image. Only allowed for base images."""),
        ("Diff", 131072, """Differencing image. Only allowed for child images."""),
        (
            "Formatted",
            536870912,
            """Special flag which requests formatting the disk image. Right now
            supported for floppy images only.""",
        ),
        (
            "NoCreateDir",
            1073741824,
            """Special flag which suppresses automatic creation of the subdirectory.
            Only used when passing the medium variant as an input parameter.""",
        ),
    ]


class DataType(Enum):
    """


    .. describe:: int32(0)



    .. describe:: int8(1)



    .. describe:: string(2)



    """

    __uuid__ = "d90ea51e-a3f1-4a01-beb1-c1723c0d3ba7"
    _enums = [
        ("Int32", 0, """"""),
        ("Int8", 1, """"""),
        ("String", 2, """"""),
    ]


class DataFlags(Enum):
    """


    .. describe:: none(0)



    .. describe:: mandatory(1)



    .. describe:: expert(2)



    .. describe:: array(4)



    .. describe:: flag_mask(7)



    """

    __uuid__ = "86884dcf-1d6b-4f1b-b4bf-f5aa44959d60"
    _enums = [
        ("None", 0, """"""),
        ("Mandatory", 1, """"""),
        ("Expert", 2, """"""),
        ("Array", 4, """"""),
        ("FlagMask", 7, """"""),
    ]


class MediumFormatCapabilities(Enum):
    """Medium format capability flags.


    .. describe:: uuid(1)

            Supports UUIDs as expected by VirtualBox code.

    .. describe:: create_fixed(2)

            Supports creating fixed size images, allocating all space instantly.

    .. describe:: create_dynamic(4)

            Supports creating dynamically growing images, allocating space on
            demand.

    .. describe:: create_split2_g(8)

            Supports creating images split in chunks of a bit less than 2 GBytes.

    .. describe:: differencing(16)

            Supports being used as a format for differencing media (see :py:func:`IMedium.create_diff_storage` ).

    .. describe:: asynchronous(32)

            Supports asynchronous I/O operations for at least some configurations.

    .. describe:: file_p(64)

            The format backend operates on files (the :py:func:`IMedium.location`
            attribute of the medium specifies a file used to store medium
            data; for a list of supported file extensions see
            :py:func:`IMediumFormat.describe_file_extensions` ).

    .. describe:: properties(128)

            The format backend uses the property interface to configure the storage
            location and properties (the :py:func:`IMediumFormat.describe_properties`
            method is used to get access to properties supported by the given medium format).

    .. describe:: tcp_networking(256)

            The format backend uses the TCP networking interface for network access.

    .. describe:: vfs(512)

            The format backend supports virtual filesystem functionality.

    .. describe:: discard(1024)

            The format backend supports discarding blocks.

    .. describe:: preferred(2048)

            Indicates that this is a frequently used format backend.

    .. describe:: capability_mask(4095)



    """

    __uuid__ = "7342ba79-7ce0-4d94-8f86-5ed5a185d9bd"
    _enums = [
        ("Uuid", 1, """Supports UUIDs as expected by VirtualBox code."""),
        (
            "CreateFixed",
            2,
            """Supports creating fixed size images, allocating all space instantly.""",
        ),
        (
            "CreateDynamic",
            4,
            """Supports creating dynamically growing images, allocating space on
            demand.""",
        ),
        (
            "CreateSplit2G",
            8,
            """Supports creating images split in chunks of a bit less than 2 GBytes.""",
        ),
        (
            "Differencing",
            16,
            """Supports being used as a format for differencing media (see :py:func:`IMedium.create_diff_storage` ).""",
        ),
        (
            "Asynchronous",
            32,
            """Supports asynchronous I/O operations for at least some configurations.""",
        ),
        (
            "File",
            64,
            """The format backend operates on files (the :py:func:`IMedium.location` 
            attribute of the medium specifies a file used to store medium
            data; for a list of supported file extensions see
            :py:func:`IMediumFormat.describe_file_extensions` ).""",
        ),
        (
            "Properties",
            128,
            """The format backend uses the property interface to configure the storage
            location and properties (the :py:func:`IMediumFormat.describe_properties` 
            method is used to get access to properties supported by the given medium format).""",
        ),
        (
            "TcpNetworking",
            256,
            """The format backend uses the TCP networking interface for network access.""",
        ),
        (
            "VFS",
            512,
            """The format backend supports virtual filesystem functionality.""",
        ),
        ("Discard", 1024, """The format backend supports discarding blocks."""),
        (
            "Preferred",
            2048,
            """Indicates that this is a frequently used format backend.""",
        ),
        ("CapabilityMask", 4095, """"""),
    ]


class PartitionTableType(Enum):
    """Partition table types.


    .. describe:: mbr(1)



    .. describe:: gpt(2)



    """

    __uuid__ = "360066eb-d19e-4fa1-57ef-fed434fbe2a9"
    _enums = [
        ("MBR", 1, """"""),
        ("GPT", 2, """"""),
    ]


class KeyboardLED(Enum):
    """Keyboard LED indicators.


    .. describe:: num_lock(1)



    .. describe:: caps_lock(2)



    .. describe:: scroll_lock(4)



    """

    __uuid__ = "ef29ea38-409b-49c7-a817-c858d426dfba"
    _enums = [
        ("NumLock", 1, """"""),
        ("CapsLock", 2, """"""),
        ("ScrollLock", 4, """"""),
    ]


class MouseButtonState(Enum):
    """Mouse button state.


    .. describe:: left_button(1)



    .. describe:: right_button(2)



    .. describe:: middle_button(4)



    .. describe:: wheel_up(8)



    .. describe:: wheel_down(16)



    .. describe:: x_button1(32)



    .. describe:: x_button2(64)



    .. describe:: mouse_state_mask(127)



    """

    __uuid__ = "9ee094b8-b28a-4d56-a166-973cb588d7f8"
    _enums = [
        ("LeftButton", 1, """"""),
        ("RightButton", 2, """"""),
        ("MiddleButton", 4, """"""),
        ("WheelUp", 8, """"""),
        ("WheelDown", 16, """"""),
        ("XButton1", 32, """"""),
        ("XButton2", 64, """"""),
        ("MouseStateMask", 127, """"""),
    ]


class TouchContactState(Enum):
    """Touch event contact state.


    .. describe:: none(0)

            The touch has finished.

    .. describe:: in_contact(1)

            Whether the touch is really touching the device.

    .. describe:: in_range(2)

            Whether the touch is close enough to the device to be detected.

    .. describe:: contact_state_mask(3)



    """

    __uuid__ = "3f942686-2506-421c-927c-90d4b45f4a38"
    _enums = [
        ("None", 0, """The touch has finished."""),
        ("InContact", 1, """Whether the touch is really touching the device."""),
        (
            "InRange",
            2,
            """Whether the touch is close enough to the device to be detected.""",
        ),
        ("ContactStateMask", 3, """"""),
    ]


class FramebufferCapabilities(Enum):
    """Framebuffer capability flags.


    .. describe:: update_image(1)

            Requires NotifyUpdateImage. NotifyUpdate must not be called.

    .. describe:: vhwa(2)

            Supports VHWA interface. If set, then
            IFramebuffer::processVHWACommand can be called. <!-- no link, otherwise trouble with javadoc -->

    .. describe:: visible_region(4)

            Supports visible region. If set, then
            IFramebuffer::setVisibleRegion can be called. <!-- no link, otherwise trouble with javadoc -->

    .. describe:: render_cursor(8)

            This framebuffer implementation can render a pointer cursor itself.  Unless the
            MoveCursor capability is also set the cursor will always be rendered at the
            location of (and usually using) the host pointer.

    .. describe:: move_cursor(16)

            Supports rendering a pointer cursor anywhere within the guest screen.  Implies
            RenderCursor.

    """

    __uuid__ = "cc395839-30fa-4ca5-ae65-e6360e3edd7a"
    _enums = [
        (
            "UpdateImage",
            1,
            """Requires NotifyUpdateImage. NotifyUpdate must not be called.""",
        ),
        (
            "VHWA",
            2,
            """Supports VHWA interface. If set, then
            IFramebuffer::processVHWACommand can be called. <!-- no link, otherwise trouble with javadoc -->""",
        ),
        (
            "VisibleRegion",
            4,
            """Supports visible region. If set, then
            IFramebuffer::setVisibleRegion can be called. <!-- no link, otherwise trouble with javadoc -->""",
        ),
        (
            "RenderCursor",
            8,
            """This framebuffer implementation can render a pointer cursor itself.  Unless the
            MoveCursor capability is also set the cursor will always be rendered at the
            location of (and usually using) the host pointer.""",
        ),
        (
            "MoveCursor",
            16,
            """Supports rendering a pointer cursor anywhere within the guest screen.  Implies
            RenderCursor.""",
        ),
    ]


class GuestMonitorStatus(Enum):
    """The current status of the guest display.


    .. describe:: disabled(0)

            The guest monitor is disabled in the guest.

    .. describe:: enabled(1)

            The guest monitor is enabled in the guest.

    .. describe:: blank(2)

            The guest monitor is enabled in the guest but should display nothing.

    """

    __uuid__ = "6b8d3f71-39cb-459e-a916-48917ed43e19"
    _enums = [
        ("Disabled", 0, """The guest monitor is disabled in the guest."""),
        ("Enabled", 1, """The guest monitor is enabled in the guest."""),
        (
            "Blank",
            2,
            """The guest monitor is enabled in the guest but should display nothing.""",
        ),
    ]


class ScreenLayoutMode(Enum):
    """How :py:func:`IDisplay.set_screen_layout`  method should work.


    .. describe:: apply_p(0)

            If the guest is already at desired mode then the API might avoid setting the mode.

    .. describe:: reset(1)

            Always set the new mode even if the guest is already at desired mode.

    .. describe:: attach(2)

            Attach new screens and always set the new mode for existing screens.

    .. describe:: silent(3)

            Do not notify the guest of the change.  Normally this is wished, but it
            might not be when re-setting monitor information from the last session
            (no hotplug happened, as it is still the same virtual monitor).

    """

    __uuid__ = "8fa1964c-8774-11e9-ae5d-1f419105e68d"
    _enums = [
        (
            "Apply",
            0,
            """If the guest is already at desired mode then the API might avoid setting the mode.""",
        ),
        (
            "Reset",
            1,
            """Always set the new mode even if the guest is already at desired mode.""",
        ),
        (
            "Attach",
            2,
            """Attach new screens and always set the new mode for existing screens.""",
        ),
        (
            "Silent",
            3,
            """Do not notify the guest of the change.  Normally this is wished, but it
            might not be when re-setting monitor information from the last session
            (no hotplug happened, as it is still the same virtual monitor).""",
        ),
    ]


class NetworkAttachmentType(Enum):
    """Network attachment type.


    .. describe:: null(0)

            Null value, also means "not attached".

    .. describe:: nat(1)



    .. describe:: bridged(2)



    .. describe:: internal(3)



    .. describe:: host_only(4)



    .. describe:: generic(5)



    .. describe:: nat_network(6)



    .. describe:: cloud(7)



    """

    __uuid__ = "524a8f9d-4b86-4b51-877d-1aa27c4ebeac"
    _enums = [
        ("Null", 0, """Null value, also means "not attached"."""),
        ("NAT", 1, """"""),
        ("Bridged", 2, """"""),
        ("Internal", 3, """"""),
        ("HostOnly", 4, """"""),
        ("Generic", 5, """"""),
        ("NATNetwork", 6, """"""),
        ("Cloud", 7, """"""),
    ]


class NetworkAdapterType(Enum):
    """Network adapter type.


    .. describe:: null(0)

            Null value (never used by the API).

    .. describe:: am79_c970_a(1)

            AMD PCNet-PCI II network card (Am79C970A).

    .. describe:: am79_c973(2)

            AMD PCNet-FAST III network card (Am79C973).

    .. describe:: i82540_em(3)

            Intel PRO/1000 MT Desktop network card (82540EM).

    .. describe:: i82543_gc(4)

            Intel PRO/1000 T Server network card (82543GC).

    .. describe:: i82545_em(5)

            Intel PRO/1000 MT Server network card (82545EM).

    .. describe:: virtio(6)

            Virtio network device.

    .. describe:: am79_c960(7)

            AMD PCnet-ISA/NE2100 network card (Am79C960).

    .. describe:: virtio_1_0(8)

            Virtio 1.0 network device.

    """

    __uuid__ = "3c2281e4-d952-4e87-8c7d-24379cb6a81c"
    _enums = [
        ("Null", 0, """Null value (never used by the API)."""),
        ("Am79C970A", 1, """AMD PCNet-PCI II network card (Am79C970A)."""),
        ("Am79C973", 2, """AMD PCNet-FAST III network card (Am79C973)."""),
        ("I82540EM", 3, """Intel PRO/1000 MT Desktop network card (82540EM)."""),
        ("I82543GC", 4, """Intel PRO/1000 T Server network card (82543GC)."""),
        ("I82545EM", 5, """Intel PRO/1000 MT Server network card (82545EM)."""),
        ("Virtio", 6, """Virtio network device."""),
        ("Am79C960", 7, """AMD PCnet-ISA/NE2100 network card (Am79C960)."""),
        ("Virtio_1_0", 8, """Virtio 1.0 network device."""),
    ]


class NetworkAdapterPromiscModePolicy(Enum):
    """The promiscuous mode policy of an interface.


    .. describe:: deny(1)

            Deny promiscuous mode requests.

    .. describe:: allow_network(2)

            Allow promiscuous mode, but restrict the scope it to the internal
            network so that it only applies to other VMs.

    .. describe:: allow_all(3)

            Allow promiscuous mode, include unrelated traffic going over the wire
            and internally on the host.

    """

    __uuid__ = "c963768a-376f-4c85-8d84-d8ced4b7269e"
    _enums = [
        ("Deny", 1, """Deny promiscuous mode requests."""),
        (
            "AllowNetwork",
            2,
            """Allow promiscuous mode, but restrict the scope it to the internal
            network so that it only applies to other VMs.""",
        ),
        (
            "AllowAll",
            3,
            """Allow promiscuous mode, include unrelated traffic going over the wire
            and internally on the host.""",
        ),
    ]


class PortMode(Enum):
    """The PortMode enumeration represents possible communication modes for
    the virtual serial port device.


    .. describe:: disconnected(0)

            Virtual device is not attached to any real host device.

    .. describe:: host_pipe(1)

            Virtual device is attached to a host pipe.

    .. describe:: host_device(2)

            Virtual device is attached to a host device.

    .. describe:: raw_file(3)

            Virtual device is attached to a raw file.

    .. describe:: tcp(4)

            Virtual device is attached to a TCP socket.

    """

    __uuid__ = "7485fcfd-d603-470a-87af-26d33beb7de9"
    _enums = [
        (
            "Disconnected",
            0,
            """Virtual device is not attached to any real host device.""",
        ),
        ("HostPipe", 1, """Virtual device is attached to a host pipe."""),
        ("HostDevice", 2, """Virtual device is attached to a host device."""),
        ("RawFile", 3, """Virtual device is attached to a raw file."""),
        ("TCP", 4, """Virtual device is attached to a TCP socket."""),
    ]


class UartType(Enum):
    """The UART type represents the emulated UART chip for the serial port device.


    .. describe:: u16450(0)

            The most basic emulated UART which doesn't support FIFO operation.

    .. describe:: u16550_a(1)

            The successor of the 16450 UART introducing a 16 byte FIFO to reduce
            operational overhead.

    .. describe:: u16750(2)

            This UART developed by Texas Instruments introduced a 64 byte FIFO
            and hardware flow control.

    """

    __uuid__ = "c8899d39-0b90-4265-9d02-1e38bd4d1b39"
    _enums = [
        (
            "U16450",
            0,
            """The most basic emulated UART which doesn't support FIFO operation.""",
        ),
        (
            "U16550A",
            1,
            """The successor of the 16450 UART introducing a 16 byte FIFO to reduce
            operational overhead.""",
        ),
        (
            "U16750",
            2,
            """This UART developed by Texas Instruments introduced a 64 byte FIFO
            and hardware flow control.""",
        ),
    ]


class VMExecutionEngine(Enum):
    """The main execution engine of a VM.


    .. describe:: not_set(0)

            Has not yet been set (try again later).

    .. describe:: raw_mode(1)

            Raw-mode.

    .. describe:: hw_virt(2)

            Hardware assisted virtualization thru HM.

    .. describe:: native_api(3)

            Hardware assisted virtualization thru native API (NEM).

    """

    __uuid__ = "56029577-31f7-44d2-3334-7ecbf95294b6"
    _enums = [
        ("NotSet", 0, """Has not yet been set (try again later)."""),
        ("RawMode", 1, """Raw-mode."""),
        ("HwVirt", 2, """Hardware assisted virtualization thru HM."""),
        ("NativeApi", 3, """Hardware assisted virtualization thru native API (NEM)."""),
    ]


class USBControllerType(Enum):
    """The USB controller type. :py:func:`IUSBController.type_p` .


    .. describe:: null(0)

            @c null value. Never used by the API.

    .. describe:: ohci(1)



    .. describe:: ehci(2)



    .. describe:: xhci(3)



    .. describe:: last(4)

            Last element (invalid). Used for parameter checks.

    """

    __uuid__ = "8fdd1c6a-5412-41da-ab07-7baed7d6e18e"
    _enums = [
        ("Null", 0, """@c null value. Never used by the API."""),
        ("OHCI", 1, """"""),
        ("EHCI", 2, """"""),
        ("XHCI", 3, """"""),
        ("Last", 4, """Last element (invalid). Used for parameter checks."""),
    ]


class USBConnectionSpeed(Enum):
    """USB device/port speed state. This enumeration represents speeds at
    which a USB device can communicate with the host.

    The speed is a function of both the device itself and the port which
    it is attached to, including hubs and cables in the path.


    Due to differences in USB stack implementations on various hosts,
    the reported speed may not exactly match the actual speed.


    :py:class:`IHostUSBDevice`


    .. describe:: null(0)

            @c null value. Never returned by the API.

    .. describe:: low(1)

            Low speed, 1.5 Mbps.

    .. describe:: full(2)

            Full speed, 12 Mbps.

    .. describe:: high(3)

            High speed, 480 Mbps.

    .. describe:: super_p(4)

            SuperSpeed, 5 Gbps.

    .. describe:: super_plus(5)

            SuperSpeedPlus, 10 Gbps.

    """

    __uuid__ = "d2915840-ea26-4fb4-b72a-21eaf6b888ff"
    _enums = [
        ("Null", 0, """@c null value. Never returned by the API."""),
        ("Low", 1, """Low speed, 1.5 Mbps."""),
        ("Full", 2, """Full speed, 12 Mbps."""),
        ("High", 3, """High speed, 480 Mbps."""),
        ("Super", 4, """SuperSpeed, 5 Gbps."""),
        ("SuperPlus", 5, """SuperSpeedPlus, 10 Gbps."""),
    ]


class USBDeviceState(Enum):
    """USB device state. This enumeration represents all possible states
    of the USB device physically attached to the host computer regarding
    its state on the host computer and availability to guest computers
    (all currently running virtual machines).

    Once a supported USB device is attached to the host, global USB
    filters (:py:func:`IHost.usb_device_filters` ) are activated. They can
    either ignore the device, or put it to USBDeviceState_Held state, or do
    nothing. Unless the device is ignored by global filters, filters of all
    currently running guests (:py:func:`IUSBDeviceFilters.device_filters` ) are
    activated that can put it to USBDeviceState_Captured state.

    If the device was ignored by global filters, or didn't match
    any filters at all (including guest ones), it is handled by the host
    in a normal way. In this case, the device state is determined by
    the host and can be one of USBDeviceState_Unavailable, USBDeviceState_Busy
    or USBDeviceState_Available, depending on the current device usage.

    Besides auto-capturing based on filters, the device can be manually
    captured by guests (:py:func:`IConsole.attach_usb_device` ) if its
    state is USBDeviceState_Busy, USBDeviceState_Available or
    USBDeviceState_Held.


    Due to differences in USB stack implementations in Linux and Win32,
    states USBDeviceState_Busy and USBDeviceState_Unavailable are applicable
    only to the Linux version of the product. This also means that (:py:func:`IConsole.attach_usb_device` ) can only succeed on Win32 if the
    device state is USBDeviceState_Held.


    :py:class:`IHostUSBDevice` , :py:class:`IHostUSBDeviceFilter`


    .. describe:: not_supported(0)

            Not supported by the VirtualBox server, not available to guests.

    .. describe:: unavailable(1)

            Being used by the host computer exclusively,
            not available to guests.

    .. describe:: busy(2)

            Being used by the host computer, potentially available to guests.

    .. describe:: available(3)

            Not used by the host computer, available to guests (the host computer
            can also start using the device at any time).

    .. describe:: held(4)

            Held by the VirtualBox server (ignored by the host computer),
            available to guests.

    .. describe:: captured(5)

            Captured by one of the guest computers, not available
            to anybody else.

    """

    __uuid__ = "b99a2e65-67fb-4882-82fd-f3e5e8193ab4"
    _enums = [
        (
            "NotSupported",
            0,
            """Not supported by the VirtualBox server, not available to guests.""",
        ),
        (
            "Unavailable",
            1,
            """Being used by the host computer exclusively,
            not available to guests.""",
        ),
        (
            "Busy",
            2,
            """Being used by the host computer, potentially available to guests.""",
        ),
        (
            "Available",
            3,
            """Not used by the host computer, available to guests (the host computer
            can also start using the device at any time).""",
        ),
        (
            "Held",
            4,
            """Held by the VirtualBox server (ignored by the host computer),
            available to guests.""",
        ),
        (
            "Captured",
            5,
            """Captured by one of the guest computers, not available
            to anybody else.""",
        ),
    ]


class USBDeviceFilterAction(Enum):
    """Actions for host USB device filters.
    :py:class:`IHostUSBDeviceFilter` , :py:class:`USBDeviceState`


    .. describe:: null(0)

            Null value (never used by the API).

    .. describe:: ignore(1)

            Ignore the matched USB device.

    .. describe:: hold(2)

            Hold the matched USB device.

    """

    __uuid__ = "cbc30a49-2f4e-43b5-9da6-121320475933"
    _enums = [
        ("Null", 0, """Null value (never used by the API)."""),
        ("Ignore", 1, """Ignore the matched USB device."""),
        ("Hold", 2, """Hold the matched USB device."""),
    ]


class AudioDriverType(Enum):
    """Host audio driver type.


    .. describe:: null(0)

            Null value, also means "dummy audio driver".

    .. describe:: win_mm(1)

            Windows multimedia (Windows hosts only, not supported at the moment).

    .. describe:: oss(2)

            Open Sound System (Linux / Unix hosts only).

    .. describe:: alsa(3)

            Advanced Linux Sound Architecture (Linux hosts only).

    .. describe:: direct_sound(4)

            DirectSound (Windows hosts only).

    .. describe:: core_audio(5)

            CoreAudio (Mac hosts only).

    .. describe:: mmpm(6)

            Reserved for historical reasons.

    .. describe:: pulse(7)

            PulseAudio (Linux hosts only).

    .. describe:: sol_audio(8)

            Solaris audio (Solaris hosts only, not supported at the moment).

    """

    __uuid__ = "4bcc3d73-c2fe-40db-b72f-0c2ca9d68496"
    _enums = [
        ("Null", 0, """Null value, also means "dummy audio driver"."""),
        (
            "WinMM",
            1,
            """Windows multimedia (Windows hosts only, not supported at the moment).""",
        ),
        ("OSS", 2, """Open Sound System (Linux / Unix hosts only)."""),
        ("ALSA", 3, """Advanced Linux Sound Architecture (Linux hosts only)."""),
        ("DirectSound", 4, """DirectSound (Windows hosts only)."""),
        ("CoreAudio", 5, """CoreAudio (Mac hosts only)."""),
        ("MMPM", 6, """Reserved for historical reasons."""),
        ("Pulse", 7, """PulseAudio (Linux hosts only)."""),
        (
            "SolAudio",
            8,
            """Solaris audio (Solaris hosts only, not supported at the moment).""",
        ),
    ]


class AudioControllerType(Enum):
    """Virtual audio controller type.


    .. describe:: ac97(0)



    .. describe:: sb16(1)



    .. describe:: hda(2)



    """

    __uuid__ = "7afd395c-42c3-444e-8788-3ce80292f36c"
    _enums = [
        ("AC97", 0, """"""),
        ("SB16", 1, """"""),
        ("HDA", 2, """"""),
    ]


class AudioCodecType(Enum):
    """The exact variant of audio codec hardware presented
    to the guest; see :py:func:`IAudioAdapter.audio_codec` .


    .. describe:: null(0)

            @c null value. Never used by the API.

    .. describe:: sb16(1)

            SB16; this is the only option for the SB16 device.

    .. describe:: stac9700(2)

            A STAC9700 AC'97 codec.

    .. describe:: ad1980(3)

            An AD1980 AC'97 codec. Recommended for Linux guests.

    .. describe:: stac9221(4)

            A STAC9221 HDA codec.

    """

    __uuid__ = "7b406301-f520-420c-9805-8ce11c086370"
    _enums = [
        ("Null", 0, """@c null value. Never used by the API."""),
        ("SB16", 1, """SB16; this is the only option for the SB16 device."""),
        ("STAC9700", 2, """A STAC9700 AC'97 codec."""),
        ("AD1980", 3, """An AD1980 AC'97 codec. Recommended for Linux guests."""),
        ("STAC9221", 4, """A STAC9221 HDA codec."""),
    ]


class AuthType(Enum):
    """VirtualBox authentication type.


    .. describe:: null(0)

            Null value, also means "no authentication".

    .. describe:: external(1)



    .. describe:: guest(2)



    """

    __uuid__ = "7eef6ef6-98c2-4dc2-ab35-10d2b292028d"
    _enums = [
        ("Null", 0, """Null value, also means "no authentication"."""),
        ("External", 1, """"""),
        ("Guest", 2, """"""),
    ]


class Reason(Enum):
    """Internal event reason type.


    .. describe:: unspecified(0)

            Null value, means "no known reason".

    .. describe:: host_suspend(1)

            Host is being suspended (power management event).

    .. describe:: host_resume(2)

            Host is being resumed (power management event).

    .. describe:: host_battery_low(3)

            Host is running low on battery (power management event).

    .. describe:: snapshot(4)

            A snapshot of the VM is being taken.

    """

    __uuid__ = "e7e8e097-299d-4e98-8bbc-c31c2d47d0cc"
    _enums = [
        ("Unspecified", 0, """Null value, means "no known reason"."""),
        ("HostSuspend", 1, """Host is being suspended (power management event)."""),
        ("HostResume", 2, """Host is being resumed (power management event)."""),
        (
            "HostBatteryLow",
            3,
            """Host is running low on battery (power management event).""",
        ),
        ("Snapshot", 4, """A snapshot of the VM is being taken."""),
    ]


class StorageBus(Enum):
    """The bus type of the storage controller (IDE, SATA, SCSI, SAS or Floppy);
    see :py:func:`IStorageController.bus` .


    .. describe:: null(0)

            @c null value. Never used by the API.

    .. describe:: ide(1)



    .. describe:: sata(2)



    .. describe:: scsi(3)



    .. describe:: floppy(4)



    .. describe:: sas(5)



    .. describe:: usb(6)



    .. describe:: pc_ie(7)



    .. describe:: virtio_scsi(8)



    """

    __uuid__ = "f9510869-7d07-46ba-96a6-6728fbf4adee"
    _enums = [
        ("Null", 0, """@c null value. Never used by the API."""),
        ("IDE", 1, """"""),
        ("SATA", 2, """"""),
        ("SCSI", 3, """"""),
        ("Floppy", 4, """"""),
        ("SAS", 5, """"""),
        ("USB", 6, """"""),
        ("PCIe", 7, """"""),
        ("VirtioSCSI", 8, """"""),
    ]


class StorageControllerType(Enum):
    """The exact variant of storage controller hardware presented
    to the guest; see :py:func:`IStorageController.controller_type` .


    .. describe:: null(0)

            @c null value. Never used by the API.

    .. describe:: lsi_logic(1)

            A SCSI controller of the LsiLogic variant.

    .. describe:: bus_logic(2)

            A SCSI controller of the BusLogic variant.

    .. describe:: intel_ahci(3)

            An Intel AHCI SATA controller; this is the only variant for SATA.

    .. describe:: piix3(4)

            An IDE controller of the PIIX3 variant.

    .. describe:: piix4(5)

            An IDE controller of the PIIX4 variant.

    .. describe:: ich6(6)

            An IDE controller of the ICH6 variant.

    .. describe:: i82078(7)

            A floppy disk controller; this is the only variant for floppy drives.

    .. describe:: lsi_logic_sas(8)

            A variant of the LsiLogic controller using SAS.

    .. describe:: usb(9)

            Special USB based storage controller.

    .. describe:: nv_me(10)

            An NVMe storage controller.

    .. describe:: virtio_scsi(11)

            Virtio SCSI storage controller.

    """

    __uuid__ = "a77d457d-66a3-4368-b24c-293d0f562a9f"
    _enums = [
        ("Null", 0, """@c null value. Never used by the API."""),
        ("LsiLogic", 1, """A SCSI controller of the LsiLogic variant."""),
        ("BusLogic", 2, """A SCSI controller of the BusLogic variant."""),
        (
            "IntelAhci",
            3,
            """An Intel AHCI SATA controller; this is the only variant for SATA.""",
        ),
        ("PIIX3", 4, """An IDE controller of the PIIX3 variant."""),
        ("PIIX4", 5, """An IDE controller of the PIIX4 variant."""),
        ("ICH6", 6, """An IDE controller of the ICH6 variant."""),
        (
            "I82078",
            7,
            """A floppy disk controller; this is the only variant for floppy drives.""",
        ),
        ("LsiLogicSas", 8, """A variant of the LsiLogic controller using SAS."""),
        ("USB", 9, """Special USB based storage controller."""),
        ("NVMe", 10, """An NVMe storage controller."""),
        ("VirtioSCSI", 11, """Virtio SCSI storage controller."""),
    ]


class ChipsetType(Enum):
    """Type of emulated chipset (mostly southbridge).


    .. describe:: null(0)

            @c null value. Never used by the API.

    .. describe:: piix3(1)

            A PIIX3 (PCI IDE ISA Xcelerator) chipset.

    .. describe:: ich9(2)

            A ICH9 (I/O Controller Hub) chipset.

    """

    __uuid__ = "8b4096a8-a7c3-4d3b-bbb1-05a0a51ec394"
    _enums = [
        ("Null", 0, """@c null value. Never used by the API."""),
        ("PIIX3", 1, """A PIIX3 (PCI IDE ISA Xcelerator) chipset."""),
        ("ICH9", 2, """A ICH9 (I/O Controller Hub) chipset."""),
    ]


class NATAliasMode(Enum):
    """


    .. describe:: alias_log(1)



    .. describe:: alias_proxy_only(2)



    .. describe:: alias_use_same_ports(4)



    """

    __uuid__ = "67772168-50d9-11df-9669-7fb714ee4fa1"
    _enums = [
        ("AliasLog", 1, """"""),
        ("AliasProxyOnly", 2, """"""),
        ("AliasUseSamePorts", 4, """"""),
    ]


class NATProtocol(Enum):
    """Protocol definitions used with NAT port-forwarding rules.


    .. describe:: udp(0)

            Port-forwarding uses UDP protocol.

    .. describe:: tcp(1)

            Port-forwarding uses TCP protocol.

    """

    __uuid__ = "e90164be-eb03-11de-94af-fff9b1c1b19f"
    _enums = [
        ("UDP", 0, """Port-forwarding uses UDP protocol."""),
        ("TCP", 1, """Port-forwarding uses TCP protocol."""),
    ]


class BandwidthGroupType(Enum):
    """Type of a bandwidth control group.


    .. describe:: null(0)

            Null type, must be first.

    .. describe:: disk(1)

            The bandwidth group controls disk I/O.

    .. describe:: network(2)

            The bandwidth group controls network I/O.

    """

    __uuid__ = "1d92b67d-dc69-4be9-ad4c-93a01e1e0c8e"
    _enums = [
        ("Null", 0, """Null type, must be first."""),
        ("Disk", 1, """The bandwidth group controls disk I/O."""),
        ("Network", 2, """The bandwidth group controls network I/O."""),
    ]


class VBoxEventType(Enum):
    """Type of an event.
    See :py:class:`IEvent`  for an introduction to VirtualBox event handling.


    .. describe:: invalid(0)

            Invalid event, must be first.

    .. describe:: any_p(1)

            Wildcard for all events.
            Events of this type are never delivered, and only used in
            :py:func:`IEventSource.register_listener`  call to simplify registration.

    .. describe:: vetoable(2)

            Wildcard for all vetoable events. Events of this type are never delivered, and only
            used in :py:func:`IEventSource.register_listener`  call to simplify registration.

    .. describe:: machine_event(3)

            Wildcard for all machine events. Events of this type are never delivered, and only used in
            :py:func:`IEventSource.register_listener`  call to simplify registration.

    .. describe:: snapshot_event(4)

            Wildcard for all snapshot events. Events of this type are never delivered, and only used in
            :py:func:`IEventSource.register_listener`  call to simplify registration.

    .. describe:: input_event(5)

            Wildcard for all input device (keyboard, mouse) events.
            Events of this type are never delivered, and only used in
            :py:func:`IEventSource.register_listener`  call to simplify registration.

    .. describe:: last_wildcard(31)

            Last wildcard.

    .. describe:: on_machine_state_changed(32)

            See :py:class:`IMachineStateChangedEvent` IMachineStateChangedEvent.

    .. describe:: on_machine_data_changed(33)

            See :py:class:`IMachineDataChangedEvent` IMachineDataChangedEvent.

    .. describe:: on_extra_data_changed(34)

            See :py:class:`IExtraDataChangedEvent` IExtraDataChangedEvent.

    .. describe:: on_extra_data_can_change(35)

            See :py:class:`IExtraDataCanChangeEvent` IExtraDataCanChangeEvent.

    .. describe:: on_medium_registered(36)

            See :py:class:`IMediumRegisteredEvent` IMediumRegisteredEvent.

    .. describe:: on_machine_registered(37)

            See :py:class:`IMachineRegisteredEvent` IMachineRegisteredEvent.

    .. describe:: on_session_state_changed(38)

            See :py:class:`ISessionStateChangedEvent` ISessionStateChangedEvent.

    .. describe:: on_snapshot_taken(39)

            See :py:class:`ISnapshotTakenEvent` ISnapshotTakenEvent.

    .. describe:: on_snapshot_deleted(40)

            See :py:class:`ISnapshotDeletedEvent` ISnapshotDeletedEvent.

    .. describe:: on_snapshot_changed(41)

            See :py:class:`ISnapshotChangedEvent` ISnapshotChangedEvent.

    .. describe:: on_guest_property_changed(42)

            See :py:class:`IGuestPropertyChangedEvent` IGuestPropertyChangedEvent.

    .. describe:: on_mouse_pointer_shape_changed(43)

            See :py:class:`IMousePointerShapeChangedEvent` IMousePointerShapeChangedEvent.

    .. describe:: on_mouse_capability_changed(44)

            See :py:class:`IMouseCapabilityChangedEvent` IMouseCapabilityChangedEvent.

    .. describe:: on_keyboard_leds_changed(45)

            See :py:class:`IKeyboardLedsChangedEvent` IKeyboardLedsChangedEvent.

    .. describe:: on_state_changed(46)

            See :py:class:`IStateChangedEvent` IStateChangedEvent.

    .. describe:: on_additions_state_changed(47)

            See :py:class:`IAdditionsStateChangedEvent` IAdditionsStateChangedEvent.

    .. describe:: on_network_adapter_changed(48)

            See :py:class:`INetworkAdapterChangedEvent` INetworkAdapterChangedEvent.

    .. describe:: on_serial_port_changed(49)

            See :py:class:`ISerialPortChangedEvent` ISerialPortChangedEvent.

    .. describe:: on_parallel_port_changed(50)

            See :py:class:`IParallelPortChangedEvent` IParallelPortChangedEvent.

    .. describe:: on_storage_controller_changed(51)

            See :py:class:`IStorageControllerChangedEvent` IStorageControllerChangedEvent.

    .. describe:: on_medium_changed(52)

            See :py:class:`IMediumChangedEvent` IMediumChangedEvent.

    .. describe:: on_vrde_server_changed(53)

            See :py:class:`IVRDEServerChangedEvent` IVRDEServerChangedEvent.

    .. describe:: on_usb_controller_changed(54)

            See :py:class:`IUSBControllerChangedEvent` IUSBControllerChangedEvent.

    .. describe:: on_usb_device_state_changed(55)

            See :py:class:`IUSBDeviceStateChangedEvent` IUSBDeviceStateChangedEvent.

    .. describe:: on_shared_folder_changed(56)

            See :py:class:`ISharedFolderChangedEvent` ISharedFolderChangedEvent.

    .. describe:: on_runtime_error(57)

            See :py:class:`IRuntimeErrorEvent` IRuntimeErrorEvent.

    .. describe:: on_can_show_window(58)

            See :py:class:`ICanShowWindowEvent` ICanShowWindowEvent.

    .. describe:: on_show_window(59)

            See :py:class:`IShowWindowEvent` IShowWindowEvent.

    .. describe:: on_cpu_changed(60)

            See :py:class:`ICPUChangedEvent` ICPUChangedEvent.

    .. describe:: on_vrde_server_info_changed(61)

            See :py:class:`IVRDEServerInfoChangedEvent` IVRDEServerInfoChangedEvent.

    .. describe:: on_event_source_changed(62)

            See :py:class:`IEventSourceChangedEvent` IEventSourceChangedEvent.

    .. describe:: on_cpu_execution_cap_changed(63)

            See :py:class:`ICPUExecutionCapChangedEvent` ICPUExecutionCapChangedEvent.

    .. describe:: on_guest_keyboard(64)

            See :py:class:`IGuestKeyboardEvent` IGuestKeyboardEvent.

    .. describe:: on_guest_mouse(65)

            See :py:class:`IGuestMouseEvent` IGuestMouseEvent.

    .. describe:: on_nat_redirect(66)

            See :py:class:`INATRedirectEvent` INATRedirectEvent.

    .. describe:: on_host_pci_device_plug(67)

            See :py:class:`IHostPCIDevicePlugEvent` IHostPCIDevicePlugEvent.

    .. describe:: on_v_box_svc_availability_changed(68)

            See :py:class:`IVBoxSVCAvailabilityChangedEvent` IVBoxSVCAvailablityChangedEvent.

    .. describe:: on_bandwidth_group_changed(69)

            See :py:class:`IBandwidthGroupChangedEvent` IBandwidthGroupChangedEvent.

    .. describe:: on_guest_monitor_changed(70)

            See :py:class:`IGuestMonitorChangedEvent` IGuestMonitorChangedEvent.

    .. describe:: on_storage_device_changed(71)

            See :py:class:`IStorageDeviceChangedEvent` IStorageDeviceChangedEvent.

    .. describe:: on_clipboard_mode_changed(72)

            See :py:class:`IClipboardModeChangedEvent` IClipboardModeChangedEvent.

    .. describe:: on_dn_d_mode_changed(73)

            See :py:class:`IDnDModeChangedEvent` IDnDModeChangedEvent.

    .. describe:: on_nat_network_changed(74)

            See :py:class:`INATNetworkChangedEvent` INATNetworkChangedEvent.

    .. describe:: on_nat_network_start_stop(75)

            See :py:class:`INATNetworkStartStopEvent` INATNetworkStartStopEvent.

    .. describe:: on_nat_network_alter(76)

            See :py:class:`INATNetworkAlterEvent` INATNetworkAlterEvent.

    .. describe:: on_nat_network_creation_deletion(77)

            See :py:class:`INATNetworkCreationDeletionEvent` INATNetworkCreationDeletionEvent.

    .. describe:: on_nat_network_setting(78)

            See :py:class:`INATNetworkSettingEvent` INATNetworkSettingEvent.

    .. describe:: on_nat_network_port_forward(79)

            See :py:class:`INATNetworkPortForwardEvent` INATNetworkPortForwardEvent.

    .. describe:: on_guest_session_state_changed(80)

            See :py:class:`IGuestSessionStateChangedEvent` IGuestSessionStateChangedEvent.

    .. describe:: on_guest_session_registered(81)

            See :py:class:`IGuestSessionRegisteredEvent` IGuestSessionRegisteredEvent.

    .. describe:: on_guest_process_registered(82)

            See :py:class:`IGuestProcessRegisteredEvent` IGuestProcessRegisteredEvent.

    .. describe:: on_guest_process_state_changed(83)

            See :py:class:`IGuestProcessStateChangedEvent` IGuestProcessStateChangedEvent.

    .. describe:: on_guest_process_input_notify(84)

            See :py:class:`IGuestProcessInputNotifyEvent` IGuestProcessInputNotifyEvent.

    .. describe:: on_guest_process_output(85)

            See :py:class:`IGuestProcessOutputEvent` IGuestProcessOutputEvent.

    .. describe:: on_guest_file_registered(86)

            See :py:class:`IGuestFileRegisteredEvent` IGuestFileRegisteredEvent.

    .. describe:: on_guest_file_state_changed(87)

            See :py:class:`IGuestFileStateChangedEvent` IGuestFileStateChangedEvent.

    .. describe:: on_guest_file_offset_changed(88)

            See :py:class:`IGuestFileOffsetChangedEvent` IGuestFileOffsetChangedEvent.

    .. describe:: on_guest_file_read(89)

            See :py:class:`IGuestFileReadEvent` IGuestFileReadEvent.

            For performance reasons this is a separate event to
            not unnecessarily overflow the event queue.

    .. describe:: on_guest_file_write(90)

            See :py:class:`IGuestFileWriteEvent` IGuestFileWriteEvent.

            For performance reasons this is a separate event to
            not unnecessarily overflow the event queue.

    .. describe:: on_recording_changed(91)

            See :py:class:`IRecordingChangedEvent` IRecordingChangeEvent.

    .. describe:: on_guest_user_state_changed(92)

            See :py:class:`IGuestUserStateChangedEvent` IGuestUserStateChangedEvent.

    .. describe:: on_guest_multi_touch(93)

            See :py:class:`IGuestMouseEvent` IGuestMouseEvent.

    .. describe:: on_host_name_resolution_configuration_change(94)

            See :py:class:`IHostNameResolutionConfigurationChangeEvent` IHostNameResolutionConfigurationChangeEvent.

    .. describe:: on_snapshot_restored(95)

            See :py:class:`ISnapshotRestoredEvent` ISnapshotRestoredEvent.

    .. describe:: on_medium_config_changed(96)

            See :py:class:`IMediumConfigChangedEvent` IMediumConfigChangedEvent.

    .. describe:: on_audio_adapter_changed(97)

            See :py:class:`IAudioAdapterChangedEvent` IAudioAdapterChangedEvent.

    .. describe:: on_progress_percentage_changed(98)

            See :py:class:`IProgressPercentageChangedEvent` IProgressPercentageChangedEvent.

    .. describe:: on_progress_task_completed(99)

            See :py:class:`IProgressTaskCompletedEvent` IProgressTaskCompletedEvent.

    .. describe:: on_cursor_position_changed(100)

            See :py:class:`ICursorPositionChangedEvent` ICursorPositionChangedEvent.

    .. describe:: on_guest_additions_status_changed(101)

            See :py:class:`IGuestAdditionsStatusChangedEvent` IGuestAdditionsStatusChangedEvent.

    .. describe:: on_guest_monitor_info_changed(102)

            See :py:class:`IGuestMonitorInfoChangedEvent` IGuestMonitorInfoChangedEvent.

    .. describe:: on_guest_file_size_changed(103)

            See :py:class:`IGuestFileSizeChangedEvent` IGuestFileSizeChangedEvent.

    .. describe:: on_clipboard_file_transfer_mode_changed(104)

            See :py:class:`IClipboardFileTransferModeChangedEvent` IClipboardFileTransferModeChangedEvent.

    .. describe:: on_cloud_provider_list_changed(105)

            See :py:class:`ICloudProviderListChangedEvent` .

    .. describe:: on_cloud_provider_registered(106)

            See :py:class:`ICloudProviderRegisteredEvent` .

    .. describe:: on_cloud_provider_uninstall(107)

            See :py:class:`ICloudProviderUninstallEvent` .

    .. describe:: on_cloud_profile_registered(108)

            See :py:class:`ICloudProfileRegisteredEvent` .

    .. describe:: on_cloud_profile_changed(109)

            See :py:class:`ICloudProfileChangedEvent` .

    .. describe:: last(110)

            Must be last event, used for iterations and structures relying on numerical event values.

    """

    __uuid__ = "2ab7c196-f232-11ea-a5d2-83b96bc30bcc"
    _enums = [
        ("Invalid", 0, """Invalid event, must be first."""),
        (
            "Any",
            1,
            """Wildcard for all events.
            Events of this type are never delivered, and only used in
            :py:func:`IEventSource.register_listener`  call to simplify registration.""",
        ),
        (
            "Vetoable",
            2,
            """Wildcard for all vetoable events. Events of this type are never delivered, and only
            used in :py:func:`IEventSource.register_listener`  call to simplify registration.""",
        ),
        (
            "MachineEvent",
            3,
            """Wildcard for all machine events. Events of this type are never delivered, and only used in
            :py:func:`IEventSource.register_listener`  call to simplify registration.""",
        ),
        (
            "SnapshotEvent",
            4,
            """Wildcard for all snapshot events. Events of this type are never delivered, and only used in
            :py:func:`IEventSource.register_listener`  call to simplify registration.""",
        ),
        (
            "InputEvent",
            5,
            """Wildcard for all input device (keyboard, mouse) events.
            Events of this type are never delivered, and only used in
            :py:func:`IEventSource.register_listener`  call to simplify registration.""",
        ),
        ("LastWildcard", 31, """Last wildcard."""),
        (
            "OnMachineStateChanged",
            32,
            """See :py:class:`IMachineStateChangedEvent` IMachineStateChangedEvent.""",
        ),
        (
            "OnMachineDataChanged",
            33,
            """See :py:class:`IMachineDataChangedEvent` IMachineDataChangedEvent.""",
        ),
        (
            "OnExtraDataChanged",
            34,
            """See :py:class:`IExtraDataChangedEvent` IExtraDataChangedEvent.""",
        ),
        (
            "OnExtraDataCanChange",
            35,
            """See :py:class:`IExtraDataCanChangeEvent` IExtraDataCanChangeEvent.""",
        ),
        (
            "OnMediumRegistered",
            36,
            """See :py:class:`IMediumRegisteredEvent` IMediumRegisteredEvent.""",
        ),
        (
            "OnMachineRegistered",
            37,
            """See :py:class:`IMachineRegisteredEvent` IMachineRegisteredEvent.""",
        ),
        (
            "OnSessionStateChanged",
            38,
            """See :py:class:`ISessionStateChangedEvent` ISessionStateChangedEvent.""",
        ),
        (
            "OnSnapshotTaken",
            39,
            """See :py:class:`ISnapshotTakenEvent` ISnapshotTakenEvent.""",
        ),
        (
            "OnSnapshotDeleted",
            40,
            """See :py:class:`ISnapshotDeletedEvent` ISnapshotDeletedEvent.""",
        ),
        (
            "OnSnapshotChanged",
            41,
            """See :py:class:`ISnapshotChangedEvent` ISnapshotChangedEvent.""",
        ),
        (
            "OnGuestPropertyChanged",
            42,
            """See :py:class:`IGuestPropertyChangedEvent` IGuestPropertyChangedEvent.""",
        ),
        (
            "OnMousePointerShapeChanged",
            43,
            """See :py:class:`IMousePointerShapeChangedEvent` IMousePointerShapeChangedEvent.""",
        ),
        (
            "OnMouseCapabilityChanged",
            44,
            """See :py:class:`IMouseCapabilityChangedEvent` IMouseCapabilityChangedEvent.""",
        ),
        (
            "OnKeyboardLedsChanged",
            45,
            """See :py:class:`IKeyboardLedsChangedEvent` IKeyboardLedsChangedEvent.""",
        ),
        (
            "OnStateChanged",
            46,
            """See :py:class:`IStateChangedEvent` IStateChangedEvent.""",
        ),
        (
            "OnAdditionsStateChanged",
            47,
            """See :py:class:`IAdditionsStateChangedEvent` IAdditionsStateChangedEvent.""",
        ),
        (
            "OnNetworkAdapterChanged",
            48,
            """See :py:class:`INetworkAdapterChangedEvent` INetworkAdapterChangedEvent.""",
        ),
        (
            "OnSerialPortChanged",
            49,
            """See :py:class:`ISerialPortChangedEvent` ISerialPortChangedEvent.""",
        ),
        (
            "OnParallelPortChanged",
            50,
            """See :py:class:`IParallelPortChangedEvent` IParallelPortChangedEvent.""",
        ),
        (
            "OnStorageControllerChanged",
            51,
            """See :py:class:`IStorageControllerChangedEvent` IStorageControllerChangedEvent.""",
        ),
        (
            "OnMediumChanged",
            52,
            """See :py:class:`IMediumChangedEvent` IMediumChangedEvent.""",
        ),
        (
            "OnVRDEServerChanged",
            53,
            """See :py:class:`IVRDEServerChangedEvent` IVRDEServerChangedEvent.""",
        ),
        (
            "OnUSBControllerChanged",
            54,
            """See :py:class:`IUSBControllerChangedEvent` IUSBControllerChangedEvent.""",
        ),
        (
            "OnUSBDeviceStateChanged",
            55,
            """See :py:class:`IUSBDeviceStateChangedEvent` IUSBDeviceStateChangedEvent.""",
        ),
        (
            "OnSharedFolderChanged",
            56,
            """See :py:class:`ISharedFolderChangedEvent` ISharedFolderChangedEvent.""",
        ),
        (
            "OnRuntimeError",
            57,
            """See :py:class:`IRuntimeErrorEvent` IRuntimeErrorEvent.""",
        ),
        (
            "OnCanShowWindow",
            58,
            """See :py:class:`ICanShowWindowEvent` ICanShowWindowEvent.""",
        ),
        ("OnShowWindow", 59, """See :py:class:`IShowWindowEvent` IShowWindowEvent."""),
        ("OnCPUChanged", 60, """See :py:class:`ICPUChangedEvent` ICPUChangedEvent."""),
        (
            "OnVRDEServerInfoChanged",
            61,
            """See :py:class:`IVRDEServerInfoChangedEvent` IVRDEServerInfoChangedEvent.""",
        ),
        (
            "OnEventSourceChanged",
            62,
            """See :py:class:`IEventSourceChangedEvent` IEventSourceChangedEvent.""",
        ),
        (
            "OnCPUExecutionCapChanged",
            63,
            """See :py:class:`ICPUExecutionCapChangedEvent` ICPUExecutionCapChangedEvent.""",
        ),
        (
            "OnGuestKeyboard",
            64,
            """See :py:class:`IGuestKeyboardEvent` IGuestKeyboardEvent.""",
        ),
        ("OnGuestMouse", 65, """See :py:class:`IGuestMouseEvent` IGuestMouseEvent."""),
        (
            "OnNATRedirect",
            66,
            """See :py:class:`INATRedirectEvent` INATRedirectEvent.""",
        ),
        (
            "OnHostPCIDevicePlug",
            67,
            """See :py:class:`IHostPCIDevicePlugEvent` IHostPCIDevicePlugEvent.""",
        ),
        (
            "OnVBoxSVCAvailabilityChanged",
            68,
            """See :py:class:`IVBoxSVCAvailabilityChangedEvent` IVBoxSVCAvailablityChangedEvent.""",
        ),
        (
            "OnBandwidthGroupChanged",
            69,
            """See :py:class:`IBandwidthGroupChangedEvent` IBandwidthGroupChangedEvent.""",
        ),
        (
            "OnGuestMonitorChanged",
            70,
            """See :py:class:`IGuestMonitorChangedEvent` IGuestMonitorChangedEvent.""",
        ),
        (
            "OnStorageDeviceChanged",
            71,
            """See :py:class:`IStorageDeviceChangedEvent` IStorageDeviceChangedEvent.""",
        ),
        (
            "OnClipboardModeChanged",
            72,
            """See :py:class:`IClipboardModeChangedEvent` IClipboardModeChangedEvent.""",
        ),
        (
            "OnDnDModeChanged",
            73,
            """See :py:class:`IDnDModeChangedEvent` IDnDModeChangedEvent.""",
        ),
        (
            "OnNATNetworkChanged",
            74,
            """See :py:class:`INATNetworkChangedEvent` INATNetworkChangedEvent.""",
        ),
        (
            "OnNATNetworkStartStop",
            75,
            """See :py:class:`INATNetworkStartStopEvent` INATNetworkStartStopEvent.""",
        ),
        (
            "OnNATNetworkAlter",
            76,
            """See :py:class:`INATNetworkAlterEvent` INATNetworkAlterEvent.""",
        ),
        (
            "OnNATNetworkCreationDeletion",
            77,
            """See :py:class:`INATNetworkCreationDeletionEvent` INATNetworkCreationDeletionEvent.""",
        ),
        (
            "OnNATNetworkSetting",
            78,
            """See :py:class:`INATNetworkSettingEvent` INATNetworkSettingEvent.""",
        ),
        (
            "OnNATNetworkPortForward",
            79,
            """See :py:class:`INATNetworkPortForwardEvent` INATNetworkPortForwardEvent.""",
        ),
        (
            "OnGuestSessionStateChanged",
            80,
            """See :py:class:`IGuestSessionStateChangedEvent` IGuestSessionStateChangedEvent.""",
        ),
        (
            "OnGuestSessionRegistered",
            81,
            """See :py:class:`IGuestSessionRegisteredEvent` IGuestSessionRegisteredEvent.""",
        ),
        (
            "OnGuestProcessRegistered",
            82,
            """See :py:class:`IGuestProcessRegisteredEvent` IGuestProcessRegisteredEvent.""",
        ),
        (
            "OnGuestProcessStateChanged",
            83,
            """See :py:class:`IGuestProcessStateChangedEvent` IGuestProcessStateChangedEvent.""",
        ),
        (
            "OnGuestProcessInputNotify",
            84,
            """See :py:class:`IGuestProcessInputNotifyEvent` IGuestProcessInputNotifyEvent.""",
        ),
        (
            "OnGuestProcessOutput",
            85,
            """See :py:class:`IGuestProcessOutputEvent` IGuestProcessOutputEvent.""",
        ),
        (
            "OnGuestFileRegistered",
            86,
            """See :py:class:`IGuestFileRegisteredEvent` IGuestFileRegisteredEvent.""",
        ),
        (
            "OnGuestFileStateChanged",
            87,
            """See :py:class:`IGuestFileStateChangedEvent` IGuestFileStateChangedEvent.""",
        ),
        (
            "OnGuestFileOffsetChanged",
            88,
            """See :py:class:`IGuestFileOffsetChangedEvent` IGuestFileOffsetChangedEvent.""",
        ),
        (
            "OnGuestFileRead",
            89,
            """See :py:class:`IGuestFileReadEvent` IGuestFileReadEvent.
            
            For performance reasons this is a separate event to
            not unnecessarily overflow the event queue.""",
        ),
        (
            "OnGuestFileWrite",
            90,
            """See :py:class:`IGuestFileWriteEvent` IGuestFileWriteEvent.
            
            For performance reasons this is a separate event to
            not unnecessarily overflow the event queue.""",
        ),
        (
            "OnRecordingChanged",
            91,
            """See :py:class:`IRecordingChangedEvent` IRecordingChangeEvent.""",
        ),
        (
            "OnGuestUserStateChanged",
            92,
            """See :py:class:`IGuestUserStateChangedEvent` IGuestUserStateChangedEvent.""",
        ),
        (
            "OnGuestMultiTouch",
            93,
            """See :py:class:`IGuestMouseEvent` IGuestMouseEvent.""",
        ),
        (
            "OnHostNameResolutionConfigurationChange",
            94,
            """See :py:class:`IHostNameResolutionConfigurationChangeEvent` IHostNameResolutionConfigurationChangeEvent.""",
        ),
        (
            "OnSnapshotRestored",
            95,
            """See :py:class:`ISnapshotRestoredEvent` ISnapshotRestoredEvent.""",
        ),
        (
            "OnMediumConfigChanged",
            96,
            """See :py:class:`IMediumConfigChangedEvent` IMediumConfigChangedEvent.""",
        ),
        (
            "OnAudioAdapterChanged",
            97,
            """See :py:class:`IAudioAdapterChangedEvent` IAudioAdapterChangedEvent.""",
        ),
        (
            "OnProgressPercentageChanged",
            98,
            """See :py:class:`IProgressPercentageChangedEvent` IProgressPercentageChangedEvent.""",
        ),
        (
            "OnProgressTaskCompleted",
            99,
            """See :py:class:`IProgressTaskCompletedEvent` IProgressTaskCompletedEvent.""",
        ),
        (
            "OnCursorPositionChanged",
            100,
            """See :py:class:`ICursorPositionChangedEvent` ICursorPositionChangedEvent.""",
        ),
        (
            "OnGuestAdditionsStatusChanged",
            101,
            """See :py:class:`IGuestAdditionsStatusChangedEvent` IGuestAdditionsStatusChangedEvent.""",
        ),
        (
            "OnGuestMonitorInfoChanged",
            102,
            """See :py:class:`IGuestMonitorInfoChangedEvent` IGuestMonitorInfoChangedEvent.""",
        ),
        (
            "OnGuestFileSizeChanged",
            103,
            """See :py:class:`IGuestFileSizeChangedEvent` IGuestFileSizeChangedEvent.""",
        ),
        (
            "OnClipboardFileTransferModeChanged",
            104,
            """See :py:class:`IClipboardFileTransferModeChangedEvent` IClipboardFileTransferModeChangedEvent.""",
        ),
        (
            "OnCloudProviderListChanged",
            105,
            """See :py:class:`ICloudProviderListChangedEvent` .""",
        ),
        (
            "OnCloudProviderRegistered",
            106,
            """See :py:class:`ICloudProviderRegisteredEvent` .""",
        ),
        (
            "OnCloudProviderUninstall",
            107,
            """See :py:class:`ICloudProviderUninstallEvent` .""",
        ),
        (
            "OnCloudProfileRegistered",
            108,
            """See :py:class:`ICloudProfileRegisteredEvent` .""",
        ),
        (
            "OnCloudProfileChanged",
            109,
            """See :py:class:`ICloudProfileChangedEvent` .""",
        ),
        (
            "Last",
            110,
            """Must be last event, used for iterations and structures relying on numerical event values.""",
        ),
    ]


class GuestMouseEventMode(Enum):
    """The mode (relative, absolute, multi-touch) of a pointer event.

    @todo A clear pattern seems to be emerging that we should usually have
    multiple input devices active for different types of reporting, so we
    should really have different event types for relative (including wheel),
    absolute (not including wheel) and multi-touch events.


    .. describe:: relative(0)

            Relative event.

    .. describe:: absolute(1)

            Absolute event.

    """

    __uuid__ = "4b500146-ebba-4b7c-bc29-69c2d57a5caf"
    _enums = [
        ("Relative", 0, """Relative event."""),
        ("Absolute", 1, """Absolute event."""),
    ]


class GuestMonitorChangedEventType(Enum):
    """How the guest monitor has been changed.


    .. describe:: enabled(0)

            The guest monitor has been enabled by the guest.

    .. describe:: disabled(1)

            The guest monitor has been disabled by the guest.

    .. describe:: new_origin(2)

            The guest monitor origin has changed in the guest.

    """

    __uuid__ = "ef172985-7e36-4297-95be-e46396968d66"
    _enums = [
        ("Enabled", 0, """The guest monitor has been enabled by the guest."""),
        ("Disabled", 1, """The guest monitor has been disabled by the guest."""),
        ("NewOrigin", 2, """The guest monitor origin has changed in the guest."""),
    ]


class FormValueType(Enum):
    """


    .. describe:: boolean(0)



    .. describe:: string(1)



    .. describe:: choice(2)



    .. describe:: ranged_integer(3)



    """

    __uuid__ = "43d794a0-7c98-11e9-a346-a36d5fa858a5"
    _enums = [
        ("Boolean", 0, """"""),
        ("String", 1, """"""),
        ("Choice", 2, """"""),
        ("RangedInteger", 3, """"""),
    ]


class CloudMachineState(Enum):
    """Cloud instance execution state


    .. describe:: invalid(0)

            Invalid state

    .. describe:: provisioning(1)

            The machine is in the process of provisioning

    .. describe:: running(2)

            The machine runs

    .. describe:: starting(3)

            The machine is in the process of starting

    .. describe:: stopping(4)

            The machine is in the process of stopping

    .. describe:: stopped(5)

            The machine was stopped

    .. describe:: creating_image(6)

            The machine is in the process of creating image

    .. describe:: terminating(7)

            The machine is in the process of terminating

    .. describe:: terminated(8)

            The machine was terminated

    """

    __uuid__ = "67b6d054-0154-4f5d-b71b-6ac406e1ff78"
    _enums = [
        ("Invalid", 0, """Invalid state"""),
        ("Provisioning", 1, """The machine is in the process of provisioning"""),
        ("Running", 2, """The machine runs"""),
        ("Starting", 3, """The machine is in the process of starting"""),
        ("Stopping", 4, """The machine is in the process of stopping"""),
        ("Stopped", 5, """The machine was stopped"""),
        ("CreatingImage", 6, """The machine is in the process of creating image"""),
        ("Terminating", 7, """The machine is in the process of terminating"""),
        ("Terminated", 8, """The machine was terminated"""),
    ]


class CloudImageState(Enum):
    """Cloud image state


    .. describe:: invalid(0)

            Invalid state

    .. describe:: provisioning(1)

            The image is in the process of provisioning

    .. describe:: importing(2)

            The image is in the process of importing

    .. describe:: available(3)

            The image is avalable

    .. describe:: exporting(4)

            The image is in the process of exporting

    .. describe:: disabled(5)

            The image is disabled

    .. describe:: deleted(6)

            The image was deleted

    """

    __uuid__ = "6e5d6762-eea2-4f2c-b104-2952d0aa8a0a"
    _enums = [
        ("Invalid", 0, """Invalid state"""),
        ("Provisioning", 1, """The image is in the process of provisioning"""),
        ("Importing", 2, """The image is in the process of importing"""),
        ("Available", 3, """The image is avalable"""),
        ("Exporting", 4, """The image is in the process of exporting"""),
        ("Disabled", 5, """The image is disabled"""),
        ("Deleted", 6, """The image was deleted"""),
    ]


class IVirtualBoxErrorInfo(Interface):
    """
    The IVirtualBoxErrorInfo interface represents extended error information.

    Extended error information can be set by VirtualBox components after
    unsuccessful or partially successful method invocation. This information
    can be retrieved by the calling party as an IVirtualBoxErrorInfo object
    and then shown to the client in addition to the plain 32-bit result code.

    In MS COM, this interface extends the IErrorInfo interface,
    in XPCOM, it extends the nsIException interface. In both cases,
    it provides a set of common attributes to retrieve error
    information.

    Sometimes invocation of some component's method may involve methods of
    other components that may also fail (independently of this method's
    failure), or a series of non-fatal errors may precede a fatal error that
    causes method failure. In cases like that, it may be desirable to preserve
    information about all errors happened during method invocation and deliver
    it to the caller. The :py:func:`next_p`  attribute is intended
    specifically for this purpose and allows to represent a chain of errors
    through a single IVirtualBoxErrorInfo object set after method invocation.

    errors are stored to a chain in the reverse order, i.e. the
    initial error object you query right after method invocation is the last
    error set by the callee, the object it points to in the @a next attribute
    is the previous error and so on, up to the first error (which is the last
    in the chain).
    """

    __uuid__ = "c1bcc6d5-7966-481d-ab0b-d0ed73e28135"
    __wsmap__ = "managed"

    @property
    def result_code(self):
        """Get int value for 'resultCode'
        Result code of the error.
        Usually, it will be the same as the result code returned
        by the method that provided this error information, but not
        always. For example, on Win32, CoCreateInstance() will most
        likely return E_NOINTERFACE upon unsuccessful component
        instantiation attempt, but not the value the component factory
        returned. Value is typed 'long', not 'result',
        to make interface usable from scripting languages.

        In MS COM, there is no equivalent.
        In XPCOM, it is the same as nsIException::result.
        """
        ret = self._get_attr("resultCode")
        return ret

    @property
    def result_detail(self):
        """Get int value for 'resultDetail'
        Optional result data of this error. This will vary depending on the
        actual error usage. By default this attribute is not being used.
        """
        ret = self._get_attr("resultDetail")
        return ret

    @property
    def interface_id(self):
        """Get str value for 'interfaceID'
        UUID of the interface that defined the error.

        In MS COM, it is the same as IErrorInfo::GetGUID, except for the
        data type.
        In XPCOM, there is no equivalent.
        """
        ret = self._get_attr("interfaceID")
        return ret

    @property
    def component(self):
        """Get str value for 'component'
        Name of the component that generated the error.

        In MS COM, it is the same as IErrorInfo::GetSource.
        In XPCOM, there is no equivalent.
        """
        ret = self._get_attr("component")
        return ret

    @property
    def text(self):
        """Get str value for 'text'
        Text description of the error.

        In MS COM, it is the same as IErrorInfo::GetDescription.
        In XPCOM, it is the same as nsIException::message.
        """
        ret = self._get_attr("text")
        return ret

    @property
    def next_p(self):
        """Get IVirtualBoxErrorInfo value for 'next'
        Next error object if there is any, or @c null otherwise.

        In MS COM, there is no equivalent.
        In XPCOM, it is the same as nsIException::inner.
        """
        ret = self._get_attr("next")
        return IVirtualBoxErrorInfo(ret)


class INATNetwork(Interface):
    """
    TBD: the idea, technically we can start any number of the NAT networks,
    but we should expect that at some point we will get collisions because of
    port-forwanding rules. so perhaps we should support only single instance of NAT
    network.
    """

    __uuid__ = "4fdebbf0-be30-49c0-b315-e9749e1bded1"
    __wsmap__ = "managed"

    @property
    def network_name(self):
        """Get or set str value for 'networkName'
        TBD: the idea, technically we can start any number of the NAT networks,
        but we should expect that at some point we will get collisions because of
        port-forwanding rules. so perhaps we should support only single instance of NAT
        network.
        """
        ret = self._get_attr("networkName")
        return ret

    @network_name.setter
    def network_name(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("networkName", value)

    @property
    def enabled(self):
        """Get or set bool value for 'enabled'"""
        ret = self._get_attr("enabled")
        return ret

    @enabled.setter
    def enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("enabled", value)

    @property
    def network(self):
        """Get or set str value for 'network'
        This is CIDR IPv4 string. Specifying it user defines IPv4 addresses
        of gateway (low address + 1) and DHCP server (= low address + 2).
        Note: If there are defined IPv4 port-forward rules update of network
        will be ignored (because new assignment could break existing rules).
        """
        ret = self._get_attr("network")
        return ret

    @network.setter
    def network(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("network", value)

    @property
    def gateway(self):
        """Get str value for 'gateway'
        This attribute is read-only. It's recalculated on changing
        network attribute (low address of network + 1).
        """
        ret = self._get_attr("gateway")
        return ret

    @property
    def i_pv6_enabled(self):
        """Get or set bool value for 'IPv6Enabled'
        This attribute define whether gateway will support IPv6 or not.
        """
        ret = self._get_attr("IPv6Enabled")
        return ret

    @i_pv6_enabled.setter
    def i_pv6_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("IPv6Enabled", value)

    @property
    def i_pv6_prefix(self):
        """Get or set str value for 'IPv6Prefix'
        This a CIDR IPv6 defining prefix for link-local addresses
        autoconfiguration within network. Note: ignored if attribute
        IPv6Enabled is false.
        """
        ret = self._get_attr("IPv6Prefix")
        return ret

    @i_pv6_prefix.setter
    def i_pv6_prefix(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("IPv6Prefix", value)

    @property
    def advertise_default_i_pv6_route_enabled(self):
        """Get or set bool value for 'advertiseDefaultIPv6RouteEnabled'"""
        ret = self._get_attr("advertiseDefaultIPv6RouteEnabled")
        return ret

    @advertise_default_i_pv6_route_enabled.setter
    def advertise_default_i_pv6_route_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("advertiseDefaultIPv6RouteEnabled", value)

    @property
    def need_dhcp_server(self):
        """Get or set bool value for 'needDhcpServer'"""
        ret = self._get_attr("needDhcpServer")
        return ret

    @need_dhcp_server.setter
    def need_dhcp_server(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("needDhcpServer", value)

    @property
    def event_source(self):
        """Get IEventSource value for 'eventSource'"""
        ret = self._get_attr("eventSource")
        return IEventSource(ret)

    @property
    def port_forward_rules4(self):
        """Get str value for 'portForwardRules4'
        Array of NAT port-forwarding rules in string representation,
        in the following format:
        "name:protocolid:[host ip]:host port:[guest ip]:guest port".
        """
        ret = self._get_attr("portForwardRules4")
        return ret

    @property
    def local_mappings(self):
        """Get str value for 'localMappings'
        Array of mappings (address,offset),e.g. ("127.0.1.1=4") maps 127.0.1.1 to networkid + 4.
        """
        ret = self._get_attr("localMappings")
        return ret

    def add_local_mapping(self, hostid, offset):
        """

        in hostid of type str

        in offset of type int

        """
        if not isinstance(hostid, basestring):
            raise TypeError("hostid can only be an instance of type basestring")
        if not isinstance(offset, baseinteger):
            raise TypeError("offset can only be an instance of type baseinteger")
        self._call("addLocalMapping", in_p=[hostid, offset])

    @property
    def loopback_ip6(self):
        """Get or set int value for 'loopbackIp6'
        Offset in ipv6 network from network id for address mapped into loopback6 interface of the host.
        """
        ret = self._get_attr("loopbackIp6")
        return ret

    @loopback_ip6.setter
    def loopback_ip6(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("loopbackIp6", value)

    @property
    def port_forward_rules6(self):
        """Get str value for 'portForwardRules6'
        Array of NAT port-forwarding rules in string representation, in the
        following format: "name:protocolid:[host ip]:host port:[guest ip]:guest port".
        """
        ret = self._get_attr("portForwardRules6")
        return ret

    def add_port_forward_rule(
        self, is_ipv6, rule_name, proto, host_ip, host_port, guest_ip, guest_port
    ):
        """Protocol handled with the rule.

        in is_ipv6 of type bool

        in rule_name of type str

        in proto of type :class:`NATProtocol`
            Protocol handled with the rule.

        in host_ip of type str
            IP of the host interface to which the rule should apply.
            An empty ip address is acceptable, in which case the NAT engine
            binds the handling socket to any interface.

        in host_port of type int
            The port number to listen on.

        in guest_ip of type str
            The IP address of the guest which the NAT engine will forward
            matching packets to. An empty IP address is not acceptable.

        in guest_port of type int
            The port number to forward.

        """
        if not isinstance(is_ipv6, bool):
            raise TypeError("is_ipv6 can only be an instance of type bool")
        if not isinstance(rule_name, basestring):
            raise TypeError("rule_name can only be an instance of type basestring")
        if not isinstance(proto, NATProtocol):
            raise TypeError("proto can only be an instance of type NATProtocol")
        if not isinstance(host_ip, basestring):
            raise TypeError("host_ip can only be an instance of type basestring")
        if not isinstance(host_port, baseinteger):
            raise TypeError("host_port can only be an instance of type baseinteger")
        if not isinstance(guest_ip, basestring):
            raise TypeError("guest_ip can only be an instance of type basestring")
        if not isinstance(guest_port, baseinteger):
            raise TypeError("guest_port can only be an instance of type baseinteger")
        self._call(
            "addPortForwardRule",
            in_p=[is_ipv6, rule_name, proto, host_ip, host_port, guest_ip, guest_port],
        )

    def remove_port_forward_rule(self, i_sipv6, rule_name):
        """

        in i_sipv6 of type bool

        in rule_name of type str

        """
        if not isinstance(i_sipv6, bool):
            raise TypeError("i_sipv6 can only be an instance of type bool")
        if not isinstance(rule_name, basestring):
            raise TypeError("rule_name can only be an instance of type basestring")
        self._call("removePortForwardRule", in_p=[i_sipv6, rule_name])

    def start(self):
        """"""
        self._call("start")

    def stop(self):
        """"""
        self._call("stop")


class ICloudNetwork(Interface):
    """
    TBD: User-friendly, descriptive name of cloud subnet. For example, domain
    names of subnet and vcn, separated by dot.
    """

    __uuid__ = "d8e3496e-735f-4fde-8a54-427d49409b5f"
    __wsmap__ = "managed"

    @property
    def network_name(self):
        """Get or set str value for 'networkName'
        TBD: User-friendly, descriptive name of cloud subnet. For example, domain
        names of subnet and vcn, separated by dot.
        """
        ret = self._get_attr("networkName")
        return ret

    @network_name.setter
    def network_name(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("networkName", value)

    @property
    def enabled(self):
        """Get or set bool value for 'enabled'"""
        ret = self._get_attr("enabled")
        return ret

    @enabled.setter
    def enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("enabled", value)

    @property
    def provider(self):
        """Get or set str value for 'provider'
        Cloud provider short name.
        """
        ret = self._get_attr("provider")
        return ret

    @provider.setter
    def provider(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("provider", value)

    @property
    def profile(self):
        """Get or set str value for 'profile'
        Cloud profile name.
        """
        ret = self._get_attr("profile")
        return ret

    @profile.setter
    def profile(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("profile", value)

    @property
    def network_id(self):
        """Get or set str value for 'networkId'
        Cloud network id.
        """
        ret = self._get_attr("networkId")
        return ret

    @network_id.setter
    def network_id(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("networkId", value)


class IDHCPServer(Interface):
    """
    The IDHCPServer interface represents the VirtualBox DHCP server configuration.

    To enumerate all the DHCP servers on the host, use the
    :py:func:`IVirtualBox.dhcp_servers`  attribute.
    """

    __uuid__ = "cadef0a2-a1a9-4ac2-8e80-c049af69dac8"
    __wsmap__ = "managed"

    @property
    def event_source(self):
        """Get IEventSource value for 'eventSource'"""
        ret = self._get_attr("eventSource")
        return IEventSource(ret)

    @property
    def enabled(self):
        """Get or set bool value for 'enabled'
        specifies if the DHCP server is enabled
        """
        ret = self._get_attr("enabled")
        return ret

    @enabled.setter
    def enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("enabled", value)

    @property
    def ip_address(self):
        """Get str value for 'IPAddress'
        specifies server IP
        """
        ret = self._get_attr("IPAddress")
        return ret

    @property
    def network_mask(self):
        """Get str value for 'networkMask'
        specifies server network mask
        """
        ret = self._get_attr("networkMask")
        return ret

    @property
    def network_name(self):
        """Get str value for 'networkName'
        specifies internal network name the server is used for
        """
        ret = self._get_attr("networkName")
        return ret

    @property
    def lower_ip(self):
        """Get str value for 'lowerIP'
        specifies from IP address in server address range
        """
        ret = self._get_attr("lowerIP")
        return ret

    @property
    def upper_ip(self):
        """Get str value for 'upperIP'
        specifies to IP address in server address range
        """
        ret = self._get_attr("upperIP")
        return ret

    @property
    def global_config(self):
        """Get IDHCPGlobalConfig value for 'globalConfig'
        Global configuration that applies to all clients.
        """
        ret = self._get_attr("globalConfig")
        return IDHCPGlobalConfig(ret)

    @property
    def group_configs(self):
        """Get IDHCPGroupConfig value for 'groupConfigs'
        Configuration groups that applies to selected clients, selection is flexible.
        """
        ret = self._get_attr("groupConfigs")
        return [IDHCPGroupConfig(a) for a in ret]

    @property
    def individual_configs(self):
        """Get IDHCPIndividualConfig value for 'individualConfigs'
        Individual NIC configurations either by MAC address or VM + NIC number.
        """
        ret = self._get_attr("individualConfigs")
        return [IDHCPIndividualConfig(a) for a in ret]

    def set_configuration(
        self, ip_address, network_mask, from_ip_address, to_ip_address
    ):
        """configures the server

        in ip_address of type str
            server IP address

        in network_mask of type str
            server network mask

        in from_ip_address of type str
            server From IP address for address range

        in to_ip_address of type str
            server To IP address for address range

        raises :class:`OleErrorInvalidarg`
            invalid configuration supplied

        """
        if not isinstance(ip_address, basestring):
            raise TypeError("ip_address can only be an instance of type basestring")
        if not isinstance(network_mask, basestring):
            raise TypeError("network_mask can only be an instance of type basestring")
        if not isinstance(from_ip_address, basestring):
            raise TypeError(
                "from_ip_address can only be an instance of type basestring"
            )
        if not isinstance(to_ip_address, basestring):
            raise TypeError("to_ip_address can only be an instance of type basestring")
        self._call(
            "setConfiguration",
            in_p=[ip_address, network_mask, from_ip_address, to_ip_address],
        )

    def start(self, trunk_name, trunk_type):
        """Starts DHCP server process.

        in trunk_name of type str
            Name of internal network trunk.

        in trunk_type of type str
            Type of internal network trunk.

        raises :class:`OleErrorFail`
            Failed to start the process.

        """
        if not isinstance(trunk_name, basestring):
            raise TypeError("trunk_name can only be an instance of type basestring")
        if not isinstance(trunk_type, basestring):
            raise TypeError("trunk_type can only be an instance of type basestring")
        self._call("start", in_p=[trunk_name, trunk_type])

    def stop(self):
        """Stops DHCP server process.

        raises :class:`OleErrorFail`
            Failed to stop the process.

        """
        self._call("stop")

    def restart(self):
        """Restart running DHCP server process.

        raises :class:`OleErrorFail`
            Failed to restart the process.

        """
        self._call("restart")

    def find_lease_by_mac(self, mac, type_p):
        """Queries the persistent lease database by MAC address.

        This is handy if the host wants to connect to a server running inside
        a VM on a host only network.

        in mac of type str
            The MAC address to look up.

        in type_p of type int
            Reserved, MBZ.

        out address of type str
            The assigned address.

        out state of type str
            The lease state.

        out issued of type int
            Timestamp of when the lease was issued, in seconds since 1970-01-01 UTC.

        out expire of type int
            Timestamp of when the lease expires/expired, in seconds since 1970-01-01 UTC.

        raises :class:`VBoxErrorObjectNotFound`
            If MAC address not in the database.

        raises :class:`VBoxErrorFileError`
            If not able to read the lease database file.

        """
        if not isinstance(mac, basestring):
            raise TypeError("mac can only be an instance of type basestring")
        if not isinstance(type_p, baseinteger):
            raise TypeError("type_p can only be an instance of type baseinteger")
        (address, state, issued, expire) = self._call(
            "findLeaseByMAC", in_p=[mac, type_p]
        )
        return (address, state, issued, expire)

    def get_config(self, scope, name, slot, may_add):
        """Gets or adds a configuration.

        in scope of type :class:`DHCPConfigScope`
            The kind of configuration being sought or added.

        in name of type str
            Meaning depends on the @a scope:
            - Ignored when the @a scope is :py:attr:`DHCPConfigScope.global_p` .
            - A VM name or UUID for :py:attr:`DHCPConfigScope.machine_nic` .
            - A MAC address for :py:attr:`DHCPConfigScope.mac` .
            - A group name for :py:attr:`DHCPConfigScope.group` .

        in slot of type int
            The NIC slot when @a scope is set to :py:attr:`DHCPConfigScope.machine_nic` ,
            must be zero for all other scope values.

        in may_add of type bool
            Set to @c TRUE if the configuration should be added if not found.
            If set to @c FALSE the method will fail with VBOX_E_OBJECT_NOT_FOUND.

        return config of type :class:`IDHCPConfig`
            The requested configuration.

        """
        if not isinstance(scope, DHCPConfigScope):
            raise TypeError("scope can only be an instance of type DHCPConfigScope")
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(slot, baseinteger):
            raise TypeError("slot can only be an instance of type baseinteger")
        if not isinstance(may_add, bool):
            raise TypeError("may_add can only be an instance of type bool")
        config = self._call("getConfig", in_p=[scope, name, slot, may_add])
        config = IDHCPConfig(config)
        return config


class IDHCPConfig(Interface):
    """
    The DHCP server has several configuration levels: global, group and
    individual MAC.  This interface implements the settings common to
    each level.
    """

    __uuid__ = "00f4a8dc-0002-4b81-0077-1dcb004571ba"
    __wsmap__ = "managed"

    @property
    def scope(self):
        """Get DHCPConfigScope value for 'scope'
        Indicates the kind of config this is (mostly for IDHCPIndividualConfig).
        """
        ret = self._get_attr("scope")
        return DHCPConfigScope(ret)

    @property
    def min_lease_time(self):
        """Get or set int value for 'minLeaseTime'
        The minimum lease time in seconds, ignored if zero.
        """
        ret = self._get_attr("minLeaseTime")
        return ret

    @min_lease_time.setter
    def min_lease_time(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("minLeaseTime", value)

    @property
    def default_lease_time(self):
        """Get or set int value for 'defaultLeaseTime'
        The default lease time in seconds, ignored if zero.
        """
        ret = self._get_attr("defaultLeaseTime")
        return ret

    @default_lease_time.setter
    def default_lease_time(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("defaultLeaseTime", value)

    @property
    def max_lease_time(self):
        """Get or set int value for 'maxLeaseTime'
        The maximum lease time in seconds, ignored if zero.
        """
        ret = self._get_attr("maxLeaseTime")
        return ret

    @max_lease_time.setter
    def max_lease_time(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("maxLeaseTime", value)

    @property
    def forced_options(self):
        """Get or set DHCPOption value for 'forcedOptions'
        List of DHCP options which should be forced upon the clients in this
        config scope when they are available, whether the clients asks for them
        or not.
        """
        ret = self._get_attr("forcedOptions")
        return [DHCPOption(a) for a in ret]

    @forced_options.setter
    def forced_options(self, value):
        if not isinstance(value, DHCPOption):
            raise TypeError("value is not an instance of DHCPOption")
        return self._set_attr("forcedOptions", value)

    @property
    def suppressed_options(self):
        """Get or set DHCPOption value for 'suppressedOptions'
        List of DHCP options which should not be sent to the clients in
        this config scope.  This is intended for cases where one client or a
        group of clients shouldn't see one or more (typically global) options.
        """
        ret = self._get_attr("suppressedOptions")
        return [DHCPOption(a) for a in ret]

    @suppressed_options.setter
    def suppressed_options(self, value):
        if not isinstance(value, DHCPOption):
            raise TypeError("value is not an instance of DHCPOption")
        return self._set_attr("suppressedOptions", value)

    def set_option(self, option, encoding, value):
        """Sets a DHCP option.

        in option of type :class:`DHCPOption`
            The DHCP option.

        in encoding of type :class:`DHCPOptionEncoding`
            The value encoding.

        in value of type str
            The DHCP option value.  The exact format depends on the DHCP
            @a option value and @a encoding, see see :py:class:`DHCPOption`
            for the :py:attr:`DHCPOptionEncoding.normal`  format.

        """
        if not isinstance(option, DHCPOption):
            raise TypeError("option can only be an instance of type DHCPOption")
        if not isinstance(encoding, DHCPOptionEncoding):
            raise TypeError(
                "encoding can only be an instance of type DHCPOptionEncoding"
            )
        if not isinstance(value, basestring):
            raise TypeError("value can only be an instance of type basestring")
        self._call("setOption", in_p=[option, encoding, value])

    def remove_option(self, option):
        """Removes the given DHCP option.

        in option of type :class:`DHCPOption`

        """
        if not isinstance(option, DHCPOption):
            raise TypeError("option can only be an instance of type DHCPOption")
        self._call("removeOption", in_p=[option])

    def remove_all_options(self):
        """Removes all the options.

        One exception here is the DhcpOpt_SubnetMask option in the global scope
        that is linked to the :py:func:`IDHCPServer.network_mask`  attribute
        and therefore cannot be removed.

        """
        self._call("removeAllOptions")

    def get_option(self, option):
        """Gets the value of a single DHCP option.

        in option of type :class:`DHCPOption`
            The DHCP option being sought.

        out encoding of type :class:`DHCPOptionEncoding`
            The value encoding.

        return value of type str
            The value of the requested DHCP option. The exact format depends on
            the DHCP @a option value and the @a encoding, see :py:class:`DHCPOption`
            for the :py:attr:`DHCPOptionEncoding.normal`  format.

        """
        if not isinstance(option, DHCPOption):
            raise TypeError("option can only be an instance of type DHCPOption")
        (value, encoding) = self._call("getOption", in_p=[option])
        encoding = DHCPOptionEncoding(encoding)
        return (value, encoding)

    def get_all_options(self):
        """Gets all DHCP options and their values

        out options of type :class:`DHCPOption`
            Array containing the DHCP option numbers.

        out encodings of type :class:`DHCPOptionEncoding`
            Array of value encodings that runs parallel to @a options.

        return values of type str
            Array of values that runs parallel to @a options and @a encodings.  The
            format depends on both of those.

        """
        (values, options, encodings) = self._call("getAllOptions")
        options = [DHCPOption(a) for a in options]
        encodings = [DHCPOptionEncoding(a) for a in encodings]
        return (values, options, encodings)

    def remove(self):
        """Remove this group or individual configuration.
        Will of course not work on global configurations.

        """
        self._call("remove")


class IDHCPGlobalConfig(IDHCPConfig):
    """
    The global DHCP server configuration, see :py:func:`IDHCPServer.global_config` .
    """

    __uuid__ = "46735de7-f4c4-4020-a185-0d2881bcfa8b"
    __wsmap__ = "managed"


class IDHCPGroupCondition(Interface):
    """
    Whether this is an inclusive or exclusive group membership condition
    """

    __uuid__ = "5ca9e537-5a1d-43f1-6f27-6a0db298a9a8"
    __wsmap__ = "managed"

    @property
    def inclusive(self):
        """Get or set bool value for 'inclusive'
        Whether this is an inclusive or exclusive group membership condition
        """
        ret = self._get_attr("inclusive")
        return ret

    @inclusive.setter
    def inclusive(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("inclusive", value)

    @property
    def type_p(self):
        """Get or set DHCPGroupConditionType value for 'type'
        Defines how the :py:func:`IDHCPGroupCondition.value`  is interpreted.
        """
        ret = self._get_attr("type")
        return DHCPGroupConditionType(ret)

    @type_p.setter
    def type_p(self, value):
        if not isinstance(value, DHCPGroupConditionType):
            raise TypeError("value is not an instance of DHCPGroupConditionType")
        return self._set_attr("type", value)

    @property
    def value(self):
        """Get or set str value for 'value'
        The condition value.
        """
        ret = self._get_attr("value")
        return ret

    @value.setter
    def value(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("value", value)

    def remove(self):
        """Remove this condition from the group."""
        self._call("remove")


class IDHCPGroupConfig(IDHCPConfig):
    """
    A configuration that applies to a group of NICs.
    """

    __uuid__ = "537707f7-ebf9-4d5c-7aea-877bfc4256ba"
    __wsmap__ = "managed"

    @property
    def name(self):
        """Get or set str value for 'name'
        The group name.
        """
        ret = self._get_attr("name")
        return ret

    @name.setter
    def name(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("name", value)

    @property
    def conditions(self):
        """Get IDHCPGroupCondition value for 'conditions'
        Group membership conditions.

        Add new conditions by calling :py:func:`IDHCPGroupConfig.add_condition`
        and use :py:func:`IDHCPGroupCondition.remove`  to remove.
        """
        ret = self._get_attr("conditions")
        return [IDHCPGroupCondition(a) for a in ret]

    def add_condition(self, inclusive, type_p, value):
        """Adds a new condition.

        in inclusive of type bool

        in type_p of type :class:`DHCPGroupConditionType`

        in value of type str

        return condition of type :class:`IDHCPGroupCondition`

        """
        if not isinstance(inclusive, bool):
            raise TypeError("inclusive can only be an instance of type bool")
        if not isinstance(type_p, DHCPGroupConditionType):
            raise TypeError(
                "type_p can only be an instance of type DHCPGroupConditionType"
            )
        if not isinstance(value, basestring):
            raise TypeError("value can only be an instance of type basestring")
        condition = self._call("addCondition", in_p=[inclusive, type_p, value])
        condition = IDHCPGroupCondition(condition)
        return condition

    def remove_all_conditions(self):
        """Removes all conditions."""
        self._call("removeAllConditions")


class IDHCPIndividualConfig(IDHCPConfig):
    """
    Configuration for a single NIC, either given directly by MAC address or by
    VM + adaptor slot number.
    """

    __uuid__ = "c40c2b86-73a5-46cc-8227-93fe57d006a6"
    __wsmap__ = "managed"

    @property
    def mac_address(self):
        """Get str value for 'MACAddress'
        The MAC address.  If a :py:attr:`DHCPConfigScope.machine_nic`  config, this
        will be queried via the VM ID.
        """
        ret = self._get_attr("MACAddress")
        return ret

    @property
    def machine_id(self):
        """Get str value for 'machineId'
        The virtual machine ID if a :py:attr:`DHCPConfigScope.machine_nic`  config,
        null UUID for :py:attr:`DHCPConfigScope.mac` .
        """
        ret = self._get_attr("machineId")
        return ret

    @property
    def slot(self):
        """Get int value for 'slot'
        The NIC slot number of the VM if a :py:attr:`DHCPConfigScope.machine_nic`  config.
        """
        ret = self._get_attr("slot")
        return ret

    @property
    def fixed_address(self):
        """Get or set str value for 'fixedAddress'
        Fixed IPv4 address assignment, dynamic if empty.
        """
        ret = self._get_attr("fixedAddress")
        return ret

    @fixed_address.setter
    def fixed_address(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("fixedAddress", value)


class IVirtualBox(Interface):
    """
    The IVirtualBox interface represents the main interface exposed by the
    product that provides virtual machine management.

    An instance of IVirtualBox is required for the product to do anything
    useful. Even though the interface does not expose this, internally,
    IVirtualBox is implemented as a singleton and actually lives in the
    process of the VirtualBox server (VBoxSVC.exe). This makes sure that
    IVirtualBox can track the state of all virtual machines on a particular
    host, regardless of which frontend started them.

    To enumerate all the virtual machines on the host, use the
    :py:func:`IVirtualBox.machines`  attribute.

    Error information handling is a bit special with IVirtualBox: creating
    an instance will always succeed. The return of the actual error
    code/information is postponed to any attribute or method call. The
    reason for this is that COM likes to mutilate the error code and lose
    the detailed error information returned by instance creation.
    """

    __uuid__ = "d0a0163f-e254-4e5b-a1f2-011cf991c38d"
    __wsmap__ = "managed"

    @property
    def version(self):
        """Get str value for 'version'
        A string representing the version number of the product. The
        format is 3 integer numbers divided by dots (e.g. 1.0.1). The
        last number represents the build number and will frequently change.

        This may be followed by a _ALPHA[0-9]*, _BETA[0-9]* or _RC[0-9]* tag
        in prerelease builds. Non-Oracle builds may (/shall) also have a
        publisher tag, at the end. The publisher tag starts with an underscore
        just like the prerelease build type tag.
        """
        ret = self._get_attr("version")
        return ret

    @property
    def version_normalized(self):
        """Get str value for 'versionNormalized'
        A string representing the version number of the product,
        without the publisher information (but still with other tags).
        See :py:func:`version` .
        """
        ret = self._get_attr("versionNormalized")
        return ret

    @property
    def revision(self):
        """Get int value for 'revision'
        The internal build revision number of the product.
        """
        ret = self._get_attr("revision")
        return ret

    @property
    def package_type(self):
        """Get str value for 'packageType'
        A string representing the package type of this product. The
        format is OS_ARCH_DIST where OS is either WINDOWS, LINUX,
        SOLARIS, DARWIN. ARCH is either 32BITS or 64BITS. DIST
        is either GENERIC, UBUNTU_606, UBUNTU_710, or something like
        this.
        """
        ret = self._get_attr("packageType")
        return ret

    @property
    def api_version(self):
        """Get str value for 'APIVersion'
        A string representing the VirtualBox API version number. The format is
        2 integer numbers divided by an underscore (e.g. 1_0). After the
        first public release of packages with a particular API version the
        API will not be changed in an incompatible way. Note that this
        guarantee does not apply to development builds, and also there is no
        guarantee that this version is identical to the first two integer
        numbers of the package version.
        """
        ret = self._get_attr("APIVersion")
        return ret

    @property
    def api_revision(self):
        """Get int value for 'APIRevision'
        This is mainly intended for the VBox Validation Kit so it can fluently
        deal with incompatible API changes and new functionality during
        development (i.e. on trunk).

        The high 7 bits (62:56) is the major version number, the next 8 bits
        (55:48) are the minor version number, the next 8 bits (47:40) are the
        build number, and the rest (39:0) is the API revision number.

        The API revision number is manually increased on trunk when making
        incompatible changes that the validation kit or others needs to be able
        to detect and cope with dynamically.  It can also be used to indicate
        the presence of new features on both trunk and branches.
        """
        ret = self._get_attr("APIRevision")
        return ret

    @property
    def home_folder(self):
        """Get str value for 'homeFolder'
        Full path to the directory where the global settings file,
        VirtualBox.xml, is stored.

        In this version of VirtualBox, the value of this property is
        always <user_dir>/.VirtualBox (where
        <user_dir> is the path to the user directory,
        as determined by the host OS), and cannot be changed.

        This path is also used as the base to resolve relative paths in
        places where relative paths are allowed (unless otherwise
        expressly indicated).
        """
        ret = self._get_attr("homeFolder")
        return ret

    @property
    def settings_file_path(self):
        """Get str value for 'settingsFilePath'
        Full name of the global settings file.
        The value of this property corresponds to the value of
        :py:func:`home_folder`  plus /VirtualBox.xml.
        """
        ret = self._get_attr("settingsFilePath")
        return ret

    @property
    def host(self):
        """Get IHost value for 'host'
        Associated host object.
        """
        ret = self._get_attr("host")
        return IHost(ret)

    @property
    def system_properties(self):
        """Get ISystemProperties value for 'systemProperties'
        Associated system information object.
        """
        ret = self._get_attr("systemProperties")
        return ISystemProperties(ret)

    @property
    def machines(self):
        """Get IMachine value for 'machines'
        Array of machine objects registered within this VirtualBox instance.
        """
        ret = self._get_attr("machines")
        return [IMachine(a) for a in ret]

    @property
    def machine_groups(self):
        """Get str value for 'machineGroups'
        Array of all machine group names which are used by the machines which
        are accessible. Each group is only listed once, however they are listed
        in no particular order and there is no guarantee that there are no gaps
        in the group hierarchy (i.e. "/", "/group/subgroup"
        is a valid result).
        """
        ret = self._get_attr("machineGroups")
        return ret

    @property
    def hard_disks(self):
        """Get IMedium value for 'hardDisks'
        Array of medium objects known to this VirtualBox installation.

        This array contains only base media. All differencing
        media of the given base medium can be enumerated using
        :py:func:`IMedium.children` .
        """
        ret = self._get_attr("hardDisks")
        return [IMedium(a) for a in ret]

    @property
    def dvd_images(self):
        """Get IMedium value for 'DVDImages'
        Array of CD/DVD image objects currently in use by this VirtualBox instance.
        """
        ret = self._get_attr("DVDImages")
        return [IMedium(a) for a in ret]

    @property
    def floppy_images(self):
        """Get IMedium value for 'floppyImages'
        Array of floppy image objects currently in use by this VirtualBox instance.
        """
        ret = self._get_attr("floppyImages")
        return [IMedium(a) for a in ret]

    @property
    def progress_operations(self):
        """Get IProgress value for 'progressOperations'"""
        ret = self._get_attr("progressOperations")
        return [IProgress(a) for a in ret]

    @property
    def guest_os_types(self):
        """Get IGuestOSType value for 'guestOSTypes'"""
        ret = self._get_attr("guestOSTypes")
        return [IGuestOSType(a) for a in ret]

    @property
    def shared_folders(self):
        """Get ISharedFolder value for 'sharedFolders'
        Collection of global shared folders. Global shared folders are
        available to all virtual machines.

        New shared folders are added to the collection using
        :py:func:`create_shared_folder` . Existing shared folders can be
        removed using :py:func:`remove_shared_folder` .


        In the current version of the product, global shared folders are not
        implemented and therefore this collection is always empty.
        """
        ret = self._get_attr("sharedFolders")
        return [ISharedFolder(a) for a in ret]

    @property
    def performance_collector(self):
        """Get IPerformanceCollector value for 'performanceCollector'
        Associated performance collector object.
        """
        ret = self._get_attr("performanceCollector")
        return IPerformanceCollector(ret)

    @property
    def dhcp_servers(self):
        """Get IDHCPServer value for 'DHCPServers'
        DHCP servers.
        """
        ret = self._get_attr("DHCPServers")
        return [IDHCPServer(a) for a in ret]

    @property
    def nat_networks(self):
        """Get INATNetwork value for 'NATNetworks'"""
        ret = self._get_attr("NATNetworks")
        return [INATNetwork(a) for a in ret]

    @property
    def event_source(self):
        """Get IEventSource value for 'eventSource'
        Event source for VirtualBox events.
        """
        ret = self._get_attr("eventSource")
        return IEventSource(ret)

    @property
    def extension_pack_manager(self):
        """Get IExtPackManager value for 'extensionPackManager'
        The extension pack manager.
        """
        ret = self._get_attr("extensionPackManager")
        return IExtPackManager(ret)

    @property
    def internal_networks(self):
        """Get str value for 'internalNetworks'
        Names of all internal networks.
        """
        ret = self._get_attr("internalNetworks")
        return ret

    @property
    def generic_network_drivers(self):
        """Get str value for 'genericNetworkDrivers'
        Names of all generic network drivers.
        """
        ret = self._get_attr("genericNetworkDrivers")
        return ret

    @property
    def cloud_networks(self):
        """Get ICloudNetwork value for 'cloudNetworks'
        Names of all configured cloud networks.
        """
        ret = self._get_attr("cloudNetworks")
        return [ICloudNetwork(a) for a in ret]

    @property
    def cloud_provider_manager(self):
        """Get ICloudProviderManager value for 'cloudProviderManager'
        The cloud provider manager (singleton).
        """
        ret = self._get_attr("cloudProviderManager")
        return ICloudProviderManager(ret)

    def compose_machine_filename(self, name, group, create_flags, base_folder):
        """Returns a recommended full path of the settings file name for a new virtual
        machine.

        This API serves two purposes:


        It gets called by :py:func:`create_machine`  if @c null or
        empty string (which is recommended) is specified for the
        @a settingsFile argument there, which means that API should use
        a recommended default file name.

        It can be called manually by a client software before creating a machine,
        e.g. if that client wants to pre-create the machine directory to create
        virtual hard disks in that directory together with the new machine
        settings file. In that case, the file name should be stripped from the
        full settings file path returned by this function to obtain the
        machine directory.


        See :py:func:`IMachine.name`  and :py:func:`create_machine`  for more
        details about the machine name.

        @a groupName defines which additional subdirectory levels should be
        included. It must be either a valid group name or @c null or empty
        string which designates that the machine will not be related to a
        machine group.

        If @a baseFolder is a @c null or empty string (which is recommended), the
        default machine settings folder
        (see :py:func:`ISystemProperties.default_machine_folder` ) will be used as
        a base folder for the created machine, resulting in a file name like
        "/home/user/VirtualBox VMs/name/name.vbox". Otherwise the given base folder
        will be used.

        This method does not access the host disks. In particular, it does not check
        for whether a machine with this name already exists.

        in name of type str
            Suggested machine name.

        in group of type str
            Machine group name for the new machine or machine group. It is
            used to determine the right subdirectory.

        in create_flags of type str
            Machine creation flags, see :py:func:`create_machine`  (optional).

        in base_folder of type str
            Base machine folder (optional).

        return file_p of type str
            Fully qualified path where the machine would be created.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(group, basestring):
            raise TypeError("group can only be an instance of type basestring")
        if not isinstance(create_flags, basestring):
            raise TypeError("create_flags can only be an instance of type basestring")
        if not isinstance(base_folder, basestring):
            raise TypeError("base_folder can only be an instance of type basestring")
        file_p = self._call(
            "composeMachineFilename", in_p=[name, group, create_flags, base_folder]
        )
        return file_p

    def create_machine(self, settings_file, name, groups, os_type_id, flags):
        """Creates a new virtual machine by creating a machine settings file at
                the given location.

                VirtualBox machine settings files use a custom XML dialect. Starting
                with VirtualBox 4.0, a ".vbox" extension is recommended, but not enforced,
                and machine files can be created at arbitrary locations.

                However, it is recommended that machines are created in the default
                machine folder (e.g. "/home/user/VirtualBox VMs/name/name.vbox"; see
                :py:func:`ISystemProperties.default_machine_folder` ). If you specify
                @c null or empty string (which is recommended) for the @a settingsFile
                argument, :py:func:`compose_machine_filename`  is called automatically
                to have such a recommended name composed based on the machine name
                given in the @a name argument and the primary group.

                If the resulting settings file already exists, this method will fail,
                unless the forceOverwrite flag is set.

                The new machine is created unregistered, with the initial configuration
                set according to the specified guest OS type. A typical sequence of
                actions to create a new virtual machine is as follows:



                Call this method to have a new machine created. The returned machine
                object will be "mutable" allowing to change any machine property.



                Configure the machine using the appropriate attributes and methods.



                Call :py:func:`IMachine.save_settings`  to write the settings
                to the machine's XML settings file. The configuration of the newly
                created machine will not be saved to disk until this method is
                called.



                Call :py:func:`register_machine`  to add the machine to the list
                of machines known to VirtualBox.



                The specified guest OS type identifier must match an ID of one of known
                guest OS types listed in the :py:func:`IVirtualBox.guest_os_types`
                array.


                :py:func:`IMachine.settings_modified`  will return
                @c false for the created machine, until any of machine settings
                are changed.



                There is no way to change the name of the settings file or
                subfolder of the created machine directly.

                in settings_file of type str
                    Fully qualified path where the settings file should be created,
                    empty string or @c null for a default folder and file based on the
                    @a name argument and the primary group.
                    (see :py:func:`compose_machine_filename` ).

                in name of type str
                    Machine name.

                in groups of type str
                    Array of group names. @c null or an empty array have the same
                    meaning as an array with just the empty string or "/", i.e.
                    create a machine without group association.

                in os_type_id of type str
                    Guest OS Type ID.

                in flags of type str
                    Additional property parameters, passed as a comma-separated list of
                    "name=value" type entries. The following ones are recognized:
                    forceOverwrite=1 to overwrite an existing machine settings
                    file, UUID=<uuid> to specify a machine UUID and
                    directoryIncludesUUID=1 to switch to a special VM directory
                    naming scheme which should not be used unless necessary.

                return machine of type :class:`IMachine`
                    Created machine object.

                raises :class:`VBoxErrorObjectNotFound`
                    @a osTypeId is invalid.

                raises :class:`VBoxErrorFileError`
                    Resulting settings file name is invalid or the settings file already
        exists or could not be created due to an I/O error.

                raises :class:`OleErrorInvalidarg`
                    @a name is empty or @c null.

        """
        if not isinstance(settings_file, basestring):
            raise TypeError("settings_file can only be an instance of type basestring")
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(groups, list):
            raise TypeError("groups can only be an instance of type list")
        for a in groups[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(os_type_id, basestring):
            raise TypeError("os_type_id can only be an instance of type basestring")
        if not isinstance(flags, basestring):
            raise TypeError("flags can only be an instance of type basestring")
        machine = self._call(
            "createMachine", in_p=[settings_file, name, groups, os_type_id, flags]
        )
        machine = IMachine(machine)
        return machine

    def open_machine(self, settings_file):
        """Opens a virtual machine from the existing settings file.
        The opened machine remains unregistered until you call
        :py:func:`register_machine` .

        The specified settings file name must be fully qualified.
        The file must exist and be a valid machine XML settings file
        whose contents will be used to construct the machine object.


        :py:func:`IMachine.settings_modified`  will return
        @c false for the opened machine, until any of machine settings
        are changed.

        in settings_file of type str
            Name of the machine settings file.

        return machine of type :class:`IMachine`
            Opened machine object.

        raises :class:`VBoxErrorFileError`
            Settings file name invalid, not found or sharing violation.

        """
        if not isinstance(settings_file, basestring):
            raise TypeError("settings_file can only be an instance of type basestring")
        machine = self._call("openMachine", in_p=[settings_file])
        machine = IMachine(machine)
        return machine

    def register_machine(self, machine):
        """Registers the machine previously created using
        :py:func:`create_machine`  or opened using
        :py:func:`open_machine`  within this VirtualBox installation. After
        successful method invocation, the
        :py:class:`IMachineRegisteredEvent`  event is fired.


        This method implicitly calls :py:func:`IMachine.save_settings`
        to save all current machine settings before registering it.

        in machine of type :class:`IMachine`

        raises :class:`VBoxErrorObjectNotFound`
            No matching virtual machine found.

        raises :class:`VBoxErrorInvalidObjectState`
            Virtual machine was not created within this VirtualBox instance.

        """
        if not isinstance(machine, IMachine):
            raise TypeError("machine can only be an instance of type IMachine")
        self._call("registerMachine", in_p=[machine])

    def find_machine(self, name_or_id):
        """Attempts to find a virtual machine given its name or UUID.

        Inaccessible machines cannot be found by name, only by UUID, because their name
        cannot safely be determined.

        in name_or_id of type str
            What to search for. This can either be the UUID or the name of a virtual machine.

        return machine of type :class:`IMachine`
            Machine object, if found.

        raises :class:`VBoxErrorObjectNotFound`
            Could not find registered machine matching @a nameOrId.

        """
        if not isinstance(name_or_id, basestring):
            raise TypeError("name_or_id can only be an instance of type basestring")
        machine = self._call("findMachine", in_p=[name_or_id])
        machine = IMachine(machine)
        return machine

    def get_machines_by_groups(self, groups):
        """Gets all machine references which are in one of the specified groups.

        in groups of type str
            What groups to match. The usual group list rules apply, i.e.
            passing an empty list will match VMs in the toplevel group, likewise
            the empty string.

        return machines of type :class:`IMachine`
            All machines which matched.

        """
        if not isinstance(groups, list):
            raise TypeError("groups can only be an instance of type list")
        for a in groups[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        machines = self._call("getMachinesByGroups", in_p=[groups])
        machines = [IMachine(a) for a in machines]
        return machines

    def get_machine_states(self, machines):
        """Gets the state of several machines in a single operation.

        in machines of type :class:`IMachine`
            Array with the machine references.

        return states of type :class:`MachineState`
            Machine states, corresponding to the machines.

        """
        if not isinstance(machines, list):
            raise TypeError("machines can only be an instance of type list")
        for a in machines[:10]:
            if not isinstance(a, IMachine):
                raise TypeError("array can only contain objects of type IMachine")
        states = self._call("getMachineStates", in_p=[machines])
        states = [MachineState(a) for a in states]
        return states

    def create_appliance(self):
        """Creates a new appliance object, which represents an appliance in the Open Virtual Machine
        Format (OVF). This can then be used to import an OVF appliance into VirtualBox or to export
        machines as an OVF appliance; see the documentation for :py:class:`IAppliance`  for details.

        return appliance of type :class:`IAppliance`
            New appliance.

        """
        appliance = self._call("createAppliance")
        appliance = IAppliance(appliance)
        return appliance

    def create_unattended_installer(self):
        """Creates a new :py:class:`IUnattended`  guest installation object.  This can be used to
        analyze an installation ISO to create and configure a new machine for it to be installed
        on.  It can also be used to (re)install an existing machine.

        return unattended of type :class:`IUnattended`
            New unattended object.

        """
        unattended = self._call("createUnattendedInstaller")
        unattended = IUnattended(unattended)
        return unattended

    def create_medium(self, format_p, location, access_mode, a_device_type_type):
        """Creates a new base medium object that will use the given storage
        format and location for medium data.

        The actual storage unit is not created by this method. In order to
        do it, and before you are able to attach the created medium to
        virtual machines, you must call one of the following methods to
        allocate a format-specific storage unit at the specified location:

        :py:func:`IMedium.create_base_storage`
        :py:func:`IMedium.create_diff_storage`


        Some medium attributes, such as :py:func:`IMedium.id_p` , may
        remain uninitialized until the medium storage unit is successfully
        created by one of the above methods.

        Depending on the given device type, the file at the storage location
        must be in one of the media formats understood by VirtualBox:


        With a "HardDisk" device type, the file must be a hard disk image
        in one of the formats supported by VirtualBox (see
        :py:func:`ISystemProperties.medium_formats` ).
        After the storage unit is successfully created and this method succeeds,
        if the medium is a base medium, it
        will be added to the :py:func:`hard_disks`  array attribute.
        With a "DVD" device type, the file must be an ISO 9960 CD/DVD image.
        After this method succeeds, the medium will be added to the
        :py:func:`dvd_images`  array attribute.
        With a "Floppy" device type, the file must be an RAW floppy image.
        After this method succeeds, the medium will be added to the
        :py:func:`floppy_images`  array attribute.


        The list of all storage formats supported by this VirtualBox
        installation can be obtained using
        :py:func:`ISystemProperties.medium_formats` . If the @a format
        attribute is empty or @c null then the default storage format
        specified by :py:func:`ISystemProperties.default_hard_disk_format`  will
        be used for disks r creating a storage unit of the medium.

        Note that the format of the location string is storage format specific.
        See :py:func:`IMedium.location`  and IMedium for more details.

        in format_p of type str
            Identifier of the storage format to use for the new medium.

        in location of type str
            Location of the storage unit for the new medium.

        in access_mode of type :class:`AccessMode`
            Whether to open the image in read/write or read-only mode. For
            a "DVD" device type, this is ignored and read-only mode is always assumed.

        in a_device_type_type of type :class:`DeviceType`
            Must be one of "HardDisk", "DVD" or "Floppy".

        return medium of type :class:`IMedium`
            Created medium object.

        raises :class:`VBoxErrorObjectNotFound`
            @a format identifier is invalid. See

        raises :class:`VBoxErrorFileError`
            @a location is a not valid file name (for file-based formats only).

        """
        if not isinstance(format_p, basestring):
            raise TypeError("format_p can only be an instance of type basestring")
        if not isinstance(location, basestring):
            raise TypeError("location can only be an instance of type basestring")
        if not isinstance(access_mode, AccessMode):
            raise TypeError("access_mode can only be an instance of type AccessMode")
        if not isinstance(a_device_type_type, DeviceType):
            raise TypeError(
                "a_device_type_type can only be an instance of type DeviceType"
            )
        medium = self._call(
            "createMedium", in_p=[format_p, location, access_mode, a_device_type_type]
        )
        medium = IMedium(medium)
        return medium

    def open_medium(self, location, device_type, access_mode, force_new_uuid):
        """Finds existing media or opens a medium from an existing storage location.

                Once a medium has been opened, it can be passed to other VirtualBox
                methods, in particular to :py:func:`IMachine.attach_device` .

                Depending on the given device type, the file at the storage location
                must be in one of the media formats understood by VirtualBox:


                With a "HardDisk" device type, the file must be a hard disk image
                in one of the formats supported by VirtualBox (see
                :py:func:`ISystemProperties.medium_formats` ).
                After this method succeeds, if the medium is a base medium, it
                will be added to the :py:func:`hard_disks`  array attribute.
                With a "DVD" device type, the file must be an ISO 9960 CD/DVD image.
                After this method succeeds, the medium will be added to the
                :py:func:`dvd_images`  array attribute.
                With a "Floppy" device type, the file must be an RAW floppy image.
                After this method succeeds, the medium will be added to the
                :py:func:`floppy_images`  array attribute.


                After having been opened, the medium can be re-found by this method
                and can be attached to virtual machines. See :py:class:`IMedium`  for
                more details.

                The UUID of the newly opened medium will either be retrieved from the
                storage location, if the format supports it (e.g. for hard disk images),
                or a new UUID will be randomly generated (e.g. for ISO and RAW files).
                If for some reason you need to change the medium's UUID, use
                :py:func:`IMedium.set_ids` .

                If a differencing hard disk medium is to be opened by this method, the
                operation will succeed only if its parent medium and all ancestors,
                if any, are already known to this VirtualBox installation (for example,
                were opened by this method before).

                This method attempts to guess the storage format of the specified medium
                by reading medium data at the specified location.

                If @a accessMode is ReadWrite (which it should be for hard disks and floppies),
                the image is opened for read/write access and must have according permissions,
                as VirtualBox may actually write status information into the disk's metadata
                sections.

                Note that write access is required for all typical hard disk usage in VirtualBox,
                since VirtualBox may need to write metadata such as a UUID into the image.
                The only exception is opening a source image temporarily for copying and
                cloning (see :py:func:`IMedium.clone_to`  when the image will be closed
                again soon.

                The format of the location string is storage format specific. See
                :py:func:`IMedium.location`  and IMedium for more details.

                in location of type str
                    Location of the storage unit that contains medium data in one of
                    the supported storage formats.

                in device_type of type :class:`DeviceType`
                    Must be one of "HardDisk", "DVD" or "Floppy".

                in access_mode of type :class:`AccessMode`
                    Whether to open the image in read/write or read-only mode. For
                    a "DVD" device type, this is ignored and read-only mode is always assumed.

                in force_new_uuid of type bool
                    Allows the caller to request a completely new medium UUID for
                    the image which is to be opened. Useful if one intends to open an exact
                    copy of a previously opened image, as this would normally fail due to
                    the duplicate UUID.

                return medium of type :class:`IMedium`
                    Opened medium object.

                raises :class:`VBoxErrorFileError`
                    Invalid medium storage file location or could not find the medium
        at the specified location.

                raises :class:`VBoxErrorIprtError`
                    Could not get medium storage format.

                raises :class:`OleErrorInvalidarg`
                    Invalid medium storage format.

                raises :class:`VBoxErrorInvalidObjectState`
                    Medium has already been added to a media registry.

        """
        if not isinstance(location, basestring):
            raise TypeError("location can only be an instance of type basestring")
        if not isinstance(device_type, DeviceType):
            raise TypeError("device_type can only be an instance of type DeviceType")
        if not isinstance(access_mode, AccessMode):
            raise TypeError("access_mode can only be an instance of type AccessMode")
        if not isinstance(force_new_uuid, bool):
            raise TypeError("force_new_uuid can only be an instance of type bool")
        medium = self._call(
            "openMedium", in_p=[location, device_type, access_mode, force_new_uuid]
        )
        medium = IMedium(medium)
        return medium

    def get_guest_os_type(self, id_p):
        """Returns an object describing the specified guest OS type.

        The requested guest OS type is specified using a string which is a
        mnemonic identifier of the guest operating system, such as
        "win31" or "ubuntu". The guest OS type ID of a
        particular virtual machine can be read or set using the
        :py:func:`IMachine.os_type_id`  attribute.

        The :py:func:`IVirtualBox.guest_os_types`  collection contains all
        available guest OS type objects. Each object has an
        :py:func:`IGuestOSType.id_p`  attribute which contains an identifier of
        the guest OS this object describes.

        While this function returns an error for unknown guest OS types, they
        can be still used without serious problems (if one accepts the fact
        that there is no default VM config information).

        in id_p of type str
            Guest OS type ID string.

        return type_p of type :class:`IGuestOSType`
            Guest OS type object.

        raises :class:`OleErrorInvalidarg`
            @a id is not a valid Guest OS type.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        type_p = self._call("getGuestOSType", in_p=[id_p])
        type_p = IGuestOSType(type_p)
        return type_p

    def create_shared_folder(
        self, name, host_path, writable, automount, auto_mount_point
    ):
        """Creates a new global shared folder by associating the given logical
        name with the given host path, adds it to the collection of shared
        folders and starts sharing it. Refer to the description of
        :py:class:`ISharedFolder`  to read more about logical names.

        In the current implementation, this operation is not
        implemented.

        in name of type str
            Unique logical name of the shared folder.

        in host_path of type str
            Full path to the shared folder in the host file system.

        in writable of type bool
            Whether the share is writable or readonly

        in automount of type bool
            Whether the share gets automatically mounted by the guest
            or not.

        in auto_mount_point of type str
            Where the guest should automatically mount the folder, if possible.
            For Windows and OS/2 guests this should be a drive letter, while other
            guests it should be a absolute directory.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(host_path, basestring):
            raise TypeError("host_path can only be an instance of type basestring")
        if not isinstance(writable, bool):
            raise TypeError("writable can only be an instance of type bool")
        if not isinstance(automount, bool):
            raise TypeError("automount can only be an instance of type bool")
        if not isinstance(auto_mount_point, basestring):
            raise TypeError(
                "auto_mount_point can only be an instance of type basestring"
            )
        self._call(
            "createSharedFolder",
            in_p=[name, host_path, writable, automount, auto_mount_point],
        )

    def remove_shared_folder(self, name):
        """Removes the global shared folder with the given name previously
        created by :py:func:`create_shared_folder`  from the collection of
        shared folders and stops sharing it.

        In the current implementation, this operation is not
        implemented.

        in name of type str
            Logical name of the shared folder to remove.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        self._call("removeSharedFolder", in_p=[name])

    def get_extra_data_keys(self):
        """Returns an array representing the global extra data keys which currently
        have values defined.

        return keys of type str
            Array of extra data keys.

        """
        keys = self._call("getExtraDataKeys")
        return keys

    def get_extra_data(self, key):
        """Returns associated global extra data.

        If the requested data @a key does not exist, this function will
        succeed and return an empty string in the @a value argument.

        in key of type str
            Name of the data key to get.

        return value of type str
            Value of the requested data key.

        raises :class:`VBoxErrorFileError`
            Settings file not accessible.

        raises :class:`VBoxErrorXmlError`
            Could not parse the settings file.

        """
        if not isinstance(key, basestring):
            raise TypeError("key can only be an instance of type basestring")
        value = self._call("getExtraData", in_p=[key])
        return value

    def set_extra_data(self, key, value):
        """Sets associated global extra data.

        If you pass @c null or an empty string as a key @a value, the given
        @a key will be deleted.


        Key must contain printable (non-control) UTF-8 characters only.


        Before performing the actual data change, this method will ask all
        registered event listeners using the
        :py:class:`IExtraDataCanChangeEvent`
        notification for a permission. If one of the listeners refuses the
        new value, the change will not be performed.


        On success, the
        :py:class:`IExtraDataChangedEvent`  notification
        is called to inform all registered listeners about a successful data
        change.

        in key of type str
            Name of the data key to set.

        in value of type str
            Value to assign to the key.

        raises :class:`VBoxErrorFileError`
            Settings file not accessible.

        raises :class:`VBoxErrorXmlError`
            Could not parse the settings file.

        raises :class:`OleErrorAccessdenied`
            Modification request refused.

        raises :class:`OleErrorInvalidarg`
            Key contains invalid characters.

        """
        if not isinstance(key, basestring):
            raise TypeError("key can only be an instance of type basestring")
        if not isinstance(value, basestring):
            raise TypeError("value can only be an instance of type basestring")
        self._call("setExtraData", in_p=[key, value])

    def set_settings_secret(self, password):
        """Unlocks the secret data by passing the unlock password to the
        server. The server will cache the password for that machine.

        in password of type str
            The cipher key.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine is not mutable.

        """
        if not isinstance(password, basestring):
            raise TypeError("password can only be an instance of type basestring")
        self._call("setSettingsSecret", in_p=[password])

    def create_dhcp_server(self, name):
        """Creates a DHCP server settings to be used for the given internal network name

        in name of type str
            server name

        return server of type :class:`IDHCPServer`
            DHCP server settings

        raises :class:`OleErrorInvalidarg`
            Host network interface @a name already exists.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        server = self._call("createDHCPServer", in_p=[name])
        server = IDHCPServer(server)
        return server

    def find_dhcp_server_by_network_name(self, name):
        """Searches a DHCP server settings to be used for the given internal network name

        in name of type str
            server name

        return server of type :class:`IDHCPServer`
            DHCP server settings

        raises :class:`OleErrorInvalidarg`
            Host network interface @a name already exists.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        server = self._call("findDHCPServerByNetworkName", in_p=[name])
        server = IDHCPServer(server)
        return server

    def remove_dhcp_server(self, server):
        """Removes the DHCP server settings

        in server of type :class:`IDHCPServer`
            DHCP server settings to be removed

        raises :class:`OleErrorInvalidarg`
            Host network interface @a name already exists.

        """
        if not isinstance(server, IDHCPServer):
            raise TypeError("server can only be an instance of type IDHCPServer")
        self._call("removeDHCPServer", in_p=[server])

    def create_nat_network(self, network_name):
        """

        in network_name of type str

        return network of type :class:`INATNetwork`

        """
        if not isinstance(network_name, basestring):
            raise TypeError("network_name can only be an instance of type basestring")
        network = self._call("createNATNetwork", in_p=[network_name])
        network = INATNetwork(network)
        return network

    def find_nat_network_by_name(self, network_name):
        """

        in network_name of type str

        return network of type :class:`INATNetwork`

        """
        if not isinstance(network_name, basestring):
            raise TypeError("network_name can only be an instance of type basestring")
        network = self._call("findNATNetworkByName", in_p=[network_name])
        network = INATNetwork(network)
        return network

    def remove_nat_network(self, network):
        """

        in network of type :class:`INATNetwork`

        """
        if not isinstance(network, INATNetwork):
            raise TypeError("network can only be an instance of type INATNetwork")
        self._call("removeNATNetwork", in_p=[network])

    def create_cloud_network(self, network_name):
        """

        in network_name of type str

        return network of type :class:`ICloudNetwork`

        """
        if not isinstance(network_name, basestring):
            raise TypeError("network_name can only be an instance of type basestring")
        network = self._call("createCloudNetwork", in_p=[network_name])
        network = ICloudNetwork(network)
        return network

    def find_cloud_network_by_name(self, network_name):
        """

        in network_name of type str

        return network of type :class:`ICloudNetwork`

        """
        if not isinstance(network_name, basestring):
            raise TypeError("network_name can only be an instance of type basestring")
        network = self._call("findCloudNetworkByName", in_p=[network_name])
        network = ICloudNetwork(network)
        return network

    def remove_cloud_network(self, network):
        """

        in network of type :class:`ICloudNetwork`

        """
        if not isinstance(network, ICloudNetwork):
            raise TypeError("network can only be an instance of type ICloudNetwork")
        self._call("removeCloudNetwork", in_p=[network])

    def check_firmware_present(self, firmware_type, version):
        """Check if this VirtualBox installation has a firmware
        of the given type available, either system-wide or per-user.
        Optionally, this may return a hint where this firmware can be
        downloaded from.

        in firmware_type of type :class:`FirmwareType`
            Type of firmware to check.

        in version of type str
            Expected version number, usually empty string (presently ignored).

        out url of type str
            Suggested URL to download this firmware from.

        out file_p of type str
            Filename of firmware, only valid if result == TRUE.

        return result of type bool
            If firmware of this type and version is available.

        """
        if not isinstance(firmware_type, FirmwareType):
            raise TypeError(
                "firmware_type can only be an instance of type FirmwareType"
            )
        if not isinstance(version, basestring):
            raise TypeError("version can only be an instance of type basestring")
        (result, url, file_p) = self._call(
            "checkFirmwarePresent", in_p=[firmware_type, version]
        )
        return (result, url, file_p)


class IVFSExplorer(Interface):
    """
    The VFSExplorer interface unifies access to different file system
    types. This includes local file systems as well remote file systems like
    S3. For a list of supported types see :py:class:`VFSType` .
    An instance of this is returned by :py:func:`IAppliance.create_vfs_explorer` .
    """

    __uuid__ = "fb220201-2fd3-47e2-a5dc-2c2431d833cc"
    __wsmap__ = "managed"

    @property
    def path(self):
        """Get str value for 'path'
        Returns the current path in the virtual file system.
        """
        ret = self._get_attr("path")
        return ret

    @property
    def type_p(self):
        """Get VFSType value for 'type'
        Returns the file system type which is currently in use.
        """
        ret = self._get_attr("type")
        return VFSType(ret)

    def update(self):
        """Updates the internal list of files/directories from the
        current directory level. Use :py:func:`entry_list`  to get the full list
        after a call to this method.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        """
        progress = self._call("update")
        progress = IProgress(progress)
        return progress

    def cd(self, dir_p):
        """Change the current directory level.

        in dir_p of type str
            The name of the directory to go in.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        """
        if not isinstance(dir_p, basestring):
            raise TypeError("dir_p can only be an instance of type basestring")
        progress = self._call("cd", in_p=[dir_p])
        progress = IProgress(progress)
        return progress

    def cd_up(self):
        """Go one directory upwards from the current directory level.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        """
        progress = self._call("cdUp")
        progress = IProgress(progress)
        return progress

    def entry_list(self):
        """Returns a list of files/directories after a call to :py:func:`update` . The user is responsible for keeping this internal
        list up do date.

        out names of type str
            The list of names for the entries.

        out types of type int
            The list of types for the entries. :py:class:`FsObjType`

        out sizes of type int
            The list of sizes (in bytes) for the entries.

        out modes of type int
            The list of file modes (in octal form) for the entries.

        """
        (names, types, sizes, modes) = self._call("entryList")
        return (names, types, sizes, modes)

    def exists(self, names):
        """Checks if the given file list exists in the current directory
        level.

        in names of type str
            The names to check.

        return exists of type str
            The names which exist.

        """
        if not isinstance(names, list):
            raise TypeError("names can only be an instance of type list")
        for a in names[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        exists = self._call("exists", in_p=[names])
        return exists

    def remove(self, names):
        """Deletes the given files in the current directory level.

        in names of type str
            The names to remove.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        """
        if not isinstance(names, list):
            raise TypeError("names can only be an instance of type list")
        for a in names[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        progress = self._call("remove", in_p=[names])
        progress = IProgress(progress)
        return progress


class ICertificate(Interface):
    """
    X.509 certificate details.
    """

    __uuid__ = "392f1de4-80e1-4a8a-93a1-67c5f92a838a"
    __wsmap__ = "managed"

    @property
    def version_number(self):
        """Get CertificateVersion value for 'versionNumber'
        Certificate version number.
        """
        ret = self._get_attr("versionNumber")
        return CertificateVersion(ret)

    @property
    def serial_number(self):
        """Get str value for 'serialNumber'
        Certificate serial number.
        """
        ret = self._get_attr("serialNumber")
        return ret

    @property
    def signature_algorithm_oid(self):
        """Get str value for 'signatureAlgorithmOID'
        The dotted OID of the signature algorithm.
        """
        ret = self._get_attr("signatureAlgorithmOID")
        return ret

    @property
    def signature_algorithm_name(self):
        """Get str value for 'signatureAlgorithmName'
        The signature algorithm name if known (if known).
        """
        ret = self._get_attr("signatureAlgorithmName")
        return ret

    @property
    def issuer_name(self):
        """Get str value for 'issuerName'
        Issuer name.  Each member of the array is on the format
        COMPONENT=NAME, e.g. "C=DE", "ST=Example", "L=For Instance", "O=Beispiel GmbH",
        "CN=beispiel.example.org".
        """
        ret = self._get_attr("issuerName")
        return ret

    @property
    def subject_name(self):
        """Get str value for 'subjectName'
        Subject name.  Same format as issuerName.
        """
        ret = self._get_attr("subjectName")
        return ret

    @property
    def friendly_name(self):
        """Get str value for 'friendlyName'
        Friendly subject name or similar.
        """
        ret = self._get_attr("friendlyName")
        return ret

    @property
    def validity_period_not_before(self):
        """Get str value for 'validityPeriodNotBefore'
        Certificate not valid before ISO timestamp.
        """
        ret = self._get_attr("validityPeriodNotBefore")
        return ret

    @property
    def validity_period_not_after(self):
        """Get str value for 'validityPeriodNotAfter'
        Certificate not valid after ISO timestamp.
        """
        ret = self._get_attr("validityPeriodNotAfter")
        return ret

    @property
    def public_key_algorithm_oid(self):
        """Get str value for 'publicKeyAlgorithmOID'
        The dotted OID of the public key algorithm.
        """
        ret = self._get_attr("publicKeyAlgorithmOID")
        return ret

    @property
    def public_key_algorithm(self):
        """Get str value for 'publicKeyAlgorithm'
        The public key algorithm name (if known).
        """
        ret = self._get_attr("publicKeyAlgorithm")
        return ret

    @property
    def subject_public_key(self):
        """Get str value for 'subjectPublicKey'
        The raw public key bytes.
        """
        ret = self._get_attr("subjectPublicKey")
        return ret

    @property
    def issuer_unique_identifier(self):
        """Get str value for 'issuerUniqueIdentifier'
        Unique identifier of the issuer (empty string if not present).
        """
        ret = self._get_attr("issuerUniqueIdentifier")
        return ret

    @property
    def subject_unique_identifier(self):
        """Get str value for 'subjectUniqueIdentifier'
        Unique identifier of this certificate (empty string if not present).
        """
        ret = self._get_attr("subjectUniqueIdentifier")
        return ret

    @property
    def certificate_authority(self):
        """Get bool value for 'certificateAuthority'
        Whether this certificate is a certificate authority.  Will return E_FAIL
        if this attribute is not present.
        """
        ret = self._get_attr("certificateAuthority")
        return ret

    @property
    def key_usage(self):
        """Get int value for 'keyUsage'
        Key usage mask.  Will return 0 if not present.
        """
        ret = self._get_attr("keyUsage")
        return ret

    @property
    def extended_key_usage(self):
        """Get str value for 'extendedKeyUsage'
        Array of dotted extended key usage OIDs.  Empty array if not present.
        """
        ret = self._get_attr("extendedKeyUsage")
        return ret

    @property
    def raw_cert_data(self):
        """Get str value for 'rawCertData'
        The raw certificate bytes.
        """
        ret = self._get_attr("rawCertData")
        return ret

    @property
    def self_signed(self):
        """Get bool value for 'selfSigned'
        Set if self signed certificate.
        """
        ret = self._get_attr("selfSigned")
        return ret

    @property
    def trusted(self):
        """Get bool value for 'trusted'
        Set if the certificate is trusted (by the parent object).
        """
        ret = self._get_attr("trusted")
        return ret

    @property
    def expired(self):
        """Get bool value for 'expired'
        Set if the certificate has expired (relevant to the parent object)/
        """
        ret = self._get_attr("expired")
        return ret

    def is_currently_expired(self):
        """Tests if the certificate has expired at the present time according to
        the X.509 validity of the certificate.

        return result of type bool

        """
        result = self._call("isCurrentlyExpired")
        return result

    def query_info(self, what):
        """Way to extend the interface.

        in what of type int

        return result of type str

        """
        if not isinstance(what, baseinteger):
            raise TypeError("what can only be an instance of type baseinteger")
        result = self._call("queryInfo", in_p=[what])
        return result


class IAppliance(Interface):
    """
    Represents a platform-independent appliance in OVF format. An instance of this is returned
    by :py:func:`IVirtualBox.create_appliance` , which can then be used to import and export
    virtual machines within an appliance with VirtualBox.

    The OVF standard suggests two different physical file formats:


    If the appliance is distributed as a set of files, there must be at least one XML descriptor
    file that conforms to the OVF standard and carries an .ovf file extension. If
    this descriptor file references other files such as disk images, as OVF appliances typically
    do, those additional files must be in the same directory as the descriptor file.

    If the appliance is distributed as a single file, it must be in TAR format and have the
    .ova file extension. This TAR file must then contain at least the OVF descriptor
    files and optionally other files.

    At this time, VirtualBox does not not yet support the packed (TAR) variant; support will
    be added with a later version.


    **Importing** an OVF appliance into VirtualBox as instances of
    :py:class:`IMachine`  involves the following sequence of API calls:


    Call :py:func:`IVirtualBox.create_appliance` . This will create an empty IAppliance object.


    On the new object, call :py:func:`read`  with the full path of the OVF file you
    would like to import. So long as this file is syntactically valid, this will succeed
    and fill the appliance object with the parsed data from the OVF file.


    Next, call :py:func:`interpret` , which analyzes the OVF data and sets up the
    contents of the IAppliance attributes accordingly. These can be inspected by a
    VirtualBox front-end such as the GUI, and the suggestions can be displayed to the
    user. In particular, the :py:func:`virtual_system_descriptions`  array contains
    instances of :py:class:`IVirtualSystemDescription`  which represent the virtual
    systems (machines) in the OVF, which in turn describe the virtual hardware prescribed
    by the OVF (network and hardware adapters, virtual disk images, memory size and so on).
    The GUI can then give the user the option to confirm and/or change these suggestions.


    If desired, call :py:func:`IVirtualSystemDescription.set_final_values`  for each
    virtual system (machine) to override the suggestions made by the :py:func:`interpret`  routine.


    Finally, call :py:func:`import_machines`  to create virtual machines in
    VirtualBox as instances of :py:class:`IMachine`  that match the information in the
    virtual system descriptions. After this call succeeded, the UUIDs of the machines created
    can be found in the :py:func:`machines`  array attribute.



    **Exporting** VirtualBox machines into an OVF appliance involves the following steps:


    As with importing, first call :py:func:`IVirtualBox.create_appliance`  to create
    an empty IAppliance object.


    For each machine you would like to export, call :py:func:`IMachine.export_to`
    with the IAppliance object you just created. Each such call creates one instance of
    :py:class:`IVirtualSystemDescription`  inside the appliance.


    If desired, call :py:func:`IVirtualSystemDescription.set_final_values`  for each
    virtual system (machine) to override the suggestions made by the :py:func:`IMachine.export_to`  routine.


    Finally, call :py:func:`write`  with a path specification to have the OVF
    file written.
    """

    __uuid__ = "86a98347-7619-41aa-aece-b21ac5c1a7e6"
    __wsmap__ = "managed"

    @property
    def path(self):
        """Get str value for 'path'
        Path to the main file of the OVF appliance, which is either the .ovf or
        the .ova file passed to :py:func:`read`  (for import) or
        :py:func:`write`  (for export).
        This attribute is empty until one of these methods has been called.
        """
        ret = self._get_attr("path")
        return ret

    @property
    def disks(self):
        """Get str value for 'disks'
        Array of virtual disk definitions. One such description exists for each
        disk definition in the OVF; each string array item represents one such piece of
        disk information, with the information fields separated by tab (\\t) characters.

        The caller should be prepared for additional fields being appended to
        this string in future versions of VirtualBox and therefore check for
        the number of tabs in the strings returned.

        In the current version, the following eight fields are returned per string
        in the array:


        Disk ID (unique string identifier given to disk)

        Capacity (unsigned integer indicating the maximum capacity of the disk)

        Populated size (optional unsigned integer indicating the current size of the
        disk; can be approximate; -1 if unspecified)

        Format (string identifying the disk format, typically
        "http://www.vmware.com/specifications/vmdk.html#sparse")

        Reference (where to find the disk image, typically a file name; if empty,
        then the disk should be created on import)

        Image size (optional unsigned integer indicating the size of the image,
        which need not necessarily be the same as the values specified above, since
        the image may be compressed or sparse; -1 if not specified)

        Chunk size (optional unsigned integer if the image is split into chunks;
        presently unsupported and always -1)

        Compression (optional string equaling "gzip" if the image is gzip-compressed)
        """
        ret = self._get_attr("disks")
        return ret

    @property
    def virtual_system_descriptions(self):
        """Get IVirtualSystemDescription value for 'virtualSystemDescriptions'
        Array of virtual system descriptions. One such description is created
        for each virtual system (machine) found in the OVF.
        This array is empty until either :py:func:`interpret`  (for import) or :py:func:`IMachine.export_to`
        (for export) has been called.
        """
        ret = self._get_attr("virtualSystemDescriptions")
        return [IVirtualSystemDescription(a) for a in ret]

    @property
    def machines(self):
        """Get str value for 'machines'
        Contains the UUIDs of the machines created from the information in this appliances. This is only
        relevant for the import case, and will only contain data after a call to :py:func:`import_machines`
        succeeded.
        """
        ret = self._get_attr("machines")
        return ret

    @property
    def certificate(self):
        """Get ICertificate value for 'certificate'
        The X.509 signing certificate, if the imported OVF was signed, @c null
        if not signed.  This is available after calling :py:func:`read` .
        """
        ret = self._get_attr("certificate")
        return ICertificate(ret)

    def read(self, file_p):
        """Reads an OVF file into the appliance object.

        This method succeeds if the OVF is syntactically valid and, by itself, without errors. The
        mere fact that this method returns successfully does not mean that VirtualBox supports all
        features requested by the appliance; this can only be examined after a call to :py:func:`interpret` .

        in file_p of type str
            Name of appliance file to open (either with an .ovf or .ova extension, depending
            on whether the appliance is distributed as a set of files or as a single file, respectively).

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        """
        if not isinstance(file_p, basestring):
            raise TypeError("file_p can only be an instance of type basestring")
        progress = self._call("read", in_p=[file_p])
        progress = IProgress(progress)
        return progress

    def interpret(self):
        """Interprets the OVF data that was read when the appliance was constructed. After
        calling this method, one can inspect the
        :py:func:`virtual_system_descriptions`  array attribute, which will then contain
        one :py:class:`IVirtualSystemDescription`  for each virtual machine found in
        the appliance.

        Calling this method is the second step of importing an appliance into VirtualBox;
        see :py:class:`IAppliance`  for an overview.

        After calling this method, one should call :py:func:`get_warnings`  to find out
        if problems were encountered during the processing which might later lead to
        errors.

        """
        self._call("interpret")

    def import_machines(self, options):
        """Imports the appliance into VirtualBox by creating instances of :py:class:`IMachine`
        and other interfaces that match the information contained in the appliance as
        closely as possible, as represented by the import instructions in the
        :py:func:`virtual_system_descriptions`  array.

        Calling this method is the final step of importing an appliance into VirtualBox;
        see :py:class:`IAppliance`  for an overview.

        Since importing the appliance will most probably involve copying and converting
        disk images, which can take a long time, this method operates asynchronously and
        returns an IProgress object to allow the caller to monitor the progress.

        After the import succeeded, the UUIDs of the IMachine instances created can be
        retrieved from the :py:func:`machines`  array attribute.

        in options of type :class:`ImportOptions`
            Options for the importing operation.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        """
        if not isinstance(options, list):
            raise TypeError("options can only be an instance of type list")
        for a in options[:10]:
            if not isinstance(a, ImportOptions):
                raise TypeError("array can only contain objects of type ImportOptions")
        progress = self._call("importMachines", in_p=[options])
        progress = IProgress(progress)
        return progress

    def create_vfs_explorer(self, uri):
        """Returns a :py:class:`IVFSExplorer`  object for the given URI.

        in uri of type str
            The URI describing the file system to use.

        return explorer of type :class:`IVFSExplorer`

        """
        if not isinstance(uri, basestring):
            raise TypeError("uri can only be an instance of type basestring")
        explorer = self._call("createVFSExplorer", in_p=[uri])
        explorer = IVFSExplorer(explorer)
        return explorer

    def write(self, format_p, options, path):
        """Writes the contents of the appliance exports into a new OVF file.

        Calling this method is the final step of exporting an appliance from VirtualBox;
        see :py:class:`IAppliance`  for an overview.

        Since exporting the appliance will most probably involve copying and converting
        disk images, which can take a long time, this method operates asynchronously and
        returns an IProgress object to allow the caller to monitor the progress.

        in format_p of type str
            Output format, as a string. Currently supported formats are "ovf-0.9", "ovf-1.0",
            "ovf-2.0" and "opc-1.0"; future versions of VirtualBox may support additional formats.
            The "opc-1.0" format is for creating tarballs for the Oracle Public Cloud.

        in options of type :class:`ExportOptions`
            Options for the exporting operation.

        in path of type str
            Name of appliance file to create.  There are certain restrictions with regard
            to the file name suffix.  If the format parameter is "opc-1.0" a .tar.gz
            suffix is required.  Otherwise the suffix must either be .ovf or
            .ova, depending on whether the appliance is distributed as a set of
            files or as a single file, respectively.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        """
        if not isinstance(format_p, basestring):
            raise TypeError("format_p can only be an instance of type basestring")
        if not isinstance(options, list):
            raise TypeError("options can only be an instance of type list")
        for a in options[:10]:
            if not isinstance(a, ExportOptions):
                raise TypeError("array can only contain objects of type ExportOptions")
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        progress = self._call("write", in_p=[format_p, options, path])
        progress = IProgress(progress)
        return progress

    def get_warnings(self):
        """Returns textual warnings which occurred during execution of :py:func:`interpret` .

        return warnings of type str

        """
        warnings = self._call("getWarnings")
        return warnings

    def get_password_ids(self):
        """Returns a list of password identifiers which must be supplied to import or export
        encrypted virtual machines.

        return identifiers of type str
            The list of password identifiers required for export on success.

        """
        identifiers = self._call("getPasswordIds")
        return identifiers

    def get_medium_ids_for_password_id(self, password_id):
        """Returns a list of medium identifiers which use the given password identifier.

        in password_id of type str
            The password identifier to get the medium identifiers for.

        return identifiers of type str
            The list of medium identifiers returned on success.

        """
        if not isinstance(password_id, basestring):
            raise TypeError("password_id can only be an instance of type basestring")
        identifiers = self._call("getMediumIdsForPasswordId", in_p=[password_id])
        return identifiers

    def add_passwords(self, identifiers, passwords):
        """Adds a list of passwords required to import or export encrypted virtual
        machines.

        in identifiers of type str
            List of identifiers.

        in passwords of type str
            List of matching passwords.

        """
        if not isinstance(identifiers, list):
            raise TypeError("identifiers can only be an instance of type list")
        for a in identifiers[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(passwords, list):
            raise TypeError("passwords can only be an instance of type list")
        for a in passwords[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        self._call("addPasswords", in_p=[identifiers, passwords])

    def create_virtual_system_descriptions(self, requested):
        """Creates a number of :py:class:`IVirtualSystemDescription`  objects and store them
        in the :py:func:`virtual_system_descriptions`  array.

        in requested of type int
            Requested number of new virtual system description objects

        return created of type int
            Actually created number of virtual system description objects

        """
        if not isinstance(requested, baseinteger):
            raise TypeError("requested can only be an instance of type baseinteger")
        created = self._call("createVirtualSystemDescriptions", in_p=[requested])
        return created


class IVirtualSystemDescription(Interface):
    """
    Represents one virtual system (machine) in an appliance. This interface is used in
    the :py:func:`IAppliance.virtual_system_descriptions`  array. After
    :py:func:`IAppliance.interpret`  has been called, that array contains information
    about how the virtual systems described in the OVF should best be imported into
    VirtualBox virtual machines. See :py:class:`IAppliance`  for the steps required to
    import an OVF into VirtualBox.
    """

    __uuid__ = "01510f40-c196-4d26-b8db-4c8c389f1f82"
    __wsmap__ = "managed"

    @property
    def count(self):
        """Get int value for 'count'
        Return the number of virtual system description entries.
        """
        ret = self._get_attr("count")
        return ret

    def get_description(self):
        """Returns information about the virtual system as arrays of instruction items. In each array, the
        items with the same indices correspond and jointly represent an import instruction for VirtualBox.

        The list below identifies the value sets that are possible depending on the
        :py:class:`VirtualSystemDescriptionType`  enum value in the array item in @a aTypes[]. In each case,
        the array item with the same index in @a OVFValues[] will contain the original value as contained
        in the OVF file (just for informational purposes), and the corresponding item in @a aVBoxValues[]
        will contain a suggested value to be used for VirtualBox. Depending on the description type,
        the @a aExtraConfigValues[] array item may also be used.



        "OS": the guest operating system type. There must be exactly one such array item on import. The
        corresponding item in @a aVBoxValues[] contains the suggested guest operating system for VirtualBox.
        This will be one of the values listed in :py:func:`IVirtualBox.guest_os_types` . The corresponding
        item in @a OVFValues[] will contain a numerical value that described the operating system in the OVF.


        "Name": the name to give to the new virtual machine. There can be at most one such array item;
        if none is present on import, then an automatic name will be created from the operating system
        type. The corresponding item im @a OVFValues[] will contain the suggested virtual machine name
        from the OVF file, and @a aVBoxValues[] will contain a suggestion for a unique VirtualBox
        :py:class:`IMachine`  name that does not exist yet.


        "Description": an arbitrary description.


        "License": the EULA section from the OVF, if present. It is the responsibility of the calling
        code to display such a license for agreement; the Main API does not enforce any such policy.


        Miscellaneous: reserved for future use.


        "CPU": the number of CPUs. There can be at most one such item, which will presently be ignored.


        "Memory": the amount of guest RAM, in bytes. There can be at most one such array item; if none
        is present on import, then VirtualBox will set a meaningful default based on the operating system
        type.


        "HardDiskControllerIDE": an IDE hard disk controller. There can be at most two such items.
        An optional value in @a OVFValues[] and @a aVBoxValues[] can be "PIIX3" or "PIIX4" to specify
        the type of IDE controller; this corresponds to the ResourceSubType element which VirtualBox
        writes into the OVF.
        The matching item in the @a aRefs[] array will contain an integer that items of the "Harddisk"
        type can use to specify which hard disk controller a virtual disk should be connected to.
        Note that in OVF, an IDE controller has two channels, corresponding to "master" and "slave"
        in traditional terminology, whereas the IDE storage controller that VirtualBox supports in
        its virtual machines supports four channels (primary master, primary slave, secondary master,
        secondary slave) and thus maps to two IDE controllers in the OVF sense.


        "HardDiskControllerSATA": an SATA hard disk controller. There can be at most one such item. This
        has no value in @a OVFValues[] or @a aVBoxValues[].
        The matching item in the @a aRefs[] array will be used as with IDE controllers (see above).


        "HardDiskControllerSCSI": a SCSI hard disk controller. There can be at most one such item.
        The items in @a OVFValues[] and @a aVBoxValues[] will either be "LsiLogic", "BusLogic" or
        "LsiLogicSas". (Note that in OVF, the LsiLogicSas controller is treated as a SCSI controller
        whereas VirtualBox considers it a class of storage controllers of its own; see
        :py:class:`StorageControllerType` ).
        The matching item in the @a aRefs[] array will be used as with IDE controllers (see above).


        "HardDiskImage": a virtual hard disk, most probably as a reference to an image file. There can be an
        arbitrary number of these items, one for each virtual disk image that accompanies the OVF.

        The array item in @a OVFValues[] will contain the file specification from the OVF file (without
        a path since the image file should be in the same location as the OVF file itself), whereas the
        item in @a aVBoxValues[] will contain a qualified path specification to where VirtualBox uses the
        hard disk image. This means that on import the image will be copied and converted from the
        "ovf" location to the "vbox" location; on export, this will be handled the other way round.

        The matching item in the @a aExtraConfigValues[] array must contain a string of the following
        format: "controller=<index>;channel=<c>"
        In this string, <index> must be an integer specifying the hard disk controller to connect
        the image to. That number must be the index of an array item with one of the hard disk controller
        types (HardDiskControllerSCSI, HardDiskControllerSATA, HardDiskControllerIDE).
        In addition, <c> must specify the channel to use on that controller. For IDE controllers,
        this can be 0 or 1 for master or slave, respectively. For compatibility with VirtualBox versions
        before 3.2, the values 2 and 3 (for secondary master and secondary slave) are also supported, but
        no longer exported. For SATA and SCSI controllers, the channel can range from 0-29.


        "CDROM": a virtual CD-ROM drive. The matching item in @a aExtraConfigValue[] contains the same
        attachment information as with "HardDiskImage" items.


        "CDROM": a virtual floppy drive. The matching item in @a aExtraConfigValue[] contains the same
        attachment information as with "HardDiskImage" items.


        "NetworkAdapter": a network adapter. The array item in @a aVBoxValues[] will specify the hardware
        for the network adapter, whereas the array item in @a aExtraConfigValues[] will have a string
        of the "type=<X>" format, where <X> must be either "NAT" or "Bridged".


        "USBController": a USB controller. There can be at most one such item. If, and only if, such an
        item is present, USB support will be enabled for the new virtual machine.


        "SoundCard": a sound card. There can be at most one such item. If and only if such an item is
        present, sound support will be enabled for the new virtual machine. Note that the virtual
        machine in VirtualBox will always be presented with the standard VirtualBox soundcard, which
        may be different from the virtual soundcard expected by the appliance.

        out types of type :class:`VirtualSystemDescriptionType`

        out refs of type str

        out ovf_values of type str

        out v_box_values of type str

        out extra_config_values of type str

        """
        (types, refs, ovf_values, v_box_values, extra_config_values) = self._call(
            "getDescription"
        )
        types = [VirtualSystemDescriptionType(a) for a in types]
        return (types, refs, ovf_values, v_box_values, extra_config_values)

    def get_description_by_type(self, type_p):
        """This is the same as :py:func:`get_description`  except that you can specify which types
        should be returned.

        in type_p of type :class:`VirtualSystemDescriptionType`

        out types of type :class:`VirtualSystemDescriptionType`

        out refs of type str

        out ovf_values of type str

        out v_box_values of type str

        out extra_config_values of type str

        """
        if not isinstance(type_p, VirtualSystemDescriptionType):
            raise TypeError(
                "type_p can only be an instance of type VirtualSystemDescriptionType"
            )
        (types, refs, ovf_values, v_box_values, extra_config_values) = self._call(
            "getDescriptionByType", in_p=[type_p]
        )
        types = [VirtualSystemDescriptionType(a) for a in types]
        return (types, refs, ovf_values, v_box_values, extra_config_values)

    def remove_description_by_type(self, type_p):
        """Delete all records which are equal to the passed type from the list

        in type_p of type :class:`VirtualSystemDescriptionType`

        """
        if not isinstance(type_p, VirtualSystemDescriptionType):
            raise TypeError(
                "type_p can only be an instance of type VirtualSystemDescriptionType"
            )
        self._call("removeDescriptionByType", in_p=[type_p])

    def get_values_by_type(self, type_p, which):
        """This is the same as :py:func:`get_description_by_type`  except that you can specify which
        value types should be returned. See :py:class:`VirtualSystemDescriptionValueType`  for possible
        values.

        in type_p of type :class:`VirtualSystemDescriptionType`

        in which of type :class:`VirtualSystemDescriptionValueType`

        return values of type str

        """
        if not isinstance(type_p, VirtualSystemDescriptionType):
            raise TypeError(
                "type_p can only be an instance of type VirtualSystemDescriptionType"
            )
        if not isinstance(which, VirtualSystemDescriptionValueType):
            raise TypeError(
                "which can only be an instance of type VirtualSystemDescriptionValueType"
            )
        values = self._call("getValuesByType", in_p=[type_p, which])
        return values

    def set_final_values(self, enabled, v_box_values, extra_config_values):
        """This method allows the appliance's user to change the configuration for the virtual
        system descriptions. For each array item returned from :py:func:`get_description` ,
        you must pass in one boolean value and one configuration value.

        Each item in the boolean array determines whether the particular configuration item
        should be enabled.
        You can only disable items of the types HardDiskControllerIDE, HardDiskControllerSATA,
        HardDiskControllerSCSI, HardDiskImage, CDROM, Floppy, NetworkAdapter, USBController
        and SoundCard.

        For the "vbox" and "extra configuration" values, if you pass in the same arrays
        as returned in the aVBoxValues and aExtraConfigValues arrays from :py:func:`get_description` ,
        the configuration remains unchanged. Please see the documentation for :py:func:`get_description`
        for valid configuration values for the individual array item types. If the
        corresponding item in the aEnabled array is @c false, the configuration value is ignored.

        in enabled of type bool

        in v_box_values of type str

        in extra_config_values of type str

        """
        if not isinstance(enabled, list):
            raise TypeError("enabled can only be an instance of type list")
        for a in enabled[:10]:
            if not isinstance(a, bool):
                raise TypeError("array can only contain objects of type bool")
        if not isinstance(v_box_values, list):
            raise TypeError("v_box_values can only be an instance of type list")
        for a in v_box_values[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(extra_config_values, list):
            raise TypeError("extra_config_values can only be an instance of type list")
        for a in extra_config_values[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        self._call("setFinalValues", in_p=[enabled, v_box_values, extra_config_values])

    def add_description(self, type_p, v_box_value, extra_config_value):
        """This method adds an additional description entry to the stack of already
        available descriptions for this virtual system. This is handy for writing
        values which aren't directly supported by VirtualBox. One example would
        be the License type of :py:class:`VirtualSystemDescriptionType` .

        in type_p of type :class:`VirtualSystemDescriptionType`

        in v_box_value of type str

        in extra_config_value of type str

        """
        if not isinstance(type_p, VirtualSystemDescriptionType):
            raise TypeError(
                "type_p can only be an instance of type VirtualSystemDescriptionType"
            )
        if not isinstance(v_box_value, basestring):
            raise TypeError("v_box_value can only be an instance of type basestring")
        if not isinstance(extra_config_value, basestring):
            raise TypeError(
                "extra_config_value can only be an instance of type basestring"
            )
        self._call("addDescription", in_p=[type_p, v_box_value, extra_config_value])


class IUnattended(Interface):
    """
    The IUnattended interface represents the pipeline for preparing
    the Guest OS for fully automated install.

    The typical workflow is:

    Call :py:func:`IVirtualBox.create_unattended_installer`  to create the object
    Set :py:func:`IUnattended.iso_path`  and call :py:func:`IUnattended.detect_iso_os`
    Create, configure and register a machine according to :py:func:`IUnattended.detected_os_type_id`
    and the other detectedOS* attributes.
    Set :py:func:`IUnattended.machine`  to the new IMachine instance.
    Set the other IUnattended attributes as desired.
    Call :py:func:`IUnattended.prepare`  for the object to check the
    attribute values and create an internal installer instance.
    Call :py:func:`IUnattended.construct_media`  to create additional
    media files (ISO/floppy) needed.
    Call :py:func:`IUnattended.reconfigure_vm`  to reconfigure the VM
    with the installation ISO, additional media files and whatnot
    Optionally call :py:func:`IUnattended.done`  to destroy the internal
    installer and allow restarting from the second step.


    Note! Step two is currently not implemented.
    """

    __uuid__ = "6f89464f-7193-426c-a41f-522e8f537fa0"
    __wsmap__ = "managed"

    @property
    def iso_path(self):
        """Get or set str value for 'isoPath'
        Guest operating system ISO image
        """
        ret = self._get_attr("isoPath")
        return ret

    @iso_path.setter
    def iso_path(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("isoPath", value)

    @property
    def machine(self):
        """Get or set IMachine value for 'machine'
        The associated machine object.

        This must be set before :py:func:`IUnattended.prepare`  is called.
        The VM must be registered.
        """
        ret = self._get_attr("machine")
        return IMachine(ret)

    @machine.setter
    def machine(self, value):
        if not isinstance(value, IMachine):
            raise TypeError("value is not an instance of IMachine")
        return self._set_attr("machine", value)

    @property
    def user(self):
        """Get or set str value for 'user'
        Assign an user login name.
        """
        ret = self._get_attr("user")
        return ret

    @user.setter
    def user(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("user", value)

    @property
    def password(self):
        """Get or set str value for 'password'
        Assign a password to the user. The password is the same for both
        normal user and for Administrator / 'root' accounts.
        """
        ret = self._get_attr("password")
        return ret

    @password.setter
    def password(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("password", value)

    @property
    def full_user_name(self):
        """Get or set str value for 'fullUserName'
        The full name of the user.  This is optional and defaults to
        :py:func:`IUnattended.user` .  Please note that not all guests picks
        up this attribute.
        """
        ret = self._get_attr("fullUserName")
        return ret

    @full_user_name.setter
    def full_user_name(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("fullUserName", value)

    @property
    def product_key(self):
        """Get or set str value for 'productKey'
        Any key which is used as authorization of access to install genuine OS
        """
        ret = self._get_attr("productKey")
        return ret

    @product_key.setter
    def product_key(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("productKey", value)

    @property
    def additions_iso_path(self):
        """Get or set str value for 'additionsIsoPath'
        Guest Additions ISO image path.  This defaults to
        :py:func:`ISystemProperties.default_additions_iso`  when the Unattended
        object is instantiated.

        This property is ignored when :py:func:`IUnattended.install_guest_additions`  is false.
        """
        ret = self._get_attr("additionsIsoPath")
        return ret

    @additions_iso_path.setter
    def additions_iso_path(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("additionsIsoPath", value)

    @property
    def install_guest_additions(self):
        """Get or set bool value for 'installGuestAdditions'
        Indicates whether the Guest Additions should be installed or not.

        Setting this to false does not affect additions shipped with the linux
        distribution, only the installation of additions pointed to by
        :py:func:`IUnattended.additions_iso_path` .
        """
        ret = self._get_attr("installGuestAdditions")
        return ret

    @install_guest_additions.setter
    def install_guest_additions(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("installGuestAdditions", value)

    @property
    def validation_kit_iso_path(self):
        """Get or set str value for 'validationKitIsoPath'
        VirtualBox ValidationKit ISO image path.  This is used when
        :py:func:`IUnattended.install_test_exec_service`  is set to true.
        """
        ret = self._get_attr("validationKitIsoPath")
        return ret

    @validation_kit_iso_path.setter
    def validation_kit_iso_path(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("validationKitIsoPath", value)

    @property
    def install_test_exec_service(self):
        """Get or set bool value for 'installTestExecService'
        Indicates whether the test execution service (TXS) from the VBox
        ValidationKit should be installed.

        The TXS binary will be taken from the ISO indicated by
        :py:func:`IUnattended.validation_kit_iso_path` .
        """
        ret = self._get_attr("installTestExecService")
        return ret

    @install_test_exec_service.setter
    def install_test_exec_service(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("installTestExecService", value)

    @property
    def time_zone(self):
        """Get or set str value for 'timeZone'
        The guest time zone specifier.

        This is unfortunately guest OS specific.

        Windows XP and earlier takes the index number from this table:
        https://support.microsoft.com/en-gb/help/973627/microsoft-time-zone-index-values

        Windows Vista and later takes the time zone string from this table:
        https://technet.microsoft.com/en-us/library/cc749073(v=ws.10).aspx

        Linux usually takes the TZ string from this table:
        https://en.wikipedia.org/wiki/List_of_tz_database_time_zones

        The default is currently UTC/GMT, but this may change to be same as
        the host later.

        TODO: Investigate automatic mapping between linux and the two windows
        time zone formats.
        TODO: Take default from host (this requires mapping).
        """
        ret = self._get_attr("timeZone")
        return ret

    @time_zone.setter
    def time_zone(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("timeZone", value)

    @property
    def locale(self):
        """Get or set str value for 'locale'
        The 5 letter locale identifier, no codesets or such.

        The format is two lower case language letters (ISO 639-1), underscore ('_'),
        and two upper case country letters (ISO 3166-1 alpha-2).  For instance
        'en_US', 'de_DE', or 'ny_NO'.

        The default is taken from the host if possible, with 'en_US' as fallback.
        """
        ret = self._get_attr("locale")
        return ret

    @locale.setter
    def locale(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("locale", value)

    @property
    def language(self):
        """Get or set str value for 'language'
        This is more or less a Windows specific setting for choosing the UI language
        setting of the installer.

        The value should be from the list availble via :py:func:`IUnattended.detected_os_languages` .
        The typical format is {language-code}-{COUNTRY} but windows may also use
        {16-bit code}:{32-bit code} or insert another component between the language
        and country codes.  We consider the format guest OS specific.

        Note that it is crucial that this is correctly specified for Windows
        installations.  If an unsupported value is given the installer will ask
        for an installation language and wait for user input.  Best to leave it
        to the default value.

        The default is the first one from :py:func:`IUnattended.detected_os_languages` .
        """
        ret = self._get_attr("language")
        return ret

    @language.setter
    def language(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("language", value)

    @property
    def country(self):
        """Get or set str value for 'country'
        The 2 upper case letter country identifier, ISO 3166-1 alpha-2.

        This is used for mirrors and such.

        The default is taken from the host when possible, falling back on
        :py:func:`IUnattended.locale` .
        """
        ret = self._get_attr("country")
        return ret

    @country.setter
    def country(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("country", value)

    @property
    def proxy(self):
        """Get or set str value for 'proxy'
        Proxy incantation to pass on to the guest OS installer.

        This is important to get right if the guest OS installer is of the type
        that goes online to fetch the packages (e.g. debian-*-netinst.iso) or
        to fetch updates during the install process.

        Format: [schema=]schema://[login@password:]proxy[:port][;...]

        The default is taken from the host proxy configuration (once implemented).
        """
        ret = self._get_attr("proxy")
        return ret

    @proxy.setter
    def proxy(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("proxy", value)

    @property
    def package_selection_adjustments(self):
        """Get or set str value for 'packageSelectionAdjustments'
        Guest OS specific package selection adjustments.

        This is a semicolon separated list of keywords, and later maybe guest OS
        package specifiers.  Currently the 'minimal' is the only recognized value,
        and this only works with a selection of linux installers.
        """
        ret = self._get_attr("packageSelectionAdjustments")
        return ret

    @package_selection_adjustments.setter
    def package_selection_adjustments(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("packageSelectionAdjustments", value)

    @property
    def hostname(self):
        """Get or set str value for 'hostname'
        The fully qualified guest hostname.

        This defaults to machine-name + ".myguest.virtualbox.org", though it may
        change to the host domain name later.
        """
        ret = self._get_attr("hostname")
        return ret

    @hostname.setter
    def hostname(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("hostname", value)

    @property
    def auxiliary_base_path(self):
        """Get or set str value for 'auxiliaryBasePath'
        The path + basename for auxiliary files generated by the unattended
        installation.  This defaults to the VM folder + Unattended + VM UUID.

        The files which gets generated depends on the OS being installed.  When
        installing Windows there is currently only a auxiliaryBasePath + "floppy.img"
        being created.  But for linux, a "cdrom.viso" and one or more configuration
        files are generate generated.
        """
        ret = self._get_attr("auxiliaryBasePath")
        return ret

    @auxiliary_base_path.setter
    def auxiliary_base_path(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("auxiliaryBasePath", value)

    @property
    def image_index(self):
        """Get or set int value for 'imageIndex'
        The image index on installation CD/DVD used to install.

        Used only with Windows installation CD/DVD:
        https://technet.microsoft.com/en-us/library/cc766022%28v=ws.10%29.aspx
        """
        ret = self._get_attr("imageIndex")
        return ret

    @image_index.setter
    def image_index(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("imageIndex", value)

    @property
    def script_template_path(self):
        """Get or set str value for 'scriptTemplatePath'
        The unattended installation script template file.

        The template default is based on the guest OS type and is determined by the
        internal installer when when :py:func:`IUnattended.prepare`  is invoked.
        Most users will want the defaults.

        After :py:func:`IUnattended.prepare`  is called, it can be read to see
        which file is being used.
        """
        ret = self._get_attr("scriptTemplatePath")
        return ret

    @script_template_path.setter
    def script_template_path(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("scriptTemplatePath", value)

    @property
    def post_install_script_template_path(self):
        """Get or set str value for 'postInstallScriptTemplatePath'
        The post installation (shell/batch) script template file.

        The template default is based on the guest OS type and is determined by the
        internal installer when when :py:func:`IUnattended.prepare`  is invoked.
        Most users will want the defaults.

        After :py:func:`IUnattended.prepare`  is called, it can be read to see
        which file is being used.
        """
        ret = self._get_attr("postInstallScriptTemplatePath")
        return ret

    @post_install_script_template_path.setter
    def post_install_script_template_path(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("postInstallScriptTemplatePath", value)

    @property
    def post_install_command(self):
        """Get or set str value for 'postInstallCommand'
        Custom post installation command.

        Exactly what is expected as input here depends on the guest OS installer
        and the post installation script template (see
        :py:func:`IUnattended.post_install_script_template_path` ).
        Most users will not need to set this attribute.
        """
        ret = self._get_attr("postInstallCommand")
        return ret

    @post_install_command.setter
    def post_install_command(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("postInstallCommand", value)

    @property
    def extra_install_kernel_parameters(self):
        """Get or set str value for 'extraInstallKernelParameters'
        Extra kernel arguments passed to the install kernel of some guests.

        This is currently only picked up by linux guests.  The exact parameters
        are specific to the guest OS being installed of course.

        After :py:func:`IUnattended.prepare`  is called, it can be read to see
        which parameters are being used.
        """
        ret = self._get_attr("extraInstallKernelParameters")
        return ret

    @extra_install_kernel_parameters.setter
    def extra_install_kernel_parameters(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("extraInstallKernelParameters", value)

    @property
    def detected_os_type_id(self):
        """Get str value for 'detectedOSTypeId'
        The detected OS type ID (:py:func:`IGuestOSType.id_p` ).

        Set by :py:func:`IUnattended.detect_iso_os`  or :py:func:`IUnattended.prepare` .

        Not yet implemented.
        """
        ret = self._get_attr("detectedOSTypeId")
        return ret

    @property
    def detected_os_version(self):
        """Get str value for 'detectedOSVersion'
        The detected OS version string.

        Set by :py:func:`IUnattended.detect_iso_os`  or :py:func:`IUnattended.prepare` .

        Not yet implemented.
        """
        ret = self._get_attr("detectedOSVersion")
        return ret

    @property
    def detected_os_flavor(self):
        """Get str value for 'detectedOSFlavor'
        The detected OS flavor (e.g. server, desktop, etc)

        Set by :py:func:`IUnattended.detect_iso_os`  or :py:func:`IUnattended.prepare` .

        Not yet implemented.
        """
        ret = self._get_attr("detectedOSFlavor")
        return ret

    @property
    def detected_os_languages(self):
        """Get str value for 'detectedOSLanguages'
        The space separated list of (Windows) installation UI languages we detected (lang.ini).

        The language specifier format is specific to the guest OS.  They are
        used to set :py:func:`IUnattended.language` .

        Set by :py:func:`IUnattended.detect_iso_os`  or :py:func:`IUnattended.prepare` .

        Partially implemented.
        """
        ret = self._get_attr("detectedOSLanguages")
        return ret

    @property
    def detected_os_hints(self):
        """Get str value for 'detectedOSHints'
        Space separated list of other stuff detected about the OS and the
        installation ISO.

        Set by :py:func:`IUnattended.detect_iso_os`  or :py:func:`IUnattended.prepare` .

        Not yet implemented.
        """
        ret = self._get_attr("detectedOSHints")
        return ret

    def detect_iso_os(self):
        """Detects the OS on the ISO given by :py:func:`IUnattended.iso_path`  and sets
        :py:func:`IUnattended.detected_os_type_id` , :py:func:`IUnattended.detected_os_version`
        :py:func:`IUnattended.detected_os_flavor` , :py:func:`IUnattended.detected_os_languages` ,
        and :py:func:`IUnattended.detected_os_hints` .

        Not really yet implemented.

        """
        self._call("detectIsoOS")

    def prepare(self):
        """Prepare for running the unattended process of installation.

        This will instantiate the installer based on the guest type associated
        with the machine (see :py:func:`IMachine.os_type_id` ).  It will also
        perform :py:func:`IUnattended.detect_iso_os`  if not yet called on the
        current :py:func:`IUnattended.iso_path`  value.

        """
        self._call("prepare")

    def construct_media(self):
        """Constructors the necessary ISO/VISO/Floppy images, with unattended scripts
        and all necessary bits on them.

        """
        self._call("constructMedia")

    def reconfigure_vm(self):
        """Reconfigures the machine to start the installation.

        This involves mounting the ISOs and floppy images created by
        :py:func:`IUnattended.construct_media` , attaching new DVD and floppy
        drives as necessary, and possibly modifying the boot order.

        """
        self._call("reconfigureVM")

    def done(self):
        """Done - time to start the VM.

        This deletes the internal installer instance that :py:func:`IUnattended.prepare`
        created.  Before done() is called, it is not possible to start over again
        from :py:func:`IUnattended.prepare` .

        """
        self._call("done")


class IInternalMachineControl(Interface):
    """
    Updates the VM state.

    This operation will also update the settings file with the correct
    information about the saved state file and delete this file from disk
    when appropriate.
    """

    __uuid__ = "0075FD6C-00C2-4484-0077-C057003D9C90"
    __wsmap__ = "suppress"

    def update_state(self, state):
        """Updates the VM state.

        This operation will also update the settings file with the correct
        information about the saved state file and delete this file from disk
        when appropriate.

        in state of type :class:`MachineState`

        """
        if not isinstance(state, MachineState):
            raise TypeError("state can only be an instance of type MachineState")
        self._call("updateState", in_p=[state])

    def begin_power_up(self, progress):
        """Tells VBoxSVC that :py:func:`IConsole.power_up`  is under ways and
        gives it the progress object that should be part of any pending
        :py:func:`IMachine.launch_vm_process`  operations. The progress
        object may be called back to reflect an early cancelation, so some care
        have to be taken with respect to any cancelation callbacks. The console
        object will call :py:func:`IInternalMachineControl.end_power_up`
        to signal the completion of the progress object.

        in progress of type :class:`IProgress`

        """
        if not isinstance(progress, IProgress):
            raise TypeError("progress can only be an instance of type IProgress")
        self._call("beginPowerUp", in_p=[progress])

    def end_power_up(self, result):
        """Tells VBoxSVC that :py:func:`IConsole.power_up`  has completed.
        This method may query status information from the progress object it
        received in :py:func:`IInternalMachineControl.begin_power_up`  and copy
        it over to any in-progress :py:func:`IMachine.launch_vm_process`
        call in order to complete that progress object.

        in result of type int

        """
        if not isinstance(result, baseinteger):
            raise TypeError("result can only be an instance of type baseinteger")
        self._call("endPowerUp", in_p=[result])

    def begin_powering_down(self):
        """Called by the VM process to inform the server it wants to
        stop the VM execution and power down.

        out progress of type :class:`IProgress`
            Progress object created by VBoxSVC to wait until
            the VM is powered down.

        """
        progress = self._call("beginPoweringDown")
        progress = IProgress(progress)
        return progress

    def end_powering_down(self, result, err_msg):
        """Called by the VM process to inform the server that powering
        down previously requested by #beginPoweringDown is either
        successfully finished or there was a failure.

        in result of type int
            @c S_OK to indicate success.

        in err_msg of type str
            @c human readable error message in case of failure.

        raises :class:`VBoxErrorFileError`
            Settings file not accessible.

        raises :class:`VBoxErrorXmlError`
            Could not parse the settings file.

        """
        if not isinstance(result, baseinteger):
            raise TypeError("result can only be an instance of type baseinteger")
        if not isinstance(err_msg, basestring):
            raise TypeError("err_msg can only be an instance of type basestring")
        self._call("endPoweringDown", in_p=[result, err_msg])

    def run_usb_device_filters(self, device):
        """Asks the server to run USB devices filters of the associated
        machine against the given USB device and tell if there is
        a match.

        Intended to be used only for remote USB devices. Local
        ones don't require to call this method (this is done
        implicitly by the Host and USBProxyService).

        in device of type :class:`IUSBDevice`

        out matched of type bool

        out masked_interfaces of type int

        """
        if not isinstance(device, IUSBDevice):
            raise TypeError("device can only be an instance of type IUSBDevice")
        (matched, masked_interfaces) = self._call("runUSBDeviceFilters", in_p=[device])
        return (matched, masked_interfaces)

    def capture_usb_device(self, id_p, capture_filename):
        """Requests a capture of the given host USB device.
        When the request is completed, the VM process will
        get a :py:func:`IInternalSessionControl.on_usb_device_attach`
        notification.

        in id_p of type str

        in capture_filename of type str

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        if not isinstance(capture_filename, basestring):
            raise TypeError(
                "capture_filename can only be an instance of type basestring"
            )
        self._call("captureUSBDevice", in_p=[id_p, capture_filename])

    def detach_usb_device(self, id_p, done):
        """Notification that a VM is going to detach (@a done = @c false) or has
        already detached (@a done = @c true) the given USB device.
        When the @a done = @c true request is completed, the VM process will
        get a :py:func:`IInternalSessionControl.on_usb_device_detach`
        notification.

        In the @a done = @c true case, the server must run its own filters
        and filters of all VMs but this one on the detached device
        as if it were just attached to the host computer.

        in id_p of type str

        in done of type bool

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        if not isinstance(done, bool):
            raise TypeError("done can only be an instance of type bool")
        self._call("detachUSBDevice", in_p=[id_p, done])

    def auto_capture_usb_devices(self):
        """Requests a capture all matching USB devices attached to the host.
        When the request is completed, the VM process will
        get a :py:func:`IInternalSessionControl.on_usb_device_attach`
        notification per every captured device.

        """
        self._call("autoCaptureUSBDevices")

    def detach_all_usb_devices(self, done):
        """Notification that a VM that is being powered down. The done
        parameter indicates whether which stage of the power down
        we're at. When @a done = @c false the VM is announcing its
        intentions, while when @a done = @c true the VM is reporting
        what it has done.

        In the @a done = @c true case, the server must run its own filters
        and filters of all VMs but this one on all detach devices as
        if they were just attached to the host computer.

        in done of type bool

        """
        if not isinstance(done, bool):
            raise TypeError("done can only be an instance of type bool")
        self._call("detachAllUSBDevices", in_p=[done])

    def on_session_end(self, session):
        """Triggered by the given session object when the session is about
        to close normally.

        in session of type :class:`ISession`
            Session that is being closed

        return progress of type :class:`IProgress`
            Used to wait until the corresponding machine is actually
            dissociated from the given session on the server.
            Returned only when this session is a direct one.

        """
        if not isinstance(session, ISession):
            raise TypeError("session can only be an instance of type ISession")
        progress = self._call("onSessionEnd", in_p=[session])
        progress = IProgress(progress)
        return progress

    def finish_online_merge_medium(self):
        """Gets called by :py:func:`IInternalSessionControl.online_merge_medium` .
        All necessary state information is available at the called object.

        """
        self._call("finishOnlineMergeMedium")

    def pull_guest_properties(self):
        """Get the list of the guest properties matching a set of patterns along
        with their values, timestamps and flags and give responsibility for
        managing properties to the console.

        out names of type str
            The names of the properties returned.

        out values of type str
            The values of the properties returned. The array entries match the
            corresponding entries in the @a name array.

        out timestamps of type int
            The timestamps of the properties returned. The array entries match
            the corresponding entries in the @a name array.

        out flags of type str
            The flags of the properties returned. The array entries match the
            corresponding entries in the @a name array.

        """
        (names, values, timestamps, flags) = self._call("pullGuestProperties")
        return (names, values, timestamps, flags)

    def clipboard_area_register(self, parms):
        """Registers a new clipboard area.

        in parms of type str
            Registration parameters. Currently not used and therefore ignored.

        out id_p of type int
            Returns the new clipboard area which got registered.

        """
        if not isinstance(parms, list):
            raise TypeError("parms can only be an instance of type list")
        for a in parms[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        id_p = self._call("clipboardAreaRegister", in_p=[parms])
        return id_p

    def clipboard_area_unregister(self, id_p):
        """Unregisters a formerly registered clipboard area.

        in id_p of type int
            Clipboard area to unregister.

        """
        if not isinstance(id_p, baseinteger):
            raise TypeError("id_p can only be an instance of type baseinteger")
        self._call("clipboardAreaUnregister", in_p=[id_p])

    def clipboard_area_attach(self, id_p):
        """Attaches to a registered clipboard area.

        in id_p of type int
            Clipboard area to attach to.

        """
        if not isinstance(id_p, baseinteger):
            raise TypeError("id_p can only be an instance of type baseinteger")
        self._call("clipboardAreaAttach", in_p=[id_p])

    def clipboard_area_detach(self, id_p):
        """Detaches from a registered clipboard area.

        in id_p of type int
            Clipboard area to detach from.

        """
        if not isinstance(id_p, baseinteger):
            raise TypeError("id_p can only be an instance of type baseinteger")
        self._call("clipboardAreaDetach", in_p=[id_p])

    def clipboard_area_get_most_recent(self):
        """Returns the most recent (last registered) clipboard area.

        out id_p of type int
            Returns the most recent clipboard area.

        """
        id_p = self._call("clipboardAreaGetMostRecent")
        return id_p

    def clipboard_area_get_ref_count(self, id_p):
        """Returns the current reference count of a clipboard area.

        in id_p of type int
            Clipboard area to return reference count for.

        out refcount of type int
            Returns the current reference count.

        """
        if not isinstance(id_p, baseinteger):
            raise TypeError("id_p can only be an instance of type baseinteger")
        refcount = self._call("clipboardAreaGetRefCount", in_p=[id_p])
        return refcount

    def push_guest_property(self, name, value, timestamp, flags):
        """Update a single guest property in IMachine.

        in name of type str
            The name of the property to be updated.

        in value of type str
            The value of the property.

        in timestamp of type int
            The timestamp of the property.

        in flags of type str
            The flags of the property.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(value, basestring):
            raise TypeError("value can only be an instance of type basestring")
        if not isinstance(timestamp, baseinteger):
            raise TypeError("timestamp can only be an instance of type baseinteger")
        if not isinstance(flags, basestring):
            raise TypeError("flags can only be an instance of type basestring")
        self._call("pushGuestProperty", in_p=[name, value, timestamp, flags])

    def lock_media(self):
        """Locks all media attached to the machine for writing and parents of
        attached differencing media (if any) for reading. This operation is
        atomic so that if it fails no media is actually locked.

        This method is intended to be called when the machine is in Starting or
        Restoring state. The locked media will be automatically unlocked when
        the machine is powered off or crashed.

        """
        self._call("lockMedia")

    def unlock_media(self):
        """Unlocks all media previously locked using
        :py:func:`IInternalMachineControl.lock_media` .

        This method is intended to be used with teleportation so that it is
        possible to teleport between processes on the same machine.

        """
        self._call("unlockMedia")

    def eject_medium(self, attachment):
        """Tells VBoxSVC that the guest has ejected the medium associated with
        the medium attachment.

        in attachment of type :class:`IMediumAttachment`
            The medium attachment where the eject happened.

        return new_attachment of type :class:`IMediumAttachment`
            A new reference to the medium attachment, as the config change can
            result in the creation of a new instance.

        """
        if not isinstance(attachment, IMediumAttachment):
            raise TypeError(
                "attachment can only be an instance of type IMediumAttachment"
            )
        new_attachment = self._call("ejectMedium", in_p=[attachment])
        new_attachment = IMediumAttachment(new_attachment)
        return new_attachment

    def report_vm_statistics(
        self,
        valid_stats,
        cpu_user,
        cpu_kernel,
        cpu_idle,
        mem_total,
        mem_free,
        mem_balloon,
        mem_shared,
        mem_cache,
        paged_total,
        mem_alloc_total,
        mem_free_total,
        mem_balloon_total,
        mem_shared_total,
        vm_net_rx,
        vm_net_tx,
    ):
        """Passes statistics collected by VM (including guest statistics) to VBoxSVC.

        in valid_stats of type int
            Mask defining which parameters are valid. For example: 0x11 means
            that cpuIdle and XXX are valid. Other parameters should be ignored.

        in cpu_user of type int
            Percentage of processor time spent in user mode as seen by the guest.

        in cpu_kernel of type int
            Percentage of processor time spent in kernel mode as seen by the guest.

        in cpu_idle of type int
            Percentage of processor time spent idling as seen by the guest.

        in mem_total of type int
            Total amount of physical guest RAM.

        in mem_free of type int
            Free amount of physical guest RAM.

        in mem_balloon of type int
            Amount of ballooned physical guest RAM.

        in mem_shared of type int
            Amount of shared physical guest RAM.

        in mem_cache of type int
            Total amount of guest (disk) cache memory.

        in paged_total of type int
            Total amount of space in the page file.

        in mem_alloc_total of type int
            Total amount of memory allocated by the hypervisor.

        in mem_free_total of type int
            Total amount of free memory available in the hypervisor.

        in mem_balloon_total of type int
            Total amount of memory ballooned by the hypervisor.

        in mem_shared_total of type int
            Total amount of shared memory in the hypervisor.

        in vm_net_rx of type int
            Network receive rate for VM.

        in vm_net_tx of type int
            Network transmit rate for VM.

        """
        if not isinstance(valid_stats, baseinteger):
            raise TypeError("valid_stats can only be an instance of type baseinteger")
        if not isinstance(cpu_user, baseinteger):
            raise TypeError("cpu_user can only be an instance of type baseinteger")
        if not isinstance(cpu_kernel, baseinteger):
            raise TypeError("cpu_kernel can only be an instance of type baseinteger")
        if not isinstance(cpu_idle, baseinteger):
            raise TypeError("cpu_idle can only be an instance of type baseinteger")
        if not isinstance(mem_total, baseinteger):
            raise TypeError("mem_total can only be an instance of type baseinteger")
        if not isinstance(mem_free, baseinteger):
            raise TypeError("mem_free can only be an instance of type baseinteger")
        if not isinstance(mem_balloon, baseinteger):
            raise TypeError("mem_balloon can only be an instance of type baseinteger")
        if not isinstance(mem_shared, baseinteger):
            raise TypeError("mem_shared can only be an instance of type baseinteger")
        if not isinstance(mem_cache, baseinteger):
            raise TypeError("mem_cache can only be an instance of type baseinteger")
        if not isinstance(paged_total, baseinteger):
            raise TypeError("paged_total can only be an instance of type baseinteger")
        if not isinstance(mem_alloc_total, baseinteger):
            raise TypeError(
                "mem_alloc_total can only be an instance of type baseinteger"
            )
        if not isinstance(mem_free_total, baseinteger):
            raise TypeError(
                "mem_free_total can only be an instance of type baseinteger"
            )
        if not isinstance(mem_balloon_total, baseinteger):
            raise TypeError(
                "mem_balloon_total can only be an instance of type baseinteger"
            )
        if not isinstance(mem_shared_total, baseinteger):
            raise TypeError(
                "mem_shared_total can only be an instance of type baseinteger"
            )
        if not isinstance(vm_net_rx, baseinteger):
            raise TypeError("vm_net_rx can only be an instance of type baseinteger")
        if not isinstance(vm_net_tx, baseinteger):
            raise TypeError("vm_net_tx can only be an instance of type baseinteger")
        self._call(
            "reportVmStatistics",
            in_p=[
                valid_stats,
                cpu_user,
                cpu_kernel,
                cpu_idle,
                mem_total,
                mem_free,
                mem_balloon,
                mem_shared,
                mem_cache,
                paged_total,
                mem_alloc_total,
                mem_free_total,
                mem_balloon_total,
                mem_shared_total,
                vm_net_rx,
                vm_net_tx,
            ],
        )

    def authenticate_external(self, auth_params):
        """Verify credentials using the external auth library.

        in auth_params of type str
            The auth parameters, credentials, etc.

        out result of type str
            The authentification result.

        """
        if not isinstance(auth_params, list):
            raise TypeError("auth_params can only be an instance of type list")
        for a in auth_params[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        result = self._call("authenticateExternal", in_p=[auth_params])
        return result


class IGraphicsAdapter(Interface):
    """
    The IGraphicsAdapter interface represents the graphics adapter
    of the virtual machine.
    """

    __uuid__ = "f692806f-febe-4049-b476-1292a8e45b09"
    __wsmap__ = "managed"

    @property
    def graphics_controller_type(self):
        """Get or set GraphicsControllerType value for 'graphicsControllerType'
        Graphics controller type.
        """
        ret = self._get_attr("graphicsControllerType")
        return GraphicsControllerType(ret)

    @graphics_controller_type.setter
    def graphics_controller_type(self, value):
        if not isinstance(value, GraphicsControllerType):
            raise TypeError("value is not an instance of GraphicsControllerType")
        return self._set_attr("graphicsControllerType", value)

    @property
    def vram_size(self):
        """Get or set int value for 'VRAMSize'
        Video memory size in megabytes.
        """
        ret = self._get_attr("VRAMSize")
        return ret

    @vram_size.setter
    def vram_size(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("VRAMSize", value)

    @property
    def accelerate3_d_enabled(self):
        """Get or set bool value for 'accelerate3DEnabled'
        This setting determines whether VirtualBox allows this machine to make
        use of the 3D graphics support available on the host.
        """
        ret = self._get_attr("accelerate3DEnabled")
        return ret

    @accelerate3_d_enabled.setter
    def accelerate3_d_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("accelerate3DEnabled", value)

    @property
    def accelerate2_d_video_enabled(self):
        """Get or set bool value for 'accelerate2DVideoEnabled'
        This setting determines whether VirtualBox allows this machine to make
        use of the 2D video acceleration support available on the host.
        """
        ret = self._get_attr("accelerate2DVideoEnabled")
        return ret

    @accelerate2_d_video_enabled.setter
    def accelerate2_d_video_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("accelerate2DVideoEnabled", value)

    @property
    def monitor_count(self):
        """Get or set int value for 'monitorCount'
        Number of virtual monitors.

        Only effective on Windows XP and later guests with
        Guest Additions installed.
        """
        ret = self._get_attr("monitorCount")
        return ret

    @monitor_count.setter
    def monitor_count(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("monitorCount", value)


class IBIOSSettings(Interface):
    """
    The IBIOSSettings interface represents BIOS settings of the virtual
    machine. This is used only in the :py:func:`IMachine.bios_settings`  attribute.
    """

    __uuid__ = "73af4152-7e67-4144-bf34-41c38e8b4cc7"
    __wsmap__ = "managed"

    @property
    def logo_fade_in(self):
        """Get or set bool value for 'logoFadeIn'
        Fade in flag for BIOS logo animation.
        """
        ret = self._get_attr("logoFadeIn")
        return ret

    @logo_fade_in.setter
    def logo_fade_in(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("logoFadeIn", value)

    @property
    def logo_fade_out(self):
        """Get or set bool value for 'logoFadeOut'
        Fade out flag for BIOS logo animation.
        """
        ret = self._get_attr("logoFadeOut")
        return ret

    @logo_fade_out.setter
    def logo_fade_out(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("logoFadeOut", value)

    @property
    def logo_display_time(self):
        """Get or set int value for 'logoDisplayTime'
        BIOS logo display time in milliseconds (0 = default).
        """
        ret = self._get_attr("logoDisplayTime")
        return ret

    @logo_display_time.setter
    def logo_display_time(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("logoDisplayTime", value)

    @property
    def logo_image_path(self):
        """Get or set str value for 'logoImagePath'
        Local file system path for external BIOS splash image. Empty string
        means the default image is shown on boot.
        """
        ret = self._get_attr("logoImagePath")
        return ret

    @logo_image_path.setter
    def logo_image_path(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("logoImagePath", value)

    @property
    def boot_menu_mode(self):
        """Get or set BIOSBootMenuMode value for 'bootMenuMode'
        Mode of the BIOS boot device menu.
        """
        ret = self._get_attr("bootMenuMode")
        return BIOSBootMenuMode(ret)

    @boot_menu_mode.setter
    def boot_menu_mode(self, value):
        if not isinstance(value, BIOSBootMenuMode):
            raise TypeError("value is not an instance of BIOSBootMenuMode")
        return self._set_attr("bootMenuMode", value)

    @property
    def acpi_enabled(self):
        """Get or set bool value for 'ACPIEnabled'
        ACPI support flag.
        """
        ret = self._get_attr("ACPIEnabled")
        return ret

    @acpi_enabled.setter
    def acpi_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("ACPIEnabled", value)

    @property
    def ioapic_enabled(self):
        """Get or set bool value for 'IOAPICEnabled'
        I/O-APIC support flag. If set, VirtualBox will provide an I/O-APIC
        and support IRQs above 15.
        """
        ret = self._get_attr("IOAPICEnabled")
        return ret

    @ioapic_enabled.setter
    def ioapic_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("IOAPICEnabled", value)

    @property
    def apic_mode(self):
        """Get or set APICMode value for 'APICMode'
        APIC mode to set up by the firmware.
        """
        ret = self._get_attr("APICMode")
        return APICMode(ret)

    @apic_mode.setter
    def apic_mode(self, value):
        if not isinstance(value, APICMode):
            raise TypeError("value is not an instance of APICMode")
        return self._set_attr("APICMode", value)

    @property
    def time_offset(self):
        """Get or set int value for 'timeOffset'
        Offset in milliseconds from the host system time. This allows for
        guests running with a different system date/time than the host.
        It is equivalent to setting the system date/time in the BIOS except
        it is not an absolute value but a relative one. Guest Additions
        time synchronization honors this offset.
        """
        ret = self._get_attr("timeOffset")
        return ret

    @time_offset.setter
    def time_offset(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("timeOffset", value)

    @property
    def pxe_debug_enabled(self):
        """Get or set bool value for 'PXEDebugEnabled'
        PXE debug logging flag. If set, VirtualBox will write extensive
        PXE trace information to the release log.
        """
        ret = self._get_attr("PXEDebugEnabled")
        return ret

    @pxe_debug_enabled.setter
    def pxe_debug_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("PXEDebugEnabled", value)

    @property
    def non_volatile_storage_file(self):
        """Get str value for 'nonVolatileStorageFile'
        The location of the file storing the non-volatile memory content when
        the VM is powered off.  The file does not always exist.
        """
        ret = self._get_attr("nonVolatileStorageFile")
        return ret

    @property
    def smbios_uuid_little_endian(self):
        """Get or set bool value for 'SMBIOSUuidLittleEndian'
        Flag to control whether the SMBIOS system UUID is presented in little endian
        form to the guest as mandated by the SMBIOS spec chapter 7.2.1.
        Before VirtualBox version 6.1 it was always presented in big endian form
        and to retain the old behavior this flag was introduced so it can be changed.
        VMs created with VBox 6.1 will default to true for this flag.
        """
        ret = self._get_attr("SMBIOSUuidLittleEndian")
        return ret

    @smbios_uuid_little_endian.setter
    def smbios_uuid_little_endian(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("SMBIOSUuidLittleEndian", value)


class IRecordingScreenSettings(Interface):
    """
    The IRecordingScreenSettings interface represents recording settings of a
    single virtual screen. This is used only in the :py:class:`IRecordingSettings`
    interface.
    """

    __uuid__ = "678fbd9a-93af-42a7-7f13-79ad6ef1a18d"
    __wsmap__ = "managed"

    def is_feature_enabled(self, feature):
        """Returns whether a particular recording feature is enabled for this
        screen or not.

        in feature of type :class:`RecordingFeature`
            Feature to check for.

        return enabled of type bool
            @c true if the feature is enabled, @c false if not.

        """
        if not isinstance(feature, RecordingFeature):
            raise TypeError("feature can only be an instance of type RecordingFeature")
        enabled = self._call("isFeatureEnabled", in_p=[feature])
        return enabled

    @property
    def id_p(self):
        """Get int value for 'id'
        This attribute contains the screen ID bound to these settings.
        """
        ret = self._get_attr("id")
        return ret

    @property
    def enabled(self):
        """Get or set bool value for 'enabled'
        This setting determines whether this screen is enabled while recording.
        """
        ret = self._get_attr("enabled")
        return ret

    @enabled.setter
    def enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("enabled", value)

    @property
    def features(self):
        """Get or set int value for 'features'
        This setting determines all enabled recording features for this
        screen.
        """
        ret = self._get_attr("features")
        return ret

    @features.setter
    def features(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("features", value)

    @property
    def destination(self):
        """Get or set RecordingDestination value for 'destination'
        This setting determines the recording destination for this
        screen.
        """
        ret = self._get_attr("destination")
        return RecordingDestination(ret)

    @destination.setter
    def destination(self, value):
        if not isinstance(value, RecordingDestination):
            raise TypeError("value is not an instance of RecordingDestination")
        return self._set_attr("destination", value)

    @property
    def filename(self):
        """Get or set str value for 'filename'
        This setting determines the filename VirtualBox uses to save
        the recorded content. This setting cannot be changed while video
        recording is enabled.

        When setting this attribute, the specified path has to be
        absolute (full path). When reading this attribute, a full path is
        always returned.
        """
        ret = self._get_attr("filename")
        return ret

    @filename.setter
    def filename(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("filename", value)

    @property
    def max_time(self):
        """Get or set int value for 'maxTime'
        This setting defines the maximum amount of time in seconds
        to record. Recording will stop as soon as the defined time
        interval has elapsed. If this value is zero, recording will not be
        limited by time. This setting cannot be changed while recording is
        enabled.
        """
        ret = self._get_attr("maxTime")
        return ret

    @max_time.setter
    def max_time(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("maxTime", value)

    @property
    def max_file_size(self):
        """Get or set int value for 'maxFileSize'
        This setting determines the maximal number of recording file
        size in MB. Recording will stop as soon as the file size has
        reached the defined value. If this value is zero, recording
        will not be limited by the file size. This setting cannot be changed
        while recording is enabled.
        """
        ret = self._get_attr("maxFileSize")
        return ret

    @max_file_size.setter
    def max_file_size(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("maxFileSize", value)

    @property
    def options(self):
        """Get or set str value for 'options'
        This setting contains any additional recording options
        required in comma-separated key=value format. This setting
        cannot be changed while recording is enabled.

        The following keys and their corresponding values are available:



        ac_enabled
        Enables audio recording when set to true, otherwise
        set to false to disable.

        **This feature is considered being experimental.**
        """
        ret = self._get_attr("options")
        return ret

    @options.setter
    def options(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("options", value)

    @property
    def audio_codec(self):
        """Get or set RecordingAudioCodec value for 'audioCodec'
        Determines the audio codec to use for encoding the
        recorded audio data. This setting cannot be changed while recording is
        enabled.
        """
        ret = self._get_attr("audioCodec")
        return RecordingAudioCodec(ret)

    @audio_codec.setter
    def audio_codec(self, value):
        if not isinstance(value, RecordingAudioCodec):
            raise TypeError("value is not an instance of RecordingAudioCodec")
        return self._set_attr("audioCodec", value)

    @property
    def audio_hz(self):
        """Get or set int value for 'audioHz'
        Determines the Hertz (Hz) rate of the recorded audio data. This setting
        cannot be changed while recording is enabled.
        """
        ret = self._get_attr("audioHz")
        return ret

    @audio_hz.setter
    def audio_hz(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("audioHz", value)

    @property
    def audio_bits(self):
        """Get or set int value for 'audioBits'
        Determines the bits per sample of the recorded audio data. This setting
        cannot be changed while recording is enabled.
        """
        ret = self._get_attr("audioBits")
        return ret

    @audio_bits.setter
    def audio_bits(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("audioBits", value)

    @property
    def audio_channels(self):
        """Get or set int value for 'audioChannels'
        Determines the audio channels of the recorded audio data.
        Specifiy 2 for stereo or 1 for mono. More than stereo (2) channels
        are not supported at the moment. This setting cannot be changed while
        recording is enabled.
        """
        ret = self._get_attr("audioChannels")
        return ret

    @audio_channels.setter
    def audio_channels(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("audioChannels", value)

    @property
    def video_codec(self):
        """Get or set RecordingVideoCodec value for 'videoCodec'
        Determines the video codec to use for encoding the recorded video data.
        This setting cannot be changed while recording is enabled.
        """
        ret = self._get_attr("videoCodec")
        return RecordingVideoCodec(ret)

    @video_codec.setter
    def video_codec(self, value):
        if not isinstance(value, RecordingVideoCodec):
            raise TypeError("value is not an instance of RecordingVideoCodec")
        return self._set_attr("videoCodec", value)

    @property
    def video_width(self):
        """Get or set int value for 'videoWidth'
        Determines the horizontal resolution of the recorded video data. This
        setting cannot be changed while recording is enabled.
        """
        ret = self._get_attr("videoWidth")
        return ret

    @video_width.setter
    def video_width(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("videoWidth", value)

    @property
    def video_height(self):
        """Get or set int value for 'videoHeight'
        Determines the vertical resolution of the recorded video data. This
        setting cannot be changed while recording is enabled.
        """
        ret = self._get_attr("videoHeight")
        return ret

    @video_height.setter
    def video_height(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("videoHeight", value)

    @property
    def video_rate(self):
        """Get or set int value for 'videoRate'
        Determines the bitrate in kilobits per second. Increasing this value
        makes the video look better for the cost of an increased file size or
        transfer rate. This setting cannot be changed while recording is enabled.
        """
        ret = self._get_attr("videoRate")
        return ret

    @video_rate.setter
    def video_rate(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("videoRate", value)

    @property
    def video_rate_control_mode(self):
        """Get or set RecordingVideoRateControlMode value for 'videoRateControlMode'
        Determines the rate control mode. This setting cannot be changed
        while recording is enabled.
        """
        ret = self._get_attr("videoRateControlMode")
        return RecordingVideoRateControlMode(ret)

    @video_rate_control_mode.setter
    def video_rate_control_mode(self, value):
        if not isinstance(value, RecordingVideoRateControlMode):
            raise TypeError("value is not an instance of RecordingVideoRateControlMode")
        return self._set_attr("videoRateControlMode", value)

    @property
    def video_fps(self):
        """Get or set int value for 'videoFPS'
        Determines the maximum number of frames per second (FPS). Frames with
        a higher frequency will be skipped. Reducing this value increases the
        number of skipped frames and reduces the file size or transfer rate.
        This setting cannot be changed while recording is enabled.
        """
        ret = self._get_attr("videoFPS")
        return ret

    @video_fps.setter
    def video_fps(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("videoFPS", value)

    @property
    def video_scaling_method(self):
        """Get or set RecordingVideoScalingMethod value for 'videoScalingMethod'
        Determines the video scaling method to use.
        This setting cannot be changed while recording is enabled.
        """
        ret = self._get_attr("videoScalingMethod")
        return RecordingVideoScalingMethod(ret)

    @video_scaling_method.setter
    def video_scaling_method(self, value):
        if not isinstance(value, RecordingVideoScalingMethod):
            raise TypeError("value is not an instance of RecordingVideoScalingMethod")
        return self._set_attr("videoScalingMethod", value)


class IRecordingSettings(Interface):
    """
    The IRecordingSettings interface represents recording settings of the virtual
    machine. This is used only in the :py:func:`IMachine.recording_settings`
    attribute.
    """

    __uuid__ = "D88F2A5A-47C7-4A3F-AAE1-1B516817DB41"
    __wsmap__ = "managed"

    def get_screen_settings(self, screen_id):
        """Returns the recording settings for a particular screen.

        in screen_id of type int
            Screen ID to retrieve recording screen settings for.

        return record_screen_settings of type :class:`IRecordingScreenSettings`
            Recording screen settings for the requested screen.

        """
        if not isinstance(screen_id, baseinteger):
            raise TypeError("screen_id can only be an instance of type baseinteger")
        record_screen_settings = self._call("getScreenSettings", in_p=[screen_id])
        record_screen_settings = IRecordingScreenSettings(record_screen_settings)
        return record_screen_settings

    @property
    def enabled(self):
        """Get or set bool value for 'enabled'
        This setting determines whether VirtualBox uses recording to record a
        VM session.
        """
        ret = self._get_attr("enabled")
        return ret

    @enabled.setter
    def enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("enabled", value)

    @property
    def screens(self):
        """Get IRecordingScreenSettings value for 'screens'
        This setting returns an array for recording settings of all configured
        virtual screens.
        """
        ret = self._get_attr("screens")
        return [IRecordingScreenSettings(a) for a in ret]


class IPCIAddress(Interface):
    """
    Address on the PCI bus.
    """

    __uuid__ = "c984d15f-e191-400b-840e-970f3dad7296"
    __wsmap__ = "managed"

    @property
    def bus(self):
        """Get or set int value for 'bus'
        Bus number.
        """
        ret = self._get_attr("bus")
        return ret

    @bus.setter
    def bus(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("bus", value)

    @property
    def device(self):
        """Get or set int value for 'device'
        Device number.
        """
        ret = self._get_attr("device")
        return ret

    @device.setter
    def device(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("device", value)

    @property
    def dev_function(self):
        """Get or set int value for 'devFunction'
        Device function number.
        """
        ret = self._get_attr("devFunction")
        return ret

    @dev_function.setter
    def dev_function(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("devFunction", value)

    def as_long(self):
        """Convert PCI address into long.

        return result of type int

        """
        result = self._call("asLong")
        return result

    def from_long(self, number):
        """Make PCI address from long.

        in number of type int

        """
        if not isinstance(number, baseinteger):
            raise TypeError("number can only be an instance of type baseinteger")
        self._call("fromLong", in_p=[number])


class IPCIDeviceAttachment(Interface):
    """
    Information about PCI attachments.
    """

    __uuid__ = "91f33d6f-e621-4f70-a77e-15f0e3c714d5"
    __wsmap__ = "struct"

    @property
    def name(self):
        """Get str value for 'name'
        Device name.
        """
        ret = self._get_attr("name")
        return ret

    @property
    def is_physical_device(self):
        """Get bool value for 'isPhysicalDevice'
        If this is physical or virtual device.
        """
        ret = self._get_attr("isPhysicalDevice")
        return ret

    @property
    def host_address(self):
        """Get int value for 'hostAddress'
        Address of device on the host, applicable only to host devices.
        """
        ret = self._get_attr("hostAddress")
        return ret

    @property
    def guest_address(self):
        """Get int value for 'guestAddress'
        Address of device in the guest.
        """
        ret = self._get_attr("guestAddress")
        return ret


class IMachine(Interface):
    """
    The IMachine interface represents a virtual machine, or guest, created
    in VirtualBox.

    This interface is used in two contexts. First of all, a collection of
    objects implementing this interface is stored in the
    :py:func:`IVirtualBox.machines`  attribute which lists all the virtual
    machines that are currently registered with this VirtualBox
    installation. Also, once a session has been opened for the given virtual
    machine (e.g. the virtual machine is running), the machine object
    associated with the open session can be queried from the session object;
    see :py:class:`ISession`  for details.

    The main role of this interface is to expose the settings of the virtual
    machine and provide methods to change various aspects of the virtual
    machine's configuration. For machine objects stored in the
    :py:func:`IVirtualBox.machines`  collection, all attributes are
    read-only unless explicitly stated otherwise in individual attribute
    and method descriptions.

    In order to change a machine setting, a session for this machine must be
    opened using one of the :py:func:`IMachine.lock_machine`  or
    :py:func:`IMachine.launch_vm_process`  methods. After the
    machine has been successfully locked for a session, a mutable machine object
    needs to be queried from the session object and then the desired settings
    changes can be applied to the returned object using IMachine attributes and
    methods. See the :py:class:`ISession`  interface description for more
    information about sessions.

    Note that IMachine does not provide methods to control virtual machine
    execution (such as start the machine, or power it down) -- these methods
    are grouped in a separate interface called :py:class:`IConsole` .

    :py:class:`ISession` , :py:class:`IConsole`
    """

    __uuid__ = "85632c68-b5bb-4316-a900-5eb28d3413df"
    __wsmap__ = "managed"

    @property
    def parent(self):
        """Get IVirtualBox value for 'parent'
        Associated parent object.
        """
        ret = self._get_attr("parent")
        return IVirtualBox(ret)

    @property
    def icon(self):
        """Get or set str value for 'icon'
        Overridden VM Icon details.
        """
        ret = self._get_attr("icon")
        return ret

    @icon.setter
    def icon(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("icon", value)

    @property
    def accessible(self):
        """Get bool value for 'accessible'
        Whether this virtual machine is currently accessible or not.

        A machine is always deemed accessible unless it is registered *and*
        its settings file cannot be read or parsed (either because the file itself
        is unavailable or has invalid XML contents).

        Every time this property is read, the accessibility state of
        this machine is re-evaluated. If the returned value is @c false,
        the :py:func:`access_error`  property may be used to get the
        detailed error information describing the reason of
        inaccessibility, including XML error messages.

        When the machine is inaccessible, only the following properties
        can be used on it:

        :py:func:`parent`
        :py:func:`id_p`
        :py:func:`settings_file_path`
        :py:func:`accessible`
        :py:func:`access_error`


        An attempt to access any other property or method will return
        an error.

        The only possible action you can perform on an inaccessible
        machine is to unregister it using the
        :py:func:`IMachine.unregister`  call (or, to check
        for the accessibility state once more by querying this
        property).


        In the current implementation, once this property returns
        @c true, the machine will never become inaccessible
        later, even if its settings file cannot be successfully
        read/written any more (at least, until the VirtualBox
        server is restarted). This limitation may be removed in
        future releases.
        """
        ret = self._get_attr("accessible")
        return ret

    @property
    def access_error(self):
        """Get IVirtualBoxErrorInfo value for 'accessError'
        Error information describing the reason of machine
        inaccessibility.

        Reading this property is only valid after the last call to
        :py:func:`accessible`  returned @c false (i.e. the
        machine is currently inaccessible). Otherwise, a @c null
        IVirtualBoxErrorInfo object will be returned.
        """
        ret = self._get_attr("accessError")
        return IVirtualBoxErrorInfo(ret)

    @property
    def name(self):
        """Get or set str value for 'name'
        Name of the virtual machine.

        Besides being used for human-readable identification purposes
        everywhere in VirtualBox, the virtual machine name is also used
        as a name of the machine's settings file and as a name of the
        subdirectory this settings file resides in. Thus, every time you
        change the value of this property, the settings file will be
        renamed once you call :py:func:`save_settings`  to confirm the
        change. The containing subdirectory will be also renamed, but
        only if it has exactly the same name as the settings file
        itself prior to changing this property (for backward compatibility
        with previous API releases). The above implies the following
        limitations:

        The machine name cannot be empty.
        The machine name can contain only characters that are valid
        file name characters according to the rules of the file
        system used to store VirtualBox configuration.
        You cannot have two or more machines with the same name
        if they use the same subdirectory for storing the machine
        settings files.
        You cannot change the name of the machine if it is running,
        or if any file in the directory containing the settings file
        is being used by another running machine or by any other
        process in the host operating system at a time when
        :py:func:`save_settings`  is called.


        If any of the above limitations are hit, :py:func:`save_settings`
        will return an appropriate error message explaining the exact
        reason and the changes you made to this machine will not be saved.

        Starting with VirtualBox 4.0, a ".vbox" extension of the settings
        file is recommended, but not enforced. (Previous versions always
        used a generic ".xml" extension.)
        """
        ret = self._get_attr("name")
        return ret

    @name.setter
    def name(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("name", value)

    @property
    def description(self):
        """Get or set str value for 'description'
        Description of the virtual machine.

        The description attribute can contain any text and is
        typically used to describe the hardware and software
        configuration of the virtual machine in detail (i.e. network
        settings, versions of the installed software and so on).
        """
        ret = self._get_attr("description")
        return ret

    @description.setter
    def description(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("description", value)

    @property
    def id_p(self):
        """Get str value for 'id'
        UUID of the virtual machine.
        """
        ret = self._get_attr("id")
        return ret

    @property
    def groups(self):
        """Get or set str value for 'groups'
        Array of machine group names of which this machine is a member.
        "" and "/" are synonyms for the toplevel group. Each
        group is only listed once, however they are listed in no particular
        order and there is no guarantee that there are no gaps in the group
        hierarchy (i.e. "/group",
        "/group/subgroup/subsubgroup" is a valid result).
        """
        ret = self._get_attr("groups")
        return ret

    @groups.setter
    def groups(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("groups", value)

    @property
    def os_type_id(self):
        """Get or set str value for 'OSTypeId'
        User-defined identifier of the Guest OS type.
        You may use :py:func:`IVirtualBox.get_guest_os_type`  to obtain
        an IGuestOSType object representing details about the given
        Guest OS type. All Guest OS types are considered valid, even those
        which are not known to :py:func:`IVirtualBox.get_guest_os_type` .

        This value may differ from the value returned by
        :py:func:`IGuest.os_type_id`  if Guest Additions are
        installed to the guest OS.
        """
        ret = self._get_attr("OSTypeId")
        return ret

    @os_type_id.setter
    def os_type_id(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("OSTypeId", value)

    @property
    def hardware_version(self):
        """Get or set str value for 'hardwareVersion'
        Hardware version identifier. Internal use only for now.
        """
        ret = self._get_attr("hardwareVersion")
        return ret

    @hardware_version.setter
    def hardware_version(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("hardwareVersion", value)

    @property
    def hardware_uuid(self):
        """Get or set str value for 'hardwareUUID'
        The UUID presented to the guest via memory tables, hardware and guest
        properties. For most VMs this is the same as the @a id, but for VMs
        which have been cloned or teleported it may be the same as the source
        VM. The latter is because the guest shouldn't notice that it was
        cloned or teleported.
        """
        ret = self._get_attr("hardwareUUID")
        return ret

    @hardware_uuid.setter
    def hardware_uuid(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("hardwareUUID", value)

    @property
    def cpu_count(self):
        """Get or set int value for 'CPUCount'
        Number of virtual CPUs in the VM.
        """
        ret = self._get_attr("CPUCount")
        return ret

    @cpu_count.setter
    def cpu_count(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("CPUCount", value)

    @property
    def cpu_hot_plug_enabled(self):
        """Get or set bool value for 'CPUHotPlugEnabled'
        This setting determines whether VirtualBox allows CPU
        hotplugging for this machine.
        """
        ret = self._get_attr("CPUHotPlugEnabled")
        return ret

    @cpu_hot_plug_enabled.setter
    def cpu_hot_plug_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("CPUHotPlugEnabled", value)

    @property
    def cpu_execution_cap(self):
        """Get or set int value for 'CPUExecutionCap'
        Means to limit the number of CPU cycles a guest can use. The unit
        is percentage of host CPU cycles per second. The valid range
        is 1 - 100. 100 (the default) implies no limit.
        """
        ret = self._get_attr("CPUExecutionCap")
        return ret

    @cpu_execution_cap.setter
    def cpu_execution_cap(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("CPUExecutionCap", value)

    @property
    def cpuid_portability_level(self):
        """Get or set int value for 'CPUIDPortabilityLevel'
        Virtual CPUID portability level, the higher number the fewer newer
        or vendor specific CPU feature is reported to the guest (via the CPUID
        instruction).  The default level of zero (0) means that all virtualized
        features supported by the host is pass thru to the guest.  While the
        three (3) is currently the level supressing the most features.

        Exactly which of the CPUID features are left out by the VMM at which
        level is subject to change with each major version.
        """
        ret = self._get_attr("CPUIDPortabilityLevel")
        return ret

    @cpuid_portability_level.setter
    def cpuid_portability_level(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("CPUIDPortabilityLevel", value)

    @property
    def memory_size(self):
        """Get or set int value for 'memorySize'
        System memory size in megabytes.
        """
        ret = self._get_attr("memorySize")
        return ret

    @memory_size.setter
    def memory_size(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("memorySize", value)

    @property
    def memory_balloon_size(self):
        """Get or set int value for 'memoryBalloonSize'
        Memory balloon size in megabytes.
        """
        ret = self._get_attr("memoryBalloonSize")
        return ret

    @memory_balloon_size.setter
    def memory_balloon_size(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("memoryBalloonSize", value)

    @property
    def page_fusion_enabled(self):
        """Get or set bool value for 'pageFusionEnabled'
        This setting determines whether VirtualBox allows page
        fusion for this machine (64-bit hosts only).
        """
        ret = self._get_attr("pageFusionEnabled")
        return ret

    @page_fusion_enabled.setter
    def page_fusion_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("pageFusionEnabled", value)

    @property
    def graphics_adapter(self):
        """Get IGraphicsAdapter value for 'graphicsAdapter'
        Graphics adapter object.
        """
        ret = self._get_attr("graphicsAdapter")
        return IGraphicsAdapter(ret)

    @property
    def bios_settings(self):
        """Get IBIOSSettings value for 'BIOSSettings'
        Object containing all BIOS settings.
        """
        ret = self._get_attr("BIOSSettings")
        return IBIOSSettings(ret)

    @property
    def recording_settings(self):
        """Get IRecordingSettings value for 'recordingSettings'
        Object containing all recording settings.
        """
        ret = self._get_attr("recordingSettings")
        return IRecordingSettings(ret)

    @property
    def firmware_type(self):
        """Get or set FirmwareType value for 'firmwareType'
        Type of firmware (such as legacy BIOS or EFI), used for initial
        bootstrap in this VM.
        """
        ret = self._get_attr("firmwareType")
        return FirmwareType(ret)

    @firmware_type.setter
    def firmware_type(self, value):
        if not isinstance(value, FirmwareType):
            raise TypeError("value is not an instance of FirmwareType")
        return self._set_attr("firmwareType", value)

    @property
    def pointing_hid_type(self):
        """Get or set PointingHIDType value for 'pointingHIDType'
        Type of pointing HID (such as mouse or tablet) used in this VM.
        The default is typically "PS2Mouse" but can vary depending on the
        requirements of the guest operating system.
        """
        ret = self._get_attr("pointingHIDType")
        return PointingHIDType(ret)

    @pointing_hid_type.setter
    def pointing_hid_type(self, value):
        if not isinstance(value, PointingHIDType):
            raise TypeError("value is not an instance of PointingHIDType")
        return self._set_attr("pointingHIDType", value)

    @property
    def keyboard_hid_type(self):
        """Get or set KeyboardHIDType value for 'keyboardHIDType'
        Type of keyboard HID used in this VM.
        The default is typically "PS2Keyboard" but can vary depending on the
        requirements of the guest operating system.
        """
        ret = self._get_attr("keyboardHIDType")
        return KeyboardHIDType(ret)

    @keyboard_hid_type.setter
    def keyboard_hid_type(self, value):
        if not isinstance(value, KeyboardHIDType):
            raise TypeError("value is not an instance of KeyboardHIDType")
        return self._set_attr("keyboardHIDType", value)

    @property
    def hpet_enabled(self):
        """Get or set bool value for 'HPETEnabled'
        This attribute controls if High Precision Event Timer (HPET) is
        enabled in this VM. Use this property if you want to provide guests
        with additional time source, or if guest requires HPET to function correctly.
        Default is false.
        """
        ret = self._get_attr("HPETEnabled")
        return ret

    @hpet_enabled.setter
    def hpet_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("HPETEnabled", value)

    @property
    def chipset_type(self):
        """Get or set ChipsetType value for 'chipsetType'
        Chipset type used in this VM.
        """
        ret = self._get_attr("chipsetType")
        return ChipsetType(ret)

    @chipset_type.setter
    def chipset_type(self, value):
        if not isinstance(value, ChipsetType):
            raise TypeError("value is not an instance of ChipsetType")
        return self._set_attr("chipsetType", value)

    @property
    def snapshot_folder(self):
        """Get or set str value for 'snapshotFolder'
        Full path to the directory used to store snapshot data
        (differencing media and saved state files) of this machine.

        The initial value of this property is
        <:py:func:`settings_file_path` path_to_settings_file>/<
        :py:func:`id_p` machine_uuid
        >.

        Currently, it is an error to try to change this property on
        a machine that has snapshots (because this would require to
        move possibly large files to a different location).
        A separate method will be available for this purpose later.


        Setting this property to @c null or to an empty string will restore
        the initial value.


        When setting this property, the specified path can be
        absolute (full path) or relative to the directory where the
        :py:func:`settings_file_path` machine settings file
        is located. When reading this property, a full path is
        always returned.


        The specified path may not exist, it will be created
        when necessary.
        """
        ret = self._get_attr("snapshotFolder")
        return ret

    @snapshot_folder.setter
    def snapshot_folder(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("snapshotFolder", value)

    @property
    def vrde_server(self):
        """Get IVRDEServer value for 'VRDEServer'
        VirtualBox Remote Desktop Extension (VRDE) server object.
        """
        ret = self._get_attr("VRDEServer")
        return IVRDEServer(ret)

    @property
    def emulated_usb_card_reader_enabled(self):
        """Get or set bool value for 'emulatedUSBCardReaderEnabled'"""
        ret = self._get_attr("emulatedUSBCardReaderEnabled")
        return ret

    @emulated_usb_card_reader_enabled.setter
    def emulated_usb_card_reader_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("emulatedUSBCardReaderEnabled", value)

    @property
    def medium_attachments(self):
        """Get IMediumAttachment value for 'mediumAttachments'
        Array of media attached to this machine.
        """
        ret = self._get_attr("mediumAttachments")
        return [IMediumAttachment(a) for a in ret]

    @property
    def usb_controllers(self):
        """Get IUSBController value for 'USBControllers'
        Array of USB controllers attached to this machine.


        If USB functionality is not available in the given edition of
        VirtualBox, this method will set the result code to @c E_NOTIMPL.
        """
        ret = self._get_attr("USBControllers")
        return [IUSBController(a) for a in ret]

    @property
    def usb_device_filters(self):
        """Get IUSBDeviceFilters value for 'USBDeviceFilters'
        Associated USB device filters object.


        If USB functionality is not available in the given edition of
        VirtualBox, this method will set the result code to @c E_NOTIMPL.
        """
        ret = self._get_attr("USBDeviceFilters")
        return IUSBDeviceFilters(ret)

    @property
    def audio_adapter(self):
        """Get IAudioAdapter value for 'audioAdapter'
        Associated audio adapter, always present.
        """
        ret = self._get_attr("audioAdapter")
        return IAudioAdapter(ret)

    @property
    def storage_controllers(self):
        """Get IStorageController value for 'storageControllers'
        Array of storage controllers attached to this machine.
        """
        ret = self._get_attr("storageControllers")
        return [IStorageController(a) for a in ret]

    @property
    def settings_file_path(self):
        """Get str value for 'settingsFilePath'
        Full name of the file containing machine settings data.
        """
        ret = self._get_attr("settingsFilePath")
        return ret

    @property
    def settings_aux_file_path(self):
        """Get str value for 'settingsAuxFilePath'
        Full name of the file containing auxiliary machine settings data.
        """
        ret = self._get_attr("settingsAuxFilePath")
        return ret

    @property
    def settings_modified(self):
        """Get bool value for 'settingsModified'
        Whether the settings of this machine have been modified
        (but neither yet saved nor discarded).

        Reading this property is only valid on instances returned
        by :py:func:`ISession.machine`  and on new machines
        created by :py:func:`IVirtualBox.create_machine`  or opened
        by :py:func:`IVirtualBox.open_machine`  but not
        yet registered, or on unregistered machines after calling
        :py:func:`IMachine.unregister` . For all other
        cases, the settings can never be modified.


        For newly created unregistered machines, the value of this
        property is always @c true until :py:func:`save_settings`
        is called (no matter if any machine settings have been
        changed after the creation or not). For opened machines
        the value is set to @c false (and then follows to normal rules).
        """
        ret = self._get_attr("settingsModified")
        return ret

    @property
    def session_state(self):
        """Get SessionState value for 'sessionState'
        Current session state for this machine.
        """
        ret = self._get_attr("sessionState")
        return SessionState(ret)

    @property
    def session_name(self):
        """Get str value for 'sessionName'
        Name of the session. If :py:func:`session_state`  is
        Spawning or Locked, this attribute contains the
        same value as passed to the
        :py:func:`IMachine.launch_vm_process`  method in the
        @a name parameter. If the session was established with
        :py:func:`IMachine.lock_machine` , it is the name of the session
        (if set, otherwise empty string). If
        :py:func:`session_state`  is SessionClosed, the value of this
        attribute is an empty string.
        """
        ret = self._get_attr("sessionName")
        return ret

    @property
    def session_pid(self):
        """Get int value for 'sessionPID'
        Identifier of the session process. This attribute contains the
        platform-dependent identifier of the process whose session was
        used with :py:func:`IMachine.lock_machine`  call. The returned
        value is only valid if :py:func:`session_state`  is Locked or
        Unlocking by the time this property is read.
        """
        ret = self._get_attr("sessionPID")
        return ret

    @property
    def state(self):
        """Get MachineState value for 'state'
        Current execution state of this machine.
        """
        ret = self._get_attr("state")
        return MachineState(ret)

    @property
    def last_state_change(self):
        """Get int value for 'lastStateChange'
        Timestamp of the last execution state change,
        in milliseconds since 1970-01-01 UTC.
        """
        ret = self._get_attr("lastStateChange")
        return ret

    @property
    def state_file_path(self):
        """Get str value for 'stateFilePath'
        Full path to the file that stores the execution state of
        the machine when it is in the :py:attr:`MachineState.saved`  state.

        When the machine is not in the Saved state, this attribute is
        an empty string.
        """
        ret = self._get_attr("stateFilePath")
        return ret

    @property
    def log_folder(self):
        """Get str value for 'logFolder'
        Full path to the folder that stores a set of rotated log files
        recorded during machine execution. The most recent log file is
        named VBox.log, the previous log file is
        named VBox.log.1 and so on (up to VBox.log.3
        in the current version).
        """
        ret = self._get_attr("logFolder")
        return ret

    @property
    def current_snapshot(self):
        """Get ISnapshot value for 'currentSnapshot'
        Current snapshot of this machine. This is @c null if the machine
        currently has no snapshots. If it is not @c null, then it was
        set by one of :py:func:`take_snapshot` , :py:func:`delete_snapshot`
        or :py:func:`restore_snapshot` , depending on which was called last.
        See :py:class:`ISnapshot`  for details.
        """
        ret = self._get_attr("currentSnapshot")
        return ISnapshot(ret)

    @property
    def snapshot_count(self):
        """Get int value for 'snapshotCount'
        Number of snapshots taken on this machine. Zero means the
        machine doesn't have any snapshots.
        """
        ret = self._get_attr("snapshotCount")
        return ret

    @property
    def current_state_modified(self):
        """Get bool value for 'currentStateModified'
        Returns @c true if the current state of the machine is not
        identical to the state stored in the current snapshot.

        The current state is identical to the current snapshot only
        directly after one of the following calls are made:


        :py:func:`restore_snapshot`

        :py:func:`take_snapshot`  (issued on a "powered off" or "saved"
        machine, for which :py:func:`settings_modified`  returns @c false)



        The current state remains identical until one of the following
        happens:

        settings of the machine are changed
        the saved state is deleted
        the current snapshot is deleted
        an attempt to execute the machine is made



        For machines that don't have snapshots, this property is
        always @c false.
        """
        ret = self._get_attr("currentStateModified")
        return ret

    @property
    def shared_folders(self):
        """Get ISharedFolder value for 'sharedFolders'
        Collection of shared folders for this machine (permanent shared
        folders). These folders are shared automatically at machine startup
        and available only to the guest OS installed within this machine.

        New shared folders are added to the collection using
        :py:func:`create_shared_folder` . Existing shared folders can be
        removed using :py:func:`remove_shared_folder` .
        """
        ret = self._get_attr("sharedFolders")
        return [ISharedFolder(a) for a in ret]

    @property
    def clipboard_mode(self):
        """Get or set ClipboardMode value for 'clipboardMode'
        Synchronization mode between the host OS clipboard
        and the guest OS clipboard.
        """
        ret = self._get_attr("clipboardMode")
        return ClipboardMode(ret)

    @clipboard_mode.setter
    def clipboard_mode(self, value):
        if not isinstance(value, ClipboardMode):
            raise TypeError("value is not an instance of ClipboardMode")
        return self._set_attr("clipboardMode", value)

    @property
    def clipboard_file_transfers_enabled(self):
        """Get or set bool value for 'clipboardFileTransfersEnabled'
        Sets or retrieves whether clipboard file transfers are allowed or not.

        When set to @a true, clipboard file transfers between supported
        host and guest OSes are allowed.
        """
        ret = self._get_attr("clipboardFileTransfersEnabled")
        return ret

    @clipboard_file_transfers_enabled.setter
    def clipboard_file_transfers_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("clipboardFileTransfersEnabled", value)

    @property
    def dn_d_mode(self):
        """Get or set DnDMode value for 'dnDMode'
        Sets or retrieves the current drag'n drop mode.
        """
        ret = self._get_attr("dnDMode")
        return DnDMode(ret)

    @dn_d_mode.setter
    def dn_d_mode(self, value):
        if not isinstance(value, DnDMode):
            raise TypeError("value is not an instance of DnDMode")
        return self._set_attr("dnDMode", value)

    @property
    def teleporter_enabled(self):
        """Get or set bool value for 'teleporterEnabled'
        When set to @a true, the virtual machine becomes a target teleporter
        the next time it is powered on. This can only set to @a true when the
        VM is in the @a PoweredOff or @a Aborted state.

        <!-- This property is automatically set to @a false when the VM is powered
        on. (bird: This doesn't work yet ) -->
        """
        ret = self._get_attr("teleporterEnabled")
        return ret

    @teleporter_enabled.setter
    def teleporter_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("teleporterEnabled", value)

    @property
    def teleporter_port(self):
        """Get or set int value for 'teleporterPort'
        The TCP port the target teleporter will listen for incoming
        teleportations on.

        0 means the port is automatically selected upon power on. The actual
        value can be read from this property while the machine is waiting for
        incoming teleportations.
        """
        ret = self._get_attr("teleporterPort")
        return ret

    @teleporter_port.setter
    def teleporter_port(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("teleporterPort", value)

    @property
    def teleporter_address(self):
        """Get or set str value for 'teleporterAddress'
        The address the target teleporter will listen on. If set to an empty
        string, it will listen on all addresses.
        """
        ret = self._get_attr("teleporterAddress")
        return ret

    @teleporter_address.setter
    def teleporter_address(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("teleporterAddress", value)

    @property
    def teleporter_password(self):
        """Get or set str value for 'teleporterPassword'
        The password to check for on the target teleporter. This is just a
        very basic measure to prevent simple hacks and operators accidentally
        beaming a virtual machine to the wrong place.

        Note that you SET a plain text password while reading back a HASHED
        password. Setting a hashed password is currently not supported.
        """
        ret = self._get_attr("teleporterPassword")
        return ret

    @teleporter_password.setter
    def teleporter_password(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("teleporterPassword", value)

    @property
    def paravirt_provider(self):
        """Get or set ParavirtProvider value for 'paravirtProvider'
        The paravirtualized guest interface provider.
        """
        ret = self._get_attr("paravirtProvider")
        return ParavirtProvider(ret)

    @paravirt_provider.setter
    def paravirt_provider(self, value):
        if not isinstance(value, ParavirtProvider):
            raise TypeError("value is not an instance of ParavirtProvider")
        return self._set_attr("paravirtProvider", value)

    @property
    def rtc_use_utc(self):
        """Get or set bool value for 'RTCUseUTC'
        When set to @a true, the RTC device of the virtual machine will run
        in UTC time, otherwise in local time. Especially Unix guests prefer
        the time in UTC.
        """
        ret = self._get_attr("RTCUseUTC")
        return ret

    @rtc_use_utc.setter
    def rtc_use_utc(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("RTCUseUTC", value)

    @property
    def io_cache_enabled(self):
        """Get or set bool value for 'IOCacheEnabled'
        When set to @a true, the builtin I/O cache of the virtual machine
        will be enabled.
        """
        ret = self._get_attr("IOCacheEnabled")
        return ret

    @io_cache_enabled.setter
    def io_cache_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("IOCacheEnabled", value)

    @property
    def io_cache_size(self):
        """Get or set int value for 'IOCacheSize'
        Maximum size of the I/O cache in MB.
        """
        ret = self._get_attr("IOCacheSize")
        return ret

    @io_cache_size.setter
    def io_cache_size(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("IOCacheSize", value)

    @property
    def pci_device_assignments(self):
        """Get IPCIDeviceAttachment value for 'PCIDeviceAssignments'
        Array of PCI devices assigned to this machine, to get list of all
        PCI devices attached to the machine use
        :py:func:`IConsole.attached_pci_devices`  attribute, as this attribute
        is intended to list only devices additional to what described in
        virtual hardware config. Usually, this list keeps host's physical
        devices assigned to the particular machine.
        """
        ret = self._get_attr("PCIDeviceAssignments")
        return [IPCIDeviceAttachment(a) for a in ret]

    @property
    def bandwidth_control(self):
        """Get IBandwidthControl value for 'bandwidthControl'
        Bandwidth control manager.
        """
        ret = self._get_attr("bandwidthControl")
        return IBandwidthControl(ret)

    @property
    def tracing_enabled(self):
        """Get or set bool value for 'tracingEnabled'
        Enables the tracing facility in the VMM (including PDM devices +
        drivers). The VMM will consume about 0.5MB of more memory when
        enabled and there may be some extra overhead from tracepoints that are
        always enabled.
        """
        ret = self._get_attr("tracingEnabled")
        return ret

    @tracing_enabled.setter
    def tracing_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("tracingEnabled", value)

    @property
    def tracing_config(self):
        """Get or set str value for 'tracingConfig'
        Tracepoint configuration to apply at startup when
        :py:func:`IMachine.tracing_enabled`  is true. The string specifies
        a space separated of tracepoint group names to enable. The special
        group 'all' enables all tracepoints. Check DBGFR3TracingConfig for
        more details on available tracepoint groups and such.

        Note that on hosts supporting DTrace (or similar), a lot of the
        tracepoints may be implemented exclusively as DTrace probes. So, the
        effect of the same config may differ between Solaris and Windows for
        example.
        """
        ret = self._get_attr("tracingConfig")
        return ret

    @tracing_config.setter
    def tracing_config(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("tracingConfig", value)

    @property
    def allow_tracing_to_access_vm(self):
        """Get or set bool value for 'allowTracingToAccessVM'
        Enables tracepoints in PDM devices and drivers to use the VMCPU or VM
        structures when firing off trace points. This is especially useful
        with DTrace tracepoints, as it allows you to use the VMCPU or VM
        pointer to obtain useful information such as guest register state.

        This is disabled by default because devices and drivers normally has no
        business accessing the VMCPU or VM structures, and are therefore unable
        to get any pointers to these.
        """
        ret = self._get_attr("allowTracingToAccessVM")
        return ret

    @allow_tracing_to_access_vm.setter
    def allow_tracing_to_access_vm(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("allowTracingToAccessVM", value)

    @property
    def autostart_enabled(self):
        """Get or set bool value for 'autostartEnabled'
        Enables autostart of the VM during system boot.
        """
        ret = self._get_attr("autostartEnabled")
        return ret

    @autostart_enabled.setter
    def autostart_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("autostartEnabled", value)

    @property
    def autostart_delay(self):
        """Get or set int value for 'autostartDelay'
        Number of seconds to wait until the VM should be started during system boot.
        """
        ret = self._get_attr("autostartDelay")
        return ret

    @autostart_delay.setter
    def autostart_delay(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("autostartDelay", value)

    @property
    def autostop_type(self):
        """Get or set AutostopType value for 'autostopType'
        Action type to do when the system is shutting down.
        """
        ret = self._get_attr("autostopType")
        return AutostopType(ret)

    @autostop_type.setter
    def autostop_type(self, value):
        if not isinstance(value, AutostopType):
            raise TypeError("value is not an instance of AutostopType")
        return self._set_attr("autostopType", value)

    @property
    def default_frontend(self):
        """Get or set str value for 'defaultFrontend'
        Selects which VM frontend should be used by default when launching
        this VM through the :py:func:`IMachine.launch_vm_process`  method.
        Empty or @c null strings do not define a particular default, it is up
        to :py:func:`IMachine.launch_vm_process`  to select one. See the
        description of :py:func:`IMachine.launch_vm_process`  for the valid
        frontend types.

        This per-VM setting overrides the default defined by
        :py:func:`ISystemProperties.default_frontend`  attribute, and is
        overridden by a frontend type passed to
        :py:func:`IMachine.launch_vm_process` .
        """
        ret = self._get_attr("defaultFrontend")
        return ret

    @default_frontend.setter
    def default_frontend(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("defaultFrontend", value)

    @property
    def usb_proxy_available(self):
        """Get bool value for 'USBProxyAvailable'
        Returns whether there is an USB proxy available.
        """
        ret = self._get_attr("USBProxyAvailable")
        return ret

    @property
    def vm_process_priority(self):
        """Get or set VMProcPriority value for 'VMProcessPriority'
        Sets the priority of the VM process. It is a VM setting which can
        be changed both before starting the VM and at runtime.

        The default value is 'Default', which selects the default
        process priority.
        """
        ret = self._get_attr("VMProcessPriority")
        return VMProcPriority(ret)

    @vm_process_priority.setter
    def vm_process_priority(self, value):
        if not isinstance(value, VMProcPriority):
            raise TypeError("value is not an instance of VMProcPriority")
        return self._set_attr("VMProcessPriority", value)

    @property
    def paravirt_debug(self):
        """Get or set str value for 'paravirtDebug'
        Debug parameters for the paravirtualized guest interface provider.
        """
        ret = self._get_attr("paravirtDebug")
        return ret

    @paravirt_debug.setter
    def paravirt_debug(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("paravirtDebug", value)

    @property
    def cpu_profile(self):
        """Get or set str value for 'CPUProfile'
        Experimental feature to select the guest CPU profile.  The default
        is "host", which indicates the host CPU.  All other names are subject
        to change.

        Use the :py:func:`ISystemProperties.get_cpu_profiles`  method to get
        currently available CPU profiles.
        """
        ret = self._get_attr("CPUProfile")
        return ret

    @cpu_profile.setter
    def cpu_profile(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("CPUProfile", value)

    def lock_machine(self, session, lock_type):
        """Locks the machine for the given session to enable the caller
        to make changes to the machine or start the VM or control
        VM execution.

        There are two ways to lock a machine for such uses:


        If you want to make changes to the machine settings,
        you must obtain an exclusive write lock on the machine
        by setting @a lockType to @c Write.

        This will only succeed if no other process has locked
        the machine to prevent conflicting changes. Only after
        an exclusive write lock has been obtained using this method, one
        can change all VM settings or execute the VM in the process
        space of the session object. (Note that the latter is only of
        interest if you actually want to write a new front-end for
        virtual machines; but this API gets called internally by
        the existing front-ends such as VBoxHeadless and the VirtualBox
        GUI to acquire a write lock on the machine that they are running.)

        On success, write-locking the machine for a session creates
        a second copy of the IMachine object. It is this second object
        upon which changes can be made; in VirtualBox terminology, the
        second copy is "mutable". It is only this second, mutable machine
        object upon which you can call methods that change the
        machine state. After having called this method, you can
        obtain this second, mutable machine object using the
        :py:func:`ISession.machine`  attribute.

        If you only want to check the machine state or control
        machine execution without actually changing machine
        settings (e.g. to get access to VM statistics or take
        a snapshot or save the machine state), then set the
        @a lockType argument to @c Shared.

        If no other session has obtained a lock, you will obtain an
        exclusive write lock as described above. However, if another
        session has already obtained such a lock, then a link to that
        existing session will be established which allows you
        to control that existing session.

        To find out which type of lock was obtained, you can
        inspect :py:func:`ISession.type_p` , which will have been
        set to either @c WriteLock or @c Shared.



        In either case, you can get access to the :py:class:`IConsole`
        object which controls VM execution.

        Also in all of the above cases, one must always call
        :py:func:`ISession.unlock_machine`  to release the lock on the machine, or
        the machine's state will eventually be set to "Aborted".

        To change settings on a machine, the following sequence is typically
        performed:


        Call this method to obtain an exclusive write lock for the current session.

        Obtain a mutable IMachine object from :py:func:`ISession.machine` .

        Change the settings of the machine by invoking IMachine methods.

        Call :py:func:`IMachine.save_settings` .

        Release the write lock by calling :py:func:`ISession.unlock_machine` .

        in session of type :class:`ISession`
            Session object for which the machine will be locked.

        in lock_type of type :class:`LockType`
            If set to @c Write, then attempt to acquire an exclusive write lock or fail.
            If set to @c Shared, then either acquire an exclusive write lock or establish
            a link to an existing session.

        raises :class:`OleErrorUnexpected`
            Virtual machine not registered.

        raises :class:`OleErrorAccessdenied`
            Process not started by

        raises :class:`VBoxErrorInvalidObjectState`
            Session already open or being opened.

        raises :class:`VBoxErrorVmError`
            Failed to assign machine to session.

        """
        if not isinstance(session, ISession):
            raise TypeError("session can only be an instance of type ISession")
        if not isinstance(lock_type, LockType):
            raise TypeError("lock_type can only be an instance of type LockType")
        self._call("lockMachine", in_p=[session, lock_type])

    def launch_vm_process(self, session, name, environment_changes):
        """Spawns a new process that will execute the virtual machine and obtains a shared
        lock on the machine for the calling session.

        If launching the VM succeeds, the new VM process will create its own session
        and write-lock the machine for it, preventing conflicting changes from other
        processes. If the machine is already locked (because it is already running or
        because another session has a write lock), launching the VM process will therefore
        fail. Reversely, future attempts to obtain a write lock will also fail while the
        machine is running.

        The caller's session object remains separate from the session opened by the new
        VM process. It receives its own :py:class:`IConsole`  object which can be used
        to control machine execution, but it cannot be used to change all VM settings
        which would be available after a :py:func:`lock_machine`  call.

        The caller must eventually release the session's shared lock by calling
        :py:func:`ISession.unlock_machine`  on the local session object once this call
        has returned. However, the session's state (see :py:func:`ISession.state` )
        will not return to "Unlocked" until the remote session has also unlocked
        the machine (i.e. the machine has stopped running).

        Launching a VM process can take some time (a new VM is started in a new process,
        for which memory and other resources need to be set up). Because of this,
        an :py:class:`IProgress`  object is returned to allow the caller to wait
        for this asynchronous operation to be completed. Until then, the caller's
        session object remains in the "Unlocked" state, and its :py:func:`ISession.machine`
        and :py:func:`ISession.console`  attributes cannot be accessed.
        It is recommended to use :py:func:`IProgress.wait_for_completion`  or
        similar calls to wait for completion. Completion is signalled when the VM
        is powered on. If launching the VM fails, error messages can be queried
        via the progress object, if available.

        The progress object will have at least 2 sub-operations. The first
        operation covers the period up to the new VM process calls powerUp.
        The subsequent operations mirror the :py:func:`IConsole.power_up`
        progress object. Because :py:func:`IConsole.power_up`  may require
        some extra sub-operations, the :py:func:`IProgress.operation_count`
        may change at the completion of operation.

        For details on the teleportation progress operation, see
        :py:func:`IConsole.power_up` .

        The @a environmentChanges argument is a list of strings where every string contains
        environment variable in the putenv style, i.e. "VAR=VALUE" for setting/replacing
        and "VAR" for unsetting. These environment variables will be applied to the environment
        of the VirtualBox server process. If an environment variable exists both in the server
        process and in this list, the value from this list takes precedence over the
        server's variable. If the value of the environment variable is omitted, this variable
        will be removed from the resulting environment. If the list is empty, the server
        environment is inherited by the started process as is.

        in session of type :class:`ISession`
            Client session object to which the VM process will be connected (this
            must be in "Unlocked" state).

        in name of type str
            Front-end to use for the new VM process. The following are currently supported:

            "gui": VirtualBox Qt GUI front-end
            "headless": VBoxHeadless (VRDE Server) front-end
            "sdl": VirtualBox SDL front-end
            "emergencystop": reserved value, used for aborting
            the currently running VM or session owner. In this case the
            @a session parameter may be @c null (if it is non-null it isn't
            used in any way), and the @a progress return value will be always
            @c null. The operation completes immediately.
            "": use the per-VM default frontend if set, otherwise
            the global default defined in the system properties. If neither
            are set, the API will launch a "gui" session, which may
            fail if there is no windowing environment available. See
            :py:func:`IMachine.default_frontend`  and
            :py:func:`ISystemProperties.default_frontend` .

        in environment_changes of type str
            The list of putenv-style changes to the VM process environment.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`OleErrorUnexpected`
            Virtual machine not registered.

        raises :class:`OleErrorInvalidarg`
            Invalid session type @a type.

        raises :class:`VBoxErrorObjectNotFound`
            No machine matching @a machineId found.

        raises :class:`VBoxErrorInvalidObjectState`
            Session already open or being opened.

        raises :class:`VBoxErrorIprtError`
            Launching process for machine failed.

        raises :class:`VBoxErrorVmError`
            Failed to assign machine to session.

        """
        if not isinstance(session, ISession):
            raise TypeError("session can only be an instance of type ISession")
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(environment_changes, list):
            raise TypeError("environment_changes can only be an instance of type list")
        for a in environment_changes[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        progress = self._call(
            "launchVMProcess", in_p=[session, name, environment_changes]
        )
        progress = IProgress(progress)
        return progress

    def set_boot_order(self, position, device):
        """Puts the given device to the specified position in
        the boot order.

        To indicate that no device is associated with the given position,
        :py:attr:`DeviceType.null`  should be used.

        @todo setHardDiskBootOrder(), setNetworkBootOrder()

        in position of type int
            Position in the boot order (@c 1 to the total number of
            devices the machine can boot from, as returned by
            :py:func:`ISystemProperties.max_boot_position` ).

        in device of type :class:`DeviceType`
            The type of the device used to boot at the given position.

        raises :class:`OleErrorInvalidarg`
            Boot @a position out of range.

        raises :class:`OleErrorNotimpl`
            Booting from USB @a device currently not supported.

        """
        if not isinstance(position, baseinteger):
            raise TypeError("position can only be an instance of type baseinteger")
        if not isinstance(device, DeviceType):
            raise TypeError("device can only be an instance of type DeviceType")
        self._call("setBootOrder", in_p=[position, device])

    def get_boot_order(self, position):
        """Returns the device type that occupies the specified
        position in the boot order.

        @todo [remove?]
        If the machine can have more than one device of the returned type
        (such as hard disks), then a separate method should be used to
        retrieve the individual device that occupies the given position.

        If here are no devices at the given position, then
        :py:attr:`DeviceType.null`  is returned.

        @todo getHardDiskBootOrder(), getNetworkBootOrder()

        in position of type int
            Position in the boot order (@c 1 to the total number of
            devices the machine can boot from, as returned by
            :py:func:`ISystemProperties.max_boot_position` ).

        return device of type :class:`DeviceType`
            Device at the given position.

        raises :class:`OleErrorInvalidarg`
            Boot @a position out of range.

        """
        if not isinstance(position, baseinteger):
            raise TypeError("position can only be an instance of type baseinteger")
        device = self._call("getBootOrder", in_p=[position])
        device = DeviceType(device)
        return device

    def attach_device(self, name, controller_port, device, type_p, medium):
        """Attaches a device and optionally mounts a medium to the given storage
                controller (:py:class:`IStorageController` , identified by @a name),
                at the indicated port and device.

                This method is intended for managing storage devices in general while a
                machine is powered off. It can be used to attach and detach fixed
                and removable media. The following kind of media can be attached
                to a machine:


                For fixed and removable media, you can pass in a medium that was
                previously opened using :py:func:`IVirtualBox.open_medium` .


                Only for storage devices supporting removable media (such as
                DVDs and floppies), you can also specify a null pointer to
                indicate an empty drive or one of the medium objects listed
                in the :py:func:`IHost.dvd_drives`  and :py:func:`IHost.floppy_drives`
                arrays to indicate a host drive.
                For removable devices, you can also use :py:func:`IMachine.mount_medium`
                to change the media while the machine is running.



                In a VM's default configuration of virtual machines, the secondary
                master of the IDE controller is used for a CD/DVD drive.

                After calling this returns successfully, a new instance of
                :py:class:`IMediumAttachment`  will appear in the machine's list of medium
                attachments (see :py:func:`IMachine.medium_attachments` ).

                See :py:class:`IMedium`  and :py:class:`IMediumAttachment`  for more
                information about attaching media.

                The specified device slot must not have a device attached to it,
                or this method will fail.


                You cannot attach a device to a newly created machine until
                this machine's settings are saved to disk using
                :py:func:`save_settings` .


                If the medium is being attached indirectly, a new differencing medium
                will implicitly be created for it and attached instead. If the
                changes made to the machine settings (including this indirect
                attachment) are later cancelled using :py:func:`discard_settings` ,
                this implicitly created differencing medium will implicitly
                be deleted.

                in name of type str
                    Name of the storage controller to attach the device to.

                in controller_port of type int
                    Port to attach the device to. For an IDE controller, 0 specifies
                    the primary controller and 1 specifies the secondary controller.
                    For a SCSI controller, this must range from 0 to 15; for a SATA controller,
                    from 0 to 29; for an SAS controller, from 0 to 7.

                in device of type int
                    Device slot in the given port to attach the device to. This is only
                    relevant for IDE controllers, for which 0 specifies the master device and
                    1 specifies the slave device. For all other controller types, this must
                    be 0.

                in type_p of type :class:`DeviceType`
                    Device type of the attached device. For media opened by
                    :py:func:`IVirtualBox.open_medium` , this must match the device type
                    specified there.

                in medium of type :class:`IMedium`
                    Medium to mount or @c null for an empty drive.

                raises :class:`OleErrorInvalidarg`
                    SATA device, SATA port, IDE port or IDE slot out of range, or
        file or UUID not found.

                raises :class:`VBoxErrorInvalidObjectState`
                    Machine must be registered before media can be attached.

                raises :class:`VBoxErrorInvalidVmState`
                    Invalid machine state.

                raises :class:`VBoxErrorObjectInUse`
                    A medium is already attached to this or another virtual machine.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        if not isinstance(type_p, DeviceType):
            raise TypeError("type_p can only be an instance of type DeviceType")
        if not isinstance(medium, IMedium):
            raise TypeError("medium can only be an instance of type IMedium")
        self._call("attachDevice", in_p=[name, controller_port, device, type_p, medium])

    def attach_device_without_medium(self, name, controller_port, device, type_p):
        """Attaches a device and optionally mounts a medium to the given storage
                controller (:py:class:`IStorageController` , identified by @a name),
                at the indicated port and device.

                This method is intended for managing storage devices in general while a
                machine is powered off. It can be used to attach and detach fixed
                and removable media. The following kind of media can be attached
                to a machine:


                For fixed and removable media, you can pass in a medium that was
                previously opened using :py:func:`IVirtualBox.open_medium` .


                Only for storage devices supporting removable media (such as
                DVDs and floppies) with an empty drive or one of the medium objects listed
                in the :py:func:`IHost.dvd_drives`  and :py:func:`IHost.floppy_drives`
                arrays to indicate a host drive.
                For removable devices, you can also use :py:func:`IMachine.mount_medium`
                to change the media while the machine is running.



                In a VM's default configuration of virtual machines, the secondary
                master of the IDE controller is used for a CD/DVD drive.
                :py:class:`IMediumAttachment`  will appear in the machine's list of medium
                attachments (see :py:func:`IMachine.medium_attachments` ).

                See :py:class:`IMedium`  and :py:class:`IMediumAttachment`  for more
                information about attaching media.

                The specified device slot must not have a device attached to it,
                or this method will fail.

                You cannot attach a device to a newly created machine until
                this machine's settings are saved to disk using
                :py:func:`save_settings` .


                If the medium is being attached indirectly, a new differencing medium
                will implicitly be created for it and attached instead. If the
                changes made to the machine settings (including this indirect
                attachment) are later cancelled using :py:func:`discard_settings` ,
                this implicitly created differencing medium will implicitly
                be deleted.

                in name of type str
                    Name of the storage controller to attach the device to.

                in controller_port of type int
                    Port to attach the device to. For an IDE controller, 0 specifies
                    the primary controller and 1 specifies the secondary controller.
                    For a SCSI controller, this must range from 0 to 15; for a SATA controller,
                    from 0 to 29; for an SAS controller, from 0 to 7.

                in device of type int
                    Device slot in the given port to attach the device to. This is only
                    relevant for IDE controllers, for which 0 specifies the master device and
                    1 specifies the slave device. For all other controller types, this must
                    be 0.

                in type_p of type :class:`DeviceType`
                    Device type of the attached device. For media opened by
                    :py:func:`IVirtualBox.open_medium` , this must match the device type
                    specified there.

                raises :class:`OleErrorInvalidarg`
                    SATA device, SATA port, IDE port or IDE slot out of range, or
        file or UUID not found.

                raises :class:`VBoxErrorInvalidObjectState`
                    Machine must be registered before media can be attached.

                raises :class:`VBoxErrorInvalidVmState`
                    Invalid machine state.

                raises :class:`VBoxErrorObjectInUse`
                    A medium is already attached to this or another virtual machine.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        if not isinstance(type_p, DeviceType):
            raise TypeError("type_p can only be an instance of type DeviceType")
        self._call(
            "attachDeviceWithoutMedium", in_p=[name, controller_port, device, type_p]
        )

    def detach_device(self, name, controller_port, device):
        """Detaches the device attached to a device slot of the specified bus.

                Detaching the device from the virtual machine is deferred. This means
                that the medium remains associated with the machine when this method
                returns and gets actually de-associated only after a successful
                :py:func:`save_settings`  call. See :py:class:`IMedium`
                for more detailed information about attaching media.


                You cannot detach a device from a running machine.


                Detaching differencing media implicitly created by :py:func:`attach_device`  for the indirect attachment using this
                method will **not** implicitly delete them. The
                :py:func:`IMedium.delete_storage`  operation should be
                explicitly performed by the caller after the medium is successfully
                detached and the settings are saved with
                :py:func:`save_settings` , if it is the desired action.

                in name of type str
                    Name of the storage controller to detach the medium from.

                in controller_port of type int
                    Port number to detach the medium from.

                in device of type int
                    Device slot number to detach the medium from.

                raises :class:`VBoxErrorInvalidVmState`
                    Attempt to detach medium from a running virtual machine.

                raises :class:`VBoxErrorObjectNotFound`
                    No medium attached to given slot/bus.

                raises :class:`VBoxErrorNotSupported`
                    Medium format does not support storage deletion (only for implicitly
        created differencing media, should not happen).

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        self._call("detachDevice", in_p=[name, controller_port, device])

    def passthrough_device(self, name, controller_port, device, passthrough):
        """Sets the passthrough mode of an existing DVD device. Changing the
        setting while the VM is running is forbidden. The setting is only used
        if at VM start the device is configured as a host DVD drive, in all
        other cases it is ignored. The device must already exist; see
        :py:func:`IMachine.attach_device`  for how to attach a new device.

        The @a controllerPort and @a device parameters specify the device slot and
        have have the same meaning as with :py:func:`IMachine.attach_device` .

        in name of type str
            Name of the storage controller.

        in controller_port of type int
            Storage controller port.

        in device of type int
            Device slot in the given port.

        in passthrough of type bool
            New value for the passthrough setting.

        raises :class:`OleErrorInvalidarg`
            SATA device, SATA port, IDE port or IDE slot out of range.

        raises :class:`VBoxErrorInvalidObjectState`
            Attempt to modify an unregistered virtual machine.

        raises :class:`VBoxErrorInvalidVmState`
            Invalid machine state.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        if not isinstance(passthrough, bool):
            raise TypeError("passthrough can only be an instance of type bool")
        self._call(
            "passthroughDevice", in_p=[name, controller_port, device, passthrough]
        )

    def temporary_eject_device(self, name, controller_port, device, temporary_eject):
        """Sets the behavior for guest-triggered medium eject. In some situations
        it is desirable that such ejects update the VM configuration, and in
        others the eject should keep the VM configuration. The device must
        already exist; see :py:func:`IMachine.attach_device`  for how to
        attach a new device.

        The @a controllerPort and @a device parameters specify the device slot and
        have have the same meaning as with :py:func:`IMachine.attach_device` .

        in name of type str
            Name of the storage controller.

        in controller_port of type int
            Storage controller port.

        in device of type int
            Device slot in the given port.

        in temporary_eject of type bool
            New value for the eject behavior.

        raises :class:`OleErrorInvalidarg`
            SATA device, SATA port, IDE port or IDE slot out of range.

        raises :class:`VBoxErrorInvalidObjectState`
            Attempt to modify an unregistered virtual machine.

        raises :class:`VBoxErrorInvalidVmState`
            Invalid machine state.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        if not isinstance(temporary_eject, bool):
            raise TypeError("temporary_eject can only be an instance of type bool")
        self._call(
            "temporaryEjectDevice",
            in_p=[name, controller_port, device, temporary_eject],
        )

    def non_rotational_device(self, name, controller_port, device, non_rotational):
        """Sets a flag in the device information which indicates that the medium
        is not based on rotational technology, i.e. that the access times are
        more or less independent of the position on the medium. This may or may
        not be supported by a particular drive, and is silently ignored in the
        latter case. At the moment only hard disks (which is a misnomer in this
        context) accept this setting. Changing the setting while the VM is
        running is forbidden. The device must already exist; see
        :py:func:`IMachine.attach_device`  for how to attach a new device.

        The @a controllerPort and @a device parameters specify the device slot and
        have have the same meaning as with :py:func:`IMachine.attach_device` .

        in name of type str
            Name of the storage controller.

        in controller_port of type int
            Storage controller port.

        in device of type int
            Device slot in the given port.

        in non_rotational of type bool
            New value for the non-rotational device flag.

        raises :class:`OleErrorInvalidarg`
            SATA device, SATA port, IDE port or IDE slot out of range.

        raises :class:`VBoxErrorInvalidObjectState`
            Attempt to modify an unregistered virtual machine.

        raises :class:`VBoxErrorInvalidVmState`
            Invalid machine state.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        if not isinstance(non_rotational, bool):
            raise TypeError("non_rotational can only be an instance of type bool")
        self._call(
            "nonRotationalDevice", in_p=[name, controller_port, device, non_rotational]
        )

    def set_auto_discard_for_device(self, name, controller_port, device, discard):
        """Sets a flag in the device information which indicates that the medium
        supports discarding unused blocks (called trimming for SATA or unmap
        for SCSI devices) .This may or may not be supported by a particular drive,
        and is silently ignored in the latter case. At the moment only hard disks
        (which is a misnomer in this context) accept this setting. Changing the
        setting while the VM is running is forbidden. The device must already
        exist; see :py:func:`IMachine.attach_device`  for how to attach a new
        device.

        The @a controllerPort and @a device parameters specify the device slot and
        have have the same meaning as with :py:func:`IMachine.attach_device` .

        in name of type str
            Name of the storage controller.

        in controller_port of type int
            Storage controller port.

        in device of type int
            Device slot in the given port.

        in discard of type bool
            New value for the discard device flag.

        raises :class:`OleErrorInvalidarg`
            SATA device, SATA port, SCSI port out of range.

        raises :class:`VBoxErrorInvalidObjectState`
            Attempt to modify an unregistered virtual machine.

        raises :class:`VBoxErrorInvalidVmState`
            Invalid machine state.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        if not isinstance(discard, bool):
            raise TypeError("discard can only be an instance of type bool")
        self._call(
            "setAutoDiscardForDevice", in_p=[name, controller_port, device, discard]
        )

    def set_hot_pluggable_for_device(
        self, name, controller_port, device, hot_pluggable
    ):
        """Sets a flag in the device information which indicates that the attached
        device is hot pluggable or not. This may or may not be supported by a
        particular controller and/or drive, and is silently ignored in the
        latter case. Changing the setting while the VM is running is forbidden.
        The device must already exist; see :py:func:`IMachine.attach_device`
        for how to attach a new device.

        The @a controllerPort and @a device parameters specify the device slot and
        have have the same meaning as with :py:func:`IMachine.attach_device` .

        in name of type str
            Name of the storage controller.

        in controller_port of type int
            Storage controller port.

        in device of type int
            Device slot in the given port.

        in hot_pluggable of type bool
            New value for the hot-pluggable device flag.

        raises :class:`OleErrorInvalidarg`
            SATA device, SATA port, IDE port or IDE slot out of range.

        raises :class:`VBoxErrorInvalidObjectState`
            Attempt to modify an unregistered virtual machine.

        raises :class:`VBoxErrorInvalidVmState`
            Invalid machine state.

        raises :class:`VBoxErrorNotSupported`
            Controller doesn't support hot plugging.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        if not isinstance(hot_pluggable, bool):
            raise TypeError("hot_pluggable can only be an instance of type bool")
        self._call(
            "setHotPluggableForDevice",
            in_p=[name, controller_port, device, hot_pluggable],
        )

    def set_bandwidth_group_for_device(
        self, name, controller_port, device, bandwidth_group
    ):
        """Sets the bandwidth group of an existing storage device.
        The device must already exist; see :py:func:`IMachine.attach_device`
        for how to attach a new device.

        The @a controllerPort and @a device parameters specify the device slot and
        have have the same meaning as with :py:func:`IMachine.attach_device` .

        in name of type str
            Name of the storage controller.

        in controller_port of type int
            Storage controller port.

        in device of type int
            Device slot in the given port.

        in bandwidth_group of type :class:`IBandwidthGroup`
            New value for the bandwidth group or @c null for no group.

        raises :class:`OleErrorInvalidarg`
            SATA device, SATA port, IDE port or IDE slot out of range.

        raises :class:`VBoxErrorInvalidObjectState`
            Attempt to modify an unregistered virtual machine.

        raises :class:`VBoxErrorInvalidVmState`
            Invalid machine state.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        if not isinstance(bandwidth_group, IBandwidthGroup):
            raise TypeError(
                "bandwidth_group can only be an instance of type IBandwidthGroup"
            )
        self._call(
            "setBandwidthGroupForDevice",
            in_p=[name, controller_port, device, bandwidth_group],
        )

    def set_no_bandwidth_group_for_device(self, name, controller_port, device):
        """Sets no bandwidth group for an existing storage device.
        The device must already exist; see :py:func:`IMachine.attach_device`
        for how to attach a new device.
        The @a controllerPort and @a device parameters specify the device slot and
        have have the same meaning as with :py:func:`IMachine.attach_device` .

        in name of type str
            Name of the storage controller.

        in controller_port of type int
            Storage controller port.

        in device of type int
            Device slot in the given port.

        raises :class:`OleErrorInvalidarg`
            SATA device, SATA port, IDE port or IDE slot out of range.

        raises :class:`VBoxErrorInvalidObjectState`
            Attempt to modify an unregistered virtual machine.

        raises :class:`VBoxErrorInvalidVmState`
            Invalid machine state.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        self._call("setNoBandwidthGroupForDevice", in_p=[name, controller_port, device])

    def unmount_medium(self, name, controller_port, device, force):
        """Unmounts any currently mounted medium (:py:class:`IMedium` ,
        identified by the given UUID @a id) to the given storage controller
        (:py:class:`IStorageController` , identified by @a name),
        at the indicated port and device. The device must already exist;

        This method is intended only for managing removable media, where the
        device is fixed but media is changeable at runtime (such as DVDs
        and floppies). It cannot be used for fixed media such as hard disks.

        The @a controllerPort and @a device parameters specify the device slot
        and have have the same meaning as with
        :py:func:`IMachine.attach_device` .

        The specified device slot must have a medium mounted, which will be
        unmounted. If there is no mounted medium it will do nothing.
        See :py:class:`IMedium`  for more detailed information about
        attaching/unmounting media.

        in name of type str
            Name of the storage controller to unmount the medium from.

        in controller_port of type int
            Port to unmount the medium from.

        in device of type int
            Device slot in the given port to unmount the medium from.

        in force of type bool
            Allows to force unmount of a medium which is locked by
            the device slot in the given port medium is attached to.

        raises :class:`OleErrorInvalidarg`
            SATA device, SATA port, IDE port or IDE slot out of range.

        raises :class:`VBoxErrorInvalidObjectState`
            Attempt to unmount medium that is not removable - not DVD or floppy.

        raises :class:`VBoxErrorInvalidVmState`
            Invalid machine state.

        raises :class:`VBoxErrorObjectInUse`
            Medium already attached to this or another virtual machine.

        raises :class:`VBoxErrorObjectNotFound`
            Medium not attached to specified port, device, controller.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        if not isinstance(force, bool):
            raise TypeError("force can only be an instance of type bool")
        self._call("unmountMedium", in_p=[name, controller_port, device, force])

    def mount_medium(self, name, controller_port, device, medium, force):
        """Mounts a medium (:py:class:`IMedium` , identified
        by the given UUID @a id) to the given storage controller
        (:py:class:`IStorageController` , identified by @a name),
        at the indicated port and device. The device must already exist;
        see :py:func:`IMachine.attach_device`  for how to attach a new device.

        This method is intended only for managing removable media, where the
        device is fixed but media is changeable at runtime (such as DVDs
        and floppies). It cannot be used for fixed media such as hard disks.

        The @a controllerPort and @a device parameters specify the device slot and
        have have the same meaning as with :py:func:`IMachine.attach_device` .

        The specified device slot can have a medium mounted, which will be
        unmounted first. Specifying a zero UUID (or an empty string) for
        @a medium does just an unmount.

        See :py:class:`IMedium`  for more detailed information about
        attaching media.

        in name of type str
            Name of the storage controller to attach the medium to.

        in controller_port of type int
            Port to attach the medium to.

        in device of type int
            Device slot in the given port to attach the medium to.

        in medium of type :class:`IMedium`
            Medium to mount or @c null for an empty drive.

        in force of type bool
            Allows to force unmount/mount of a medium which is locked by
            the device slot in the given port to attach the medium to.

        raises :class:`OleErrorInvalidarg`
            SATA device, SATA port, IDE port or IDE slot out of range.

        raises :class:`VBoxErrorInvalidObjectState`
            Attempt to attach medium to an unregistered virtual machine.

        raises :class:`VBoxErrorInvalidVmState`
            Invalid machine state.

        raises :class:`VBoxErrorObjectInUse`
            Medium already attached to this or another virtual machine.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        if not isinstance(medium, IMedium):
            raise TypeError("medium can only be an instance of type IMedium")
        if not isinstance(force, bool):
            raise TypeError("force can only be an instance of type bool")
        self._call("mountMedium", in_p=[name, controller_port, device, medium, force])

    def get_medium(self, name, controller_port, device):
        """Returns the virtual medium attached to a device slot of the specified
        bus.

        Note that if the medium was indirectly attached by
        :py:func:`mount_medium`  to the given device slot then this
        method will return not the same object as passed to the
        :py:func:`mount_medium`  call. See :py:class:`IMedium`  for
        more detailed information about mounting a medium.

        in name of type str
            Name of the storage controller the medium is attached to.

        in controller_port of type int
            Port to query.

        in device of type int
            Device slot in the given port to query.

        return medium of type :class:`IMedium`
            Attached medium object.

        raises :class:`VBoxErrorObjectNotFound`
            No medium attached to given slot/bus.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        medium = self._call("getMedium", in_p=[name, controller_port, device])
        medium = IMedium(medium)
        return medium

    def get_medium_attachments_of_controller(self, name):
        """Returns an array of medium attachments which are attached to the
        the controller with the given name.

        in name of type str

        return medium_attachments of type :class:`IMediumAttachment`

        raises :class:`VBoxErrorObjectNotFound`
            A storage controller with given name doesn't exist.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        medium_attachments = self._call("getMediumAttachmentsOfController", in_p=[name])
        medium_attachments = [IMediumAttachment(a) for a in medium_attachments]
        return medium_attachments

    def get_medium_attachment(self, name, controller_port, device):
        """Returns a medium attachment which corresponds to the controller with
        the given name, on the given port and device slot.

        in name of type str

        in controller_port of type int

        in device of type int

        return attachment of type :class:`IMediumAttachment`

        raises :class:`VBoxErrorObjectNotFound`
            No attachment exists for the given controller/port/device combination.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(controller_port, baseinteger):
            raise TypeError(
                "controller_port can only be an instance of type baseinteger"
            )
        if not isinstance(device, baseinteger):
            raise TypeError("device can only be an instance of type baseinteger")
        attachment = self._call(
            "getMediumAttachment", in_p=[name, controller_port, device]
        )
        attachment = IMediumAttachment(attachment)
        return attachment

    def attach_host_pci_device(
        self, host_address, desired_guest_address, try_to_unbind
    ):
        """Attaches host PCI device with the given (host) PCI address to the
        PCI bus of the virtual machine. Please note, that this operation
        is two phase, as real attachment will happen when VM will start,
        and most information will be delivered as IHostPCIDevicePlugEvent
        on IVirtualBox event source.

        :py:class:`IHostPCIDevicePlugEvent`

        in host_address of type int
            Address of the host PCI device.

        in desired_guest_address of type int
            Desired position of this device on guest PCI bus.

        in try_to_unbind of type bool
            If VMM shall try to unbind existing drivers from the
            device before attaching it to the guest.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine state is not stopped (PCI hotplug not yet implemented).

        raises :class:`VBoxErrorPdmError`
            Virtual machine does not have a PCI controller allowing attachment of physical devices.

        raises :class:`VBoxErrorNotSupported`
            Hardware or host OS doesn't allow PCI device passthrough.

        """
        if not isinstance(host_address, baseinteger):
            raise TypeError("host_address can only be an instance of type baseinteger")
        if not isinstance(desired_guest_address, baseinteger):
            raise TypeError(
                "desired_guest_address can only be an instance of type baseinteger"
            )
        if not isinstance(try_to_unbind, bool):
            raise TypeError("try_to_unbind can only be an instance of type bool")
        self._call(
            "attachHostPCIDevice",
            in_p=[host_address, desired_guest_address, try_to_unbind],
        )

    def detach_host_pci_device(self, host_address):
        """Detach host PCI device from the virtual machine.
        Also HostPCIDevicePlugEvent on IVirtualBox event source
        will be delivered. As currently we don't support hot device
        unplug, IHostPCIDevicePlugEvent event is delivered immediately.

        :py:class:`IHostPCIDevicePlugEvent`

        in host_address of type int
            Address of the host PCI device.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine state is not stopped (PCI hotplug not yet implemented).

        raises :class:`VBoxErrorObjectNotFound`
            This host device is not attached to this machine.

        raises :class:`VBoxErrorPdmError`
            Virtual machine does not have a PCI controller allowing attachment of physical devices.

        raises :class:`VBoxErrorNotSupported`
            Hardware or host OS doesn't allow PCI device passthrough.

        """
        if not isinstance(host_address, baseinteger):
            raise TypeError("host_address can only be an instance of type baseinteger")
        self._call("detachHostPCIDevice", in_p=[host_address])

    def get_network_adapter(self, slot):
        """Returns the network adapter associated with the given slot.
        Slots are numbered sequentially, starting with zero. The total
        number of adapters per machine is defined by the
        :py:func:`ISystemProperties.get_max_network_adapters`  property,
        so the maximum slot number is one less than that property's value.

        in slot of type int

        return adapter of type :class:`INetworkAdapter`

        raises :class:`OleErrorInvalidarg`
            Invalid @a slot number.

        """
        if not isinstance(slot, baseinteger):
            raise TypeError("slot can only be an instance of type baseinteger")
        adapter = self._call("getNetworkAdapter", in_p=[slot])
        adapter = INetworkAdapter(adapter)
        return adapter

    def add_storage_controller(self, name, connection_type):
        """Adds a new storage controller (SCSI, SAS or SATA controller) to the
        machine and returns it as an instance of
        :py:class:`IStorageController` .

        @a name identifies the controller for subsequent calls such as
        :py:func:`get_storage_controller_by_name` ,
        :py:func:`get_storage_controller_by_instance` ,
        :py:func:`remove_storage_controller` ,
        :py:func:`attach_device`  or :py:func:`mount_medium` .

        After the controller has been added, you can set its exact
        type by setting the :py:func:`IStorageController.controller_type` .

        in name of type str

        in connection_type of type :class:`StorageBus`

        return controller of type :class:`IStorageController`

        raises :class:`VBoxErrorObjectInUse`
            A storage controller with given name exists already.

        raises :class:`OleErrorInvalidarg`
            Invalid @a controllerType.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(connection_type, StorageBus):
            raise TypeError(
                "connection_type can only be an instance of type StorageBus"
            )
        controller = self._call("addStorageController", in_p=[name, connection_type])
        controller = IStorageController(controller)
        return controller

    def get_storage_controller_by_name(self, name):
        """Returns a storage controller with the given name.

        in name of type str

        return storage_controller of type :class:`IStorageController`

        raises :class:`VBoxErrorObjectNotFound`
            A storage controller with given name doesn't exist.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        storage_controller = self._call("getStorageControllerByName", in_p=[name])
        storage_controller = IStorageController(storage_controller)
        return storage_controller

    def get_storage_controller_by_instance(self, connection_type, instance):
        """Returns a storage controller of a specific storage bus
        with the given instance number.

        in connection_type of type :class:`StorageBus`

        in instance of type int

        return storage_controller of type :class:`IStorageController`

        raises :class:`VBoxErrorObjectNotFound`
            A storage controller with given instance number doesn't exist.

        """
        if not isinstance(connection_type, StorageBus):
            raise TypeError(
                "connection_type can only be an instance of type StorageBus"
            )
        if not isinstance(instance, baseinteger):
            raise TypeError("instance can only be an instance of type baseinteger")
        storage_controller = self._call(
            "getStorageControllerByInstance", in_p=[connection_type, instance]
        )
        storage_controller = IStorageController(storage_controller)
        return storage_controller

    def remove_storage_controller(self, name):
        """Removes a storage controller from the machine with all devices attached to it.

                in name of type str

                raises :class:`VBoxErrorObjectNotFound`
                    A storage controller with given name doesn't exist.

                raises :class:`VBoxErrorNotSupported`
                    Medium format does not support storage deletion (only for implicitly
        created differencing media, should not happen).

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        self._call("removeStorageController", in_p=[name])

    def set_storage_controller_bootable(self, name, bootable):
        """Sets the bootable flag of the storage controller with the given name.

        in name of type str

        in bootable of type bool

        raises :class:`VBoxErrorObjectNotFound`
            A storage controller with given name doesn't exist.

        raises :class:`VBoxErrorObjectInUse`
            Another storage controller is marked as bootable already.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(bootable, bool):
            raise TypeError("bootable can only be an instance of type bool")
        self._call("setStorageControllerBootable", in_p=[name, bootable])

    def add_usb_controller(self, name, type_p):
        """Adds a new USB controller to the machine and returns it as an instance of
        :py:class:`IUSBController` .

        in name of type str

        in type_p of type :class:`USBControllerType`

        return controller of type :class:`IUSBController`

        raises :class:`VBoxErrorObjectInUse`
            A USB controller with given type exists already.

        raises :class:`OleErrorInvalidarg`
            Invalid @a controllerType.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(type_p, USBControllerType):
            raise TypeError("type_p can only be an instance of type USBControllerType")
        controller = self._call("addUSBController", in_p=[name, type_p])
        controller = IUSBController(controller)
        return controller

    def remove_usb_controller(self, name):
        """Removes a USB controller from the machine.

        in name of type str

        raises :class:`VBoxErrorObjectNotFound`
            A USB controller with given type doesn't exist.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        self._call("removeUSBController", in_p=[name])

    def get_usb_controller_by_name(self, name):
        """Returns a USB controller with the given type.

        in name of type str

        return controller of type :class:`IUSBController`

        raises :class:`VBoxErrorObjectNotFound`
            A USB controller with given name doesn't exist.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        controller = self._call("getUSBControllerByName", in_p=[name])
        controller = IUSBController(controller)
        return controller

    def get_usb_controller_count_by_type(self, type_p):
        """Returns the number of USB controllers of the given type attached to the VM.

        in type_p of type :class:`USBControllerType`

        return controllers of type int

        """
        if not isinstance(type_p, USBControllerType):
            raise TypeError("type_p can only be an instance of type USBControllerType")
        controllers = self._call("getUSBControllerCountByType", in_p=[type_p])
        return controllers

    def get_serial_port(self, slot):
        """Returns the serial port associated with the given slot.
        Slots are numbered sequentially, starting with zero. The total
        number of serial ports per machine is defined by the
        :py:func:`ISystemProperties.serial_port_count`  property,
        so the maximum slot number is one less than that property's value.

        in slot of type int

        return port of type :class:`ISerialPort`

        raises :class:`OleErrorInvalidarg`
            Invalid @a slot number.

        """
        if not isinstance(slot, baseinteger):
            raise TypeError("slot can only be an instance of type baseinteger")
        port = self._call("getSerialPort", in_p=[slot])
        port = ISerialPort(port)
        return port

    def get_parallel_port(self, slot):
        """Returns the parallel port associated with the given slot.
        Slots are numbered sequentially, starting with zero. The total
        number of parallel ports per machine is defined by the
        :py:func:`ISystemProperties.parallel_port_count`  property,
        so the maximum slot number is one less than that property's value.

        in slot of type int

        return port of type :class:`IParallelPort`

        raises :class:`OleErrorInvalidarg`
            Invalid @a slot number.

        """
        if not isinstance(slot, baseinteger):
            raise TypeError("slot can only be an instance of type baseinteger")
        port = self._call("getParallelPort", in_p=[slot])
        port = IParallelPort(port)
        return port

    def get_extra_data_keys(self):
        """Returns an array representing the machine-specific extra data keys
        which currently have values defined.

        return keys of type str
            Array of extra data keys.

        """
        keys = self._call("getExtraDataKeys")
        return keys

    def get_extra_data(self, key):
        """Returns associated machine-specific extra data.

        If the requested data @a key does not exist, this function will
        succeed and return an empty string in the @a value argument.

        in key of type str
            Name of the data key to get.

        return value of type str
            Value of the requested data key.

        raises :class:`VBoxErrorFileError`
            Settings file not accessible.

        raises :class:`VBoxErrorXmlError`
            Could not parse the settings file.

        """
        if not isinstance(key, basestring):
            raise TypeError("key can only be an instance of type basestring")
        value = self._call("getExtraData", in_p=[key])
        return value

    def set_extra_data(self, key, value):
        """Sets associated machine-specific extra data.

        If you pass @c null or an empty string as a key @a value, the given
        @a key will be deleted.


        Key must contain printable (non-control) UTF-8 characters only.


        Before performing the actual data change, this method will ask all
        registered event listeners using the
        :py:class:`IExtraDataCanChangeEvent`
        notification for a permission. If one of the listeners refuses the
        new value, the change will not be performed.


        On success, the
        :py:class:`IExtraDataChangedEvent`  notification
        is called to inform all registered listeners about a successful data
        change.


        This method can be called outside the machine session and therefore
        it's a caller's responsibility to handle possible race conditions
        when several clients change the same key at the same time.

        in key of type str
            Name of the data key to set.

        in value of type str
            Value to assign to the key.

        raises :class:`VBoxErrorFileError`
            Settings file not accessible.

        raises :class:`VBoxErrorXmlError`
            Could not parse the settings file.

        raises :class:`OleErrorInvalidarg`
            Key contains invalid characters.

        """
        if not isinstance(key, basestring):
            raise TypeError("key can only be an instance of type basestring")
        if not isinstance(value, basestring):
            raise TypeError("value can only be an instance of type basestring")
        self._call("setExtraData", in_p=[key, value])

    def get_cpu_property(self, property_p):
        """Returns the virtual CPU boolean value of the specified property.

        in property_p of type :class:`CPUPropertyType`
            Property type to query.

        return value of type bool
            Property value.

        raises :class:`OleErrorInvalidarg`
            Invalid property.

        """
        if not isinstance(property_p, CPUPropertyType):
            raise TypeError(
                "property_p can only be an instance of type CPUPropertyType"
            )
        value = self._call("getCPUProperty", in_p=[property_p])
        return value

    def set_cpu_property(self, property_p, value):
        """Sets the virtual CPU boolean value of the specified property.

        in property_p of type :class:`CPUPropertyType`
            Property type to query.

        in value of type bool
            Property value.

        raises :class:`OleErrorInvalidarg`
            Invalid property.

        """
        if not isinstance(property_p, CPUPropertyType):
            raise TypeError(
                "property_p can only be an instance of type CPUPropertyType"
            )
        if not isinstance(value, bool):
            raise TypeError("value can only be an instance of type bool")
        self._call("setCPUProperty", in_p=[property_p, value])

    def get_cpuid_leaf_by_ordinal(self, ordinal):
        """Used to enumerate CPUID information override values.

        in ordinal of type int
            The ordinal number of the leaf to get.

        out idx of type int
            CPUID leaf index.

        out idx_sub of type int
            CPUID leaf sub-index.

        out val_eax of type int
            CPUID leaf value for register eax.

        out val_ebx of type int
            CPUID leaf value for register ebx.

        out val_ecx of type int
            CPUID leaf value for register ecx.

        out val_edx of type int
            CPUID leaf value for register edx.

        raises :class:`OleErrorInvalidarg`
            Invalid ordinal number is out of range.

        """
        if not isinstance(ordinal, baseinteger):
            raise TypeError("ordinal can only be an instance of type baseinteger")
        (idx, idx_sub, val_eax, val_ebx, val_ecx, val_edx) = self._call(
            "getCPUIDLeafByOrdinal", in_p=[ordinal]
        )
        return (idx, idx_sub, val_eax, val_ebx, val_ecx, val_edx)

    def get_cpuid_leaf(self, idx, idx_sub):
        """Returns the virtual CPU cpuid information for the specified leaf.

        Currently supported index values for cpuid:
        Standard CPUID leaves: 0 - 0x1f
        Extended CPUID leaves: 0x80000000 - 0x8000001f
        VIA CPUID leaves:      0xc0000000 - 0xc000000f

        See the Intel, AMD and VIA programmer's manuals for detailed information
        about the CPUID instruction and its leaves.

        in idx of type int
            CPUID leaf index.

        in idx_sub of type int
            CPUID leaf sub-index (ECX).  Set to 0xffffffff (or 0) if not applicable.

        out val_eax of type int
            CPUID leaf value for register eax.

        out val_ebx of type int
            CPUID leaf value for register ebx.

        out val_ecx of type int
            CPUID leaf value for register ecx.

        out val_edx of type int
            CPUID leaf value for register edx.

        raises :class:`OleErrorInvalidarg`
            Invalid index.

        """
        if not isinstance(idx, baseinteger):
            raise TypeError("idx can only be an instance of type baseinteger")
        if not isinstance(idx_sub, baseinteger):
            raise TypeError("idx_sub can only be an instance of type baseinteger")
        (val_eax, val_ebx, val_ecx, val_edx) = self._call(
            "getCPUIDLeaf", in_p=[idx, idx_sub]
        )
        return (val_eax, val_ebx, val_ecx, val_edx)

    def set_cpuid_leaf(self, idx, idx_sub, val_eax, val_ebx, val_ecx, val_edx):
        """Sets the virtual CPU cpuid information for the specified leaf. Note that these values
        are not passed unmodified. VirtualBox clears features that it doesn't support.

        Currently supported index values for cpuid:
        Standard CPUID leaves: 0 - 0x1f
        Extended CPUID leaves: 0x80000000 - 0x8000001f
        VIA CPUID leaves:      0xc0000000 - 0xc000000f

        The subleaf index is only applicable to certain leaves (see manuals as this is
        subject to change).

        See the Intel, AMD and VIA programmer's manuals for detailed information
        about the cpuid instruction and its leaves.

        Do not use this method unless you know exactly what you're doing. Misuse can lead to
        random crashes inside VMs.

        in idx of type int
            CPUID leaf index.

        in idx_sub of type int
            CPUID leaf sub-index (ECX).  Set to 0xffffffff (or 0) if not applicable.
            The 0xffffffff causes it to remove all other subleaves before adding one
            with sub-index 0.

        in val_eax of type int
            CPUID leaf value for register eax.

        in val_ebx of type int
            CPUID leaf value for register ebx.

        in val_ecx of type int
            CPUID leaf value for register ecx.

        in val_edx of type int
            CPUID leaf value for register edx.

        raises :class:`OleErrorInvalidarg`
            Invalid index.

        """
        if not isinstance(idx, baseinteger):
            raise TypeError("idx can only be an instance of type baseinteger")
        if not isinstance(idx_sub, baseinteger):
            raise TypeError("idx_sub can only be an instance of type baseinteger")
        if not isinstance(val_eax, baseinteger):
            raise TypeError("val_eax can only be an instance of type baseinteger")
        if not isinstance(val_ebx, baseinteger):
            raise TypeError("val_ebx can only be an instance of type baseinteger")
        if not isinstance(val_ecx, baseinteger):
            raise TypeError("val_ecx can only be an instance of type baseinteger")
        if not isinstance(val_edx, baseinteger):
            raise TypeError("val_edx can only be an instance of type baseinteger")
        self._call(
            "setCPUIDLeaf", in_p=[idx, idx_sub, val_eax, val_ebx, val_ecx, val_edx]
        )

    def remove_cpuid_leaf(self, idx, idx_sub):
        """Removes the virtual CPU cpuid leaf for the specified index

        in idx of type int
            CPUID leaf index.

        in idx_sub of type int
            CPUID leaf sub-index (ECX).  Set to 0xffffffff (or 0) if not applicable.
            The 0xffffffff value works like a wildcard.

        raises :class:`OleErrorInvalidarg`
            Invalid index.

        """
        if not isinstance(idx, baseinteger):
            raise TypeError("idx can only be an instance of type baseinteger")
        if not isinstance(idx_sub, baseinteger):
            raise TypeError("idx_sub can only be an instance of type baseinteger")
        self._call("removeCPUIDLeaf", in_p=[idx, idx_sub])

    def remove_all_cpuid_leaves(self):
        """Removes all the virtual CPU cpuid leaves"""
        self._call("removeAllCPUIDLeaves")

    def get_hw_virt_ex_property(self, property_p):
        """Returns the value of the specified hardware virtualization boolean property.

        in property_p of type :class:`HWVirtExPropertyType`
            Property type to query.

        return value of type bool
            Property value.

        raises :class:`OleErrorInvalidarg`
            Invalid property.

        """
        if not isinstance(property_p, HWVirtExPropertyType):
            raise TypeError(
                "property_p can only be an instance of type HWVirtExPropertyType"
            )
        value = self._call("getHWVirtExProperty", in_p=[property_p])
        return value

    def set_hw_virt_ex_property(self, property_p, value):
        """Sets a new value for the specified hardware virtualization boolean property.

        in property_p of type :class:`HWVirtExPropertyType`
            Property type to set.

        in value of type bool
            New property value.

        raises :class:`OleErrorInvalidarg`
            Invalid property.

        """
        if not isinstance(property_p, HWVirtExPropertyType):
            raise TypeError(
                "property_p can only be an instance of type HWVirtExPropertyType"
            )
        if not isinstance(value, bool):
            raise TypeError("value can only be an instance of type bool")
        self._call("setHWVirtExProperty", in_p=[property_p, value])

    def set_settings_file_path(self, settings_file_path):
        """Currently, it is an error to change this property on any machine.
        Later this will allow setting a new path for the settings file, with
        automatic relocation of all files (including snapshots and disk images)
        which are inside the base directory. This operation is only allowed
        when there are no pending unsaved settings.


        Setting this property to @c null or to an empty string is forbidden.
        When setting this property, the specified path must be absolute.
        The specified path may not exist, it will be created when necessary.

        in settings_file_path of type str
            New settings file path, will be used to determine the new
            location for the attached media if it is in the same directory or
            below as the original settings file.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`OleErrorNotimpl`
            The operation is not implemented yet.

        """
        if not isinstance(settings_file_path, basestring):
            raise TypeError(
                "settings_file_path can only be an instance of type basestring"
            )
        progress = self._call("setSettingsFilePath", in_p=[settings_file_path])
        progress = IProgress(progress)
        return progress

    def save_settings(self):
        """Saves any changes to machine settings made since the session
        has been opened or a new machine has been created, or since the
        last call to :py:func:`save_settings`  or :py:func:`discard_settings` .
        For registered machines, new settings become visible to all
        other VirtualBox clients after successful invocation of this
        method.

        The method sends :py:class:`IMachineDataChangedEvent`
        notification event after the configuration has been successfully
        saved (only for registered machines).


        Calling this method is only valid on instances returned
        by :py:func:`ISession.machine`  and on new machines
        created by :py:func:`IVirtualBox.create_machine`  but not
        yet registered, or on unregistered machines after calling
        :py:func:`IMachine.unregister` .

        raises :class:`VBoxErrorFileError`
            Settings file not accessible.

        raises :class:`VBoxErrorXmlError`
            Could not parse the settings file.

        raises :class:`OleErrorAccessdenied`
            Modification request refused.

        """
        self._call("saveSettings")

    def discard_settings(self):
        """Discards any changes to the machine settings made since the session
        has been opened or since the last call to :py:func:`save_settings`
        or :py:func:`discard_settings` .

        Calling this method is only valid on instances returned
        by :py:func:`ISession.machine`  and on new machines
        created by :py:func:`IVirtualBox.create_machine`  or
        opened by :py:func:`IVirtualBox.open_machine`  but not
        yet registered, or on unregistered machines after calling
        :py:func:`IMachine.unregister` .

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine is not mutable.

        """
        self._call("discardSettings")

    def unregister(self, cleanup_mode):
        """Unregisters a machine previously registered with
        :py:func:`IVirtualBox.register_machine`  and optionally do additional
        cleanup before the machine is unregistered.

        This method does not delete any files. It only changes the machine configuration and
        the list of registered machines in the VirtualBox object. To delete the files which
        belonged to the machine, including the XML file of the machine itself, call
        :py:func:`delete_config` , optionally with the array of IMedium objects which was returned
        from this method.

        How thoroughly this method cleans up the machine configuration before unregistering
        the machine depends on the @a cleanupMode argument.


        With "UnregisterOnly", the machine will only be unregistered, but no additional
        cleanup will be performed. The call will fail if the machine is in "Saved" state
        or has any snapshots or any media attached (see :py:class:`IMediumAttachment` ).
        It is the responsibility of the caller to delete all such configuration in this mode.
        In this mode, the API behaves like the former @c IVirtualBox::unregisterMachine() API
        which it replaces.
        With "DetachAllReturnNone", the call will succeed even if the machine is in "Saved"
        state or if it has snapshots or media attached. All media attached to the current machine
        state or in snapshots will be detached. No medium objects will be returned;
        all of the machine's media will remain open.
        With "DetachAllReturnHardDisksOnly", the call will behave like with "DetachAllReturnNone",
        except that all the hard disk medium objects which were detached from the machine will
        be returned as an array. This allows for quickly passing them to the :py:func:`delete_config`
        API for closing and deletion.
        With "Full", the call will behave like with "DetachAllReturnHardDisksOnly", except
        that all media will be returned in the array, including removable media like DVDs and
        floppies. This might be useful if the user wants to inspect in detail which media were
        attached to the machine. Be careful when passing the media array to :py:func:`delete_config`
        in that case because users will typically want to preserve ISO and RAW image files.


        A typical implementation will use "DetachAllReturnHardDisksOnly" and then pass the
        resulting IMedium array to :py:func:`delete_config` . This way, the machine is completely
        deleted with all its saved states and hard disk images, but images for removable
        drives (such as ISO and RAW files) will remain on disk.

        This API does not verify whether the media files returned in the array are still
        attached to other machines (i.e. shared between several machines). If such a shared
        image is passed to :py:func:`delete_config`  however, closing the image will fail there
        and the image will be silently skipped.

        This API may, however, move media from this machine's media registry to other media
        registries (see :py:class:`IMedium`  for details on media registries). For machines
        created with VirtualBox 4.0 or later, if media from this machine's media registry
        are also attached to another machine (shared attachments), each such medium will be
        moved to another machine's registry. This is because without this machine's media
        registry, the other machine cannot find its media any more and would become inaccessible.

        This API implicitly calls :py:func:`save_settings`  to save all current machine settings
        before unregistering it. It may also silently call :py:func:`save_settings`  on other machines
        if media are moved to other machines' media registries.

        After successful method invocation, the :py:class:`IMachineRegisteredEvent`  event
        is fired.

        The call will fail if the machine is currently locked (see :py:class:`ISession` ).


        If the given machine is inaccessible (see :py:func:`accessible` ), it
        will be unregistered and fully uninitialized right afterwards. As a result,
        the returned machine object will be unusable and an attempt to call
        **any** method will return the "Object not ready" error.

        in cleanup_mode of type :class:`CleanupMode`
            How to clean up after the machine has been unregistered.

        return media of type :class:`IMedium`
            List of media detached from the machine, depending on the @a cleanupMode parameter.

        raises :class:`VBoxErrorInvalidObjectState`
            Machine is currently locked for a session.

        """
        if not isinstance(cleanup_mode, CleanupMode):
            raise TypeError("cleanup_mode can only be an instance of type CleanupMode")
        media = self._call("unregister", in_p=[cleanup_mode])
        media = [IMedium(a) for a in media]
        return media

    def delete_config(self, media):
        """Deletes the files associated with this machine from disk. If medium objects are passed
        in with the @a aMedia argument, they are closed and, if closing was successful, their
        storage files are deleted as well. For convenience, this array of media files can be
        the same as the one returned from a previous :py:func:`unregister`  call.

        This method must only be called on machines which are either write-locked (i.e. on instances
        returned by :py:func:`ISession.machine` ) or on unregistered machines (i.e. not yet
        registered machines created by :py:func:`IVirtualBox.create_machine`  or opened by
        :py:func:`IVirtualBox.open_machine` , or after having called :py:func:`unregister` ).

        The following files will be deleted by this method:

        If :py:func:`unregister`  had been previously called with a @a cleanupMode
        argument other than "UnregisterOnly", this will delete all saved state files that
        the machine had in use; possibly one if the machine was in "Saved" state and one
        for each online snapshot that the machine had.
        On each medium object passed in the @a aMedia array, this will call
        :py:func:`IMedium.close` . If that succeeds, this will attempt to delete the
        medium's storage on disk. Since the :py:func:`IMedium.close`  call will fail if the medium is still
        in use, e.g. because it is still attached to a second machine; in that case the
        storage will not be deleted.
        Finally, the machine's own XML file will be deleted.


        Since deleting large disk image files can be a time-consuming I/O operation, this
        method operates asynchronously and returns an IProgress object to allow the caller
        to monitor the progress. There will be one sub-operation for each file that is
        being deleted (saved state or medium storage file).


        :py:func:`settings_modified`  will return @c true after this
        method successfully returns.

        in media of type :class:`IMedium`
            List of media to be closed and whose storage files will be deleted.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorInvalidVmState`
            Machine is registered but not write-locked.

        raises :class:`VBoxErrorIprtError`
            Could not delete the settings file.

        """
        if not isinstance(media, list):
            raise TypeError("media can only be an instance of type list")
        for a in media[:10]:
            if not isinstance(a, IMedium):
                raise TypeError("array can only contain objects of type IMedium")
        progress = self._call("deleteConfig", in_p=[media])
        progress = IProgress(progress)
        return progress

    def export_to(self, appliance, location):
        """Exports the machine to an OVF appliance. See :py:class:`IAppliance`  for the
        steps required to export VirtualBox machines to OVF.

        in appliance of type :class:`IAppliance`
            Appliance to export this machine to.

        in location of type str
            The target location.

        return description of type :class:`IVirtualSystemDescription`
            VirtualSystemDescription object which is created for this machine.

        """
        if not isinstance(appliance, IAppliance):
            raise TypeError("appliance can only be an instance of type IAppliance")
        if not isinstance(location, basestring):
            raise TypeError("location can only be an instance of type basestring")
        description = self._call("exportTo", in_p=[appliance, location])
        description = IVirtualSystemDescription(description)
        return description

    def find_snapshot(self, name_or_id):
        """Returns a snapshot of this machine with the given name or UUID.

        Returns a snapshot of this machine with the given UUID.
        A @c null argument can be used to obtain the first snapshot
        taken on this machine. To traverse the whole tree of snapshots
        starting from the root, inspect the root snapshot's
        :py:func:`ISnapshot.children`  attribute and recurse over those children.

        in name_or_id of type str
            What to search for. Name or UUID of the snapshot to find

        return snapshot of type :class:`ISnapshot`
            Snapshot object with the given name.

        raises :class:`VBoxErrorObjectNotFound`
            Virtual machine has no snapshots or snapshot not found.

        """
        if not isinstance(name_or_id, basestring):
            raise TypeError("name_or_id can only be an instance of type basestring")
        snapshot = self._call("findSnapshot", in_p=[name_or_id])
        snapshot = ISnapshot(snapshot)
        return snapshot

    def create_shared_folder(
        self, name, host_path, writable, automount, auto_mount_point
    ):
        """Creates a new permanent shared folder by associating the given logical
        name with the given host path, adds it to the collection of shared
        folders and starts sharing it. Refer to the description of
        :py:class:`ISharedFolder`  to read more about logical names.

        in name of type str
            Unique logical name of the shared folder.

        in host_path of type str
            Full path to the shared folder in the host file system.

        in writable of type bool
            Whether the share is writable or read-only.

        in automount of type bool
            Whether the share gets automatically mounted by the guest
            or not.

        in auto_mount_point of type str
            Where the guest should automatically mount the folder, if possible.
            For Windows and OS/2 guests this should be a drive letter, while other
            guests it should be a absolute directory.

        raises :class:`VBoxErrorObjectInUse`
            Shared folder already exists.

        raises :class:`VBoxErrorFileError`
            Shared folder @a hostPath not accessible.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(host_path, basestring):
            raise TypeError("host_path can only be an instance of type basestring")
        if not isinstance(writable, bool):
            raise TypeError("writable can only be an instance of type bool")
        if not isinstance(automount, bool):
            raise TypeError("automount can only be an instance of type bool")
        if not isinstance(auto_mount_point, basestring):
            raise TypeError(
                "auto_mount_point can only be an instance of type basestring"
            )
        self._call(
            "createSharedFolder",
            in_p=[name, host_path, writable, automount, auto_mount_point],
        )

    def remove_shared_folder(self, name):
        """Removes the permanent shared folder with the given name previously
        created by :py:func:`create_shared_folder`  from the collection of
        shared folders and stops sharing it.

        in name of type str
            Logical name of the shared folder to remove.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine is not mutable.

        raises :class:`VBoxErrorObjectNotFound`
            Shared folder @a name does not exist.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        self._call("removeSharedFolder", in_p=[name])

    def can_show_console_window(self):
        """Returns @c true if the VM console process can activate the
        console window and bring it to foreground on the desktop of
        the host PC.

        This method will fail if a session for this machine is not
        currently open.

        return can_show of type bool
            @c true if the console window can be shown and @c false otherwise.

        raises :class:`VBoxErrorInvalidVmState`
            Machine session is not open.

        """
        can_show = self._call("canShowConsoleWindow")
        return can_show

    def show_console_window(self):
        """Activates the console window and brings it to foreground on
        the desktop of the host PC. Many modern window managers on
        many platforms implement some sort of focus stealing
        prevention logic, so that it may be impossible to activate
        a window without the help of the currently active
        application. In this case, this method will return a non-zero
        identifier that represents the top-level window of the VM
        console process. The caller, if it represents a currently
        active process, is responsible to use this identifier (in a
        platform-dependent manner) to perform actual window
        activation.

        This method will fail if a session for this machine is not
        currently open.

        return win_id of type int
            Platform-dependent identifier of the top-level VM console
            window, or zero if this method has performed all actions
            necessary to implement the *show window* semantics for
            the given platform and/or VirtualBox front-end.

        raises :class:`VBoxErrorInvalidVmState`
            Machine session is not open.

        """
        win_id = self._call("showConsoleWindow")
        return win_id

    def get_guest_property(self, name):
        """Reads an entry from the machine's guest property store.

        in name of type str
            The name of the property to read.

        out value of type str
            The value of the property. If the property does not exist then this
            will be empty.

        out timestamp of type int
            The time at which the property was last modified, as seen by the
            server process.

        out flags of type str
            Additional property parameters, passed as a comma-separated list of
            "name=value" type entries.

        raises :class:`VBoxErrorInvalidVmState`
            Machine session is not open.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        (value, timestamp, flags) = self._call("getGuestProperty", in_p=[name])
        return (value, timestamp, flags)

    def get_guest_property_value(self, property_p):
        """Reads a value from the machine's guest property store.

        in property_p of type str
            The name of the property to read.

        return value of type str
            The value of the property. If the property does not exist then this
            will be empty.

        raises :class:`VBoxErrorInvalidVmState`
            Machine session is not open.

        """
        if not isinstance(property_p, basestring):
            raise TypeError("property_p can only be an instance of type basestring")
        value = self._call("getGuestPropertyValue", in_p=[property_p])
        return value

    def get_guest_property_timestamp(self, property_p):
        """Reads a property timestamp from the machine's guest property store.

        in property_p of type str
            The name of the property to read.

        return value of type int
            The timestamp. If the property does not exist then this will be
            empty.

        raises :class:`VBoxErrorInvalidVmState`
            Machine session is not open.

        """
        if not isinstance(property_p, basestring):
            raise TypeError("property_p can only be an instance of type basestring")
        value = self._call("getGuestPropertyTimestamp", in_p=[property_p])
        return value

    def set_guest_property(self, property_p, value, flags):
        """Sets, changes or deletes an entry in the machine's guest property
        store.

        in property_p of type str
            The name of the property to set, change or delete.

        in value of type str
            The new value of the property to set, change or delete. If the
            property does not yet exist and value is non-empty, it will be
            created. If the value is @c null or empty, the property will be
            deleted if it exists.

        in flags of type str
            Additional property parameters, passed as a comma-separated list of
            "name=value" type entries.

        raises :class:`OleErrorAccessdenied`
            Property cannot be changed.

        raises :class:`OleErrorInvalidarg`
            Invalid @a flags.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine is not mutable or session not open.

        raises :class:`VBoxErrorInvalidObjectState`
            Cannot set transient property when machine not running.

        """
        if not isinstance(property_p, basestring):
            raise TypeError("property_p can only be an instance of type basestring")
        if not isinstance(value, basestring):
            raise TypeError("value can only be an instance of type basestring")
        if not isinstance(flags, basestring):
            raise TypeError("flags can only be an instance of type basestring")
        self._call("setGuestProperty", in_p=[property_p, value, flags])

    def set_guest_property_value(self, property_p, value):
        """Sets or changes a value in the machine's guest property
        store. The flags field will be left unchanged or created empty for a
        new property.

        in property_p of type str
            The name of the property to set or change.

        in value of type str
            The new value of the property to set or change. If the
            property does not yet exist and value is non-empty, it will be
            created.

        raises :class:`OleErrorAccessdenied`
            Property cannot be changed.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine is not mutable or session not open.

        raises :class:`VBoxErrorInvalidObjectState`
            Cannot set transient property when machine not running.

        """
        if not isinstance(property_p, basestring):
            raise TypeError("property_p can only be an instance of type basestring")
        if not isinstance(value, basestring):
            raise TypeError("value can only be an instance of type basestring")
        self._call("setGuestPropertyValue", in_p=[property_p, value])

    def delete_guest_property(self, name):
        """Deletes an entry from the machine's guest property store.

        in name of type str
            The name of the property to delete.

        raises :class:`VBoxErrorInvalidVmState`
            Machine session is not open.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        self._call("deleteGuestProperty", in_p=[name])

    def enumerate_guest_properties(self, patterns):
        """Return a list of the guest properties matching a set of patterns along
        with their values, timestamps and flags.

        in patterns of type str
            The patterns to match the properties against, separated by '|'
            characters. If this is empty or @c null, all properties will match.

        out names of type str
            The names of the properties returned.

        out values of type str
            The values of the properties returned. The array entries match the
            corresponding entries in the @a name array.

        out timestamps of type int
            The timestamps of the properties returned. The array entries match
            the corresponding entries in the @a name array.

        out flags of type str
            The flags of the properties returned. The array entries match the
            corresponding entries in the @a name array.

        """
        if not isinstance(patterns, basestring):
            raise TypeError("patterns can only be an instance of type basestring")
        (names, values, timestamps, flags) = self._call(
            "enumerateGuestProperties", in_p=[patterns]
        )
        return (names, values, timestamps, flags)

    def query_saved_guest_screen_info(self, screen_id):
        """Returns the guest dimensions from the saved state.

        in screen_id of type int
            Saved guest screen to query info from.

        out origin_x of type int
            The X position of the guest monitor top left corner.

        out origin_y of type int
            The Y position of the guest monitor top left corner.

        out width of type int
            Guest width at the time of the saved state was taken.

        out height of type int
            Guest height at the time of the saved state was taken.

        out enabled of type bool
            Whether the monitor is enabled in the guest.

        """
        if not isinstance(screen_id, baseinteger):
            raise TypeError("screen_id can only be an instance of type baseinteger")
        (origin_x, origin_y, width, height, enabled) = self._call(
            "querySavedGuestScreenInfo", in_p=[screen_id]
        )
        return (origin_x, origin_y, width, height, enabled)

    def read_saved_thumbnail_to_array(self, screen_id, bitmap_format):
        """Thumbnail is retrieved to an array of bytes in the requested format.

        in screen_id of type int
            Saved guest screen to read from.

        in bitmap_format of type :class:`BitmapFormat`
            The requested format.

        out width of type int
            Bitmap width.

        out height of type int
            Bitmap height.

        return data of type str
            Array with resulting bitmap data.

        """
        if not isinstance(screen_id, baseinteger):
            raise TypeError("screen_id can only be an instance of type baseinteger")
        if not isinstance(bitmap_format, BitmapFormat):
            raise TypeError(
                "bitmap_format can only be an instance of type BitmapFormat"
            )
        (data, width, height) = self._call(
            "readSavedThumbnailToArray", in_p=[screen_id, bitmap_format]
        )
        return (data, width, height)

    def query_saved_screenshot_info(self, screen_id):
        """Returns available formats and size of the screenshot from saved state.

        in screen_id of type int
            Saved guest screen to query info from.

        out width of type int
            Image width.

        out height of type int
            Image height.

        return bitmap_formats of type :class:`BitmapFormat`
            Formats supported by readSavedScreenshotToArray.

        """
        if not isinstance(screen_id, baseinteger):
            raise TypeError("screen_id can only be an instance of type baseinteger")
        (bitmap_formats, width, height) = self._call(
            "querySavedScreenshotInfo", in_p=[screen_id]
        )
        bitmap_formats = [BitmapFormat(a) for a in bitmap_formats]
        return (bitmap_formats, width, height)

    def read_saved_screenshot_to_array(self, screen_id, bitmap_format):
        """Screenshot in requested format is retrieved to an array of bytes.

        in screen_id of type int
            Saved guest screen to read from.

        in bitmap_format of type :class:`BitmapFormat`
            The requested format.

        out width of type int
            Image width.

        out height of type int
            Image height.

        return data of type str
            Array with resulting image data.

        """
        if not isinstance(screen_id, baseinteger):
            raise TypeError("screen_id can only be an instance of type baseinteger")
        if not isinstance(bitmap_format, BitmapFormat):
            raise TypeError(
                "bitmap_format can only be an instance of type BitmapFormat"
            )
        (data, width, height) = self._call(
            "readSavedScreenshotToArray", in_p=[screen_id, bitmap_format]
        )
        return (data, width, height)

    def hot_plug_cpu(self, cpu):
        """Plugs a CPU into the machine.

        in cpu of type int
            The CPU id to insert.

        """
        if not isinstance(cpu, baseinteger):
            raise TypeError("cpu can only be an instance of type baseinteger")
        self._call("hotPlugCPU", in_p=[cpu])

    def hot_unplug_cpu(self, cpu):
        """Removes a CPU from the machine.

        in cpu of type int
            The CPU id to remove.

        """
        if not isinstance(cpu, baseinteger):
            raise TypeError("cpu can only be an instance of type baseinteger")
        self._call("hotUnplugCPU", in_p=[cpu])

    def get_cpu_status(self, cpu):
        """Returns the current status of the given CPU.

        in cpu of type int
            The CPU id to check for.

        return attached of type bool
            Status of the CPU.

        """
        if not isinstance(cpu, baseinteger):
            raise TypeError("cpu can only be an instance of type baseinteger")
        attached = self._call("getCPUStatus", in_p=[cpu])
        return attached

    def get_effective_paravirt_provider(self):
        """Returns the effective paravirtualization provider for this VM.

        return paravirt_provider of type :class:`ParavirtProvider`
            The effective paravirtualization provider for this VM.

        """
        paravirt_provider = self._call("getEffectiveParavirtProvider")
        paravirt_provider = ParavirtProvider(paravirt_provider)
        return paravirt_provider

    def query_log_filename(self, idx):
        """Queries for the VM log file name of an given index. Returns an empty
        string if a log file with that index doesn't exists.

        in idx of type int
            Which log file name to query. 0=current log file.

        return filename of type str
            On return the full path to the log file or an empty string on error.

        """
        if not isinstance(idx, baseinteger):
            raise TypeError("idx can only be an instance of type baseinteger")
        filename = self._call("queryLogFilename", in_p=[idx])
        return filename

    def read_log(self, idx, offset, size):
        """Reads the VM log file. The chunk size is limited, so even if you
        ask for a big piece there might be less data returned.

        in idx of type int
            Which log file to read. 0=current log file.

        in offset of type int
            Offset in the log file.

        in size of type int
            Chunk size to read in the log file.

        return data of type str
            Data read from the log file. A data size of 0 means end of file
            if the requested chunk size was not 0. This is the unprocessed
            file data, i.e. the line ending style depends on the platform of
            the system the server is running on.

        """
        if not isinstance(idx, baseinteger):
            raise TypeError("idx can only be an instance of type baseinteger")
        if not isinstance(offset, baseinteger):
            raise TypeError("offset can only be an instance of type baseinteger")
        if not isinstance(size, baseinteger):
            raise TypeError("size can only be an instance of type baseinteger")
        data = self._call("readLog", in_p=[idx, offset, size])
        return data

    def clone_to(self, target, mode, options):
        """Creates a clone of this machine, either as a full clone (which means
        creating independent copies of the hard disk media, save states and so
        on), or as a linked clone (which uses its own differencing media,
        sharing the parent media with the source machine).

        The target machine object must have been created previously with :py:func:`IVirtualBox.create_machine` , and all the settings will be
        transferred except the VM name and the hardware UUID. You can set the
        VM name and the new hardware UUID when creating the target machine. The
        network MAC addresses are newly created for all enabled network
        adapters. You can change that behaviour with the options parameter.
        The operation is performed asynchronously, so the machine object will
        be not be usable until the @a progress object signals completion.

        in target of type :class:`IMachine`
            Target machine object.

        in mode of type :class:`CloneMode`
            Which states should be cloned.

        in options of type :class:`CloneOptions`
            Options for the cloning operation.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`OleErrorInvalidarg`
            @a target is @c null.

        """
        if not isinstance(target, IMachine):
            raise TypeError("target can only be an instance of type IMachine")
        if not isinstance(mode, CloneMode):
            raise TypeError("mode can only be an instance of type CloneMode")
        if not isinstance(options, list):
            raise TypeError("options can only be an instance of type list")
        for a in options[:10]:
            if not isinstance(a, CloneOptions):
                raise TypeError("array can only contain objects of type CloneOptions")
        progress = self._call("cloneTo", in_p=[target, mode, options])
        progress = IProgress(progress)
        return progress

    def move_to(self, folder, type_p):
        """Move machine on to new place/folder

        in folder of type str
            Target folder where machine is moved.

        in type_p of type str
            Type of moving.
            Possible values:
            basic - Only the files which belong solely to this machine
            are moved from the original machine's folder to
            a new folder.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`OleErrorInvalidarg`
            @a target is @c null.

        """
        if not isinstance(folder, basestring):
            raise TypeError("folder can only be an instance of type basestring")
        if not isinstance(type_p, basestring):
            raise TypeError("type_p can only be an instance of type basestring")
        progress = self._call("moveTo", in_p=[folder, type_p])
        progress = IProgress(progress)
        return progress

    def save_state(self):
        """Saves the current execution state of a running virtual machine
        and stops its execution.

        After this operation completes, the machine will go to the
        Saved state. Next time it is powered up, this state will
        be restored and the machine will continue its execution from
        the place where it was saved.

        This operation differs from taking a snapshot to the effect
        that it doesn't create new differencing media. Also, once
        the machine is powered up from the state saved using this method,
        the saved state is deleted, so it will be impossible to return
        to this state later.


        On success, this method implicitly calls
        :py:func:`save_settings`  to save all current machine
        settings (including runtime changes to the DVD medium, etc.).
        Together with the impossibility to change any VM settings when it is
        in the Saved state, this guarantees adequate hardware
        configuration of the machine when it is restored from the saved
        state file.



        The machine must be in the Running or Paused state, otherwise
        the operation will fail.



        :py:func:`take_snapshot`

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine state neither Running nor Paused.

        raises :class:`VBoxErrorFileError`
            Failed to create directory for saved state file.

        """
        progress = self._call("saveState")
        progress = IProgress(progress)
        return progress

    def adopt_saved_state(self, saved_state_file):
        """Associates the given saved state file to the virtual machine.

        On success, the machine will go to the Saved state. Next time it is
        powered up, it will be restored from the adopted saved state and
        continue execution from the place where the saved state file was
        created.

        The specified saved state file path may be absolute or relative to the
        folder the VM normally saves the state to (usually,
        :py:func:`snapshot_folder` ).


        It's a caller's responsibility to make sure the given saved state
        file is compatible with the settings of this virtual machine that
        represent its virtual hardware (memory size, storage disk configuration
        etc.). If there is a mismatch, the behavior of the virtual machine
        is undefined.

        in saved_state_file of type str
            Path to the saved state file to adopt.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine state neither PoweredOff nor Aborted.

        """
        if not isinstance(saved_state_file, basestring):
            raise TypeError(
                "saved_state_file can only be an instance of type basestring"
            )
        self._call("adoptSavedState", in_p=[saved_state_file])

    def discard_saved_state(self, f_remove_file):
        """Forcibly resets the machine to "Powered Off" state if it is
        currently in the "Saved" state (previously created by :py:func:`save_state` ).
        Next time the machine is powered up, a clean boot will occur.

        This operation is equivalent to resetting or powering off
        the machine without doing a proper shutdown of the guest
        operating system; as with resetting a running phyiscal
        computer, it can can lead to data loss.

        If @a fRemoveFile is @c true, the file in the machine directory
        into which the machine state was saved is also deleted. If
        this is @c false, then the state can be recovered and later
        re-inserted into a machine using :py:func:`adopt_saved_state` .
        The location of the file can be found in the
        :py:func:`state_file_path`  attribute.

        in f_remove_file of type bool
            Whether to also remove the saved state file.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine not in state Saved.

        """
        if not isinstance(f_remove_file, bool):
            raise TypeError("f_remove_file can only be an instance of type bool")
        self._call("discardSavedState", in_p=[f_remove_file])

    def take_snapshot(self, name, description, pause):
        """Saves the current execution state
        and all settings of the machine and creates differencing images
        for all normal (non-independent) media.
        See :py:class:`ISnapshot`  for an introduction to snapshots.

        This method can be called for a PoweredOff, Saved (see
        :py:func:`save_state` ), Running or
        Paused virtual machine. When the machine is PoweredOff, an
        offline snapshot is created. When the machine is Running a live
        snapshot is created, and an online snapshot is created when Paused.

        The taken snapshot is always based on the
        :py:func:`current_snapshot` current snapshot
        of the associated virtual machine and becomes a new current snapshot.


        This method implicitly calls :py:func:`save_settings`  to
        save all current machine settings before taking an offline snapshot.

        in name of type str
            Short name for the snapshot.

        in description of type str
            Optional description of the snapshot.

        in pause of type bool
            Whether the VM should be paused while taking the snapshot. Only
            relevant when the VM is running, and distinguishes between online
            (@c true) and live (@c false) snapshots. When the VM is not running
            the result is always an offline snapshot.

        out id_p of type str
            UUID of the snapshot which will be created. Useful for follow-up
            operations after the snapshot has been created.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine currently changing state.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(description, basestring):
            raise TypeError("description can only be an instance of type basestring")
        if not isinstance(pause, bool):
            raise TypeError("pause can only be an instance of type bool")
        (progress, id_p) = self._call("takeSnapshot", in_p=[name, description, pause])
        progress = IProgress(progress)
        return (progress, id_p)

    def delete_snapshot(self, id_p):
        """Starts deleting the specified snapshot asynchronously.
                See :py:class:`ISnapshot`  for an introduction to snapshots.

                The execution state and settings of the associated machine stored in
                the snapshot will be deleted. The contents of all differencing media of
                this snapshot will be merged with the contents of their dependent child
                media to keep the medium chain valid (in other words, all changes
                represented by media being deleted will be propagated to their child
                medium). After that, this snapshot's differencing medium will be
                deleted. The parent of this snapshot will become a new parent for all
                its child snapshots.

                If the deleted snapshot is the current one, its parent snapshot will
                become a new current snapshot. The current machine state is not directly
                affected in this case, except that currently attached differencing
                media based on media of the deleted snapshot will be also merged as
                described above.

                If the deleted snapshot is the first or current snapshot, then the
                respective IMachine attributes will be adjusted. Deleting the current
                snapshot will also implicitly call :py:func:`save_settings`
                to make all current machine settings permanent.

                Deleting a snapshot has the following preconditions:


                Child media of all normal media of the deleted snapshot
                must be accessible (see :py:func:`IMedium.state` ) for this
                operation to succeed. If only one running VM refers to all images
                which participates in merging the operation can be performed while
                the VM is running. Otherwise all virtual machines whose media are
                directly or indirectly based on the media of deleted snapshot must
                be powered off. In any case, online snapshot deleting usually is
                slower than the same operation without any running VM.

                You cannot delete the snapshot if a medium attached to it has
                more than one child medium (differencing images) because otherwise
                merging would be impossible. This might be the case if there is
                more than one child snapshot or differencing images were created
                for other reason (e.g. implicitly because of multiple machine
                attachments).


                The virtual machine's :py:func:`state` state is
                changed to "DeletingSnapshot", "DeletingSnapshotOnline" or
                "DeletingSnapshotPaused" while this operation is in progress.


                Merging medium contents can be very time and disk space
                consuming, if these media are big in size and have many
                children. However, if the snapshot being deleted is the last
                (head) snapshot on the branch, the operation will be rather
                quick.

                in id_p of type str
                    UUID of the snapshot to delete.

                return progress of type :class:`IProgress`
                    Progress object to track the operation completion.

                raises :class:`VBoxErrorInvalidVmState`
                    The running virtual machine prevents deleting this snapshot. This
        happens only in very specific situations, usually snapshots can be
        deleted without trouble while a VM is running. The error message
        text explains the reason for the failure.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        progress = self._call("deleteSnapshot", in_p=[id_p])
        progress = IProgress(progress)
        return progress

    def delete_snapshot_and_all_children(self, id_p):
        """Starts deleting the specified snapshot and all its children
                asynchronously. See :py:class:`ISnapshot`  for an introduction to
                snapshots. The conditions and many details are the same as with
                :py:func:`delete_snapshot` .

                This operation is very fast if the snapshot subtree does not include
                the current state. It is still significantly faster than deleting the
                snapshots one by one if the current state is in the subtree and there
                are more than one snapshots from current state to the snapshot which
                marks the subtree, since it eliminates the incremental image merging.

                This API method is right now not implemented!

                in id_p of type str
                    UUID of the snapshot to delete, including all its children.

                return progress of type :class:`IProgress`
                    Progress object to track the operation completion.

                raises :class:`VBoxErrorInvalidVmState`
                    The running virtual machine prevents deleting this snapshot. This
        happens only in very specific situations, usually snapshots can be
        deleted without trouble while a VM is running. The error message
        text explains the reason for the failure.

                raises :class:`OleErrorNotimpl`
                    The method is not implemented yet.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        progress = self._call("deleteSnapshotAndAllChildren", in_p=[id_p])
        progress = IProgress(progress)
        return progress

    def delete_snapshot_range(self, start_id, end_id):
        """Starts deleting the specified snapshot range. This is limited to
                linear snapshot lists, which means there may not be any other child
                snapshots other than the direct sequence between the start and end
                snapshot. If the start and end snapshot point to the same snapshot this
                method is completely equivalent to :py:func:`delete_snapshot` . See
                :py:class:`ISnapshot`  for an introduction to snapshots. The
                conditions and many details are the same as with
                :py:func:`delete_snapshot` .

                This operation is generally faster than deleting snapshots one by one
                and often also needs less extra disk space before freeing up disk space
                by deleting the removed disk images corresponding to the snapshot.

                This API method is right now not implemented!

                in start_id of type str
                    UUID of the first snapshot to delete.

                in end_id of type str
                    UUID of the last snapshot to delete.

                return progress of type :class:`IProgress`
                    Progress object to track the operation completion.

                raises :class:`VBoxErrorInvalidVmState`
                    The running virtual machine prevents deleting this snapshot. This
        happens only in very specific situations, usually snapshots can be
        deleted without trouble while a VM is running. The error message
        text explains the reason for the failure.

                raises :class:`OleErrorNotimpl`
                    The method is not implemented yet.

        """
        if not isinstance(start_id, basestring):
            raise TypeError("start_id can only be an instance of type basestring")
        if not isinstance(end_id, basestring):
            raise TypeError("end_id can only be an instance of type basestring")
        progress = self._call("deleteSnapshotRange", in_p=[start_id, end_id])
        progress = IProgress(progress)
        return progress

    def restore_snapshot(self, snapshot):
        """Starts resetting the machine's current state to the state contained
        in the given snapshot, asynchronously. All current settings of the
        machine will be reset and changes stored in differencing media
        will be lost.
        See :py:class:`ISnapshot`  for an introduction to snapshots.

        After this operation is successfully completed, new empty differencing
        media are created for all normal media of the machine.

        If the given snapshot is an online snapshot, the machine will go to
        the :py:attr:`MachineState.saved` saved state, so that the
        next time it is powered on, the execution state will be restored
        from the state of the snapshot.


        The machine must not be running, otherwise the operation will fail.



        If the machine state is :py:attr:`MachineState.saved` Saved
        prior to this operation, the saved state file will be implicitly
        deleted (as if :py:func:`IMachine.discard_saved_state`  were
        called).

        in snapshot of type :class:`ISnapshot`
            The snapshot to restore the VM state from.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine is running.

        """
        if not isinstance(snapshot, ISnapshot):
            raise TypeError("snapshot can only be an instance of type ISnapshot")
        progress = self._call("restoreSnapshot", in_p=[snapshot])
        progress = IProgress(progress)
        return progress

    def apply_defaults(self, flags):
        """Applies the defaults for the configured guest OS type. This is
                primarily for getting sane settings straight after creating a
                new VM, but it can also be applied later.


                This is primarily a shortcut, centralizing the tedious job of
                getting the recommended settings and translating them into
                settings updates. The settings are made at the end of the call,
                but not saved.

                in flags of type str
                    Additional flags, to be defined later.

                raises :class:`OleErrorFail`
                    General error.

                raises :class:`VBoxErrorInvalidVmState`
                    The machine is in invalid state.

                raises :class:`VBoxErrorObjectInUse`
                    Some of the applied objects already exist. The method has been
        called to already configured machine.

        """
        if not isinstance(flags, basestring):
            raise TypeError("flags can only be an instance of type basestring")
        self._call("applyDefaults", in_p=[flags])


class IEmulatedUSB(Interface):
    """
    Manages emulated USB devices.
    """

    __uuid__ = "6e253ee8-477a-2497-6759-88b8292a5af0"
    __wsmap__ = "managed"

    def webcam_attach(self, path, settings):
        """Attaches the emulated USB webcam to the VM, which will use a host video capture device.

        in path of type str
            The host path of the capture device to use.

        in settings of type str
            Optional settings.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(settings, basestring):
            raise TypeError("settings can only be an instance of type basestring")
        self._call("webcamAttach", in_p=[path, settings])

    def webcam_detach(self, path):
        """Detaches the emulated USB webcam from the VM

        in path of type str
            The host path of the capture device to detach.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        self._call("webcamDetach", in_p=[path])

    @property
    def webcams(self):
        """Get str value for 'webcams'
        Lists attached virtual webcams.
        """
        ret = self._get_attr("webcams")
        return ret


class IVRDEServerInfo(Interface):
    """
    Contains information about the remote desktop (VRDE) server capabilities and status.
    This is used in the :py:func:`IConsole.vrde_server_info`  attribute.
    """

    __uuid__ = "c39ef4d6-7532-45e8-96da-eb5986ae76e4"
    __wsmap__ = "struct"

    @property
    def active(self):
        """Get bool value for 'active'
        Whether the remote desktop connection is active.
        """
        ret = self._get_attr("active")
        return ret

    @property
    def port(self):
        """Get int value for 'port'
        VRDE server port number. If this property is equal to 0, then
        the VRDE server failed to start, usually because there are no free IP
        ports to bind to. If this property is equal to -1, then the VRDE
        server has not yet been started.
        """
        ret = self._get_attr("port")
        return ret

    @property
    def number_of_clients(self):
        """Get int value for 'numberOfClients'
        How many times a client connected.
        """
        ret = self._get_attr("numberOfClients")
        return ret

    @property
    def begin_time(self):
        """Get int value for 'beginTime'
        When the last connection was established, in milliseconds since 1970-01-01 UTC.
        """
        ret = self._get_attr("beginTime")
        return ret

    @property
    def end_time(self):
        """Get int value for 'endTime'
        When the last connection was terminated or the current time, if
        connection is still active, in milliseconds since 1970-01-01 UTC.
        """
        ret = self._get_attr("endTime")
        return ret

    @property
    def bytes_sent(self):
        """Get int value for 'bytesSent'
        How many bytes were sent in last or current, if still active, connection.
        """
        ret = self._get_attr("bytesSent")
        return ret

    @property
    def bytes_sent_total(self):
        """Get int value for 'bytesSentTotal'
        How many bytes were sent in all connections.
        """
        ret = self._get_attr("bytesSentTotal")
        return ret

    @property
    def bytes_received(self):
        """Get int value for 'bytesReceived'
        How many bytes were received in last or current, if still active, connection.
        """
        ret = self._get_attr("bytesReceived")
        return ret

    @property
    def bytes_received_total(self):
        """Get int value for 'bytesReceivedTotal'
        How many bytes were received in all connections.
        """
        ret = self._get_attr("bytesReceivedTotal")
        return ret

    @property
    def user(self):
        """Get str value for 'user'
        Login user name supplied by the client.
        """
        ret = self._get_attr("user")
        return ret

    @property
    def domain(self):
        """Get str value for 'domain'
        Login domain name supplied by the client.
        """
        ret = self._get_attr("domain")
        return ret

    @property
    def client_name(self):
        """Get str value for 'clientName'
        The client name supplied by the client.
        """
        ret = self._get_attr("clientName")
        return ret

    @property
    def client_ip(self):
        """Get str value for 'clientIP'
        The IP address of the client.
        """
        ret = self._get_attr("clientIP")
        return ret

    @property
    def client_version(self):
        """Get int value for 'clientVersion'
        The client software version number.
        """
        ret = self._get_attr("clientVersion")
        return ret

    @property
    def encryption_style(self):
        """Get int value for 'encryptionStyle'
        Public key exchange method used when connection was established.
        Values: 0 - RDP4 public key exchange scheme.
        1 - X509 certificates were sent to client.
        """
        ret = self._get_attr("encryptionStyle")
        return ret


class IConsole(Interface):
    """
    The IConsole interface represents an interface to control virtual
    machine execution.

    A console object gets created when a machine has been locked for a
    particular session (client process) using :py:func:`IMachine.lock_machine`
    or :py:func:`IMachine.launch_vm_process` . The console object can
    then be found in the session's :py:func:`ISession.console`  attribute.

    Methods of the IConsole interface allow the caller to query the current
    virtual machine execution state, pause the machine or power it down, save
    the machine state or take a snapshot, attach and detach removable media
    and so on.

    :py:class:`ISession`
    """

    __uuid__ = "872da645-4a9b-1727-bee2-5585105b9eed"
    __wsmap__ = "managed"

    @property
    def machine(self):
        """Get IMachine value for 'machine'
        Machine object for this console session.

        This is a convenience property, it has the same value as
        :py:func:`ISession.machine`  of the corresponding session
        object.
        """
        ret = self._get_attr("machine")
        return IMachine(ret)

    @property
    def state(self):
        """Get MachineState value for 'state'
        Current execution state of the machine.

        This property always returns the same value as the corresponding
        property of the IMachine object for this console session.
        For the process that owns (executes) the VM, this is the
        preferable way of querying the VM state, because no IPC
        calls are made.
        """
        ret = self._get_attr("state")
        return MachineState(ret)

    @property
    def guest(self):
        """Get IGuest value for 'guest'
        Guest object.
        """
        ret = self._get_attr("guest")
        return IGuest(ret)

    @property
    def keyboard(self):
        """Get IKeyboard value for 'keyboard'
        Virtual keyboard object.

        If the machine is not running, any attempt to use
        the returned object will result in an error.
        """
        ret = self._get_attr("keyboard")
        return IKeyboard(ret)

    @property
    def mouse(self):
        """Get IMouse value for 'mouse'
        Virtual mouse object.

        If the machine is not running, any attempt to use
        the returned object will result in an error.
        """
        ret = self._get_attr("mouse")
        return IMouse(ret)

    @property
    def display(self):
        """Get IDisplay value for 'display'
        Virtual display object.

        If the machine is not running, any attempt to use
        the returned object will result in an error.
        """
        ret = self._get_attr("display")
        return IDisplay(ret)

    @property
    def debugger(self):
        """Get IMachineDebugger value for 'debugger'
        Debugging interface.
        """
        ret = self._get_attr("debugger")
        return IMachineDebugger(ret)

    @property
    def usb_devices(self):
        """Get IUSBDevice value for 'USBDevices'
        Collection of USB devices currently attached to the virtual
        USB controller.

        The collection is empty if the machine is not running.
        """
        ret = self._get_attr("USBDevices")
        return [IUSBDevice(a) for a in ret]

    @property
    def remote_usb_devices(self):
        """Get IHostUSBDevice value for 'remoteUSBDevices'
        List of USB devices currently attached to the remote VRDE client.
        Once a new device is physically attached to the remote host computer,
        it appears in this list and remains there until detached.
        """
        ret = self._get_attr("remoteUSBDevices")
        return [IHostUSBDevice(a) for a in ret]

    @property
    def shared_folders(self):
        """Get ISharedFolder value for 'sharedFolders'
        Collection of shared folders for the current session. These folders
        are called transient shared folders because they are available to the
        guest OS running inside the associated virtual machine only for the
        duration of the session (as opposed to
        :py:func:`IMachine.shared_folders`  which represent permanent shared
        folders). When the session is closed (e.g. the machine is powered down),
        these folders are automatically discarded.

        New shared folders are added to the collection using
        :py:func:`create_shared_folder` . Existing shared folders can be
        removed using :py:func:`remove_shared_folder` .
        """
        ret = self._get_attr("sharedFolders")
        return [ISharedFolder(a) for a in ret]

    @property
    def vrde_server_info(self):
        """Get IVRDEServerInfo value for 'VRDEServerInfo'
        Interface that provides information on Remote Desktop Extension (VRDE) connection.
        """
        ret = self._get_attr("VRDEServerInfo")
        return IVRDEServerInfo(ret)

    @property
    def event_source(self):
        """Get IEventSource value for 'eventSource'
        Event source for console events.
        """
        ret = self._get_attr("eventSource")
        return IEventSource(ret)

    @property
    def attached_pci_devices(self):
        """Get IPCIDeviceAttachment value for 'attachedPCIDevices'
        Array of PCI devices attached to this machine.
        """
        ret = self._get_attr("attachedPCIDevices")
        return [IPCIDeviceAttachment(a) for a in ret]

    @property
    def use_host_clipboard(self):
        """Get or set bool value for 'useHostClipboard'
        Whether the guest clipboard should be connected to the host one or
        whether it should only be allowed access to the VRDE clipboard. This
        setting may not affect existing guest clipboard connections which
        are already connected to the host clipboard.
        """
        ret = self._get_attr("useHostClipboard")
        return ret

    @use_host_clipboard.setter
    def use_host_clipboard(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("useHostClipboard", value)

    @property
    def emulated_usb(self):
        """Get IEmulatedUSB value for 'emulatedUSB'
        Interface that manages emulated USB devices.
        """
        ret = self._get_attr("emulatedUSB")
        return IEmulatedUSB(ret)

    def power_up(self):
        """Starts the virtual machine execution using the current machine
        state (that is, its current execution state, current settings and
        current storage devices).


        This method is only useful for front-ends that want to actually
        execute virtual machines in their own process (like the VirtualBox
        or VBoxSDL front-ends). Unless you are intending to write such a
        front-end, do not call this method. If you simply want to
        start virtual machine execution using one of the existing front-ends
        (for example the VirtualBox GUI or headless server), use
        :py:func:`IMachine.launch_vm_process`  instead; these
        front-ends will power up the machine automatically for you.


        If the machine is powered off or aborted, the execution will
        start from the beginning (as if the real hardware were just
        powered on).

        If the machine is in the :py:attr:`MachineState.saved`  state,
        it will continue its execution the point where the state has
        been saved.

        If the machine :py:func:`IMachine.teleporter_enabled`  property is
        enabled on the machine being powered up, the machine will wait for an
        incoming teleportation in the :py:attr:`MachineState.teleporting_in`
        state. The returned progress object will have at least three
        operations where the last three are defined as: (1) powering up and
        starting TCP server, (2) waiting for incoming teleportations, and
        (3) perform teleportation. These operations will be reflected as the
        last three operations of the progress objected returned by
        :py:func:`IMachine.launch_vm_process`  as well.

        :py:func:`IMachine.save_state`

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine already running.

        raises :class:`VBoxErrorHostError`
            Host interface does not exist or name not set.

        raises :class:`VBoxErrorFileError`
            Invalid saved state file.

        """
        progress = self._call("powerUp")
        progress = IProgress(progress)
        return progress

    def power_up_paused(self):
        """Identical to powerUp except that the VM will enter the
        :py:attr:`MachineState.paused`  state, instead of
        :py:attr:`MachineState.running` .

        :py:func:`power_up`

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine already running.

        raises :class:`VBoxErrorHostError`
            Host interface does not exist or name not set.

        raises :class:`VBoxErrorFileError`
            Invalid saved state file.

        """
        progress = self._call("powerUpPaused")
        progress = IProgress(progress)
        return progress

    def power_down(self):
        """Initiates the power down procedure to stop the virtual machine
        execution.

        The completion of the power down procedure is tracked using the returned
        IProgress object. After the operation is complete, the machine will go
        to the PoweredOff state.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine must be Running, Paused or Stuck to be powered down.

        """
        progress = self._call("powerDown")
        progress = IProgress(progress)
        return progress

    def reset(self):
        """Resets the virtual machine.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine not in Running state.

        raises :class:`VBoxErrorVmError`
            Virtual machine error in reset operation.

        """
        self._call("reset")

    def pause(self):
        """Pauses the virtual machine execution.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine not in Running state.

        raises :class:`VBoxErrorVmError`
            Virtual machine error in suspend operation.

        """
        self._call("pause")

    def resume(self):
        """Resumes the virtual machine execution.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine not in Paused state.

        raises :class:`VBoxErrorVmError`
            Virtual machine error in resume operation.

        """
        self._call("resume")

    def power_button(self):
        """Sends the ACPI power button event to the guest.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine not in Running state.

        raises :class:`VBoxErrorPdmError`
            Controlled power off failed.

        """
        self._call("powerButton")

    def sleep_button(self):
        """Sends the ACPI sleep button event to the guest.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine not in Running state.

        raises :class:`VBoxErrorPdmError`
            Sending sleep button event failed.

        """
        self._call("sleepButton")

    def get_power_button_handled(self):
        """Checks if the last power button event was handled by guest.

        return handled of type bool

        raises :class:`VBoxErrorPdmError`
            Checking if the event was handled by the guest OS failed.

        """
        handled = self._call("getPowerButtonHandled")
        return handled

    def get_guest_entered_acpi_mode(self):
        """Checks if the guest entered the ACPI mode G0 (working) or
        G1 (sleeping). If this method returns @c false, the guest will
        most likely not respond to external ACPI events.

        return entered of type bool

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine not in Running state.

        """
        entered = self._call("getGuestEnteredACPIMode")
        return entered

    def get_device_activity(self, type_p):
        """Gets the current activity type of given devices or device groups.

        in type_p of type :class:`DeviceType`

        return activity of type :class:`DeviceActivity`

        raises :class:`OleErrorInvalidarg`
            Invalid device type.

        """
        if not isinstance(type_p, list):
            raise TypeError("type_p can only be an instance of type list")
        for a in type_p[:10]:
            if not isinstance(a, DeviceType):
                raise TypeError("array can only contain objects of type DeviceType")
        activity = self._call("getDeviceActivity", in_p=[type_p])
        activity = [DeviceActivity(a) for a in activity]
        return activity

    def attach_usb_device(self, id_p, capture_filename):
        """Attaches a host USB device with the given UUID to the
        USB controller of the virtual machine.

        The device needs to be in one of the following states:
        :py:attr:`USBDeviceState.busy` ,
        :py:attr:`USBDeviceState.available`  or
        :py:attr:`USBDeviceState.held` ,
        otherwise an error is immediately returned.

        When the device state is
        :py:attr:`USBDeviceState.busy` Busy, an error may also
        be returned if the host computer refuses to release it for some reason.

        :py:func:`IUSBDeviceFilters.device_filters` ,
        :py:class:`USBDeviceState`

        in id_p of type str
            UUID of the host USB device to attach.

        in capture_filename of type str
            Filename to capture the USB traffic to.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine state neither Running nor Paused.

        raises :class:`VBoxErrorPdmError`
            Virtual machine does not have a USB controller.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        if not isinstance(capture_filename, basestring):
            raise TypeError(
                "capture_filename can only be an instance of type basestring"
            )
        self._call("attachUSBDevice", in_p=[id_p, capture_filename])

    def detach_usb_device(self, id_p):
        """Detaches an USB device with the given UUID from the USB controller
        of the virtual machine.

        After this method succeeds, the VirtualBox server re-initiates
        all USB filters as if the device were just physically attached
        to the host, but filters of this machine are ignored to avoid
        a possible automatic re-attachment.

        :py:func:`IUSBDeviceFilters.device_filters` ,
        :py:class:`USBDeviceState`

        in id_p of type str
            UUID of the USB device to detach.

        return device of type :class:`IUSBDevice`
            Detached USB device.

        raises :class:`VBoxErrorPdmError`
            Virtual machine does not have a USB controller.

        raises :class:`OleErrorInvalidarg`
            USB device not attached to this virtual machine.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        device = self._call("detachUSBDevice", in_p=[id_p])
        device = IUSBDevice(device)
        return device

    def find_usb_device_by_address(self, name):
        """Searches for a USB device with the given host address.



        :py:func:`IUSBDevice.address`

        in name of type str
            Address of the USB device (as assigned by the host) to
            search for.

        return device of type :class:`IUSBDevice`
            Found USB device object.

        raises :class:`VBoxErrorObjectNotFound`
            Given @c name does not correspond to any USB device.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        device = self._call("findUSBDeviceByAddress", in_p=[name])
        device = IUSBDevice(device)
        return device

    def find_usb_device_by_id(self, id_p):
        """Searches for a USB device with the given UUID.



        :py:func:`IUSBDevice.id_p`

        in id_p of type str
            UUID of the USB device to search for.

        return device of type :class:`IUSBDevice`
            Found USB device object.

        raises :class:`VBoxErrorObjectNotFound`
            Given @c id does not correspond to any USB device.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        device = self._call("findUSBDeviceById", in_p=[id_p])
        device = IUSBDevice(device)
        return device

    def create_shared_folder(
        self, name, host_path, writable, automount, auto_mount_point
    ):
        """Creates a transient new shared folder by associating the given logical
        name with the given host path, adds it to the collection of shared
        folders and starts sharing it. Refer to the description of
        :py:class:`ISharedFolder`  to read more about logical names.

        in name of type str
            Unique logical name of the shared folder.

        in host_path of type str
            Full path to the shared folder in the host file system.

        in writable of type bool
            Whether the share is writable or readonly

        in automount of type bool
            Whether the share gets automatically mounted by the guest
            or not.

        in auto_mount_point of type str
            Where the guest should automatically mount the folder, if possible.
            For Windows and OS/2 guests this should be a drive letter, while other
            guests it should be a absolute directory.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine in Saved state or currently changing state.

        raises :class:`VBoxErrorFileError`
            Shared folder already exists or not accessible.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(host_path, basestring):
            raise TypeError("host_path can only be an instance of type basestring")
        if not isinstance(writable, bool):
            raise TypeError("writable can only be an instance of type bool")
        if not isinstance(automount, bool):
            raise TypeError("automount can only be an instance of type bool")
        if not isinstance(auto_mount_point, basestring):
            raise TypeError(
                "auto_mount_point can only be an instance of type basestring"
            )
        self._call(
            "createSharedFolder",
            in_p=[name, host_path, writable, automount, auto_mount_point],
        )

    def remove_shared_folder(self, name):
        """Removes a transient shared folder with the given name previously
        created by :py:func:`create_shared_folder`  from the collection of
        shared folders and stops sharing it.

        in name of type str
            Logical name of the shared folder to remove.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine in Saved state or currently changing state.

        raises :class:`VBoxErrorFileError`
            Shared folder does not exists.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        self._call("removeSharedFolder", in_p=[name])

    def teleport(self, hostname, tcpport, password, max_downtime):
        """Teleport the VM to a different host machine or process.

        @todo Explain the details.

        in hostname of type str
            The name or IP of the host to teleport to.

        in tcpport of type int
            The TCP port to connect to (1..65535).

        in password of type str
            The password.

        in max_downtime of type int
            The maximum allowed downtime given as milliseconds. 0 is not a valid
            value. Recommended value: 250 ms.

            The higher the value is, the greater the chance for a successful
            teleportation. A small value may easily result in the teleportation
            process taking hours and eventually fail.


            The current implementation treats this a guideline, not as an
            absolute rule.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorInvalidVmState`
            Virtual machine not running or paused.

        """
        if not isinstance(hostname, basestring):
            raise TypeError("hostname can only be an instance of type basestring")
        if not isinstance(tcpport, baseinteger):
            raise TypeError("tcpport can only be an instance of type baseinteger")
        if not isinstance(password, basestring):
            raise TypeError("password can only be an instance of type basestring")
        if not isinstance(max_downtime, baseinteger):
            raise TypeError("max_downtime can only be an instance of type baseinteger")
        progress = self._call(
            "teleport", in_p=[hostname, tcpport, password, max_downtime]
        )
        progress = IProgress(progress)
        return progress

    def add_disk_encryption_password(self, id_p, password, clear_on_suspend):
        """Adds a password used for hard disk encryption/decryption.

                in id_p of type str
                    The identifier used for the password. Must match the identifier
                    used when the encrypted medium was created.

                in password of type str
                    The password.

                in clear_on_suspend of type bool
                    Flag whether to clear the password on VM suspend (due to a suspending host
                    for example). The password must be supplied again before the VM can resume.

                raises :class:`VBoxErrorPasswordIncorrect`
                    The password provided wasn't correct for at least one disk using the provided
        ID.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        if not isinstance(password, basestring):
            raise TypeError("password can only be an instance of type basestring")
        if not isinstance(clear_on_suspend, bool):
            raise TypeError("clear_on_suspend can only be an instance of type bool")
        self._call("addDiskEncryptionPassword", in_p=[id_p, password, clear_on_suspend])

    def add_disk_encryption_passwords(self, ids, passwords, clear_on_suspend):
        """Adds a password used for hard disk encryption/decryption.

                in ids of type str
                    List of identifiers for the passwords. Must match the identifier
                    used when the encrypted medium was created.

                in passwords of type str
                    List of passwords.

                in clear_on_suspend of type bool
                    Flag whether to clear the given passwords on VM suspend (due to a suspending host
                    for example). The passwords must be supplied again before the VM can resume.

                raises :class:`VBoxErrorPasswordIncorrect`
                    The password provided wasn't correct for at least one disk using the provided
        ID.

        """
        if not isinstance(ids, list):
            raise TypeError("ids can only be an instance of type list")
        for a in ids[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(passwords, list):
            raise TypeError("passwords can only be an instance of type list")
        for a in passwords[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(clear_on_suspend, bool):
            raise TypeError("clear_on_suspend can only be an instance of type bool")
        self._call(
            "addDiskEncryptionPasswords", in_p=[ids, passwords, clear_on_suspend]
        )

    def remove_disk_encryption_password(self, id_p):
        """Removes a password used for hard disk encryption/decryption from
        the running VM. As soon as the medium requiring this password
        is accessed the VM is paused with an error and the password must be
        provided again.

        in id_p of type str
            The identifier used for the password. Must match the identifier
            used when the encrypted medium was created.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        self._call("removeDiskEncryptionPassword", in_p=[id_p])

    def clear_all_disk_encryption_passwords(self):
        """Clears all provided supplied disk encryption passwords."""
        self._call("clearAllDiskEncryptionPasswords")


class IHostNetworkInterface(Interface):
    """
    Represents one of host's network interfaces. IP V6 address and network
    mask are strings of 32 hexadecimal digits grouped by four. Groups are
    separated by colons.
    For example, fe80:0000:0000:0000:021e:c2ff:fed2:b030.
    """

    __uuid__ = "455f8c45-44a0-a470-ba20-27890b96dba9"
    __wsmap__ = "managed"

    @property
    def name(self):
        """Get str value for 'name'
        Returns the host network interface name.
        """
        ret = self._get_attr("name")
        return ret

    @property
    def short_name(self):
        """Get str value for 'shortName'
        Returns the host network interface short name.
        """
        ret = self._get_attr("shortName")
        return ret

    @property
    def id_p(self):
        """Get str value for 'id'
        Returns the interface UUID.
        """
        ret = self._get_attr("id")
        return ret

    @property
    def network_name(self):
        """Get str value for 'networkName'
        Returns the name of a virtual network the interface gets attached to.
        """
        ret = self._get_attr("networkName")
        return ret

    @property
    def dhcp_enabled(self):
        """Get bool value for 'DHCPEnabled'
        Specifies whether the DHCP is enabled for the interface.
        """
        ret = self._get_attr("DHCPEnabled")
        return ret

    @property
    def ip_address(self):
        """Get str value for 'IPAddress'
        Returns the IP V4 address of the interface.
        """
        ret = self._get_attr("IPAddress")
        return ret

    @property
    def network_mask(self):
        """Get str value for 'networkMask'
        Returns the network mask of the interface.
        """
        ret = self._get_attr("networkMask")
        return ret

    @property
    def ipv6_supported(self):
        """Get bool value for 'IPV6Supported'
        Specifies whether the IP V6 is supported/enabled for the interface.
        """
        ret = self._get_attr("IPV6Supported")
        return ret

    @property
    def ipv6_address(self):
        """Get str value for 'IPV6Address'
        Returns the IP V6 address of the interface.
        """
        ret = self._get_attr("IPV6Address")
        return ret

    @property
    def ipv6_network_mask_prefix_length(self):
        """Get int value for 'IPV6NetworkMaskPrefixLength'
        Returns the length IP V6 network mask prefix of the interface.
        """
        ret = self._get_attr("IPV6NetworkMaskPrefixLength")
        return ret

    @property
    def hardware_address(self):
        """Get str value for 'hardwareAddress'
        Returns the hardware address. For Ethernet it is MAC address.
        """
        ret = self._get_attr("hardwareAddress")
        return ret

    @property
    def medium_type(self):
        """Get HostNetworkInterfaceMediumType value for 'mediumType'
        Type of protocol encapsulation used.
        """
        ret = self._get_attr("mediumType")
        return HostNetworkInterfaceMediumType(ret)

    @property
    def status(self):
        """Get HostNetworkInterfaceStatus value for 'status'
        Status of the interface.
        """
        ret = self._get_attr("status")
        return HostNetworkInterfaceStatus(ret)

    @property
    def interface_type(self):
        """Get HostNetworkInterfaceType value for 'interfaceType'
        specifies the host interface type.
        """
        ret = self._get_attr("interfaceType")
        return HostNetworkInterfaceType(ret)

    @property
    def wireless(self):
        """Get bool value for 'wireless'
        Specifies whether the interface is wireless.
        """
        ret = self._get_attr("wireless")
        return ret

    def enable_static_ip_config(self, ip_address, network_mask):
        """sets and enables the static IP V4 configuration for the given interface.

        in ip_address of type str
            IP address.

        in network_mask of type str
            network mask.

        """
        if not isinstance(ip_address, basestring):
            raise TypeError("ip_address can only be an instance of type basestring")
        if not isinstance(network_mask, basestring):
            raise TypeError("network_mask can only be an instance of type basestring")
        self._call("enableStaticIPConfig", in_p=[ip_address, network_mask])

    def enable_static_ip_config_v6(self, ipv6_address, ipv6_network_mask_prefix_length):
        """sets and enables the static IP V6 configuration for the given interface.

        in ipv6_address of type str
            IP address.

        in ipv6_network_mask_prefix_length of type int
            network mask.

        """
        if not isinstance(ipv6_address, basestring):
            raise TypeError("ipv6_address can only be an instance of type basestring")
        if not isinstance(ipv6_network_mask_prefix_length, baseinteger):
            raise TypeError(
                "ipv6_network_mask_prefix_length can only be an instance of type baseinteger"
            )
        self._call(
            "enableStaticIPConfigV6",
            in_p=[ipv6_address, ipv6_network_mask_prefix_length],
        )

    def enable_dynamic_ip_config(self):
        """enables the dynamic IP configuration."""
        self._call("enableDynamicIPConfig")

    def dhcp_rediscover(self):
        """refreshes the IP configuration for DHCP-enabled interface."""
        self._call("DHCPRediscover")


class IHostVideoInputDevice(Interface):
    """
    Represents one of host's video capture devices, for example a webcam.
    """

    __uuid__ = "e8c25d4d-ac97-4c16-b3e2-81bd8a57cc27"
    __wsmap__ = "managed"

    @property
    def name(self):
        """Get str value for 'name'
        User friendly name.
        """
        ret = self._get_attr("name")
        return ret

    @property
    def path(self):
        """Get str value for 'path'
        The host path of the device.
        """
        ret = self._get_attr("path")
        return ret

    @property
    def alias(self):
        """Get str value for 'alias'
        An alias which can be used for :py:func:`IEmulatedUSB.webcam_attach`
        """
        ret = self._get_attr("alias")
        return ret


class IHostUpdate(Interface):
    """
    Represents the state of the update checking logic (a singleton returned
    by :py:func:`IHost.update`  attribute).
    """

    __uuid__ = "6fa2671b-0547-448e-bc7c-94e9e173bf57"
    __wsmap__ = "managed"

    def update_check(self, check_type):
        """Check for a newer version of software based on the 'checkType' value.

        in check_type of type :class:`UpdateCheckType`
            The type of software to check.

        return progress of type :class:`IProgress`
            Progress object to track operation completion.

        """
        if not isinstance(check_type, UpdateCheckType):
            raise TypeError(
                "check_type can only be an instance of type UpdateCheckType"
            )
        progress = self._call("updateCheck", in_p=[check_type])
        progress = IProgress(progress)
        return progress

    @property
    def update_response(self):
        """Get bool value for 'updateResponse'
        The response from the :py:func:`IHostUpdate.update_check`  method.
        """
        ret = self._get_attr("updateResponse")
        return ret

    @property
    def update_version(self):
        """Get str value for 'updateVersion'
        The newer version of the software returned by calling the update check
        :py:func:`IHostUpdate.update_check`  method.
        """
        ret = self._get_attr("updateVersion")
        return ret

    @property
    def update_url(self):
        """Get str value for 'updateURL'
        The download URL of the newer software version returned by calling the
        update check :py:func:`IHostUpdate.update_check`  method.
        """
        ret = self._get_attr("updateURL")
        return ret

    @property
    def update_check_needed(self):
        """Get bool value for 'updateCheckNeeded'
        Is it time to check for a newer version of software?
        """
        ret = self._get_attr("updateCheckNeeded")
        return ret


class IHostDrivePartition(Interface):
    """
    The IHostDrivePartition interface represents the partition of the host drive.
    To enumerate all available drives partitions in the host, use the
    :py:func:`IHost.host_drives`  attribute.
    """

    __uuid__ = "4f529a14-ace3-407c-9c49-066e8e8027f0"
    __wsmap__ = "struct"

    @property
    def number(self):
        """Get int value for 'number'
        The number of the partition. Represents the system number of the
        partition, e.g. /dev/sdX in the linux, where X is the number
        returned.
        """
        ret = self._get_attr("number")
        return ret

    @property
    def size(self):
        """Get int value for 'size'
        The partition size in bytes.
        """
        ret = self._get_attr("size")
        return ret

    @property
    def start(self):
        """Get int value for 'start'
        The start byte offset of this partition in bytes relative to the
        beginning of the hard disk.
        """
        ret = self._get_attr("start")
        return ret

    @property
    def type_p(self):
        """Get PartitionType value for 'type'
        A translation of :py:func:`IHostDrivePartition.type_mbr`  and
        :py:func:`IHostDrivePartition.type_uuid`  when possible, otherwise
        set to :py:attr:`PartitionType.unknown` .
        """
        ret = self._get_attr("type")
        return PartitionType(ret)

    @property
    def active(self):
        """Get bool value for 'active'
        The partition is bootable when TRUE.
        """
        ret = self._get_attr("active")
        return ret

    @property
    def type_mbr(self):
        """Get int value for 'typeMBR'
        The raw MBR partition type, 0 for non-MBR disks.
        """
        ret = self._get_attr("typeMBR")
        return ret

    @property
    def start_cylinder(self):
        """Get int value for 'startCylinder'
        The cylinder (0..1023) of the first sector in the partition on an MBR disk, zero for not an MBR disk.
        """
        ret = self._get_attr("startCylinder")
        return ret

    @property
    def start_head(self):
        """Get int value for 'startHead'
        The head (0..255) of the first sector in the partition on an MBR disk, zero for not an MBR disk.
        """
        ret = self._get_attr("startHead")
        return ret

    @property
    def start_sector(self):
        """Get int value for 'startSector'
        The sector (0..63) of the first sector in the partition on an MBR disk, zero for not an MBR disk.
        """
        ret = self._get_attr("startSector")
        return ret

    @property
    def end_cylinder(self):
        """Get int value for 'endCylinder'
        The cylinder (0..1023) of the last sector (inclusive) in the partition on an MBR disk, zero for not an MBR disk.
        """
        ret = self._get_attr("endCylinder")
        return ret

    @property
    def end_head(self):
        """Get int value for 'endHead'
        The head (0..255) of the last sector (inclusive) in the partition on an MBR disk, zero for not an MBR disk.
        """
        ret = self._get_attr("endHead")
        return ret

    @property
    def end_sector(self):
        """Get int value for 'endSector'
        The sector (1..63) of the last sector (inclusive) in the partition on an MBR disk, zero for not an MBR disk.
        """
        ret = self._get_attr("endSector")
        return ret

    @property
    def type_uuid(self):
        """Get str value for 'typeUuid'
        The partition type when GUID partitioning scheme is used, NULL UUID value for not a GPT disks.
        """
        ret = self._get_attr("typeUuid")
        return ret

    @property
    def uuid(self):
        """Get str value for 'uuid'
        The GUID of the partition when GUID partitioning scheme is used, NULL UUID value for not a GPT disks.
        """
        ret = self._get_attr("uuid")
        return ret

    @property
    def name(self):
        """Get str value for 'name'
        The name of the partition if GPT partitioning is used, empty if not a GPT disk.
        """
        ret = self._get_attr("name")
        return ret


class IHostDrive(Interface):
    """
    The IHostDrive interface represents the drive of the physical machine.
    It is not a complete medium description and, therefore, it is not IMedium
    based. The interface is used to get information about a host drive and
    its partitioning.

    The object operates in limited mode if the user cannot open the drive
    and parse the partition table.  In limited mode on the
    :py:func:`IHostDrive.drive_path`  and :py:func:`IHostDrive.model`
    attributes can be accessed, the rest will fail with E_ACCESSDENIED.
    """

    __uuid__ = "70e2e0c3-332c-4d72-b822-2db16e2cb31b"
    __wsmap__ = "managed"

    @property
    def drive_path(self):
        """Get str value for 'drivePath'
        The path of the drive. Platform dependent.
        """
        ret = self._get_attr("drivePath")
        return ret

    @property
    def partitioning_type(self):
        """Get PartitioningType value for 'partitioningType'
        The scheme of the partitions the disk has.
        """
        ret = self._get_attr("partitioningType")
        return PartitioningType(ret)

    @property
    def uuid(self):
        """Get str value for 'uuid'
        The GUID of the disk.
        """
        ret = self._get_attr("uuid")
        return ret

    @property
    def sector_size(self):
        """Get int value for 'sectorSize'
        The size of the sector in bytes.
        """
        ret = self._get_attr("sectorSize")
        return ret

    @property
    def size(self):
        """Get int value for 'size'
        The size of the disk in bytes.
        """
        ret = self._get_attr("size")
        return ret

    @property
    def model(self):
        """Get str value for 'model'
        The model string of the drive if available.
        """
        ret = self._get_attr("model")
        return ret

    @property
    def partitions(self):
        """Get IHostDrivePartition value for 'partitions'
        List of partitions available on the host drive.
        """
        ret = self._get_attr("partitions")
        return [IHostDrivePartition(a) for a in ret]


class IHost(Interface):
    """
    The IHost interface represents the physical machine that this VirtualBox
    installation runs on.

    An object implementing this interface is returned by the
    :py:func:`IVirtualBox.host`  attribute. This interface contains
    read-only information about the host's physical hardware (such as what
    processors and disks are available, what the host operating system is,
    and so on) and also allows for manipulating some of the host's hardware,
    such as global USB device filters and host interface networking.
    """

    __uuid__ = "fc0759a6-a5e2-41e1-93ca-64776335eb2d"
    __wsmap__ = "managed"

    @property
    def dvd_drives(self):
        """Get IMedium value for 'DVDDrives'
        List of DVD drives available on the host.
        """
        ret = self._get_attr("DVDDrives")
        return [IMedium(a) for a in ret]

    @property
    def floppy_drives(self):
        """Get IMedium value for 'floppyDrives'
        List of floppy drives available on the host.
        """
        ret = self._get_attr("floppyDrives")
        return [IMedium(a) for a in ret]

    @property
    def usb_devices(self):
        """Get IHostUSBDevice value for 'USBDevices'
        List of USB devices currently attached to the host.
        Once a new device is physically attached to the host computer,
        it appears in this list and remains there until detached.


        If USB functionality is not available in the given edition of
        VirtualBox, this method will set the result code to @c E_NOTIMPL.
        """
        ret = self._get_attr("USBDevices")
        return [IHostUSBDevice(a) for a in ret]

    @property
    def usb_device_filters(self):
        """Get IHostUSBDeviceFilter value for 'USBDeviceFilters'
        List of USB device filters in action.
        When a new device is physically attached to the host computer,
        filters from this list are applied to it (in order they are stored
        in the list). The first matched filter will determine the
        :py:func:`IHostUSBDeviceFilter.action` action
        performed on the device.

        Unless the device is ignored by these filters, filters of all
        currently running virtual machines
        (:py:func:`IUSBDeviceFilters.device_filters` ) are applied to it.


        If USB functionality is not available in the given edition of
        VirtualBox, this method will set the result code to @c E_NOTIMPL.


        :py:class:`IHostUSBDeviceFilter` ,
        :py:class:`USBDeviceState`
        """
        ret = self._get_attr("USBDeviceFilters")
        return [IHostUSBDeviceFilter(a) for a in ret]

    @property
    def network_interfaces(self):
        """Get IHostNetworkInterface value for 'networkInterfaces'
        List of host network interfaces currently defined on the host.
        """
        ret = self._get_attr("networkInterfaces")
        return [IHostNetworkInterface(a) for a in ret]

    @property
    def name_servers(self):
        """Get str value for 'nameServers'
        The list of nameservers registered in host's name resolving system.
        """
        ret = self._get_attr("nameServers")
        return ret

    @property
    def domain_name(self):
        """Get str value for 'domainName'
        Domain name used for name resolving.
        """
        ret = self._get_attr("domainName")
        return ret

    @property
    def search_strings(self):
        """Get str value for 'searchStrings'
        Search string registered for name resolving.
        """
        ret = self._get_attr("searchStrings")
        return ret

    @property
    def processor_count(self):
        """Get int value for 'processorCount'
        Number of (logical) CPUs installed in the host system.
        """
        ret = self._get_attr("processorCount")
        return ret

    @property
    def processor_online_count(self):
        """Get int value for 'processorOnlineCount'
        Number of (logical) CPUs online in the host system.
        """
        ret = self._get_attr("processorOnlineCount")
        return ret

    @property
    def processor_core_count(self):
        """Get int value for 'processorCoreCount'
        Number of physical processor cores installed in the host system.
        """
        ret = self._get_attr("processorCoreCount")
        return ret

    @property
    def processor_online_core_count(self):
        """Get int value for 'processorOnlineCoreCount'
        Number of physical processor cores online in the host system.
        """
        ret = self._get_attr("processorOnlineCoreCount")
        return ret

    @property
    def host_drives(self):
        """Get IHostDrive value for 'hostDrives'
        List of the host drive available to use in the VirtualBox.
        """
        ret = self._get_attr("hostDrives")
        return [IHostDrive(a) for a in ret]

    def get_processor_speed(self, cpu_id):
        """Query the (approximate) maximum speed of a specified host CPU in
        Megahertz.

        in cpu_id of type int
            Identifier of the CPU.

        return speed of type int
            Speed value. 0 is returned if value is not known or @a cpuId is
            invalid.

        """
        if not isinstance(cpu_id, baseinteger):
            raise TypeError("cpu_id can only be an instance of type baseinteger")
        speed = self._call("getProcessorSpeed", in_p=[cpu_id])
        return speed

    def get_processor_feature(self, feature):
        """Query whether a CPU feature is supported or not.

        in feature of type :class:`ProcessorFeature`
            CPU Feature identifier.

        return supported of type bool
            Feature is supported or not.

        """
        if not isinstance(feature, ProcessorFeature):
            raise TypeError("feature can only be an instance of type ProcessorFeature")
        supported = self._call("getProcessorFeature", in_p=[feature])
        return supported

    def get_processor_description(self, cpu_id):
        """Query the model string of a specified host CPU.

        in cpu_id of type int
            Identifier of the CPU.

            The current implementation might not necessarily return the
            description for this exact CPU.

        return description of type str
            Model string. An empty string is returned if value is not known or
            @a cpuId is invalid.

        """
        if not isinstance(cpu_id, baseinteger):
            raise TypeError("cpu_id can only be an instance of type baseinteger")
        description = self._call("getProcessorDescription", in_p=[cpu_id])
        return description

    def get_processor_cpuid_leaf(self, cpu_id, leaf, sub_leaf):
        """Returns the CPU cpuid information for the specified leaf.

        in cpu_id of type int
            Identifier of the CPU. The CPU most be online.

            The current implementation might not necessarily return the
            description for this exact CPU.

        in leaf of type int
            CPUID leaf index (eax).

        in sub_leaf of type int
            CPUID leaf sub index (ecx). This currently only applies to cache
            information on Intel CPUs. Use 0 if retrieving values for
            :py:func:`IMachine.set_cpuid_leaf` .

        out val_eax of type int
            CPUID leaf value for register eax.

        out val_ebx of type int
            CPUID leaf value for register ebx.

        out val_ecx of type int
            CPUID leaf value for register ecx.

        out val_edx of type int
            CPUID leaf value for register edx.

        """
        if not isinstance(cpu_id, baseinteger):
            raise TypeError("cpu_id can only be an instance of type baseinteger")
        if not isinstance(leaf, baseinteger):
            raise TypeError("leaf can only be an instance of type baseinteger")
        if not isinstance(sub_leaf, baseinteger):
            raise TypeError("sub_leaf can only be an instance of type baseinteger")
        (val_eax, val_ebx, val_ecx, val_edx) = self._call(
            "getProcessorCPUIDLeaf", in_p=[cpu_id, leaf, sub_leaf]
        )
        return (val_eax, val_ebx, val_ecx, val_edx)

    @property
    def memory_size(self):
        """Get int value for 'memorySize'
        Amount of system memory in megabytes installed in the host system.
        """
        ret = self._get_attr("memorySize")
        return ret

    @property
    def memory_available(self):
        """Get int value for 'memoryAvailable'
        Available system memory in the host system.
        """
        ret = self._get_attr("memoryAvailable")
        return ret

    @property
    def operating_system(self):
        """Get str value for 'operatingSystem'
        Name of the host system's operating system.
        """
        ret = self._get_attr("operatingSystem")
        return ret

    @property
    def os_version(self):
        """Get str value for 'OSVersion'
        Host operating system's version string.
        """
        ret = self._get_attr("OSVersion")
        return ret

    @property
    def utc_time(self):
        """Get int value for 'UTCTime'
        Returns the current host time in milliseconds since 1970-01-01 UTC.
        """
        ret = self._get_attr("UTCTime")
        return ret

    @property
    def acceleration3_d_available(self):
        """Get bool value for 'acceleration3DAvailable'
        Returns @c true when the host supports 3D hardware acceleration.
        """
        ret = self._get_attr("acceleration3DAvailable")
        return ret

    def create_host_only_network_interface(self):
        """Creates a new adapter for Host Only Networking.

        out host_interface of type :class:`IHostNetworkInterface`
            Created host interface object.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`OleErrorInvalidarg`
            Host network interface @a name already exists.

        """
        (progress, host_interface) = self._call("createHostOnlyNetworkInterface")
        progress = IProgress(progress)
        host_interface = IHostNetworkInterface(host_interface)
        return (progress, host_interface)

    def remove_host_only_network_interface(self, id_p):
        """Removes the given Host Only Networking interface.

        in id_p of type str
            Adapter GUID.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorObjectNotFound`
            No host network interface matching @a id found.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        progress = self._call("removeHostOnlyNetworkInterface", in_p=[id_p])
        progress = IProgress(progress)
        return progress

    def create_usb_device_filter(self, name):
        """Creates a new USB device filter. All attributes except
        the filter name are set to empty (any match),
        *active* is @c false (the filter is not active).

        The created filter can be added to the list of filters using
        :py:func:`insert_usb_device_filter` .

        :py:func:`usb_device_filters`

        in name of type str
            Filter name. See :py:func:`IUSBDeviceFilter.name`  for more information.

        return filter_p of type :class:`IHostUSBDeviceFilter`
            Created filter object.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        filter_p = self._call("createUSBDeviceFilter", in_p=[name])
        filter_p = IHostUSBDeviceFilter(filter_p)
        return filter_p

    def insert_usb_device_filter(self, position, filter_p):
        """Inserts the given USB device to the specified position
        in the list of filters.

        Positions are numbered starting from @c 0. If the specified
        position is equal to or greater than the number of elements in
        the list, the filter is added at the end of the collection.


        Duplicates are not allowed, so an attempt to insert a
        filter already in the list is an error.


        If USB functionality is not available in the given edition of
        VirtualBox, this method will set the result code to @c E_NOTIMPL.


        :py:func:`usb_device_filters`

        in position of type int
            Position to insert the filter to.

        in filter_p of type :class:`IHostUSBDeviceFilter`
            USB device filter to insert.

        raises :class:`VBoxErrorInvalidObjectState`
            USB device filter is not created within this VirtualBox instance.

        raises :class:`OleErrorInvalidarg`
            USB device filter already in list.

        """
        if not isinstance(position, baseinteger):
            raise TypeError("position can only be an instance of type baseinteger")
        if not isinstance(filter_p, IHostUSBDeviceFilter):
            raise TypeError(
                "filter_p can only be an instance of type IHostUSBDeviceFilter"
            )
        self._call("insertUSBDeviceFilter", in_p=[position, filter_p])

    def remove_usb_device_filter(self, position):
        """Removes a USB device filter from the specified position in the
        list of filters.

        Positions are numbered starting from @c 0. Specifying a
        position equal to or greater than the number of elements in
        the list will produce an error.


        If USB functionality is not available in the given edition of
        VirtualBox, this method will set the result code to @c E_NOTIMPL.


        :py:func:`usb_device_filters`

        in position of type int
            Position to remove the filter from.

        raises :class:`OleErrorInvalidarg`
            USB device filter list empty or invalid @a position.

        """
        if not isinstance(position, baseinteger):
            raise TypeError("position can only be an instance of type baseinteger")
        self._call("removeUSBDeviceFilter", in_p=[position])

    def find_host_dvd_drive(self, name):
        """Searches for a host DVD drive with the given @c name.

        in name of type str
            Name of the host drive to search for

        return drive of type :class:`IMedium`
            Found host drive object

        raises :class:`VBoxErrorObjectNotFound`
            Given @c name does not correspond to any host drive.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        drive = self._call("findHostDVDDrive", in_p=[name])
        drive = IMedium(drive)
        return drive

    def find_host_floppy_drive(self, name):
        """Searches for a host floppy drive with the given @c name.

        in name of type str
            Name of the host floppy drive to search for

        return drive of type :class:`IMedium`
            Found host floppy drive object

        raises :class:`VBoxErrorObjectNotFound`
            Given @c name does not correspond to any host floppy drive.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        drive = self._call("findHostFloppyDrive", in_p=[name])
        drive = IMedium(drive)
        return drive

    def find_host_network_interface_by_name(self, name):
        """Searches through all host network interfaces for an interface with
        the given @c name.

        The method returns an error if the given @c name does not
        correspond to any host network interface.

        in name of type str
            Name of the host network interface to search for.

        return network_interface of type :class:`IHostNetworkInterface`
            Found host network interface object.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        network_interface = self._call("findHostNetworkInterfaceByName", in_p=[name])
        network_interface = IHostNetworkInterface(network_interface)
        return network_interface

    def find_host_network_interface_by_id(self, id_p):
        """Searches through all host network interfaces for an interface with
        the given GUID.

        The method returns an error if the given GUID does not
        correspond to any host network interface.

        in id_p of type str
            GUID of the host network interface to search for.

        return network_interface of type :class:`IHostNetworkInterface`
            Found host network interface object.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        network_interface = self._call("findHostNetworkInterfaceById", in_p=[id_p])
        network_interface = IHostNetworkInterface(network_interface)
        return network_interface

    def find_host_network_interfaces_of_type(self, type_p):
        """Searches through all host network interfaces and returns a list of interfaces of the specified type

        in type_p of type :class:`HostNetworkInterfaceType`
            type of the host network interfaces to search for.

        return network_interfaces of type :class:`IHostNetworkInterface`
            Found host network interface objects.

        """
        if not isinstance(type_p, HostNetworkInterfaceType):
            raise TypeError(
                "type_p can only be an instance of type HostNetworkInterfaceType"
            )
        network_interfaces = self._call(
            "findHostNetworkInterfacesOfType", in_p=[type_p]
        )
        network_interfaces = [IHostNetworkInterface(a) for a in network_interfaces]
        return network_interfaces

    def find_usb_device_by_id(self, id_p):
        """Searches for a USB device with the given UUID.



        :py:func:`IUSBDevice.id_p`

        in id_p of type str
            UUID of the USB device to search for.

        return device of type :class:`IHostUSBDevice`
            Found USB device object.

        raises :class:`VBoxErrorObjectNotFound`
            Given @c id does not correspond to any USB device.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        device = self._call("findUSBDeviceById", in_p=[id_p])
        device = IHostUSBDevice(device)
        return device

    def find_usb_device_by_address(self, name):
        """Searches for a USB device with the given host address.



        :py:func:`IUSBDevice.address`

        in name of type str
            Address of the USB device (as assigned by the host) to
            search for.

        return device of type :class:`IHostUSBDevice`
            Found USB device object.

        raises :class:`VBoxErrorObjectNotFound`
            Given @c name does not correspond to any USB device.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        device = self._call("findUSBDeviceByAddress", in_p=[name])
        device = IHostUSBDevice(device)
        return device

    def generate_mac_address(self):
        """Generates a valid Ethernet MAC address, 12 hexadecimal characters.

        return address of type str
            New Ethernet MAC address.

        """
        address = self._call("generateMACAddress")
        return address

    @property
    def video_input_devices(self):
        """Get IHostVideoInputDevice value for 'videoInputDevices'
        List of currently available host video capture devices.
        """
        ret = self._get_attr("videoInputDevices")
        return [IHostVideoInputDevice(a) for a in ret]

    def add_usb_device_source(
        self, backend, id_p, address, property_names, property_values
    ):
        """Adds a new USB device source.

        in backend of type str
            The backend to use as the new device source.

        in id_p of type str
            Unique ID to identify the source.

        in address of type str
            Address to use, the format is dependent on the backend.
            For USB/IP backends for example the notation is host[:port].

        in property_names of type str
            Array of property names for more detailed configuration. Not used at the moment.

        in property_values of type str
            Array of property values for more detailed configuration. Not used at the moment.

        """
        if not isinstance(backend, basestring):
            raise TypeError("backend can only be an instance of type basestring")
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        if not isinstance(address, basestring):
            raise TypeError("address can only be an instance of type basestring")
        if not isinstance(property_names, list):
            raise TypeError("property_names can only be an instance of type list")
        for a in property_names[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(property_values, list):
            raise TypeError("property_values can only be an instance of type list")
        for a in property_values[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        self._call(
            "addUSBDeviceSource",
            in_p=[backend, id_p, address, property_names, property_values],
        )

    def remove_usb_device_source(self, id_p):
        """Removes a previously added USB device source.

        in id_p of type str
            The identifier used when the source was added.

        """
        if not isinstance(id_p, basestring):
            raise TypeError("id_p can only be an instance of type basestring")
        self._call("removeUSBDeviceSource", in_p=[id_p])

    @property
    def update(self):
        """Get IHostUpdate value for 'update'
        The check for newer software object (singleton).
        """
        ret = self._get_attr("update")
        return IHostUpdate(ret)


class ICPUProfile(Interface):
    """
    CPU profile. Immutable.
    """

    __uuid__ = "b7fda727-7a08-46ee-8dd8-f8d7308b519c"
    __wsmap__ = "managed"

    @property
    def name(self):
        """Get str value for 'name'
        The name.
        """
        ret = self._get_attr("name")
        return ret

    @property
    def full_name(self):
        """Get str value for 'fullName'
        The full name.
        """
        ret = self._get_attr("fullName")
        return ret

    @property
    def architecture(self):
        """Get CPUArchitecture value for 'architecture'
        The CPU architecture.
        """
        ret = self._get_attr("architecture")
        return CPUArchitecture(ret)


class ISystemProperties(Interface):
    """
    The ISystemProperties interface represents global properties of the given
    VirtualBox installation.

    These properties define limits and default values for various attributes
    and parameters. Most of the properties are read-only, but some can be
    changed by a user.
    """

    __uuid__ = "81c55eb1-d584-41a7-aa0b-08b71cddd773"
    __wsmap__ = "managed"

    @property
    def min_guest_ram(self):
        """Get int value for 'minGuestRAM'
        Minimum guest system memory in Megabytes.
        """
        ret = self._get_attr("minGuestRAM")
        return ret

    @property
    def max_guest_ram(self):
        """Get int value for 'maxGuestRAM'
        Maximum guest system memory in Megabytes.
        """
        ret = self._get_attr("maxGuestRAM")
        return ret

    @property
    def min_guest_vram(self):
        """Get int value for 'minGuestVRAM'
        Minimum guest video memory in Megabytes.
        """
        ret = self._get_attr("minGuestVRAM")
        return ret

    @property
    def max_guest_vram(self):
        """Get int value for 'maxGuestVRAM'
        Maximum guest video memory in Megabytes.
        """
        ret = self._get_attr("maxGuestVRAM")
        return ret

    @property
    def min_guest_cpu_count(self):
        """Get int value for 'minGuestCPUCount'
        Minimum CPU count.
        """
        ret = self._get_attr("minGuestCPUCount")
        return ret

    @property
    def max_guest_cpu_count(self):
        """Get int value for 'maxGuestCPUCount'
        Maximum CPU count.
        """
        ret = self._get_attr("maxGuestCPUCount")
        return ret

    @property
    def max_guest_monitors(self):
        """Get int value for 'maxGuestMonitors'
        Maximum of monitors which could be connected.
        """
        ret = self._get_attr("maxGuestMonitors")
        return ret

    @property
    def info_vd_size(self):
        """Get int value for 'infoVDSize'
        Maximum size of a virtual disk image in bytes. Informational value,
        does not reflect the limits of any virtual disk image format.
        """
        ret = self._get_attr("infoVDSize")
        return ret

    @property
    def serial_port_count(self):
        """Get int value for 'serialPortCount'
        Maximum number of serial ports associated with every
        :py:class:`IMachine`  instance.
        """
        ret = self._get_attr("serialPortCount")
        return ret

    @property
    def parallel_port_count(self):
        """Get int value for 'parallelPortCount'
        Maximum number of parallel ports associated with every
        :py:class:`IMachine`  instance.
        """
        ret = self._get_attr("parallelPortCount")
        return ret

    @property
    def max_boot_position(self):
        """Get int value for 'maxBootPosition'
        Maximum device position in the boot order. This value corresponds
        to the total number of devices a machine can boot from, to make it
        possible to include all possible devices to the boot list.
        :py:func:`IMachine.set_boot_order`
        """
        ret = self._get_attr("maxBootPosition")
        return ret

    @property
    def raw_mode_supported(self):
        """Get bool value for 'rawModeSupported'
        Indicates whether VirtualBox was built with raw-mode support.

        When this reads as False, the :py:attr:`HWVirtExPropertyType.enabled`
        setting will be ignored and assumed to be True.
        """
        ret = self._get_attr("rawModeSupported")
        return ret

    @property
    def exclusive_hw_virt(self):
        """Get or set bool value for 'exclusiveHwVirt'
        Exclusive use of hardware virtualization by VirtualBox. When enabled,
        VirtualBox assumes it can obtain full and exclusive access to the VT-x
        or AMD-V feature of the host. To share hardware virtualization with
        other hypervisors, this property must be disabled.

        This is ignored on OS X, the kernel mediates hardware
        access there.
        """
        ret = self._get_attr("exclusiveHwVirt")
        return ret

    @exclusive_hw_virt.setter
    def exclusive_hw_virt(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("exclusiveHwVirt", value)

    @property
    def default_machine_folder(self):
        """Get or set str value for 'defaultMachineFolder'
        Full path to the default directory used to create new or open
        existing machines when a machine settings file name contains no
        path.

        Starting with VirtualBox 4.0, by default, this attribute contains
        the full path of folder named "VirtualBox VMs" in the user's
        home directory, which depends on the host platform.

        When setting this attribute, a full path must be specified.
        Setting this property to @c null or an empty string or the
        special value "Machines" (for compatibility reasons) will restore
        that default value.

        If the folder specified herein does not exist, it will be created
        automatically as needed.


        :py:func:`IVirtualBox.create_machine` ,
        :py:func:`IVirtualBox.open_machine`
        """
        ret = self._get_attr("defaultMachineFolder")
        return ret

    @default_machine_folder.setter
    def default_machine_folder(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("defaultMachineFolder", value)

    @property
    def logging_level(self):
        """Get or set str value for 'loggingLevel'
        Specifies the logging level in current use by VirtualBox.
        """
        ret = self._get_attr("loggingLevel")
        return ret

    @logging_level.setter
    def logging_level(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("loggingLevel", value)

    @property
    def medium_formats(self):
        """Get IMediumFormat value for 'mediumFormats'
        List of all medium storage formats supported by this VirtualBox
        installation.

        Keep in mind that the medium format identifier
        (:py:func:`IMediumFormat.id_p` ) used in other API calls like
        :py:func:`IVirtualBox.create_medium`  to refer to a particular
        medium format is a case-insensitive string. This means that, for
        example, all of the following strings:

        ::

             "VDI"
             "vdi"
             "VdI"


        refer to the same medium format.

        Note that the virtual medium framework is backend-based, therefore
        the list of supported formats depends on what backends are currently
        installed.

        :py:class:`IMediumFormat`
        """
        ret = self._get_attr("mediumFormats")
        return [IMediumFormat(a) for a in ret]

    @property
    def default_hard_disk_format(self):
        """Get or set str value for 'defaultHardDiskFormat'
        Identifier of the default medium format used by VirtualBox.

        The medium format set by this attribute is used by VirtualBox
        when the medium format was not specified explicitly. One example is
        :py:func:`IVirtualBox.create_medium`  with the empty
        format argument. A more complex example is implicit creation of
        differencing media when taking a snapshot of a virtual machine:
        this operation will try to use a format of the parent medium first
        and if this format does not support differencing media the default
        format specified by this argument will be used.

        The list of supported medium formats may be obtained by the
        :py:func:`medium_formats`  call. Note that the default medium
        format must have a capability to create differencing media;
        otherwise operations that create media implicitly may fail
        unexpectedly.

        The initial value of this property is "VDI" in the current
        version of the VirtualBox product, but may change in the future.


        Setting this property to @c null or empty string will restore the
        initial value.



        :py:func:`medium_formats` ,
        :py:func:`IMediumFormat.id_p` ,
        :py:func:`IVirtualBox.create_medium`
        """
        ret = self._get_attr("defaultHardDiskFormat")
        return ret

    @default_hard_disk_format.setter
    def default_hard_disk_format(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("defaultHardDiskFormat", value)

    @property
    def free_disk_space_warning(self):
        """Get or set int value for 'freeDiskSpaceWarning'
        Issue a warning if the free disk space is below (or in some disk
        intensive operation is expected to go below) the given size in
        bytes.
        """
        ret = self._get_attr("freeDiskSpaceWarning")
        return ret

    @free_disk_space_warning.setter
    def free_disk_space_warning(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("freeDiskSpaceWarning", value)

    @property
    def free_disk_space_percent_warning(self):
        """Get or set int value for 'freeDiskSpacePercentWarning'
        Issue a warning if the free disk space is below (or in some disk
        intensive operation is expected to go below) the given percentage.
        """
        ret = self._get_attr("freeDiskSpacePercentWarning")
        return ret

    @free_disk_space_percent_warning.setter
    def free_disk_space_percent_warning(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("freeDiskSpacePercentWarning", value)

    @property
    def free_disk_space_error(self):
        """Get or set int value for 'freeDiskSpaceError'
        Issue an error if the free disk space is below (or in some disk
        intensive operation is expected to go below) the given size in
        bytes.
        """
        ret = self._get_attr("freeDiskSpaceError")
        return ret

    @free_disk_space_error.setter
    def free_disk_space_error(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("freeDiskSpaceError", value)

    @property
    def free_disk_space_percent_error(self):
        """Get or set int value for 'freeDiskSpacePercentError'
        Issue an error if the free disk space is below (or in some disk
        intensive operation is expected to go below) the given percentage.
        """
        ret = self._get_attr("freeDiskSpacePercentError")
        return ret

    @free_disk_space_percent_error.setter
    def free_disk_space_percent_error(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("freeDiskSpacePercentError", value)

    @property
    def vrde_auth_library(self):
        """Get or set str value for 'VRDEAuthLibrary'
        Library that provides authentication for Remote Desktop clients. The library
        is used if a virtual machine's authentication type is set to "external"
        in the VM RemoteDisplay configuration.

        The system library extension (".DLL" or ".so") must be omitted.
        A full path can be specified; if not, then the library must reside on the
        system's default library path.

        The default value of this property is "VBoxAuth". There is a library
        of that name in one of the default VirtualBox library directories.

        For details about VirtualBox authentication libraries and how to implement
        them, please refer to the VirtualBox manual.


        Setting this property to @c null or empty string will restore the
        initial value.
        """
        ret = self._get_attr("VRDEAuthLibrary")
        return ret

    @vrde_auth_library.setter
    def vrde_auth_library(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("VRDEAuthLibrary", value)

    @property
    def web_service_auth_library(self):
        """Get or set str value for 'webServiceAuthLibrary'
        Library that provides authentication for webservice clients. The library
        is used if a virtual machine's authentication type is set to "external"
        in the VM RemoteDisplay configuration and will be called from
        within the :py:func:`IWebsessionManager.logon`  implementation.

        As opposed to :py:func:`ISystemProperties.vrde_auth_library` ,
        there is no per-VM setting for this, as the webservice is a global
        resource (if it is running). Only for this setting (for the webservice),
        setting this value to a literal "null" string disables authentication,
        meaning that :py:func:`IWebsessionManager.logon`  will always succeed,
        no matter what user name and password are supplied.

        The initial value of this property is "VBoxAuth",
        meaning that the webservice will use the same authentication
        library that is used by default for VRDE (again, see
        :py:func:`ISystemProperties.vrde_auth_library` ).
        The format and calling convention of authentication libraries
        is the same for the webservice as it is for VRDE.


        Setting this property to @c null or empty string will restore the
        initial value.
        """
        ret = self._get_attr("webServiceAuthLibrary")
        return ret

    @web_service_auth_library.setter
    def web_service_auth_library(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("webServiceAuthLibrary", value)

    @property
    def default_vrde_ext_pack(self):
        """Get or set str value for 'defaultVRDEExtPack'
        The name of the extension pack providing the default VRDE.

        This attribute is for choosing between multiple extension packs
        providing VRDE. If only one is installed, it will automatically be the
        default one. The attribute value can be empty if no VRDE extension
        pack is installed.

        For details about VirtualBox Remote Desktop Extension and how to
        implement one, please refer to the VirtualBox SDK.
        """
        ret = self._get_attr("defaultVRDEExtPack")
        return ret

    @default_vrde_ext_pack.setter
    def default_vrde_ext_pack(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("defaultVRDEExtPack", value)

    @property
    def log_history_count(self):
        """Get or set int value for 'logHistoryCount'
        This value specifies how many old release log files are kept.
        """
        ret = self._get_attr("logHistoryCount")
        return ret

    @log_history_count.setter
    def log_history_count(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("logHistoryCount", value)

    @property
    def default_audio_driver(self):
        """Get AudioDriverType value for 'defaultAudioDriver'
        This value hold the default audio driver for the current
        system.
        """
        ret = self._get_attr("defaultAudioDriver")
        return AudioDriverType(ret)

    @property
    def autostart_database_path(self):
        """Get or set str value for 'autostartDatabasePath'
        The path to the autostart database. Depending on the host this might
        be a filesystem path or something else.
        """
        ret = self._get_attr("autostartDatabasePath")
        return ret

    @autostart_database_path.setter
    def autostart_database_path(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("autostartDatabasePath", value)

    @property
    def default_additions_iso(self):
        """Get or set str value for 'defaultAdditionsISO'
        The path to the default Guest Additions ISO image. Can be empty if
        the location is not known in this installation.
        """
        ret = self._get_attr("defaultAdditionsISO")
        return ret

    @default_additions_iso.setter
    def default_additions_iso(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("defaultAdditionsISO", value)

    @property
    def default_frontend(self):
        """Get or set str value for 'defaultFrontend'
        Selects which VM frontend should be used by default when launching
        a VM through the :py:func:`IMachine.launch_vm_process`  method.
        Empty or @c null strings do not define a particular default, it is up
        to :py:func:`IMachine.launch_vm_process`  to select one. See the
        description of :py:func:`IMachine.launch_vm_process`  for the valid
        frontend types.

        This global setting is overridden by the per-VM attribute
        :py:func:`IMachine.default_frontend`  or a frontend type
        passed to :py:func:`IMachine.launch_vm_process` .
        """
        ret = self._get_attr("defaultFrontend")
        return ret

    @default_frontend.setter
    def default_frontend(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("defaultFrontend", value)

    @property
    def screen_shot_formats(self):
        """Get BitmapFormat value for 'screenShotFormats'
        Supported bitmap formats which can be used with takeScreenShot
        and takeScreenShotToArray methods.
        """
        ret = self._get_attr("screenShotFormats")
        return [BitmapFormat(a) for a in ret]

    @property
    def proxy_mode(self):
        """Get or set ProxyMode value for 'proxyMode'
        The proxy mode setting: System, NoProxy or Manual.
        """
        ret = self._get_attr("proxyMode")
        return ProxyMode(ret)

    @proxy_mode.setter
    def proxy_mode(self, value):
        if not isinstance(value, ProxyMode):
            raise TypeError("value is not an instance of ProxyMode")
        return self._set_attr("proxyMode", value)

    @property
    def proxy_url(self):
        """Get or set str value for 'proxyURL'
        Proxy server URL for the :py:attr:`ProxyMode.manual`  proxy mode.

        The format is: [{type}"://"][{userid}[@{password}]:]{server}[":"{port}]

        Valid types are: http (default), https, socks4, socks4a, socks5, socks5h and direct.
        Please note that these are proxy types defining how the proxy operates rather than
        how to proxy any similarly named protocol (i.e. don't confuse a http-proxy with
        proxying the http protocol, as a http-proxy usually can proxy https and other protocols too).

        The port number defaults to 80 for http, 443 for https and 1080 for the socks ones.

        The password is currently stored as plain text!  Use the :py:attr:`ProxyMode.system`
        mode if you consider the proxy password to be sensitive.

        An empty string will cause the behavior to be identical to :py:attr:`ProxyMode.system` .
        For compatibility with libproxy, an URL starting with "direct://" will cause
        :py:attr:`ProxyMode.no_proxy`  behavior.
        """
        ret = self._get_attr("proxyURL")
        return ret

    @proxy_url.setter
    def proxy_url(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("proxyURL", value)

    @property
    def supported_paravirt_providers(self):
        """Get ParavirtProvider value for 'supportedParavirtProviders'
        Returns an array of officially supported values for enum :py:class:`ParavirtProvider` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedParavirtProviders")
        return [ParavirtProvider(a) for a in ret]

    @property
    def supported_clipboard_modes(self):
        """Get ClipboardMode value for 'supportedClipboardModes'
        Returns an array of officially supported values for enum :py:class:`ClipboardMode` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedClipboardModes")
        return [ClipboardMode(a) for a in ret]

    @property
    def supported_dn_d_modes(self):
        """Get DnDMode value for 'supportedDnDModes'
        Returns an array of officially supported values for enum :py:class:`DnDMode` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedDnDModes")
        return [DnDMode(a) for a in ret]

    @property
    def supported_firmware_types(self):
        """Get FirmwareType value for 'supportedFirmwareTypes'
        Returns an array of officially supported values for enum :py:class:`FirmwareType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedFirmwareTypes")
        return [FirmwareType(a) for a in ret]

    @property
    def supported_pointing_hid_types(self):
        """Get PointingHIDType value for 'supportedPointingHIDTypes'
        Returns an array of officially supported values for enum :py:class:`PointingHIDType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedPointingHIDTypes")
        return [PointingHIDType(a) for a in ret]

    @property
    def supported_keyboard_hid_types(self):
        """Get KeyboardHIDType value for 'supportedKeyboardHIDTypes'
        Returns an array of officially supported values for enum :py:class:`KeyboardHIDType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedKeyboardHIDTypes")
        return [KeyboardHIDType(a) for a in ret]

    @property
    def supported_vfs_types(self):
        """Get VFSType value for 'supportedVFSTypes'
        Returns an array of officially supported values for enum :py:class:`VFSType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedVFSTypes")
        return [VFSType(a) for a in ret]

    @property
    def supported_import_options(self):
        """Get ImportOptions value for 'supportedImportOptions'
        Returns an array of officially supported values for enum :py:class:`ImportOptions` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedImportOptions")
        return [ImportOptions(a) for a in ret]

    @property
    def supported_export_options(self):
        """Get ExportOptions value for 'supportedExportOptions'
        Returns an array of officially supported values for enum :py:class:`ExportOptions` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedExportOptions")
        return [ExportOptions(a) for a in ret]

    @property
    def supported_recording_audio_codecs(self):
        """Get RecordingAudioCodec value for 'supportedRecordingAudioCodecs'
        Returns an array of officially supported values for enum :py:class:`RecordingAudioCodec` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedRecordingAudioCodecs")
        return [RecordingAudioCodec(a) for a in ret]

    @property
    def supported_recording_video_codecs(self):
        """Get RecordingVideoCodec value for 'supportedRecordingVideoCodecs'
        Returns an array of officially supported values for enum :py:class:`RecordingVideoCodec` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedRecordingVideoCodecs")
        return [RecordingVideoCodec(a) for a in ret]

    @property
    def supported_recording_vs_methods(self):
        """Get RecordingVideoScalingMethod value for 'supportedRecordingVSMethods'
        Returns an array of officially supported values for enum :py:class:`RecordingVideoScalingMethod` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedRecordingVSMethods")
        return [RecordingVideoScalingMethod(a) for a in ret]

    @property
    def supported_recording_vrc_modes(self):
        """Get RecordingVideoRateControlMode value for 'supportedRecordingVRCModes'
        Returns an array of officially supported values for enum :py:class:`RecordingVideoRateControlMode` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedRecordingVRCModes")
        return [RecordingVideoRateControlMode(a) for a in ret]

    @property
    def supported_graphics_controller_types(self):
        """Get GraphicsControllerType value for 'supportedGraphicsControllerTypes'
        Returns an array of officially supported values for enum :py:class:`GraphicsControllerType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedGraphicsControllerTypes")
        return [GraphicsControllerType(a) for a in ret]

    @property
    def supported_clone_options(self):
        """Get CloneOptions value for 'supportedCloneOptions'
        Returns an array of officially supported values for enum :py:class:`CloneOptions` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedCloneOptions")
        return [CloneOptions(a) for a in ret]

    @property
    def supported_autostop_types(self):
        """Get AutostopType value for 'supportedAutostopTypes'
        Returns an array of officially supported values for enum :py:class:`AutostopType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedAutostopTypes")
        return [AutostopType(a) for a in ret]

    @property
    def supported_vm_proc_priorities(self):
        """Get VMProcPriority value for 'supportedVMProcPriorities'
        Returns an array of officially supported values for enum :py:class:`VMProcPriority` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedVMProcPriorities")
        return [VMProcPriority(a) for a in ret]

    @property
    def supported_network_attachment_types(self):
        """Get NetworkAttachmentType value for 'supportedNetworkAttachmentTypes'
        Returns an array of officially supported values for enum :py:class:`NetworkAttachmentType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedNetworkAttachmentTypes")
        return [NetworkAttachmentType(a) for a in ret]

    @property
    def supported_network_adapter_types(self):
        """Get NetworkAdapterType value for 'supportedNetworkAdapterTypes'
        Returns an array of officially supported values for enum :py:class:`NetworkAdapterType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedNetworkAdapterTypes")
        return [NetworkAdapterType(a) for a in ret]

    @property
    def supported_port_modes(self):
        """Get PortMode value for 'supportedPortModes'
        Returns an array of officially supported values for enum :py:class:`PortMode` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedPortModes")
        return [PortMode(a) for a in ret]

    @property
    def supported_uart_types(self):
        """Get UartType value for 'supportedUartTypes'
        Returns an array of officially supported values for enum :py:class:`UartType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedUartTypes")
        return [UartType(a) for a in ret]

    @property
    def supported_usb_controller_types(self):
        """Get USBControllerType value for 'supportedUSBControllerTypes'
        Returns an array of officially supported values for enum :py:class:`USBControllerType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedUSBControllerTypes")
        return [USBControllerType(a) for a in ret]

    @property
    def supported_audio_driver_types(self):
        """Get AudioDriverType value for 'supportedAudioDriverTypes'
        Returns an array of officially supported values for enum :py:class:`AudioDriverType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedAudioDriverTypes")
        return [AudioDriverType(a) for a in ret]

    @property
    def supported_audio_controller_types(self):
        """Get AudioControllerType value for 'supportedAudioControllerTypes'
        Returns an array of officially supported values for enum :py:class:`AudioControllerType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedAudioControllerTypes")
        return [AudioControllerType(a) for a in ret]

    @property
    def supported_storage_buses(self):
        """Get StorageBus value for 'supportedStorageBuses'
        Returns an array of officially supported values for enum :py:class:`StorageBus` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedStorageBuses")
        return [StorageBus(a) for a in ret]

    @property
    def supported_storage_controller_types(self):
        """Get StorageControllerType value for 'supportedStorageControllerTypes'
        Returns an array of officially supported values for enum :py:class:`StorageControllerType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedStorageControllerTypes")
        return [StorageControllerType(a) for a in ret]

    @property
    def supported_chipset_types(self):
        """Get ChipsetType value for 'supportedChipsetTypes'
        Returns an array of officially supported values for enum :py:class:`ChipsetType` ,
        in the sense of what is e.g. worth offering in the VirtualBox GUI.
        """
        ret = self._get_attr("supportedChipsetTypes")
        return [ChipsetType(a) for a in ret]

    @property
    def v_box_update_enabled(self):
        """Get or set bool value for 'VBoxUpdateEnabled'
        Is the VirtualBox update check enabled?
        """
        ret = self._get_attr("VBoxUpdateEnabled")
        return ret

    @v_box_update_enabled.setter
    def v_box_update_enabled(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("VBoxUpdateEnabled", value)

    @property
    def v_box_update_frequency(self):
        """Get or set int value for 'VBoxUpdateFrequency'
        How often should a check for a newer version of VirtualBox be made? (in days)
        """
        ret = self._get_attr("VBoxUpdateFrequency")
        return ret

    @v_box_update_frequency.setter
    def v_box_update_frequency(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("VBoxUpdateFrequency", value)

    @property
    def v_box_update_last_check_date(self):
        """Get or set str value for 'VBoxUpdateLastCheckDate'
        When was the update check last performed? If updating this attribute the
        string must be in ISO 8601 format (e.g. 2020-05-11T21:13:39.348416000Z).
        """
        ret = self._get_attr("VBoxUpdateLastCheckDate")
        return ret

    @v_box_update_last_check_date.setter
    def v_box_update_last_check_date(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("VBoxUpdateLastCheckDate", value)

    @property
    def v_box_update_target(self):
        """Get or set VBoxUpdateTarget value for 'VBoxUpdateTarget'
        The preferred release type used for determining whether a newer version of VirtualBox is available.
        """
        ret = self._get_attr("VBoxUpdateTarget")
        return VBoxUpdateTarget(ret)

    @v_box_update_target.setter
    def v_box_update_target(self, value):
        if not isinstance(value, VBoxUpdateTarget):
            raise TypeError("value is not an instance of VBoxUpdateTarget")
        return self._set_attr("VBoxUpdateTarget", value)

    @property
    def v_box_update_count(self):
        """Get or set int value for 'VBoxUpdateCount'
        The count of update check attempts.
        """
        ret = self._get_attr("VBoxUpdateCount")
        return ret

    @v_box_update_count.setter
    def v_box_update_count(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("VBoxUpdateCount", value)

    @property
    def supported_v_box_update_target_types(self):
        """Get VBoxUpdateTarget value for 'supportedVBoxUpdateTargetTypes'
        Returns an array of officially supported values for enum :py:class:`VBoxUpdateTarget` .
        """
        ret = self._get_attr("supportedVBoxUpdateTargetTypes")
        return [VBoxUpdateTarget(a) for a in ret]

    def get_max_network_adapters(self, chipset):
        """Maximum total number of network adapters associated with every
        :py:class:`IMachine`  instance.

        in chipset of type :class:`ChipsetType`
            The chipset type to get the value for.

        return max_network_adapters of type int
            The maximum total number of network adapters allowed.

        """
        if not isinstance(chipset, ChipsetType):
            raise TypeError("chipset can only be an instance of type ChipsetType")
        max_network_adapters = self._call("getMaxNetworkAdapters", in_p=[chipset])
        return max_network_adapters

    def get_max_network_adapters_of_type(self, chipset, type_p):
        """Maximum number of network adapters of a given attachment type,
        associated with every :py:class:`IMachine`  instance.

        in chipset of type :class:`ChipsetType`
            The chipset type to get the value for.

        in type_p of type :class:`NetworkAttachmentType`
            Type of attachment.

        return max_network_adapters of type int
            The maximum number of network adapters allowed for
            particular chipset and attachment type.

        """
        if not isinstance(chipset, ChipsetType):
            raise TypeError("chipset can only be an instance of type ChipsetType")
        if not isinstance(type_p, NetworkAttachmentType):
            raise TypeError(
                "type_p can only be an instance of type NetworkAttachmentType"
            )
        max_network_adapters = self._call(
            "getMaxNetworkAdaptersOfType", in_p=[chipset, type_p]
        )
        return max_network_adapters

    def get_max_devices_per_port_for_storage_bus(self, bus):
        """Returns the maximum number of devices which can be attached to a port
        for the given storage bus.

        in bus of type :class:`StorageBus`
            The storage bus type to get the value for.

        return max_devices_per_port of type int
            The maximum number of devices which can be attached to the port for the given
            storage bus.

        """
        if not isinstance(bus, StorageBus):
            raise TypeError("bus can only be an instance of type StorageBus")
        max_devices_per_port = self._call(
            "getMaxDevicesPerPortForStorageBus", in_p=[bus]
        )
        return max_devices_per_port

    def get_min_port_count_for_storage_bus(self, bus):
        """Returns the minimum number of ports the given storage bus supports.

        in bus of type :class:`StorageBus`
            The storage bus type to get the value for.

        return min_port_count of type int
            The minimum number of ports for the given storage bus.

        """
        if not isinstance(bus, StorageBus):
            raise TypeError("bus can only be an instance of type StorageBus")
        min_port_count = self._call("getMinPortCountForStorageBus", in_p=[bus])
        return min_port_count

    def get_max_port_count_for_storage_bus(self, bus):
        """Returns the maximum number of ports the given storage bus supports.

        in bus of type :class:`StorageBus`
            The storage bus type to get the value for.

        return max_port_count of type int
            The maximum number of ports for the given storage bus.

        """
        if not isinstance(bus, StorageBus):
            raise TypeError("bus can only be an instance of type StorageBus")
        max_port_count = self._call("getMaxPortCountForStorageBus", in_p=[bus])
        return max_port_count

    def get_max_instances_of_storage_bus(self, chipset, bus):
        """Returns the maximum number of storage bus instances which
        can be configured for each VM. This corresponds to the number of
        storage controllers one can have. Value may depend on chipset type
        used.

        in chipset of type :class:`ChipsetType`
            The chipset type to get the value for.

        in bus of type :class:`StorageBus`
            The storage bus type to get the value for.

        return max_instances of type int
            The maximum number of instances for the given storage bus.

        """
        if not isinstance(chipset, ChipsetType):
            raise TypeError("chipset can only be an instance of type ChipsetType")
        if not isinstance(bus, StorageBus):
            raise TypeError("bus can only be an instance of type StorageBus")
        max_instances = self._call("getMaxInstancesOfStorageBus", in_p=[chipset, bus])
        return max_instances

    def get_device_types_for_storage_bus(self, bus):
        """Returns list of all the supported device types
        (:py:class:`DeviceType` ) for the given type of storage
        bus.

        in bus of type :class:`StorageBus`
            The storage bus type to get the value for.

        return device_types of type :class:`DeviceType`
            The list of all supported device types for the given storage bus.

        """
        if not isinstance(bus, StorageBus):
            raise TypeError("bus can only be an instance of type StorageBus")
        device_types = self._call("getDeviceTypesForStorageBus", in_p=[bus])
        device_types = [DeviceType(a) for a in device_types]
        return device_types

    def get_storage_bus_for_storage_controller_type(self, storage_controller_type):
        """Returns the :py:class:`StorageBus`  enum value
        for a given storage controller type.

        in storage_controller_type of type :class:`StorageControllerType`
            The storage controller type to get the value for.

        return storage_bus of type :class:`StorageBus`
            The storage bus which is applicable.

        """
        if not isinstance(storage_controller_type, StorageControllerType):
            raise TypeError(
                "storage_controller_type can only be an instance of type StorageControllerType"
            )
        storage_bus = self._call(
            "getStorageBusForStorageControllerType", in_p=[storage_controller_type]
        )
        storage_bus = StorageBus(storage_bus)
        return storage_bus

    def get_storage_controller_types_for_storage_bus(self, storage_bus):
        """Returns the possible :py:class:`StorageControllerType`  enum values
        for a given storage bus.

        in storage_bus of type :class:`StorageBus`
            The storage bus type to get the values for.

        return storage_controller_type of type :class:`StorageControllerType`
            The enum values (sorted by what should be a sensible decreasing
            importance of the type) which are valid.

        """
        if not isinstance(storage_bus, StorageBus):
            raise TypeError("storage_bus can only be an instance of type StorageBus")
        storage_controller_type = self._call(
            "getStorageControllerTypesForStorageBus", in_p=[storage_bus]
        )
        storage_controller_type = [
            StorageControllerType(a) for a in storage_controller_type
        ]
        return storage_controller_type

    def get_default_io_cache_setting_for_storage_controller(self, controller_type):
        """Returns the default I/O cache setting for the
        given storage controller

        in controller_type of type :class:`StorageControllerType`
            The storage controller type to get the setting for.

        return enabled of type bool
            Returned flag indicating the default value

        """
        if not isinstance(controller_type, StorageControllerType):
            raise TypeError(
                "controller_type can only be an instance of type StorageControllerType"
            )
        enabled = self._call(
            "getDefaultIoCacheSettingForStorageController", in_p=[controller_type]
        )
        return enabled

    def get_storage_controller_hotplug_capable(self, controller_type):
        """Returns whether the given storage controller supports
        hot-plugging devices.

        in controller_type of type :class:`StorageControllerType`
            The storage controller to check the setting for.

        return hotplug_capable of type bool
            Returned flag indicating whether the controller is hotplug capable

        """
        if not isinstance(controller_type, StorageControllerType):
            raise TypeError(
                "controller_type can only be an instance of type StorageControllerType"
            )
        hotplug_capable = self._call(
            "getStorageControllerHotplugCapable", in_p=[controller_type]
        )
        return hotplug_capable

    def get_max_instances_of_usb_controller_type(self, chipset, type_p):
        """Returns the maximum number of USB controller instances which
        can be configured for each VM. This corresponds to the number of
        USB controllers one can have. Value may depend on chipset type
        used.

        in chipset of type :class:`ChipsetType`
            The chipset type to get the value for.

        in type_p of type :class:`USBControllerType`
            The USB controller type to get the value for.

        return max_instances of type int
            The maximum number of instances for the given USB controller type.

        """
        if not isinstance(chipset, ChipsetType):
            raise TypeError("chipset can only be an instance of type ChipsetType")
        if not isinstance(type_p, USBControllerType):
            raise TypeError("type_p can only be an instance of type USBControllerType")
        max_instances = self._call(
            "getMaxInstancesOfUSBControllerType", in_p=[chipset, type_p]
        )
        return max_instances

    def get_cpu_profiles(self, architecture, name_pattern):
        """Returns CPU profiles matching the given criteria.

        in architecture of type :class:`CPUArchitecture`
            The architecture to get profiles for. Required.

        in name_pattern of type str
            Name pattern. Simple wildcard matching using asterisk (*) and
            question mark (?).

        return profiles of type :class:`ICPUProfile`
            The matching CPU profiles.

        """
        if not isinstance(architecture, CPUArchitecture):
            raise TypeError(
                "architecture can only be an instance of type CPUArchitecture"
            )
        if not isinstance(name_pattern, basestring):
            raise TypeError("name_pattern can only be an instance of type basestring")
        profiles = self._call("getCPUProfiles", in_p=[architecture, name_pattern])
        profiles = [ICPUProfile(a) for a in profiles]
        return profiles


class IGuestOSType(Interface):
    """"""

    __uuid__ = "d0d6c6d8-e5db-4d2c-baaa-c71053a6236d"
    __wsmap__ = "struct"

    @property
    def family_id(self):
        """Get str value for 'familyId'
        Guest OS family identifier string.
        """
        ret = self._get_attr("familyId")
        return ret

    @property
    def family_description(self):
        """Get str value for 'familyDescription'
        Human readable description of the guest OS family.
        """
        ret = self._get_attr("familyDescription")
        return ret

    @property
    def id_p(self):
        """Get str value for 'id'
        Guest OS identifier string.
        """
        ret = self._get_attr("id")
        return ret

    @property
    def description(self):
        """Get str value for 'description'
        Human readable description of the guest OS.
        """
        ret = self._get_attr("description")
        return ret

    @property
    def is64_bit(self):
        """Get bool value for 'is64Bit'
        Returns @c true if the given OS is 64-bit
        """
        ret = self._get_attr("is64Bit")
        return ret

    @property
    def recommended_ioapic(self):
        """Get bool value for 'recommendedIOAPIC'
        Returns @c true if I/O-APIC recommended for this OS type.
        """
        ret = self._get_attr("recommendedIOAPIC")
        return ret

    @property
    def recommended_virt_ex(self):
        """Get bool value for 'recommendedVirtEx'
        Returns @c true if VT-x or AMD-V recommended for this OS type.
        """
        ret = self._get_attr("recommendedVirtEx")
        return ret

    @property
    def recommended_ram(self):
        """Get int value for 'recommendedRAM'
        Recommended RAM size in Megabytes.
        """
        ret = self._get_attr("recommendedRAM")
        return ret

    @property
    def recommended_graphics_controller(self):
        """Get GraphicsControllerType value for 'recommendedGraphicsController'
        Recommended graphics controller type.
        """
        ret = self._get_attr("recommendedGraphicsController")
        return GraphicsControllerType(ret)

    @property
    def recommended_vram(self):
        """Get int value for 'recommendedVRAM'
        Recommended video RAM size in Megabytes.
        """
        ret = self._get_attr("recommendedVRAM")
        return ret

    @property
    def recommended2_d_video_acceleration(self):
        """Get bool value for 'recommended2DVideoAcceleration'
        Returns @c true if 2D video acceleration is recommended for this OS type.
        """
        ret = self._get_attr("recommended2DVideoAcceleration")
        return ret

    @property
    def recommended3_d_acceleration(self):
        """Get bool value for 'recommended3DAcceleration'
        Returns @c true if 3D acceleration is recommended for this OS type.
        """
        ret = self._get_attr("recommended3DAcceleration")
        return ret

    @property
    def recommended_hdd(self):
        """Get int value for 'recommendedHDD'
        Recommended hard disk size in bytes.
        """
        ret = self._get_attr("recommendedHDD")
        return ret

    @property
    def adapter_type(self):
        """Get NetworkAdapterType value for 'adapterType'
        Returns recommended network adapter for this OS type.
        """
        ret = self._get_attr("adapterType")
        return NetworkAdapterType(ret)

    @property
    def recommended_pae(self):
        """Get bool value for 'recommendedPAE'
        Returns @c true if using PAE is recommended for this OS type.
        """
        ret = self._get_attr("recommendedPAE")
        return ret

    @property
    def recommended_dvd_storage_controller(self):
        """Get StorageControllerType value for 'recommendedDVDStorageController'
        Recommended storage controller type for DVD/CD drives.
        """
        ret = self._get_attr("recommendedDVDStorageController")
        return StorageControllerType(ret)

    @property
    def recommended_dvd_storage_bus(self):
        """Get StorageBus value for 'recommendedDVDStorageBus'
        Recommended storage bus type for DVD/CD drives.
        """
        ret = self._get_attr("recommendedDVDStorageBus")
        return StorageBus(ret)

    @property
    def recommended_hd_storage_controller(self):
        """Get StorageControllerType value for 'recommendedHDStorageController'
        Recommended storage controller type for HD drives.
        """
        ret = self._get_attr("recommendedHDStorageController")
        return StorageControllerType(ret)

    @property
    def recommended_hd_storage_bus(self):
        """Get StorageBus value for 'recommendedHDStorageBus'
        Recommended storage bus type for HD drives.
        """
        ret = self._get_attr("recommendedHDStorageBus")
        return StorageBus(ret)

    @property
    def recommended_firmware(self):
        """Get FirmwareType value for 'recommendedFirmware'
        Recommended firmware type.
        """
        ret = self._get_attr("recommendedFirmware")
        return FirmwareType(ret)

    @property
    def recommended_usbhid(self):
        """Get bool value for 'recommendedUSBHID'
        Returns @c true if using USB Human Interface Devices, such as keyboard and mouse recommended.
        """
        ret = self._get_attr("recommendedUSBHID")
        return ret

    @property
    def recommended_hpet(self):
        """Get bool value for 'recommendedHPET'
        Returns @c true if using HPET is recommended for this OS type.
        """
        ret = self._get_attr("recommendedHPET")
        return ret

    @property
    def recommended_usb_tablet(self):
        """Get bool value for 'recommendedUSBTablet'
        Returns @c true if using a USB Tablet is recommended.
        """
        ret = self._get_attr("recommendedUSBTablet")
        return ret

    @property
    def recommended_rtc_use_utc(self):
        """Get bool value for 'recommendedRTCUseUTC'
        Returns @c true if the RTC of this VM should be set to UTC
        """
        ret = self._get_attr("recommendedRTCUseUTC")
        return ret

    @property
    def recommended_chipset(self):
        """Get ChipsetType value for 'recommendedChipset'
        Recommended chipset type.
        """
        ret = self._get_attr("recommendedChipset")
        return ChipsetType(ret)

    @property
    def recommended_audio_controller(self):
        """Get AudioControllerType value for 'recommendedAudioController'
        Recommended audio controller type.
        """
        ret = self._get_attr("recommendedAudioController")
        return AudioControllerType(ret)

    @property
    def recommended_audio_codec(self):
        """Get AudioCodecType value for 'recommendedAudioCodec'
        Recommended audio codec type.
        """
        ret = self._get_attr("recommendedAudioCodec")
        return AudioCodecType(ret)

    @property
    def recommended_floppy(self):
        """Get bool value for 'recommendedFloppy'
        Returns @c true a floppy drive is recommended for this OS type.
        """
        ret = self._get_attr("recommendedFloppy")
        return ret

    @property
    def recommended_usb(self):
        """Get bool value for 'recommendedUSB'
        Returns @c true a USB controller is recommended for this OS type.
        """
        ret = self._get_attr("recommendedUSB")
        return ret

    @property
    def recommended_usb3(self):
        """Get bool value for 'recommendedUSB3'
        Returns @c true an xHCI (USB 3) controller is recommended for this OS type.
        """
        ret = self._get_attr("recommendedUSB3")
        return ret

    @property
    def recommended_tf_reset(self):
        """Get bool value for 'recommendedTFReset'
        Returns @c true if using VCPU reset on triple fault is recommended for this OS type.
        """
        ret = self._get_attr("recommendedTFReset")
        return ret

    @property
    def recommended_x2_apic(self):
        """Get bool value for 'recommendedX2APIC'
        Returns @c true if X2APIC is recommended for this OS type.
        """
        ret = self._get_attr("recommendedX2APIC")
        return ret


class IAdditionsFacility(Interface):
    """
    Structure representing a Guest Additions facility.
    """

    __uuid__ = "f2f7fae4-4a06-81fc-a916-78b2da1fa0e5"
    __wsmap__ = "struct"

    @property
    def class_type(self):
        """Get AdditionsFacilityClass value for 'classType'
        The class this facility is part of.
        """
        ret = self._get_attr("classType")
        return AdditionsFacilityClass(ret)

    @property
    def last_updated(self):
        """Get int value for 'lastUpdated'
        Timestamp of the last status update,
        in milliseconds since 1970-01-01 UTC.
        """
        ret = self._get_attr("lastUpdated")
        return ret

    @property
    def name(self):
        """Get str value for 'name'
        The facility's friendly name.
        """
        ret = self._get_attr("name")
        return ret

    @property
    def status(self):
        """Get AdditionsFacilityStatus value for 'status'
        The current status.
        """
        ret = self._get_attr("status")
        return AdditionsFacilityStatus(ret)

    @property
    def type_p(self):
        """Get AdditionsFacilityType value for 'type'
        The facility's type ID.
        """
        ret = self._get_attr("type")
        return AdditionsFacilityType(ret)


class IDnDBase(Interface):
    """
    Base abstract interface for drag'n drop.
    """

    __uuid__ = "00727A73-000A-4C4A-006D-E7D300351186"
    __wsmap__ = "managed"

    @property
    def formats(self):
        """Get str value for 'formats'
        Returns all supported drag'n drop formats.
        """
        ret = self._get_attr("formats")
        return ret

    def is_format_supported(self, format_p):
        """Checks if a specific drag'n drop MIME / Content-type format is supported.

        in format_p of type str
            Format to check for.

        return supported of type bool
            Returns @c true if the specified format is supported, @c false if not.

        """
        if not isinstance(format_p, basestring):
            raise TypeError("format_p can only be an instance of type basestring")
        supported = self._call("isFormatSupported", in_p=[format_p])
        return supported

    def add_formats(self, formats):
        """Adds MIME / Content-type formats to the supported formats.

        in formats of type str
            Collection of formats to add.

        """
        if not isinstance(formats, list):
            raise TypeError("formats can only be an instance of type list")
        for a in formats[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        self._call("addFormats", in_p=[formats])

    def remove_formats(self, formats):
        """Removes MIME / Content-type formats from the supported formats.

        in formats of type str
            Collection of formats to remove.

        """
        if not isinstance(formats, list):
            raise TypeError("formats can only be an instance of type list")
        for a in formats[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        self._call("removeFormats", in_p=[formats])


class IDnDSource(IDnDBase):
    """
    Abstract interface for handling drag'n drop sources.
    """

    __uuid__ = "d23a9ca3-42da-c94b-8aec-21968e08355d"
    __wsmap__ = "managed"

    def drag_is_pending(self, screen_id):
        """Ask the source if there is any drag and drop operation pending.
        If no drag and drop operation is pending currently, DnDAction_Ignore is returned.

        in screen_id of type int
            The screen ID where the drag and drop event occurred.

        out formats of type str
            On return the supported mime types.

        out allowed_actions of type :class:`DnDAction`
            On return the actions which are allowed.

        return default_action of type :class:`DnDAction`
            On return the default action to use.

        raises :class:`VBoxErrorVmError`
            VMM device is not available.

        """
        if not isinstance(screen_id, baseinteger):
            raise TypeError("screen_id can only be an instance of type baseinteger")
        (default_action, formats, allowed_actions) = self._call(
            "dragIsPending", in_p=[screen_id]
        )
        default_action = DnDAction(default_action)
        allowed_actions = [DnDAction(a) for a in allowed_actions]
        return (default_action, formats, allowed_actions)

    def drop(self, format_p, action):
        """Informs the source that a drop event occurred for a pending
        drag and drop operation.

        in format_p of type str
            The mime type the data must be in.

        in action of type :class:`DnDAction`
            The action to use.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorVmError`
            VMM device is not available.

        """
        if not isinstance(format_p, basestring):
            raise TypeError("format_p can only be an instance of type basestring")
        if not isinstance(action, DnDAction):
            raise TypeError("action can only be an instance of type DnDAction")
        progress = self._call("drop", in_p=[format_p, action])
        progress = IProgress(progress)
        return progress

    def receive_data(self):
        """Receive the data of a previously drag and drop event from the source.

        return data of type str
            The actual data.

        raises :class:`VBoxErrorVmError`
            VMM device is not available.

        """
        data = self._call("receiveData")
        return data


class IGuestDnDSource(IDnDSource):
    """
    Implementation of the :py:class:`IDnDSource`  object
    for source drag'n drop operations on the guest.
    """

    __uuid__ = "dedfb5d9-4c1b-edf7-fdf3-c1be6827dc28"
    __wsmap__ = "managed"

    @property
    def midl_does_not_like_empty_interfaces(self):
        """Get bool value for 'midlDoesNotLikeEmptyInterfaces'"""
        ret = self._get_attr("midlDoesNotLikeEmptyInterfaces")
        return ret


class IDnDTarget(IDnDBase):
    """
    Abstract interface for handling drag'n drop targets.
    """

    __uuid__ = "ff5befc3-4ba3-7903-2aa4-43988ba11554"
    __wsmap__ = "managed"

    def enter(self, screen_id, y, x, default_action, allowed_actions, formats):
        """Informs the target about a drag and drop enter event.

        in screen_id of type int
            The screen ID where the drag and drop event occurred.

        in y of type int
            Y-position of the event.

        in x of type int
            X-position of the event.

        in default_action of type :class:`DnDAction`
            The default action to use.

        in allowed_actions of type :class:`DnDAction`
            The actions which are allowed.

        in formats of type str
            The supported MIME types.

        return result_action of type :class:`DnDAction`
            The resulting action of this event.

        raises :class:`VBoxErrorVmError`
            VMM device is not available.

        """
        if not isinstance(screen_id, baseinteger):
            raise TypeError("screen_id can only be an instance of type baseinteger")
        if not isinstance(y, baseinteger):
            raise TypeError("y can only be an instance of type baseinteger")
        if not isinstance(x, baseinteger):
            raise TypeError("x can only be an instance of type baseinteger")
        if not isinstance(default_action, DnDAction):
            raise TypeError("default_action can only be an instance of type DnDAction")
        if not isinstance(allowed_actions, list):
            raise TypeError("allowed_actions can only be an instance of type list")
        for a in allowed_actions[:10]:
            if not isinstance(a, DnDAction):
                raise TypeError("array can only contain objects of type DnDAction")
        if not isinstance(formats, list):
            raise TypeError("formats can only be an instance of type list")
        for a in formats[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        result_action = self._call(
            "enter", in_p=[screen_id, y, x, default_action, allowed_actions, formats]
        )
        result_action = DnDAction(result_action)
        return result_action

    def move(self, screen_id, x, y, default_action, allowed_actions, formats):
        """Informs the target about a drag and drop move event.

        in screen_id of type int
            The screen ID where the drag and drop event occurred.

        in x of type int
            X-position of the event.

        in y of type int
            Y-position of the event.

        in default_action of type :class:`DnDAction`
            The default action to use.

        in allowed_actions of type :class:`DnDAction`
            The actions which are allowed.

        in formats of type str
            The supported MIME types.

        return result_action of type :class:`DnDAction`
            The resulting action of this event.

        raises :class:`VBoxErrorVmError`
            VMM device is not available.

        """
        if not isinstance(screen_id, baseinteger):
            raise TypeError("screen_id can only be an instance of type baseinteger")
        if not isinstance(x, baseinteger):
            raise TypeError("x can only be an instance of type baseinteger")
        if not isinstance(y, baseinteger):
            raise TypeError("y can only be an instance of type baseinteger")
        if not isinstance(default_action, DnDAction):
            raise TypeError("default_action can only be an instance of type DnDAction")
        if not isinstance(allowed_actions, list):
            raise TypeError("allowed_actions can only be an instance of type list")
        for a in allowed_actions[:10]:
            if not isinstance(a, DnDAction):
                raise TypeError("array can only contain objects of type DnDAction")
        if not isinstance(formats, list):
            raise TypeError("formats can only be an instance of type list")
        for a in formats[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        result_action = self._call(
            "move", in_p=[screen_id, x, y, default_action, allowed_actions, formats]
        )
        result_action = DnDAction(result_action)
        return result_action

    def leave(self, screen_id):
        """Informs the target about a drag and drop leave event.

        in screen_id of type int
            The screen ID where the drag and drop event occurred.

        raises :class:`VBoxErrorVmError`
            VMM device is not available.

        """
        if not isinstance(screen_id, baseinteger):
            raise TypeError("screen_id can only be an instance of type baseinteger")
        self._call("leave", in_p=[screen_id])

    def drop(self, screen_id, x, y, default_action, allowed_actions, formats):
        """Informs the target about a drop event.

        in screen_id of type int
            The screen ID where the Drag and Drop event occurred.

        in x of type int
            X-position of the event.

        in y of type int
            Y-position of the event.

        in default_action of type :class:`DnDAction`
            The default action to use.

        in allowed_actions of type :class:`DnDAction`
            The actions which are allowed.

        in formats of type str
            The supported MIME types.

        out format_p of type str
            The resulting format of this event.

        return result_action of type :class:`DnDAction`
            The resulting action of this event.

        raises :class:`VBoxErrorVmError`
            VMM device is not available.

        """
        if not isinstance(screen_id, baseinteger):
            raise TypeError("screen_id can only be an instance of type baseinteger")
        if not isinstance(x, baseinteger):
            raise TypeError("x can only be an instance of type baseinteger")
        if not isinstance(y, baseinteger):
            raise TypeError("y can only be an instance of type baseinteger")
        if not isinstance(default_action, DnDAction):
            raise TypeError("default_action can only be an instance of type DnDAction")
        if not isinstance(allowed_actions, list):
            raise TypeError("allowed_actions can only be an instance of type list")
        for a in allowed_actions[:10]:
            if not isinstance(a, DnDAction):
                raise TypeError("array can only contain objects of type DnDAction")
        if not isinstance(formats, list):
            raise TypeError("formats can only be an instance of type list")
        for a in formats[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        (result_action, format_p) = self._call(
            "drop", in_p=[screen_id, x, y, default_action, allowed_actions, formats]
        )
        result_action = DnDAction(result_action)
        return (result_action, format_p)

    def send_data(self, screen_id, format_p, data):
        """Initiates sending data to the target.

        in screen_id of type int
            The screen ID where the drag and drop event occurred.

        in format_p of type str
            The MIME type the data is in.

        in data of type str
            The actual data.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorVmError`
            VMM device is not available.

        """
        if not isinstance(screen_id, baseinteger):
            raise TypeError("screen_id can only be an instance of type baseinteger")
        if not isinstance(format_p, basestring):
            raise TypeError("format_p can only be an instance of type basestring")
        if not isinstance(data, list):
            raise TypeError("data can only be an instance of type list")
        for a in data[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        progress = self._call("sendData", in_p=[screen_id, format_p, data])
        progress = IProgress(progress)
        return progress

    def cancel(self):
        """Requests cancelling the current operation. The target can veto
        the request in case the operation is not cancelable at the moment.

        return veto of type bool
            Whether the target has vetoed cancelling the operation.

        raises :class:`VBoxErrorVmError`
            VMM device is not available.

        """
        veto = self._call("cancel")
        return veto


class IGuestDnDTarget(IDnDTarget):
    """
    Implementation of the :py:class:`IDnDTarget`  object
    for target drag'n drop operations on the guest.
    """

    __uuid__ = "50ce4b51-0ff7-46b7-a138-3c6e5ac946b4"
    __wsmap__ = "managed"

    @property
    def midl_does_not_like_empty_interfaces(self):
        """Get bool value for 'midlDoesNotLikeEmptyInterfaces'"""
        ret = self._get_attr("midlDoesNotLikeEmptyInterfaces")
        return ret


class IGuestSession(Interface):
    """
    A guest session represents one impersonated user account in the guest, so
    every operation will use the same credentials specified when creating
    the session object via :py:func:`IGuest.create_session` .

    There can be a maximum of 32 sessions at once per VM, whereas session 0
    always is reserved for the root session (the root session is part of that
    limit).

    This root session is controlling all other guest sessions and also is
    responsible for actions which require system level privileges.

    Each guest session keeps track of the guest directories and files that
    it opened as well as guest processes it has created.  To work on guest
    files or directories a guest session offers methods to open or create
    such objects (see :py:func:`IGuestSession.file_open`  or
    :py:func:`IGuestSession.directory_open`  for instance).  Similarly,
    there a methods for creating guest processes.

    There can be up to 2048 objects (guest processes, files and directories)
    a time per guest session.  Exceeding the limit will result in an error (see
    the corresponding functions for more).

    When done with either of these objects, including the guest session itself,
    use the appropriate close() method to let the object do its cleanup work.

    Closing a session via :py:func:`IGuestSession.close`  will try to close
    all the mentioned objects above unless these objects are still used by
    a client.

    A set of environment variables changes is associated with each session
    (:py:func:`IGuestSession.environment_changes` ).  These are applied to
    the base environment of the impersonated guest user when creating a new
    guest process.  For additional flexibility the :py:func:`IGuestSession.process_create`
    and :py:func:`IGuestSession.process_create_ex`  methods allows you to
    specify individual environment changes for each process you create.
    With newer guest addition versions, the base environment is also made
    available via :py:func:`IGuestSession.environment_base` .  (One reason
    for why we record changes to a base environment instead of working
    directly on an environment block is that we need to be compatible
    with older Guest Additions.  Another reason is that this way it is always
    possible to undo all the changes you've scheduled.)
    """

    __uuid__ = "3E14C189-4A75-437E-B0BB-7E7C90D0DF2A"
    __wsmap__ = "managed"

    @property
    def user(self):
        """Get str value for 'user'
        Returns the user name used by this session to impersonate
        users in the guest.
        """
        ret = self._get_attr("user")
        return ret

    @property
    def domain(self):
        """Get str value for 'domain'
        Returns the domain name used by this session to impersonate
        users in the guest.
        """
        ret = self._get_attr("domain")
        return ret

    @property
    def name(self):
        """Get str value for 'name'
        Returns the session's friendly name.
        """
        ret = self._get_attr("name")
        return ret

    @property
    def id_p(self):
        """Get int value for 'id'
        Returns the internal session ID.
        """
        ret = self._get_attr("id")
        return ret

    @property
    def timeout(self):
        """Get or set int value for 'timeout'
        <!-- r=bird: Using 'Returns' for writable attributes is misleading. -->
        Returns the session timeout (in ms).
        """
        ret = self._get_attr("timeout")
        return ret

    @timeout.setter
    def timeout(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("timeout", value)

    @property
    def protocol_version(self):
        """Get int value for 'protocolVersion'
        Returns the protocol version which is used by this session to
        communicate with the guest.
        """
        ret = self._get_attr("protocolVersion")
        return ret

    @property
    def status(self):
        """Get GuestSessionStatus value for 'status'
        Returns the current session status.
        """
        ret = self._get_attr("status")
        return GuestSessionStatus(ret)

    @property
    def environment_changes(self):
        """Get or set str value for 'environmentChanges'
        The set of scheduled environment changes to the base environment of the
        session.  They are in putenv format, i.e. "VAR=VALUE" for setting and
        "VAR" for unsetting.  One entry per variable (change).  The changes are
        applied when creating new guest processes.

        This is writable, so to undo all the scheduled changes, assign it an
        empty array.
        """
        ret = self._get_attr("environmentChanges")
        return ret

    @environment_changes.setter
    def environment_changes(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("environmentChanges", value)

    @property
    def environment_base(self):
        """Get str value for 'environmentBase'
        The base environment of the session.  They are on the "VAR=VALUE" form,
        one array entry per variable.
        <!-- @todo/TODO/FIXME: This doesn't end up in the PDF.
        -->

        Access fails with VBOX_E_NOT_SUPPORTED if the Guest Additions does not
        support the session base environment feature.  Support for this was
        introduced with protocol version XXXX.

        Access fails with VBOX_E_INVALID_OBJECT_STATE if the Guest Additions
        has yet to report the session base environment.
        """
        ret = self._get_attr("environmentBase")
        return ret

    @property
    def processes(self):
        """Get IGuestProcess value for 'processes'
        Returns all current guest processes.
        """
        ret = self._get_attr("processes")
        return [IGuestProcess(a) for a in ret]

    @property
    def path_style(self):
        """Get PathStyle value for 'pathStyle'
        The style of paths used by the guest.  Handy for giving the right kind
        of path specifications to :py:func:`IGuestSession.file_open`  and similar methods.
        """
        ret = self._get_attr("pathStyle")
        return PathStyle(ret)

    @property
    def current_directory(self):
        """Get or set str value for 'currentDirectory'
        Gets or sets the current directory of the session.  Guest path style.
        """
        ret = self._get_attr("currentDirectory")
        return ret

    @current_directory.setter
    def current_directory(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("currentDirectory", value)

    @property
    def user_home(self):
        """Get str value for 'userHome'
        Returns the user's home / profile directory.  Guest path style.
        """
        ret = self._get_attr("userHome")
        return ret

    @property
    def user_documents(self):
        """Get str value for 'userDocuments'
        Returns the user's documents directory.  Guest path style.
        """
        ret = self._get_attr("userDocuments")
        return ret

    @property
    def directories(self):
        """Get IGuestDirectory value for 'directories'
        Returns all currently opened guest directories.
        """
        ret = self._get_attr("directories")
        return [IGuestDirectory(a) for a in ret]

    @property
    def files(self):
        """Get IGuestFile value for 'files'
        Returns all currently opened guest files.
        """
        ret = self._get_attr("files")
        return [IGuestFile(a) for a in ret]

    @property
    def event_source(self):
        """Get IEventSource value for 'eventSource'
        Event source for guest session events.
        """
        ret = self._get_attr("eventSource")
        return IEventSource(ret)

    def close(self):
        """Closes this session. All opened guest directories, files and
        processes which are not referenced by clients anymore will be
        closed. Guest processes which fall into this category and still
        are running in the guest will be terminated automatically.

        """
        self._call("close")

    def copy_from_guest(self, sources, filters, flags, destination):
        """Copies directories and/or files from guest to the host.

        This function requires several parallel arrays to be supplied, one
        set for each source.

        in sources of type str
            Paths to directories and/or files on the guest side that should be
            copied to the host. If the path ends with a path delimiter, only
            the directory's content is being copied. Guest path style.

        in filters of type str
            Array of source filters. This uses the
            DOS/NT style wildcard characters '?' and '*'.

        in flags of type str
            Array of comma-separated list of source flags.

            The following flags are available for directory sources:



            CopyIntoExisting
            Allow copying into an existing destination directory.



            The following flags are available for file sources:



            NoReplace
            Do not replace any destination object.


            FollowLinks
            Follows (and handles) (symbolic) links.


            Update
            Only copy when the source file is newer than the destination
            file or when the destination file is missing.

        in destination of type str
            Where to put the sources on the host. Host path style.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        """
        if not isinstance(sources, list):
            raise TypeError("sources can only be an instance of type list")
        for a in sources[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(filters, list):
            raise TypeError("filters can only be an instance of type list")
        for a in filters[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(destination, basestring):
            raise TypeError("destination can only be an instance of type basestring")
        progress = self._call(
            "copyFromGuest", in_p=[sources, filters, flags, destination]
        )
        progress = IProgress(progress)
        return progress

    def copy_to_guest(self, sources, filters, flags, destination):
        """Copies directories and/or files from host to the guest.

        This function requires several parallel arrays to be supplied, one
        set for each source.

        in sources of type str
            Paths to directories and/or files on the host side that should be
            copied to the guest. If the path ends with a path delimiter, only
            the directory's content is being copied. Host path style.

        in filters of type str
            Array of source filters. This uses the
            DOS/NT style wildcard characters '?' and '*'.

        in flags of type str
            Array of comma-separated list of source flags.

            The following flags are available for directory sources:



            CopyIntoExisting
            Allow copying into an existing destination directory.



            The following flags are available for file sources:



            NoReplace
            Do not replace any destination object.


            FollowLinks
            Follows (and handles) (symbolic) links.


            Update
            Only copy when the source file is newer than the destination
            file or when the destination file is missing.

        in destination of type str
            Where to put the sources on the guest. Guest path style.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        """
        if not isinstance(sources, list):
            raise TypeError("sources can only be an instance of type list")
        for a in sources[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(filters, list):
            raise TypeError("filters can only be an instance of type list")
        for a in filters[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(destination, basestring):
            raise TypeError("destination can only be an instance of type basestring")
        progress = self._call(
            "copyToGuest", in_p=[sources, filters, flags, destination]
        )
        progress = IProgress(progress)
        return progress

    def directory_copy(self, source, destination, flags):
        """Recursively copies a directory from one guest location to another.

        in source of type str
            The path to the directory to copy (in the guest).  Guest path style.

        in destination of type str
            The path to the target directory (in the guest).  Unless the
            :py:attr:`DirectoryCopyFlag.copy_into_existing`  flag is given, the
            directory shall not already exist.  Guest path style.

        in flags of type :class:`DirectoryCopyFlag`
            Zero or more :py:class:`DirectoryCopyFlag`  values.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        raises :class:`OleErrorNotimpl`
            Not yet implemented.

        """
        if not isinstance(source, basestring):
            raise TypeError("source can only be an instance of type basestring")
        if not isinstance(destination, basestring):
            raise TypeError("destination can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, DirectoryCopyFlag):
                raise TypeError(
                    "array can only contain objects of type DirectoryCopyFlag"
                )
        progress = self._call("directoryCopy", in_p=[source, destination, flags])
        progress = IProgress(progress)
        return progress

    def directory_copy_from_guest(self, source, destination, flags):
        """Recursively copies a directory from the guest to the host.

        in source of type str
            Path to the directory on the guest side that should be copied to
            the host. Guest path style.

        in destination of type str
            Where to put the directory on the host.  Unless the
            :py:attr:`DirectoryCopyFlag.copy_into_existing`  flag is given, the
            directory shall not already exist.  Host path style.

        in flags of type :class:`DirectoryCopyFlag`
            Zero or more :py:class:`DirectoryCopyFlag`  values.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        """
        if not isinstance(source, basestring):
            raise TypeError("source can only be an instance of type basestring")
        if not isinstance(destination, basestring):
            raise TypeError("destination can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, DirectoryCopyFlag):
                raise TypeError(
                    "array can only contain objects of type DirectoryCopyFlag"
                )
        progress = self._call(
            "directoryCopyFromGuest", in_p=[source, destination, flags]
        )
        progress = IProgress(progress)
        return progress

    def directory_copy_to_guest(self, source, destination, flags):
        """Recursively copies a directory from the host to the guest.

        in source of type str
            Path to the directory on the host side that should be copied to
            the guest.  Host path style.

        in destination of type str
            Where to put the file in the guest. Unless the
            :py:attr:`DirectoryCopyFlag.copy_into_existing`  flag is given, the
            directory shall not already exist. Guest style path.

        in flags of type :class:`DirectoryCopyFlag`
            Zero or more :py:class:`DirectoryCopyFlag`  values.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        """
        if not isinstance(source, basestring):
            raise TypeError("source can only be an instance of type basestring")
        if not isinstance(destination, basestring):
            raise TypeError("destination can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, DirectoryCopyFlag):
                raise TypeError(
                    "array can only contain objects of type DirectoryCopyFlag"
                )
        progress = self._call("directoryCopyToGuest", in_p=[source, destination, flags])
        progress = IProgress(progress)
        return progress

    def directory_create(self, path, mode, flags):
        """Creates a directory in the guest.

        in path of type str
            Path to the directory directory to be created. Guest path style.

        in mode of type int
            The UNIX-style access mode mask to create the directory with.
            Whether/how all three access groups and associated access rights are
            realized is guest OS dependent.  The API does the best it can on each
            OS.

        in flags of type :class:`DirectoryCreateFlag`
            Zero or more :py:class:`DirectoryCreateFlag`  flags.

        raises :class:`VBoxErrorIprtError`
            Error while creating the directory.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(mode, baseinteger):
            raise TypeError("mode can only be an instance of type baseinteger")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, DirectoryCreateFlag):
                raise TypeError(
                    "array can only contain objects of type DirectoryCreateFlag"
                )
        self._call("directoryCreate", in_p=[path, mode, flags])

    def directory_create_temp(self, template_name, mode, path, secure):
        """Creates a temporary directory in the guest.

                in template_name of type str
                    Template for the name of the directory to create. This must
                    contain at least one 'X' character. The first group of consecutive
                    'X' characters in the template will be replaced by a random
                    alphanumeric string to produce a unique name.

                in mode of type int
                    The UNIX-style access mode mask to create the directory with.
                    Whether/how all three access groups and associated access rights are
                    realized is guest OS dependent.  The API does the best it can on each
                    OS.

                    This parameter is ignore if the @a secure parameter is set to @c true.
                    It is strongly recommended to use 0700.

                in path of type str
                    The path to the directory in which the temporary directory should
                    be created. Guest path style.

                in secure of type bool
                    Whether to fail if the directory can not be securely created.
                    Currently this means that another unprivileged user cannot
                    manipulate the path specified or remove the temporary directory
                    after it has been created. Also causes the mode specified to be
                    ignored. May not be supported on all guest types.

                return directory of type str
                    On success this will contain the full path to the created
                    directory. Guest path style.

                raises :class:`VBoxErrorNotSupported`
                    The operation is not possible as requested on this particular
        guest type.

                raises :class:`OleErrorInvalidarg`
                    Invalid argument. This includes an incorrectly formatted template,
        or a non-absolute path.

                raises :class:`VBoxErrorIprtError`
                    The temporary directory could not be created. Possible reasons
        include a non-existing path or an insecure path when the secure
        option was requested.

        """
        if not isinstance(template_name, basestring):
            raise TypeError("template_name can only be an instance of type basestring")
        if not isinstance(mode, baseinteger):
            raise TypeError("mode can only be an instance of type baseinteger")
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(secure, bool):
            raise TypeError("secure can only be an instance of type bool")
        directory = self._call(
            "directoryCreateTemp", in_p=[template_name, mode, path, secure]
        )
        return directory

    def directory_exists(self, path, follow_symlinks):
        """Checks whether a directory exists in the guest or not.

        in path of type str
            Path to the directory to check if exists. Guest path style.

        in follow_symlinks of type bool
            If @c true, symbolic links in the final component will be followed
            and the existance of the symlink target made the question for this method.
            If @c false, a symbolic link in the final component will make the
            method return @c false (because a symlink isn't a directory).

        return exists of type bool
            Returns @c true if the directory exists, @c false if not, or is not a directory.

        raises :class:`VBoxErrorIprtError`
            Error while checking existence of the directory specified.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(follow_symlinks, bool):
            raise TypeError("follow_symlinks can only be an instance of type bool")
        exists = self._call("directoryExists", in_p=[path, follow_symlinks])
        return exists

    def directory_open(self, path, filter_p, flags):
        """Opens a directory in the guest and creates a :py:class:`IGuestDirectory`
        object that can be used for further operations.

        This method follows symbolic links by default at the moment, this
        may change in the future.

        One idiosyncrasy of the current implementation is that you will NOT
        get VBOX_E_OBJECT_NOT_FOUND returned here if the directory doesn't exist.
        Instead the read function will fail with VBOX_E_IPRT_ERROR.  This will
        be fixed soon.

        in path of type str
            Path to the directory to open. Guest path style.

        in filter_p of type str
            Optional directory listing filter to apply.  This uses the DOS/NT
            style wildcard characters '?' and '*'.

        in flags of type :class:`DirectoryOpenFlag`
            Zero or more :py:class:`DirectoryOpenFlag`  flags.

        return directory of type :class:`IGuestDirectory`
            :py:class:`IGuestDirectory`  object containing the opened directory.

        raises :class:`VBoxErrorObjectNotFound`
            Directory to open was not found.

        raises :class:`VBoxErrorIprtError`
            Error while opening the directory.

        raises :class:`VBoxErrorMaximumReached`
            The maximum of concurrent guest directories has been reached.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(filter_p, basestring):
            raise TypeError("filter_p can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, DirectoryOpenFlag):
                raise TypeError(
                    "array can only contain objects of type DirectoryOpenFlag"
                )
        directory = self._call("directoryOpen", in_p=[path, filter_p, flags])
        directory = IGuestDirectory(directory)
        return directory

    def directory_remove(self, path):
        """Removes a guest directory if empty.

        Symbolic links in the final component will not be followed,
        instead an not-a-directory error is reported.

        in path of type str
            Path to the directory that should be removed. Guest path style.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        self._call("directoryRemove", in_p=[path])

    def directory_remove_recursive(self, path, flags):
        """Removes a guest directory recursively.

        <!--  Add this back when the warning can be removed:
        Unless :py:attr:`DirectoryRemoveRecFlag.content_and_dir`  or
        :py:attr:`DirectoryRemoveRecFlag.content_only`  is given, only the
        directory structure is removed.  Which means it will fail if there are
        directories which are not empty in the directory tree @a path points to.
        -->

        WARNING!! THE FLAGS ARE NOT CURRENTLY IMPLEMENTED.  THE IMPLEMENTATION
        WORKS AS IF FLAGS WAS SET TO :py:attr:`DirectoryRemoveRecFlag.content_and_dir` .


        If the final path component is a symbolic link, this method will
        fail as it can only be applied to directories.

        in path of type str
            Path of the directory that is to be removed recursively. Guest
            path style.

        in flags of type :class:`DirectoryRemoveRecFlag`
            Zero or more :py:class:`DirectoryRemoveRecFlag`  flags.
            WARNING! SPECIFYING :py:attr:`DirectoryRemoveRecFlag.content_and_dir`  IS
            MANDATORY AT THE MOMENT!!

        return progress of type :class:`IProgress`
            Progress object to track the operation completion. This is not implemented
            yet and therefore this method call will block until deletion is completed.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, DirectoryRemoveRecFlag):
                raise TypeError(
                    "array can only contain objects of type DirectoryRemoveRecFlag"
                )
        progress = self._call("directoryRemoveRecursive", in_p=[path, flags])
        progress = IProgress(progress)
        return progress

    def environment_schedule_set(self, name, value):
        """Schedules setting an environment variable when creating the next guest
        process.  This affects the :py:func:`IGuestSession.environment_changes`
        attribute.

        in name of type str
            Name of the environment variable to set.  This cannot be empty
            nor can it contain any equal signs.

        in value of type str
            Value to set the session environment variable to.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(value, basestring):
            raise TypeError("value can only be an instance of type basestring")
        self._call("environmentScheduleSet", in_p=[name, value])

    def environment_schedule_unset(self, name):
        """Schedules unsetting (removing) an environment variable when creating
        the next guest process.  This affects the
        :py:func:`IGuestSession.environment_changes`  attribute.

        in name of type str
            Name of the environment variable to unset.  This cannot be empty
            nor can it contain any equal signs.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        self._call("environmentScheduleUnset", in_p=[name])

    def environment_get_base_variable(self, name):
        """Gets an environment variable from the session's base environment
                (:py:func:`IGuestSession.environment_base` ).

                in name of type str
                    Name of the environment variable to   get.This cannot be empty
                    nor can it contain any equal signs.

                return value of type str
                    The value of the variable.  Empty if not found.  To deal with
                    variables that may have empty values, use
                    :py:func:`IGuestSession.environment_does_base_variable_exist` .

                raises :class:`VBoxErrorNotSupported`
                    If the Guest Additions does not
        support the session base environment feature.  Support for this was
        introduced with protocol version XXXX.

                raises :class:`VBoxErrorInvalidObjectState`
                    If the Guest Additions has
        yet to report the session base environment.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        value = self._call("environmentGetBaseVariable", in_p=[name])
        return value

    def environment_does_base_variable_exist(self, name):
        """Checks if the given environment variable exists in the session's base
                environment (:py:func:`IGuestSession.environment_base` ).

                in name of type str
                    Name of the environment variable to look for.  This cannot be
                    empty nor can it contain any equal signs.

                return exists of type bool
                    TRUE if the variable exists, FALSE if not.

                raises :class:`VBoxErrorNotSupported`
                    If the Guest Additions does not
        support the session base environment feature.  Support for this was
        introduced with protocol version XXXX.

                raises :class:`VBoxErrorInvalidObjectState`
                    If the Guest Additions has
        yet to report the session base environment.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        exists = self._call("environmentDoesBaseVariableExist", in_p=[name])
        return exists

    def file_copy(self, source, destination, flags):
        """Copies a file from one guest location to another.

        Will overwrite the destination file unless
        :py:attr:`FileCopyFlag.no_replace`  is specified.

        in source of type str
            The path to the file to copy (in the guest).  Guest path style.

        in destination of type str
            The path to the target file (in the guest).  This cannot be a
            directory.  Guest path style.

        in flags of type :class:`FileCopyFlag`
            Zero or more :py:class:`FileCopyFlag`  values.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        raises :class:`OleErrorNotimpl`
            Not yet implemented.

        """
        if not isinstance(source, basestring):
            raise TypeError("source can only be an instance of type basestring")
        if not isinstance(destination, basestring):
            raise TypeError("destination can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, FileCopyFlag):
                raise TypeError("array can only contain objects of type FileCopyFlag")
        progress = self._call("fileCopy", in_p=[source, destination, flags])
        progress = IProgress(progress)
        return progress

    def file_copy_from_guest(self, source, destination, flags):
        """Copies a file from the guest to the host.

        Will overwrite the destination file unless
        :py:attr:`FileCopyFlag.no_replace`  is specified.

        in source of type str
            Path to the file on the guest side that should be copied to the
            host.  Guest path style.

        in destination of type str
            Where to put the file on the host (file, not directory). Host
            path style.

        in flags of type :class:`FileCopyFlag`
            Zero or more :py:class:`FileCopyFlag`  values.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        raises :class:`VBoxErrorIprtError`
            Error starting the copy operation.

        """
        if not isinstance(source, basestring):
            raise TypeError("source can only be an instance of type basestring")
        if not isinstance(destination, basestring):
            raise TypeError("destination can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, FileCopyFlag):
                raise TypeError("array can only contain objects of type FileCopyFlag")
        progress = self._call("fileCopyFromGuest", in_p=[source, destination, flags])
        progress = IProgress(progress)
        return progress

    def file_copy_to_guest(self, source, destination, flags):
        """Copies a file from the host to the guest.

        Will overwrite the destination file unless
        :py:attr:`FileCopyFlag.no_replace`  is specified.

        in source of type str
            Path to the file on the host side that should be copied to the
            guest.  Host path style.

        in destination of type str
            Where to put the file in the guest (file, not directory).  Guest
            style path.

        in flags of type :class:`FileCopyFlag`
            Zero or more :py:class:`FileCopyFlag`  values.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        raises :class:`VBoxErrorIprtError`
            Error starting the copy operation.

        """
        if not isinstance(source, basestring):
            raise TypeError("source can only be an instance of type basestring")
        if not isinstance(destination, basestring):
            raise TypeError("destination can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, FileCopyFlag):
                raise TypeError("array can only contain objects of type FileCopyFlag")
        progress = self._call("fileCopyToGuest", in_p=[source, destination, flags])
        progress = IProgress(progress)
        return progress

    def file_create_temp(self, template_name, mode, path, secure):
        """Creates a temporary file in the guest.

                in template_name of type str
                    Template for the name of the file to create. This must contain
                    at least one 'X' character. The first group of consecutive 'X'
                    characters in the template will be replaced by a random
                    alphanumeric string to produce a unique name.

                in mode of type int
                    The UNIX-style access mode mask to create the file with.
                    Whether/how all three access groups and associated access rights are
                    realized is guest OS dependent.  The API does the best it can on each
                    OS.

                    This parameter is ignore if the @a secure parameter is set to @c true.
                    It is strongly recommended to use 0600.

                in path of type str
                    The path to the directory in which the temporary file should be
                    created.

                in secure of type bool
                    Whether to fail if the file can not be securely created.
                    Currently this means that another unprivileged user cannot
                    manipulate the path specified or remove the temporary file after
                    it has been created. Also causes the mode specified to be ignored.
                    May not be supported on all guest types.

                return file_p of type :class:`IGuestFile`
                    On success this will contain an open file object for the new
                    temporary file.

                raises :class:`VBoxErrorNotSupported`
                    The operation is not possible as requested on this particular
        guest OS.

                raises :class:`OleErrorInvalidarg`
                    Invalid argument. This includes an incorrectly formatted template,
        or a non-absolute path.

                raises :class:`VBoxErrorIprtError`
                    The temporary file could not be created. Possible reasons include
        a non-existing path or an insecure path when the secure
        option was requested.

        """
        if not isinstance(template_name, basestring):
            raise TypeError("template_name can only be an instance of type basestring")
        if not isinstance(mode, baseinteger):
            raise TypeError("mode can only be an instance of type baseinteger")
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(secure, bool):
            raise TypeError("secure can only be an instance of type bool")
        file_p = self._call("fileCreateTemp", in_p=[template_name, mode, path, secure])
        file_p = IGuestFile(file_p)
        return file_p

    def file_exists(self, path, follow_symlinks):
        """Checks whether a regular file exists in the guest or not.

        in path of type str
            Path to the alleged regular file.  Guest path style.

        in follow_symlinks of type bool
            If @c true, symbolic links in the final component will be followed
            and the existance of the symlink target made the question for this method.
            If @c false, a symbolic link in the final component will make the
            method return @c false (because a symlink isn't a regular file).

        return exists of type bool
            Returns @c true if the file exists, @c false if not.  @c false is
            also return if this @a path does not point to a file object.

        raises :class:`VBoxErrorIprtError`
            Error while checking existence of the file specified.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(follow_symlinks, bool):
            raise TypeError("follow_symlinks can only be an instance of type bool")
        exists = self._call("fileExists", in_p=[path, follow_symlinks])
        return exists

    def file_open(self, path, access_mode, open_action, creation_mode):
        """Opens a file and creates a :py:class:`IGuestFile`  object that
        can be used for further operations.

        in path of type str
            Path to file to open.  Guest path style.

        in access_mode of type :class:`FileAccessMode`
            The file access mode (read, write and/or append).
            See :py:class:`FileAccessMode`  for details.

        in open_action of type :class:`FileOpenAction`
            What action to take depending on whether the file exists or not.
            See :py:class:`FileOpenAction`  for details.

        in creation_mode of type int
            The UNIX-style access mode mask to create the file with if @a openAction
            requested the file to be created (otherwise ignored).  Whether/how all
            three access groups and associated access rights are realized is guest
            OS dependent.  The API does the best it can on each OS.

        return file_p of type :class:`IGuestFile`
            :py:class:`IGuestFile`  object representing the opened file.

        raises :class:`VBoxErrorObjectNotFound`
            File to open was not found.

        raises :class:`VBoxErrorIprtError`
            Error while opening the file.

        raises :class:`VBoxErrorMaximumReached`
            The maximum of concurrent guest files has been reached.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(access_mode, FileAccessMode):
            raise TypeError(
                "access_mode can only be an instance of type FileAccessMode"
            )
        if not isinstance(open_action, FileOpenAction):
            raise TypeError(
                "open_action can only be an instance of type FileOpenAction"
            )
        if not isinstance(creation_mode, baseinteger):
            raise TypeError("creation_mode can only be an instance of type baseinteger")
        file_p = self._call(
            "fileOpen", in_p=[path, access_mode, open_action, creation_mode]
        )
        file_p = IGuestFile(file_p)
        return file_p

    def file_open_ex(
        self, path, access_mode, open_action, sharing_mode, creation_mode, flags
    ):
        """Opens a file and creates a :py:class:`IGuestFile`  object that
        can be used for further operations, extended version.

        in path of type str
            Path to file to open.  Guest path style.

        in access_mode of type :class:`FileAccessMode`
            The file access mode (read, write and/or append).
            See :py:class:`FileAccessMode`  for details.

        in open_action of type :class:`FileOpenAction`
            What action to take depending on whether the file exists or not.
            See :py:class:`FileOpenAction`  for details.

        in sharing_mode of type :class:`FileSharingMode`
            The file sharing mode in the guest. This parameter is currently
            ignore for all guest OSes.  It will in the future be implemented for
            Windows, OS/2 and maybe Solaris guests only, the others will ignore it.
            Use :py:attr:`FileSharingMode.all_p` .

        in creation_mode of type int
            The UNIX-style access mode mask to create the file with if @a openAction
            requested the file to be created (otherwise ignored).  Whether/how all
            three access groups and associated access rights are realized is guest
            OS dependent.  The API does the best it can on each OS.

        in flags of type :class:`FileOpenExFlag`
            Zero or more :py:class:`FileOpenExFlag`  values.

        return file_p of type :class:`IGuestFile`
            :py:class:`IGuestFile`  object representing the opened file.

        raises :class:`VBoxErrorObjectNotFound`
            File to open was not found.

        raises :class:`VBoxErrorIprtError`
            Error while opening the file.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(access_mode, FileAccessMode):
            raise TypeError(
                "access_mode can only be an instance of type FileAccessMode"
            )
        if not isinstance(open_action, FileOpenAction):
            raise TypeError(
                "open_action can only be an instance of type FileOpenAction"
            )
        if not isinstance(sharing_mode, FileSharingMode):
            raise TypeError(
                "sharing_mode can only be an instance of type FileSharingMode"
            )
        if not isinstance(creation_mode, baseinteger):
            raise TypeError("creation_mode can only be an instance of type baseinteger")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, FileOpenExFlag):
                raise TypeError("array can only contain objects of type FileOpenExFlag")
        file_p = self._call(
            "fileOpenEx",
            in_p=[path, access_mode, open_action, sharing_mode, creation_mode, flags],
        )
        file_p = IGuestFile(file_p)
        return file_p

    def file_query_size(self, path, follow_symlinks):
        """Queries the size of a regular file in the guest.

        in path of type str
            Path to the file which size is requested.  Guest path style.

        in follow_symlinks of type bool
            It @c true, symbolic links in the final path component will be
            followed to their target, and the size of the target is returned.
            If @c false, symbolic links in the final path component will make
            the method call fail (symblink is not a regular file).

        return size of type int
            Queried file size.

        raises :class:`VBoxErrorObjectNotFound`
            File to was not found.

        raises :class:`VBoxErrorIprtError`
            Error querying file size.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(follow_symlinks, bool):
            raise TypeError("follow_symlinks can only be an instance of type bool")
        size = self._call("fileQuerySize", in_p=[path, follow_symlinks])
        return size

    def fs_obj_exists(self, path, follow_symlinks):
        """Checks whether a file system object (file, directory, etc) exists in
        the guest or not.

        in path of type str
            Path to the file system object to check the existance of.  Guest
            path style.

        in follow_symlinks of type bool
            If @c true, symbolic links in the final component will be followed
            and the method will instead check if the target exists.
            If @c false, symbolic links in the final component will satisfy the
            method and it will return @c true in @a exists.

        return exists of type bool
            Returns @c true if the file exists, @c false if not.

        raises :class:`VBoxErrorIprtError`
            Error while checking existence of the file specified.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(follow_symlinks, bool):
            raise TypeError("follow_symlinks can only be an instance of type bool")
        exists = self._call("fsObjExists", in_p=[path, follow_symlinks])
        return exists

    def fs_obj_query_info(self, path, follow_symlinks):
        """Queries information about a file system object (file, directory, etc)
        in the guest.

        in path of type str
            Path to the file system object to gather information about.
            Guest path style.

        in follow_symlinks of type bool
            Information about symbolic links is returned if @c false.  Otherwise,
            symbolic links are followed and the returned information concerns
            itself with the symlink target if @c true.

        return info of type :class:`IGuestFsObjInfo`
            :py:class:`IGuestFsObjInfo`  object containing the information.

        raises :class:`VBoxErrorObjectNotFound`
            The file system object was not found.

        raises :class:`VBoxErrorIprtError`
            Error while querying information.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(follow_symlinks, bool):
            raise TypeError("follow_symlinks can only be an instance of type bool")
        info = self._call("fsObjQueryInfo", in_p=[path, follow_symlinks])
        info = IGuestFsObjInfo(info)
        return info

    def fs_obj_remove(self, path):
        """Removes a file system object (file, symlink, etc) in the guest.  Will
        not work on directories, use :py:func:`IGuestSession.directory_remove`
        to remove directories.

        This method will remove symbolic links in the final path
        component, not follow them.

        in path of type str
            Path to the file system object to remove.  Guest style path.

        raises :class:`OleErrorNotimpl`
            The method has not been implemented yet.

        raises :class:`VBoxErrorObjectNotFound`
            The file system object was not found.

        raises :class:`VBoxErrorIprtError`
            For most other errors. We know this is unhelpful, will fix shortly...

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        self._call("fsObjRemove", in_p=[path])

    def fs_obj_remove_array(self, path):
        """Removes multiple file system objects (files, directories, symlinks, etc)
        in the guest. Use with caution.

        This method is not implemented yet and will return E_NOTIMPL.

        This method will remove symbolic links in the final path
        component, not follow them.

        in path of type str
            Array of paths to the file system objects to remove.  Guest style path.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        raises :class:`OleErrorNotimpl`
            The method has not been implemented yet.

        """
        if not isinstance(path, list):
            raise TypeError("path can only be an instance of type list")
        for a in path[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        progress = self._call("fsObjRemoveArray", in_p=[path])
        progress = IProgress(progress)
        return progress

    def fs_obj_rename(self, old_path, new_path, flags):
        """Renames a file system object (file, directory, symlink, etc) in the
        guest.

        in old_path of type str
            The current path to the object.  Guest path style.

        in new_path of type str
            The new path to the object.  Guest path style.

        in flags of type :class:`FsObjRenameFlag`
            Zero or more :py:class:`FsObjRenameFlag`  values.

        raises :class:`VBoxErrorObjectNotFound`
            The file system object was not found.

        raises :class:`VBoxErrorIprtError`
            For most other errors. We know this is unhelpful, will fix shortly...

        """
        if not isinstance(old_path, basestring):
            raise TypeError("old_path can only be an instance of type basestring")
        if not isinstance(new_path, basestring):
            raise TypeError("new_path can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, FsObjRenameFlag):
                raise TypeError(
                    "array can only contain objects of type FsObjRenameFlag"
                )
        self._call("fsObjRename", in_p=[old_path, new_path, flags])

    def fs_obj_move(self, source, destination, flags):
        """Moves a file system object (file, directory, symlink, etc) from one
        guest location to another.

        This differs from :py:func:`IGuestSession.fs_obj_rename`  in that it
        can move accross file system boundraries.  In that case it will
        perform a copy and then delete the original.  For directories, this
        can take a while and is subject to races.

        in source of type str
            Path to the file to move.  Guest path style.

        in destination of type str
            Where to move the file to (file, not directory).  Guest path
            style.

        in flags of type :class:`FsObjMoveFlag`
            Zero or more :py:class:`FsObjMoveFlag`  values.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        raises :class:`OleErrorNotimpl`
            Not yet implemented.

        """
        if not isinstance(source, basestring):
            raise TypeError("source can only be an instance of type basestring")
        if not isinstance(destination, basestring):
            raise TypeError("destination can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, FsObjMoveFlag):
                raise TypeError("array can only contain objects of type FsObjMoveFlag")
        progress = self._call("fsObjMove", in_p=[source, destination, flags])
        progress = IProgress(progress)
        return progress

    def fs_obj_move_array(self, source, destination, flags):
        """Moves file system objects (files, directories, symlinks, etc) from one
        guest location to another.

        in source of type str
            Array of paths to the file system objects to move.  Guest style path.

        in destination of type str
            Where to move the file system objects to (directory).  Guest path
            style.

        in flags of type :class:`FsObjMoveFlag`
            Zero or more :py:class:`FsObjMoveFlag`  values.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        raises :class:`OleErrorNotimpl`
            Not yet implemented.

        """
        if not isinstance(source, list):
            raise TypeError("source can only be an instance of type list")
        for a in source[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(destination, basestring):
            raise TypeError("destination can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, FsObjMoveFlag):
                raise TypeError("array can only contain objects of type FsObjMoveFlag")
        progress = self._call("fsObjMoveArray", in_p=[source, destination, flags])
        progress = IProgress(progress)
        return progress

    def fs_obj_copy_array(self, source, destination, flags):
        """Copies file system objects (files, directories, symlinks, etc) from one
        guest location to another.

        in source of type str
            Array of paths to the file system objects to copy.  Guest style path.

        in destination of type str
            Where to copy the file system objects to (directory).  Guest path
            style.

        in flags of type :class:`FileCopyFlag`
            Zero or more :py:class:`FileCopyFlag`  values.

        return progress of type :class:`IProgress`
            Progress object to track the operation to completion.

        raises :class:`OleErrorNotimpl`
            Not yet implemented.

        """
        if not isinstance(source, list):
            raise TypeError("source can only be an instance of type list")
        for a in source[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(destination, basestring):
            raise TypeError("destination can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, FileCopyFlag):
                raise TypeError("array can only contain objects of type FileCopyFlag")
        progress = self._call("fsObjCopyArray", in_p=[source, destination, flags])
        progress = IProgress(progress)
        return progress

    def fs_obj_set_acl(self, path, follow_symlinks, acl, mode):
        """Sets the access control list (ACL) of a file system object (file,
        directory, etc) in the guest.

        in path of type str
            Full path of the file system object which ACL to set

        in follow_symlinks of type bool
            If @c true symbolic links in the final component will be followed,
            otherwise, if @c false, the method will work directly on a symbolic
            link in the final component.

        in acl of type str
            The ACL specification string. To-be-defined.

        in mode of type int
            UNIX-style mode mask to use if @a acl is empty. As mention in
            :py:func:`IGuestSession.directory_create`  this is realized on
            a best effort basis and the exact behavior depends on the Guest OS.

        raises :class:`OleErrorNotimpl`
            The method is not implemented yet.

        """
        if not isinstance(path, basestring):
            raise TypeError("path can only be an instance of type basestring")
        if not isinstance(follow_symlinks, bool):
            raise TypeError("follow_symlinks can only be an instance of type bool")
        if not isinstance(acl, basestring):
            raise TypeError("acl can only be an instance of type basestring")
        if not isinstance(mode, baseinteger):
            raise TypeError("mode can only be an instance of type baseinteger")
        self._call("fsObjSetACL", in_p=[path, follow_symlinks, acl, mode])

    def process_create(
        self, executable, arguments, environment_changes, flags, timeout_ms
    ):
        """Creates a new process running in the guest. The new process will be
        started asynchronously, meaning on return of this function it is not
        be guaranteed that the guest process is in a started state. To wait for
        successful startup, use the :py:func:`IProcess.wait_for`  call.


        Starting at VirtualBox 4.2 guest process execution by is default limited
        to serve up to 255 guest processes at a time. If all 255 guest processes
        are active and running, creating a new guest process will result in an
        error.

        If ProcessCreateFlag_WaitForStdOut and/or ProcessCreateFlag_WaitForStdErr
        are set, the guest process will not enter the terminated state until
        all data from the specified streams have been read read.

        in executable of type str
            Full path to the file to execute in the guest.  The file has to
            exists in the guest VM with executable right to the session user in
            order to succeed.  If empty/null, the first entry in the
            @a arguments array will be used instead (i.e. argv[0]).

        in arguments of type str
            Array of arguments passed to the new process.

            Starting with VirtualBox 5.0 this array starts with argument 0
            instead of argument 1 as in previous versions.  Whether the zeroth
            argument can be passed to the guest depends on the VBoxService
            version running there.  If you depend on this, check that the
            :py:func:`IGuestSession.protocol_version`  is 3 or higher.

        in environment_changes of type str
            Set of environment changes to complement
            :py:func:`IGuestSession.environment_changes` .  Takes precedence
            over the session ones.  The changes are in putenv format, i.e.
            "VAR=VALUE" for setting and "VAR" for unsetting.

            The changes are applied to the base environment of the impersonated
            guest user (:py:func:`IGuestSession.environment_base` ) when
            creating the process.  (This is done on the guest side of things in
            order to be compatible with older Guest Additions.  That is one of
            the motivations for not passing in the whole environment here.)

        in flags of type :class:`ProcessCreateFlag`
            Process creation flags;
            see :py:class:`ProcessCreateFlag`  for more information.

        in timeout_ms of type int
            Timeout (in ms) for limiting the guest process' running time.
            Pass 0 for an infinite timeout. On timeout the guest process will be
            killed and its status will be put to an appropriate value. See
            :py:class:`ProcessStatus`  for more information.

        return guest_process of type :class:`IGuestProcess`
            Guest process object of the newly created process.

        raises :class:`VBoxErrorIprtError`
            Error creating guest process.

        raises :class:`VBoxErrorMaximumReached`
            The maximum of concurrent guest processes has been reached.

        """
        if not isinstance(executable, basestring):
            raise TypeError("executable can only be an instance of type basestring")
        if not isinstance(arguments, list):
            raise TypeError("arguments can only be an instance of type list")
        for a in arguments[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(environment_changes, list):
            raise TypeError("environment_changes can only be an instance of type list")
        for a in environment_changes[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, ProcessCreateFlag):
                raise TypeError(
                    "array can only contain objects of type ProcessCreateFlag"
                )
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        guest_process = self._call(
            "processCreate",
            in_p=[executable, arguments, environment_changes, flags, timeout_ms],
        )
        guest_process = IGuestProcess(guest_process)
        return guest_process

    def process_create_ex(
        self,
        executable,
        arguments,
        environment_changes,
        flags,
        timeout_ms,
        priority,
        affinity,
    ):
        """Creates a new process running in the guest with the extended options
        for setting the process priority and affinity.

        See :py:func:`IGuestSession.process_create`  for more information.

        in executable of type str
            Full path to the file to execute in the guest.  The file has to
            exists in the guest VM with executable right to the session user in
            order to succeed.  If empty/null, the first entry in the
            @a arguments array will be used instead (i.e. argv[0]).

        in arguments of type str
            Array of arguments passed to the new process.

            Starting with VirtualBox 5.0 this array starts with argument 0
            instead of argument 1 as in previous versions.  Whether the zeroth
            argument can be passed to the guest depends on the VBoxService
            version running there.  If you depend on this, check that the
            :py:func:`IGuestSession.protocol_version`  is 3 or higher.

        in environment_changes of type str
            Set of environment changes to complement
            :py:func:`IGuestSession.environment_changes` .  Takes precedence
            over the session ones.  The changes are in putenv format, i.e.
            "VAR=VALUE" for setting and "VAR" for unsetting.

            The changes are applied to the base environment of the impersonated
            guest user (:py:func:`IGuestSession.environment_base` ) when
            creating the process.  (This is done on the guest side of things in
            order to be compatible with older Guest Additions.  That is one of
            the motivations for not passing in the whole environment here.)

        in flags of type :class:`ProcessCreateFlag`
            Process creation flags, see :py:class:`ProcessCreateFlag`  for
            detailed description of available flags.

        in timeout_ms of type int
            Timeout (in ms) for limiting the guest process' running time.
            Pass 0 for an infinite timeout. On timeout the guest process will be
            killed and its status will be put to an appropriate value. See
            :py:class:`ProcessStatus`  for more information.

        in priority of type :class:`ProcessPriority`
            Process priority to use for execution, see :py:class:`ProcessPriority`
            for available priority levels.
            This is silently ignored if not supported by Guest Additions.

        in affinity of type int
            Processor affinity to set for the new process.  This is a list of
            guest CPU numbers the process is allowed to run on.

            This is silently ignored if the guest does not support setting the
            affinity of processes, or if the Guest Additions does not implemet
            this feature.

        return guest_process of type :class:`IGuestProcess`
            Guest process object of the newly created process.

        """
        if not isinstance(executable, basestring):
            raise TypeError("executable can only be an instance of type basestring")
        if not isinstance(arguments, list):
            raise TypeError("arguments can only be an instance of type list")
        for a in arguments[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(environment_changes, list):
            raise TypeError("environment_changes can only be an instance of type list")
        for a in environment_changes[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, ProcessCreateFlag):
                raise TypeError(
                    "array can only contain objects of type ProcessCreateFlag"
                )
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        if not isinstance(priority, ProcessPriority):
            raise TypeError("priority can only be an instance of type ProcessPriority")
        if not isinstance(affinity, list):
            raise TypeError("affinity can only be an instance of type list")
        for a in affinity[:10]:
            if not isinstance(a, baseinteger):
                raise TypeError("array can only contain objects of type baseinteger")
        guest_process = self._call(
            "processCreateEx",
            in_p=[
                executable,
                arguments,
                environment_changes,
                flags,
                timeout_ms,
                priority,
                affinity,
            ],
        )
        guest_process = IGuestProcess(guest_process)
        return guest_process

    def process_get(self, pid):
        """Gets a certain guest process by its process ID (PID).

        in pid of type int
            Process ID (PID) to get guest process for.

        return guest_process of type :class:`IGuestProcess`
            Guest process of specified process ID (PID).

        """
        if not isinstance(pid, baseinteger):
            raise TypeError("pid can only be an instance of type baseinteger")
        guest_process = self._call("processGet", in_p=[pid])
        guest_process = IGuestProcess(guest_process)
        return guest_process

    def symlink_create(self, symlink, target, type_p):
        """Creates a symbolic link in the guest.

        in symlink of type str
            Path to the symbolic link that should be created.  Guest path
            style.

        in target of type str
            The path to the symbolic link target.  If not an absolute, this will
            be relative to the @a symlink location at access time.  Guest path
            style.

        in type_p of type :class:`SymlinkType`
            The symbolic link type (mainly for Windows). See :py:class:`SymlinkType`
            for more information.

        raises :class:`OleErrorNotimpl`
            The method is not implemented yet.

        """
        if not isinstance(symlink, basestring):
            raise TypeError("symlink can only be an instance of type basestring")
        if not isinstance(target, basestring):
            raise TypeError("target can only be an instance of type basestring")
        if not isinstance(type_p, SymlinkType):
            raise TypeError("type_p can only be an instance of type SymlinkType")
        self._call("symlinkCreate", in_p=[symlink, target, type_p])

    def symlink_exists(self, symlink):
        """Checks whether a symbolic link exists in the guest.

        in symlink of type str
            Path to the alleged symbolic link.  Guest path style.

        return exists of type bool
            Returns @c true if the symbolic link exists.  Returns @c false if it
            does not exist, if the file system object identified by the path is
            not a symbolic link, or if the object type is inaccessible to the
            user, or if the @a symlink argument is empty.

        raises :class:`OleErrorNotimpl`
            The method is not implemented yet.

        """
        if not isinstance(symlink, basestring):
            raise TypeError("symlink can only be an instance of type basestring")
        exists = self._call("symlinkExists", in_p=[symlink])
        return exists

    def symlink_read(self, symlink, flags):
        """Reads the target value of a symbolic link in the guest.

        in symlink of type str
            Path to the symbolic link to read.

        in flags of type :class:`SymlinkReadFlag`
            Zero or more :py:class:`SymlinkReadFlag`  values.

        return target of type str
            Target value of the symbolic link.  Guest path style.

        raises :class:`OleErrorNotimpl`
            The method is not implemented yet.

        """
        if not isinstance(symlink, basestring):
            raise TypeError("symlink can only be an instance of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, SymlinkReadFlag):
                raise TypeError(
                    "array can only contain objects of type SymlinkReadFlag"
                )
        target = self._call("symlinkRead", in_p=[symlink, flags])
        return target

    def wait_for(self, wait_for, timeout_ms):
        """Waits for one or more events to happen.

        in wait_for of type int
            Specifies what to wait for;
            see :py:class:`GuestSessionWaitForFlag`  for more information.

        in timeout_ms of type int
            Timeout (in ms) to wait for the operation to complete.
            Pass 0 for an infinite timeout.

        return reason of type :class:`GuestSessionWaitResult`
            The overall wait result;
            see :py:class:`GuestSessionWaitResult`  for more information.

        """
        if not isinstance(wait_for, baseinteger):
            raise TypeError("wait_for can only be an instance of type baseinteger")
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        reason = self._call("waitFor", in_p=[wait_for, timeout_ms])
        reason = GuestSessionWaitResult(reason)
        return reason

    def wait_for_array(self, wait_for, timeout_ms):
        """Waits for one or more events to happen.
        Scriptable version of :py:func:`wait_for` .

        in wait_for of type :class:`GuestSessionWaitForFlag`
            Specifies what to wait for;
            see :py:class:`GuestSessionWaitForFlag`  for more information.

        in timeout_ms of type int
            Timeout (in ms) to wait for the operation to complete.
            Pass 0 for an infinite timeout.

        return reason of type :class:`GuestSessionWaitResult`
            The overall wait result;
            see :py:class:`GuestSessionWaitResult`  for more information.

        """
        if not isinstance(wait_for, list):
            raise TypeError("wait_for can only be an instance of type list")
        for a in wait_for[:10]:
            if not isinstance(a, GuestSessionWaitForFlag):
                raise TypeError(
                    "array can only contain objects of type GuestSessionWaitForFlag"
                )
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        reason = self._call("waitForArray", in_p=[wait_for, timeout_ms])
        reason = GuestSessionWaitResult(reason)
        return reason


class IProcess(Interface):
    """
    Abstract parent interface for processes handled by VirtualBox.
    """

    __uuid__ = "bc68370c-8a02-45f3-a07d-a67aa72756aa"
    __wsmap__ = "managed"

    @property
    def arguments(self):
        """Get str value for 'arguments'
        The arguments this process is using for execution.
        """
        ret = self._get_attr("arguments")
        return ret

    @property
    def environment(self):
        """Get str value for 'environment'
        The initial process environment.  Not yet implemented.
        """
        ret = self._get_attr("environment")
        return ret

    @property
    def event_source(self):
        """Get IEventSource value for 'eventSource'
        Event source for process events.
        """
        ret = self._get_attr("eventSource")
        return IEventSource(ret)

    @property
    def executable_path(self):
        """Get str value for 'executablePath'
        Full path of the actual executable image.
        """
        ret = self._get_attr("executablePath")
        return ret

    @property
    def exit_code(self):
        """Get int value for 'exitCode'
        The exit code. Only available when the process has been
        terminated normally.
        """
        ret = self._get_attr("exitCode")
        return ret

    @property
    def name(self):
        """Get str value for 'name'
        The friendly name of this process.
        """
        ret = self._get_attr("name")
        return ret

    @property
    def pid(self):
        """Get int value for 'PID'
        The process ID (PID).
        """
        ret = self._get_attr("PID")
        return ret

    @property
    def status(self):
        """Get ProcessStatus value for 'status'
        The current process status; see :py:class:`ProcessStatus`
        for more information.
        """
        ret = self._get_attr("status")
        return ProcessStatus(ret)

    def wait_for(self, wait_for, timeout_ms):
        """Waits for one or more events to happen.

        in wait_for of type int
            Specifies what to wait for;
            see :py:class:`ProcessWaitForFlag`  for more information.

        in timeout_ms of type int
            Timeout (in ms) to wait for the operation to complete.
            Pass 0 for an infinite timeout.

        return reason of type :class:`ProcessWaitResult`
            The overall wait result;
            see :py:class:`ProcessWaitResult`  for more information.

        """
        if not isinstance(wait_for, baseinteger):
            raise TypeError("wait_for can only be an instance of type baseinteger")
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        reason = self._call("waitFor", in_p=[wait_for, timeout_ms])
        reason = ProcessWaitResult(reason)
        return reason

    def wait_for_array(self, wait_for, timeout_ms):
        """Waits for one or more events to happen.
        Scriptable version of :py:func:`wait_for` .

        in wait_for of type :class:`ProcessWaitForFlag`
            Specifies what to wait for;
            see :py:class:`ProcessWaitForFlag`  for more information.

        in timeout_ms of type int
            Timeout (in ms) to wait for the operation to complete.
            Pass 0 for an infinite timeout.

        return reason of type :class:`ProcessWaitResult`
            The overall wait result;
            see :py:class:`ProcessWaitResult`  for more information.

        """
        if not isinstance(wait_for, list):
            raise TypeError("wait_for can only be an instance of type list")
        for a in wait_for[:10]:
            if not isinstance(a, ProcessWaitForFlag):
                raise TypeError(
                    "array can only contain objects of type ProcessWaitForFlag"
                )
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        reason = self._call("waitForArray", in_p=[wait_for, timeout_ms])
        reason = ProcessWaitResult(reason)
        return reason

    def read(self, handle, to_read, timeout_ms):
        """Reads data from a running process.

        in handle of type int
            Handle to read from. Usually 0 is stdin.

        in to_read of type int
            Number of bytes to read.

        in timeout_ms of type int
            Timeout (in ms) to wait for the operation to complete.
            Pass 0 for an infinite timeout.

        return data of type str
            Array of data read.

        """
        if not isinstance(handle, baseinteger):
            raise TypeError("handle can only be an instance of type baseinteger")
        if not isinstance(to_read, baseinteger):
            raise TypeError("to_read can only be an instance of type baseinteger")
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        data = self._call("read", in_p=[handle, to_read, timeout_ms])
        return data

    def write(self, handle, flags, data, timeout_ms):
        """Writes data to a running process.

        in handle of type int
            Handle to write to. Usually 0 is stdin, 1 is stdout and 2 is stderr.

        in flags of type int
            A combination of :py:class:`ProcessInputFlag`  flags.

        in data of type str
            Array of bytes to write. The size of the array also specifies
            how much to write.

        in timeout_ms of type int
            Timeout (in ms) to wait for the operation to complete.
            Pass 0 for an infinite timeout.

        return written of type int
            How many bytes were written.

        """
        if not isinstance(handle, baseinteger):
            raise TypeError("handle can only be an instance of type baseinteger")
        if not isinstance(flags, baseinteger):
            raise TypeError("flags can only be an instance of type baseinteger")
        if not isinstance(data, list):
            raise TypeError("data can only be an instance of type list")
        for a in data[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        written = self._call("write", in_p=[handle, flags, data, timeout_ms])
        return written

    def write_array(self, handle, flags, data, timeout_ms):
        """Writes data to a running process.
        Scriptable version of :py:func:`write` .

        in handle of type int
            Handle to write to. Usually 0 is stdin, 1 is stdout and 2 is stderr.

        in flags of type :class:`ProcessInputFlag`
            A combination of :py:class:`ProcessInputFlag`  flags.

        in data of type str
            Array of bytes to write. The size of the array also specifies
            how much to write.

        in timeout_ms of type int
            Timeout (in ms) to wait for the operation to complete.
            Pass 0 for an infinite timeout.

        return written of type int
            How may bytes were written.

        """
        if not isinstance(handle, baseinteger):
            raise TypeError("handle can only be an instance of type baseinteger")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, ProcessInputFlag):
                raise TypeError(
                    "array can only contain objects of type ProcessInputFlag"
                )
        if not isinstance(data, list):
            raise TypeError("data can only be an instance of type list")
        for a in data[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        written = self._call("writeArray", in_p=[handle, flags, data, timeout_ms])
        return written

    def terminate(self):
        """Terminates (kills) a running process.
        It can take up to 30 seconds to get a guest process killed. In
        case a guest process could not be killed an appropriate error is
        returned.

        """
        self._call("terminate")


class IGuestProcess(IProcess):
    """
    Implementation of the :py:class:`IProcess`  object
    for processes the host has started in the guest.
    """

    __uuid__ = "35cf4b3f-4453-4f3e-c9b8-5686939c80b6"
    __wsmap__ = "managed"

    @property
    def midl_does_not_like_empty_interfaces(self):
        """Get bool value for 'midlDoesNotLikeEmptyInterfaces'"""
        ret = self._get_attr("midlDoesNotLikeEmptyInterfaces")
        return ret


class IDirectory(Interface):
    """
    Abstract parent interface for directories handled by VirtualBox.
    """

    __uuid__ = "758d7eac-e4b1-486a-8f2e-747ae346c3e9"
    __wsmap__ = "managed"

    @property
    def directory_name(self):
        """Get str value for 'directoryName'
        The path specified when opening the directory.
        """
        ret = self._get_attr("directoryName")
        return ret

    @property
    def filter_p(self):
        """Get str value for 'filter'
        Directory listing filter to (specified when opening the directory).
        """
        ret = self._get_attr("filter")
        return ret

    def close(self):
        """Closes this directory. After closing operations like reading the next
        directory entry will not be possible anymore.

        """
        self._call("close")

    def read(self):
        """Reads the next directory entry of this directory.

        return obj_info of type :class:`IFsObjInfo`
            Object information of the current directory entry read. Also see
            :py:class:`IFsObjInfo` .

        raises :class:`VBoxErrorObjectNotFound`
            No more directory entries to read.

        """
        obj_info = self._call("read")
        obj_info = IFsObjInfo(obj_info)
        return obj_info


class IGuestDirectory(IDirectory):
    """
    Implementation of the :py:class:`IDirectory`  object
    for directories in the guest.
    """

    __uuid__ = "cc830458-4974-a19c-4dc6-cc98c2269626"
    __wsmap__ = "managed"

    @property
    def midl_does_not_like_empty_interfaces(self):
        """Get bool value for 'midlDoesNotLikeEmptyInterfaces'"""
        ret = self._get_attr("midlDoesNotLikeEmptyInterfaces")
        return ret


class IFile(Interface):
    """
    Abstract parent interface for files handled by VirtualBox.
    """

    __uuid__ = "59a235ac-2f1a-4d6c-81fc-e3fa843f49ae"
    __wsmap__ = "managed"

    @property
    def event_source(self):
        """Get IEventSource value for 'eventSource'
        Event source for file events.
        """
        ret = self._get_attr("eventSource")
        return IEventSource(ret)

    @property
    def id_p(self):
        """Get int value for 'id'
        The ID VirtualBox internally assigned to the open file.
        """
        ret = self._get_attr("id")
        return ret

    @property
    def initial_size(self):
        """Get int value for 'initialSize'
        The initial size in bytes when opened.
        """
        ret = self._get_attr("initialSize")
        return ret

    @property
    def offset(self):
        """Get int value for 'offset'
        The current file position.

        The file current position always applies to the :py:func:`IFile.read`
        method, which updates it upon return.  Same goes for the :py:func:`IFile.write`
        method except when :py:func:`IFile.access_mode`  is :py:attr:`FileAccessMode.append_only`
        or :py:attr:`FileAccessMode.append_read` , where it will always write
        to the end of the file and will leave this attribute unchanged.

        The :py:func:`IFile.seek`  is used to change this attribute without
        transfering any file data like read and write does.

        This will not always be correct with older Guest Additions
        (version 5.2.30 and earlier, as well as versions 6.0.0 thru 6.0.8)
        after a calling :py:func:`IFile.read_at`  or :py:func:`IFile.write_at` ,
        or after calling :py:func:`IFile.write`  on a file in append mode.
        The correct file offset can be obtained using :py:func:`IFile.seek` .
        """
        ret = self._get_attr("offset")
        return ret

    @property
    def status(self):
        """Get FileStatus value for 'status'
        Current file status.
        """
        ret = self._get_attr("status")
        return FileStatus(ret)

    @property
    def filename(self):
        """Get str value for 'filename'
        Full path of the actual file name of this file.
        <!-- r=bird: The 'actual' file name is too tough, we cannot guarentee
        that on unix guests.  Seeing how IGuestDirectory did things,
        I'm questioning the 'Full path' part too.   Not urgent to check. -->
        """
        ret = self._get_attr("filename")
        return ret

    @property
    def creation_mode(self):
        """Get int value for 'creationMode'
        The UNIX-style creation mode specified when opening the file.
        """
        ret = self._get_attr("creationMode")
        return ret

    @property
    def open_action(self):
        """Get FileOpenAction value for 'openAction'
        The opening action specified when opening the file.
        """
        ret = self._get_attr("openAction")
        return FileOpenAction(ret)

    @property
    def access_mode(self):
        """Get FileAccessMode value for 'accessMode'
        The file access mode.
        """
        ret = self._get_attr("accessMode")
        return FileAccessMode(ret)

    def close(self):
        """Closes this file. After closing operations like reading data,
        writing data or querying information will not be possible anymore.

        """
        self._call("close")

    def query_info(self):
        """Queries information about this file.

        return obj_info of type :class:`IFsObjInfo`
            Object information of this file. Also see
            :py:class:`IFsObjInfo` .

        """
        obj_info = self._call("queryInfo")
        obj_info = IFsObjInfo(obj_info)
        return obj_info

    def query_size(self):
        """Queries the current file size.

        return size of type int
            Queried file size.

        """
        size = self._call("querySize")
        return size

    def read(self, to_read, timeout_ms):
        """Reads data from this file.

        The file current position (:py:func:`IFile.offset` ) is updated on success.

        in to_read of type int
            Number of bytes to read.

        in timeout_ms of type int
            Timeout (in ms) to wait for the operation to complete.
            Pass 0 for an infinite timeout.

        return data of type str
            Array of data read.

        """
        if not isinstance(to_read, baseinteger):
            raise TypeError("to_read can only be an instance of type baseinteger")
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        data = self._call("read", in_p=[to_read, timeout_ms])
        return data

    def read_at(self, offset, to_read, timeout_ms):
        """Reads data from an offset of this file.

        The file current position (:py:func:`IFile.offset` ) is updated on success.

        in offset of type int
            Offset in bytes to start reading.

        in to_read of type int
            Number of bytes to read.

        in timeout_ms of type int
            Timeout (in ms) to wait for the operation to complete.
            Pass 0 for an infinite timeout.

        return data of type str
            Array of data read.

        """
        if not isinstance(offset, baseinteger):
            raise TypeError("offset can only be an instance of type baseinteger")
        if not isinstance(to_read, baseinteger):
            raise TypeError("to_read can only be an instance of type baseinteger")
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        data = self._call("readAt", in_p=[offset, to_read, timeout_ms])
        return data

    def seek(self, offset, whence):
        """Changes the current file position of this file.

        The file current position always applies to the :py:func:`IFile.read`
        method.  Same for the :py:func:`IFile.write`  method it except when
        the :py:func:`IFile.access_mode`  is :py:attr:`FileAccessMode.append_only`
        or :py:attr:`FileAccessMode.append_read` .

        in offset of type int
            Offset to seek relative to the position specified by @a whence.

        in whence of type :class:`FileSeekOrigin`
            One of the :py:class:`FileSeekOrigin`  seek starting points.

        return new_offset of type int
            The new file offset after the seek operation.

        """
        if not isinstance(offset, baseinteger):
            raise TypeError("offset can only be an instance of type baseinteger")
        if not isinstance(whence, FileSeekOrigin):
            raise TypeError("whence can only be an instance of type FileSeekOrigin")
        new_offset = self._call("seek", in_p=[offset, whence])
        return new_offset

    def set_acl(self, acl, mode):
        """Sets the ACL of this file.

        in acl of type str
            The ACL specification string. To-be-defined.

        in mode of type int
            UNIX-style mode mask to use if @a acl is empty. As mention in
            :py:func:`IGuestSession.directory_create`  this is realized on
            a best effort basis and the exact behavior depends on the Guest OS.

        raises :class:`OleErrorNotimpl`
            The method is not implemented yet.

        """
        if not isinstance(acl, basestring):
            raise TypeError("acl can only be an instance of type basestring")
        if not isinstance(mode, baseinteger):
            raise TypeError("mode can only be an instance of type baseinteger")
        self._call("setACL", in_p=[acl, mode])

    def set_size(self, size):
        """Changes the file size.

        in size of type int
            The new file size.

        """
        if not isinstance(size, baseinteger):
            raise TypeError("size can only be an instance of type baseinteger")
        self._call("setSize", in_p=[size])

    def write(self, data, timeout_ms):
        """Writes bytes to this file.

        The file current position (:py:func:`IFile.offset` ) is updated on success.

        in data of type str
            Array of bytes to write. The size of the array also specifies
            how much to write.

        in timeout_ms of type int
            Timeout (in ms) to wait for the operation to complete.
            Pass 0 for an infinite timeout.

        return written of type int
            How many bytes were written.

        """
        if not isinstance(data, list):
            raise TypeError("data can only be an instance of type list")
        for a in data[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        written = self._call("write", in_p=[data, timeout_ms])
        return written

    def write_at(self, offset, data, timeout_ms):
        """Writes bytes at a certain offset to this file.

        The file current position (:py:func:`IFile.offset` ) is updated on success.

        in offset of type int
            Offset in bytes to start writing.  If the file was opened with the
            :py:func:`IFile.access_mode`  set to :py:attr:`FileAccessMode.append_only`
            or :py:attr:`FileAccessMode.append_read` , the offset is ignored and the
            write always goes to the end of the file.

        in data of type str
            Array of bytes to write. The size of the array also specifies
            how much to write.

        in timeout_ms of type int
            Timeout (in ms) to wait for the operation to complete.
            Pass 0 for an infinite timeout.

        return written of type int
            How many bytes were written.

        """
        if not isinstance(offset, baseinteger):
            raise TypeError("offset can only be an instance of type baseinteger")
        if not isinstance(data, list):
            raise TypeError("data can only be an instance of type list")
        for a in data[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        written = self._call("writeAt", in_p=[offset, data, timeout_ms])
        return written


class IGuestFile(IFile):
    """
    Implementation of the :py:class:`IFile`  object
    for files in the guest.
    """

    __uuid__ = "92f21dc0-44de-1653-b717-2ebf0ca9b664"
    __wsmap__ = "managed"

    @property
    def midl_does_not_like_empty_interfaces(self):
        """Get bool value for 'midlDoesNotLikeEmptyInterfaces'"""
        ret = self._get_attr("midlDoesNotLikeEmptyInterfaces")
        return ret


class IFsObjInfo(Interface):
    """
    Abstract parent interface for VirtualBox file system object information.
    This can be information about a file or a directory, for example.
    """

    __uuid__ = "081fc833-c6fa-430e-6020-6a505d086387"
    __wsmap__ = "managed"

    @property
    def name(self):
        """Get str value for 'name'
        The object's name.
        """
        ret = self._get_attr("name")
        return ret

    @property
    def type_p(self):
        """Get FsObjType value for 'type'
        The object type. See :py:class:`FsObjType`  for more.
        """
        ret = self._get_attr("type")
        return FsObjType(ret)

    @property
    def file_attributes(self):
        """Get str value for 'fileAttributes'
        File attributes. Not implemented yet.
        """
        ret = self._get_attr("fileAttributes")
        return ret

    @property
    def object_size(self):
        """Get int value for 'objectSize'
        The logical size (st_size). For normal files this is the size of the file.
        For symbolic links, this is the length of the path name contained in the
        symbolic link. For other objects this fields needs to be specified.
        """
        ret = self._get_attr("objectSize")
        return ret

    @property
    def allocated_size(self):
        """Get int value for 'allocatedSize'
        Disk allocation size (st_blocks * DEV_BSIZE).
        """
        ret = self._get_attr("allocatedSize")
        return ret

    @property
    def access_time(self):
        """Get int value for 'accessTime'
        Time of last access (st_atime).
        """
        ret = self._get_attr("accessTime")
        return ret

    @property
    def birth_time(self):
        """Get int value for 'birthTime'
        Time of file birth (st_birthtime).
        """
        ret = self._get_attr("birthTime")
        return ret

    @property
    def change_time(self):
        """Get int value for 'changeTime'
        Time of last status change (st_ctime).
        """
        ret = self._get_attr("changeTime")
        return ret

    @property
    def modification_time(self):
        """Get int value for 'modificationTime'
        Time of last data modification (st_mtime).
        """
        ret = self._get_attr("modificationTime")
        return ret

    @property
    def uid(self):
        """Get int value for 'UID'
        The user owning the filesystem object (st_uid).  This is -1 if not available.
        """
        ret = self._get_attr("UID")
        return ret

    @property
    def user_name(self):
        """Get str value for 'userName'
        The user name.
        """
        ret = self._get_attr("userName")
        return ret

    @property
    def gid(self):
        """Get int value for 'GID'
        The group the filesystem object is assigned (st_gid).  This is -1 if not available.
        """
        ret = self._get_attr("GID")
        return ret

    @property
    def group_name(self):
        """Get str value for 'groupName'
        The group name.
        """
        ret = self._get_attr("groupName")
        return ret

    @property
    def node_id(self):
        """Get int value for 'nodeId'
        The unique identifier (within the filesystem) of this filesystem object (st_ino).
        This is zero if not availalbe.
        """
        ret = self._get_attr("nodeId")
        return ret

    @property
    def node_id_device(self):
        """Get int value for 'nodeIdDevice'
        The device number of the device which this filesystem object resides on (st_dev).
        """
        ret = self._get_attr("nodeIdDevice")
        return ret

    @property
    def hard_links(self):
        """Get int value for 'hardLinks'
        Number of hard links to this filesystem object (st_nlink).
        """
        ret = self._get_attr("hardLinks")
        return ret

    @property
    def device_number(self):
        """Get int value for 'deviceNumber'
        The device number of a character or block device type object (st_rdev).
        """
        ret = self._get_attr("deviceNumber")
        return ret

    @property
    def generation_id(self):
        """Get int value for 'generationId'
        The current generation number (st_gen).
        """
        ret = self._get_attr("generationId")
        return ret

    @property
    def user_flags(self):
        """Get int value for 'userFlags'
        User flags (st_flags).
        """
        ret = self._get_attr("userFlags")
        return ret


class IGuestFsObjInfo(IFsObjInfo):
    """
    Represents the guest implementation of the
    :py:class:`IFsObjInfo`  object.
    """

    __uuid__ = "6620db85-44e0-ca69-e9e0-d4907ceccbe5"
    __wsmap__ = "managed"

    @property
    def midl_does_not_like_empty_interfaces(self):
        """Get bool value for 'midlDoesNotLikeEmptyInterfaces'"""
        ret = self._get_attr("midlDoesNotLikeEmptyInterfaces")
        return ret


class IGuest(Interface):
    """
    The IGuest interface represents information about the operating system
    running inside the virtual machine. Used in
    :py:func:`IConsole.guest` .

    IGuest provides information about the guest operating system, whether
    Guest Additions are installed and other OS-specific virtual machine
    properties.
    """

    __uuid__ = "00892186-A4AF-4627-B21F-FC561CE4473C"
    __wsmap__ = "managed"

    @property
    def os_type_id(self):
        """Get str value for 'OSTypeId'
        Identifier of the Guest OS type as reported by the Guest
        Additions.
        You may use :py:func:`IVirtualBox.get_guest_os_type`  to obtain
        an IGuestOSType object representing details about the given
        Guest OS type.

        If Guest Additions are not installed, this value will be
        the same as :py:func:`IMachine.os_type_id` .
        """
        ret = self._get_attr("OSTypeId")
        return ret

    @property
    def additions_run_level(self):
        """Get AdditionsRunLevelType value for 'additionsRunLevel'
        Current run level of the installed Guest Additions.
        """
        ret = self._get_attr("additionsRunLevel")
        return AdditionsRunLevelType(ret)

    @property
    def additions_version(self):
        """Get str value for 'additionsVersion'
        Version of the installed Guest Additions in the same format as
        :py:func:`IVirtualBox.version` .
        """
        ret = self._get_attr("additionsVersion")
        return ret

    @property
    def additions_revision(self):
        """Get int value for 'additionsRevision'
        The internal build revision number of the installed Guest Additions.

        See also :py:func:`IVirtualBox.revision` .
        """
        ret = self._get_attr("additionsRevision")
        return ret

    @property
    def dn_d_source(self):
        """Get IGuestDnDSource value for 'dnDSource'
        Retrieves the drag'n drop source implementation for the guest side, that
        is, handling and retrieving drag'n drop data from the guest.
        """
        ret = self._get_attr("dnDSource")
        return IGuestDnDSource(ret)

    @property
    def dn_d_target(self):
        """Get IGuestDnDTarget value for 'dnDTarget'
        Retrieves the drag'n drop source implementation for the host side. This
        will allow the host to handle and initiate a drag'n drop operation to copy
        data from the host to the guest.
        """
        ret = self._get_attr("dnDTarget")
        return IGuestDnDTarget(ret)

    @property
    def event_source(self):
        """Get IEventSource value for 'eventSource'
        Event source for guest events.
        """
        ret = self._get_attr("eventSource")
        return IEventSource(ret)

    @property
    def facilities(self):
        """Get IAdditionsFacility value for 'facilities'
        Returns a collection of current known facilities. Only returns facilities where
        a status is known, e.g. facilities with an unknown status will not be returned.
        """
        ret = self._get_attr("facilities")
        return [IAdditionsFacility(a) for a in ret]

    @property
    def sessions(self):
        """Get IGuestSession value for 'sessions'
        Returns a collection of all opened guest sessions.
        """
        ret = self._get_attr("sessions")
        return [IGuestSession(a) for a in ret]

    @property
    def memory_balloon_size(self):
        """Get or set int value for 'memoryBalloonSize'
        Guest system memory balloon size in megabytes (transient property).
        """
        ret = self._get_attr("memoryBalloonSize")
        return ret

    @memory_balloon_size.setter
    def memory_balloon_size(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("memoryBalloonSize", value)

    @property
    def statistics_update_interval(self):
        """Get or set int value for 'statisticsUpdateInterval'
        Interval to update guest statistics in seconds.
        """
        ret = self._get_attr("statisticsUpdateInterval")
        return ret

    @statistics_update_interval.setter
    def statistics_update_interval(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("statisticsUpdateInterval", value)

    def internal_get_statistics(self):
        """Internal method; do not use as it might change at any time.

        out cpu_user of type int
            Percentage of processor time spent in user mode as seen by the guest.

        out cpu_kernel of type int
            Percentage of processor time spent in kernel mode as seen by the guest.

        out cpu_idle of type int
            Percentage of processor time spent idling as seen by the guest.

        out mem_total of type int
            Total amount of physical guest RAM.

        out mem_free of type int
            Free amount of physical guest RAM.

        out mem_balloon of type int
            Amount of ballooned physical guest RAM.

        out mem_shared of type int
            Amount of shared physical guest RAM.

        out mem_cache of type int
            Total amount of guest (disk) cache memory.

        out paged_total of type int
            Total amount of space in the page file.

        out mem_alloc_total of type int
            Total amount of memory allocated by the hypervisor.

        out mem_free_total of type int
            Total amount of free memory available in the hypervisor.

        out mem_balloon_total of type int
            Total amount of memory ballooned by the hypervisor.

        out mem_shared_total of type int
            Total amount of shared memory in the hypervisor.

        """
        (
            cpu_user,
            cpu_kernel,
            cpu_idle,
            mem_total,
            mem_free,
            mem_balloon,
            mem_shared,
            mem_cache,
            paged_total,
            mem_alloc_total,
            mem_free_total,
            mem_balloon_total,
            mem_shared_total,
        ) = self._call("internalGetStatistics")
        return (
            cpu_user,
            cpu_kernel,
            cpu_idle,
            mem_total,
            mem_free,
            mem_balloon,
            mem_shared,
            mem_cache,
            paged_total,
            mem_alloc_total,
            mem_free_total,
            mem_balloon_total,
            mem_shared_total,
        )

    def get_facility_status(self, facility):
        """Get the current status of a Guest Additions facility.

        in facility of type :class:`AdditionsFacilityType`
            Facility to check status for.

        out timestamp of type int
            Timestamp (in ms) of last status update seen by the host.

        return status of type :class:`AdditionsFacilityStatus`
            The current (latest) facility status.

        """
        if not isinstance(facility, AdditionsFacilityType):
            raise TypeError(
                "facility can only be an instance of type AdditionsFacilityType"
            )
        (status, timestamp) = self._call("getFacilityStatus", in_p=[facility])
        status = AdditionsFacilityStatus(status)
        return (status, timestamp)

    def get_additions_status(self, level):
        """Retrieve the current status of a certain Guest Additions run level.

        in level of type :class:`AdditionsRunLevelType`
            Status level to check

        return active of type bool
            Flag whether the status level has been reached or not

        raises :class:`VBoxErrorNotSupported`
            Wrong status level specified.

        """
        if not isinstance(level, AdditionsRunLevelType):
            raise TypeError(
                "level can only be an instance of type AdditionsRunLevelType"
            )
        active = self._call("getAdditionsStatus", in_p=[level])
        return active

    def set_credentials(self, user_name, password, domain, allow_interactive_logon):
        """Store login credentials that can be queried by guest operating
        systems with Additions installed. The credentials are transient
        to the session and the guest may also choose to erase them. Note
        that the caller cannot determine whether the guest operating system
        has queried or made use of the credentials.

        in user_name of type str
            User name string, can be empty

        in password of type str
            Password string, can be empty

        in domain of type str
            Domain name (guest logon scheme specific), can be empty

        in allow_interactive_logon of type bool
            Flag whether the guest should alternatively allow the user to
            interactively specify different credentials. This flag might
            not be supported by all versions of the Additions.

        raises :class:`VBoxErrorVmError`
            VMM device is not available.

        """
        if not isinstance(user_name, basestring):
            raise TypeError("user_name can only be an instance of type basestring")
        if not isinstance(password, basestring):
            raise TypeError("password can only be an instance of type basestring")
        if not isinstance(domain, basestring):
            raise TypeError("domain can only be an instance of type basestring")
        if not isinstance(allow_interactive_logon, bool):
            raise TypeError(
                "allow_interactive_logon can only be an instance of type bool"
            )
        self._call(
            "setCredentials",
            in_p=[user_name, password, domain, allow_interactive_logon],
        )

    def create_session(self, user, password, domain, session_name):
        """Creates a new guest session for controlling the guest. The new session
        will be started asynchronously, meaning on return of this function it is
        not guaranteed that the guest session is in a started and/or usable state.
        To wait for successful startup, use the :py:func:`IGuestSession.wait_for`
        call.

        A guest session represents one impersonated user account in the guest, so
        every operation will use the same credentials specified when creating
        the session object via :py:func:`IGuest.create_session` . Anonymous
        sessions, that is, sessions without specifying a valid
        user account in the guest are not allowed reasons of security.

        There can be a maximum of 32 sessions at once per VM.  An error will
        be returned if this has been reached.

        For more information please consult :py:class:`IGuestSession`

        in user of type str
            User name this session will be using to control the guest; has to exist
            and have the appropriate rights to execute programs in the VM. Must not
            be empty.

        in password of type str
            Password of the user account to be used. Empty passwords are allowed.

        in domain of type str
            Domain name of the user account to be used if the guest is part of
            a domain. Optional. This feature is not implemented yet.

        in session_name of type str
            The session's friendly name. Optional, can be empty.

        return guest_session of type :class:`IGuestSession`
            The newly created session object.

        raises :class:`VBoxErrorIprtError`
            Error creating guest session.

        raises :class:`VBoxErrorMaximumReached`
            The maximum of concurrent guest sessions has been reached.

        """
        if not isinstance(user, basestring):
            raise TypeError("user can only be an instance of type basestring")
        if not isinstance(password, basestring):
            raise TypeError("password can only be an instance of type basestring")
        if not isinstance(domain, basestring):
            raise TypeError("domain can only be an instance of type basestring")
        if not isinstance(session_name, basestring):
            raise TypeError("session_name can only be an instance of type basestring")
        guest_session = self._call(
            "createSession", in_p=[user, password, domain, session_name]
        )
        guest_session = IGuestSession(guest_session)
        return guest_session

    def find_session(self, session_name):
        """Finds guest sessions by their friendly name and returns an interface
        array with all found guest sessions.

        in session_name of type str
            The session's friendly name to find. Wildcards like ? and * are allowed.

        return sessions of type :class:`IGuestSession`
            Array with all guest sessions found matching the name specified.

        """
        if not isinstance(session_name, basestring):
            raise TypeError("session_name can only be an instance of type basestring")
        sessions = self._call("findSession", in_p=[session_name])
        sessions = [IGuestSession(a) for a in sessions]
        return sessions

    def shutdown(self, flags):
        """Shuts down (and optionally halts and/or reboots) the guest.
                Needs supported Guest Additions installed.

                in flags of type :class:`GuestShutdownFlag`
                    :py:class:`GuestShutdownFlag`  flags.

                raises :class:`VBoxErrorNotSupported`
                    Guest OS is not supported for shutting down, or the
        already installed Guest Additions are not supported.

                raises :class:`VBoxErrorIprtError`
                    Error while shutting down.

        """
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, GuestShutdownFlag):
                raise TypeError(
                    "array can only contain objects of type GuestShutdownFlag"
                )
        self._call("shutdown", in_p=[flags])

    def update_guest_additions(self, source, arguments, flags):
        """Automatically updates already installed Guest Additions in a VM.

                At the moment only Windows and Linux guests are supported.

                Because the VirtualBox Guest Additions drivers are not WHQL-certified
                yet there might be warning dialogs during the actual Guest Additions
                update. These need to be confirmed manually in order to continue the
                installation process. This applies to Windows 2000 and Windows XP guests
                and therefore these guests can't be updated in a fully automated fashion
                without user interaction. However, to start a Guest Additions update for
                the mentioned Windows versions anyway, the flag
                AdditionsUpdateFlag_WaitForUpdateStartOnly can be specified. See
                :py:class:`AdditionsUpdateFlag`  for more information.

                The guest needs to be restarted in order to make use of the updated
                Guest Additions.

                in source of type str
                    Path to the Guest Additions .ISO file to use for the update.

                in arguments of type str
                    Optional command line arguments to use for the Guest Additions
                    installer. Useful for retrofitting features which weren't installed
                    before in the guest.

                in flags of type :class:`AdditionsUpdateFlag`
                    :py:class:`AdditionsUpdateFlag`  flags.

                return progress of type :class:`IProgress`
                    Progress object to track the operation completion.

                raises :class:`VBoxErrorNotSupported`
                    Guest OS is not supported for automated Guest Additions updates or the
        already installed Guest Additions are not ready yet.

                raises :class:`VBoxErrorIprtError`
                    Error while updating.

        """
        if not isinstance(source, basestring):
            raise TypeError("source can only be an instance of type basestring")
        if not isinstance(arguments, list):
            raise TypeError("arguments can only be an instance of type list")
        for a in arguments[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(flags, list):
            raise TypeError("flags can only be an instance of type list")
        for a in flags[:10]:
            if not isinstance(a, AdditionsUpdateFlag):
                raise TypeError(
                    "array can only contain objects of type AdditionsUpdateFlag"
                )
        progress = self._call("updateGuestAdditions", in_p=[source, arguments, flags])
        progress = IProgress(progress)
        return progress


class IProgress(Interface):
    """
    The IProgress interface is used to track and control
    asynchronous tasks within VirtualBox.

    An instance of this is returned every time VirtualBox starts
    an asynchronous task (in other words, a separate thread) which
    continues to run after a method call returns. For example,
    :py:func:`IMachine.save_state` , which saves the state of
    a running virtual machine, can take a long time to complete.
    To be able to display a progress bar, a user interface such as
    the VirtualBox graphical user interface can use the IProgress
    object returned by that method.

    Note that IProgress is a "read-only" interface in the sense
    that only the VirtualBox internals behind the Main API can
    create and manipulate progress objects, whereas client code
    can only use the IProgress object to monitor a task's
    progress and, if :py:func:`cancelable`  is @c true,
    cancel the task by calling :py:func:`cancel` .

    A task represented by IProgress consists of either one or
    several sub-operations that run sequentially, one by one (see
    :py:func:`operation`  and :py:func:`operation_count` ).
    Every operation is identified by a number (starting from 0)
    and has a separate description.

    You can find the individual percentage of completion of the current
    operation in :py:func:`operation_percent`  and the
    percentage of completion of the task as a whole
    in :py:func:`percent` .

    Similarly, you can wait for the completion of a particular
    operation via :py:func:`wait_for_operation_completion`  or
    for the completion of the whole task via
    :py:func:`wait_for_completion` .
    """

    __uuid__ = "d7b98d2b-30e8-447e-99cb-e31becae6ae4"
    __wsmap__ = "managed"

    @property
    def id_p(self):
        """Get str value for 'id'
        ID of the task.
        """
        ret = self._get_attr("id")
        return ret

    @property
    def description(self):
        """Get str value for 'description'
        Description of the task.
        """
        ret = self._get_attr("description")
        return ret

    @property
    def initiator(self):
        """Get Interface value for 'initiator'
        Initiator of the task.
        """
        ret = self._get_attr("initiator")
        return Interface(ret)

    @property
    def cancelable(self):
        """Get bool value for 'cancelable'
        Whether the task can be interrupted.
        """
        ret = self._get_attr("cancelable")
        return ret

    @property
    def percent(self):
        """Get int value for 'percent'
        Current progress value of the task as a whole, in percent.
        This value depends on how many operations are already complete.
        Returns 100 if :py:func:`completed`  is @c true.
        """
        ret = self._get_attr("percent")
        return ret

    @property
    def time_remaining(self):
        """Get int value for 'timeRemaining'
        Estimated remaining time until the task completes, in
        seconds. Returns 0 once the task has completed; returns -1
        if the remaining time cannot be computed, in particular if
        the current progress is 0.

        Even if a value is returned, the estimate will be unreliable
        for low progress values. It will become more reliable as the
        task progresses; it is not recommended to display an ETA
        before at least 20% of a task have completed.
        """
        ret = self._get_attr("timeRemaining")
        return ret

    @property
    def completed(self):
        """Get bool value for 'completed'
        Whether the task has been completed.
        """
        ret = self._get_attr("completed")
        return ret

    @property
    def canceled(self):
        """Get bool value for 'canceled'
        Whether the task has been canceled.
        """
        ret = self._get_attr("canceled")
        return ret

    @property
    def result_code(self):
        """Get int value for 'resultCode'
        Result code of the progress task.
        Valid only if :py:func:`completed`  is @c true.
        """
        ret = self._get_attr("resultCode")
        return ret

    @property
    def error_info(self):
        """Get IVirtualBoxErrorInfo value for 'errorInfo'
        Extended information about the unsuccessful result of the
        progress operation. May be @c null if no extended information
        is available.
        Valid only if :py:func:`completed`  is @c true and
        :py:func:`result_code`  indicates a failure.
        """
        ret = self._get_attr("errorInfo")
        return IVirtualBoxErrorInfo(ret)

    @property
    def operation_count(self):
        """Get int value for 'operationCount'
        Number of sub-operations this task is divided into.
        Every task consists of at least one suboperation.
        """
        ret = self._get_attr("operationCount")
        return ret

    @property
    def operation(self):
        """Get int value for 'operation'
        Number of the sub-operation being currently executed.
        """
        ret = self._get_attr("operation")
        return ret

    @property
    def operation_description(self):
        """Get str value for 'operationDescription'
        Description of the sub-operation being currently executed.
        """
        ret = self._get_attr("operationDescription")
        return ret

    @property
    def operation_percent(self):
        """Get int value for 'operationPercent'
        Progress value of the current sub-operation only, in percent.
        """
        ret = self._get_attr("operationPercent")
        return ret

    @property
    def operation_weight(self):
        """Get int value for 'operationWeight'
        Weight value of the current sub-operation only.
        """
        ret = self._get_attr("operationWeight")
        return ret

    @property
    def timeout(self):
        """Get or set int value for 'timeout'
        When non-zero, this specifies the number of milliseconds after which
        the operation will automatically be canceled. This can only be set on
        cancelable objects.
        """
        ret = self._get_attr("timeout")
        return ret

    @timeout.setter
    def timeout(self, value):
        if not isinstance(value, baseinteger):
            raise TypeError("value is not an instance of baseinteger")
        return self._set_attr("timeout", value)

    @property
    def event_source(self):
        """Get IEventSource value for 'eventSource'"""
        ret = self._get_attr("eventSource")
        return IEventSource(ret)

    def wait_for_completion(self, timeout):
        """Waits until the task is done (including all sub-operations)
        with a given timeout in milliseconds; specify -1 for an indefinite wait.

        Note that the VirtualBox/XPCOM/COM/native event queues of the calling
        thread are not processed while waiting. Neglecting event queues may
        have dire consequences (degrade performance, resource hogs,
        deadlocks, etc.), this is specially so for the main thread on
        platforms using XPCOM. Callers are advised wait for short periods
        and service their event queues between calls, or to create a worker
        thread to do the waiting.

        in timeout of type int
            Maximum time in milliseconds to wait or -1 to wait indefinitely.

        raises :class:`VBoxErrorIprtError`
            Failed to wait for task completion.

        """
        if not isinstance(timeout, baseinteger):
            raise TypeError("timeout can only be an instance of type baseinteger")
        self._call("waitForCompletion", in_p=[timeout])

    def wait_for_operation_completion(self, operation, timeout):
        """Waits until the given operation is done with a given timeout in
        milliseconds; specify -1 for an indefinite wait.

        See :py:func:`wait_for_completion` for event queue considerations.

        in operation of type int
            Number of the operation to wait for.
            Must be less than :py:func:`operation_count` .

        in timeout of type int
            Maximum time in milliseconds to wait or -1 to wait indefinitely.

        raises :class:`VBoxErrorIprtError`
            Failed to wait for operation completion.

        """
        if not isinstance(operation, baseinteger):
            raise TypeError("operation can only be an instance of type baseinteger")
        if not isinstance(timeout, baseinteger):
            raise TypeError("timeout can only be an instance of type baseinteger")
        self._call("waitForOperationCompletion", in_p=[operation, timeout])

    def cancel(self):
        """Cancels the task.

        If :py:func:`cancelable`  is @c false, then this method will fail.

        raises :class:`VBoxErrorInvalidObjectState`
            Operation cannot be canceled.

        """
        self._call("cancel")


class IInternalProgressControl(Interface):
    """
    Internal method, not to be called externally.
    """

    __uuid__ = "41a033b8-cc87-4f6e-a0e9-47bb7f2d4be5"
    __wsmap__ = "suppress"

    def set_current_operation_progress(self, percent):
        """Internal method, not to be called externally.

        in percent of type int

        """
        if not isinstance(percent, baseinteger):
            raise TypeError("percent can only be an instance of type baseinteger")
        self._call("setCurrentOperationProgress", in_p=[percent])

    def wait_for_other_progress_completion(self, progress_other, timeout_ms):
        """Internal method, not to be called externally.

        Waits until the other task is completed (including all sub-operations)
        and forward all changes from the other progress to this progress. This
        means sub-operation number, description, percent and so on.

        The caller is responsible for having at least the same count of
        sub-operations in this progress object as there are in the other
        progress object.

        If the other progress object supports cancel and this object gets any
        cancel request (when here enabled as well), it will be forwarded to
        the other progress object.

        Error information is automatically preserved (by transferring it to
        the current thread's error information). If the caller wants to set it
        as the completion state of this progress it needs to be done separately.

        in progress_other of type :class:`IProgress`
            Other progress object to be "cloned".

        in timeout_ms of type int
            Timeout (in ms). Pass 0 for an infinite timeout.

        raises :class:`VBoxErrorTimeout`
            Waiting time has expired.

        """
        if not isinstance(progress_other, IProgress):
            raise TypeError("progress_other can only be an instance of type IProgress")
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        self._call("waitForOtherProgressCompletion", in_p=[progress_other, timeout_ms])

    def set_next_operation(self, next_operation_description, next_operations_weight):
        """Internal method, not to be called externally.

        in next_operation_description of type str

        in next_operations_weight of type int

        """
        if not isinstance(next_operation_description, basestring):
            raise TypeError(
                "next_operation_description can only be an instance of type basestring"
            )
        if not isinstance(next_operations_weight, baseinteger):
            raise TypeError(
                "next_operations_weight can only be an instance of type baseinteger"
            )
        self._call(
            "setNextOperation",
            in_p=[next_operation_description, next_operations_weight],
        )

    def notify_point_of_no_return(self):
        """Internal method, not to be called externally."""
        self._call("notifyPointOfNoReturn")

    def notify_complete(self, result_code, error_info):
        """Internal method, not to be called externally.

        in result_code of type int

        in error_info of type :class:`IVirtualBoxErrorInfo`

        """
        if not isinstance(result_code, baseinteger):
            raise TypeError("result_code can only be an instance of type baseinteger")
        if not isinstance(error_info, IVirtualBoxErrorInfo):
            raise TypeError(
                "error_info can only be an instance of type IVirtualBoxErrorInfo"
            )
        self._call("notifyComplete", in_p=[result_code, error_info])


class ISnapshot(Interface):
    """
    The ISnapshot interface represents a snapshot of the virtual
    machine.

    Together with the differencing media that are created
    when a snapshot is taken, a machine can be brought back to
    the exact state it was in when the snapshot was taken.

    The ISnapshot interface has no methods, only attributes; snapshots
    are controlled through methods of the :py:class:`IMachine`  interface
    which also manage the media associated with the snapshot.
    The following operations exist:


    :py:func:`IMachine.take_snapshot`  creates a new snapshot
    by creating new, empty differencing images for the machine's
    media and saving the VM settings and (if the VM is running)
    the current VM state in the snapshot.

    The differencing images will then receive all data written to
    the machine's media, while their parent (base) images
    remain unmodified after the snapshot has been taken (see
    :py:class:`IMedium`  for details about differencing images).
    This simplifies restoring a machine to the state of a snapshot:
    only the differencing images need to be deleted.

    The current machine state is not changed by taking a snapshot
    except that :py:func:`IMachine.current_snapshot`  is set to
    the newly created snapshot, which is also added to the machine's
    snapshots tree.


    :py:func:`IMachine.restore_snapshot`  resets a machine to
    the state of a previous snapshot by deleting the differencing
    image of each of the machine's media and setting the machine's
    settings and state to the state that was saved in the snapshot (if any).

    This destroys the machine's current state. After calling this,
    :py:func:`IMachine.current_snapshot`  points to the snapshot
    that was restored.


    :py:func:`IMachine.delete_snapshot`  deletes a snapshot
    without affecting the current machine state.

    This does not change the current machine state, but instead frees the
    resources allocated when the snapshot was taken: the settings and machine
    state file are deleted (if any), and the snapshot's differencing image for
    each of the machine's media gets merged with its parent image.

    Neither the current machine state nor other snapshots are affected
    by this operation, except that parent media will be modified
    to contain the disk data associated with the snapshot being deleted.

    When deleting the current snapshot, the :py:func:`IMachine.current_snapshot`
    attribute is set to the current snapshot's parent or @c null if it
    has no parent. Otherwise the attribute is unchanged.



    Each snapshot contains a copy of virtual machine's settings (hardware
    configuration etc.). This copy is contained in an immutable (read-only)
    instance of :py:class:`IMachine`  which is available from the snapshot's
    :py:func:`machine`  attribute. When restoring the snapshot, these
    settings are copied back to the original machine.

    In addition, if the machine was running when the
    snapshot was taken (:py:func:`IMachine.state`  is :py:attr:`MachineState.running` ),
    the current VM state is saved in the snapshot (similarly to what happens
    when a VM's state is saved). The snapshot is then said to be *online*
    because when restoring it, the VM will be running.

    If the machine was in :py:attr:`MachineState.saved` saved saved,
    the snapshot receives a copy of the execution state file
    (:py:func:`IMachine.state_file_path` ).

    Otherwise, if the machine was not running (:py:attr:`MachineState.powered_off`
    or :py:attr:`MachineState.aborted` ), the snapshot is *offline*;
    it then contains a so-called "zero execution state", representing a
    machine that is powered off.
    """

    __uuid__ = "6cc49055-dad4-4496-85cf-3f76bcb3b5fa"
    __wsmap__ = "managed"

    @property
    def id_p(self):
        """Get str value for 'id'
        UUID of the snapshot.
        """
        ret = self._get_attr("id")
        return ret

    @property
    def name(self):
        """Get or set str value for 'name'
        Short name of the snapshot.
        Setting this attribute causes :py:func:`IMachine.save_settings`  to
        be called implicitly.
        """
        ret = self._get_attr("name")
        return ret

    @name.setter
    def name(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("name", value)

    @property
    def description(self):
        """Get or set str value for 'description'
        Optional description of the snapshot.
        Setting this attribute causes :py:func:`IMachine.save_settings`  to
        be called implicitly.
        """
        ret = self._get_attr("description")
        return ret

    @description.setter
    def description(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("description", value)

    @property
    def time_stamp(self):
        """Get int value for 'timeStamp'
        Timestamp of the snapshot, in milliseconds since 1970-01-01 UTC.
        """
        ret = self._get_attr("timeStamp")
        return ret

    @property
    def online(self):
        """Get bool value for 'online'
        @c true if this snapshot is an online snapshot and @c false otherwise.

        When this attribute is @c true, the
        :py:func:`IMachine.state_file_path`  attribute of the
        :py:func:`machine`  object associated with this snapshot
        will point to the saved state file. Otherwise, it will be
        an empty string.
        """
        ret = self._get_attr("online")
        return ret

    @property
    def machine(self):
        """Get IMachine value for 'machine'
        Virtual machine this snapshot is taken on. This object
        stores all settings the machine had when taking this snapshot.

        The returned machine object is immutable, i.e. no
        any settings can be changed.
        """
        ret = self._get_attr("machine")
        return IMachine(ret)

    @property
    def parent(self):
        """Get ISnapshot value for 'parent'
        Parent snapshot (a snapshot this one is based on), or
        @c null if the snapshot has no parent (i.e. is the first snapshot).
        """
        ret = self._get_attr("parent")
        return ISnapshot(ret)

    @property
    def children(self):
        """Get ISnapshot value for 'children'
        Child snapshots (all snapshots having this one as a parent).
        By inspecting this attribute starting with a machine's root snapshot
        (which can be obtained by calling :py:func:`IMachine.find_snapshot`
        with a @c null UUID), a machine's snapshots tree can be iterated over.
        """
        ret = self._get_attr("children")
        return [ISnapshot(a) for a in ret]

    @property
    def children_count(self):
        """Get int value for 'childrenCount'
        Returns the number of direct children of this snapshot.
        """
        ret = self._get_attr("childrenCount")
        return ret


class IMediumAttachment(Interface):
    """
    The IMediumAttachment interface links storage media to virtual machines.
    For each medium (:py:class:`IMedium` ) which has been attached to a
    storage controller (:py:class:`IStorageController` ) of a machine
    (:py:class:`IMachine` ) via the :py:func:`IMachine.attach_device`
    method, one instance of IMediumAttachment is added to the machine's
    :py:func:`IMachine.medium_attachments`  array attribute.

    Each medium attachment specifies the storage controller as well as a
    port and device number and the IMedium instance representing a virtual
    hard disk or floppy or DVD image.

    For removable media (DVDs or floppies), there are two additional
    options. For one, the IMedium instance can be @c null to represent
    an empty drive with no media inserted (see :py:func:`IMachine.mount_medium` );
    secondly, the medium can be one of the pseudo-media for host drives
    listed in :py:func:`IHost.dvd_drives`  or :py:func:`IHost.floppy_drives` .

    Attaching Hard Disks

    Hard disks are attached to virtual machines using the
    :py:func:`IMachine.attach_device`  method and detached using the
    :py:func:`IMachine.detach_device`  method. Depending on a medium's
    type (see :py:func:`IMedium.type_p` ), hard disks are attached either
    *directly* or *indirectly*.

    When a hard disk is being attached directly, it is associated with the
    virtual machine and used for hard disk operations when the machine is
    running. When a hard disk is being attached indirectly, a new differencing
    hard disk linked to it is implicitly created and this differencing hard
    disk is associated with the machine and used for hard disk operations.
    This also means that if :py:func:`IMachine.attach_device`  performs
    a direct attachment then the same hard disk will be returned in response
    to the subsequent :py:func:`IMachine.get_medium`  call; however if
    an indirect attachment is performed then
    :py:func:`IMachine.get_medium`  will return the implicitly created
    differencing hard disk, not the original one passed to :py:func:`IMachine.attach_device` . In detail:


    **Normal base** hard disks that do not have children (i.e.
    differencing hard disks linked to them) and that are not already
    attached to virtual machines in snapshots are attached **directly**.
    Otherwise, they are attached **indirectly** because having
    dependent children or being part of the snapshot makes it impossible
    to modify hard disk contents without breaking the integrity of the
    dependent party. The :py:func:`IMedium.read_only`  attribute allows to
    quickly determine the kind of the attachment for the given hard
    disk. Note that if a normal base hard disk is to be indirectly
    attached to a virtual machine with snapshots then a special
    procedure called *smart attachment* is performed (see below).
    **Normal differencing** hard disks are like normal base hard disks:
    they are attached **directly** if they do not have children and are
    not attached to virtual machines in snapshots, and **indirectly**
    otherwise. Note that the smart attachment procedure is never performed
    for differencing hard disks.
    **Immutable** hard disks are always attached **indirectly** because
    they are designed to be non-writable. If an immutable hard disk is
    attached to a virtual machine with snapshots then a special
    procedure called smart attachment is performed (see below).
    **Writethrough** hard disks are always attached **directly**,
    also as designed. This also means that writethrough hard disks cannot
    have other hard disks linked to them at all.
    **Shareable** hard disks are always attached **directly**,
    also as designed. This also means that shareable hard disks cannot
    have other hard disks linked to them at all. They behave almost
    like writethrough hard disks, except that shareable hard disks can
    be attached to several virtual machines which are running, allowing
    concurrent accesses. You need special cluster software running in
    the virtual machines to make use of such disks.


    Note that the same hard disk, regardless of its type, may be attached to
    more than one virtual machine at a time. In this case, the machine that is
    started first gains exclusive access to the hard disk and attempts to
    start other machines having this hard disk attached will fail until the
    first machine is powered down.

    Detaching hard disks is performed in a *deferred* fashion. This means
    that the given hard disk remains associated with the given machine after a
    successful :py:func:`IMachine.detach_device`  call until
    :py:func:`IMachine.save_settings`  is called to save all changes to
    machine settings to disk. This deferring is necessary to guarantee that
    the hard disk configuration may be restored at any time by a call to
    :py:func:`IMachine.discard_settings`  before the settings
    are saved (committed).

    Note that if :py:func:`IMachine.discard_settings`  is called after
    indirectly attaching some hard disks to the machine but before a call to
    :py:func:`IMachine.save_settings`  is made, it will implicitly delete
    all differencing hard disks implicitly created by
    :py:func:`IMachine.attach_device`  for these indirect attachments.
    Such implicitly created hard disks will also be immediately deleted when
    detached explicitly using the :py:func:`IMachine.detach_device`
    call if it is made before :py:func:`IMachine.save_settings` . This
    implicit deletion is safe because newly created differencing hard
    disks do not contain any user data.

    However, keep in mind that detaching differencing hard disks that were
    implicitly created by :py:func:`IMachine.attach_device`
    before the last :py:func:`IMachine.save_settings`  call will
    **not** implicitly delete them as they may already contain some data
    (for example, as a result of virtual machine execution). If these hard
    disks are no more necessary, the caller can always delete them explicitly
    using :py:func:`IMedium.delete_storage`  after they are actually de-associated
    from this machine by the :py:func:`IMachine.save_settings`  call.

    Smart Attachment

    When normal base or immutable hard disks are indirectly attached to a
    virtual machine then some additional steps are performed to make sure the
    virtual machine will have the most recent "view" of the hard disk being
    attached. These steps include walking through the machine's snapshots
    starting from the current one and going through ancestors up to the first
    snapshot. Hard disks attached to the virtual machine in all
    of the encountered snapshots are checked whether they are descendants of
    the given normal base or immutable hard disk. The first found child (which
    is the differencing hard disk) will be used instead of the normal base or
    immutable hard disk as a parent for creating a new differencing hard disk
    that will be actually attached to the machine. And only if no descendants
    are found or if the virtual machine does not have any snapshots then the
    normal base or immutable hard disk will be used itself as a parent for
    this differencing hard disk.

    It is easier to explain what smart attachment does using the
    following example:

    ::

         BEFORE attaching B.vdi:       AFTER attaching B.vdi:

         Snapshot 1 (B.vdi)            Snapshot 1 (B.vdi)
         Snapshot 2 (D1->B.vdi)        Snapshot 2 (D1->B.vdi)
         Snapshot 3 (D2->D1.vdi)       Snapshot 3 (D2->D1.vdi)
         Snapshot 4 (none)             Snapshot 4 (none)
         CurState   (none)             CurState   (D3->D2.vdi)

         NOT
         ...
         CurState   (D3->B.vdi)


    The first column is the virtual machine configuration before the base hard
    disk B.vdi is attached, the second column shows the machine after
    this hard disk is attached. Constructs like D1->B.vdi and similar
    mean that the hard disk that is actually attached to the machine is a
    differencing hard disk, D1.vdi, which is linked to (based on)
    another hard disk, B.vdi.

    As we can see from the example, the hard disk B.vdi was detached
    from the machine before taking Snapshot 4. Later, after Snapshot 4 was
    taken, the user decides to attach B.vdi again. B.vdi has
    dependent child hard disks (D1.vdi, D2.vdi), therefore
    it cannot be attached directly and needs an indirect attachment (i.e.
    implicit creation of a new differencing hard disk). Due to the smart
    attachment procedure, the new differencing hard disk
    (D3.vdi) will be based on D2.vdi, not on
    B.vdi itself, since D2.vdi is the most recent view of
    B.vdi existing for this snapshot branch of the given virtual
    machine.

    Note that if there is more than one descendant hard disk of the given base
    hard disk found in a snapshot, and there is an exact device, channel and
    bus match, then this exact match will be used. Otherwise, the youngest
    descendant will be picked up.

    There is one more important aspect of the smart attachment procedure which
    is not related to snapshots at all. Before walking through the snapshots
    as described above, the backup copy of the current list of hard disk
    attachment is searched for descendants. This backup copy is created when
    the hard disk configuration is changed for the first time after the last
    :py:func:`IMachine.save_settings`  call and used by
    :py:func:`IMachine.discard_settings`  to undo the recent hard disk
    changes. When such a descendant is found in this backup copy, it will be
    simply re-attached back, without creating a new differencing hard disk for
    it. This optimization is necessary to make it possible to re-attach the
    base or immutable hard disk to a different bus, channel or device slot
    without losing the contents of the differencing hard disk actually
    attached to the machine in place of it.
    """

    __uuid__ = "8d095cb0-0126-43e0-b05d-326e74abb356"
    __wsmap__ = "struct"

    @property
    def machine(self):
        """Get IMachine value for 'machine'
        Machine object for this medium attachment.
        """
        ret = self._get_attr("machine")
        return IMachine(ret)

    @property
    def medium(self):
        """Get IMedium value for 'medium'
        Medium object associated with this attachment; it
        can be @c null for removable devices.
        """
        ret = self._get_attr("medium")
        return IMedium(ret)

    @property
    def controller(self):
        """Get str value for 'controller'
        Name of the storage controller of this attachment; this
        refers to one of the controllers in :py:func:`IMachine.storage_controllers`
        by name.
        """
        ret = self._get_attr("controller")
        return ret

    @property
    def port(self):
        """Get int value for 'port'
        Port number of this attachment.
        See :py:func:`IMachine.attach_device`  for the meaning of this value for the different controller types.
        """
        ret = self._get_attr("port")
        return ret

    @property
    def device(self):
        """Get int value for 'device'
        Device slot number of this attachment.
        See :py:func:`IMachine.attach_device`  for the meaning of this value for the different controller types.
        """
        ret = self._get_attr("device")
        return ret

    @property
    def type_p(self):
        """Get DeviceType value for 'type'
        Device type of this attachment.
        """
        ret = self._get_attr("type")
        return DeviceType(ret)

    @property
    def passthrough(self):
        """Get bool value for 'passthrough'
        Pass I/O requests through to a device on the host.
        """
        ret = self._get_attr("passthrough")
        return ret

    @property
    def temporary_eject(self):
        """Get bool value for 'temporaryEject'
        Whether guest-triggered eject results in unmounting the medium.
        """
        ret = self._get_attr("temporaryEject")
        return ret

    @property
    def is_ejected(self):
        """Get bool value for 'isEjected'
        Signals that the removable medium has been ejected. This is not
        necessarily equivalent to having a @c null medium association.
        """
        ret = self._get_attr("isEjected")
        return ret

    @property
    def non_rotational(self):
        """Get bool value for 'nonRotational'
        Whether the associated medium is non-rotational.
        """
        ret = self._get_attr("nonRotational")
        return ret

    @property
    def discard(self):
        """Get bool value for 'discard'
        Whether the associated medium supports discarding unused blocks.
        """
        ret = self._get_attr("discard")
        return ret

    @property
    def hot_pluggable(self):
        """Get bool value for 'hotPluggable'
        Whether this attachment is hot pluggable or not.
        """
        ret = self._get_attr("hotPluggable")
        return ret

    @property
    def bandwidth_group(self):
        """Get IBandwidthGroup value for 'bandwidthGroup'
        The bandwidth group this medium attachment is assigned to.
        """
        ret = self._get_attr("bandwidthGroup")
        return IBandwidthGroup(ret)


class IMedium(Interface):
    """
    The IMedium interface represents virtual storage for a machine's
    hard disks, CD/DVD or floppy drives. It will typically represent
    a disk image on the host, for example a VDI or VMDK file representing
    a virtual hard disk, or an ISO or RAW file representing virtual
    removable media, but can also point to a network location (e.g.
    for iSCSI targets).

    Instances of IMedium are connected to virtual machines by way of medium
    attachments, which link the storage medium to a particular device slot
    of a storage controller of the virtual machine.
    In the VirtualBox API, virtual storage is therefore always represented
    by the following chain of object links:


    :py:func:`IMachine.storage_controllers`  contains an array of
    storage controllers (IDE, SATA, SCSI, SAS or a floppy controller;
    these are instances of :py:class:`IStorageController` ).
    :py:func:`IMachine.medium_attachments`  contains an array of
    medium attachments (instances of :py:class:`IMediumAttachment`
    created by :py:func:`IMachine.attach_device` ),
    each containing a storage controller from the above array, a
    port/device specification, and an instance of IMedium representing
    the medium storage (image file).

    For removable media, the storage medium is optional; a medium
    attachment with no medium represents a CD/DVD or floppy drive
    with no medium inserted. By contrast, hard disk attachments
    will always have an IMedium object attached.
    Each IMedium in turn points to a storage unit (such as a file
    on the host computer or a network resource) that holds actual
    data. This location is represented by the :py:func:`location`
    attribute.


    Existing media are opened using :py:func:`IVirtualBox.open_medium` ;
    new hard disk media can be created with the VirtualBox API using the
    :py:func:`IVirtualBox.create_medium`  method. Differencing hard
    disks (see below) are usually implicitly created by VirtualBox as
    needed, but may also be created explicitly using :py:func:`create_diff_storage` .
    VirtualBox cannot create CD/DVD or floppy images (ISO and RAW files); these
    should be created with external tools and then opened from within VirtualBox.

    Only for CD/DVDs and floppies, an IMedium instance can also represent a host
    drive. In that case the :py:func:`id_p`  attribute contains the UUID of
    one of the drives in :py:func:`IHost.dvd_drives`  or :py:func:`IHost.floppy_drives` .

    Media registries

    When a medium has been opened or created using one of the aforementioned
    APIs, it becomes "known" to VirtualBox. Known media can be attached
    to virtual machines and re-found through :py:func:`IVirtualBox.open_medium` .
    They also appear in the global
    :py:func:`IVirtualBox.hard_disks` ,
    :py:func:`IVirtualBox.dvd_images`  and
    :py:func:`IVirtualBox.floppy_images`  arrays.

    Prior to VirtualBox 4.0, opening a medium added it to a global media registry
    in the VirtualBox.xml file, which was shared between all machines and made
    transporting machines and their media from one host to another difficult.

    Starting with VirtualBox 4.0, media are only added to a registry when they are
    *attached* to a machine using :py:func:`IMachine.attach_device` . For
    backwards compatibility, which registry a medium is added to depends on which
    VirtualBox version created a machine:


    If the medium has first been attached to a machine which was created by
    VirtualBox 4.0 or later, it is added to that machine's media registry in
    the machine XML settings file. This way all information about a machine's
    media attachments is contained in a single file and can be transported
    easily.
    For older media attachments (i.e. if the medium was first attached to a
    machine which was created with a VirtualBox version before 4.0), media
    continue to be registered in the global VirtualBox settings file, for
    backwards compatibility.


    See :py:func:`IVirtualBox.open_medium`  for more information.

    Media are removed from media registries by the :py:func:`IMedium.close` ,
    :py:func:`delete_storage`  and :py:func:`merge_to`  methods.

    Accessibility checks

    VirtualBox defers media accessibility checks until the :py:func:`refresh_state`
    method is called explicitly on a medium. This is done to make the VirtualBox object
    ready for serving requests as fast as possible and let the end-user
    application decide if it needs to check media accessibility right away or not.

    As a result, when VirtualBox starts up (e.g. the VirtualBox
    object gets created for the first time), all known media are in the
    "Inaccessible" state, but the value of the :py:func:`last_access_error`
    attribute is an empty string because no actual accessibility check has
    been made yet.

    After calling :py:func:`refresh_state` , a medium is considered
    *accessible* if its storage unit can be read. In that case, the
    :py:func:`state`  attribute has a value of "Created". If the storage
    unit cannot be read (for example, because it is located on a disconnected
    network resource, or was accidentally deleted outside VirtualBox),
    the medium is considered *inaccessible*, which is indicated by the
    "Inaccessible" state. The exact reason why the medium is inaccessible can be
    obtained by reading the :py:func:`last_access_error`  attribute.

    Medium types

    There are five types of medium behavior which are stored in the
    :py:func:`type_p`  attribute (see :py:class:`MediumType` ) and
    which define the medium's behavior with attachments and snapshots.

    All media can be also divided in two groups: *base* media and
    *differencing* media. A base medium contains all sectors of the
    medium data in its own storage and therefore can be used independently.
    In contrast, a differencing medium is a "delta" to some other medium and
    contains only those sectors which differ from that other medium, which is
    then called a *parent*. The differencing medium is said to be
    *linked to* that parent. The parent may be itself a differencing
    medium, thus forming a chain of linked media. The last element in that
    chain must always be a base medium. Note that several differencing
    media may be linked to the same parent medium.

    Differencing media can be distinguished from base media by querying the
    :py:func:`parent`  attribute: base media do not have parents they would
    depend on, so the value of this attribute is always @c null for them.
    Using this attribute, it is possible to walk up the medium tree (from the
    child medium to its parent). It is also possible to walk down the tree
    using the :py:func:`children`  attribute.

    Note that the type of all differencing media is "normal"; all other
    values are meaningless for them. Base media may be of any type.

    Automatic composition of the file name part

    Another extension to the :py:func:`IMedium.location`  attribute is that
    there is a possibility to cause VirtualBox to compose a unique value for
    the file name part of the location using the UUID of the hard disk. This
    applies only to hard disks in :py:attr:`MediumState.not_created`  state,
    e.g. before the storage unit is created, and works as follows. You set the
    value of the :py:func:`IMedium.location`  attribute to a location
    specification which only contains the path specification but not the file
    name part and ends with either a forward slash or a backslash character.
    In response, VirtualBox will generate a new UUID for the hard disk and
    compose the file name using the following pattern:

    ::

         <path>/{<uuid>}.<ext>


    where <path> is the supplied path specification,
    <uuid> is the newly generated UUID and <ext>
    is the default extension for the storage format of this hard disk. After
    that, you may call any of the methods that create a new hard disk storage
    unit and they will use the generated UUID and file name.
    """

    __uuid__ = "ad47ad09-787b-44ab-b343-a082a3f2dfb1"
    __wsmap__ = "managed"

    @property
    def id_p(self):
        """Get str value for 'id'
        UUID of the medium. For a newly created medium, this value is a randomly
        generated UUID.


        For media in one of MediumState_NotCreated, MediumState_Creating or
        MediumState_Deleting states, the value of this property is undefined
        and will most likely be an empty UUID.
        """
        ret = self._get_attr("id")
        return ret

    @property
    def description(self):
        """Get or set str value for 'description'
        Optional description of the medium. For a newly created medium the value
        of this attribute is an empty string.

        Medium types that don't support this attribute will return E_NOTIMPL in
        attempt to get or set this attribute's value.


        For some storage types, reading this attribute may return an outdated
        (last known) value when :py:func:`state`  is :py:attr:`MediumState.inaccessible`  or :py:attr:`MediumState.locked_write`  because the value of this attribute is
        stored within the storage unit itself. Also note that changing the
        attribute value is not possible in such case, as well as when the
        medium is the :py:attr:`MediumState.locked_read`  state.
        """
        ret = self._get_attr("description")
        return ret

    @description.setter
    def description(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("description", value)

    @property
    def state(self):
        """Get MediumState value for 'state'
        Returns the current medium state, which is the last state set by
        the accessibility check performed by :py:func:`refresh_state` .
        If that method has not yet been called on the medium, the state
        is "Inaccessible"; as opposed to truly inaccessible media, the
        value of :py:func:`last_access_error`  will be an empty string in
        that case.

        As of version 3.1, this no longer performs an accessibility check
        automatically; call :py:func:`refresh_state`  for that.
        """
        ret = self._get_attr("state")
        return MediumState(ret)

    @property
    def variant(self):
        """Get MediumVariant value for 'variant'
        Returns the storage format variant information for this medium
        as an array of the flags described at :py:class:`MediumVariant` .
        Before :py:func:`refresh_state`  is called this method returns
        an undefined value.
        """
        ret = self._get_attr("variant")
        return [MediumVariant(a) for a in ret]

    @property
    def location(self):
        """Get or set str value for 'location'
        Location of the storage unit holding medium data.

        The format of the location string is medium type specific. For medium
        types using regular files in a host's file system, the location
        string is the full file name.
        """
        ret = self._get_attr("location")
        return ret

    @location.setter
    def location(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value is not an instance of basestring")
        return self._set_attr("location", value)

    @property
    def name(self):
        """Get str value for 'name'
        Name of the storage unit holding medium data.

        The returned string is a short version of the :py:func:`location`
        attribute that is suitable for representing the medium in situations
        where the full location specification is too long (such as lists
        and comboboxes in GUI frontends). This string is also used by frontends
        to sort the media list alphabetically when needed.

        For example, for locations that are regular files in the host's file
        system, the value of this attribute is just the file name (+ extension),
        without the path specification.

        Note that as opposed to the :py:func:`location`  attribute, the name
        attribute will not necessary be unique for a list of media of the
        given type and format.
        """
        ret = self._get_attr("name")
        return ret

    @property
    def device_type(self):
        """Get DeviceType value for 'deviceType'
        Kind of device (DVD/Floppy/HardDisk) which is applicable to this
        medium.
        """
        ret = self._get_attr("deviceType")
        return DeviceType(ret)

    @property
    def host_drive(self):
        """Get bool value for 'hostDrive'
        True if this corresponds to a drive on the host.
        """
        ret = self._get_attr("hostDrive")
        return ret

    @property
    def size(self):
        """Get int value for 'size'
        Physical size of the storage unit used to hold medium data (in bytes).


        For media whose :py:func:`state`  is :py:attr:`MediumState.inaccessible` , the value of this property is the
        last known size. For :py:attr:`MediumState.not_created`  media,
        the returned value is zero.
        """
        ret = self._get_attr("size")
        return ret

    @property
    def format_p(self):
        """Get str value for 'format'
        Storage format of this medium.

        The value of this attribute is a string that specifies a backend used
        to store medium data. The storage format is defined when you create a
        new medium or automatically detected when you open an existing medium,
        and cannot be changed later.

        The list of all storage formats supported by this VirtualBox
        installation can be obtained using
        :py:func:`ISystemProperties.medium_formats` .
        """
        ret = self._get_attr("format")
        return ret

    @property
    def medium_format(self):
        """Get IMediumFormat value for 'mediumFormat'
        Storage medium format object corresponding to this medium.

        The value of this attribute is a reference to the medium format object
        that specifies the backend properties used to store medium data. The
        storage format is defined when you create a new medium or automatically
        detected when you open an existing medium, and cannot be changed later.

        @c null is returned if there is no associated medium format
        object. This can e.g. happen for medium objects representing host
        drives and other special medium objects.
        """
        ret = self._get_attr("mediumFormat")
        return IMediumFormat(ret)

    @property
    def type_p(self):
        """Get or set MediumType value for 'type'
        Type (role) of this medium.

        The following constraints apply when changing the value of this
        attribute:

        If a medium is attached to a virtual machine (either in the
        current state or in one of the snapshots), its type cannot be
        changed.

        As long as the medium has children, its type cannot be set
        to :py:attr:`MediumType.writethrough` .

        The type of all differencing media is
        :py:attr:`MediumType.normal`  and cannot be changed.



        The type of a newly created or opened medium is set to
        :py:attr:`MediumType.normal` , except for DVD and floppy media,
        which have a type of :py:attr:`MediumType.writethrough` .
        """
        ret = self._get_attr("type")
        return MediumType(ret)

    @type_p.setter
    def type_p(self, value):
        if not isinstance(value, MediumType):
            raise TypeError("value is not an instance of MediumType")
        return self._set_attr("type", value)

    @property
    def allowed_types(self):
        """Get MediumType value for 'allowedTypes'
        Returns which medium types can selected for this medium.
        """
        ret = self._get_attr("allowedTypes")
        return [MediumType(a) for a in ret]

    @property
    def parent(self):
        """Get IMedium value for 'parent'
        Parent of this medium (the medium this medium is directly based
        on).

        Only differencing media have parents. For base (non-differencing)
        media, @c null is returned.
        """
        ret = self._get_attr("parent")
        return IMedium(ret)

    @property
    def children(self):
        """Get IMedium value for 'children'
        Children of this medium (all differencing media directly based
        on this medium). A @c null array is returned if this medium
        does not have any children.
        """
        ret = self._get_attr("children")
        return [IMedium(a) for a in ret]

    @property
    def base(self):
        """Get IMedium value for 'base'
        Base medium of this medium.

        If this is a differencing medium, its base medium is the medium
        the given medium branch starts from. For all other types of media, this
        property returns the medium object itself (i.e. the same object this
        property is read on).
        """
        ret = self._get_attr("base")
        return IMedium(ret)

    @property
    def read_only(self):
        """Get bool value for 'readOnly'
        Returns @c true if this medium is read-only and @c false otherwise.

        A medium is considered to be read-only when its contents cannot be
        modified without breaking the integrity of other parties that depend on
        this medium such as its child media or snapshots of virtual machines
        where this medium is attached to these machines. If there are no
        children and no such snapshots then there is no dependency and the
        medium is not read-only.

        The value of this attribute can be used to determine the kind of the
        attachment that will take place when attaching this medium to a
        virtual machine. If the value is @c false then the medium will
        be attached directly. If the value is @c true then the medium
        will be attached indirectly by creating a new differencing child
        medium for that. See the interface description for more information.

        Note that all :py:attr:`MediumType.immutable` Immutable media
        are always read-only while all
        :py:attr:`MediumType.writethrough` Writethrough media are
        always not.


        The read-only condition represented by this attribute is related to
        the medium type and usage, not to the current
        :py:func:`IMedium.state` medium state and not to the read-only
        state of the storage unit.
        """
        ret = self._get_attr("readOnly")
        return ret

    @property
    def logical_size(self):
        """Get int value for 'logicalSize'
        Logical size of this medium (in bytes), as reported to the
        guest OS running inside the virtual machine this medium is
        attached to. The logical size is defined when the medium is created
        and cannot be changed later.


        For media whose state is :py:func:`state`  is :py:attr:`MediumState.inaccessible` , the value of this property is the
        last known logical size. For :py:attr:`MediumState.not_created`
        media, the returned value is zero.
        """
        ret = self._get_attr("logicalSize")
        return ret

    @property
    def auto_reset(self):
        """Get or set bool value for 'autoReset'
        Whether this differencing medium will be automatically reset each
        time a virtual machine it is attached to is powered up. This
        attribute is automatically set to @c true for the last
        differencing image of an "immutable" medium (see
        :py:class:`MediumType` ).

        See :py:func:`reset`  for more information about resetting
        differencing media.


        Reading this property on a base (non-differencing) medium will
        always @c false. Changing the value of this property in this
        case is not supported.
        """
        ret = self._get_attr("autoReset")
        return ret

    @auto_reset.setter
    def auto_reset(self, value):
        if not isinstance(value, bool):
            raise TypeError("value is not an instance of bool")
        return self._set_attr("autoReset", value)

    @property
    def last_access_error(self):
        """Get str value for 'lastAccessError'
        Text message that represents the result of the last accessibility
        check performed by :py:func:`refresh_state` .

        An empty string is returned if the last accessibility check
        was successful or has not yet been called. As a result, if
        :py:func:`state`  is "Inaccessible" and this attribute is empty,
        then :py:func:`refresh_state`  has yet to be called; this is the
        default value of media after VirtualBox initialization.
        A non-empty string indicates a failure and should normally describe
        a reason of the failure (for example, a file read error).
        """
        ret = self._get_attr("lastAccessError")
        return ret

    @property
    def machine_ids(self):
        """Get str value for 'machineIds'
        Array of UUIDs of all machines this medium is attached to.

        A @c null array is returned if this medium is not attached to any
        machine or to any machine's snapshot.


        The returned array will include a machine even if this medium is not
        attached to that machine in the current state but attached to it in
        one of the machine's snapshots. See :py:func:`get_snapshot_ids`  for
        details.
        """
        ret = self._get_attr("machineIds")
        return ret

    def set_ids(self, set_image_id, image_id, set_parent_id, parent_id):
        """Changes the UUID and parent UUID for a hard disk medium.

        in set_image_id of type bool
            Select whether a new image UUID is set or not.

        in image_id of type str
            New UUID for the image. If an empty string is passed, then a new
            UUID is automatically created, provided that @a setImageId is @c true.
            Specifying a zero UUID is not allowed.

        in set_parent_id of type bool
            Select whether a new parent UUID is set or not.

        in parent_id of type str
            New parent UUID for the image. If an empty string is passed, then a
            new UUID is automatically created, provided @a setParentId is
            @c true. A zero UUID is valid.

        raises :class:`OleErrorInvalidarg`
            Invalid parameter combination.

        raises :class:`VBoxErrorNotSupported`
            Medium is not a hard disk medium.

        """
        if not isinstance(set_image_id, bool):
            raise TypeError("set_image_id can only be an instance of type bool")
        if not isinstance(image_id, basestring):
            raise TypeError("image_id can only be an instance of type basestring")
        if not isinstance(set_parent_id, bool):
            raise TypeError("set_parent_id can only be an instance of type bool")
        if not isinstance(parent_id, basestring):
            raise TypeError("parent_id can only be an instance of type basestring")
        self._call("setIds", in_p=[set_image_id, image_id, set_parent_id, parent_id])

    def refresh_state(self):
        """If the current medium state (see :py:class:`MediumState` ) is one of
        "Created", "Inaccessible" or "LockedRead", then this performs an
        accessibility check on the medium and sets the value of the :py:func:`state`
        attribute accordingly; that value is also returned for convenience.

        For all other state values, this does not perform a refresh but returns
        the state only.

        The refresh, if performed, may take a long time (several seconds or even
        minutes, depending on the storage unit location and format) because it performs an
        accessibility check of the storage unit. This check may cause a significant
        delay if the storage unit of the given medium is, for example, a file located
        on a network share which is not currently accessible due to connectivity
        problems. In that case, the call will not return until a timeout
        interval defined by the host OS for this operation expires. For this reason,
        it is recommended to never read this attribute on the main UI thread to avoid
        making the UI unresponsive.

        If the last known state of the medium is "Created" and the accessibility
        check fails, then the state would be set to "Inaccessible", and
        :py:func:`last_access_error`  may be used to get more details about the
        failure. If the state of the medium is "LockedRead", then it remains the
        same, and a non-empty value of :py:func:`last_access_error`  will
        indicate a failed accessibility check in this case.

        Note that not all medium states are applicable to all medium types.

        return state of type :class:`MediumState`
            New medium state.

        """
        state = self._call("refreshState")
        state = MediumState(state)
        return state

    def get_snapshot_ids(self, machine_id):
        """Returns an array of UUIDs of all snapshots of the given machine where
        this medium is attached to.

        If the medium is attached to the machine in the current state, then the
        first element in the array will always be the ID of the queried machine
        (i.e. the value equal to the @c machineId argument), followed by
        snapshot IDs (if any).

        If the medium is not attached to the machine in the current state, then
        the array will contain only snapshot IDs.

        The returned array may be @c null if this medium is not attached
        to the given machine at all, neither in the current state nor in one of
        the snapshots.

        in machine_id of type str
            UUID of the machine to query.

        return snapshot_ids of type str
            Array of snapshot UUIDs of the given machine using this medium.

        """
        if not isinstance(machine_id, basestring):
            raise TypeError("machine_id can only be an instance of type basestring")
        snapshot_ids = self._call("getSnapshotIds", in_p=[machine_id])
        return snapshot_ids

    def lock_read(self):
        """Locks this medium for reading.

                A read lock is shared: many clients can simultaneously lock the
                same medium for reading unless it is already locked for writing (see
                :py:func:`lock_write` ) in which case an error is returned.

                When the medium is locked for reading, it cannot be modified
                from within VirtualBox. This means that any method that changes
                the properties of this medium or contents of the storage unit
                will return an error (unless explicitly stated otherwise). That
                includes an attempt to start a virtual machine that wants to
                write to the medium.

                When the virtual machine is started up, it locks for reading all
                media it uses in read-only mode. If some medium cannot be locked
                for reading, the startup procedure will fail.
                A medium is typically locked for reading while it is used by a running
                virtual machine but has a depending differencing image that receives
                the actual write operations. This way one base medium can have
                multiple child differencing images which can be written to
                simultaneously. Read-only media such as DVD and floppy images are
                also locked for reading only (so they can be in use by multiple
                machines simultaneously).

                A medium is also locked for reading when it is the source of a
                write operation such as :py:func:`clone_to`  or :py:func:`merge_to` .

                The medium locked for reading must be unlocked by abandoning the
                returned token object, see :py:class:`IToken` . Calls to
                :py:func:`lock_read`  can be nested and the lock is actually released
                when all callers have abandoned the token.

                This method sets the medium state (see :py:func:`state` ) to
                "LockedRead" on success. The medium's previous state must be
                one of "Created", "Inaccessible" or "LockedRead".

                Locking an inaccessible medium is not an error; this method performs
                a logical lock that prevents modifications of this medium through
                the VirtualBox API, not a physical file-system lock of the underlying
                storage unit.

                This method returns the current state of the medium
                *before* the operation.

                return token of type :class:`IToken`
                    Token object, when this is released (reference count reaches 0) then
                    the lock count is decreased. The lock is released when the lock count
                    reaches 0.

                raises :class:`VBoxErrorInvalidObjectState`
                    Invalid medium state (e.g. not created, locked, inaccessible,
        creating, deleting).

        """
        token = self._call("lockRead")
        token = IToken(token)
        return token

    def lock_write(self):
        """Locks this medium for writing.

                A write lock, as opposed to :py:func:`lock_read` , is
                exclusive: there may be only one client holding a write lock,
                and there may be no read locks while the write lock is held.
                As a result, read-locking fails if a write lock is held, and
                write-locking fails if either a read or another write lock is held.

                When a medium is locked for writing, it cannot be modified
                from within VirtualBox, and it is not guaranteed that the values
                of its properties are up-to-date. Any method that changes the
                properties of this medium or contents of the storage unit will
                return an error (unless explicitly stated otherwise).

                When a virtual machine is started up, it locks for writing all
                media it uses to write data to. If any medium could not be locked
                for writing, the startup procedure will fail. If a medium has
                differencing images, then while the machine is running, only
                the last ("leaf") differencing image is locked for writing,
                whereas its parents are locked for reading only.

                A medium is also locked for writing when it is the target of a
                write operation such as :py:func:`clone_to`  or :py:func:`merge_to` .

                The medium locked for writing must be unlocked by abandoning the
                returned token object, see :py:class:`IToken` . Write locks
                *cannot* be nested.

                This method sets the medium state (see :py:func:`state` ) to
                "LockedWrite" on success. The medium's previous state must be
                either "Created" or "Inaccessible".

                Locking an inaccessible medium is not an error; this method performs
                a logical lock that prevents modifications of this medium through
                the VirtualBox API, not a physical file-system lock of the underlying
                storage unit.

                return token of type :class:`IToken`
                    Token object, when this is released (reference count reaches 0) then
                    the lock is released.

                raises :class:`VBoxErrorInvalidObjectState`
                    Invalid medium state (e.g. not created, locked, inaccessible,
        creating, deleting).

        """
        token = self._call("lockWrite")
        token = IToken(token)
        return token

    def close(self):
        """Closes this medium.

                The medium must not be attached to any known virtual machine
                and must not have any known child media, otherwise the
                operation will fail.

                When the medium is successfully closed, it is removed from
                the list of registered media, but its storage unit is not
                deleted. In particular, this means that this medium can
                later be opened again using the :py:func:`IVirtualBox.open_medium`
                call.

                Note that after this method successfully returns, the given medium
                object becomes uninitialized. This means that any attempt
                to call any of its methods or attributes will fail with the
                "Object not ready" (E_ACCESSDENIED) error.

                raises :class:`VBoxErrorInvalidObjectState`
                    Invalid medium state (other than not created, created or
        inaccessible).

                raises :class:`VBoxErrorObjectInUse`
                    Medium attached to virtual machine.

                raises :class:`VBoxErrorFileError`
                    Settings file not accessible.

                raises :class:`VBoxErrorXmlError`
                    Could not parse the settings file.

        """
        self._call("close")

    def get_property(self, name):
        """Returns the value of the custom medium property with the given name.

        The list of all properties supported by the given medium format can
        be obtained with :py:func:`IMediumFormat.describe_properties` .

        If this method returns an empty string in @a value, the requested
        property is supported but currently not assigned any value.

        in name of type str
            Name of the property to get.

        return value of type str
            Current property value.

        raises :class:`VBoxErrorObjectNotFound`
            Requested property does not exist (not supported by the format).

        raises :class:`OleErrorInvalidarg`
            @a name is @c null or empty.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        value = self._call("getProperty", in_p=[name])
        return value

    def set_property(self, name, value):
        """Sets the value of the custom medium property with the given name.

        The list of all properties supported by the given medium format can
        be obtained with :py:func:`IMediumFormat.describe_properties` .

        Setting the property value to @c null or an empty string is
        equivalent to deleting the existing value. A default value (if it is
        defined for this property) will be used by the format backend in this
        case.

        in name of type str
            Name of the property to set.

        in value of type str
            Property value to set.

        raises :class:`VBoxErrorObjectNotFound`
            Requested property does not exist (not supported by the format).

        raises :class:`OleErrorInvalidarg`
            @a name is @c null or empty.

        """
        if not isinstance(name, basestring):
            raise TypeError("name can only be an instance of type basestring")
        if not isinstance(value, basestring):
            raise TypeError("value can only be an instance of type basestring")
        self._call("setProperty", in_p=[name, value])

    def get_properties(self, names):
        """Returns values for a group of properties in one call.

        The names of the properties to get are specified using the @a names
        argument which is a list of comma-separated property names or
        an empty string if all properties are to be returned.
        Currently the value of this argument is ignored and the method
        always returns all existing properties.

        The list of all properties supported by the given medium format can
        be obtained with :py:func:`IMediumFormat.describe_properties` .

        The method returns two arrays, the array of property names corresponding
        to the @a names argument and the current values of these properties.
        Both arrays have the same number of elements with each element at the
        given index in the first array corresponds to an element at the same
        index in the second array.

        For properties that do not have assigned values, an empty string is
        returned at the appropriate index in the @a returnValues array.

        in names of type str
            Names of properties to get.

        out return_names of type str
            Names of returned properties.

        return return_values of type str
            Values of returned properties.

        """
        if not isinstance(names, basestring):
            raise TypeError("names can only be an instance of type basestring")
        (return_values, return_names) = self._call("getProperties", in_p=[names])
        return (return_values, return_names)

    def set_properties(self, names, values):
        """Sets values for a group of properties in one call.

        The names of the properties to set are passed in the @a names
        array along with the new values for them in the @a values array. Both
        arrays have the same number of elements with each element at the given
        index in the first array corresponding to an element at the same index
        in the second array.

        If there is at least one property name in @a names that is not valid,
        the method will fail before changing the values of any other properties
        from the @a names array.

        Using this method over :py:func:`set_property`  is preferred if you
        need to set several properties at once since it is more efficient.

        The list of all properties supported by the given medium format can
        be obtained with :py:func:`IMediumFormat.describe_properties` .

        Setting the property value to @c null or an empty string is equivalent
        to deleting the existing value. A default value (if it is defined for
        this property) will be used by the format backend in this case.

        in names of type str
            Names of properties to set.

        in values of type str
            Values of properties to set.

        """
        if not isinstance(names, list):
            raise TypeError("names can only be an instance of type list")
        for a in names[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        if not isinstance(values, list):
            raise TypeError("values can only be an instance of type list")
        for a in values[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        self._call("setProperties", in_p=[names, values])

    def create_base_storage(self, logical_size, variant):
        """Starts creating a hard disk storage unit (fixed/dynamic, according
        to the variant flags) in the background. The previous storage unit
        created for this object, if any, must first be deleted using
        :py:func:`delete_storage` , otherwise the operation will fail.

        Before the operation starts, the medium is placed in
        :py:attr:`MediumState.creating`  state. If the create operation
        fails, the medium will be placed back in :py:attr:`MediumState.not_created`
        state.

        After the returned progress object reports that the operation has
        successfully completed, the medium state will be set to :py:attr:`MediumState.created` , the medium will be remembered by this
        VirtualBox installation and may be attached to virtual machines.

        in logical_size of type int
            Maximum logical size of the medium in bytes.

        in variant of type :class:`MediumVariant`
            Exact image variant which should be created (as a combination of
            :py:class:`MediumVariant`  flags).

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorNotSupported`
            The variant of storage creation operation is not supported. See

        """
        if not isinstance(logical_size, baseinteger):
            raise TypeError("logical_size can only be an instance of type baseinteger")
        if not isinstance(variant, list):
            raise TypeError("variant can only be an instance of type list")
        for a in variant[:10]:
            if not isinstance(a, MediumVariant):
                raise TypeError("array can only contain objects of type MediumVariant")
        progress = self._call("createBaseStorage", in_p=[logical_size, variant])
        progress = IProgress(progress)
        return progress

    def delete_storage(self):
        """Starts deleting the storage unit of this medium.

                The medium must not be attached to any known virtual machine and must
                not have any known child media, otherwise the operation will fail.
                It will also fail if there is no storage unit to delete or if deletion
                is already in progress, or if the medium is being in use (locked for
                read or for write) or inaccessible. Therefore, the only valid state for
                this operation to succeed is :py:attr:`MediumState.created` .

                Before the operation starts, the medium is placed in
                :py:attr:`MediumState.deleting`  state and gets removed from the list
                of remembered hard disks (media registry). If the delete operation
                fails, the medium will be remembered again and placed back to
                :py:attr:`MediumState.created`  state.

                After the returned progress object reports that the operation is
                complete, the medium state will be set to
                :py:attr:`MediumState.not_created`  and you will be able to use one of
                the storage creation methods to create it again.

                :py:func:`close`




                If the deletion operation fails, it is not guaranteed that the storage
                unit still exists. You may check the :py:func:`IMedium.state`  value
                to answer this question.

                return progress of type :class:`IProgress`
                    Progress object to track the operation completion.

                raises :class:`VBoxErrorObjectInUse`
                    Medium is attached to a virtual machine.

                raises :class:`VBoxErrorNotSupported`
                    Storage deletion is not allowed because neither of storage creation
        operations are supported. See

        """
        progress = self._call("deleteStorage")
        progress = IProgress(progress)
        return progress

    def create_diff_storage(self, target, variant):
        """Starts creating an empty differencing storage unit based on this
        medium in the format and at the location defined by the @a target
        argument.

        The target medium must be in :py:attr:`MediumState.not_created`
        state (i.e. must not have an existing storage unit). Upon successful
        completion, this operation will set the type of the target medium to
        :py:attr:`MediumType.normal`  and create a storage unit necessary to
        represent the differencing medium data in the given format (according
        to the storage format of the target object).

        After the returned progress object reports that the operation is
        successfully complete, the target medium gets remembered by this
        VirtualBox installation and may be attached to virtual machines.


        The medium will be set to :py:attr:`MediumState.locked_read`
        state for the duration of this operation.

        in target of type :class:`IMedium`
            Target medium.

        in variant of type :class:`MediumVariant`
            Exact image variant which should be created (as a combination of
            :py:class:`MediumVariant`  flags).

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorObjectInUse`
            Medium not in @c NotCreated state.

        """
        if not isinstance(target, IMedium):
            raise TypeError("target can only be an instance of type IMedium")
        if not isinstance(variant, list):
            raise TypeError("variant can only be an instance of type list")
        for a in variant[:10]:
            if not isinstance(a, MediumVariant):
                raise TypeError("array can only contain objects of type MediumVariant")
        progress = self._call("createDiffStorage", in_p=[target, variant])
        progress = IProgress(progress)
        return progress

    def merge_to(self, target):
        """Starts merging the contents of this medium and all intermediate
        differencing media in the chain to the given target medium.

        The target medium must be either a descendant of this medium or
        its ancestor (otherwise this method will immediately return a failure).
        It follows that there are two logical directions of the merge operation:
        from ancestor to descendant (*forward merge*) and from descendant to
        ancestor (*backward merge*). Let us consider the following medium
        chain:

        Base <- Diff_1 <- Diff_2

        Here, calling this method on the Base medium object with
        Diff_2 as an argument will be a forward merge; calling it on
        Diff_2 with Base as an argument will be a backward
        merge. Note that in both cases the contents of the resulting medium
        will be the same, the only difference is the medium object that takes
        the result of the merge operation. In case of the forward merge in the
        above example, the result will be written to Diff_2; in case of
        the backward merge, the result will be written to Base. In
        other words, the result of the operation is always stored in the target
        medium.

        Upon successful operation completion, the storage units of all media in
        the chain between this (source) medium and the target medium, including
        the source medium itself, will be automatically deleted and the
        relevant medium objects (including this medium) will become
        uninitialized. This means that any attempt to call any of
        their methods or attributes will fail with the
        "Object not ready" (E_ACCESSDENIED) error. Applied to the above
        example, the forward merge of Base to Diff_2 will
        delete and uninitialize both Base and Diff_1 media.
        Note that Diff_2 in this case will become a base medium
        itself since it will no longer be based on any other medium.

        Considering the above, all of the following conditions must be met in
        order for the merge operation to succeed:


        Neither this (source) medium nor any intermediate
        differencing medium in the chain between it and the target
        medium is attached to any virtual machine.


        Neither the source medium nor the target medium is an
        :py:attr:`MediumType.immutable`  medium.


        The part of the medium tree from the source medium to the
        target medium is a linear chain, i.e. all medium in this
        chain have exactly one child which is the next medium in this
        chain. The only exception from this rule is the target medium in
        the forward merge operation; it is allowed to have any number of
        child media because the merge operation will not change its
        logical contents (as it is seen by the guest OS or by children).


        None of the involved media are in
        :py:attr:`MediumState.locked_read`  or
        :py:attr:`MediumState.locked_write`  state.




        This (source) medium and all intermediates will be placed to :py:attr:`MediumState.deleting`  state and the target medium will be
        placed to :py:attr:`MediumState.locked_write`  state and for the
        duration of this operation.

        in target of type :class:`IMedium`
            Target medium.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        """
        if not isinstance(target, IMedium):
            raise TypeError("target can only be an instance of type IMedium")
        progress = self._call("mergeTo", in_p=[target])
        progress = IProgress(progress)
        return progress

    def clone_to(self, target, variant, parent):
        """Starts creating a clone of this medium in the format and at the
        location defined by the @a target argument.

        The target medium must be either in :py:attr:`MediumState.not_created`
        state (i.e. must not have an existing storage unit) or in
        :py:attr:`MediumState.created`  state (i.e. created and not locked, and
        big enough to hold the data or else the copy will be partial). Upon
        successful completion, the cloned medium will contain exactly the
        same sector data as the medium being cloned, except that in the
        first case a new UUID for the clone will be randomly generated, and in
        the second case the UUID will remain unchanged.

        The @a parent argument defines which medium will be the parent
        of the clone. Passing a @c null reference indicates that the clone will
        be a base image, i.e. completely independent. It is possible to specify
        an arbitrary medium for this parameter, including the parent of the
        medium which is being cloned. Even cloning to a child of the source
        medium is possible. Note that when cloning to an existing image, the
        @a parent argument is ignored.

        After the returned progress object reports that the operation is
        successfully complete, the target medium gets remembered by this
        VirtualBox installation and may be attached to virtual machines.


        This medium will be placed to :py:attr:`MediumState.locked_read`
        state for the duration of this operation.

        in target of type :class:`IMedium`
            Target medium.

        in variant of type :class:`MediumVariant`
            Exact image variant which should be created (as a combination of
            :py:class:`MediumVariant`  flags).

        in parent of type :class:`IMedium`
            Parent of the cloned medium.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`OleErrorNotimpl`
            The specified cloning variant is not supported at the moment.

        """
        if not isinstance(target, IMedium):
            raise TypeError("target can only be an instance of type IMedium")
        if not isinstance(variant, list):
            raise TypeError("variant can only be an instance of type list")
        for a in variant[:10]:
            if not isinstance(a, MediumVariant):
                raise TypeError("array can only contain objects of type MediumVariant")
        if not isinstance(parent, IMedium):
            raise TypeError("parent can only be an instance of type IMedium")
        progress = self._call("cloneTo", in_p=[target, variant, parent])
        progress = IProgress(progress)
        return progress

    def clone_to_base(self, target, variant):
        """Starts creating a clone of this medium in the format and at the
        location defined by the @a target argument.

        The target medium must be either in :py:attr:`MediumState.not_created`
        state (i.e. must not have an existing storage unit) or in
        :py:attr:`MediumState.created`  state (i.e. created and not locked, and
        big enough to hold the data or else the copy will be partial). Upon
        successful completion, the cloned medium will contain exactly the
        same sector data as the medium being cloned, except that in the
        first case a new UUID for the clone will be randomly generated, and in
        the second case the UUID will remain unchanged.

        The @a parent argument defines which medium will be the parent
        of the clone. In this case the clone will be a base image, i.e.
        completely independent. It is possible to specify an arbitrary
        medium for this parameter, including the parent of the
        medium which is being cloned. Even cloning to a child of the source
        medium is possible. Note that when cloning to an existing image, the
        @a parent argument is ignored.

        After the returned progress object reports that the operation is
        successfully complete, the target medium gets remembered by this
        VirtualBox installation and may be attached to virtual machines.


        This medium will be placed to :py:attr:`MediumState.locked_read`
        state for the duration of this operation.

        in target of type :class:`IMedium`
            Target medium.

        in variant of type :class:`MediumVariant`
            :py:class:`MediumVariant`  flags).

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`OleErrorNotimpl`
            The specified cloning variant is not supported at the moment.

        """
        if not isinstance(target, IMedium):
            raise TypeError("target can only be an instance of type IMedium")
        if not isinstance(variant, list):
            raise TypeError("variant can only be an instance of type list")
        for a in variant[:10]:
            if not isinstance(a, MediumVariant):
                raise TypeError("array can only contain objects of type MediumVariant")
        progress = self._call("cloneToBase", in_p=[target, variant])
        progress = IProgress(progress)
        return progress

    def move_to(self, location):
        """Changes the location of this medium. Some medium types may support
        changing the storage unit location by simply changing the value of the
        associated property. In this case the operation is performed
        immediately, and @a progress is returning a @c null reference.
        Otherwise on success there is a progress object returned, which
        signals progress and completion of the operation. This distinction is
        necessary because for some formats the operation is very fast, while
        for others it can be very slow (moving the image file by copying all
        data), and in the former case it'd be a waste of resources to create
        a progress object which will immediately signal completion.

        When setting a location for a medium which corresponds to a/several
        regular file(s) in the host's file system, the given file name may be
        either relative to the :py:func:`IVirtualBox.home_folder` VirtualBox
        home folder or absolute. Note that if the given location
        specification does not contain the file extension part then a proper
        default extension will be automatically appended by the implementation
        depending on the medium type.

        in location of type str
            New location.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`OleErrorNotimpl`
            The operation is not implemented yet.

        raises :class:`VBoxErrorNotSupported`
            Medium format does not support changing the location.

        """
        if not isinstance(location, basestring):
            raise TypeError("location can only be an instance of type basestring")
        progress = self._call("moveTo", in_p=[location])
        progress = IProgress(progress)
        return progress

    def compact(self):
        """Starts compacting of this medium. This means that the medium is
                transformed into a possibly more compact storage representation.
                This potentially creates temporary images, which can require a
                substantial amount of additional disk space.

                This medium will be placed to :py:attr:`MediumState.locked_write`
                state and all its parent media (if any) will be placed to
                :py:attr:`MediumState.locked_read`  state for the duration of this
                operation.

                Please note that the results can be either returned straight away,
                or later as the result of the background operation via the object
                returned via the @a progress parameter.

                return progress of type :class:`IProgress`
                    Progress object to track the operation completion.

                raises :class:`VBoxErrorNotSupported`
                    Medium format does not support compacting (but potentially
        needs it).

        """
        progress = self._call("compact")
        progress = IProgress(progress)
        return progress

    def resize(self, logical_size):
        """Starts resizing this medium. This means that the nominal size of the
        medium is set to the new value. Both increasing and decreasing the
        size is possible, and there are no safety checks, since VirtualBox
        does not make any assumptions about the medium contents.

        Resizing usually needs additional disk space, and possibly also
        some temporary disk space. Note that resize does not create a full
        temporary copy of the medium, so the additional disk space requirement
        is usually much lower than using the clone operation.

        This medium will be placed to :py:attr:`MediumState.locked_write`
        state for the duration of this operation.

        Please note that the results can be either returned straight away,
        or later as the result of the background operation via the object
        returned via the @a progress parameter.

        in logical_size of type int
            New nominal capacity of the medium in bytes.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorNotSupported`
            Medium format does not support resizing.

        """
        if not isinstance(logical_size, baseinteger):
            raise TypeError("logical_size can only be an instance of type baseinteger")
        progress = self._call("resize", in_p=[logical_size])
        progress = IProgress(progress)
        return progress

    def reset(self):
        """Starts erasing the contents of this differencing medium.

        This operation will reset the differencing medium to its initial
        state when it does not contain any sector data and any read operation is
        redirected to its parent medium. This automatically gets called
        during VM power-up for every medium whose :py:func:`auto_reset`
        attribute is @c true.

        The medium will be write-locked for the duration of this operation (see
        :py:func:`lock_write` ).

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorNotSupported`
            This is not a differencing medium.

        raises :class:`VBoxErrorInvalidObjectState`
            Medium is not in

        """
        progress = self._call("reset")
        progress = IProgress(progress)
        return progress

    def change_encryption(
        self, current_password, cipher, new_password, new_password_id
    ):
        """Starts encryption of this medium. This means that the stored data in the
                medium is encrypted.

                This medium will be placed to :py:attr:`MediumState.locked_write`
                state.

                Please note that the results can be either returned straight away,
                or later as the result of the background operation via the object
                returned via the @a progress parameter.

                in current_password of type str
                    The current password the medium is protected with. Use an empty string to indicate
                    that the medium isn't encrypted.

                in cipher of type str
                    The cipher to use for encryption. An empty string indicates no encryption for the
                    result.

                in new_password of type str
                    The new password the medium should be protected with. An empty password and password ID
                    will result in the medium being encrypted with the current password.

                in new_password_id of type str
                    The ID of the new password when unlocking the medium.

                return progress of type :class:`IProgress`
                    Progress object to track the operation completion.

                raises :class:`VBoxErrorNotSupported`
                    Encryption is not supported for this medium because it is attached to more than one VM
        or has children.

        """
        if not isinstance(current_password, basestring):
            raise TypeError(
                "current_password can only be an instance of type basestring"
            )
        if not isinstance(cipher, basestring):
            raise TypeError("cipher can only be an instance of type basestring")
        if not isinstance(new_password, basestring):
            raise TypeError("new_password can only be an instance of type basestring")
        if not isinstance(new_password_id, basestring):
            raise TypeError(
                "new_password_id can only be an instance of type basestring"
            )
        progress = self._call(
            "changeEncryption",
            in_p=[current_password, cipher, new_password, new_password_id],
        )
        progress = IProgress(progress)
        return progress

    def get_encryption_settings(self):
        """Returns the encryption settings for this medium.

        out cipher of type str
            The cipher used for encryption.

        return password_id of type str
            The ID of the password when unlocking the medium.

        raises :class:`VBoxErrorNotSupported`
            Encryption is not configured for this medium.

        """
        (password_id, cipher) = self._call("getEncryptionSettings")
        return (password_id, cipher)

    def check_encryption_password(self, password):
        """Checks whether the supplied password is correct for the medium.

        in password of type str
            The password to check.

        raises :class:`VBoxErrorNotSupported`
            Encryption is not configured for this medium.

        raises :class:`VBoxErrorPasswordIncorrect`
            The given password is incorrect.

        """
        if not isinstance(password, basestring):
            raise TypeError("password can only be an instance of type basestring")
        self._call("checkEncryptionPassword", in_p=[password])

    def open_for_io(self, writable, password):
        """Open the medium for I/O.

        in writable of type bool
            Set this to open the medium for both reading and writing.  When
            not set the medium is opened readonly.

        in password of type str
            Password for accessing an encrypted medium. Must be empty if not encrypted.

        return medium_io of type :class:`IMediumIO`
            Medium I/O object.

        """
        if not isinstance(writable, bool):
            raise TypeError("writable can only be an instance of type bool")
        if not isinstance(password, basestring):
            raise TypeError("password can only be an instance of type basestring")
        medium_io = self._call("openForIO", in_p=[writable, password])
        medium_io = IMediumIO(medium_io)
        return medium_io


class IMediumFormat(Interface):
    """
    The IMediumFormat interface represents a medium format.

    Each medium format has an associated backend which is used to handle
    media stored in this format. This interface provides information
    about the properties of the associated backend.

    Each medium format is identified by a string represented by the
    :py:func:`id_p`  attribute. This string is used in calls like
    :py:func:`IVirtualBox.create_medium`  to specify the desired
    format.

    The list of all supported medium formats can be obtained using
    :py:func:`ISystemProperties.medium_formats` .

    :py:class:`IMedium`
    """

    __uuid__ = "11be93c7-a862-4dc9-8c89-bf4ba74a886a"
    __wsmap__ = "managed"

    @property
    def id_p(self):
        """Get str value for 'id'
        Identifier of this format.

        The format identifier is a non-@c null non-empty ASCII string. Note that
        this string is case-insensitive. This means that, for example, all of
        the following strings:

        ::

             "VDI"
             "vdi"
             "VdI"


        refer to the same medium format.

        This string is used in methods of other interfaces where it is necessary
        to specify a medium format, such as
        :py:func:`IVirtualBox.create_medium` .
        """
        ret = self._get_attr("id")
        return ret

    @property
    def name(self):
        """Get str value for 'name'
        Human readable description of this format.

        Mainly for use in file open dialogs.
        """
        ret = self._get_attr("name")
        return ret

    @property
    def capabilities(self):
        """Get MediumFormatCapabilities value for 'capabilities'
        Capabilities of the format as an array of the flags.

        For the meaning of individual capability flags see
        :py:class:`MediumFormatCapabilities` .
        """
        ret = self._get_attr("capabilities")
        return [MediumFormatCapabilities(a) for a in ret]

    def describe_file_extensions(self):
        """Returns two arrays describing the supported file extensions.

        The first array contains the supported extensions and the seconds one
        the type each extension supports. Both have the same size.

        Note that some backends do not work on files, so this array may be
        empty.

        :py:func:`IMediumFormat.capabilities`

        out extensions of type str
            The array of supported extensions.

        out types of type :class:`DeviceType`
            The array which indicates the device type for every given extension.

        """
        (extensions, types) = self._call("describeFileExtensions")
        types = [DeviceType(a) for a in types]
        return (extensions, types)

    def describe_properties(self):
        """Returns several arrays describing the properties supported by this
        format.

        An element with the given index in each array describes one
        property. Thus, the number of elements in each returned array is the
        same and corresponds to the number of supported properties.

        The returned arrays are filled in only if the
        :py:attr:`MediumFormatCapabilities.properties`  flag is set.
        All arguments must be non-@c null.

        :py:class:`DataType` , :py:class:`DataFlags`

        out names of type str
            Array of property names.

        out descriptions of type str
            Array of property descriptions.

        out types of type :class:`DataType`
            Array of property types.

        out flags of type int
            Array of property flags.

        out defaults of type str
            Array of default property values.

        """
        (names, descriptions, types, flags, defaults) = self._call("describeProperties")
        types = [DataType(a) for a in types]
        return (names, descriptions, types, flags, defaults)


class IDataStream(Interface):
    """
    The IDataStream interface is used to retrieve a data stream. It is
    returned by :py:func:`IMediumIO.convert_to_stream` .
    """

    __uuid__ = "a338ed20-58d9-43ae-8b03-c1fd7088ef15"
    __wsmap__ = "managed"

    @property
    def read_size(self):
        """Get int value for 'readSize'
        Recommended size of a read. Requesting a larger read may be
        possible in certain situations, but it is not guaranteed.
        """
        ret = self._get_attr("readSize")
        return ret

    def read(self, size, timeout_ms):
        """Read data from the stream.

        in size of type int
            How many bytes to try read.

        in timeout_ms of type int
            Timeout (in ms) for limiting the wait time for data to be available.
            Pass 0 for an infinite timeout.

        return data of type str
            Array of data read. This may be shorter than the specified size.
            Returning a zero-sized array indicates the end of the stream, if the
            status is successful.

        raises :class:`VBoxErrorTimeout`
            Waiting time has expired.

        """
        if not isinstance(size, baseinteger):
            raise TypeError("size can only be an instance of type baseinteger")
        if not isinstance(timeout_ms, baseinteger):
            raise TypeError("timeout_ms can only be an instance of type baseinteger")
        data = self._call("read", in_p=[size, timeout_ms])
        return data


class IMediumIO(Interface):
    """
    The IMediumIO interface is used to access and modify the content of a
    medium.  It is returned by :py:func:`IMedium.open_for_io` .
    """

    __uuid__ = "e4b301a9-5f86-4d65-ad1b-87ca284fb1c8"
    __wsmap__ = "managed"

    @property
    def medium(self):
        """Get IMedium value for 'medium'
        The open medium.
        """
        ret = self._get_attr("medium")
        return IMedium(ret)

    @property
    def writable(self):
        """Get bool value for 'writable'
        Whether the medium can be written to. (It can always be read from.)
        """
        ret = self._get_attr("writable")
        return ret

    @property
    def explorer(self):
        """Get IVFSExplorer value for 'explorer'
        Returns the virtual file system explorer for the medium.

        This will attempt to recognize the format of the medium content and
        present it as a virtual directory structure to the API user.

        A FAT floppy image will be represented will a single root subdir 'fat12'
        that gives access to the file system content.

        A ISO-9660 image will have one subdir in the root for each format present
        in the image, so the API user can select which data view to access (iso9660,
        rockridge, joliet, udf, hfs, ...).

        A partitioned harddisk image will have subdirs for each partition.  The
        the filesystem content of each partition can be accessed thru the subdirs
        if we have a file system interpreter for it.  There will also be raw files
        for each subdirectory, to provide a simple way of accessing raw partition
        data from an API client.

        Please note that the explorer may show inconsistent information if
        the API user modifies the raw image content after it was opened.
        """
        ret = self._get_attr("explorer")
        return IVFSExplorer(ret)

    def read(self, offset, size):
        """Read data from the medium.

        in offset of type int
            The byte offset into the medium to start reading at.

        in size of type int
            How many bytes to try read.

        return data of type str
            Array of data read. This may be shorter than the specified size.

        """
        if not isinstance(offset, baseinteger):
            raise TypeError("offset can only be an instance of type baseinteger")
        if not isinstance(size, baseinteger):
            raise TypeError("size can only be an instance of type baseinteger")
        data = self._call("read", in_p=[offset, size])
        return data

    def write(self, offset, data):
        """Write data to the medium.

        in offset of type int
            The byte offset into the medium to start reading at.

        in data of type str
            Array of data to write.

        return written of type int
            How many bytes were actually written.

        """
        if not isinstance(offset, baseinteger):
            raise TypeError("offset can only be an instance of type baseinteger")
        if not isinstance(data, list):
            raise TypeError("data can only be an instance of type list")
        for a in data[:10]:
            if not isinstance(a, basestring):
                raise TypeError("array can only contain objects of type basestring")
        written = self._call("write", in_p=[offset, data])
        return written

    def format_fat(self, quick):
        """Formats the medium as FAT.  Generally only useful for floppy images as
        no partition table will be created.

        in quick of type bool
            Quick format it when set.

        """
        if not isinstance(quick, bool):
            raise TypeError("quick can only be an instance of type bool")
        self._call("formatFAT", in_p=[quick])

    def initialize_partition_table(self, format_p, whole_disk_in_one_entry):
        """Writes an empty partition table to the disk.

        in format_p of type :class:`PartitionTableType`
            The partition table format.

        in whole_disk_in_one_entry of type bool
            When @c true a partition table entry for the whole disk is created.
            Otherwise the partition table is empty.

        """
        if not isinstance(format_p, PartitionTableType):
            raise TypeError(
                "format_p can only be an instance of type PartitionTableType"
            )
        if not isinstance(whole_disk_in_one_entry, bool):
            raise TypeError(
                "whole_disk_in_one_entry can only be an instance of type bool"
            )
        self._call("initializePartitionTable", in_p=[format_p, whole_disk_in_one_entry])

    def convert_to_stream(self, format_p, variant, buffer_size):
        """Converts the currently opened image into a stream of the specified
        image type/variant. It is sufficient to open the image in read-only
        mode. Only few types and variants are supported due to the inherent
        restrictions of the output style.

        in format_p of type str
            Identifier of the storage format to use for output.

        in variant of type :class:`MediumVariant`
            The partition table format.

        in buffer_size of type int
            Requested buffer size (in bytes) for efficient conversion. Sizes
            which are too small or too large are silently truncated to suitable
            values. Tens to hundreds of Megabytes are a good choice.

        out stream of type :class:`IDataStream`
            Data stream object for reading the target image.

        return progress of type :class:`IProgress`
            Progress object to track the operation completion.

        raises :class:`VBoxErrorNotSupported`
            The requested format/variant combination cannot handle stream output.

        raises :class:`VBoxErrorFileError`
            An error occurred during the conversion.

        """
        if not isinstance(format_p, basestring):
            raise TypeError("format_p can only be an instance of type basestring")
        if not isinstance(variant, list):
            raise TypeError("variant can only be an instance of type list")
        for a in variant[:10]:
            if not isinstance(a, MediumVariant):
                raise TypeError("array can only contain objects of type MediumVariant")
        if not isinstance(buffer_size, baseinteger):
            raise TypeError("buffer_size can only be an instance of type baseinteger")
        (progress, stream) = self._call(
            "convertToStream", in_p=[format_p, variant, buffer_size]
        )
        progress = IProgress(progress)
        stream = IDataStream(stream)
        return (progress, stream)

    def close(self):
        """Explictly close the medium I/O rather than waiting for garbage
        collection and the destructor.

        This will wait for any pending reads and writes to complete and then
        close down the I/O access without regard for open explorer instances or
        anything like that.

        """
        self._call("close")


class IToken(Interface):
    """
    The IToken interface represents a token passed to an API client, which
    triggers cleanup actions when it is explicitly released by calling the
    :py:func:`abandon`  method (preferred, as it is accurately defined
    when the release happens), or when the object reference count drops
    to 0. The latter way is implicitly used when an API client crashes,
    however the discovery that there was a crash can take rather long,
    depending on the platform (COM needs 6 minutes). So better don't rely
    on the crash behavior too much.
    """

    __uuid__ = "20479eaf-d8ed-44cf-85ac-c83a26c95a4d"
    __wsmap__ = "managed"

    def abandon(self):
        """Releases this token. Cannot be undone in any way, and makes the
        token object unusable (even the :py:func:`dummy`  method will return
        an error), ready for releasing. It is a more defined way than just
        letting the reference count drop to 0, because the latter (depending
        on the platform) can trigger asynchronous cleanup activity.

        """
        self._call("abandon")

    def dummy(self):
        """Purely a NOOP. Useful when using proxy type API bindings (e.g. the
        webservice) which manage objects on behalf of the actual client, using
        an object reference expiration time based garbage collector.

        """
        self._call("dummy")


class IKeyboard(Interface):
    """
    The IKeyboard interface represents the virtual machine's keyboard. Used
    in :py:func:`IConsole.keyboard` .

    Use this interface to send keystrokes or the Ctrl-Alt-Del sequence
    to the virtual machine.
    """

    __uuid__ = "755e6bdf-1640-41f9-bd74-3ef5fd653250"
    __wsmap__ = "managed"

    @property
    def keyboard_le_ds(self):
        """Get KeyboardLED value for 'keyboardLEDs'
        Current status of the guest keyboard LEDs.
        """
        ret = self._get_attr("keyboardLEDs")
        return [KeyboardLED(a) for a in ret]

    def put_scancode(self, scancode):
        """Sends a scancode to the keyboard.

        in scancode of type int

        raises :class:`VBoxErrorIprtError`
            Could not send scan code to virtual keyboard.

        """
        if not isinstance(scancode, baseinteger):
            raise TypeError("scancode can only be an instance of type baseinteger")
        self._call("putScancode", in_p=[scancode])

    def put_scancodes(self, scancodes):
        """Sends an array of scancodes to the keyboard.

        in scancodes of type int

        return codes_stored of type int

        raises :class:`VBoxErrorIprtError`
            Could not send all scan codes to virtual keyboard.

        """
        if not isinstance(scancodes, list):
            raise TypeError("scancodes can only be an instance of type list")
        for a in scancodes[:10]:
            if not isinstance(a, baseinteger):
                raise TypeError("array can only contain objects of type baseinteger")
        codes_stored = self._call("putScancodes", in_p=[scancodes])
        return codes_stored

    def put_cad(self):
        """Sends the Ctrl-Alt-Del sequence to the keyboard. This
        function is nothing special, it is just a convenience function
        calling :py:func:`IKeyboard.put_scancodes`  with the proper scancodes.

        raises :class:`VBoxErrorIprtError`
            Could not send all scan codes to virtual keyboard.

        """
        self._call("putCAD")

    def release_keys(self):
        """Causes the virtual keyboard to release any keys which are
        currently pressed. Useful when host and guest keyboard may be out
        of sync.

        raises :class:`VBoxErrorIprtError`
            Could not release some or all keys.

        """
        self._call("releaseKeys")

    def put_usage_code(self, usage_code, usage_page, key_release):
        """Sends a USB HID usage code and page to the keyboard. The
        keyRelease flag is set when the key is being released.

        in usage_code of type int

        in usage_page of type int

        in key_release of type bool

        raises :class:`VBoxErrorIprtError`
            Could not send usage code to virtual keyboard.

        """
        if not isinstance(usage_code, baseinteger):
            raise TypeError("usage_code can only be an instance of type baseinteger")
        if not isinstance(usage_page, baseinteger):
            raise TypeError("usage_page can only be an instance of type baseinteger")
        if not isinstance(key_release, bool):
            raise TypeError("key_release can only be an instance of type bool")
        self._call("putUsageCode", in_p=[usage_code, usage_page, key_release])

    @property
    def event_source(self):
        """Get IEventSource value for 'eventSource'
        Event source for keyboard events.
        """
        ret = self._get_attr("eventSource")
        return IEventSource(ret)


class IMousePointerShape(Interface):
    """
    The guest mouse pointer description.
    """

    __uuid__ = "1e775ea3-9070-4f9c-b0d5-53054496dbe0"
    __wsmap__ = "managed"

    @property
    def visible(self):
        """Get bool value for 'visible'
        Flag whether the pointer is visible.
        """
        ret = self._get_attr("visible")
        return ret

    @property
    def alpha(self):
        """Get bool value for 'alpha'
        Flag whether the pointer has an alpha channel.
        """
        ret = self._get_attr("alpha")
        return ret

    @property
    def hot_x(self):
        """Get int value for 'hotX'
        The pointer hot spot X coordinate.
        """
        ret = self._get_attr("hotX")
        return ret

    @property
    def hot_y(self):
        """Get int value for 'hotY'
        The pointer hot spot Y coordinate.
        """
        ret = self._get_attr("hotY")
        return ret

    @property
    def width(self):
        """Get int value for 'width'
        Width of the pointer shape in pixels.
        """
        ret = self._get_attr("width")
        return ret

    @property
    def height(self):
        """Get int value for 'height'
        Height of the pointer shape in pixels.
        """
        ret = self._get_attr("height")
        return ret

    @property
    def shape(self):
        """Get str value for 'shape'
        Shape bitmaps.

        The @a shape buffer contains a 1bpp (bits per pixel) AND mask
        followed by a 32bpp XOR (color) mask.

        For pointers without alpha channel the XOR mask pixels are
        32 bit values: (lsb)BGR0(msb). For pointers with alpha channel
        the XOR mask consists of (lsb)BGRA(msb) 32 bit values.

        An AND mask is provided for pointers with alpha channel, so if the
        client does not support alpha, the pointer could be
        displayed as a normal color pointer.

        The AND mask is a 1bpp bitmap with byte aligned scanlines. The
        size of the AND mask therefore is cbAnd = (width + 7) / 8 *
        height. The padding bits at the end of each scanline are
        undefined.

        The XOR mask follows the AND mask on the next 4-byte aligned
        offset: uint8_t *pu8Xor = pu8And + (cbAnd + 3) & ~3.
        Bytes in the gap between the AND and the XOR mask are undefined.
        The XOR mask scanlines have no gap between them and the size of
        the XOR mask is: cbXor = width * 4 * height.


        If @a shape size is 0, then the shape is not known or did not change.
        This can happen if only the pointer visibility is changed.
        """
        ret = self._get_attr("shape")
        return ret


class IMouse(Interface):
    """
    The IMouse interface represents the virtual machine's mouse. Used in
    :py:func:`IConsole.mouse` .

    Through this interface, the virtual machine's virtual mouse can be
    controlled.
    """

    __uuid__ = "10cd08d0-e8b8-4838-b10c-45ba193734c1"
    __wsmap__ = "managed"

    @property
    def absolute_supported(self):
        """Get bool value for 'absoluteSupported'
        Whether the guest OS supports absolute mouse pointer positioning
        or not.

        You can use the :py:class:`IMouseCapabilityChangedEvent`
        event to be instantly informed about changes of this attribute
        during virtual machine execution.

        :py:func:`put_mouse_event_absolute`
        """
        ret = self._get_attr("absoluteSupported")
        return ret

    @property
    def relative_supported(self):
        """Get bool value for 'relativeSupported'
        Whether the guest OS supports relative mouse pointer positioning
        or not.

        You can use the :py:class:`IMouseCapabilityChangedEvent`
        event to be instantly informed about changes of this attribute
        during virtual machine execution.

        :py:func:`put_mouse_event`
        """
        ret = self._get_attr("relativeSupported")
        return ret

    @property
    def multi_touch_supported(self):
        """Get bool value for 'multiTouchSupported'
        Whether the guest OS has enabled the multi-touch reporting device.

        You can use the :py:class:`IMouseCapabilityChangedEvent`
        event to be instantly informed about changes of this attribute
        during virtual machine execution.

        :py:func:`put_mouse_event`
        """
        ret = self._get_attr("multiTouchSupported")
        return ret

    @property
    def needs_host_cursor(self):
        """Get bool value for 'needsHostCursor'
        Whether the guest OS can currently switch to drawing it's own mouse
        cursor on demand.

        You can use the :py:class:`IMouseCapabilityChangedEvent`
        event to be instantly informed about changes of this attribute
        during virtual machine execution.

        :py:func:`put_mouse_event`
        """
        ret = self._get_attr("needsHostCursor")
        return ret

    @property
    def pointer_shape(self):
        """Get IMousePointerShape value for 'pointerShape'
        The current mouse pointer used by the guest.
        """
        ret = self._get_attr("pointerShape")
        return IMousePointerShape(ret)

    def put_mouse_event(self, dx, dy, dz, dw, button_state):
        """Initiates a mouse event using relative pointer movements
        along x and y axis.

        in dx of type int
            Amount of pixels the mouse should move to the right.
            Negative values move the mouse to the left.

        in dy of type int
            Amount of pixels the mouse should move downwards.
            Negative values move the mouse upwards.

        in dz of type int
            Amount of mouse wheel moves.
            Positive values describe clockwise wheel rotations,
            negative values describe counterclockwise rotations.

        in dw of type int
            Amount of horizontal mouse wheel moves.
            Positive values describe a movement to the left,
            negative values describe a movement to the right.

        in button_state of type int
            The current state of mouse buttons. Every bit represents
            a mouse button as follows:

            Bit 0 (0x01)left mouse button
            Bit 1 (0x02)right mouse button
            Bit 2 (0x04)middle mouse button

            A value of 1 means the corresponding button is pressed.
            otherwise it is released.

        raises :class:`OleErrorAccessdenied`
            Console not powered up.

        raises :class:`VBoxErrorIprtError`
            Could not send mouse event to virtual mouse.

        """
        if not isinstance(dx, baseinteger):
            raise TypeError("dx can only be an instance of type baseinteger")
        if not isinstance(dy, baseinteger):
            raise TypeError("dy can only be an instance of type baseinteger")
        if not isinstance(dz, baseinteger):
            raise TypeError("dz can only be an instance of type baseinteger")
        if not isinstance(dw, baseinteger):
            raise TypeError("dw can only be an instance of type baseinteger")
        if not isinstance(button_state, baseinteger):
            raise TypeError("button_state can only be an instance of type baseinteger")
        self._call("putMouseEvent", in_p=[dx, dy, dz, dw, button_state])

    def put_mouse_event_absolute(self, x, y, dz, dw, button_state):
        """Positions the mouse pointer using absolute x and y coordinates.
        These coordinates are expressed in pixels and
        start from [1,1] which corresponds to the top left
        corner of the virtual display.  The values [-1,-1] and
        [0x7fffffff,0x7fffffff] have special meanings as
        respectively "no data" (to signal that the host wishes to report
        absolute pointer data in future) and "out of range" (the host
        pointer is outside of all guest windows).




        This method will have effect only if absolute mouse
        positioning is supported by the guest OS.


        :py:func:`absolute_supported`

        in x of type int
            X coordinate of the pointer in pixels, starting from @c 1.

        in y of type int
            Y coordinate of the pointer in pixels, starting from @c 1.

        in dz of type int
            Amount of mouse wheel moves.
            Positive values describe clockwise wheel rotations,
            negative values describe counterclockwise rotations.

        in dw of type int
            Amount of horizontal mouse wheel moves.
            Positive values describe a movement to the left,
            negative values describe a movement to the right.

        in button_state of type int
            The current state of mouse buttons. Every bit represents
            a mouse button as follows:

            Bit 0 (0x01)left mouse button
            Bit 1 (0x02)right mouse button
            Bit 2 (0x04)middle mouse button

            A value of @c 1 means the corresponding button is pressed.
            otherwise it is released.

        raises :class:`OleErrorAccessdenied`
            Console not powered up.

        raises :class:`VBoxErrorIprtError`
            Could not send mouse event to virtual mouse.

        """
        if not isinstance(x, baseinteger):
            raise TypeError("x can only be an instance of type baseinteger")
        if not isinstance(y, baseinteger):
            raise TypeError("y can only be an instance of type baseinteger")
        if not isinstance(dz, baseinteger):
            raise TypeError("dz can only be an instance of type baseinteger")
        if not isinstance(dw, baseinteger):
            raise TypeError("dw can only be an instance of type baseinteger")
        if not isinstance(button_state, baseinteger):
            raise TypeError("button_state can only be an instance of type baseinteger")
        self._call("putMouseEventAbsolute", in_p=[x, y, dz, dw, button_state])

    def put_event_multi_touch(self, count, contacts, scan_time):
        """Sends a multi-touch pointer event. The coordinates are expressed in
        pixels and start from [1,1] which corresponds to the top left
        corner of the virtual display.




        The guest may not understand or may choose to ignore this event.


        :py:func:`multi_touch_supported`

        in count of type int
            Number of contacts in the event.

        in contacts of type int
            Each array element contains packed information about one contact.
            Bits 0..15: X coordinate in pixels.
            Bits 16..31: Y coordinate in pixels.
            Bits 32..39: contact identifier.
            Bit 40: "in contact" flag, which indicates that there is a contact with the touch surface.
            Bit 41: "in range" flag, the contact is close enough to the touch surface.
            All other bits are reserved for future use and must be set to 0.

        in scan_time of type int
            Timestamp of the event in milliseconds. Only relative time between events is important.

        raises :class:`OleErrorAccessdenied`
            Console not powered up.

        raises :class:`VBoxErrorIprtError`
            Could not send event to virtual device.

        """
        if not isinstance(count, baseinteger):
            raise TypeError("count can only be an instance of type baseinteger")
        if not isinstance(contacts, list):
            raise TypeError("contacts can only be an instance of type list")
        for a in contacts[:10]:
            if not isinstance(a, baseinteger):
                raise TypeError("array can only contain objects of type baseinteger")
        if not isinstance(scan_time, baseinteger):
            raise TypeError("scan_time can only be an instance of type baseinteger")
        self._call("putEventMultiTouch", in_p=[count, contacts, scan_time])

    def put_event_multi_touch_string(self, count, contacts, scan_time):
        """:py:func:`put_event_multi_touch`

        in count of type int
            :py:func:`put_event_multi_touch`

        in contacts of type str
            Contains information about all contacts:
            "id1,x1,y1,inContact1,inRange1;...;idN,xN,yN,inContactN,inRangeN".
            For example for two contacts: "0,10,20,1,1;1,30,40,1,1"

        in scan_time of type int
            :py:func:`put_event_multi_touch`

        """
        if not isinstance(count, baseinteger):
            raise TypeError("count can only be an instance of type baseinteger")
        if not isinstance(contacts, basestring):
            raise TypeError("contacts can only be an instance of type basestring")
        if not isinstance(scan_time, baseinteger):
            raise TypeError("scan_time can only be an instance of type baseinteger")
        self._call("putEventMultiTouchString", in_p=[count, contacts, scan_time])

    @property
    def event_source(self):
        """Get IEventSource value for 'eventSource'
        Event source for mouse events.
        """
        ret = self._get_attr("eventSource")
        return IEventSource(ret)


class IDisplaySourceBitmap(Interface):
    """
    Information about the screen bitmap.
    """

    __uuid__ = "5094f67a-8084-11e9-b185-dbe296e54799"
    __wsmap__ = "suppress"

    @property
    def screen_id(self):
        """Get int value for 'screenId'"""
        ret = self._get_attr("screenId")
        return ret

    def query_bitmap_info(self):
        """Information about the screen bitmap.

        out address of type str

        out width of type int

        out height of type int

        out bits_per_pixel of type int

        out bytes_per_line of type int

        out bitmap_format of type :class:`BitmapFormat`

        """
        (
            address,
            width,
            height,
            bits_per_pixel,
            bytes_per_line,
            bitmap_format,
        ) = self._call("queryBitmapInfo")
        bitmap_format = BitmapFormat(bitmap_format)
        return (address, width, height, bits_per_pixel, bytes_per_line, bitmap_format)


class IFramebuffer(Interface):
    """
    Frame buffer width, in pixels.
    """

    __uuid__ = "1e8d3f27-b45c-48ae-8b36-d35e83d207aa"
    __wsmap__ = "managed"

    @property
    def width(self):
        """Get int value for 'width'
        Frame buffer width, in pixels.
        """
        ret = self._get_attr("width")
        return ret

    @property
    def height(self):
        """Get int value for 'height'
        Frame buffer height, in pixels.
        """
        ret = self._get_attr("height")
        return ret

    @property
    def bits_per_pixel(self):
        """Get int value for 'bitsPerPixel'
        Color depth, in bits per pixel.
        """
        ret = self._get_attr("bitsPerPixel")
        return ret

    @property
    def bytes_per_line(self):
        """Get int value for 'bytesPerLine'
        Scan line size, in bytes.
        """
        ret = self._get_attr("bytesPerLine")
        return ret

    @property
    def pixel_format(self):
        """Get BitmapFormat value for 'pixelFormat'
        Frame buffer pixel format. It's one of the values defined by :py:class:`BitmapFormat` .

        This attribute must never (and will never) return :py:attr:`BitmapFormat.opaque`  -- the format of the frame
        buffer must be always known.
        """
        ret = self._get_attr("pixelFormat")
        return BitmapFormat(ret)

    @property
    def height_reduction(self):
        """Get int value for 'heightReduction'
        Hint from the frame buffer about how much of the standard
        screen height it wants to use for itself. This information is
        exposed to the guest through the VESA BIOS and VMMDev interface
        so that it can use it for determining its video mode table. It
        is not guaranteed that the guest respects the value.
        """
        ret = self._get_attr("heightReduction")
        return ret

    @property
    def overlay(self):
        """Get IFramebufferOverlay value for 'overlay'
        An alpha-blended overlay which is superposed over the frame buffer.
        The initial purpose is to allow the display of icons providing
        information about the VM state, including disk activity, in front
        ends which do not have other means of doing that. The overlay is
        designed to controlled exclusively by IDisplay. It has no locking
        of its own, and any changes made to it are not guaranteed to be
        visible until the affected portion of IFramebuffer is updated. The
        overlay can be created lazily the first time it is requested. This
        attribute can also return @c null to signal that the overlay is not
        implemented.
        """
        ret = self._get_attr("overlay")
        return IFramebufferOverlay(ret)

    @property
    def win_id(self):
        """Get int value for 'winId'
        Platform-dependent identifier of the window where context of this
  