"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ChangeController = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const change_controller_function_1 = require("./change-controller-function");
/**
 * A change controller. When added to a stage in a pipeline, this will check against
 * a calendar and enable or disable the stage transition based off that calendar,
 * defaulting to closed when the calendar cannot be found or when
 * the check against it fails. It also checks to against alarms.
 */
class ChangeController extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new change_controller_function_1.ChangeControllerFunction(this, 'ChangeControllerLambda');
        // Grant permission for stage transitions
        fn.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [`${props.stage.pipeline.pipelineArn}*`],
            actions: [
                'codepipeline:EnableStageTransition',
                'codepipeline:DisableStageTransition',
                'codepipeline:GetPipelineState',
            ],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        // Grant permission to retrieve calendars
        fn.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [props.calendar.calendarArn],
            actions: ['ssm:GetCalendarState'],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        // Grant permisssion to check alarm states
        fn.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: ['cloudwatch:DescribeAlarms'],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        // Any error in the lambda function will close the time window
        const alarmDescription = 'change-controller-error-alarm';
        fn.metricErrors({ statistic: 'sum' }).createAlarm(this, 'change-controller-alarm', {
            alarmName: `ChangeController-${props.stage.pipeline.pipelineName}${props.stage.stageName}`,
            evaluationPeriods: 1,
            threshold: 1,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
            alarmDescription,
        });
        // Create a rule to run the lambda on a schedule defined by the user
        const searchTerms = props.searchTerms.concat(alarmDescription);
        new aws_events_1.Rule(this, 'Scheduler', {
            ruleName: fn.functionName,
            schedule: props.schedule,
            targets: [
                new aws_events_targets_1.LambdaFunction(fn, {
                    event: aws_events_1.RuleTargetInput.fromObject({
                        calendar: props.calendar,
                        searchTerms,
                        stageName: props.stage.stageName,
                        pipelineName: props.stage.pipeline.pipelineName,
                    }),
                }),
            ],
        });
    }
}
exports.ChangeController = ChangeController;
_a = JSII_RTTI_SYMBOL_1;
ChangeController[_a] = { fqn: "@cdklabs/cdk-codepipeline-extensions.ChangeController", version: "0.0.67" };
//# sourceMappingURL=data:application/json;base64,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