"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SlackNotifier = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_iam_1 = require("@aws-cdk/aws-iam");
class SlackNotifier extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { slackBotToken, slackSigningSecret, slackChannel, slackChannelId, slackBotName, slackBotIcon, pipeline, stageNames, } = props;
        this.environment = {
            SLACK_BOT_TOKEN: slackBotToken,
            SLACK_SIGNING_SECRET: slackSigningSecret,
            SLACK_CHANNEL: slackChannel || '',
            SLACK_CHANNEL_ID: slackChannelId || '',
        };
        if (slackBotName) {
            this.environment.SLACK_BOT_NAME = slackBotName;
        }
        if (slackBotIcon) {
            this.environment.SLACK_BOT_ICON = slackBotIcon;
        }
        const notifier = new aws_lambda_1.Function(scope, 'SlackNotifierFunction', {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code: aws_lambda_1.Code.asset(path.join(__dirname, 'lambdas', 'notifier')),
            environment: this.environment,
        });
        notifier.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [pipeline.pipelineArn],
            actions: [
                'codepipeline:GetPipelineState',
                'codepipeline:GetPipelineExecution',
            ],
        }));
        pipeline.onStateChange('SlackPipelineNotifierRule', {
            target: new aws_events_targets_1.LambdaFunction(notifier),
        });
        const stageRule = new aws_events_1.Rule(this, 'SlackStageNotifierRule');
        stageRule.addTarget(new aws_events_targets_1.LambdaFunction(notifier));
        stageRule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [pipeline.pipelineArn],
            detailType: ['CodePipeline Stage Execution State Change'],
        });
        if (stageNames) {
            stageRule.addEventPattern({
                detail: {
                    stage: stageNames,
                },
            });
        }
    }
    validate() {
        if (this.environment.SLACK_CHANNEL && this.environment.SLACK_CHANNEL_ID) {
            return [
                'Redundant Configuration: Please configure slackChannel by id (prop slackChannelId) OR name (prop slackChannel)',
            ];
        }
        if (!this.environment.SLACK_CHANNEL && !this.environment.SLACK_CHANNEL_ID) {
            return [
                'Missing Configuration: Please configure slackChannel by id (prop slackChannelId) or name (prop slackChannel)',
            ];
        }
        return [];
    }
}
exports.SlackNotifier = SlackNotifier;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2xhY2stbm90aWZpZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvc2xhY2stbm90aWZpZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBQzdCLHdDQUEwQztBQUMxQyxvREFBOEQ7QUFFOUQsb0VBQTZEO0FBQzdELG9EQUEyQztBQUMzQyw4Q0FBbUQ7QUFhbkQsTUFBYSxhQUFjLFNBQVEsZ0JBQVM7SUFHMUMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF5QjtRQUNqRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sRUFDSixhQUFhLEVBQ2Isa0JBQWtCLEVBQ2xCLFlBQVksRUFDWixjQUFjLEVBQ2QsWUFBWSxFQUNaLFlBQVksRUFDWixRQUFRLEVBQ1IsVUFBVSxHQUNYLEdBQUcsS0FBSyxDQUFDO1FBRVYsSUFBSSxDQUFDLFdBQVcsR0FBRztZQUNqQixlQUFlLEVBQUUsYUFBYTtZQUM5QixvQkFBb0IsRUFBRSxrQkFBa0I7WUFDeEMsYUFBYSxFQUFFLFlBQVksSUFBSSxFQUFFO1lBQ2pDLGdCQUFnQixFQUFFLGNBQWMsSUFBSSxFQUFFO1NBQ3ZDLENBQUM7UUFFRixJQUFJLFlBQVksRUFBRTtZQUNoQixJQUFJLENBQUMsV0FBVyxDQUFDLGNBQWMsR0FBRyxZQUFZLENBQUM7U0FDaEQ7UUFFRCxJQUFJLFlBQVksRUFBRTtZQUNoQixJQUFJLENBQUMsV0FBVyxDQUFDLGNBQWMsR0FBRyxZQUFZLENBQUM7U0FDaEQ7UUFFRCxNQUFNLFFBQVEsR0FBRyxJQUFJLHFCQUFRLENBQUMsS0FBSyxFQUFFLHVCQUF1QixFQUFFO1lBQzVELE9BQU8sRUFBRSxvQkFBTyxDQUFDLFdBQVc7WUFDNUIsT0FBTyxFQUFFLGVBQWU7WUFDeEIsSUFBSSxFQUFFLGlCQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUM3RCxXQUFXLEVBQUUsSUFBSSxDQUFDLFdBQVc7U0FDOUIsQ0FBQyxDQUFDO1FBQ0gsUUFBUSxDQUFDLGVBQWUsQ0FDdEIsSUFBSSx5QkFBZSxDQUFDO1lBQ2xCLFNBQVMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUM7WUFDakMsT0FBTyxFQUFFO2dCQUNQLCtCQUErQjtnQkFDL0IsbUNBQW1DO2FBQ3BDO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFFRixRQUFRLENBQUMsYUFBYSxDQUFDLDJCQUEyQixFQUFFO1lBQ2xELE1BQU0sRUFBRSxJQUFJLG1DQUFjLENBQUMsUUFBUSxDQUFDO1NBQ3JDLENBQUMsQ0FBQztRQUVILE1BQU0sU0FBUyxHQUFHLElBQUksaUJBQUksQ0FBQyxJQUFJLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztRQUUzRCxTQUFTLENBQUMsU0FBUyxDQUFDLElBQUksbUNBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBRWxELFNBQVMsQ0FBQyxlQUFlLENBQUM7WUFDeEIsTUFBTSxFQUFFLENBQUMsa0JBQWtCLENBQUM7WUFDNUIsU0FBUyxFQUFFLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQztZQUNqQyxVQUFVLEVBQUUsQ0FBQywyQ0FBMkMsQ0FBQztTQUMxRCxDQUFDLENBQUM7UUFFSCxJQUFJLFVBQVUsRUFBRTtZQUNkLFNBQVMsQ0FBQyxlQUFlLENBQUM7Z0JBQ3hCLE1BQU0sRUFBRTtvQkFDTixLQUFLLEVBQUUsVUFBVTtpQkFDbEI7YUFDRixDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFUyxRQUFRO1FBQ2hCLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsRUFBRTtZQUN2RSxPQUFPO2dCQUNMLGdIQUFnSDthQUNqSCxDQUFDO1NBQ0g7UUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxhQUFhLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLGdCQUFnQixFQUFFO1lBQ3pFLE9BQU87Z0JBQ0wsOEdBQThHO2FBQy9HLENBQUM7U0FDSDtRQUNELE9BQU8sRUFBRSxDQUFDO0lBQ1osQ0FBQztDQUNGO0FBcEZELHNDQW9GQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvZGUsIEZ1bmN0aW9uLCBSdW50aW1lIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBJUGlwZWxpbmUgfSBmcm9tICdAYXdzLWNkay9hd3MtY29kZXBpcGVsaW5lJztcbmltcG9ydCB7IExhbWJkYUZ1bmN0aW9uIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWV2ZW50cy10YXJnZXRzJztcbmltcG9ydCB7IFJ1bGUgfSBmcm9tICdAYXdzLWNkay9hd3MtZXZlbnRzJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuXG5leHBvcnQgaW50ZXJmYWNlIFNsYWNrTm90aWZpZXJQcm9wcyB7XG4gIHJlYWRvbmx5IHNsYWNrQm90VG9rZW46IHN0cmluZztcbiAgcmVhZG9ubHkgc2xhY2tTaWduaW5nU2VjcmV0OiBzdHJpbmc7XG4gIHJlYWRvbmx5IHNsYWNrQ2hhbm5lbD86IHN0cmluZztcbiAgcmVhZG9ubHkgc2xhY2tDaGFubmVsSWQ/OiBzdHJpbmc7XG4gIHJlYWRvbmx5IHNsYWNrQm90TmFtZT86IHN0cmluZztcbiAgcmVhZG9ubHkgc2xhY2tCb3RJY29uPzogc3RyaW5nO1xuICByZWFkb25seSBwaXBlbGluZTogSVBpcGVsaW5lO1xuICByZWFkb25seSBzdGFnZU5hbWVzPzogc3RyaW5nW107XG59XG5cbmV4cG9ydCBjbGFzcyBTbGFja05vdGlmaWVyIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgcHJvdGVjdGVkIGVudmlyb25tZW50OiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+O1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTbGFja05vdGlmaWVyUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3Qge1xuICAgICAgc2xhY2tCb3RUb2tlbixcbiAgICAgIHNsYWNrU2lnbmluZ1NlY3JldCxcbiAgICAgIHNsYWNrQ2hhbm5lbCxcbiAgICAgIHNsYWNrQ2hhbm5lbElkLFxuICAgICAgc2xhY2tCb3ROYW1lLFxuICAgICAgc2xhY2tCb3RJY29uLFxuICAgICAgcGlwZWxpbmUsXG4gICAgICBzdGFnZU5hbWVzLFxuICAgIH0gPSBwcm9wcztcblxuICAgIHRoaXMuZW52aXJvbm1lbnQgPSB7XG4gICAgICBTTEFDS19CT1RfVE9LRU46IHNsYWNrQm90VG9rZW4sXG4gICAgICBTTEFDS19TSUdOSU5HX1NFQ1JFVDogc2xhY2tTaWduaW5nU2VjcmV0LFxuICAgICAgU0xBQ0tfQ0hBTk5FTDogc2xhY2tDaGFubmVsIHx8ICcnLFxuICAgICAgU0xBQ0tfQ0hBTk5FTF9JRDogc2xhY2tDaGFubmVsSWQgfHwgJycsXG4gICAgfTtcblxuICAgIGlmIChzbGFja0JvdE5hbWUpIHtcbiAgICAgIHRoaXMuZW52aXJvbm1lbnQuU0xBQ0tfQk9UX05BTUUgPSBzbGFja0JvdE5hbWU7XG4gICAgfVxuXG4gICAgaWYgKHNsYWNrQm90SWNvbikge1xuICAgICAgdGhpcy5lbnZpcm9ubWVudC5TTEFDS19CT1RfSUNPTiA9IHNsYWNrQm90SWNvbjtcbiAgICB9XG5cbiAgICBjb25zdCBub3RpZmllciA9IG5ldyBGdW5jdGlvbihzY29wZSwgJ1NsYWNrTm90aWZpZXJGdW5jdGlvbicsIHtcbiAgICAgIHJ1bnRpbWU6IFJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBjb2RlOiBDb2RlLmFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdsYW1iZGFzJywgJ25vdGlmaWVyJykpLFxuICAgICAgZW52aXJvbm1lbnQ6IHRoaXMuZW52aXJvbm1lbnQsXG4gICAgfSk7XG4gICAgbm90aWZpZXIuYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIHJlc291cmNlczogW3BpcGVsaW5lLnBpcGVsaW5lQXJuXSxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdjb2RlcGlwZWxpbmU6R2V0UGlwZWxpbmVTdGF0ZScsXG4gICAgICAgICAgJ2NvZGVwaXBlbGluZTpHZXRQaXBlbGluZUV4ZWN1dGlvbicsXG4gICAgICAgIF0sXG4gICAgICB9KSxcbiAgICApO1xuXG4gICAgcGlwZWxpbmUub25TdGF0ZUNoYW5nZSgnU2xhY2tQaXBlbGluZU5vdGlmaWVyUnVsZScsIHtcbiAgICAgIHRhcmdldDogbmV3IExhbWJkYUZ1bmN0aW9uKG5vdGlmaWVyKSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHN0YWdlUnVsZSA9IG5ldyBSdWxlKHRoaXMsICdTbGFja1N0YWdlTm90aWZpZXJSdWxlJyk7XG5cbiAgICBzdGFnZVJ1bGUuYWRkVGFyZ2V0KG5ldyBMYW1iZGFGdW5jdGlvbihub3RpZmllcikpO1xuXG4gICAgc3RhZ2VSdWxlLmFkZEV2ZW50UGF0dGVybih7XG4gICAgICBzb3VyY2U6IFsnYXdzLmNvZGVwaXBlbGluZSddLFxuICAgICAgcmVzb3VyY2VzOiBbcGlwZWxpbmUucGlwZWxpbmVBcm5dLFxuICAgICAgZGV0YWlsVHlwZTogWydDb2RlUGlwZWxpbmUgU3RhZ2UgRXhlY3V0aW9uIFN0YXRlIENoYW5nZSddLFxuICAgIH0pO1xuXG4gICAgaWYgKHN0YWdlTmFtZXMpIHtcbiAgICAgIHN0YWdlUnVsZS5hZGRFdmVudFBhdHRlcm4oe1xuICAgICAgICBkZXRhaWw6IHtcbiAgICAgICAgICBzdGFnZTogc3RhZ2VOYW1lcyxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIHByb3RlY3RlZCB2YWxpZGF0ZSh0aGlzOiBTbGFja05vdGlmaWVyKTogc3RyaW5nW10ge1xuICAgIGlmICh0aGlzLmVudmlyb25tZW50LlNMQUNLX0NIQU5ORUwgJiYgdGhpcy5lbnZpcm9ubWVudC5TTEFDS19DSEFOTkVMX0lEKSB7XG4gICAgICByZXR1cm4gW1xuICAgICAgICAnUmVkdW5kYW50IENvbmZpZ3VyYXRpb246IFBsZWFzZSBjb25maWd1cmUgc2xhY2tDaGFubmVsIGJ5IGlkIChwcm9wIHNsYWNrQ2hhbm5lbElkKSBPUiBuYW1lIChwcm9wIHNsYWNrQ2hhbm5lbCknLFxuICAgICAgXTtcbiAgICB9XG4gICAgaWYgKCF0aGlzLmVudmlyb25tZW50LlNMQUNLX0NIQU5ORUwgJiYgIXRoaXMuZW52aXJvbm1lbnQuU0xBQ0tfQ0hBTk5FTF9JRCkge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAgJ01pc3NpbmcgQ29uZmlndXJhdGlvbjogUGxlYXNlIGNvbmZpZ3VyZSBzbGFja0NoYW5uZWwgYnkgaWQgKHByb3Agc2xhY2tDaGFubmVsSWQpIG9yIG5hbWUgKHByb3Agc2xhY2tDaGFubmVsKScsXG4gICAgICBdO1xuICAgIH1cbiAgICByZXR1cm4gW107XG4gIH1cbn1cbiJdfQ==