"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const util_1 = require("./util");
async function handler(props, event) {
    const tableNamePrefix = props.tableName.prefix;
    const tableNameSuffix = props.tableName.generateSuffix ? `${event.RequestId.substring(0, 8)}` : '';
    const tableColumns = props.tableColumns;
    const clusterProps = props;
    if (event.RequestType === 'Create') {
        const tableName = await createTable(tableNamePrefix, tableNameSuffix, tableColumns, clusterProps);
        return { PhysicalResourceId: tableName };
    }
    else if (event.RequestType === 'Delete') {
        await dropTable(event.PhysicalResourceId, clusterProps);
        return;
    }
    else if (event.RequestType === 'Update') {
        const tableName = await updateTable(event.PhysicalResourceId, tableNamePrefix, tableNameSuffix, tableColumns, clusterProps, event.OldResourceProperties);
        return { PhysicalResourceId: tableName };
    }
    else {
        /* eslint-disable-next-line dot-notation */
        throw new Error(`Unrecognized event type: ${event['RequestType']}`);
    }
}
exports.handler = handler;
async function createTable(tableNamePrefix, tableNameSuffix, tableColumns, clusterProps) {
    const tableName = tableNamePrefix + tableNameSuffix;
    const tableColumnsString = tableColumns.map(column => `${column.name} ${column.dataType}`).join();
    await util_1.executeStatement(`CREATE TABLE ${tableName} (${tableColumnsString})`, clusterProps);
    return tableName;
}
async function dropTable(tableName, clusterProps) {
    await util_1.executeStatement(`DROP TABLE ${tableName}`, clusterProps);
}
async function updateTable(tableName, tableNamePrefix, tableNameSuffix, tableColumns, clusterProps, oldResourceProperties) {
    const oldClusterProps = oldResourceProperties;
    if (clusterProps.clusterName !== oldClusterProps.clusterName || clusterProps.databaseName !== oldClusterProps.databaseName) {
        return createTable(tableNamePrefix, tableNameSuffix, tableColumns, clusterProps);
    }
    const oldTableNamePrefix = oldResourceProperties.tableName.prefix;
    if (tableNamePrefix !== oldTableNamePrefix) {
        return createTable(tableNamePrefix, tableNameSuffix, tableColumns, clusterProps);
    }
    const oldTableColumns = oldResourceProperties.tableColumns;
    if (!oldTableColumns.every(oldColumn => tableColumns.some(column => column.name === oldColumn.name && column.dataType === oldColumn.dataType))) {
        return createTable(tableNamePrefix, tableNameSuffix, tableColumns, clusterProps);
    }
    const additions = tableColumns.filter(column => {
        return !oldTableColumns.some(oldColumn => column.name === oldColumn.name && column.dataType === oldColumn.dataType);
    }).map(column => `ADD ${column.name} ${column.dataType}`);
    await Promise.all(additions.map(addition => util_1.executeStatement(`ALTER TABLE ${tableName} ${addition}`, clusterProps)));
    return tableName;
}
//# sourceMappingURL=data:application/json;base64,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