"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'can add a vpn connection to a vpc with a vpn gateway'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    asn: 65001,
                    ip: '192.0.2.1',
                },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::CustomerGateway', {
            BgpAsn: 65001,
            IpAddress: '192.0.2.1',
            Type: 'ipsec.1',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnection', {
            CustomerGatewayId: {
                Ref: 'VpcNetworkVpnConnectionCustomerGateway8B56D9AF',
            },
            Type: 'ipsec.1',
            VpnGatewayId: {
                Ref: 'VpcNetworkVpnGateway501295FA',
            },
            StaticRoutesOnly: false,
        }));
        test.done();
    },
    'with static routing'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                static: {
                    ip: '192.0.2.1',
                    staticRoutes: [
                        '192.168.10.0/24',
                        '192.168.20.0/24',
                    ],
                },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnection', {
            CustomerGatewayId: {
                Ref: 'VpcNetworkstaticCustomerGatewayAF2651CC',
            },
            Type: 'ipsec.1',
            VpnGatewayId: {
                Ref: 'VpcNetworkVpnGateway501295FA',
            },
            StaticRoutesOnly: true,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnectionRoute', {
            DestinationCidrBlock: '192.168.10.0/24',
            VpnConnectionId: {
                Ref: 'VpcNetworkstaticE33EA98C',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnectionRoute', {
            DestinationCidrBlock: '192.168.20.0/24',
            VpnConnectionId: {
                Ref: 'VpcNetworkstaticE33EA98C',
            },
        }));
        test.done();
    },
    'with tunnel options'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            preSharedKey: 'secretkey1234',
                            tunnelInsideCidr: '169.254.10.0/30',
                        },
                    ],
                },
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnection', {
            CustomerGatewayId: {
                Ref: 'VpcNetworkVpnConnectionCustomerGateway8B56D9AF',
            },
            Type: 'ipsec.1',
            VpnGatewayId: {
                Ref: 'VpcNetworkVpnGateway501295FA',
            },
            StaticRoutesOnly: false,
            VpnTunnelOptionsSpecifications: [
                {
                    PreSharedKey: 'secretkey1234',
                    TunnelInsideCidr: '169.254.10.0/30',
                },
            ],
        }));
        test.done();
    },
    'fails when ip is invalid'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.256',
                },
            },
        }), /`ip`.+IPv4/);
        test.done();
    },
    'fails when specifying more than two tunnel options'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            preSharedKey: 'secretkey1234',
                        },
                        {
                            preSharedKey: 'secretkey1234',
                        },
                        {
                            preSharedKey: 'secretkey1234',
                        },
                    ],
                },
            },
        }), /two.+`tunnelOptions`/);
        test.done();
    },
    'fails with duplicate tunnel inside cidr'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            tunnelInsideCidr: '169.254.10.0/30',
                        },
                        {
                            tunnelInsideCidr: '169.254.10.0/30',
                        },
                    ],
                },
            },
        }), /`tunnelInsideCidr`.+both tunnels/);
        test.done();
    },
    'fails when specifying an invalid pre-shared key'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            preSharedKey: '0invalid',
                        },
                    ],
                },
            },
        }), /`preSharedKey`/);
        test.done();
    },
    'fails when specifying a reserved tunnel inside cidr'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            tunnelInsideCidr: '169.254.1.0/30',
                        },
                    ],
                },
            },
        }), /`tunnelInsideCidr`.+reserved/);
        test.done();
    },
    'fails when specifying an invalid tunnel inside cidr'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            tunnelInsideCidr: '169.200.10.0/30',
                        },
                    ],
                },
            },
        }), /`tunnelInsideCidr`.+size/);
        test.done();
    },
    'can use metricTunnelState on a vpn connection'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnGateway: true,
        });
        const vpn = vpc.addVpnConnection('Vpn', {
            ip: '192.0.2.1',
        });
        // THEN
        test.deepEqual(stack.resolve(vpn.metricTunnelState()), {
            dimensions: { VpnId: { Ref: 'VpcNetworkVpnA476C58D' } },
            namespace: 'AWS/VPN',
            metricName: 'TunnelState',
            period: core_1.Duration.minutes(5),
            statistic: 'Average',
        });
        test.done();
    },
    'can use metricAllTunnelDataOut'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.deepEqual(stack.resolve(lib_1.VpnConnection.metricAllTunnelDataOut()), {
            namespace: 'AWS/VPN',
            metricName: 'TunnelDataOut',
            period: core_1.Duration.minutes(5),
            statistic: 'Sum',
        });
        test.done();
    },
    'fails when enabling vpnGateway without having subnets'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnGateway: true,
            subnetConfiguration: [],
        }), /VPN gateway/);
        test.done();
    },
    'can add a vpn connection later to a vpc that initially had no subnets'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const vpc = new lib_1.Vpc(stack, 'VpcNetwork', {
            subnetConfiguration: [],
        });
        const subnet = new lib_1.PublicSubnet(stack, 'Subnet', {
            vpcId: vpc.vpcId,
            availabilityZone: 'eu-central-1a',
            cidrBlock: '10.0.0.0/28',
        });
        vpc.publicSubnets.push(subnet);
        vpc.addVpnConnection('VPNConnection', {
            ip: '1.2.3.4',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::CustomerGateway', {
            Type: 'ipsec.1',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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