"use strict";
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
function mockVpcContextProviderWith(test, response, paramValidator) {
    const previous = core_1.ContextProvider.getValue;
    core_1.ContextProvider.getValue = (_scope, options) => {
        // do some basic sanity checks
        test.equal(options.provider, cxschema.ContextProvider.VPC_PROVIDER, `Expected provider to be: '${cxschema.ContextProvider.VPC_PROVIDER}', got: '${options.provider}'`);
        test.equal((options.props || {}).returnAsymmetricSubnets, true, `Expected options.props.returnAsymmetricSubnets to be true, got: '${(options.props || {}).returnAsymmetricSubnets}'`);
        if (paramValidator) {
            paramValidator(options.props);
        }
        return {
            value: {
                availabilityZones: [],
                isolatedSubnetIds: undefined,
                isolatedSubnetNames: undefined,
                isolatedSubnetRouteTableIds: undefined,
                privateSubnetIds: undefined,
                privateSubnetNames: undefined,
                privateSubnetRouteTableIds: undefined,
                publicSubnetIds: undefined,
                publicSubnetNames: undefined,
                publicSubnetRouteTableIds: undefined,
                ...response,
            },
        };
    };
    return previous;
}
function restoreContextProvider(previous) {
    core_1.ContextProvider.getValue = previous;
}
module.exports = {
    'Vpc.fromLookup()': {
        'requires concrete values'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => {
                lib_1.Vpc.fromLookup(stack, 'Vpc', {
                    vpcId: core_1.Lazy.stringValue({ produce: () => 'some-id' }),
                });
            }, 'All arguments to Vpc.fromLookup() must be concrete');
            test.done();
        },
        'selecting subnets by name from a looked-up VPC does not throw'(test) {
            // GIVEN
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'us-east-1', account: '123456789012' } });
            const vpc = lib_1.Vpc.fromLookup(stack, 'VPC', {
                vpcId: 'vpc-1234',
            });
            // WHEN
            vpc.selectSubnets({ subnetName: 'Bleep' });
            // THEN: no exception
            test.done();
        },
        'accepts asymmetric subnets'(test) {
            const previous = mockVpcContextProviderWith(test, {
                vpcId: 'vpc-1234',
                subnetGroups: [
                    {
                        name: 'Public',
                        type: cxapi.VpcSubnetGroupType.PUBLIC,
                        subnets: [
                            {
                                subnetId: 'pub-sub-in-us-east-1a',
                                availabilityZone: 'us-east-1a',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pub-sub-in-us-east-1b',
                                availabilityZone: 'us-east-1b',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                    {
                        name: 'Private',
                        type: cxapi.VpcSubnetGroupType.PRIVATE,
                        subnets: [
                            {
                                subnetId: 'pri-sub-1-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-1-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                ],
            }, options => {
                test.deepEqual(options.filter, {
                    isDefault: 'true',
                });
                test.equal(options.subnetGroupNameTag, undefined);
            });
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromLookup(stack, 'Vpc', {
                isDefault: true,
            });
            test.deepEqual(vpc.availabilityZones, ['us-east-1a', 'us-east-1b', 'us-east-1c', 'us-east-1d']);
            test.equal(vpc.publicSubnets.length, 2);
            test.equal(vpc.privateSubnets.length, 4);
            test.equal(vpc.isolatedSubnets.length, 0);
            restoreContextProvider(previous);
            test.done();
        },
        'selectSubnets onePerAz works on imported VPC'(test) {
            const previous = mockVpcContextProviderWith(test, {
                vpcId: 'vpc-1234',
                subnetGroups: [
                    {
                        name: 'Public',
                        type: cxapi.VpcSubnetGroupType.PUBLIC,
                        subnets: [
                            {
                                subnetId: 'pub-sub-in-us-east-1a',
                                availabilityZone: 'us-east-1a',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pub-sub-in-us-east-1b',
                                availabilityZone: 'us-east-1b',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                    {
                        name: 'Private',
                        type: cxapi.VpcSubnetGroupType.PRIVATE,
                        subnets: [
                            {
                                subnetId: 'pri-sub-1-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-1-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                ],
            }, options => {
                test.deepEqual(options.filter, {
                    isDefault: 'true',
                });
                test.equal(options.subnetGroupNameTag, undefined);
            });
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromLookup(stack, 'Vpc', {
                isDefault: true,
            });
            // WHEN
            const subnets = vpc.selectSubnets({ subnetType: lib_1.SubnetType.PRIVATE, onePerAz: true });
            // THEN: we got 2 subnets and not 4
            test.deepEqual(subnets.subnets.map(s => s.availabilityZone), ['us-east-1c', 'us-east-1d']);
            restoreContextProvider(previous);
            test.done();
        },
        'AZ in dummy lookup VPC matches AZ in Stack'(test) {
            // GIVEN
            const stack = new core_1.Stack(undefined, 'MyTestStack', { env: { account: '1234567890', region: 'dummy' } });
            const vpc = lib_1.Vpc.fromLookup(stack, 'vpc', { isDefault: true });
            // WHEN
            const subnets = vpc.selectSubnets({
                availabilityZones: stack.availabilityZones,
            });
            // THEN
            test.equals(subnets.subnets.length, 2);
            test.done();
        },
        'don\'t crash when using subnetgroup name in lookup VPC'(test) {
            // GIVEN
            const stack = new core_1.Stack(undefined, 'MyTestStack', { env: { account: '1234567890', region: 'dummy' } });
            const vpc = lib_1.Vpc.fromLookup(stack, 'vpc', { isDefault: true });
            // WHEN
            new lib_1.Instance(stack, 'Instance', {
                vpc,
                instanceType: new lib_1.InstanceType('t2.large'),
                machineImage: new lib_1.GenericLinuxImage({ dummy: 'ami-1234' }),
                vpcSubnets: {
                    subnetGroupName: 'application_layer',
                },
            });
            // THEN -- no exception occurred
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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