"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EbsDeviceVolumeType = exports.synthesizeBlockDeviceMappings = exports.BlockDeviceVolume = void 0;
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info
     * @param virtualName Virtual device name
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * Creates a new Elastic Block Storage device
     *
     * @param volumeSize The volume size, in Gibibytes (GiB)
     * @param options additional device options
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot
     *
     * @param snapshotId The snapshot ID of the volume to use
     * @param options additional device options
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index. Must be equal or greater than 0
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== EbsDeviceVolumeType.IO1) {
                construct.node.addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
            noDevice: mappingEnabled === false ? {} : undefined,
        };
    });
}
exports.synthesizeBlockDeviceMappings = synthesizeBlockDeviceMappings;
/**
 * Supported EBS volume types for blockDevices
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    /**
     * Magnetic
     */
    EbsDeviceVolumeType["STANDARD"] = "standard";
    /**
     *  Provisioned IOPS SSD
     */
    EbsDeviceVolumeType["IO1"] = "io1";
    /**
     * General Purpose SSD
     */
    EbsDeviceVolumeType["GP2"] = "gp2";
    /**
     * Throughput Optimized HDD
     */
    EbsDeviceVolumeType["ST1"] = "st1";
    /**
     * Cold HDD
     */
    EbsDeviceVolumeType["SC1"] = "sc1";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
//# sourceMappingURL=data:application/json;base64,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